import { createMessage } from '@formkit/core';
import { createObserver, removeListeners, isKilled, applyListeners, diffDeps } from '@formkit/observer';
import { cloneAny, token, eq, empty, clone, has } from '@formkit/utils';

/**
 * Message that gets set when the node is awaiting validation.
 */
const validatingMessage = createMessage({
    type: 'state',
    blocking: true,
    visible: false,
    value: true,
    key: 'validating',
});
/**
 * The actual validation plugin function. Everything must be bootstrapped here.
 *
 * @param baseRules - Base validation rules to include in the plugin. By default,
 * FormKit makes all rules in the \@formkit/rules package available via the
 * defaultConfig.
 *
 * @public
 */
function createValidationPlugin(baseRules = {}) {
    return function validationPlugin(node) {
        let propRules = cloneAny(node.props.validationRules || {});
        let availableRules = { ...baseRules, ...propRules };
        // create an observed node
        let observedNode = createObserver(node);
        const state = { input: token(), rerun: null, isPassing: true };
        let validation = cloneAny(node.props.validation);
        // If the node's validation props change, reboot:
        node.on('prop:validation', ({ payload }) => reboot(payload, propRules));
        node.on('prop:validationRules', ({ payload }) => reboot(validation, payload));
        /**
         * Reboots the validation using new rules or declarations/intents.
         * @param newValidation - New validation declaration to use
         * @param newRules - New validation rules to use
         * @returns
         */
        function reboot(newValidation, newRules) {
            var _a;
            if (eq(Object.keys(propRules || {}), Object.keys(newRules || {})) &&
                eq(validation, newValidation))
                return;
            propRules = cloneAny(newRules);
            validation = cloneAny(newValidation);
            availableRules = { ...baseRules, ...propRules };
            // Destroy all observers that may re-trigger validation on an old stack
            removeListeners(observedNode.receipts);
            // Clear existing message observers
            (_a = node.props.parsedRules) === null || _a === void 0 ? void 0 : _a.forEach((validation) => {
                var _a;
                validation.messageObserver = (_a = validation.messageObserver) === null || _a === void 0 ? void 0 : _a.kill();
            });
            // Remove all existing messages before re-validating
            node.store.filter(() => false, 'validation');
            node.props.parsedRules = parseRules(newValidation, availableRules);
            observedNode.kill();
            observedNode = createObserver(node);
            validate(observedNode, node.props.parsedRules, state);
        }
        // Validate the field when this plugin is initialized
        node.props.parsedRules = parseRules(validation, availableRules);
        validate(observedNode, node.props.parsedRules, state);
    };
}
/**
 * Given parsed validations, a value and a node, run the validations and set
 * the appropriate store messages on the node.
 * @param value - The value being validated
 * @param node - The Node this value belongs to
 * @param rules - The rules
 */
function validate(node, validations, state) {
    if (isKilled(node))
        return;
    state.input = token();
    state.isPassing = true;
    node.store.filter((message) => !message.meta.removeImmediately, 'validation');
    validations.forEach((validation) => validation.debounce && clearTimeout(validation.timer));
    if (validations.length) {
        node.store.set(validatingMessage);
        run(0, validations, node, state, false, () => {
            node.store.remove(validatingMessage.key);
        });
    }
}
/**
 * Runs validation rules recursively while collecting dependencies allowing for
 * cross-node validation rules that automatically re-trigger when a foreign
 * value is changed.
 * @param current - The index of the current validation rule
 * @param validations - The remaining validation rule stack to run
 * @param node - An observed node, the owner of this validation stack
 * @param state - An object of state information about this run
 * @param removeImmediately - Should messages created during this call be removed immediately when a new commit takes place?
 * @returns
 */
function run(current, validations, node, state, removeImmediately, complete) {
    const validation = validations[current];
    if (!validation)
        return complete();
    const currentRun = state.input;
    validation.state = null;
    function next(async, result) {
        state.isPassing = state.isPassing && !!result;
        validation.queued = false;
        const newDeps = node.stopObserve();
        applyListeners(node, diffDeps(validation.deps, newDeps), () => {
            // Event callback for when the deps change:
            try {
                node.store.set(validatingMessage);
            }
            catch (e) { }
            validation.queued = true;
            if (state.rerun)
                clearTimeout(state.rerun);
            state.rerun = setTimeout(validate, 0, node, validations, state);
        });
        validation.deps = newDeps;
        if (state.input === currentRun) {
            validation.state = result;
            if (result === false) {
                createFailedMessage(node, validation, removeImmediately || async);
            }
            else {
                removeMessage(node, validation);
            }
            if (validations.length > current + 1) {
                run(current + 1, validations, node, state, removeImmediately || async, complete);
            }
            else {
                // The validation has completed
                complete();
            }
        }
    }
    if ((!empty(node.value) || !validation.skipEmpty) &&
        (state.isPassing || validation.force)) {
        if (validation.queued) {
            runRule(validation, node, (result) => {
                result instanceof Promise
                    ? result.then((r) => next(true, r))
                    : next(false, result);
            });
        }
        else {
            // In this case our rule is not queued, so literally nothing happened that
            // would affect it, we just need to move past this rule and make no
            // modifications to state
            run(current + 1, validations, node, state, removeImmediately, complete);
        }
    }
    else {
        // This rule is not being run because either:
        //  1. The field is empty and this rule should not run when empty
        //  2. A previous validation rule is failing and this one is not forced
        // In this case we should call next validation.
        if (empty(node.value) && validation.skipEmpty && state.isPassing) {
            // This node has an empty value so its validation was skipped. So we
            // need to queue it up, we do that by starting an observation and just
            // touching the value attribute.
            node.observe();
            node.value;
            // Because this validation rule is skipped when the node's value is empty
            // so we keep the current value `state.isPassing` to the next rule execution
            // if we pass null it will be typecasted to false and all following rules
            // will be ignored including `required` rule which cause odds behavior
            next(false, state.isPassing);
        }
        else {
            next(false, null);
        }
    }
}
/**
 * Run a validation rule debounced or not.
 * @param validation - A validation to debounce
 */
function runRule(validation, node, after) {
    if (validation.debounce) {
        validation.timer = setTimeout(() => {
            node.observe();
            after(validation.rule(node, ...validation.args));
        }, validation.debounce);
    }
    else {
        node.observe();
        after(validation.rule(node, ...validation.args));
    }
}
/**
 * The messages given to this function have already been set on the node, but
 * any other validation messages on the node that are not included in this
 * stack should be removed because they have been resolved.
 * @param node - The node to operate on.
 * @param messages - A new stack of messages
 */
function removeMessage(node, validation) {
    const key = `rule_${validation.name}`;
    if (validation.messageObserver) {
        validation.messageObserver = validation.messageObserver.kill();
    }
    if (has(node.store, key)) {
        node.store.remove(key);
    }
}
/**
 *
 * @param value - The value that is failing
 * @param validation - The validation object
 */
function createFailedMessage(node, validation, removeImmediately) {
    if (isKilled(node))
        return;
    if (!validation.messageObserver) {
        validation.messageObserver = createObserver(node._node);
    }
    validation.messageObserver.watch((node) => {
        const i18nArgs = createI18nArgs(node, validation);
        return i18nArgs;
    }, (i18nArgs) => {
        const customMessage = createCustomMessage(node, validation, i18nArgs);
        // Here we short circuit the i18n system to force the output.
        const message = createMessage({
            blocking: validation.blocking,
            key: `rule_${validation.name}`,
            meta: {
                /**
                 * Use this key instead of the message root key to produce i18n validation
                 * messages.
                 */
                messageKey: validation.name,
                /**
                 * For messages that were created *by or after* a debounced or async
                 * validation rule — we make note of it so we can immediately remove them
                 * as soon as the next commit happens.
                 */
                removeImmediately,
                /**
                 * Determines if this message should be passed to localization.
                 */
                localize: !customMessage,
                /**
                 * The arguments that will be passed to the validation rules
                 */
                i18nArgs,
            },
            type: 'validation',
            value: customMessage || 'This field is not valid.',
        });
        node.store.set(message);
    });
}
/**
 * Returns a custom validation message if applicable.
 * @param node - FormKit Node
 * @param validation - The validation rule being processed.
 */
function createCustomMessage(node, validation, i18nArgs) {
    const customMessage = node.props.validationMessages &&
        has(node.props.validationMessages, validation.name)
        ? node.props.validationMessages[validation.name]
        : undefined;
    if (typeof customMessage === 'function') {
        return customMessage(...i18nArgs);
    }
    return customMessage;
}
/**
 * Creates the arguments passed to the i18n
 * @param node - The node that performed the validation
 * @param validation - The validation that failed
 */
function createI18nArgs(node, validation) {
    // If a custom message has been found, short circuit the i18n system.
    return [
        {
            node,
            name: createMessageName(node),
            args: validation.args,
        },
    ];
}
/**
 * Given a node, this returns the name that should be used in validation
 * messages. This is either the `validationLabel` prop, the `label` prop, or
 * the name of the input (in that order).
 * @param node - The node to display
 * @returns
 * @public
 */
function createMessageName(node) {
    if (typeof node.props.validationLabel === 'function') {
        return node.props.validationLabel(node);
    }
    return (node.props.validationLabel ||
        node.props.label ||
        node.props.name ||
        String(node.name));
}
/**
 * Describes hints, must also be changed in the debounceExtractor.
 */
const hintPattern = '(?:[\\*+?()0-9]+)';
/**
 * A pattern to describe rule names. Rules names can only contain letters,
 * numbers, and underscores and must start with a letter.
 */
const rulePattern = '[a-zA-Z][a-zA-Z0-9_]+';
/**
 * Regular expression for extracting rule data.
 */
const ruleExtractor = new RegExp(`^(${hintPattern}?${rulePattern})(?:\\:(.*)+)?$`, 'i');
/**
 * Validation hints are special characters preceding a validation rule, like
 * !phone
 */
const hintExtractor = new RegExp(`^(${hintPattern})(${rulePattern})$`, 'i');
/**
 * Given a hint string like ^(200)? or ^? or (200)?^ extract the hints to
 * matches.
 */
const debounceExtractor = /([\*+?]+)?(\(\d+\))([\*+?]+)?/;
/**
 * Determines if a given string is in the proper debounce format.
 */
const hasDebounce = /\(\d+\)/;
/**
 * The default values of the available validation hints.
 */
const defaultHints = {
    blocking: true,
    debounce: 0,
    force: false,
    skipEmpty: true,
    name: '',
};
/**
 * Parse validation intents and strings into validation rule stacks.
 * @param validation - Either a string a validation rules, or proper array of structured rules.
 * @internal
 */
function parseRules(validation, rules) {
    if (!validation)
        return [];
    const intents = typeof validation === 'string'
        ? extractRules(validation)
        : clone(validation);
    return intents.reduce((validations, args) => {
        let rule = args.shift();
        const hints = {};
        if (typeof rule === 'string') {
            const [ruleName, parsedHints] = parseHints(rule);
            if (has(rules, ruleName)) {
                rule = rules[ruleName];
                Object.assign(hints, parsedHints);
            }
        }
        if (typeof rule === 'function') {
            validations.push({
                rule,
                args,
                timer: 0,
                state: null,
                queued: true,
                deps: new Map(),
                ...defaultHints,
                ...fnHints(hints, rule),
            });
        }
        return validations;
    }, []);
}
/**
 * A string of validation rules written in FormKitRule notation.
 * @param validation - The string of rules
 * @internal
 */
function extractRules(validation) {
    return validation.split('|').reduce((rules, rule) => {
        const parsedRule = parseRule(rule);
        if (parsedRule) {
            rules.push(parsedRule);
        }
        return rules;
    }, []);
}
/**
 * Given a rule like confirm:password_confirm produce a FormKitValidationIntent
 * @param rule - A string representing a validation rule.
 * @returns
 */
function parseRule(rule) {
    const trimmed = rule.trim();
    if (trimmed) {
        const matches = trimmed.match(ruleExtractor);
        if (matches && typeof matches[1] === 'string') {
            const ruleName = matches[1].trim();
            const args = matches[2] && typeof matches[2] === 'string'
                ? matches[2].split(',').map((s) => s.trim())
                : [];
            return [ruleName, ...args];
        }
    }
    return false;
}
/**
 * Given a rule name, detect if there are any additional hints like !
 * @param ruleName - string representing a rule name
 * @returns
 */
function parseHints(ruleName) {
    const matches = ruleName.match(hintExtractor);
    if (!matches) {
        return [ruleName, { name: ruleName }];
    }
    const map = {
        '*': { force: true },
        '+': { skipEmpty: false },
        '?': { blocking: false },
    };
    const [, hints, rule] = matches;
    const hintGroups = hasDebounce.test(hints)
        ? hints.match(debounceExtractor) || []
        : [, hints];
    return [
        rule,
        [hintGroups[1], hintGroups[2], hintGroups[3]].reduce((hints, group) => {
            if (!group)
                return hints;
            if (hasDebounce.test(group)) {
                hints.debounce = parseInt(group.substr(1, group.length - 1));
            }
            else {
                group
                    .split('')
                    .forEach((hint) => has(map, hint) && Object.assign(hints, map[hint]));
            }
            return hints;
        }, { name: rule }),
    ];
}
/**
 * Extracts hint properties from the validation rule function itself and applies
 * them if they are not already in the set of validation hints extracted from
 * strings.
 * @param existingHints - An existing set of hints already parsed
 * @param rule - The actual rule function, which can contain hint properties
 * @returns
 */
function fnHints(existingHints, rule) {
    if (!existingHints.name) {
        existingHints.name = rule.ruleName || rule.name;
    }
    return ['skipEmpty', 'force', 'debounce', 'blocking'].reduce((hints, hint) => {
        if (has(rule, hint) && !has(hints, hint)) {
            Object.assign(hints, {
                [hint]: rule[hint],
            });
        }
        return hints;
    }, existingHints);
}
/**
 * Extracts all validation messages from the given node and all its descendants.
 * This is not reactive and must be re-called each time the messages change.
 * @param node - The FormKit node to extract validation rules from — as well as its descendants.
 * @public
 */
function getValidationMessages(node) {
    const messages = new Map();
    const extract = (n) => {
        const nodeMessages = [];
        for (const key in n.store) {
            const message = n.store[key];
            if (message.type === 'validation' &&
                message.blocking &&
                message.visible &&
                typeof message.value === 'string') {
                nodeMessages.push(message);
            }
        }
        if (nodeMessages.length) {
            messages.set(n, nodeMessages);
        }
        return n;
    };
    extract(node).walk(extract);
    return messages;
}

export { createMessageName, createValidationPlugin, getValidationMessages };
