'use strict';

var core = require('@formkit/core');
var vue = require('vue');
var utils = require('@formkit/utils');
var inputs = require('@formkit/inputs');
var defaultRules = require('@formkit/rules');
var validation = require('@formkit/validation');
var i18n = require('@formkit/i18n');
var themes = require('@formkit/themes');
var observer = require('@formkit/observer');
var dev = require('@formkit/dev');

function _interopNamespaceDefault(e) {
    var n = Object.create(null);
    if (e) {
        Object.keys(e).forEach(function (k) {
            if (k !== 'default') {
                var d = Object.getOwnPropertyDescriptor(e, k);
                Object.defineProperty(n, k, d.get ? d : {
                    enumerable: true,
                    get: function () { return e[k]; }
                });
            }
        });
    }
    n.default = e;
    return Object.freeze(n);
}

var defaultRules__namespace = /*#__PURE__*/_interopNamespaceDefault(defaultRules);

/**
 * A flag indicating if this is (likely) a server context.
 */
const isServer$2 = typeof window === 'undefined';
/**
 * A map of Vue applications to a set of callbacks to be flushed after SSR is
 * complete.
 */
const ssrCompleteRegistry = new Map();
/**
 * Flush all callbacks registered with onSSRComplete for a given app.
 * @param app - The Vue application.
 * @public
 */
function ssrComplete(app) {
    if (!isServer$2)
        return;
    const callbacks = ssrCompleteRegistry.get(app);
    if (!callbacks)
        return;
    for (const callback of callbacks) {
        callback();
    }
    callbacks.clear();
    ssrCompleteRegistry.delete(app);
}
/**
 * Register a callback for when SSR is complete. No-op if not in a server
 * context.
 * @param app - The Vue application.
 * @param callback - The callback to be called after SSR is complete.
 * @public
 */
function onSSRComplete(app, callback) {
    var _a;
    if (!isServer$2 || !app)
        return;
    if (!ssrCompleteRegistry.has(app))
        ssrCompleteRegistry.set(app, new Set());
    (_a = ssrCompleteRegistry.get(app)) === null || _a === void 0 ? void 0 : _a.add(callback);
}

/**
 * A simple flag to tell if we are running on the server or not.
 */
const isServer$1 = typeof window === 'undefined';
/**
 * A registry of memoized schemas (in JSON) to their respective render function
 * and provider registry.
 */
const memo = {};
/**
 * A map of memoized keys to how many instances of that memo are currently in
 * use.
 */
const memoKeys = {};
/**
 * This object represents the current component instance during render. It is
 * critical for linking the current instance to the data required for render.
 */
let instanceKey;
/**
 * A registry of scoped data produced during runtime that is keyed by the
 * instance object. For example data from: for-loop instances and slot data.
 */
// NOTE: This is a hack to get around the fact that the TS compiler doesn't
// understand WeakMap's allowing us to use a object as a keys, see:
// https://github.com/microsoft/TypeScript/issues/52534
// eslint-disable-next-line @typescript-eslint/ban-ts-comment
// @ts-ignore
const instanceScopes = new WeakMap();
/**
 * Indicates the a section of the schema is raw.
 */
const raw = '__raw__';
/**
 * Is a class prop.
 */
const isClassProp = /[a-zA-Z0-9\-][cC]lass$/;
/**
 * Returns a reference as a placeholder to a specific location on an object.
 * @param data - A reactive data object
 * @param token - A dot-syntax string representing the object path
 * @returns
 */
function getRef(token, data) {
    const value = vue.ref(null);
    if (token === 'get') {
        const nodeRefs = {};
        value.value = get.bind(null, nodeRefs);
        return value;
    }
    const path = token.split('.');
    vue.watchEffect(() => {
        value.value = getValue(vue.isRef(data) ? data.value : data, path);
    });
    return value;
}
/**
 * Returns a value inside a set of data objects.
 * @param sets - An array of objects to search through
 * @param path - A array of string paths easily produced by split()
 * @returns
 */
function getValue(set, path) {
    if (Array.isArray(set)) {
        for (const subset of set) {
            const value = subset !== false && getValue(subset, path);
            if (value !== undefined)
                return value;
        }
        return undefined;
    }
    let foundValue = undefined;
    let obj = set;
    for (const i in path) {
        const key = path[i];
        if (typeof obj !== 'object' || obj === null) {
            foundValue = undefined;
            break;
        }
        const currentValue = obj[key];
        if (Number(i) === path.length - 1 && currentValue !== undefined) {
            // When the value is a function, we need to bind the `this` value
            // before providing this back to the compiler.
            foundValue =
                typeof currentValue === 'function'
                    ? currentValue.bind(obj)
                    : currentValue;
            break;
        }
        obj = currentValue;
    }
    return foundValue;
}
/**
 * Get the node from the global registry
 * @param id - A dot-syntax string where the node is located.
 */
function get(nodeRefs, id) {
    if (typeof id !== 'string')
        return core.warn(650);
    if (!(id in nodeRefs))
        nodeRefs[id] = vue.ref(undefined);
    if (nodeRefs[id].value === undefined) {
        nodeRefs[id].value = null;
        const root = core.getNode(id);
        if (root)
            nodeRefs[id].value = root.context;
        core.watchRegistry(id, ({ payload: node }) => {
            nodeRefs[id].value = core.isNode(node) ? node.context : node;
        });
    }
    return nodeRefs[id].value;
}
/**
 *
 * @param library - A library of concrete components to use
 * @param schema -
 * @returns
 */
function parseSchema(library, schema, memoKey) {
    /**
     * Given an if/then/else schema node, pre-compile the node and return the
     * artifacts for the render function.
     * @param data - The schema context object
     * @param library - The available components
     * @param node - The node to parse
     */
    function parseCondition(library, node) {
        const condition = provider(core.compile(node.if), { if: true });
        const children = createElements(library, node.then);
        const alternate = node.else ? createElements(library, node.else) : null;
        return [condition, children, alternate];
    }
    /**
     * Parses a conditional if/then/else attribute statement.
     * @param data - The data object
     * @param attr - The attribute
     * @param _default - The default value
     * @returns
     */
    function parseConditionAttr(attr, _default) {
        var _a, _b;
        const condition = provider(core.compile(attr.if));
        let b = () => _default;
        let a = () => _default;
        if (typeof attr.then === 'object') {
            a = parseAttrs(attr.then, undefined);
        }
        else if (typeof attr.then === 'string' && ((_a = attr.then) === null || _a === void 0 ? void 0 : _a.startsWith('$'))) {
            a = provider(core.compile(attr.then));
        }
        else {
            a = () => attr.then;
        }
        if (utils.has(attr, 'else')) {
            if (typeof attr.else === 'object') {
                b = parseAttrs(attr.else);
            }
            else if (typeof attr.else === 'string' && ((_b = attr.else) === null || _b === void 0 ? void 0 : _b.startsWith('$'))) {
                b = provider(core.compile(attr.else));
            }
            else {
                b = () => attr.else;
            }
        }
        return () => (condition() ? a() : b());
    }
    /**
     * Parse attributes for dynamic content.
     * @param attrs - Object of attributes
     * @returns
     */
    function parseAttrs(unparsedAttrs, bindExp, _default = {}) {
        const explicitAttrs = new Set(Object.keys(unparsedAttrs || {}));
        const boundAttrs = bindExp ? provider(core.compile(bindExp)) : () => ({});
        const setters = [
            (attrs) => {
                const bound = boundAttrs();
                for (const attr in bound) {
                    if (!explicitAttrs.has(attr)) {
                        attrs[attr] = bound[attr];
                    }
                }
            },
        ];
        if (unparsedAttrs) {
            if (core.isConditional(unparsedAttrs)) {
                // This is a root conditional object that must produce an object of
                // attributes.
                const condition = parseConditionAttr(unparsedAttrs, _default);
                return condition;
            }
            // Some attributes are explicitly bound, we need to parse those ones
            // using the compiler and create a dynamic "setter".
            for (let attr in unparsedAttrs) {
                const value = unparsedAttrs[attr];
                let getValue;
                const isStr = typeof value === 'string';
                if (attr.startsWith(raw)) {
                    // attributes prefixed with __raw__ should not be parsed
                    attr = attr.substring(7);
                    getValue = () => value;
                }
                else if (isStr &&
                    value.startsWith('$') &&
                    value.length > 1 &&
                    !(value.startsWith('$reset') && isClassProp.test(attr))) {
                    // Most attribute values starting with $ should be compiled
                    // -class attributes starting with `$reset` should not be compiled
                    getValue = provider(core.compile(value));
                }
                else if (typeof value === 'object' && core.isConditional(value)) {
                    // Conditional attrs require further processing
                    getValue = parseConditionAttr(value, undefined);
                }
                else if (typeof value === 'object' && utils.isPojo(value)) {
                    // Sub-parse pojos
                    getValue = parseAttrs(value);
                }
                else {
                    // In all other cases, the value is static
                    getValue = () => value;
                }
                setters.push((attrs) => {
                    attrs[attr] = getValue();
                });
            }
        }
        return () => {
            const attrs = Array.isArray(unparsedAttrs) ? [] : {};
            setters.forEach((setter) => setter(attrs));
            return attrs;
        };
    }
    /**
     * Given a single schema node, parse it and extract the value.
     * @param data - A state object provided to each node
     * @param node - The schema node being parsed
     * @returns
     */
    function parseNode(library, _node) {
        let element = null;
        let attrs = () => null;
        let condition = false;
        let children = null;
        let alternate = null;
        let iterator = null;
        let resolve = false;
        const node = core.sugar(_node);
        if (core.isDOM(node)) {
            // This is an actual HTML DOM element
            element = node.$el;
            attrs =
                node.$el !== 'text' ? parseAttrs(node.attrs, node.bind) : () => null;
        }
        else if (core.isComponent(node)) {
            // This is a Vue Component
            if (typeof node.$cmp === 'string') {
                if (utils.has(library, node.$cmp)) {
                    element = library[node.$cmp];
                }
                else {
                    element = node.$cmp;
                    resolve = true;
                }
            }
            else {
                // in this case it must be an actual component
                element = node.$cmp;
            }
            attrs = parseAttrs(node.props, node.bind);
        }
        else if (core.isConditional(node)) {
            [condition, children, alternate] = parseCondition(library, node);
        }
        // This is the same as a "v-if" statement — not an if/else statement
        if (!core.isConditional(node) && 'if' in node) {
            condition = provider(core.compile(node.if));
        }
        else if (!core.isConditional(node) && element === null) {
            // In this odd case our element is actually a partial and
            // we only want to render the children.
            condition = () => true;
        }
        // Compile children down to a function
        if ('children' in node && node.children) {
            if (typeof node.children === 'string') {
                // We are dealing with a raw string value
                if (node.children.startsWith('$slots.')) {
                    // this is a lone text node, turn it into a slot
                    element = element === 'text' ? 'slot' : element;
                    children = provider(core.compile(node.children));
                }
                else if (node.children.startsWith('$') && node.children.length > 1) {
                    const value = provider(core.compile(node.children));
                    children = () => String(value());
                }
                else {
                    children = () => String(node.children);
                }
            }
            else if (Array.isArray(node.children)) {
                // We are dealing with node sub-children
                children = createElements(library, node.children);
            }
            else {
                // This is a conditional if/else clause
                const [childCondition, c, a] = parseCondition(library, node.children);
                children = (iterationData) => childCondition && childCondition()
                    ? c && c(iterationData)
                    : a && a(iterationData);
            }
        }
        if (core.isComponent(node)) {
            if (children) {
                // Children of components need to be provided as an object of slots
                // so we provide an object with the default slot provided as children.
                // We also create a new scope for this default slot, and then on each
                // render pass the scoped slot props to the scope.
                const produceChildren = children;
                children = (iterationData) => {
                    return {
                        default(slotData, key) {
                            var _a, _b, _c, _d;
                            // We need to switch the current instance key back to the one that
                            // originally called this component's render function.
                            const currentKey = instanceKey;
                            if (key)
                                instanceKey = key;
                            if (slotData)
                                (_a = instanceScopes.get(instanceKey)) === null || _a === void 0 ? void 0 : _a.unshift(slotData);
                            if (iterationData)
                                (_b = instanceScopes.get(instanceKey)) === null || _b === void 0 ? void 0 : _b.unshift(iterationData);
                            const c = produceChildren(iterationData);
                            // Ensure our instance key never changed during runtime
                            if (slotData)
                                (_c = instanceScopes.get(instanceKey)) === null || _c === void 0 ? void 0 : _c.shift();
                            if (iterationData)
                                (_d = instanceScopes.get(instanceKey)) === null || _d === void 0 ? void 0 : _d.shift();
                            instanceKey = currentKey;
                            return c;
                        },
                    };
                };
                children.slot = true;
            }
            else {
                // If we dont have any children, we still need to provide an object
                // instead of an empty array (which raises a warning in vue)
                children = () => ({});
            }
        }
        // Compile the for loop down
        if ('for' in node && node.for) {
            const values = node.for.length === 3 ? node.for[2] : node.for[1];
            const getValues = typeof values === 'string' && values.startsWith('$')
                ? provider(core.compile(values))
                : () => values;
            iterator = [
                getValues,
                node.for[0],
                node.for.length === 3 ? String(node.for[1]) : null,
            ];
        }
        return [condition, element, attrs, children, alternate, iterator, resolve];
    }
    /**
     * Given a particular function that produces children, ensure that the second
     * argument of all these slots is the original instance key being used to
     * render the slots.
     * @param children - The children() function that will produce slots
     */
    function createSlots(children, iterationData) {
        const slots = children(iterationData);
        const currentKey = instanceKey;
        return Object.keys(slots).reduce((allSlots, slotName) => {
            const slotFn = slots && slots[slotName];
            allSlots[slotName] = (data) => {
                return (slotFn && slotFn(data, currentKey)) || null;
            };
            return allSlots;
        }, {});
    }
    /**
     * Creates an element
     * @param data - The context data available to the node
     * @param node - The schema node to render
     * @returns
     */
    function createElement(library, node) {
        // Parses the schema node into pertinent parts
        const [condition, element, attrs, children, alternate, iterator, resolve] = parseNode(library, node);
        // This is a sub-render function (called within a render function). It must
        // only use pre-compiled features, and be organized in the most efficient
        // manner possible.
        let createNodes = ((iterationData) => {
            if (condition && element === null && children) {
                // Handle conditional if/then statements
                return condition()
                    ? children(iterationData)
                    : alternate && alternate(iterationData);
            }
            if (element && (!condition || condition())) {
                // handle text nodes
                if (element === 'text' && children) {
                    return vue.createTextVNode(String(children()));
                }
                // Handle lone slots
                if (element === 'slot' && children)
                    return children(iterationData);
                // Handle resolving components
                const el = resolve ? vue.resolveComponent(element) : element;
                // If we are rendering slots as children, ensure their instanceKey is properly added
                const slots = (children === null || children === void 0 ? void 0 : children.slot)
                    ? createSlots(children, iterationData)
                    : null;
                // Handle dom elements and components
                return vue.h(el, attrs(), (slots || (children ? children(iterationData) : [])));
            }
            return typeof alternate === 'function'
                ? alternate(iterationData)
                : alternate;
        });
        if (iterator) {
            const repeatedNode = createNodes;
            const [getValues, valueName, keyName] = iterator;
            createNodes = (() => {
                const _v = getValues();
                const values = Number.isFinite(_v)
                    ? Array(Number(_v))
                        .fill(0)
                        .map((_, i) => i)
                    : _v;
                const fragment = [];
                if (typeof values !== 'object')
                    return null;
                const instanceScope = instanceScopes.get(instanceKey) || [];
                const isArray = Array.isArray(values);
                for (const key in values) {
                    if (isArray && key in Array.prototype)
                        continue; // Fix #299
                    const iterationData = Object.defineProperty({
                        ...instanceScope.reduce((previousIterationData, scopedData) => {
                            if (previousIterationData.__idata) {
                                return { ...previousIterationData, ...scopedData };
                            }
                            return scopedData;
                        }, {}),
                        [valueName]: values[key],
                        ...(keyName !== null
                            ? { [keyName]: isArray ? Number(key) : key }
                            : {}),
                    }, '__idata', { enumerable: false, value: true });
                    instanceScope.unshift(iterationData);
                    fragment.push(repeatedNode.bind(null, iterationData)());
                    instanceScope.shift();
                }
                return fragment;
            });
        }
        return createNodes;
    }
    /**
     * Given a schema, parse it and return the resulting renderable nodes.
     * @param data - The schema context object
     * @param library - The available components
     * @param node - The node to parse
     * @returns
     */
    function createElements(library, schema) {
        if (Array.isArray(schema)) {
            const els = schema.map(createElement.bind(null, library));
            return (iterationData) => els.map((element) => element(iterationData));
        }
        // Single node to render
        const element = createElement(library, schema);
        return (iterationData) => element(iterationData);
    }
    /**
     * Data providers produced as a result of the compiler.
     */
    const providers = [];
    /**
     * Append the requisite compiler provider and return the compiled function.
     * @param compiled - A compiled function
     * @returns
     */
    function provider(compiled, hints = {}) {
        const compiledFns = new WeakMap();
        providers.push((callback, key) => {
            compiledFns.set(key, compiled.provide((tokens) => callback(tokens, hints)));
        });
        // eslint-disable-next-line @typescript-eslint/no-non-null-assertion
        return () => compiledFns.get(instanceKey)();
    }
    /**
     * Creates a new instance of a given schema — this either comes from a
     * memoized copy of the parsed schema or a freshly parsed version. An object
     * instance key, and dataProvider functions are passed in.
     * @param providerCallback - A function that is called for each required provider
     * @param key - a object representing the current instance
     */
    function createInstance(providerCallback, key) {
        var _a;
        memoKey !== null && memoKey !== void 0 ? memoKey : (memoKey = JSON.stringify(schema));
        const [render, compiledProviders] = utils.has(memo, memoKey)
            ? memo[memoKey]
            : [createElements(library, schema), providers];
        if (!isServer$1) {
            (_a = memoKeys[memoKey]) !== null && _a !== void 0 ? _a : (memoKeys[memoKey] = 0);
            memoKeys[memoKey]++;
            memo[memoKey] = [render, compiledProviders];
        }
        compiledProviders.forEach((compiledProvider) => {
            compiledProvider(providerCallback, key);
        });
        return () => {
            // Set the instance key for this pass of rendering.
            instanceKey = key;
            return render();
        };
    }
    return createInstance;
}
/**
 * Checks the current runtime scope for data.
 * @param token - The token to lookup in the current scope
 * @param defaultValue - The default ref value to use if no scope is found.
 */
function useScope(token, defaultValue) {
    const scopedData = instanceScopes.get(instanceKey) || [];
    let scopedValue = undefined;
    if (scopedData.length) {
        scopedValue = getValue(scopedData, token.split('.'));
    }
    return scopedValue === undefined ? defaultValue : scopedValue;
}
/**
 * Get the current scoped data and flatten it.
 */
function slotData(data, key) {
    return new Proxy(data, {
        get(...args) {
            let data = undefined;
            const property = args[1];
            if (typeof property === 'string') {
                const prevKey = instanceKey;
                instanceKey = key;
                data = useScope(property, undefined);
                instanceKey = prevKey;
            }
            return data !== undefined ? data : Reflect.get(...args);
        },
    });
}
/**
 * Provides data to a parsed schema.
 * @param provider - The SchemaProvider (output of calling parseSchema)
 * @param data - Data to fetch values from
 * @returns
 */
function createRenderFn(instanceCreator, data, instanceKey) {
    return instanceCreator((requirements, hints = {}) => {
        return requirements.reduce((tokens, token) => {
            if (token.startsWith('slots.')) {
                const slot = token.substring(6);
                const hasSlot = () => data.slots &&
                    utils.has(data.slots, slot) &&
                    typeof data.slots[slot] === 'function';
                if (hints.if) {
                    // If statement — dont render the slot, check if it exists
                    tokens[token] = hasSlot;
                }
                else if (data.slots) {
                    // Render the slot with current scope data
                    const scopedData = slotData(data, instanceKey);
                    tokens[token] = () => hasSlot() ? data.slots[slot](scopedData) : null;
                }
            }
            else {
                const value = getRef(token, data);
                tokens[token] = () => useScope(token, value.value);
            }
            return tokens;
        }, {});
    }, instanceKey);
}
/**
 * Removes the schema from the memo and cleans up the instance scope.
 * @param schema - The schema to remove from memo.
 * @param instanceKey - The instance key to remove.
 */
function clean(schema, memoKey, instanceKey) {
    memoKey !== null && memoKey !== void 0 ? memoKey : (memoKey = JSON.stringify(schema));
    memoKeys[memoKey]--;
    if (memoKeys[memoKey] === 0) {
        delete memoKeys[memoKey];
        const [, providers] = memo[memoKey];
        delete memo[memoKey];
        providers.length = 0;
    }
    instanceScopes.delete(instanceKey);
}
/**
 * The FormKitSchema vue component:
 *
 * @public
 */
const FormKitSchema = /* #__PURE__ */ vue.defineComponent({
    name: 'FormKitSchema',
    props: {
        schema: {
            type: [Array, Object],
            required: true,
        },
        data: {
            type: Object,
            default: () => ({}),
        },
        library: {
            type: Object,
            default: () => ({}),
        },
        memoKey: {
            type: String,
            required: false,
        },
    },
    setup(props, context) {
        var _a;
        const instance = vue.getCurrentInstance();
        let instanceKey = {};
        instanceScopes.set(instanceKey, []);
        const library = { FormKit: vue.markRaw(formkitComponent), ...props.library };
        let provider = parseSchema(library, props.schema, props.memoKey);
        let render;
        let data;
        // // Re-parse the schema if it changes:
        if (!isServer$1) {
            vue.watch(() => props.schema, (newSchema, oldSchema) => {
                var _a;
                const oldKey = instanceKey;
                instanceKey = {};
                instanceScopes.set(instanceKey, []);
                provider = parseSchema(library, props.schema, props.memoKey);
                render = createRenderFn(provider, data, instanceKey);
                if (newSchema === oldSchema) {
                    ((_a = instance === null || instance === void 0 ? void 0 : instance.proxy) === null || _a === void 0 ? void 0 : _a.$forceUpdate)();
                }
                clean(props.schema, props.memoKey, oldKey);
            }, { deep: true });
        }
        // // Watch the data object explicitly
        vue.watchEffect(() => {
            var _a;
            data = Object.assign(vue.reactive((_a = props.data) !== null && _a !== void 0 ? _a : {}), {
                slots: context.slots,
            });
            context.slots;
            render = createRenderFn(provider, data, instanceKey);
        });
        /**
         * Perform cleanup operations when the component is unmounted. This should
         * remove any memory allocations that were made during the render process.
         */
        function cleanUp() {
            // Perform cleanup operations
            clean(props.schema, props.memoKey, instanceKey);
            /* eslint-disable @typescript-eslint/no-non-null-assertion */
            if (data.node)
                data.node.destroy();
            data.slots = null;
            data = null;
            render = null;
            /* eslint-enable @typescript-eslint/no-non-null-assertion */
        }
        // For browser rendering:
        vue.onUnmounted(cleanUp);
        // For SSR rendering:
        onSSRComplete((_a = vue.getCurrentInstance()) === null || _a === void 0 ? void 0 : _a.appContext.app, cleanUp);
        return () => (render ? render() : null);
    },
});

/**
 * Flag to determine if we are running on the server.
 */
const isServer = typeof window === 'undefined';
/**
 * The symbol that represents the formkit parent injection value.
 *
 * @public
 */
const parentSymbol = Symbol('FormKitParent');
/**
 * This variable is set to the node that is currently having its schema created.
 *
 * @internal
 */
let currentSchemaNode = null;
/**
 * Returns the node that is currently having its schema created.
 *
 * @public
 */
const getCurrentSchemaNode = () => currentSchemaNode;
/**
 * The actual runtime setup function for the FormKit component.
 *
 * @param props - The props passed to the component.
 * @param context - The context passed to the component.
 */
function FormKit(props, context) {
    const node = useInput(props, context);
    if (!node.props.definition)
        core.error(600, node);
    if (node.props.definition.component) {
        return () => {
            var _a;
            return vue.h((_a = node.props.definition) === null || _a === void 0 ? void 0 : _a.component, {
                context: node.context,
            }, { ...context.slots });
        };
    }
    const schema = vue.ref([]);
    let memoKey = node.props.definition.schemaMemoKey;
    const generateSchema = () => {
        var _a, _b;
        const schemaDefinition = (_b = (_a = node.props) === null || _a === void 0 ? void 0 : _a.definition) === null || _b === void 0 ? void 0 : _b.schema;
        if (!schemaDefinition)
            core.error(601, node);
        if (typeof schemaDefinition === 'function') {
            currentSchemaNode = node;
            schema.value = schemaDefinition({ ...props.sectionsSchema });
            currentSchemaNode = null;
            if ((memoKey && props.sectionsSchema) ||
                ('memoKey' in schemaDefinition &&
                    typeof schemaDefinition.memoKey === 'string')) {
                memoKey =
                    (memoKey !== null && memoKey !== void 0 ? memoKey : schemaDefinition === null || schemaDefinition === void 0 ? void 0 : schemaDefinition.memoKey) +
                        JSON.stringify(props.sectionsSchema);
            }
        }
        else {
            schema.value = schemaDefinition;
        }
    };
    generateSchema();
    // // If someone emits the schema event, we re-generate the schema
    if (!isServer) {
        node.on('schema', () => {
            memoKey += '♻️';
            generateSchema();
        });
    }
    context.emit('node', node);
    const definitionLibrary = node.props.definition.library;
    const library = {
        FormKit: vue.markRaw(formkitComponent),
        ...definitionLibrary,
    };
    // // Expose the FormKitNode to template refs.
    context.expose({ node });
    return () => vue.h(FormKitSchema, { schema: schema.value, data: node.context, library, memoKey }, { ...context.slots });
}
/**
 * The root FormKit component. Use it to craft all inputs and structure of your
 * forms. For example:
 *
 * ```vue
 * <FormKit
 *  type="text"
 *  label="Name"
 *  help="Please enter your name"
 *  validation="required|length:2"
 * />
 * ```
 *
 * @public
 */
const formkitComponent = /* #__PURE__ */ vue.defineComponent(FormKit, {
    props: inputs.runtimeProps,
    inheritAttrs: false,
});

/**
 * The symbol that represents the formkit’s root element injection value.
 *
 * @public
 */
const rootSymbol = Symbol();
/**
 * The FormKitRoot wrapper component used to provide context to FormKit about
 * whether a FormKit input is booting in a Document or ShadowRoot. This is
 * generally only necessary when booting FormKit nodes in contexts that do not
 * have a document. For example, if running code like this:
 *
 * ```ts
 * document.getElementById(node.props.id)
 * ```
 *
 * does not work because the `document` is not available or is not in the same
 * scope, you can place a `<FormKitRoot>` component somewhere near the root of
 * of your shadowRoot and it will inform any FormKitNode child (at any depth)
 * that it is running in a shadow root. The "root" (`Document` or `ShadowRoot`)
 * will be made available to all child nodes at `node.context._root`
 *
 * @public
 */
const FormKitRoot = /* #__PURE__ */ vue.defineComponent((_p, context) => {
    const boundary = vue.ref(null);
    const showBody = vue.ref(false);
    const shadowRoot = vue.ref(undefined);
    const stopWatch = vue.watch(boundary, (el) => {
        let parent = el;
        let root = null;
        while ((parent = parent === null || parent === void 0 ? void 0 : parent.parentNode)) {
            root = parent;
            if (root instanceof ShadowRoot || root instanceof Document) {
                foundRoot(root);
                break;
            }
        }
        stopWatch();
        showBody.value = true;
    });
    vue.provide(rootSymbol, shadowRoot);
    function foundRoot(root) {
        shadowRoot.value = root;
    }
    return () => showBody.value && context.slots.default
        ? context.slots.default()
        : vue.h('template', { ref: boundary });
});

/**
 * The Create a new instance of the FormKit plugin for Vue.
 *
 * @param app - A Vue application
 * @param config - FormKit Vue plugin configuration options
 *
 * @internal
 */
function createPlugin(app, options) {
    app
        .component(options.alias || 'FormKit', formkitComponent)
        .component(options.schemaAlias || 'FormKitSchema', FormKitSchema);
    return {
        get: core.getNode,
        setLocale: (locale) => {
            var _a;
            if ((_a = options.config) === null || _a === void 0 ? void 0 : _a.rootConfig) {
                options.config.rootConfig.locale = locale;
            }
        },
        clearErrors: core.clearErrors,
        setErrors: core.setErrors,
        submit: core.submitForm,
        reset: core.reset,
    };
}
/**
 * The symbol key for accessing the FormKit node options.
 *
 * @public
 */
const optionsSymbol = Symbol.for('FormKitOptions');
/**
 * The symbol key for accessing FormKit root configuration.
 *
 * @public
 */
const configSymbol = Symbol.for('FormKitConfig');
/**
 * Create the FormKit plugin.
 *
 * @public
 */
const plugin = {
    install(app, _options) {
        /**
         * Extend the default configuration options.
         */
        const options = Object.assign({
            alias: 'FormKit',
            schemaAlias: 'FormKitSchema',
        }, typeof _options === 'function' ? _options() : _options);
        /**
         * The root configuration options.
         */
        const rootConfig = core.createConfig(options.config || {});
        /**
         * We dont want to explicitly provide any "config" options, only a root
         * config option — so here we override the existing config options.
         */
        options.config = { rootConfig };
        /**
         * Register the global $formkit plugin property.
         */
        app.config.globalProperties.$formkit = createPlugin(app, options);
        /**
         * Provide the config to the application for injection.
         */
        app.provide(optionsSymbol, options);
        /**
         * Provide the root config to the application.
         */
        app.provide(configSymbol, rootConfig);
        /**
         * Register the FormKit component globally.
         */
        if (typeof window !== 'undefined') {
            globalThis.__FORMKIT_CONFIGS__ = (globalThis.__FORMKIT_CONFIGS__ || []).concat([rootConfig]);
        }
    },
};

const isBrowser = typeof window !== 'undefined';
/**
 * Props that are extracted from the attrs object.
 * TODO: Currently local, this should probably exported to a inputs or another
 * package.
 */
const pseudoProps = [
    'help',
    'label',
    'ignore',
    'disabled',
    'preserve',
    /^preserve(-e|E)rrors/,
    /^[a-z]+(?:-visibility|Visibility|-behavior|Behavior)$/,
    /^[a-zA-Z-]+(?:-class|Class)$/,
    'prefixIcon',
    'suffixIcon',
    /^[a-zA-Z-]+(?:-icon|Icon)$/,
];
/**
 * Given some props, map those props to individualized props internally.
 * @param node - A formkit node
 * @param props - Some props that may include a classes object
 */
function classesToNodeProps(node, props) {
    if (props.classes) {
        Object.keys(props.classes).forEach((key) => {
            if (typeof key === 'string') {
                node.props[`_${key}Class`] = props.classes[key];
                // We need to ensure Vue is aware that we want to actually observe the
                // child values too, so we touch them here.
                if (utils.isObject(props.classes[key]) && key === 'inner')
                    Object.values(props.classes[key]);
            }
        });
    }
}
/**
 * Extracts known FormKit listeners.
 * @param props - Extract known FormKit listeners.
 * @returns
 */
function onlyListeners(props) {
    if (!props)
        return {};
    const knownListeners = ['Submit', 'SubmitRaw', 'SubmitInvalid'].reduce((listeners, listener) => {
        const name = `on${listener}`;
        if (name in props) {
            if (typeof props[name] === 'function') {
                listeners[name] = props[name];
            }
        }
        return listeners;
    }, {});
    return knownListeners;
}
/**
 * A composable for creating a new FormKit node.
 *
 * @param type - The type of node (input, group, list)
 * @param attrs - The FormKit "props" — which is really the attrs list.
 *
 * @returns {@link @formkit/core#FormKitNode | FormKitNode}
 *
 * @public
 */
function useInput(props, context, options = {}) {
    var _a;
    /**
     * The configuration options, these are provided by either the plugin or by
     * explicit props.
     */
    const config = Object.assign({}, vue.inject(optionsSymbol) || {}, options);
    /**
     * The root element — generally this is either a Document or ShadowRoot.
     */
    const __root = vue.inject(rootSymbol, vue.ref(isBrowser ? document : undefined));
    /**
     * The current instance.
     */
    const instance = vue.getCurrentInstance();
    /**
     * Extracts the listeners.
     */
    const listeners = onlyListeners(instance === null || instance === void 0 ? void 0 : instance.vnode.props);
    /**
     * Determines if the prop is v-modeled. Credit to:
     * {@link https://github.com/LinusBorg | Thorsten Lünborg}
     * for coming up with this solution.
     */
    const isVModeled = ['modelValue', 'model-value'].some((prop) => { var _a; return prop in ((_a = instance === null || instance === void 0 ? void 0 : instance.vnode.props) !== null && _a !== void 0 ? _a : {}); });
    // Track if the input has mounted or not.
    let isMounted = false;
    vue.onMounted(() => {
        isMounted = true;
    });
    /**
     * Determines if the object being passed as a v-model is reactive.
     */
    // const isReactiveVModel = isVModeled && isReactive(props.modelValue)
    /**
     * Define the initial component
     */
    const value = props.modelValue !== undefined
        ? props.modelValue
        : utils.cloneAny(context.attrs.value);
    /**
     * Creates the node's initial props from the context, props, and definition
     * @returns
     */
    function createInitialProps() {
        var _a;
        const initialProps = {
            ...utils.nodeProps(props),
            ...listeners,
            type: (_a = props.type) !== null && _a !== void 0 ? _a : 'text',
            __root: __root.value,
            __slots: context.slots,
        };
        const attrs = utils.except(utils.nodeProps(context.attrs), pseudoProps);
        if (!attrs.key)
            attrs.key = utils.token();
        initialProps.attrs = attrs;
        const propValues = utils.only(utils.nodeProps(context.attrs), pseudoProps);
        for (const propName in propValues) {
            initialProps[utils.camel(propName)] = propValues[propName];
        }
        const classesProps = { props: {} };
        classesToNodeProps(classesProps, props);
        Object.assign(initialProps, classesProps.props);
        if (typeof initialProps.type !== 'string') {
            initialProps.definition = initialProps.type;
            delete initialProps.type;
        }
        return initialProps;
    }
    /**
     * Create the FormKitNode.
     */
    const initialProps = createInitialProps();
    /**
     * The parent node.
     */
    const parent = initialProps.ignore
        ? null
        : props.parent || vue.inject(parentSymbol, null);
    const node = core.createNode(utils.extend(config || {}, {
        name: props.name || undefined,
        value,
        parent,
        plugins: (config.plugins || []).concat((_a = props.plugins) !== null && _a !== void 0 ? _a : []),
        config: props.config || {},
        props: initialProps,
        index: props.index,
        sync: !!utils.undefine(context.attrs.sync || context.attrs.dynamic),
    }, false, true));
    /**
     * If no definition has been assigned at this point — we're out!
     */
    if (!node.props.definition)
        core.error(600, node);
    /**
     * All props that are bound "late" (after node creation) — are added to a set
     * which is used to watch the context.attrs object.
     */
    const lateBoundProps = vue.ref(new Set(node.props.definition.props || []));
    /**
     * Any additional props added at a "later" time should also be part of the
     * late bound props.
     */
    node.on('added-props', ({ payload: lateProps }) => {
        if (Array.isArray(lateProps))
            lateProps.forEach((newProp) => lateBoundProps.value.add(newProp));
    });
    /**
     * These prop names must be assigned.
     */
    const pseudoPropNames = vue.computed(() => pseudoProps.concat([...lateBoundProps.value]).reduce((names, prop) => {
        if (typeof prop === 'string') {
            names.push(utils.camel(prop));
            names.push(utils.kebab(prop));
        }
        else {
            names.push(prop);
        }
        return names;
    }, []));
    /* Splits Classes object into discrete props for each key */
    vue.watchEffect(() => classesToNodeProps(node, props));
    /**
     * The props object already has properties even if they start as "undefined"
     * so we can loop over them and individual watchEffect to prevent responding
     * inappropriately.
     */
    const passThrough = utils.nodeProps(props);
    for (const prop in passThrough) {
        vue.watch(() => props[prop], () => {
            if (props[prop] !== undefined) {
                node.props[prop] = props[prop];
            }
        });
    }
    // Ensure the root always stays up to date.
    vue.watchEffect(() => {
        node.props.__root = __root.value;
    });
    /**
     * Watch "pseudoProp" attributes explicitly.
     */
    const attributeWatchers = new Set();
    const possibleProps = utils.nodeProps(context.attrs);
    vue.watchEffect(() => {
        watchAttributes(utils.only(possibleProps, pseudoPropNames.value));
    });
    /**
     * Defines attributes that should be used as props.
     * @param attrProps - Attributes that should be used as props instead
     */
    function watchAttributes(attrProps) {
        attributeWatchers.forEach((stop) => {
            stop();
            attributeWatchers.delete(stop);
        });
        for (const prop in attrProps) {
            const camelName = utils.camel(prop);
            attributeWatchers.add(vue.watch(() => context.attrs[prop], () => {
                node.props[camelName] = context.attrs[prop];
            }));
        }
    }
    /**
     * Watch and dynamically set attribute values, those values that are not
     * props and are not pseudoProps
     */
    vue.watchEffect(() => {
        const attrs = utils.except(utils.nodeProps(context.attrs), pseudoPropNames.value);
        // An explicit exception to ensure naked "multiple" attributes appear on the
        // outer wrapper as data-multiple="true"
        if ('multiple' in attrs)
            attrs.multiple = utils.undefine(attrs.multiple);
        if (typeof attrs.onBlur === 'function') {
            attrs.onBlur = utils.oncePerTick(attrs.onBlur);
        }
        node.props.attrs = Object.assign({}, node.props.attrs || {}, attrs);
    });
    /**
     * Add any/all "prop" errors to the store.
     */
    vue.watchEffect(() => {
        var _a;
        const messages = ((_a = props.errors) !== null && _a !== void 0 ? _a : []).map((error) => core.createMessage({
            key: utils.slugify(error),
            type: 'error',
            value: error,
            meta: { source: 'prop' },
        }));
        node.store.apply(messages, (message) => message.type === 'error' && message.meta.source === 'prop');
    });
    /**
     * Add input errors.
     */
    if (node.type !== 'input') {
        const sourceKey = `${node.name}-prop`;
        vue.watchEffect(() => {
            var _a;
            const inputErrors = (_a = props.inputErrors) !== null && _a !== void 0 ? _a : {};
            const keys = Object.keys(inputErrors);
            if (!keys.length)
                node.clearErrors(true, sourceKey);
            const messages = keys.reduce((messages, key) => {
                let value = inputErrors[key];
                if (typeof value === 'string')
                    value = [value];
                if (Array.isArray(value)) {
                    messages[key] = value.map((error) => core.createMessage({
                        key: error,
                        type: 'error',
                        value: error,
                        meta: { source: sourceKey },
                    }));
                }
                return messages;
            }, {});
            node.store.apply(messages, (message) => message.type === 'error' && message.meta.source === sourceKey);
        });
    }
    /**
     * Watch the config prop for any changes.
     */
    vue.watchEffect(() => Object.assign(node.config, props.config));
    /**
     * Produce another parent object.
     */
    if (node.type !== 'input') {
        vue.provide(parentSymbol, node);
    }
    // let inputTimeout: number | undefined
    let clonedValueBeforeVmodel = undefined;
    /**
     * Explicitly watch the input value, and emit changes (lazy)
     */
    node.on('modelUpdated', () => {
        var _a, _b;
        // Emit the values after commit
        context.emit('inputRaw', (_a = node.context) === null || _a === void 0 ? void 0 : _a.value, node);
        if (isMounted) {
            context.emit('input', (_b = node.context) === null || _b === void 0 ? void 0 : _b.value, node);
        }
        if (isVModeled && node.context) {
            clonedValueBeforeVmodel = utils.cloneAny(node.value);
            context.emit('update:modelValue', utils.shallowClone(node.value));
        }
    });
    /**
     * Enabled support for v-model, using this for groups/lists is not recommended
     */
    if (isVModeled) {
        vue.watch(vue.toRef(props, 'modelValue'), (value) => {
            if (!utils.eq(clonedValueBeforeVmodel, value)) {
                node.input(value, false);
            }
        }, { deep: true });
        /**
         * On initialization, if the node’s value was updated (like in a plugin
         * hook) then we should emit a `modelUpdated` event.
         */
        if (node.value !== value) {
            node.emit('modelUpdated');
        }
    }
    /**
     * When this input shuts down, we need to "delete" the node too.
     */
    vue.onBeforeUnmount(() => node.destroy());
    return node;
}

let totalCreated = 1;
/**
 * Determine if the given object is a vue component.
 *
 * @param obj - Object or function
 * @returns
 * @public
 */
function isComponent(obj) {
    return ((typeof obj === 'function' && obj.length === 2) ||
        (typeof obj === 'object' &&
            !Array.isArray(obj) &&
            !('$el' in obj) &&
            !('$cmp' in obj) &&
            !('if' in obj)));
}
/**
 * Creates a new input from schema or a Vue component with the "standard"
 * FormKit features in place such as labels, help text, validation messages, and
 * class support.
 *
 * @param schemaOrComponent - The actual schema of the input or the component.
 * @param definitionOptions - Any options in the FormKitTypeDefinition you want
 * to define.
 *
 * @returns {@link @formkit/core#FormKitTypeDefinition | FormKitTypeDefinition}
 *
 * @public
 */
function createInput(schemaOrComponent, definitionOptions = {}) {
    const definition = {
        type: 'input',
        ...definitionOptions,
    };
    let schema;
    if (isComponent(schemaOrComponent)) {
        const cmpName = `SchemaComponent${totalCreated++}`;
        schema = inputs.createSection('input', () => ({
            $cmp: cmpName,
            props: {
                context: '$node.context',
            },
        }));
        definition.library = { [cmpName]: vue.markRaw(schemaOrComponent) };
    }
    else if (typeof schemaOrComponent === 'function') {
        schema = schemaOrComponent;
    }
    else {
        schema = inputs.createSection('input', () => utils.cloneAny(schemaOrComponent));
    }
    // Use the default wrapping schema
    definition.schema = inputs.useSchema(schema || 'Schema undefined');
    if (!definition.schemaMemoKey) {
        definition.schemaMemoKey = `${Math.random()}`;
    }
    return definition;
}

function defineFormKitConfig(config) {
    return () => (typeof config === 'function' ? config() : config);
}

/**
 * Fetches the list of inputs from the remote schema repository
 */
async function fetchInputList() {
    const response = await fetch('https://raw.githubusercontent.com/formkit/input-schemas/master/index.json');
    const json = await response.json();
    return json;
}
/**
 * Fetches the list of inputs from the remote schema repository
 */
async function fetchInputSchema(input) {
    try {
        const response = await fetch(`https://raw.githubusercontent.com/formkit/input-schemas/master/schemas/${input}.json`);
        const json = await response.json();
        return json;
    }
    catch (error) {
        console.error(error);
    }
}
/**
 * Renders FormKit components fetched from a remote schema repository.
 * This is a kitchen sink component that is used for testing purposes.
 * It shows inputs in various states and configurations.
 *
 * @public
 */
const FormKitKitchenSink = /* #__PURE__ */ vue.defineComponent({
    name: 'FormKitKitchenSink',
    props: {
        pro: {
            type: Boolean,
            default: true,
        },
        schemas: {
            type: Array,
            required: false,
        },
    },
    async setup(props) {
        const inputList = await fetchInputList();
        const schemas = {};
        const promises = [];
        const activeTab = vue.ref('');
        if (!props.schemas) {
            const coreInputPromises = inputList.core.map(async (schema) => {
                const response = await fetchInputSchema(schema);
                schemas[schema] = response;
            });
            promises.push(...coreInputPromises);
            if (props.pro) {
                const proInputPromises = inputList.pro.map(async (schema) => {
                    const response = await fetchInputSchema(schema);
                    schemas[schema] = response;
                });
                promises.push(...proInputPromises);
            }
        }
        else {
            const schemaPromises = props.schemas.map(async (schema) => {
                const response = await fetchInputSchema(`${schema}`);
                schemas[`${schema}`] = response;
            });
            promises.push(...schemaPromises);
        }
        // supporting schema functions for async input states
        const data = {
            asyncLoader: async () => {
                // eslint-disable-next-line @typescript-eslint/no-empty-function
                return await new Promise(() => { });
            },
            paginatedLoader: async ({ page, hasNextPage, }) => {
                const base = (page - 1) * 10;
                hasNextPage();
                return Array.from({ length: 10 }, (_, i) => `Option ${base + i + 1}`);
            },
        };
        await Promise.all(promises);
        const inputKeys = Object.keys(schemas);
        // create friendly labels for use in tabs
        const formNames = inputKeys.map((key) => {
            if (key.startsWith('form/')) {
                switch (key) {
                    case 'form/tshirt':
                        return {
                            id: key,
                            name: 'Order Form',
                        };
                    default:
                        const name = key.replace('form/', '');
                        return {
                            id: key,
                            name: name.charAt(0).toUpperCase() + name.slice(1) + ' Form',
                        };
                }
            }
            return {
                id: key,
                name: '',
            };
        });
        const filteredFormNames = formNames.filter((form) => form.name !== '');
        const forms = inputKeys.filter((schema) => {
            return schema.startsWith('form/');
        });
        const inputs = inputKeys.filter((schema) => !schema.startsWith('form/'));
        const tabs = [];
        if (inputs.length) {
            tabs.push({
                id: 'kitchen-sink',
                name: 'Kitchen Sink',
            });
        }
        if (forms.length) {
            tabs.push(...filteredFormNames);
        }
        if (tabs.length) {
            activeTab.value = tabs[0].id;
        }
        // collection of all inputs to be rendered in the "kitchen sink" tab
        const kitchenSinkRenders = inputs.map((inputName) => {
            const schemaDefinitions = schemas[inputName];
            const schemaRenders = schemaDefinitions.map((schema) => {
                return vue.h('div', {
                    class: 'formkit-specimen flex flex-col p-2 max-w-[75vw]',
                }, [
                    vue.h(FormKitSchema, {
                        schema: schema,
                        data: data,
                    }),
                ]);
            });
            return vue.h('div', {
                class: 'formkit-input-section mr-[min(350px,25vw)]',
                'data-type': inputName,
            }, [
                vue.h('span', {
                    class: `
                formkit-input-type block font-bold text-neutral-900 border-b border-neutral-100 text-3xl mb-8 pb-2 capitalize
                dark:border-neutral-800 dark:text-neutral-50
            `,
                }, inputName),
                vue.h('div', {
                    class: 'formkit-specimen-group grid grid-cols-[repeat(auto-fit,400px)] mb-16',
                }, schemaRenders),
            ]);
        });
        const formRenders = vue.computed(() => {
            return filteredFormNames.map((form) => {
                const schemaDefinition = schemas[form.id];
                return vue.h('div', {}, activeTab.value === form.id
                    ? [
                        vue.h('div', {
                            class: 'w-full bg-white rounded border border-neutral-100 shadow-lg max-w-[800px] p-[min(5vw,5rem)] dark:bg-neutral-900 dark:border-neutral-800 dark:shadow-3xl',
                        }, [
                            vue.h(FormKitSchema, {
                                schema: schemaDefinition[0],
                                data: data,
                            }),
                        ]),
                    ]
                    : '');
            });
        });
        const tabBar = vue.computed(() => {
            return vue.h('div', {
                class: 'formkit-kitchen-sink-tabs mt-4 mb-8',
            }, tabs.map((tab) => {
                return vue.h('span', {
                    class: 'formkit-kitchen-sink-tab inline-block mr-4 cursor-pointer px-4 py-2 border border-neutral-100 text-neutral-800 rounded data-[active]:bg-neutral-800 data-[active]:text-neutral-50 hover:bg-neutral-100 hover:text-neutral-900 dark:border-neutral-800 dark:text-neutral-50 dark:data-[active]:bg-neutral-100 dark:data-[active]:text-neutral-800 dark:hover:bg-neutral-800 dark:hover:text-neutral-50',
                    'data-tab': tab.id,
                    'data-active': activeTab.value === tab.id || undefined,
                    onClick: () => {
                        console.log(tab.id);
                        activeTab.value = tab.id;
                    },
                }, tab.name);
            }));
        });
        return () => {
            return vue.h('div', {
                class: 'formkit-kitchen-sink my-12',
            }, [
                tabs.length > 1 ? tabBar.value : '',
                vue.h('div', {
                    class: 'formkit-inputs',
                }, kitchenSinkRenders.length && activeTab.value === 'kitchen-sink'
                    ? kitchenSinkRenders
                    : []),
                ...formRenders.value,
            ]);
        };
    },
});

/**
 * @internal
 */
const messages$1 = inputs.createSection('messages', () => ({
    $el: 'ul',
    if: '$fns.length($messages)',
}));
/**
 * @internal
 */
const message$1 = inputs.createSection('message', () => ({
    $el: 'li',
    for: ['message', '$messages'],
    attrs: {
        key: '$message.key',
        id: `$id + '-' + $message.key`,
        'data-message-type': '$message.type',
    },
}));
/**
 * The actual schema to render for the messages.
 */
const definition$1 = /* #__PURE__ */ messages$1(/* #__PURE__ */ message$1('$message.value'));
/**
 * Renders the messages for a parent node, or any node explicitly passed to it.
 * @public
 */
const FormKitMessages = /* #__PURE__ */ vue.defineComponent({
    props: {
        node: {
            type: Object,
            required: false,
        },
        sectionsSchema: {
            type: Object,
            default: {},
        },
        defaultPosition: {
            type: [String, Boolean],
            default: false,
        },
    },
    setup(props, context) {
        const node = vue.computed(() => {
            return props.node || vue.inject(parentSymbol, undefined);
        });
        vue.watch(node, () => {
            var _a;
            if (((_a = node.value) === null || _a === void 0 ? void 0 : _a.context) && !utils.undefine(props.defaultPosition)) {
                node.value.context.defaultMessagePlacement = false;
            }
        }, { immediate: true });
        const schema = definition$1(props.sectionsSchema || {});
        const data = vue.computed(() => {
            var _a, _b, _c, _d, _e, _f;
            return {
                messages: ((_b = (_a = node.value) === null || _a === void 0 ? void 0 : _a.context) === null || _b === void 0 ? void 0 : _b.messages) || {},
                fns: ((_d = (_c = node.value) === null || _c === void 0 ? void 0 : _c.context) === null || _d === void 0 ? void 0 : _d.fns) || {},
                classes: ((_f = (_e = node.value) === null || _e === void 0 ? void 0 : _e.context) === null || _f === void 0 ? void 0 : _f.classes) || {},
            };
        });
        return () => {
            var _a;
            return ((_a = node.value) === null || _a === void 0 ? void 0 : _a.context)
                ? vue.h(FormKitSchema, { schema, data: data.value }, { ...context.slots })
                : null;
        };
    },
});

/**
 * A composable to provide a given configuration to all children.
 * @param config - A FormKit configuration object or a function
 */
function useConfig(config) {
    const options = Object.assign({
        alias: 'FormKit',
        schemaAlias: 'FormKitSchema',
    }, typeof config === 'function' ? config() : config);
    /**
     * The root configuration options.
     */
    const rootConfig = core.createConfig(options.config || {});
    /**
     * We dont want to explicitly provide any "config" options, only a root
     * config option — so here we override the existing config options.
     */
    options.config = { rootConfig };
    /**
     * Provide the config to children.
     */
    vue.provide(optionsSymbol, options);
    /**
     * Provide the root config to the children.
     */
    vue.provide(configSymbol, rootConfig);
    /**
     * Register the FormKit component globally.
     */
    if (typeof window !== 'undefined') {
        globalThis.__FORMKIT_CONFIGS__ = (globalThis.__FORMKIT_CONFIGS__ || []).concat([rootConfig]);
    }
}
/**
 * The FormKitProvider component provides the FormKit config to the children.
 *
 * @public
 */
const FormKitProvider = /* #__PURE__ */ vue.defineComponent(function FormKitProvider(props, { slots }) {
    const options = {};
    if (props.config) {
        useConfig(props.config);
    }
    return () => (slots.default ? slots.default(options) : null);
}, { props: ['config'], name: 'FormKitProvider' });
/**
 * The FormKitConfigLoader is an async component (meaning it needs a parent or
 * grandparent Suspense component to render) that loads the FormKit config and
 * provides it to the children.
 *
 * @internal
 */
const FormKitConfigLoader = /* #__PURE__ */ vue.defineComponent(async function FormKitConfigLoader(props, context) {
    var _a;
    let config = {};
    if (props.configFile) {
        const configFile = await import(
        /*@__formkit.config.ts__*/ /* @vite-ignore */ /* webpackIgnore: true */ props.configFile);
        config = 'default' in configFile ? configFile.default : configFile;
    }
    // Ensure this a factory function for runtimeConfig in nuxt.
    if (typeof config === 'function') {
        config = config();
    }
    /* @__default-config__ */
    const useDefaultConfig = (_a = props.defaultConfig) !== null && _a !== void 0 ? _a : true;
    if (useDefaultConfig) {
        const { defaultConfig } = await Promise.resolve().then(function () { return defaultConfig$1; });
        config = /* @__PURE__ */ defaultConfig(config);
    }
    /* @__default-config__ */
    return () => vue.h(FormKitProvider, { config }, context.slots);
}, {
    props: ['defaultConfig', 'configFile'],
});
/**
 * The FormKitLazyProvider component performs 2 HOC functions:
 *
 * 1. It checks if a FormKit config has already been provided, if it has it will
 *   render the children immediately.
 * 2. If a config has not been provided, it will render a Suspense component
 *    which will render the children once the config has been loaded by using
 *    the FormKitConfigLoader component.
 *
 * @public
 */
const FormKitLazyProvider = /* #__PURE__ */ vue.defineComponent(function FormKitLazyProvider(props, context) {
    const config = vue.inject(optionsSymbol, null);
    if (config) {
        // If there is already a config provided, render the children immediately.
        return () => { var _a; return (((_a = context.slots) === null || _a === void 0 ? void 0 : _a.default) ? context.slots.default() : null); };
    }
    const instance = vue.getCurrentInstance();
    if (instance.suspense) {
        // If there is a suspense boundary already in place, we can render the
        // config loader without another suspense boundary.
        return () => vue.h(FormKitConfigLoader, props, {
            default: () => { var _a; return ((_a = context.slots) === null || _a === void 0 ? void 0 : _a.default) ? context.slots.default() : null; },
        });
    }
    // If there is no suspense boundary, and no config, we render the suspense
    // boundary and the config loader.
    return () => vue.h(vue.Suspense, null, {
        ...context.slots,
        default: () => vue.h(FormKitConfigLoader, props, context.slots),
    });
}, {
    props: ['defaultConfig', 'configFile'],
});

const summary = inputs.createSection('summary', () => ({
    $el: 'div',
    attrs: {
        'aria-live': 'polite',
    },
}));
const summaryInner = inputs.createSection('summaryInner', () => ({
    $el: 'div',
    if: '$summaries.length && $showSummaries',
}));
/**
 * @internal
 */
const messages = inputs.createSection('messages', () => ({
    $el: 'ul',
    if: '$summaries.length && $showSummaries',
}));
/**
 * @internal
 */
const message = inputs.createSection('message', () => ({
    $el: 'li',
    for: ['summary', '$summaries'],
    attrs: {
        key: '$summary.key',
        'data-message-type': '$summary.type',
    },
}));
const summaryHeader = inputs.createSection('summaryHeader', () => ({
    $el: 'h2',
    attrs: {
        id: '$id',
    },
}));
const messageLink = inputs.createSection('messageLink', () => ({
    $el: 'a',
    attrs: {
        id: '$summary.key',
        href: '$: "#" + $summary.id',
        onClick: '$jumpLink',
    },
}));
/**
 * The actual schema to render for the messages.
 */
const definition = /* #__PURE__ */ summary(/* #__PURE__ */ summaryInner(/* #__PURE__ */ summaryHeader('$summaryHeader'), /* #__PURE__ */ messages(/* #__PURE__ */ message(/* #__PURE__ */ messageLink('$summary.message')))));
/**
 * Renders the messages for a parent node, or any node explicitly passed to it.
 * @public
 */
const FormKitSummary = /* #__PURE__ */ vue.defineComponent({
    props: {
        node: {
            type: Object,
            required: false,
        },
        sectionsSchema: {
            type: Object,
            default: {},
        },
    },
    emits: {
        /* eslint-disable-next-line @typescript-eslint/no-unused-vars */
        show: (_summaries) => true,
    },
    setup(props, context) {
        var _a, _b;
        const id = `summary-${utils.token()}`;
        const node = vue.computed(() => {
            return props.node || vue.inject(parentSymbol, undefined);
        });
        if (!node)
            throw new Error('FormKitSummary must have a FormKit parent or use the node prop.');
        const summaryContexts = vue.ref([]);
        const showSummaries = vue.ref(false);
        const summaries = vue.computed(() => {
            const summarizedMessages = [];
            summaryContexts.value.forEach((context) => {
                for (const idx in context.messages) {
                    const message = context.messages[idx];
                    if (typeof message.value !== 'string')
                        continue;
                    summarizedMessages.push({
                        message: message.value,
                        id: context.id,
                        key: `${context.id}-${message.key}`,
                        type: message.type,
                    });
                }
            });
            return summarizedMessages;
        });
        const addContexts = () => {
            var _a;
            summaryContexts.value = [];
            (_a = node.value) === null || _a === void 0 ? void 0 : _a.walk((child) => child.context && summaryContexts.value.push(child.context));
        };
        (_a = node.value) === null || _a === void 0 ? void 0 : _a.on('submit-raw', async () => {
            var _a, _b;
            addContexts();
            if (summaries.value.length === 0)
                return;
            context.emit('show', summaries.value);
            showSummaries.value = true;
            await vue.nextTick();
            if (typeof window !== 'undefined') {
                (_a = document.getElementById(id)) === null || _a === void 0 ? void 0 : _a.scrollIntoView({ behavior: 'smooth' });
                if (summaries.value[0]) {
                    (_b = document.getElementById(summaries.value[0].key)) === null || _b === void 0 ? void 0 : _b.focus();
                }
            }
        });
        (_b = node.value) === null || _b === void 0 ? void 0 : _b.on('child', addContexts);
        function jumpLink(e) {
            var _a, _b, _c;
            if (e.target instanceof HTMLAnchorElement) {
                e.preventDefault();
                const id = (_a = e.target.getAttribute('href')) === null || _a === void 0 ? void 0 : _a.substring(1);
                if (id) {
                    (_b = document.getElementById(id)) === null || _b === void 0 ? void 0 : _b.scrollIntoView({ behavior: 'smooth' });
                    (_c = document.getElementById(id)) === null || _c === void 0 ? void 0 : _c.focus();
                }
            }
        }
        inputs.localize('summaryHeader', 'There were errors in your form.')(node.value);
        const schema = definition(props.sectionsSchema || {});
        const data = vue.computed(() => {
            var _a, _b, _c, _d, _e, _f, _g, _h;
            return {
                id,
                fns: ((_b = (_a = node.value) === null || _a === void 0 ? void 0 : _a.context) === null || _b === void 0 ? void 0 : _b.fns) || {},
                classes: ((_d = (_c = node.value) === null || _c === void 0 ? void 0 : _c.context) === null || _d === void 0 ? void 0 : _d.classes) || {},
                summaries: summaries.value,
                showSummaries: showSummaries.value,
                summaryHeader: ((_h = (_g = (_f = (_e = node.value) === null || _e === void 0 ? void 0 : _e.context) === null || _f === void 0 ? void 0 : _f.ui) === null || _g === void 0 ? void 0 : _g.summaryHeader) === null || _h === void 0 ? void 0 : _h.value) || '',
                jumpLink,
            };
        });
        return () => {
            var _a;
            return ((_a = node.value) === null || _a === void 0 ? void 0 : _a.context)
                ? vue.h(FormKitSchema, { schema, data: data.value }, { ...context.slots })
                : null;
        };
    },
});

/**
 * A plugin that creates Vue-specific context object on each given node.
 *
 * @param node - FormKitNode to create the context on.
 *
 * @public
 */
const vueBindings = function vueBindings(node) {
    /**
     * Start a validity counter on all blocking messages.
     */
    node.ledger.count('blocking', (m) => m.blocking);
    const isValid = vue.ref(!node.ledger.value('blocking'));
    /**
     * Start an error message counter.
     */
    node.ledger.count('errors', (m) => m.type === 'error');
    const hasErrors = vue.ref(!!node.ledger.value('errors'));
    /**
     * Keep track of the first time a Vue tick cycle has passed.
     */
    let hasTicked = false;
    vue.nextTick(() => {
        hasTicked = true;
    });
    /**
     * All messages with the visibility state set to true.
     */
    const availableMessages = vue.reactive(node.store.reduce((store, message) => {
        if (message.visible) {
            store[message.key] = message;
        }
        return store;
    }, {}));
    /**
     * A flag that determines when validation messages should be displayed.
     */
    const validationVisibility = vue.ref(node.props.validationVisibility ||
        (node.props.type === 'checkbox' ? 'dirty' : 'blur'));
    node.on('prop:validationVisibility', ({ payload }) => {
        validationVisibility.value = payload;
    });
    /**
     * Keep track of if this input has ever shown validation errors.
     */
    const hasShownErrors = vue.ref(validationVisibility.value === 'live');
    /**
     * An array of unique identifiers that should only be used for iterating
     * inside a synced list.
     */
    const items = vue.ref(node.children.map((child) => child.uid));
    /**
     * The current visibility state of validation messages.
     */
    const validationVisible = vue.computed(() => {
        if (!context.state)
            return false;
        if (context.state.submitted)
            return true;
        if (!hasShownErrors.value && !context.state.settled) {
            return false;
        }
        switch (validationVisibility.value) {
            case 'live':
                return true;
            case 'blur':
                return context.state.blurred;
            case 'dirty':
                return context.state.dirty;
            default:
                return false;
        }
    });
    /**
     * Determines if the input should be considered "complete".
     */
    const isComplete = vue.computed(() => {
        return context && hasValidation.value
            ? isValid.value && !hasErrors.value
            : context.state.dirty && !utils.empty(context.value);
    });
    /**
     * If the input has validation rules or not.
     */
    const hasValidation = vue.ref(Array.isArray(node.props.parsedRules) && node.props.parsedRules.length > 0);
    node.on('prop:parsedRules', ({ payload: rules }) => {
        hasValidation.value = Array.isArray(rules) && rules.length > 0;
    });
    /**
     * All messages that are currently on display to an end user. This changes
     * based on the current message type visibility, like errorVisibility.
     */
    const messages = vue.computed(() => {
        const visibleMessages = {};
        for (const key in availableMessages) {
            const message = availableMessages[key];
            if (message.type !== 'validation' || validationVisible.value) {
                visibleMessages[key] = message;
            }
        }
        return visibleMessages;
    });
    /**
     * UI Messages.
     */
    const ui = vue.reactive(node.store.reduce((messages, message) => {
        if (message.type === 'ui' && message.visible)
            messages[message.key] = message;
        return messages;
    }, {}));
    /**
     * This is the reactive data object that is provided to all schemas and
     * forms. It is a subset of data in the core node object.
     */
    const cachedClasses = vue.reactive({});
    const classes = new Proxy(cachedClasses, {
        get(...args) {
            const [target, property] = args;
            let className = Reflect.get(...args);
            if (!className && typeof property === 'string') {
                if (!utils.has(target, property) && !property.startsWith('__v')) {
                    const observedNode = observer.createObserver(node);
                    observedNode.watch((node) => {
                        const rootClasses = typeof node.config.rootClasses === 'function'
                            ? node.config.rootClasses(property, node)
                            : {};
                        const globalConfigClasses = node.config.classes
                            ? core.createClasses(property, node, node.config.classes[property])
                            : {};
                        const classesPropClasses = core.createClasses(property, node, node.props[`_${property}Class`]);
                        const sectionPropClasses = core.createClasses(property, node, node.props[`${property}Class`]);
                        className = core.generateClassList(node, property, rootClasses, globalConfigClasses, classesPropClasses, sectionPropClasses);
                        target[property] = className !== null && className !== void 0 ? className : '';
                    });
                }
            }
            return className;
        },
    });
    node.on('prop:rootClasses', () => {
        const keys = Object.keys(cachedClasses);
        for (const key of keys) {
            delete cachedClasses[key];
        }
    });
    const describedBy = vue.computed(() => {
        const describers = [];
        if (context.help) {
            describers.push(`help-${node.props.id}`);
        }
        for (const key in messages.value) {
            describers.push(`${node.props.id}-${key}`);
        }
        return describers.length ? describers.join(' ') : undefined;
    });
    const value = vue.ref(node.value);
    const _value = vue.ref(node.value);
    const context = vue.reactive({
        _value,
        attrs: node.props.attrs,
        disabled: node.props.disabled,
        describedBy,
        fns: {
            length: (obj) => Object.keys(obj).length,
            number: (value) => Number(value),
            string: (value) => String(value),
            json: (value) => JSON.stringify(value),
            eq: utils.eq,
        },
        handlers: {
            blur: (e) => {
                if (!node)
                    return;
                node.store.set(core.createMessage({ key: 'blurred', visible: false, value: true }));
                if (typeof node.props.attrs.onBlur === 'function') {
                    node.props.attrs.onBlur(e);
                }
            },
            touch: () => {
                var _a;
                const doCompare = context.dirtyBehavior === 'compare';
                if (((_a = node.store.dirty) === null || _a === void 0 ? void 0 : _a.value) && !doCompare)
                    return;
                const isDirty = !utils.eq(node.props._init, node._value);
                if (!isDirty && !doCompare)
                    return;
                node.store.set(core.createMessage({ key: 'dirty', visible: false, value: isDirty }));
            },
            DOMInput: (e) => {
                node.input(e.target.value);
                node.emit('dom-input-event', e);
            },
        },
        help: node.props.help,
        id: node.props.id,
        items,
        label: node.props.label,
        messages,
        node: vue.markRaw(node),
        options: node.props.options,
        defaultMessagePlacement: true,
        slots: node.props.__slots,
        state: {
            blurred: false,
            complete: isComplete,
            dirty: false,
            empty: utils.empty(value),
            submitted: false,
            settled: node.isSettled,
            valid: isValid,
            errors: hasErrors,
            rules: hasValidation,
            validationVisible,
        },
        type: node.props.type,
        family: node.props.family,
        ui,
        value,
        classes,
    });
    /**
     * Ensure the context object is properly configured after booting up.
     */
    node.on('created', () => {
        if (!utils.eq(context.value, node.value)) {
            _value.value = node.value;
            value.value = node.value;
            vue.triggerRef(value);
            vue.triggerRef(_value);
        }
        (async () => {
            await node.settled;
            if (node)
                node.props._init = utils.cloneAny(node.value);
        })();
    });
    /**
     * Sets the settled state.
     */
    node.on('settled', ({ payload: isSettled }) => {
        context.state.settled = isSettled;
    });
    /**
     * Observes node.props properties explicitly and updates them in the context
     * object.
     * @param observe - Props to observe and register as context data.
     */
    function observeProps(observe) {
        observe.forEach((prop) => {
            prop = utils.camel(prop);
            if (!utils.has(context, prop)) {
                context[prop] = node.props[prop];
            }
            node.on(`prop:${prop}`, ({ payload }) => {
                context[prop] = payload;
            });
        });
    }
    /**
     * We use a node observer to individually observe node props.
     */
    const rootProps = () => {
        const props = [
            '__root',
            'help',
            'label',
            'disabled',
            'options',
            'type',
            'attrs',
            'preserve',
            'preserveErrors',
            'id',
            'dirtyBehavior',
        ];
        const iconPattern = /^[a-zA-Z-]+(?:-icon|Icon)$/;
        const matchingProps = Object.keys(node.props).filter((prop) => {
            return iconPattern.test(prop);
        });
        return props.concat(matchingProps);
    };
    observeProps(rootProps());
    /**
     * Once the input is defined, deal with it.
     * @param definition - Type definition.
     */
    function definedAs(definition) {
        if (definition.props)
            observeProps(definition.props);
    }
    node.props.definition && definedAs(node.props.definition);
    /**
     * When new props are added to the core node as "props" (ie not attrs) then
     * we automatically need to start tracking them here.
     */
    node.on('added-props', ({ payload }) => observeProps(payload));
    /**
     * Watch for input events from core.
     */
    node.on('input', ({ payload }) => {
        if (node.type !== 'input' && !vue.isRef(payload) && !vue.isReactive(payload)) {
            _value.value = utils.shallowClone(payload);
        }
        else {
            _value.value = payload;
            vue.triggerRef(_value);
        }
    });
    /**
     * Model updates from core. This is the raw value and should emitted as a
     * model update even if the value did not update internally. Why? Because
     * the model that created this event may have not be the same value as our
     * internal value.
     *
     * See test: "emits a modelUpdated event even when the value results in the
     * same value"
     */
    node.on('commitRaw', ({ payload }) => {
        if (node.type !== 'input' && !vue.isRef(payload) && !vue.isReactive(payload)) {
            value.value = _value.value = utils.shallowClone(payload);
        }
        else {
            value.value = _value.value = payload;
            vue.triggerRef(value);
        }
        node.emit('modelUpdated');
    });
    /**
     * Watch for input commits from core.
     */
    node.on('commit', ({ payload }) => {
        // The input is dirty after a value has been input by a user
        if ((!context.state.dirty || context.dirtyBehavior === 'compare') &&
            node.isCreated &&
            hasTicked) {
            context.handlers.touch();
        }
        if (isComplete &&
            node.type === 'input' &&
            hasErrors.value &&
            !utils.undefine(node.props.preserveErrors)) {
            node.store.filter((message) => { var _a; return !(message.type === 'error' && ((_a = message.meta) === null || _a === void 0 ? void 0 : _a.autoClear) === true); });
        }
        if (node.type === 'list' && node.sync) {
            items.value = node.children.map((child) => child.uid);
        }
        context.state.empty = utils.empty(payload);
    });
    /**
     * Update the local state in response to messages.
     * @param message - A formkit message
     */
    const updateState = async (message) => {
        if (message.type === 'ui' &&
            message.visible &&
            !message.meta.showAsMessage) {
            ui[message.key] = message;
        }
        else if (message.visible) {
            availableMessages[message.key] = message;
        }
        else if (message.type === 'state') {
            // await node.settled
            context.state[message.key] = !!message.value;
        }
    };
    /**
     * Listen to message events and modify the local message data values.
     */
    node.on('message-added', (e) => updateState(e.payload));
    node.on('message-updated', (e) => updateState(e.payload));
    node.on('message-removed', ({ payload: message }) => {
        delete ui[message.key];
        delete availableMessages[message.key];
        delete context.state[message.key];
    });
    node.on('settled:blocking', () => {
        isValid.value = true;
    });
    node.on('unsettled:blocking', () => {
        isValid.value = false;
    });
    node.on('settled:errors', () => {
        hasErrors.value = false;
    });
    node.on('unsettled:errors', () => {
        hasErrors.value = true;
    });
    /**
     * Watch the validation visible prop and set the hasShownErrors state.
     */
    vue.watch(validationVisible, (value) => {
        if (value) {
            hasShownErrors.value = true;
        }
    });
    node.context = context;
    // The context is complete
    node.emit('context', node, false);
    node.on('destroyed', () => {
        node.context = undefined;
        /* @ts-ignore */ // eslint-disable-line
        node = null;
    });
};

/**
 * Default configuration options. Includes all validation rules,
 * en i18n messages.
 *
 * @public
 */
const defaultConfig = (options = {}) => {
    dev.register();
    const { rules = {}, locales = {}, inputs: inputs$1 = {}, messages = {}, locale = undefined, theme = undefined, iconLoaderUrl = undefined, iconLoader = undefined, icons = {}, ...nodeOptions } = options;
    /**
     * The default configuration includes the validation plugin,
     * with all core-available validation rules.
     */
    const validation$1 = validation.createValidationPlugin({
        ...defaultRules__namespace,
        ...(rules || {}),
    });
    /**
     * Includes the i18n plugin with only the english language
     * messages.
     */
    const i18n$1 = i18n.createI18nPlugin(utils.extend({ en: i18n.en, ...(locales || {}) }, messages));
    /**
     * Create the library of inputs that are generally available. This default
     * config imports all "native" inputs by default, but
     */
    const library = inputs.createLibraryPlugin(inputs.inputs, inputs$1);
    /**
     * Create the theme plugin for the user provided theme
     */
    const themePlugin = themes.createThemePlugin(theme, icons, iconLoaderUrl, iconLoader);
    return utils.extend({
        plugins: [library, themePlugin, vueBindings, i18n$1, validation$1],
        ...(!locale ? {} : { config: { locale } }),
    }, nodeOptions || {}, true);
};

var defaultConfig$1 = /*#__PURE__*/Object.freeze({
    __proto__: null,
    defaultConfig: defaultConfig
});

/**
 * Renders an icon using the current IconLoader set at the root FormKit config
 *
 * @public
 */
const FormKitIcon = /* #__PURE__ */ vue.defineComponent({
    name: 'FormKitIcon',
    props: {
        icon: {
            type: String,
            default: '',
        },
        iconLoader: {
            type: Function,
            default: null,
        },
        iconLoaderUrl: {
            type: Function,
            default: null,
        },
    },
    setup(props) {
        var _a, _b;
        const icon = vue.ref(undefined);
        const config = vue.inject(optionsSymbol, {});
        const parent = vue.inject(parentSymbol, null);
        let iconHandler = undefined;
        function loadIcon() {
            if (!iconHandler || typeof iconHandler !== 'function')
                return;
            const iconOrPromise = iconHandler(props.icon);
            if (iconOrPromise instanceof Promise) {
                iconOrPromise.then((iconValue) => {
                    icon.value = iconValue;
                });
            }
            else {
                icon.value = iconOrPromise;
            }
        }
        if (props.iconLoader && typeof props.iconLoader === 'function') {
            // if we have a locally supplied loader, then use it
            iconHandler = themes.createIconHandler(props.iconLoader);
        }
        else if (parent && ((_a = parent.props) === null || _a === void 0 ? void 0 : _a.iconLoader)) {
            // otherwise try to inherit from a parent
            iconHandler = themes.createIconHandler(parent.props.iconLoader);
        }
        else if (props.iconLoaderUrl &&
            typeof props.iconLoaderUrl === 'function') {
            iconHandler = themes.createIconHandler(iconHandler, props.iconLoaderUrl);
        }
        else {
            // grab our iconHandler from the global config
            const iconPlugin = (_b = config === null || config === void 0 ? void 0 : config.plugins) === null || _b === void 0 ? void 0 : _b.find((plugin) => {
                return (typeof plugin
                    .iconHandler === 'function');
            });
            if (iconPlugin) {
                iconHandler = iconPlugin.iconHandler;
            }
        }
        vue.watch(() => props.icon, () => {
            loadIcon();
        }, { immediate: true });
        return () => {
            if (props.icon && icon.value) {
                return vue.h('span', {
                    class: 'formkit-icon',
                    innerHTML: icon.value,
                });
            }
            return null;
        };
    },
});

Object.defineProperty(exports, 'clearErrors', {
    enumerable: true,
    get: function () { return core.clearErrors; }
});
Object.defineProperty(exports, 'errorHandler', {
    enumerable: true,
    get: function () { return core.errorHandler; }
});
Object.defineProperty(exports, 'reset', {
    enumerable: true,
    get: function () { return core.reset; }
});
Object.defineProperty(exports, 'resetCount', {
    enumerable: true,
    get: function () { return core.resetCount; }
});
Object.defineProperty(exports, 'setErrors', {
    enumerable: true,
    get: function () { return core.setErrors; }
});
Object.defineProperty(exports, 'submitForm', {
    enumerable: true,
    get: function () { return core.submitForm; }
});
exports.FormKit = formkitComponent;
exports.FormKitIcon = FormKitIcon;
exports.FormKitKitchenSink = FormKitKitchenSink;
exports.FormKitLazyProvider = FormKitLazyProvider;
exports.FormKitMessages = FormKitMessages;
exports.FormKitProvider = FormKitProvider;
exports.FormKitRoot = FormKitRoot;
exports.FormKitSchema = FormKitSchema;
exports.FormKitSummary = FormKitSummary;
exports.bindings = vueBindings;
exports.configSymbol = configSymbol;
exports.createInput = createInput;
exports.defaultConfig = defaultConfig;
exports.defineFormKitConfig = defineFormKitConfig;
exports.getCurrentSchemaNode = getCurrentSchemaNode;
exports.onSSRComplete = onSSRComplete;
exports.optionsSymbol = optionsSymbol;
exports.parentSymbol = parentSymbol;
exports.plugin = plugin;
exports.rootSymbol = rootSymbol;
exports.ssrComplete = ssrComplete;
exports.useConfig = useConfig;
exports.useInput = useInput;
