"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
var Apply_1 = require("./Apply");
var Option_1 = require("./Option");
var option = require("./Option");
var Ord_1 = require("./Ord");
var function_1 = require("./function");
exports.URI = 'Array';
/** @function */
exports.empty = function () {
    return [];
};
/** @function */
exports.concat = function (x) { return function (y) {
    return x.concat(y);
}; };
/** @function */
exports.map = function (f, fa) {
    return fa.map(f);
};
/** @function */
exports.of = function (a) {
    return [a];
};
/** @function */
exports.ap = function (fab, fa) {
    return fab.reduce(function (acc, f) { return acc.concat(fa.map(f)); }, []);
};
/** @function */
exports.chain = function (f, fa) {
    return fa.reduce(function (acc, a) { return acc.concat(f(a)); }, []);
};
/** @function */
exports.reduce = function (f, b, fa) {
    return fa.reduce(f, b);
};
/** @function */
function traverse(F) {
    var liftedSnoc = Apply_1.liftA2(F)(exports.snoc);
    return function (f, ta) { return exports.reduce(function (fab, a) { return liftedSnoc(fab)(f(a)); }, F.of(exports.empty()), ta); };
}
exports.traverse = traverse;
/**
 * @function
 * @alias empty
 */
exports.zero = exports.empty;
/** @function */
exports.alt = function (x, y) {
    return x.concat(y);
};
/** @function */
exports.unfoldr = function (f, b) {
    var ret = [];
    var bb = b;
    while (true) {
        var mt = f(bb);
        if (option.isSome(mt)) {
            var _a = mt.value, a = _a[0], b_1 = _a[1];
            ret.push(a);
            bb = b_1;
        }
        else {
            break;
        }
    }
    return ret;
};
/** @function */
exports.extend = function (f, fa) {
    return fa.map(function (_, i, as) { return f(as.slice(i)); });
};
/** @function */
exports.partitionMap = function (f, fa) {
    var left = [];
    var right = [];
    for (var i = 0; i < fa.length; i++) {
        f(fa[i]).fold(function (l) { return left.push(l); }, function (r) { return right.push(r); });
    }
    return { left: left, right: right };
};
/**
 * Example
 *
 * ```ts
 * flatten([[1], [2], [3]]) // [1, 2, 3]
 * ```
 *
 * @function
 */
exports.flatten = function (ffa) {
    return exports.chain(function (as) { return as; }, ffa);
};
/**
 * Break an array into its first element and remaining elements
 *
 * Example
 *
 * ```ts
 * const length = <A>(xs: Array<A>): number => fold(() => 0, (head, tail) => 1 + length(tail), xs)
 * ```
 *
 * @function
 */
exports.fold = function (nil, cons, as) {
    return as.length === 0 ? nil() : cons(as[0], as.slice(1));
};
/**
 * Get the number of elements in an array
 * @function
 */
exports.length = function (as) {
    return as.length;
};
/**
 * Test whether an array is empty
 * @function
 */
exports.isEmpty = function (as) {
    return exports.length(as) === 0;
};
/**
 * Test whether an array contains a particular index
 * @function
 */
exports.isOutOfBound = function (i) { return function (as) {
    return i < 0 || i >= as.length;
}; };
/**
 * This function provides a safe way to read a value at a particular index from an array
 * @function
 */
exports.index = function (i) { return function (as) {
    return exports.isOutOfBound(i)(as) ? option.none : option.some(as[i]);
}; };
/**
 * Attaches an element to the front of an array, creating a new array
 */
exports.cons = function (a) { return function (as) {
    return [a].concat(as);
}; };
/**
 * Append an element to the end of an array, creating a new array
 * @function
 */
exports.snoc = function (as) { return function (a) {
    return as.concat([a]);
}; };
/**
 * Get the first element in an array, or `None` if the array is empty
 * @function
 */
exports.head = function (as) {
    return exports.isEmpty(as) ? option.none : option.some(as[0]);
};
/**
 * Get the last element in an array, or `None` if the array is empty
 * @function
 */
exports.last = function (as) {
    return exports.index(exports.length(as) - 1)(as);
};
/**
 * Get all but the first element of an array, creating a new array, or `None` if the array is empty
 * @function
 */
exports.tail = function (as) {
    return as.length === 0 ? option.none : option.some(as.slice(1));
};
/**
 * Extract a subarray by a start and end index
 * @function
 */
exports.slice = function (start, end) { return function (as) {
    return as.slice(start, end);
}; };
/**
 * Get all but the last element of an array, creating a new array, or `None` if the array is empty
 * @function
 */
exports.init = function (as) {
    var len = as.length;
    return len === 0 ? option.none : option.some(as.slice(0, len - 1));
};
/**
 * Keep only a number of elements from the start of an array, creating a new array
 * @function
 */
exports.take = function (n) { return function (as) {
    return exports.slice(0, n)(as);
}; };
/**
 * Split an array into two parts:
 * 1. the longest initial subarray for which all elements satisfy the specified predicate
 * 2. the remaining elements
 * @function
 */
exports.span = function (predicate) { return function (as) {
    var init = [];
    var i = 0;
    for (; i < as.length; i++) {
        if (predicate(as[i])) {
            init.push(as[i]);
        }
        else {
            break;
        }
    }
    return { init: init, rest: as.slice(i) };
}; };
/**
 * Calculate the longest initial subarray for which all element satisfy the
 * specified predicate, creating a new array
 * @function
 */
exports.takeWhile = function (predicate) { return function (as) {
    return exports.span(predicate)(as).init;
}; };
/**
 * Drop a number of elements from the start of an array, creating a new array
 * @function
 */
exports.drop = function (n) { return function (as) {
    return exports.slice(n, exports.length(as))(as);
}; };
/**
 * Remove the longest initial subarray for which all element satisfy the
 * specified predicate, creating a new array
 * @function
 */
exports.dropWhile = function (predicate) { return function (as) {
    return exports.span(predicate)(as).rest;
}; };
/**
 * Find the first index for which a predicate holds
 * @function
 */
exports.findIndex = function (predicate) { return function (as) {
    var len = as.length;
    for (var i = 0; i < len; i++) {
        if (predicate(as[i])) {
            return option.some(i);
        }
    }
    return option.none;
}; };
/**
 * Find the first element which satisfies a predicate function
 * @function
 */
exports.findFirst = function (predicate) { return function (as) {
    return Option_1.fromNullable(as.find(predicate));
}; };
/**
 * Find the last element which satisfies a predicate function
 * @function
 */
exports.findLast = function (predicate) { return function (as) {
    var len = as.length;
    var a = null;
    for (var i = len - 1; i >= 0; i--) {
        if (predicate(as[i])) {
            a = as[i];
            break;
        }
    }
    return Option_1.fromNullable(a);
}; };
/**
 * Filter an array, keeping the elements which satisfy a predicate function, creating a new array
 * @function
 */
exports.filter = function (predicate) { return function (as) {
    return as.filter(predicate);
}; };
/** @function */
exports.refine = function (as) { return function (refinement) {
    return as.filter(refinement);
}; };
/** @function */
exports.copy = function (as) {
    return as.slice();
};
/** @function */
exports.unsafeInsertAt = function (i) { return function (a) { return function (as) {
    var xs = exports.copy(as);
    xs.splice(i, 0, a);
    return xs;
}; }; };
/**
 * Insert an element at the specified index, creating a new array, or
 * returning `None` if the index is out of bounds
 * @function
 */
exports.insertAt = function (i) { return function (a) { return function (as) {
    return i < 0 || i > as.length ? option.none : option.some(exports.unsafeInsertAt(i)(a)(as));
}; }; };
/** @function */
exports.unsafeUpdateAt = function (i) { return function (a) { return function (as) {
    var xs = exports.copy(as);
    xs[i] = a;
    return xs;
}; }; };
/**
 * Change the element at the specified index, creating a new array, or
 * returning `None` if the index is out of bounds
 * @function
 */
exports.updateAt = function (i) { return function (a) { return function (as) {
    return exports.isOutOfBound(i)(as) ? option.none : option.some(exports.unsafeUpdateAt(i)(a)(as));
}; }; };
/** @function */
exports.unsafeDeleteAt = function (i) { return function (as) {
    var xs = exports.copy(as);
    xs.splice(i, 1);
    return xs;
}; };
/**
 * Delete the element at the specified index, creating a new array, or
 * returning `None` if the index is out of bounds
 * @function
 */
exports.deleteAt = function (i) { return function (as) {
    return exports.isOutOfBound(i)(as) ? option.none : option.some(exports.unsafeDeleteAt(i)(as));
}; };
/**
 * Apply a function to the element at the specified index, creating a new
 * array, or returning `None` if the index is out of bounds
 * @function
 */
exports.modifyAt = function (i) { return function (f) { return function (as) {
    return exports.isOutOfBound(i)(as) ? option.none : exports.updateAt(i)(f(as[i]))(as);
}; }; };
/**
 * Reverse an array, creating a new array
 * @function
 */
exports.reverse = function (as) {
    return exports.copy(as).reverse();
};
/**
 * Apply a function to each element in an array, keeping only the results
 * which contain a value, creating a new array
 * @function
 */
exports.mapOption = function (f) { return function (as) {
    return exports.chain(function (a) { return f(a).fold(exports.empty, exports.of); }, as);
}; };
/**
 * Filter an array of optional values, keeping only the elements which contain
 * a value, creating a new array
 * @function
 */
exports.catOptions = function (as) {
    return exports.mapOption(function_1.identity)(as);
};
/**
 * Extracts from a list of `Either` all the `Right` elements. All the `Right` elements are extracted in order
 * @function
 */
exports.rights = function (as) {
    return exports.chain(function (a) { return a.fold(exports.empty, exports.of); }, as);
};
/**
 * Extracts from a list of `Either` all the `Left` elements. All the `Left` elements are extracted in order
 * @function
 */
exports.lefts = function (as) {
    return exports.chain(function (a) { return a.fold(exports.of, exports.empty); }, as);
};
/**
 * Sort the elements of an array in increasing order, creating a new array
 * @function
 */
exports.sort = function (ord) {
    var comparator = Ord_1.toNativeComparator(ord.compare);
    return function (as) { return exports.copy(as).sort(comparator); };
};
/**
 * Apply a function to pairs of elements at the same index in two arrays,
 * collecting the results in a new array.
 * If one input array is short, excess elements of the longer array are discarded.
 * @function
 */
exports.zipWith = function (f) { return function (fa) { return function (fb) {
    var fc = [];
    var len = Math.min(fa.length, fb.length);
    for (var i = 0; i < len; i++) {
        fc[i] = f(fa[i], fb[i]);
    }
    return fc;
}; }; };
/**
 * Takes two arrays and returns an array of corresponding pairs.
 * If one input array is short, excess elements of the longer array are discarded
 * @function
 */
exports.zip = function (fa) { return function (fb) {
    return exports.zipWith(function_1.tuple)(fa)(fb);
}; };
exports.array = {
    URI: exports.URI,
    empty: exports.empty,
    concat: exports.concat,
    map: exports.map,
    of: exports.of,
    ap: exports.ap,
    chain: exports.chain,
    reduce: exports.reduce,
    unfoldr: exports.unfoldr,
    traverse: traverse,
    zero: exports.zero,
    alt: exports.alt,
    extend: exports.extend
};
//# sourceMappingURL=Array.js.map