var assert = require('assert');

var Fraction = require('../fraction.min');

var tests = [{
        set: "foo",
        expect: "Invalid Param"
    }, {
        set: " 123",
        expect: "Invalid Param"
    }, {
        set: 0,
        expect: 0
    }, {
        set: .2,
        expect: "0.2"
    }, {
        set: .333,
        expect: "0.333"
    }, {
        set: 1.1,
        expect: "1.1"
    }, {
        set: 1.2,
        expect: "1.2"
    }, {
        set: 1.3,
        expect: "1.3"
    }, {
        set: 1.4,
        expect: "1.4"
    }, {
        set: 1.5,
        expect: "1.5"
    }, {
        set: 2.555,
        expect: "2.555"
    }, {
        set: " - ",
        expect: "Invalid Param"
    }, {
        set: ".5",
        expect: "0.5"
    }, {
        set: "-.5",
        expect: "-0.5"
    }, {
        set: "123",
        expect: "123"
    }, {
        set: "-123",
        expect: "-123"
    }, {
        set: "123.4",
        expect: "123.4"
    }, {
        set: "-123.4",
        expect: "-123.4"
    }, {
        set: "123.",
        expect: "123"
    }, {
        set: "-123.",
        expect: "-123"
    }, {
        set: "123.4(56)",
        expect: "123.4(56)"
    }, {
        set: "-123.4(56)",
        expect: "-123.4(56)"
    }, {
        set: "123.(4)",
        expect: "123.(4)"
    }, {
        set: "-123.(4)",
        expect: "-123.(4)"
    }, {
        set: "123/456",
        expect: "0.269(736842105263157894)"
    }, {
        set: "-123/456",
        expect: "-0.269(736842105263157894)"
    }, {
        set: "19 123/456",
        expect: "19.269(736842105263157894)"
    }, {
        set: "-19 123/456",
        expect: "-19.269(736842105263157894)"
    }, {
        set: "123.(22)123",
        expect: "Invalid Param"
    }, {
        set: "+33.3(3)",
        expect: "33.(3)"
    }, {
        set: "3.'09009'",
        expect: "3.(09009)"
    }, {
        set: "123.(((",
        expect: "Invalid Param"
    }, {
        set: "123.((",
        expect: "Invalid Param"
    }, {
        set: "123.()",
        expect: "Invalid Param"
    }, {
        set: null,
        expect: "0" // I would say it's just fine
    }, {
        set: [22, 7],
        expect: '3.(142857)' // We got Pi! - almost ;o
    }, {
        set: "355/113",
        expect: "3.(1415929203539823008849557522123893805309734513274336283185840707964601769911504424778761061946902654867256637168)" // Yay, a better PI
    }, {
        set: "3 1/7",
        expect: '3.(142857)'
    }, {
        set: [36, -36],
        expect: "-1"
    }, {
        set: "9/12",
        expect: "0.75"
    }, {
        set: "0.09(33)",
        expect: "0.09(3)"
    }, {
        set: 1 / 2,
        expect: "0.5"
    }, {
        set: 1 / 3,
        expect: "0.(3)"
    }, {
        set: "0.'3'",
        expect: "0.(3)"
    }, {
        set: "0.00002",
        expect: "0.00002"
    }, {
        set: 7 / 8,
        expect: "0.875"
    }, {
        set: 0.003,
        expect: "0.003"
    }, {
        set: 4,
        expect: "4"
    }, {
        set: -99,
        expect: "-99"
    }, {
        set: "-92332.1192",
        expect: "-92332.1192"
    }, {
        set: '88.92933(12111)',
        expect: "88.92933(12111)"
    }, {
        set: '-192322.823(123)',
        expect: "-192322.8(231)"
    }, {
        label: "-99.12 % 0.09(34)",
        set: '-99.12',
        fn: "mod",
        param: "0.09(34)",
        expect: "-0.07(95)"
    }, {
        label: "0.4 / 0.1",
        set: .4,
        fn: "div",
        param: ".1",
        expect: "4"
    }, {
        label: "1 / -.1",
        set: 1,
        fn: "div",
        param: "-.1",
        expect: "-10"
    }, {
        label: "1 - (-1)",
        set: 1,
        fn: "sub",
        param: "-1",
        expect: "2"
    }, {
        label: "1 + (-1)",
        set: 1,
        fn: "add",
        param: "-1",
        expect: "0"
    }, {
        label: "-187 % 12",
        set: '-187',
        fn: "mod",
        param: "12",
        expect: "-7"
    }, {
        label: "Negate by 99 * -1",
        set: '99',
        fn: "mul",
        param: "-1",
        expect: "-99"
    }, {
        label: "99.(9) + 66",
        set: '99.(999999)',
        fn: "add",
        param: "66",
        expect: "166"
    }, {
        label: "-82.124 / 66.(3)",
        set: '-82.124',
        fn: "div",
        param: "66.(3)",
        expect: "-1.238(050251256281407035175879396984924623115577889447236180904522613065326633165829145728643216080402010)"
    }, {
        label: "100 - .91",
        set: '100',
        fn: "sub",
        param: ".91",
        expect: "99.09"
    }, {
        label: "381.(33411) % 11.119(356)",
        set: '381.(33411)',
        fn: "mod",
        param: "11.119(356)",
        expect: "3.275(997225017295217)"
    }, {
        label: "13/26 mod 1",
        set: '13/26',
        fn: "mod",
        param: "1.000",
        expect: "0.5"
    }, {
        label: "381.(33411) % 1", // Extract fraction part of a number
        set: '381.(33411)',
        fn: "mod",
        param: "1",
        expect: "0.(33411)"
    }, {
        label: "-222/3",
        set: {
            n: 3,
            d: 222,
            s: -1
        },
        fn: "inverse",
        param: null,
        expect: "-74"
    }, {
        label: "inverse",
        set: 1 / 2,
        fn: "inverse",
        param: null,
        expect: "2"
    }, {
        label: "abs(-222/3)",
        set: {
            n: -222,
            d: 3
        },
        fn: "abs",
        param: null,
        expect: "74"
    }, {
        label: "9 % -2",
        set: 9,
        fn: "mod",
        param: "-2",
        expect: "1"
    }, {
        label: "-9 % 2",
        set: '-9',
        fn: "mod",
        param: "-2",
        expect: "-1"
    }, {
        label: "10 / 0",
        set: 10,
        fn: "div",
        param: 0,
        expect: "DIV/0"
    }, {
        label: "-3 / 4",
        set: [-3, 4],
        fn: "inverse",
        param: null,
        expect: "-1.(3)"
    }, {
        label: "-19.6",
        set: [-98, 5],
        fn: "equals",
        param: '-19.6',
        expect: "true" // actually, we get a real bool but we call toString() in the test below
    }, {
        label: "-19.6",
        set: [98, -5],
        fn: "equals",
        param: '-19.6',
        expect: "true"
    }, {
        label: "99/88",
        set: [99, 88],
        fn: "equals",
        param: [88, 99],
        expect: "false"
    }, {
        label: "99/88",
        set: [99, -88],
        fn: "equals",
        param: [9, 8],
        expect: "false"
    }, {
        label: "12.5",
        set: 12.5,
        fn: "add",
        param: 0,
        expect: "12.5"
    }, {
        label: "0/1 -> 1/0",
        set: 0,
        fn: "inverse",
        param: null,
        expect: "DIV/0"
    }, {
        label: "abs(-100.25)",
        set: -100.25,
        fn: "abs",
        param: null,
        expect: "100.25"
    }, {
        label: "0.022222222",
        set: '0.0(22222222)',
        fn: "abs",
        param: null,
        expect: "0.0(2)"
    }, {
        label: "1.5 | 100.5",
        set: 100.5,
        fn: "divisible",
        param: '1.5',
        expect: "true"
    }, {
        label: "1.5 | 100.6",
        set: 100.6,
        fn: "divisible",
        param: 1.6,
        expect: "false"
    }, {
        label: "(1/6) | (2/3)", // == 4
        set: [2, 3],
        fn: "divisible",
        param: [1, 6],
        expect: "true"
    }, {
        label: "(1/6) | (2/5)",
        set: [2, 5],
        fn: "divisible",
        param: [1, 6],
        expect: "false"
    }, {
        label: "0 | (2/5)",
        set: [2, 5],
        fn: "divisible",
        param: 0,
        expect: "false"
    }, {
        label: "6 | 0",
        set: 0,
        fn: "divisible",
        param: 6,
        expect: "true"
    }, {
        label: "fmod(4.55, 0.05)", // http://phpjs.org/functions/fmod/ (comment section)
        set: 4.55,
        fn: "mod",
        param: 0.05,
        expect: "0"
    }, {
        label: "fmod(99.12, 0.4)",
        set: 99.12,
        fn: "mod",
        param: "0.4",
        expect: "0.32"
    }, {
        label: "fmod(fmod(1.0,0.1))", // http://stackoverflow.com/questions/4218961/why-fmod1-0-0-1-1
        set: 1.0,
        fn: "mod",
        param: 0.1,
        expect: "0"
    }, {
        label: "bignum",
        set: [5385020324, 1673196525],
        fn: "add",
        param: 0,
        expect: "3.(218402765927331817761216065160068390651241640607638723132060054929889362518249313241909822876305579226564554334106090735515960983722459021960973771446244188201382978607369507894477607763379738073505740755707103802405996510182807127214180653405313521076073236525518124656635896372065439234641011461579505730804694326029633608042545988433725679653799185364671971213901487154953301137175144443955858681932177692037700114157241630656625945359287666462252543824760812242303694719901477203940523364402755976319039988443676692431571957753139608032594975655952907265331548545978482712901881026796897035152520412986155347172980771042421331827712228842932840779118878459301127224131666183086293464540873344211613157635502500221843336663635492549209065563891247024912390372075390247418784233967973367623387814530633214170702392535748303684768888699431168134896765937282830538988837548536027469935129108638329260216459031911986549219016576669019797300858008894083735919783840096129771725410438561602917505461589456743582467098418101244861239476934725285781955589466694595244871190489712498058170423226285388083745870796617868902160193047257255091418505067717613147684489722449070948195998673855720564564285118868508288349451359277715449474771052372344605485001231400477597812366960300733352288070285108917495510576678970810078630781282551372738477328597129377853566842663625541536431292791502779388093696883574390641290627829865950743592418111195874017249707113753418774282955195594850999346893814520682201392929620147280666866075400198431562006740361835260206508019134213776830548939850326308799)"
    }, {
        label: "ceil(0.4)",
        set: 0.4,
        fn: "ceil",
        param: null,
        expect: "1"
    }, {
        label: "ceil(0.5)",
        set: 0.5,
        fn: "ceil",
        param: null,
        expect: "1"
    }, {
        label: "ceil(0.6)",
        set: 0.6,
        fn: "ceil",
        param: null,
        expect: "1"
    }, {
        label: "ceil(-0.4)",
        set: -0.4,
        fn: "ceil",
        param: null,
        expect: "0"
    }, {
        label: "ceil(-0.5)",
        set: -0.5,
        fn: "ceil",
        param: null,
        expect: "0"
    }, {
        label: "ceil(-0.6)",
        set: -0.6,
        fn: "ceil",
        param: null,
        expect: "0"
    }, {
        label: "floor(0.4)",
        set: 0.4,
        fn: "floor",
        param: null,
        expect: "0"
    }, {
        label: "floor(0.5)",
        set: 0.5,
        fn: "floor",
        param: null,
        expect: "0"
    }, {
        label: "floor(0.6)",
        set: 0.6,
        fn: "floor",
        param: null,
        expect: "0"
    }, {
        label: "floor(-0.4)",
        set: -0.4,
        fn: "floor",
        param: null,
        expect: "-1"
    }, {
        label: "floor(-0.5)",
        set: -0.5,
        fn: "floor",
        param: null,
        expect: "-1"
    }, {
        label: "floor(-0.6)",
        set: -0.6,
        fn: "floor",
        param: null,
        expect: "-1"
    }, {
        label: "round(0.4)",
        set: 0.4,
        fn: "round",
        param: null,
        expect: "0"
    }, {
        label: "round(0.5)",
        set: 0.5,
        fn: "round",
        param: null,
        expect: "1"
    }, {
        label: "round(0.6)",
        set: 0.6,
        fn: "round",
        param: null,
        expect: "1"
    }, {
        label: "round(-0.4)",
        set: -0.4,
        fn: "round",
        param: null,
        expect: "0"
    }, {
        label: "round(-0.5)",
        set: -0.5,
        fn: "round",
        param: null,
        expect: "0"
    }, {
        label: "round(-0.6)",
        set: -0.6,
        fn: "round",
        param: null,
        expect: "-1"
    }, {
        label: "17402216385200408/5539306332998545",
        set: [17402216385200408, 5539306332998545],
        fn: "add",
        param: 0,
        expect: "3.1415876535"
    }, {
        label: "99+1",
        set: [99, 1],
        fn: "add",
        param: 1,
        expect: "100"
    }, {
        label: "gcd(5/8, 3/7)",
        set: [5, 8],
        fn: "gcd",
        param: [3, 7],
        expect: "0.017(857142)" // == 1/56
    }, {
        label: "gcd(2/3, 7/5)",
        set: [2, 3],
        fn: "gcd",
        param: [7, 5],
        expect: "0.0(6)" // == 1/15
    }, {
        label: "lcm(-3, 3)",
        set: -3,
        fn: "lcm",
        param: 3,
        expect: "3"
    }, {
        label: "lcm(3,-3)",
        set: 3,
        fn: "lcm",
        param: -3,
        expect: "3"
    }, {
        label: "lcm(0,3)",
        set: 0,
        fn: "lcm",
        param: 3,
        expect: "0"
    }, {
        label: "lcm(3, 0)",
        set: 3,
        fn: "lcm",
        param: 0,
        expect: "0"
    }, {
        label: "lcm(0, 0)",
        set: 0,
        fn: "lcm",
        param: 0,
        expect: "0"
    }, {
        label: "lcm(200, 333)",
        set: 200,
        fn: "lcm",
        param: 333, expect: "66600"
    }, {
        label: "3.5 < 4.1",
        set: 3.5,
        fn: "compare",
        param: 4.1,
        expect: "-1"
    }, {
        label: "3.5 > 4.1",
        set: 4.1,
        fn: "compare",
        param: 3.1,
        expect: "1"
    }, {
        label: "-3.5 > -4.1",
        set: -3.5,
        fn: "compare",
        param: -4.1,
        expect: "1"
    }, {
        label: "-3.5 > -4.1",
        set: -4.1,
        fn: "compare",
        param: -3.5,
        expect: "-1"
    }, {
        label: "4.3 == 4.3",
        set: 4.3,
        fn: "compare",
        param: 4.3,
        expect: "0"
    }, {
        label: "-4.3 == -4.3",
        set: -4.3,
        fn: "compare",
        param: -4.3,
        expect: "0"
    }, {
        label: "-4.3 < 4.3",
        set: -4.3,
        fn: "compare",
        param: 4.3,
        expect: "-1"
    }, {
        label: "4.3 == -4.3",
        set: 4.3,
        fn: "compare",
        param: -4.3,
        expect: "1"
    }, {
        label: "-0.5^-3",
        set: -0.5,
        fn: "pow",
        param: -3,
        expect: "-8"
    }, {
        label: "-3",
        set: -3,
        fn: "pow",
        param: 2,
        expect: "9"
    }, {
        label: "0^0",
        set: 0,
        fn: "pow",
        param: 0,
        expect: "1"
    }, {
        label: "2/3^7",
        set: [2, 3],
        fn: "pow",
        param: 7,
        expect: "0.(058527663465935070873342478280749885688157293095564700502972107910379515317786922725194330132601737540009144947416552354823959762231367169638774577046181984453589391860996799268404206675811614083219021490626428898033836305441243712848651120256)"
    }, {
        label: "-0.6^4",
        set: -0.6,
        fn: "pow",
        param: 4,
        expect: "0.1296"
    }, {
        label: "8128371:12394 - 8128371/12394",
        set: "8128371:12394",
        fn: "sub",
        param: "8128371/12394",
        expect: "0"
    }, {
        label: "3/4 + 1/4",
        set: "3/4",
        fn: "add",
        param: "1/4",
        expect: "1"
    }, {
        label: "1/10 + 2/10",
        set: "1/10",
        fn: "add",
        param: "2/10",
        expect: "0.3"
    }, {
        label: "5/10 + 2/10",
        set: "5/10",
        fn: "add",
        param: "2/10",
        expect: "0.7"
    }, {
        label: "18/10 + 2/10",
        set: "18/10",
        fn: "add",
        param: "2/10",
        expect: "2"
    }, {
        label: "1/3 + 1/6",
        set: "1/3",
        fn: "add",
        param: "1/6",
        expect: "0.5"
    }, {
        label: "1/3 + 2/6",
        set: "1/3",
        fn: "add",
        param: "2/6",
        expect: "0.(6)"
    }, {
        label: "3/4 / 1/4",
        set: "3/4",
        fn: "div",
        param: "1/4",
        expect: "3"
    }, {
        label: "1/10 / 2/10",
        set: "1/10",
        fn: "div",
        param: "2/10",
        expect: "0.5"
    }, {
        label: "5/10 / 2/10",
        set: "5/10",
        fn: "div",
        param: "2/10",
        expect: "2.5"
    }, {
        label: "18/10 / 2/10",
        set: "18/10",
        fn: "div",
        param: "2/10",
        expect: "9"
    }, {
        label: "1/3 / 1/6",
        set: "1/3",
        fn: "div",
        param: "1/6",
        expect: "2"
    }, {
        label: "1/3 / 2/6",
        set: "1/3",
        fn: "div",
        param: "2/6",
        expect: "1"
    }, {
        label: "3/4 * 1/4",
        set: "3/4",
        fn: "mul",
        param: "1/4",
        expect: "0.1875"
    }, {
        label: "1/10 * 2/10",
        set: "1/10",
        fn: "mul",
        param: "2/10",
        expect: "0.02"
    }, {
        label: "5/10 * 2/10",
        set: "5/10",
        fn: "mul",
        param: "2/10",
        expect: "0.1"
    }, {
        label: "18/10 * 2/10",
        set: "18/10",
        fn: "mul",
        param: "2/10",
        expect: "0.36"
    }, {
        label: "1/3 * 1/6",
        set: "1/3",
        fn: "mul",
        param: "1/6",
        expect: "0.0(5)"
    }, {
        label: "1/3 * 2/6",
        set: "1/3",
        fn: "mul",
        param: "2/6",
        expect: "0.(1)"
    }, {
        label: "5/4 - 1/4",
        set: "5/4",
        fn: "sub",
        param: "1/4",
        expect: "1"
    }, {
        label: "5/10 - 2/10",
        set: "5/10",
        fn: "sub",
        param: "2/10",
        expect: "0.3"
    }, {
        label: "9/10 - 2/10",
        set: "9/10",
        fn: "sub",
        param: "2/10",
        expect: "0.7"
    }, {
        label: "22/10 - 2/10",
        set: "22/10",
        fn: "sub",
        param: "2/10",
        expect: "2"
    }, {
        label: "2/3 - 1/6",
        set: "2/3",
        fn: "sub",
        param: "1/6",
        expect: "0.5"
    }, {
        label: "3/3 - 2/6",
        set: "3/3",
        fn: "sub",
        param: "2/6",
        expect: "0.(6)"
    }, {
        label: "0.006999999999999999",
        set: 0.006999999999999999,
        fn: "add",
        param: 0,
        expect: "0.007"
    }, {
        label: "1/7 - 1",
        set: 1 / 7,
        fn: "add",
        param: -1,
        expect: "-0.(857142)"
    }, {
        label: "0.0065 + 0.0005",
        set: 0.0065,
        fn: "add",
        param: 0.0005,
        expect: "0.007"
    }, {
        label: "6.5/.5",
        set: 6.5,
        fn: "div",
        param: .5,
        expect: "13"
    }, {
        label: "0.999999999999999999999999999",
        set: 0.999999999999999999999999999,
        fn: "sub",
        param: 1,
        expect: "0"
    }, {
        label: "0.5833333333333334",
        set: 0.5833333333333334,
        fn: "add",
        param: 0,
        expect: "0.58(3)"
    }, {
        label: "1.75/3",
        set: 1.75 / 3,
        fn: "add",
        param: 0,
        expect: "0.58(3)"
    }, {
        label: "3.3333333333333",
        set: 3.3333333333333,
        fn: "add",
        param: 0,
        expect: "3.(3)"
    }, {
        label: "4.285714285714285714285714",
        set: 4.285714285714285714285714,
        fn: "add",
        param: 0,
        expect: "4.(285714)"
    }, {
        label: "-4",
        set: -4,
        fn: "neg",
        param: 0,
        expect: "4"
    }, {
        label: "4",
        set: 4,
        fn: "neg",
        param: 0,
        expect: "-4"
    }, {
        label: "0",
        set: 0,
        fn: "neg",
        param: 0,
        expect: "0"
    }
];

describe('Fraction', function() {

    for (var i = 0; i < tests.length; i++) {

        (function(i) {

            if (tests[i].fn) {

                it(tests[i].label || tests[i].set, function() {
                    try {
                        assert.equal(new Fraction(tests[i].set)[tests[i].fn](tests[i].param).toString(), tests[i].expect);
                    } catch (e) {
                        assert.equal(e.toString(), tests[i].expect);
                    }
                });

            } else {

                it(tests[i].label || tests[i].set, function() {
                    try {
                        assert.equal(new Fraction(tests[i].set).toString(), tests[i].expect);
                    } catch (e) {
                        assert.equal(e.toString(), tests[i].expect);
                    }
                });
            }

        })(i);
    }
});

describe('JSON', function() {

    it("Should be possible to stringify the object", function() {

        assert.equal('{"s":1,"n":14623,"d":330}', JSON.stringify(new Fraction("44.3(12)")));

        assert.equal('{"s":-1,"n":2,"d":1}', JSON.stringify(new Fraction(-1 / 2).inverse()));

    });
});

describe('Arguments', function() {

    it("Should be possible to use different kind of params", function() {

        // String
        var fraction = new Fraction("0.1");
        assert.equal("1/10", fraction.n + "/" + fraction.d);

        var fraction = new Fraction("6234/6460");
        assert.equal("3117/3230", fraction.n + "/" + fraction.d);

        // Two params
        var fraction = new Fraction(1, 2);
        assert.equal("1/2", fraction.n + "/" + fraction.d);

        // Object
        var fraction = new Fraction({n: 1, d: 3});
        assert.equal("1/3", fraction.n + "/" + fraction.d);

        // Array
        var fraction = new Fraction([1, 4]);
        assert.equal("1/4", fraction.n + "/" + fraction.d);
    });
});

describe('fractions', function() {

    it("Should pass 0.08 = 2/25", function() {

        var fraction = new Fraction("0.08");
        assert.equal("2/25", fraction.n + "/" + fraction.d);
    });

    it("Should pass 0.200 = 1/5", function() {

        var fraction = new Fraction("0.200");
        assert.equal("1/5", fraction.n + "/" + fraction.d);
    });

    it("Should pass 0.125 = 1/8", function() {

        var fraction = new Fraction("0.125");
        assert.equal("1/8", fraction.n + "/" + fraction.d);
    });

    it("Should pass 8.36 = 209/25", function() {

        var fraction = new Fraction(8.36);
        assert.equal("209/25", fraction.n + "/" + fraction.d);
    });

});

describe('constructors', function() {

    it("Should pass 0.08 = 2/25", function() {

        var tmp = new Fraction({d: 4, n: 2, s: -1});
        assert.equal("-1/2", tmp.s * tmp.n + "/" + tmp.d);

        var tmp = new Fraction(-88.3);
        assert.equal("-883/10", tmp.s * tmp.n + "/" + tmp.d);

        var tmp = new Fraction(-88.3).clone();
        assert.equal("-883/10", tmp.s * tmp.n + "/" + tmp.d);

        var tmp = new Fraction("123.'3'");
        assert.equal("370/3", tmp.s * tmp.n + "/" + tmp.d);

        var tmp = new Fraction("123.'3'").clone();
        assert.equal("370/3", tmp.s * tmp.n + "/" + tmp.d);

        var tmp = new Fraction([-1023461776, 334639305]);
        tmp = tmp.add([4, 25]);
        assert.equal("-4849597436/1673196525", tmp.s * tmp.n + "/" + tmp.d);
    });
});

describe('Latex Output', function() {

    it("Should pass 123.'3' = \\frac{370}{3}", function() {

        var tmp = new Fraction("123.'3'");
        assert.equal("\\frac{370}{3}", tmp.toLatex());
    });

    it("Should pass 1.'3' = \\frac{4}{3}", function() {

        var tmp = new Fraction("1.'3'");
        assert.equal("\\frac{4}{3}", tmp.toLatex());
    });

    it("Should pass -1.0000000000 = -1", function() {

        var tmp = new Fraction("-1.0000000000");
        assert.equal('-1', tmp.toLatex());
    });

    it("Should pass -0.0000000000 = 0", function() {

        var tmp = new Fraction("-0.0000000000");
        assert.equal('0', tmp.toLatex());
    });
});

describe('Fraction Output', function() {

    it("Should pass 123.'3' = 123 1/3", function() {

        var tmp = new Fraction("123.'3'");
        assert.equal('370/3', tmp.toFraction());
    });

    it("Should pass 1.'3' = 1 1/3", function() {

        var tmp = new Fraction("1.'3'");
        assert.equal('4/3', tmp.toFraction());
    });

    it("Should pass -1.0000000000 = -1", function() {

        var tmp = new Fraction("-1.0000000000");
        assert.equal('-1', tmp.toFraction());
    });

    it("Should pass -0.0000000000 = 0", function() {

        var tmp = new Fraction("-0.0000000000");
        assert.equal('0', tmp.toFraction());
    });

    it("Should pass 1/-99/293 = -1/29007", function() {

        var tmp = new Fraction(-99).inverse().div(293);
        assert.equal('-1/29007', tmp.toFraction());
    });

    it('Should work with large calculations', function() {
        var x = Fraction(1123875);
        var y = Fraction(1238750184);
        var z = Fraction(1657134);
        var r = Fraction(77344464613500, 92063);
        assert.equal(x.mul(y).div(z).toFraction(), r.toFraction());
    });
});
