"use strict";
// Copyright 2018, Google, LLC.
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//    http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : new P(function (resolve) { resolve(result.value); }).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
Object.defineProperty(exports, "__esModule", { value: true });
const extend = require("extend");
const node_fetch_1 = require("node-fetch");
const qs = require("querystring");
const url_1 = require("url");
const common_1 = require("./common");
const retry_1 = require("./retry");
// tslint:disable-next-line variable-name
const HttpsProxyAgent = require('https-proxy-agent');
class Gaxios {
    /**
     * The Gaxios class is responsible for making HTTP requests.
     * @param defaults The default set of options to be used for this instance.
     */
    constructor(defaults) {
        this.agentCache = new Map();
        this.defaults = defaults || {};
    }
    /**
     * Perform an HTTP request with the given options.
     * @param opts Set of HTTP options that will be used for this HTTP request.
     */
    request(opts) {
        return __awaiter(this, void 0, void 0, function* () {
            opts = this.validateOpts(opts);
            try {
                const res = yield node_fetch_1.default(opts.url, opts);
                const data = yield this.getResponseData(opts, res);
                const translatedResponse = this.translateResponse(opts, res, data);
                if (!opts.validateStatus(res.status)) {
                    throw new common_1.GaxiosError(`Request failed with status code ${res.status}`, opts, translatedResponse);
                }
                return this.translateResponse(opts, res, data);
            }
            catch (e) {
                const err = e;
                err.config = opts;
                const { shouldRetry, config } = yield retry_1.getRetryConfig(e);
                if (shouldRetry && config) {
                    err.config.retryConfig.currentRetryAttempt =
                        config.retryConfig.currentRetryAttempt;
                    return this.request(err.config);
                }
                throw err;
            }
        });
    }
    getResponseData(opts, res) {
        return __awaiter(this, void 0, void 0, function* () {
            switch (opts.responseType) {
                case 'stream':
                    return res.body;
                case 'json':
                    let data = yield res.text();
                    try {
                        data = JSON.parse(data);
                    }
                    catch (e) {
                    }
                    return data;
                case 'arraybuffer':
                    return res.arrayBuffer();
                case 'blob':
                    return res.blob();
                default:
                    return res.text();
            }
        });
    }
    /**
     * Validate the options, and massage them to match the fetch format.
     * @param opts The original options passed from the client.
     */
    validateOpts(options) {
        const opts = extend(true, {}, this.defaults, options);
        if (!opts.url) {
            throw new Error('URL is required.');
        }
        if (opts.baseUrl) {
            opts.url = (new url_1.URL(opts.url, opts.baseUrl)).toString();
        }
        opts.headers = opts.headers || {};
        if (opts.data) {
            opts.body = JSON.stringify(opts.data);
            opts.headers['Content-Type'] = 'application/json';
        }
        opts.validateStatus = opts.validateStatus || this.validateStatus;
        opts.paramsSerializer = opts.paramsSerializer || this.paramsSerializer;
        opts.responseType = opts.responseType || 'json';
        if (!opts.headers['Accept'] && opts.responseType === 'json') {
            opts.headers['Accept'] = 'application/json';
        }
        opts.method = opts.method || 'GET';
        if (opts.params) {
            const parts = new url_1.URL(opts.url);
            parts.search = opts.paramsSerializer(opts.params);
            opts.url = parts.href;
        }
        const proxy = process.env.HTTPS_PROXY || process.env.https_proxy ||
            process.env.HTTP_PROXY || process.env.http_proxy;
        if (proxy) {
            if (this.agentCache.has(proxy)) {
                opts.agent = this.agentCache.get(proxy);
            }
            else {
                opts.agent = new HttpsProxyAgent(proxy);
                this.agentCache.set(proxy, opts.agent);
            }
        }
        return opts;
    }
    /**
     * By default, throw for any non-2xx status code
     * @param status status code from the HTTP response
     */
    validateStatus(status) {
        return status >= 200 && status < 300;
    }
    /**
     * Encode a set of key/value pars into a querystring format (?foo=bar&baz=boo)
     * @param params key value pars to encode
     */
    paramsSerializer(params) {
        return qs.stringify(params);
    }
    translateResponse(opts, res, data) {
        // headers need to be converted from a map to an obj
        const headers = {};
        res.headers.forEach((value, key) => {
            headers[key] = value;
        });
        return { config: opts, data: data, headers, status: res.status };
    }
}
exports.Gaxios = Gaxios;
//# sourceMappingURL=gaxios.js.map