/*!
 * Copyright 2015 Google Inc. All Rights Reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/*!
 * @module logging/entry
 */

'use strict';

var extend = require('extend');
var is = require('is');

/**
 * Create an entry object to define new data to insert into a log.
 *
 * @resource [LogEntry JSON representation]{@link https://cloud.google.com/logging/docs/api/ref_v2beta1/rest/v2beta1/LogEntry}
 *
 * @param {object=|string=} resource - See a
 *     [Monitored Resource](https://cloud.google.com/logging/docs/api/ref_v2beta1/rest/v2beta1/MonitoredResource).
 * @param {object|string} data - The data to use as the value for this log
 *     entry.
 * @return {module:logging/entry}
 *
 * @example
 * var gcloud = require('gcloud')({
 *   keyFilename: '/path/to/keyfile.json',
 *   projectId: 'grape-spaceship-123'
 * });
 *
 * var logging = gcloud.logging();
 *
 * var syslog = logging.log('syslog');
 *
 * var resource = {
 *   type: 'gce_instance',
 *   labels: {
 *     zone: 'global',
 *     instance_id: 3
 *   }
 * };
 *
 * var entry = syslog.entry(resource, {
 *   delegate: process.env.user
 * });
 *
 * syslog.alert(entry, function(err, apiResponse) {
 *   if (!error) {
 *     // Log entry inserted successfully.
 *   }
 * });
 *
 * //-
 * // You will also receive `Entry` objects when using
 * // {module:logging#getEntries} and {module:logging/log#getEntries}.
 * //-
 * logging.getEntries(function(err, entries) {
 *   if (!err) {
 *     // entries[0].data = The data value from the log entry.
 *   }
 * });
 */
function Entry(resource, data) {
  if (!data) {
    this.data = resource;
    return;
  }

  this.resource = resource;
  this.data = data;
}

/**
 * Create an Entry object from an API response, such as `entries:list`.
 *
 * @private
 *
 * @param {object} entry - An API representation of an entry. See a
 *     [LogEntry](https://cloud.google.com/logging/docs/api/ref_v2beta1/rest/v2beta1/LogEntry).
 * @return {module:logging/entity}
 */
Entry.fromApiResponse_ = function(entry) {
  // Only one of these are populated.
  var data = entry.protoPayload || entry.jsonPayload || entry.textPayload;
  return extend(new Entry(entry.resource, data), entry);
};

/**
 * Serialize an entry to the format the API expects.
 *
 * @private
 */
Entry.prototype.toJSON = function() {
  var entry = extend(true, {}, this);

  var whitelist = [
    'logName',
    'resource',
    'timestamp',
    'severity',
    'insertId',
    'httpRequest',
    'labels',
    'operation'
  ];

  for (var prop in entry) {
    if (whitelist.indexOf(prop) === -1) {
      delete entry[prop];
    }
  }

  if (is.string(this.resource)) {
    entry.resource = {
      type: this.resource
    };
  }

  if (is.object(this.data)) {
    entry.jsonPayload = this.data;
  } else if (is.string(this.data)) {
    entry.textPayload = this.data;
  }

  return entry;
};

module.exports = Entry;
