import { tokenize as htmlTokenize } from '@nfrasser/simple-html-tokenizer';
import { tokenize, Options } from 'linkifyjs';

const LinkifyResult = 'LinkifyResult';
const StartTag = 'StartTag';
const EndTag = 'EndTag';
const Chars = 'Chars';
const Comment = 'Comment';
const Doctype = 'Doctype';

/**
 * @param {string} str html string to link
 * @param {import('linkifyjs').Opts} [opts] linkify options
 * @returns {string} resulting string
 */
export default function linkifyHtml(str, opts = {}) {
	// `tokens` and `token` in this section refer to tokens generated by the
	// HTML parser, not linkify's parser
	const tokens = htmlTokenize(str);
	const linkifiedTokens = [];
	const linkified = [];

	const options = new Options(opts, defaultRender);

	// Linkify the tokens given by the parser
	for (let i = 0; i < tokens.length; i++) {
		const token = tokens[i];

		if (token.type === StartTag) {
			linkifiedTokens.push(token);

			// Ignore all the contents of ignored tags
			const tagName = token.tagName.toUpperCase();
			const isIgnored = tagName === 'A' || options.ignoreTags.indexOf(tagName) >= 0;
			if (!isIgnored) {
				continue;
			}

			let preskipLen = linkifiedTokens.length;
			skipTagTokens(tagName, tokens, ++i, linkifiedTokens);
			i += linkifiedTokens.length - preskipLen - 1;
		} else if (token.type !== Chars) {
			// Skip this token, it's not important
			linkifiedTokens.push(token);
		} else {
			// Valid text token, linkify it!
			const linkifedChars = linkifyChars(token.chars, options);
			linkifiedTokens.push.apply(linkifiedTokens, linkifedChars);
		}
	}

	// Convert the tokens back into a string
	for (let i = 0; i < linkifiedTokens.length; i++) {
		const token = linkifiedTokens[i];
		switch (token.type) {
			case LinkifyResult:
				linkified.push(token.rendered);
				break;
			case StartTag: {
				let link = '<' + token.tagName;
				if (token.attributes.length > 0) {
					link += ' ' + attributeArrayToStrings(token.attributes).join(' ');
				}
				if (token.selfClosing) {
					link += ' /';
				}
				link += '>';
				linkified.push(link);
				break;
			}
			case EndTag:
				linkified.push(`</${token.tagName}>`);
				break;
			case Chars:
				linkified.push(escapeText(token.chars));
				break;
			case Comment:
				linkified.push(`<!--${escapeText(token.chars)}-->`);
				break;
			case Doctype: {
				let doctype = `<!DOCTYPE ${token.name}`;
				if (token.publicIdentifier) {
					doctype += ` PUBLIC "${token.publicIdentifier}"`;
				}
				if (token.systemIdentifier) {
					doctype += ` "${token.systemIdentifier}"`;
				}
				doctype += '>';
				linkified.push(doctype);
				break;
			}
		}
	}

	return linkified.join('');
}

/**
	`tokens` and `token` in this section referes to tokens returned by
	`linkify.tokenize`. `linkified` will contain HTML Parser-style tokens
	@param {string}
	@param {import('linkifyjs').Options}
*/
function linkifyChars(str, options) {
	const tokens = tokenize(str);
	const result = [];

	for (let i = 0; i < tokens.length; i++) {
		const token = tokens[i];
		if (token.t === 'nl' && options.get('nl2br')) {
			result.push({
				type: StartTag,
				tagName: 'br',
				attributes: [],
				selfClosing: true,
			});
		} else if (!token.isLink || !options.check(token)) {
			result.push({ type: Chars, chars: token.toString() });
		} else {
			result.push({
				type: LinkifyResult,
				rendered: options.render(token),
			});
		}
	}

	return result;
}

/**
	Returns a list of tokens skipped until the closing tag of tagName.

	* `tagName` is the closing tag which will prompt us to stop skipping
	* `tokens` is the array of tokens generated by HTML5Tokenizer which
	* `i` is the index immediately after the opening tag to skip
	* `skippedTokens` is an array which skipped tokens are being pushed into

	Caveats

	* Assumes that i is the first token after the given opening tagName
	* The closing tag will be skipped, but nothing after it
	* Will track whether there is a nested tag of the same type
*/
function skipTagTokens(tagName, tokens, i, skippedTokens) {
	// number of tokens of this type on the [fictional] stack
	let stackCount = 1;

	while (i < tokens.length && stackCount > 0) {
		let token = tokens[i];

		if (token.type === StartTag && token.tagName.toUpperCase() === tagName) {
			// Nested tag of the same type, "add to stack"
			stackCount++;
		} else if (token.type === EndTag && token.tagName.toUpperCase() === tagName) {
			// Closing tag
			stackCount--;
		}

		skippedTokens.push(token);
		i++;
	}

	// Note that if stackCount > 0 here, the HTML is probably invalid
	return skippedTokens;
}

function defaultRender({ tagName, attributes, content }) {
	return `<${tagName} ${attributesToString(attributes)}>${escapeText(content)}</${tagName}>`;
}

function escapeText(text) {
	return text.replace(/</g, '&lt;').replace(/>/g, '&gt;');
}

function escapeAttr(attr) {
	return attr.replace(/"/g, '&quot;');
}

function attributesToString(attributes) {
	const result = [];
	for (const attr in attributes) {
		const val = attributes[attr] + '';
		result.push(`${attr}="${escapeAttr(val)}"`);
	}
	return result.join(' ');
}

function attributeArrayToStrings(attrs) {
	const attrStrs = [];
	for (let i = 0; i < attrs.length; i++) {
		const name = attrs[i][0];
		const value = attrs[i][1] + '';
		attrStrs.push(`${name}="${escapeAttr(value)}"`);
	}
	return attrStrs;
}
