const http = require('https'); // or 'https' for https:// URLs
const fs = require('fs');
const punycode = require('punycode/');

const tldsListUrl = 'https://data.iana.org/TLD/tlds-alpha-by-domain.txt';
const tldsjs = 'packages/linkifyjs/src/tlds.js';
let tldsListContents = '';


/**
 * Given a list of TLDs, encodes into a compact string that may be decoded
 * with decodeTlds. Works best when input is sorted alphabetically
 *
 * Example input: ['cat', 'cats', 'car', 'cars']
 * Example output: 'cat0s2r0s4'
 */
function encodeTlds(tlds) {
	return encodeTrie(createTrie(tlds));
}

/**
 * Given a list of words, encodes into an object-based trie.
 *
 * Example input: ['cat', 'cats', 'car', 'cars']
 * Example output: {
 *   c: {
 *     a: {
 *       t: {
 *         isWord: true,
 *         s: { isWord: true }
 *       },
 *       r: {
 *         isWord: true,
 *         s: { isWord: true }
 *       }
 *     }
 *   }
 * }
 */
function createTrie(words) {
	const root = {};
	for (const word of words) {
		let current = root;
		for (const letter of word) {
			if (!(letter in current)) {
				current[letter] = {};
			}
			current = current[letter];
		}
		current.isWord = true;
	}
	return root;
}


/**
 * Given an object trie created by `createTrie`, encodes into a compact string
 * that can later be decoded back into a list of strings.
 *
 * Using the example trie above, output would be: 'cat0s2r0s4'
 *
 * NOTE: Does not work if trie contains worlds with digits 0-9
 */
function encodeTrie(trie) {
	return encodeTrieHelper(trie).join('');
}

function encodeTrieHelper(trie) {
	const output = [];
	for (const k in trie) {
		if (k === 'isWord') {
			output.push(0); // Zero means previous steps into trie make a word
			continue;
		}
		output.push(k); // Push child node means drop down a level into the trie
		output.push(...encodeTrieHelper(trie[k]));
		// increment the number of times we have to go back up to get to this
		// level of the trie.
		if (typeof output[output.length - 1] === 'number') {
			output[output.length - 1] += 1;
		} else {
			output.push(1);
		}
	}
	return output;
}

/**
 * Converts a string of Top-Level Domain names back into a list of strings.
 *
 * Example input: 'cat0s2r0s4'
 * Example output: ['cat', 'cats', 'car', 'cars']
 */
function decodeTlds(encoded) {
	const words = [];
	const stack = [];
	let i = 0;
	let digits = '0123456789';
	while (i < encoded.length) {
		let popDigitCount = 0;
		while (digits.indexOf(encoded[i + popDigitCount]) >= 0) {
			popDigitCount++; // encountered some digits, have to pop to go one level up trie
		}
		if (popDigitCount > 0) {
			words.push(stack.join('')); // whatever preceded the pop digits must be a word
			for (let popCount = parseInt(encoded.substring(i, i + popDigitCount), 10); popCount > 0; popCount--) {
				stack.pop();
			}
			i += popDigitCount;
		} else {
			stack.push(encoded[i]); // drop down a level into the trie
			i++;
		}
	}
	return words;
}

http.get(tldsListUrl, (response) => {
	console.log(`Downloading ${tldsListUrl}...`);
	response.on('data', (chunk) => { tldsListContents += chunk; });
	response.on('end', () => {
		console.log(`Downloaded. Re-generating ${tldsjs}...`);

		// NOTE: punycode versions of IDNs (e.g., `XN--...`) do not get included
		// in the TLDs list because these will not be as commonly used without
		// the http prefix anyway and linkify will already force-encode those.
		let tlds = [];
		let utlds = [];

		// NOTE: vermögensberater vermögensberatung are special cases because
		// they're the only ones that contain a mix of ASCII and non-ASCII
		// characters.
		const specialTlds = ['XN--VERMGENSBERATER-CTB', 'XN--VERMGENSBERATUNG-PWB'];
		const specialUtlds = specialTlds.map(tld => punycode.toUnicode(tld.toLowerCase()));

		for (const line of tldsListContents.split('\n').map(line => line.trim())) {
			if (!line || line[0] === '#' || specialTlds.includes(line)) { continue; }
			if (/^XN--/.test(line)) {
				utlds.push(punycode.toUnicode(line.toLowerCase()));
			} else {
				tlds.push(line.toLowerCase());
			}
		}
		tlds = tlds.concat(specialUtlds).sort();
		utlds = utlds.sort();

		console.log('Encoding...');
		const encodedTlds = encodeTlds(tlds);
		const encodedUtlds = encodeTlds(utlds);

		console.log('Testing decode...');
		const decodedTlds = decodeTlds(encodedTlds);
		console.assert(JSON.stringify(decodedTlds) === JSON.stringify(tlds), 'Invalid encode/decode routine');

		const jsFile = fs.openSync(tldsjs, 'w');
		fs.writeSync(jsFile, '// THIS FILE IS AUTOMATICALLY GENERATED DO NOT EDIT DIRECTLY\n');
		fs.writeSync(jsFile, '// See update-tlds.js for encoding/decoding format\n');
		fs.writeSync(jsFile, `// ${tldsListUrl}\n`);

		// Write TLDs
		fs.writeSync(jsFile, 'export const encodedTlds = \'');
		fs.writeSync(jsFile, encodedTlds);
		fs.writeSync(jsFile, '\';\n');
		fs.writeSync(jsFile, '// Internationalized domain names containing non-ASCII\n');
		fs.writeSync(jsFile, 'export const encodedUtlds = \'');
		fs.writeSync(jsFile, encodedUtlds);
		fs.writeSync(jsFile, '\';\n');
		fs.closeSync(jsFile);

		console.log('Done');
	});
});
