/* Copyright 2017 Google Inc. All Rights Reserved.

Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
==============================================================================*/

import {Tensor} from '../graph';
import {NDArrayMath} from '../math/math';
import {Array1D, Scalar} from '../math/ndarray';
import {TensorArrayMap} from '../tensor_array_map';
import * as util from '../util';

import {Operation} from './op';

export class Softmax extends Operation {
  constructor(private logitsTensor: Tensor, private output: Tensor) {
    super();
  }

  feedForward(math: NDArrayMath, inferenceArrays: TensorArrayMap) {
    const logits = inferenceArrays.get(this.logitsTensor) as Array1D;
    return math.scope((keep) => {
      inferenceArrays.set(this.output, keep(math.softmax(logits)));
    });
  }

  backProp() {
    throw Error('Softmax backprop is not yet implemented');
  }
}

export class SoftmaxCrossEntropyCost extends Operation {
  constructor(
      private logitsTensor: Tensor, private labelTensor: Tensor,
      private yTensor: Tensor) {
    super();
    this.softmaxTensor = new Tensor(logitsTensor.shape);
  }

  feedForward(math: NDArrayMath, inferenceArrays: TensorArrayMap) {
    const logits = inferenceArrays.get(this.logitsTensor) as Array1D;
    const label = inferenceArrays.get(this.labelTensor) as Array1D;

    math.scope((keep) => {
      const softmaxResult = math.softmax(logits);

      inferenceArrays.set(this.softmaxTensor, keep(softmaxResult));
      inferenceArrays.set(
          this.yTensor,
          keep(crossEntropyCost(math, softmaxResult, label, this.epsilon)));
    });
  }

  backProp(
      math: NDArrayMath, inferenceArrays: TensorArrayMap,
      gradientArrays: TensorArrayMap) {
    const softmax = inferenceArrays.get(this.softmaxTensor);
    const label = inferenceArrays.get(this.labelTensor);

    math.scope((keep) => {
      gradientArrays.set(this.logitsTensor, keep(math.sub(softmax, label)));
    });
  }

  disposeTransientArrays(
      inferenceArrays: TensorArrayMap, gradientArrays: TensorArrayMap) {
    inferenceArrays.disposeArray(this.softmaxTensor);
  }

  dispose() {
    this.epsilon.dispose();
  }

  private softmaxTensor: Tensor;
  private epsilon = Scalar.new(1e-5);
}

export function crossEntropyCost(
    math: NDArrayMath, y: Array1D, target: Array1D, epsilon: Scalar): Scalar {
  util.assert(
      y.size === target.size, 'The output and target must be the same size');

  return math.scope(() => {
    const yPlusEps = math.scalarPlusArray(epsilon, y);
    const logOutput = math.log(yPlusEps);
    const tarLogOutput = math.elementWiseMul(target, logOutput);
    const costVector = math.neg(tarLogOutput);
    return math.sum(costVector);
  });
}
