# Webpack

This library is built with some dynamic logic to determine if it is invoked in a
script tag or in nodejs.  Webpack does not understand those feature tests, so by
default it will do some strange things.

## Basic Usage

`webpack.app.js` demonstrates bundling an entire app script in a bundle.  For
basic projects requiring the module from the npm package, it is sufficient to
suppress the node shims:

```js
/* webpack config for app.out.js */
{
	/* entry point app.js */
	entry: './app.js',

	/* write to app.out.js */
	output: { path:__dirname, filename: './app.out.js' },

	/* suppress node shims */
	node: {
		fs: false,
		process: false,
		Buffer: false
	}
}
```

## Suppressing the Node shims

The library properly guards against accidental leakage of node features in the
browser but webpack disregards those.  The config should explicitly suppress:

```js
	node: {
		fs: false,
		process: false,
		Buffer: false
	}
```

## Omitting optional dependencies

The `codepage` is needed in certain special cases, including files generated by
non-US-English versions of Excel, but may not be needed.  To reduce build size,
the module can be omitted by aliasing the dependency:

```js
	resolve: {
		alias: { "./dist/cpexcel.js": "" }
	},
```

Alternatively, bundling the `xlsx.core.min.js` script always omits dependencies.

## Bower and minified versions

Webpack may show a message like "This seems to be a pre-built javascript file"
when processing minified files (like the default Bower script).  The message is
harmless.  To suppress the message, set `module.noParse` in the webpack config:

```js
	module: {
		noParse: [
			/xlsx.core.min.js/,
			/xlsx.full.min.js/
		]
	}
```

## Other Demos

This demo also attempts to demonstrate bundling of the library as well as the
core and full distribution versions.  `app.js` is the common app code (it will
not be bundled).  The individual bundles merely wrap and reflect `XLSX`.  The
app code uses the bundles with script tag inclusion in the main HTML files. The
worker scripts use the bundles with `importScripts` references.

| required script |   HTML page |     entry |   worker script |
|----------------:|------------:|----------:|----------------:|
| main `xlsx` lib | `main.html` | `main.js` | `mainworker.js` |
| `xlsx.core.min` | `core.html` | `core.js` | `coreworker.js` |
| `xlsx.full.min` | `full.html` | `full.js` | `fullworker.js` |

The entry points in the demo merely require and re-export the library:

```js
/* main.js */
var XLSX = require('../../');
console.log("it works!");
module.exports = XLSX;
```

The main advantage of reflecting the library is deduplication: the library code
is only downloaded once.  The basic example builds a separate worker script and
eventually ships the library twice.

### Reflecting the XLSX variable

This library will not assign to `module.exports` if it is run in the browser. To
convince webpack, the demo webpack config sets `output`:

```js
	output: {
		libraryTarget: 'var',
		library: 'XLSX'
	}
```

[![Analytics](https://ga-beacon.appspot.com/UA-36810333-1/SheetJS/js-xlsx?pixel)](https://github.com/SheetJS/js-xlsx)
