/**
 * @license
 * Copyright Google Inc. All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */

import {StaticReflector} from '../aot/static_reflector';
import {CompileDirectiveMetadata, CompileIdentifierMetadata, CompileNgModuleMetadata, CompilePipeSummary, CompileTypeMetadata} from '../compile_metadata';
import {DEFAULT_INTERPOLATION_CONFIG, DomElementSchemaRegistry, Lexer, ParseError, Parser} from '../compiler';
import {CompileMetadataResolver} from '../metadata_resolver';
import * as o from '../output/output_ast';
import {BindingParser} from '../template_parser/binding_parser';
import {TemplateAst} from '../template_parser/template_ast';
import {OutputContext} from '../util';

import {compilePipe} from './r3_pipe_compiler';
import {BUILD_OPTIMIZER_REMOVE, OutputMode} from './r3_types';
import {compileComponent, compileDirective} from './r3_view_compiler';

export const enum ModuleKind {
  Renderer2,
  Renderer3,
}

/**
 * Produce the back-patching function for the given module to the output context.
 */
export function compileModuleBackPatch(
    outputCtx: OutputContext, name: string, module: CompileNgModuleMetadata, kind: ModuleKind,
    backPatchReferenceOf: (module: CompileTypeMetadata) => o.Expression,
    parseTemplate: (
        compMeta: CompileDirectiveMetadata, ngModule: CompileNgModuleMetadata,
        directiveIdentifiers: CompileIdentifierMetadata[]) => {
      template: TemplateAst[],
      pipes: CompilePipeSummary[]
    },
    reflector: StaticReflector, resolver: CompileMetadataResolver) {
  const imports: o.Statement[] = [];
  let statements: o.Statement[] = [];

  // Call dependent back patching
  for (const importedModule of module.importedModules) {
    const importBackPatchFunction = backPatchReferenceOf(importedModule.type);

    // e.g. // @BUILD_OPTIMIZER_REMOVE
    imports.push(new o.CommentStmt(BUILD_OPTIMIZER_REMOVE));

    // e.g. ngBackPatch_some_other_module_Module();
    imports.push(importBackPatchFunction.callFn([]).toStmt());
  }

  // The local output context allows collecting the back-patch statements that
  // are generated by the various compilers which allows putting placing them
  // into the body of a function instead of at global scope.
  const localCtx: OutputContext = {
    statements,
    constantPool: outputCtx.constantPool,
    genFilePath: outputCtx.genFilePath,
    importExpr: outputCtx.importExpr
  };

  // e.g. export function ngBackPatch_some_module_Lib1Module()
  if (kind === ModuleKind.Renderer2) {
    // For all Renderer2 modules generate back-patching code for all the components, directives,
    // pipes, and injectables as well as the injector def for the module itself.

    const expressionParser = new Parser(new Lexer());
    const elementSchemaRegistry = new DomElementSchemaRegistry();
    const errors: ParseError[] = [];
    const hostBindingParser = new BindingParser(
        expressionParser, DEFAULT_INTERPOLATION_CONFIG, elementSchemaRegistry, [], errors);

    // Back-patch all declared directive and components
    for (const declaredDirective of module.declaredDirectives) {
      const declaredDirectiveMetadata = resolver.getDirectiveMetadata(declaredDirective.reference);
      if (declaredDirectiveMetadata.isComponent) {
        const {template: parsedTemplate, pipes: parsedPipes} =
            parseTemplate(declaredDirectiveMetadata, module, module.transitiveModule.directives);
        compileComponent(
            localCtx, declaredDirectiveMetadata, parsedPipes, parsedTemplate, reflector,
            hostBindingParser, OutputMode.BackPatch);
      } else {
        compileDirective(
            localCtx, declaredDirectiveMetadata, reflector, hostBindingParser,
            OutputMode.BackPatch);
      }
    }

    // Back-patch all pipes declared in the module.
    for (const pipeType of module.declaredPipes) {
      const pipeMetadata = resolver.getPipeMetadata(pipeType.reference);
      if (pipeMetadata) {
        compilePipe(localCtx, pipeMetadata, reflector, OutputMode.BackPatch);
      }
    }

    if (errors.length) {
      throw new Error(errors.map(e => e.toString()).join('\n'));
    }
  }

  outputCtx.statements.push(new o.DeclareFunctionStmt(
      name, [], [...imports, ...statements], o.INFERRED_TYPE, [o.StmtModifier.Exported]));
}
