The `a11y` package provides a number of tools to improve accessibility, described below.

### ListKeyManager
`ListKeyManager` manages the active option in a list of items based on keyboard interaction.
Intended to be used with components that correspond to a `role="menu"` or `role="listbox"` pattern.

#### Basic usage
Any component that uses a `ListKeyManager` will generally do three things:
* Create a `@ViewChildren` query for the options being managed.
* Initialize the `ListKeyManager`, passing in the options.
* Forward keyboard events from the managed component to the `ListKeyManager`.

Each option should implement the `ListKeyManagerOption` interface:
```ts
interface ListKeyManagerOption {
  disabled?: boolean;
  getLabel?(): string;
}
```

#### Wrapping
Navigation through options can be made to wrap via the `withWrap` method
```ts
this.keyManager = new FocusKeyManager(...).withWrap();
``` 

#### Types of key managers
There are two varieties of `ListKeyManager`, `FocusKeyManager` and `ActiveDescendantKeyManager`.

##### FocusKeyManager
Used when options will directly receive browser focus. Each item managed must implement the
`FocusableOption` interface:
```ts
interface FocusableOption extends ListKeyManagerOption {
  focus(): void;
}
```

##### ActiveDescendantKeyManager
Used when options will be marked as active via `aria-activedescendant`. 
Each item managed must implement the
`Highlightable` interface:
```ts
interface Highlightable extends ListKeyManagerOption {
  setActiveStyles(): void;
  setInactiveStyles(): void;
}
```

Each item must also have an ID bound to the listbox's or menu's `aria-activedescendant`.


### FocusTrap
The `cdkTrapFocus` directive traps <kbd>Tab</kbd> key focus within an element. This is intended to
be used to create accessible experience for components like
[modal dialogs](https://www.w3.org/TR/wai-aria-practices-1.1/#dialog_modal), where focus must be
constrained.

This directive is declared in `A11yModule`.

#### Example
```html
<div class="my-inner-dialog-content" cdkTrapFocus>
  <!-- Tab and Shift + Tab will not leave this element. -->
</div>
```

This directive will not prevent focus from moving out of the trapped region due to mouse
interaction.

#### Regions
Regions can be declared explicitly with an initial focus element by using 
the `cdkFocusRegionStart`, `cdkFocusRegionEnd` and `cdkFocusInitial` DOM attributes. 
`cdkFocusInitial` specifies the element that will receive focus upon initialization of the region. 
`cdkFocusRegionStart` and `cdkFocusRegionEnd` define the region within which focus will be 
trapped. When using the tab key, focus will move through this region and wrap around on either end.

For example:

```html
<a mat-list-item routerLink cdkFocusRegionStart>Focus region start</a>
<a mat-list-item routerLink>Link</a>
<a mat-list-item routerLink cdkFocusInitial>Initially focused</a>
<a mat-list-item routerLink cdkFocusRegionEnd>Focus region end</a>
```


### InteractivityChecker
`InteractivityChecker` is used to check the interactivity of an element, capturing disabled,
visible, tabbable, and focusable states for accessibility purposes. See the API docs for more
details.


### LiveAnnouncer
`LiveAnnouncer` is used to announce messages for screen-reader users using an `aria-live` region.
See [the W3C's WAI-ARIA](https://www.w3.org/TR/wai-aria/states_and_properties#aria-live)
for more information on aria-live regions. 

#### Example
```ts
@Component({...})
export class MyComponent {

 constructor(liveAnnouncer: LiveAnnouncer) {
   liveAnnouncer.announce("Hey Google");
 }
}
```
