/**
 * Copyright 2013-present, Facebook, Inc.
 * All rights reserved.
 *
 * This source code is licensed under the BSD-style license found in the
 * LICENSE file in the root directory of this source tree. An additional grant
 * of patent rights can be found in the PATENTS file in the same directory.
 */

'use strict';

var React = require('react');

if (typeof React === 'undefined') {
  throw Error(
    'react-linked-input could not find the React object. If you are using script tags, ' +
      'make sure that React is being loaded before react-linked-input.'
  );
}

var invariant = require('fbjs/lib/invariant');

function _assertSingleLink(inputProps) {
  invariant(
    inputProps.checkedLink == null || inputProps.valueLink == null,
    'Cannot provide a checkedLink and a valueLink. If you want to use ' +
      "checkedLink, you probably don't want to use valueLink and vice versa."
  );
}
function _assertValueLink(inputProps) {
  _assertSingleLink(inputProps);
  invariant(
    inputProps.value == null && inputProps.onChange == null,
    'Cannot provide a valueLink and a value or onChange event. If you want ' +
      "to use value or onChange, you probably don't want to use valueLink."
  );
}

function _assertCheckedLink(inputProps) {
  _assertSingleLink(inputProps);
  invariant(
    inputProps.checked == null && inputProps.onChange == null,
    'Cannot provide a checkedLink and a checked property or onChange event. ' +
      "If you want to use checked or onChange, you probably don't want to " +
      'use checkedLink'
  );
}

/**
 * Provide a linked `value` attribute for controlled forms. You should not use
 * this outside of the ReactDOM controlled form components.
 */
var LinkedValueUtils = {
  /**
   * @param {object} inputProps Props for form component
   * @return {*} current value of the input either from value prop or link.
   */
  getValue: function(inputProps) {
    if (inputProps.valueLink) {
      _assertValueLink(inputProps);
      return inputProps.valueLink.value;
    }
    return inputProps.value;
  },

  /**
   * @param {object} inputProps Props for form component
   * @return {*} current checked status of the input either from checked prop
   *             or link.
   */
  getChecked: function(inputProps) {
    if (inputProps.checkedLink) {
      _assertCheckedLink(inputProps);
      return inputProps.checkedLink.value;
    }
    return inputProps.checked;
  },

  /**
   * @param {object} inputProps Props for form component
   * @param {SyntheticEvent} event change event to handle
   */
  executeOnChange: function(inputProps, event) {
    if (inputProps.valueLink) {
      _assertValueLink(inputProps);
      return inputProps.valueLink.requestChange(event.target.value);
    } else if (inputProps.checkedLink) {
      _assertCheckedLink(inputProps);
      return inputProps.checkedLink.requestChange(event.target.checked);
    } else if (inputProps.onChange) {
      return inputProps.onChange.call(undefined, event);
    }
  }
};

function _classCallCheck(instance, Constructor) {
  if (!(instance instanceof Constructor)) {
    throw new TypeError('Cannot call a class as a function');
  }
}

function _possibleConstructorReturn(self, call) {
  if (!self) {
    throw new ReferenceError(
      "this hasn't been initialised - super() hasn't been called"
    );
  }
  return call && (typeof call === 'object' || typeof call === 'function')
    ? call
    : self;
}

function _inherits(subClass, superClass) {
  if (typeof superClass !== 'function' && superClass !== null) {
    throw new TypeError(
      'Super expression must either be null or a function, not ' +
        typeof superClass
    );
  }
  subClass.prototype = Object.create(superClass && superClass.prototype, {
    constructor: {
      value: subClass,
      enumerable: false,
      writable: true,
      configurable: true
    }
  });
  if (superClass) {
    if (Object.setPrototypeOf) {
      Object.setPrototypeOf(subClass, superClass);
    } else {
      // eslint-disable-next-line no-proto
      subClass.__proto__ = superClass;
    }
  }
}

var LI = (function(_React$Component) {
  _inherits(LinkedInput, _React$Component);

  function LinkedInput() {
    _classCallCheck(this, LinkedInput);

    var _this = _possibleConstructorReturn(this, _React$Component.call(this));

    _this.handleChange = _this.handleChange.bind(_this);
    return _this;
  }

  LinkedInput.prototype.handleChange = function handleChange(e) {
    LinkedValueUtils.executeOnChange(this.props, e);
  };

  LinkedInput.prototype.render = function render() {
    var newProps = Object.assign({}, this.props);
    newProps.value = LinkedValueUtils.getValue(this.props);
    newProps.checked = LinkedValueUtils.getChecked(this.props);
    newProps.onChange = this.handleChange;
    delete newProps.valueLink;
    delete newProps.checkedLink;
    return React.createElement('input', newProps);
  };

  return LinkedInput;
})(React.Component);

module.exports = LI;
