// bwipjs/lib/xhr-fonts.js
//
// Creates the browser global bwipjs_fonts font manager and demand-loads
// fonts using XHR.
//
// This is part of the bwip-js project available at:
//
// 		http://metafloor.github.io/bwip-js
//
// Copyright (c) 2011-2018 Mark Warren
//
// The MIT License
//
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to deal
// in the Software without restriction, including without limitation the rights
// to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
// copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
// OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
// THE SOFTWARE.
//
"use strict"

var bwipjs_fonts = {
	monocolor: false,	// aka monochrome
	fontsets: [],
	names: {},
	toload:	{},
};

// Get the url to this script.  The fonts will be in ../fonts/
(function() {
	var srcpath = document.querySelector('script[src$="xhr-fonts.js"]').getAttribute('src');
	bwipjs_fonts.fontdir = srcpath.replace(/lib\/xhr-fonts.js$/, 'fonts/');
})();


bwipjs_fonts.lookup = function lookup(name) {
	var fontid = this.names[name.toUpperCase()];
	return fontid === undefined ? 1 : fontid;		// OCR B default
};

bwipjs_fonts.monochrome = function monochrome(mono) {
	this.monocolor = mono;
};

bwipjs_fonts.loadfonts = function loadfonts(callback) {
	var count = 0;
	for (var temp in this.toload) {
		(function(fontpath, font) {
			var xhr = new XMLHttpRequest;
			xhr.open('GET', bwipjs_fonts.fontdir + fontpath, true);
			xhr.responseType = 'arraybuffer';
			xhr.onload = function(e) {
				if (xhr.status == 200) {
					font.bytes = new Uint8Array(xhr.response);
					font.view  = new DataView(xhr.response);
				}
				if (--count == 0) {
					callback();
				}
			}

			count++;
			xhr.send(null);
		})(temp, this.toload[temp]);
	}
	this.toload = {};

	if (!count) {
		callback();
	}
};


// `width` and `height` are the same size, scaled respectively by the
// x,y factors.  Because bar code width is what we care about, font
// width is the only metric we use to match the font.
bwipjs_fonts.getglyph = function getglyph(fontid, charcode, width, height) {
	if (fontid < 0 || fontid >= this.fontsets.length) {
		fontid = 1;		// OCR B default
	}
	var fontset = this.fontsets[fontid];
	if (width < fontset.minsize) {
		width = fontset.minsize;
	} else if (width > fontset.maxsize) {
		width = fontset.maxsize;
	}

	// Find the largest size not exceeding
	var size = width;
	while (size > 5 && !fontset.sizes[size]) {
		size--;
	}
	if (size == 5) {
		throw 'font-size not supported (' + width + ')';
	}

	var key  = (this.monocolor ? 'm' : 'a') + size;
	var font = fontset.fonts[key];

	// Need to demand-load the font bitmap?
	if (!font) {
		fontset.fonts[key] = font = { glyphs:[] };
		this.toload['fnt' + fontid + (this.monocolor ? 'm-' : 'a-') + size + '.bin'] = font;
	}

	// Convert the charcode to a glyph index.  Simple 7-bit printable ascii.  No special mapping supported.
	if (charcode < fontset.minchar || charcode > fontset.maxchar) {
		charcode = fontset.minchar;
	}
	charcode -= fontset.minchar;

	var glyph = font.glyphs[charcode];
	if (!glyph) {
		var glidx = charcode * 5;
		var metrics = fontset.metrics[key];
		font.glyphs[charcode] = glyph = {
				advance: metrics[glidx + 0],
				top:	 metrics[glidx + 1],
				left:	 metrics[glidx + 2],
				width:	 metrics[glidx + 3],
				height:	 metrics[glidx + 4],
			};

		// The maxicode bulls-eye is generated with left=0, top=height, but needs to be
		// centered on 0,0
		//if (fontid == 2 && charcode == 4) {
		//	glyph.top  -= ~~(glyph.height / 2);
		//	glyph.left -= ~~(glyph.width / 2);
		//}

	}

	// Once the glyph bitmaps have been loaded, we can initialize the remainder of each glyph
	if (!glyph.offset && font.bytes) {
		glyph.offset = font.view.getUint32(charcode * 4, true) + 10;
		glyph.bytes  = font.bytes;
	}

	return glyph;
};

//--APPENDFONTS-MARKER--
// fnt0-desc.js
// $ node genfont 0
(function() {
  var desc = {"name":"OCR-A","minsize":8,"maxsize":108,"minchar":32,"maxchar":126,"sizes":{"8":8,"9":9,"10":10,"12":12,"16":16,"18":18,"20":20,"24":24,"27":27,"30":30,"32":32,"36":36,"40":40,"45":45,"48":48,"50":50,"54":54,"56":56,"60":60,"63":63,"64":64,"70":70,"72":72,"80":80,"81":81,"84":84,"90":90,"96":96,"108":108},"metrics":{"m8":[5,0,0,0,0,5,6,2,1,6,5,6,0,5,3,5,6,0,5,6,5,6,1,3,6,5,6,1,3,6,5,5,1,4,5,5,6,2,1,4,5,6,2,3,6,5,6,0,3,6,5,5,0,5,5,5,5,0,5,5,5,3,2,2,3,5,3,0,5,1,5,2,2,1,1,5,6,0,5,6,5,6,1,3,6,5,6,1,3,6,5,6,1,3,6,5,6,0,4,6,5,6,1,4,6,5,6,0,4,6,5,6,1,3,6,5,6,0,4,6,5,6,1,4,6,5,6,1,3,6,5,4,2,1,3,5,5,2,2,5,5,5,0,5,5,5,4,0,5,2,5,5,0,5,5,5,6,0,5,6,5,6,1,3,6,5,6,0,5,6,5,6,1,3,6,5,6,1,4,6,5,6,0,4,6,5,6,1,4,6,5,6,1,4,6,5,6,1,3,6,5,6,1,3,6,5,6,0,5,6,5,6,1,3,6,5,6,1,4,6,5,6,1,4,6,5,6,1,3,6,5,6,1,3,6,5,6,1,3,6,5,6,1,3,6,5,6,1,3,6,5,6,1,3,6,5,6,1,3,6,5,6,1,3,6,5,6,1,3,6,5,6,1,3,6,5,6,1,3,6,5,6,0,5,6,5,6,1,3,6,5,6,0,5,6,5,6,1,4,6,5,6,0,5,6,5,6,0,4,6,5,5,0,5,5,5,0,0,5,1,5,6,1,3,3,5,4,1,3,4,5,6,1,3,6,5,4,1,4,4,5,6,1,3,6,5,4,1,3,4,5,6,1,4,6,5,4,1,3,6,5,6,1,3,6,5,6,1,4,6,5,6,1,3,8,5,6,1,4,6,5,6,1,4,6,5,4,1,3,4,5,4,1,3,4,5,4,1,3,4,5,4,1,3,6,5,4,1,3,6,5,4,1,3,4,5,4,1,3,4,5,6,0,5,6,5,4,1,3,4,5,4,1,3,4,5,4,1,4,4,5,4,0,5,4,5,4,1,3,6,5,4,0,5,4,5,6,0,5,6,5,6,2,1,6,5,6,0,5,6,5,6,1,3,3],"a8":[5,0,0,0,0,5,6,2,1,6,5,6,0,5,3,5,6,0,5,6,5,6,1,3,6,5,6,0,4,6,5,5,1,4,5,5,6,1,2,4,5,6,2,3,6,5,6,0,3,6,5,5,0,5,5,5,5,0,5,5,5,3,1,3,3,5,3,0,5,1,5,2,1,2,2,5,6,0,5,6,5,6,1,3,6,5,6,0,4,6,5,6,1,3,6,5,6,0,4,6,5,6,1,4,6,5,6,0,4,6,5,6,1,3,6,5,6,0,4,6,5,6,1,4,6,5,6,1,3,6,5,5,1,2,5,5,5,1,3,5,5,5,0,5,5,5,4,0,5,2,5,5,0,5,5,5,6,0,5,6,5,6,1,3,6,5,6,0,5,6,5,6,1,3,6,5,6,1,4,6,5,6,0,4,6,5,6,1,4,6,5,6,1,4,6,5,6,1,3,6,5,6,1,3,6,5,6,0,5,6,5,6,1,3,6,5,6,1,4,6,5,6,1,4,6,5,6,1,3,6,5,6,1,3,6,5,6,1,3,6,5,6,1,3,6,5,6,1,3,6,5,6,1,3,6,5,6,1,3,6,5,6,1,3,6,5,6,1,3,6,5,6,1,3,6,5,6,1,3,6,5,6,0,5,6,5,6,1,3,6,5,6,0,5,6,5,6,1,4,6,5,6,0,5,6,5,6,0,4,6,5,5,0,5,5,5,0,0,5,1,5,6,1,3,3,5,4,1,3,4,5,6,1,3,6,5,4,1,4,4,5,6,1,3,6,5,4,1,3,4,5,6,1,4,6,5,4,1,3,6,5,6,1,3,6,5,6,1,3,6,5,6,1,3,8,5,6,1,4,6,5,6,1,4,6,5,4,1,3,4,5,4,1,3,4,5,4,1,3,4,5,4,1,3,6,5,4,1,3,6,5,4,1,3,4,5,4,1,3,4,5,6,0,5,6,5,4,1,3,4,5,4,1,3,4,5,4,1,4,4,5,4,0,5,4,5,4,1,3,6,5,4,0,5,4,5,6,0,5,6,5,6,2,1,6,5,6,0,5,6,5,6,1,3,3],"m9":[6,0,0,0,0,6,6,2,1,6,6,6,0,5,4,6,6,1,4,6,6,6,1,4,6,6,6,1,4,6,6,6,1,4,6,6,6,2,2,4,6,6,2,3,6,6,6,1,3,6,6,6,0,5,6,6,6,0,5,5,6,3,2,2,3,6,3,0,5,1,6,3,2,2,2,6,6,0,5,6,6,6,1,4,6,6,6,1,4,6,6,6,1,4,6,6,6,0,5,6,6,6,1,4,6,6,6,1,4,6,6,6,1,4,6,6,6,0,5,6,6,6,1,4,6,6,6,1,4,6,6,6,2,2,5,6,6,2,3,6,6,6,0,5,6,6,5,0,5,3,6,6,0,5,6,6,6,1,5,6,6,6,1,4,6,6,6,0,5,6,6,6,1,4,6,6,6,1,5,6,6,6,1,4,6,6,6,1,5,6,6,6,1,5,6,6,6,1,4,6,6,6,1,4,6,6,6,0,5,6,6,6,1,4,6,6,6,1,4,6,6,6,1,5,6,6,6,1,4,6,6,6,1,4,6,6,6,1,4,6,6,6,1,4,6,6,6,1,4,6,6,6,1,4,6,6,6,1,4,6,6,6,1,4,6,6,6,1,4,6,6,6,1,4,6,6,6,1,4,6,6,6,0,5,6,6,6,1,4,6,6,6,0,5,6,6,6,1,5,6,6,6,0,5,6,6,6,1,4,6,6,5,0,5,4,6,0,0,5,1,6,6,1,4,3,6,5,1,4,5,6,6,1,4,6,6,5,1,5,5,6,6,1,4,6,6,5,1,4,5,6,6,1,4,6,6,5,1,4,7,6,6,1,4,6,6,6,1,4,6,6,6,1,3,8,6,6,1,5,6,6,6,1,4,6,6,5,1,4,5,6,5,1,4,5,6,5,1,4,5,6,5,1,4,7,6,5,1,4,7,6,5,1,4,5,6,5,1,4,5,6,7,1,5,7,6,5,1,4,5,6,5,1,4,5,6,5,1,4,5,6,5,0,5,5,6,5,1,4,7,6,5,0,5,5,6,6,0,5,6,6,6,2,1,6,6,6,0,5,6,6,6,1,4,3],"a9":[6,0,0,0,0,6,6,2,1,6,6,6,0,5,3,6,6,1,4,6,6,6,1,4,6,6,6,1,4,6,6,6,1,4,6,6,6,2,2,4,6,6,2,3,6,6,6,1,3,6,6,6,0,5,6,6,6,0,5,5,6,3,1,3,3,6,3,0,5,1,6,3,2,2,2,6,6,0,5,6,6,6,1,4,6,6,6,1,4,6,6,6,1,4,6,6,6,0,5,6,6,6,1,4,6,6,6,1,4,6,6,6,1,4,6,6,6,0,5,6,6,6,1,4,6,6,6,1,4,6,6,6,2,2,5,6,6,2,3,6,6,6,0,5,6,6,5,0,5,3,6,6,0,5,6,6,6,1,5,6,6,6,1,4,6,6,6,0,5,6,6,6,1,4,6,6,6,1,5,6,6,6,1,4,6,6,6,1,5,6,6,6,1,5,6,6,6,1,4,6,6,6,1,4,6,6,6,0,5,6,6,6,1,4,6,6,6,1,4,6,6,6,1,5,6,6,6,1,4,6,6,6,1,4,6,6,6,1,4,6,6,6,1,4,6,6,6,1,4,6,6,6,1,4,6,6,6,1,4,6,6,6,1,4,6,6,6,1,4,6,6,6,1,4,6,6,6,1,4,6,6,6,0,5,6,6,6,1,4,6,6,6,0,5,6,6,6,1,5,6,6,6,0,5,6,6,6,1,4,6,6,5,0,5,4,6,0,0,5,1,6,6,1,4,3,6,5,1,4,5,6,6,1,4,6,6,5,1,5,5,6,6,1,4,6,6,5,1,4,5,6,6,1,4,6,6,5,1,4,7,6,6,1,4,6,6,6,1,4,6,6,6,1,4,8,6,6,1,5,6,6,6,1,4,6,6,5,1,4,5,6,5,1,4,5,6,5,1,4,5,6,5,1,4,7,6,5,1,4,7,6,5,1,4,5,6,5,1,4,5,6,7,1,5,7,6,5,1,4,5,6,5,1,4,5,6,5,1,4,5,6,5,0,5,5,6,5,1,4,7,6,5,0,5,5,6,6,0,5,6,6,6,2,1,6,6,6,0,5,6,6,6,1,4,3],"m10":[7,0,0,0,0,7,7,2,1,7,7,7,0,6,4,7,7,1,5,6,7,7,1,4,7,7,7,1,5,7,7,6,1,5,6,7,7,2,2,4,7,7,2,3,7,7,7,1,3,7,7,7,1,5,7,7,6,1,5,5,7,3,2,3,3,7,4,1,5,1,7,3,2,2,2,7,7,1,5,7,7,7,1,4,7,7,7,1,4,7,7,7,1,4,7,7,7,0,5,7,7,7,1,5,7,7,7,1,4,7,7,7,1,4,7,7,7,1,4,7,7,7,1,5,7,7,7,1,4,7,7,6,2,2,5,7,7,2,3,7,7,7,1,5,7,7,5,1,5,3,7,7,1,5,7,7,7,1,5,7,7,7,1,4,7,7,7,0,6,7,7,7,1,4,7,7,7,1,5,7,7,7,1,4,7,7,7,1,5,7,7,7,1,5,7,7,7,1,4,7,7,7,1,4,7,7,7,1,5,7,7,7,1,4,7,7,7,1,5,7,7,7,1,5,7,7,7,1,4,7,7,7,1,4,7,7,7,1,4,7,7,7,1,4,7,7,7,1,4,7,7,7,1,4,7,7,7,1,4,7,7,7,1,4,7,7,7,1,4,7,7,7,1,4,7,7,7,1,4,7,7,7,1,5,7,7,7,1,4,7,7,7,0,6,7,7,7,1,5,7,7,7,1,5,7,7,7,0,5,7,7,6,0,6,5,7,0,1,5,1,7,7,1,4,4,7,5,1,4,5,7,7,1,4,7,7,5,1,5,5,7,7,1,4,7,7,5,1,4,5,7,7,1,5,7,7,5,1,4,7,7,7,1,4,7,7,7,1,4,7,7,7,1,4,9,7,7,1,5,7,7,7,1,4,7,7,5,1,4,5,7,5,1,4,5,7,5,1,4,5,7,5,1,4,7,7,5,1,4,7,7,5,1,4,5,7,5,1,4,5,7,7,1,5,7,7,5,1,4,5,7,5,1,4,5,7,5,1,5,5,7,5,1,5,5,7,5,1,4,7,7,5,1,5,5,7,7,1,5,7,7,7,3,1,7,7,7,1,5,7,7,7,1,4,3],"a10":[7,0,0,0,0,7,7,2,1,7,7,7,0,6,4,7,7,1,5,6,7,7,1,4,7,7,7,1,5,7,7,6,1,5,6,7,7,2,2,4,7,7,2,3,7,7,7,1,3,7,7,7,1,5,7,7,6,1,5,5,7,3,2,3,3,7,4,1,5,1,7,3,2,2,2,7,7,1,5,7,7,7,1,4,7,7,7,1,4,7,7,7,1,4,7,7,7,0,5,7,7,7,1,5,7,7,7,1,4,7,7,7,1,4,7,7,7,1,4,7,7,7,1,5,7,7,7,1,4,7,7,6,2,2,5,7,7,2,3,7,7,7,1,5,7,7,5,1,5,3,7,7,1,5,7,7,7,1,5,7,7,7,1,4,7,7,7,0,6,7,7,7,1,4,7,7,7,1,5,7,7,7,1,4,7,7,7,1,5,7,7,7,1,5,7,7,7,1,4,7,7,7,1,4,7,7,7,1,5,7,7,7,1,4,7,7,7,1,5,7,7,7,1,5,7,7,7,1,4,7,7,7,1,4,7,7,7,1,4,7,7,7,1,4,7,7,7,1,4,7,7,7,1,4,7,7,7,1,4,7,7,7,1,4,7,7,7,1,4,7,7,7,1,4,7,7,7,1,4,7,7,7,1,5,7,7,7,1,4,7,7,7,0,6,7,7,7,1,5,7,7,7,1,5,7,7,7,0,5,7,7,6,0,6,5,7,0,1,5,1,7,7,1,4,4,7,5,1,4,5,7,7,1,4,7,7,5,1,5,5,7,7,1,4,7,7,5,1,4,5,7,7,1,5,7,7,5,1,4,7,7,7,1,4,7,7,7,1,4,7,7,7,1,4,9,7,7,1,5,7,7,7,1,4,7,7,5,1,4,5,7,5,1,4,5,7,5,1,4,5,7,5,1,4,7,7,5,1,4,7,7,5,1,4,5,7,5,1,4,5,7,7,1,5,7,7,5,1,4,5,7,5,1,4,5,7,5,1,5,5,7,5,1,5,5,7,5,1,4,7,7,5,1,5,5,7,7,1,5,7,7,7,3,1,7,7,7,1,5,7,7,7,1,4,3],"m12":[8,0,0,0,0,8,8,3,1,8,8,8,1,6,4,8,8,0,7,7,8,8,1,5,8,8,8,1,5,8,8,7,1,6,7,8,8,3,2,5,8,8,3,3,8,8,8,2,3,8,8,8,1,5,8,8,8,0,7,7,8,4,2,3,4,8,4,1,6,1,8,3,3,2,2,8,8,1,6,8,8,8,1,5,8,8,8,1,5,8,8,8,1,5,8,8,8,0,6,8,8,8,1,6,8,8,8,0,6,8,8,8,1,5,8,8,8,0,6,8,8,8,1,5,8,8,8,1,5,8,8,7,3,2,6,8,8,2,3,8,8,8,1,6,8,8,6,1,6,3,8,8,1,6,8,8,8,0,7,8,8,8,1,5,8,8,8,1,6,8,8,8,1,5,8,8,8,1,6,8,8,8,0,6,8,8,8,1,6,8,8,8,1,6,8,8,8,1,6,8,8,8,1,5,8,8,8,0,7,8,8,8,2,4,8,8,8,1,5,8,8,8,1,6,8,8,8,1,5,8,8,8,1,5,8,8,8,1,5,8,8,8,1,5,8,8,8,1,5,8,8,8,1,5,8,8,8,1,5,8,8,8,1,5,8,8,8,1,5,8,8,8,1,5,8,8,8,1,5,8,8,8,1,6,8,8,8,1,5,8,8,8,1,6,8,8,8,1,6,8,8,8,1,6,8,8,8,1,5,8,8,7,1,6,6,8,0,1,6,1,8,8,1,5,4,8,6,1,5,6,8,8,1,5,8,8,6,1,6,6,8,8,1,5,8,8,6,1,5,6,8,8,1,6,8,8,6,1,5,8,8,8,1,5,8,8,8,2,4,8,8,8,1,5,10,8,8,1,6,8,8,8,2,5,8,8,6,1,5,6,8,6,1,5,6,8,6,1,5,6,8,6,1,5,8,8,6,1,5,8,8,6,1,5,6,8,6,1,5,6,8,8,0,7,8,8,6,1,5,6,8,6,1,5,6,8,6,1,6,6,8,6,1,6,6,8,6,1,5,8,8,6,1,6,6,8,8,0,7,8,8,8,3,1,8,8,8,0,7,8,8,8,1,5,3],"a12":[8,0,0,0,0,8,8,3,2,8,8,8,1,6,4,8,8,0,7,7,8,8,1,6,8,8,8,1,6,8,8,8,1,6,8,8,8,2,3,5,8,8,2,4,8,8,8,2,3,8,8,8,1,5,8,8,8,0,7,7,8,4,2,4,4,8,5,1,6,2,8,4,2,3,3,8,8,1,6,8,8,8,1,6,8,8,8,1,6,8,8,8,1,6,8,8,8,0,7,8,8,8,1,6,8,8,8,0,7,8,8,8,0,7,8,8,8,0,7,8,8,8,1,6,8,8,8,1,6,8,8,8,2,3,7,8,8,2,4,8,8,8,1,6,8,8,6,1,6,4,8,8,1,6,8,8,8,0,7,8,8,8,1,6,8,8,8,1,6,8,8,8,1,6,8,8,8,1,6,8,8,8,0,7,8,8,8,1,6,8,8,8,1,6,8,8,8,1,6,8,8,8,1,6,8,8,8,0,7,8,8,8,1,5,8,8,8,1,5,8,8,8,1,6,8,8,8,1,6,8,8,8,1,6,8,8,8,1,6,8,8,8,1,6,8,8,8,1,6,8,8,8,1,6,8,8,8,1,6,8,8,8,1,6,8,8,8,1,6,8,8,8,1,6,8,8,8,1,6,8,8,8,1,6,8,8,8,1,6,8,8,8,1,6,8,8,8,1,6,8,8,8,1,6,8,8,8,1,6,8,8,7,1,6,6,8,0,1,6,2,8,8,1,5,4,8,6,1,6,6,8,8,1,6,8,8,6,1,6,6,8,8,1,6,8,8,6,1,6,6,8,8,1,6,8,8,6,1,6,8,8,8,1,6,8,8,9,2,4,9,8,9,1,6,11,8,8,1,6,8,8,8,2,5,8,8,6,1,6,6,8,6,1,6,6,8,6,1,6,6,8,7,1,6,9,8,6,1,6,8,8,6,1,6,6,8,6,1,6,6,8,8,0,7,8,8,6,1,6,6,8,6,1,6,6,8,6,1,6,6,8,6,1,6,6,8,6,1,6,8,8,6,1,6,6,8,8,0,7,8,8,8,3,2,8,8,8,0,7,8,8,8,1,6,4],"m16":[10,0,0,0,0,10,11,4,2,11,10,11,1,8,5,10,11,1,8,10,10,11,2,6,11,10,11,2,7,11,10,10,2,7,10,10,11,4,3,7,10,11,4,4,11,10,11,2,4,11,10,10,1,7,9,10,9,1,7,7,10,5,3,4,5,10,6,1,8,2,10,5,4,3,3,10,11,1,8,11,10,11,2,6,11,10,11,2,6,11,10,11,2,6,11,10,11,1,7,11,10,11,2,6,11,10,11,1,7,11,10,11,2,6,11,10,11,1,7,11,10,11,2,6,11,10,11,2,7,11,10,10,4,3,8,10,10,3,4,10,10,10,1,8,9,10,8,1,8,4,10,10,1,8,9,10,11,1,8,11,10,11,2,6,11,10,11,1,8,11,10,11,2,6,11,10,11,2,7,11,10,11,1,7,11,10,11,2,7,11,10,11,2,7,11,10,11,2,6,11,10,11,2,6,11,10,11,1,7,11,10,11,2,6,11,10,11,2,7,11,10,11,2,7,11,10,11,2,6,11,10,11,2,6,11,10,11,2,6,11,10,11,2,6,11,10,11,2,6,11,10,11,2,6,11,10,11,2,6,11,10,11,2,6,11,10,11,2,6,11,10,11,2,6,11,10,11,2,6,11,10,11,1,8,11,10,11,2,6,11,10,11,1,8,11,10,11,2,7,11,10,11,1,8,11,10,11,1,7,11,10,9,1,8,8,10,0,1,8,1,10,11,2,6,6,10,8,2,6,8,10,11,2,6,11,10,8,2,7,8,10,11,2,6,11,10,8,2,6,8,10,11,1,7,11,10,8,2,6,11,10,11,2,6,11,10,11,2,7,11,10,11,2,6,14,10,11,2,7,11,10,11,2,7,11,10,8,2,6,8,10,8,2,6,8,10,8,2,6,8,10,8,2,6,11,10,8,2,6,11,10,8,2,6,8,10,8,2,6,8,10,11,1,8,11,10,8,2,6,8,10,8,2,6,8,10,8,2,6,8,10,8,1,8,8,10,8,2,6,11,10,8,1,8,8,10,11,1,7,11,10,11,4,1,11,10,11,1,7,11,10,11,2,6,5],"a16":[10,0,0,0,0,10,11,4,3,11,10,11,1,8,6,10,11,1,8,10,10,11,1,8,11,10,11,2,7,11,10,10,1,8,10,10,11,4,3,7,10,11,3,5,11,10,11,2,5,11,10,10,1,7,9,10,9,1,7,8,10,5,3,4,5,10,7,1,8,3,10,5,4,3,3,10,11,1,8,11,10,11,1,8,11,10,11,2,7,11,10,11,1,8,11,10,11,1,8,11,10,11,2,6,11,10,11,1,8,11,10,11,1,8,11,10,11,1,8,11,10,11,1,8,11,10,11,1,8,11,10,10,4,3,8,10,10,3,4,10,10,10,1,8,9,10,8,1,8,5,10,10,1,8,9,10,11,1,8,11,10,11,1,8,11,10,11,1,8,11,10,11,1,8,11,10,11,1,8,11,10,11,1,8,11,10,11,1,8,11,10,11,1,8,11,10,11,1,8,11,10,11,1,8,11,10,11,1,7,11,10,11,2,6,11,10,11,1,8,11,10,11,1,8,11,10,11,1,8,11,10,11,1,8,11,10,11,1,8,11,10,11,1,8,11,10,11,1,8,11,10,11,1,8,11,10,11,1,8,11,10,11,1,8,11,10,11,1,8,11,10,11,1,8,11,10,11,1,8,11,10,11,1,8,11,10,11,1,8,11,10,11,1,8,11,10,11,1,8,11,10,11,1,8,11,10,11,1,8,11,10,9,1,8,8,10,0,1,8,2,10,11,2,6,6,10,8,1,8,8,10,11,1,8,11,10,8,1,8,8,10,11,1,8,11,10,8,1,8,8,10,11,1,7,11,10,8,1,8,11,10,11,1,8,11,10,11,2,7,11,10,11,2,7,14,10,11,1,8,11,10,11,2,7,11,10,8,1,8,8,10,8,1,8,8,10,8,1,8,8,10,9,1,8,12,10,8,1,8,11,10,8,1,8,8,10,8,1,8,8,10,11,1,8,11,10,8,1,8,8,10,8,1,8,8,10,8,1,8,8,10,8,1,8,8,10,8,1,8,11,10,8,1,8,8,10,11,1,7,11,10,11,4,2,11,10,11,1,7,11,10,11,1,8,5],"m18":[12,0,0,0,0,12,13,5,2,13,12,13,1,9,7,12,13,1,9,11,12,13,2,8,13,12,13,2,8,13,12,12,2,8,12,12,13,4,3,8,12,13,4,5,13,12,13,3,5,13,12,12,2,8,11,12,11,1,10,10,12,6,3,5,6,12,7,1,9,2,12,5,4,3,3,12,13,1,9,13,12,13,2,8,13,12,13,2,8,13,12,13,2,8,13,12,13,1,9,13,12,13,2,8,13,12,13,1,9,13,12,13,2,8,13,12,13,1,9,13,12,13,2,8,13,12,13,2,8,13,12,11,4,3,9,12,11,3,5,11,12,12,1,9,11,12,9,1,9,5,12,12,1,9,11,12,13,1,9,13,12,13,2,8,13,12,13,1,9,13,12,13,2,8,13,12,13,2,9,13,12,13,1,9,13,12,13,2,9,13,12,13,2,9,13,12,13,2,8,13,12,13,2,8,13,12,13,1,10,13,12,13,2,7,13,12,13,2,8,13,12,13,2,9,13,12,13,2,8,13,12,13,2,8,13,12,13,2,8,13,12,13,2,8,13,12,13,2,8,13,12,13,2,8,13,12,13,2,8,13,12,13,2,8,13,12,13,2,8,13,12,13,2,8,13,12,13,2,8,13,12,13,1,9,13,12,13,2,8,13,12,13,1,9,13,12,13,2,8,13,12,13,1,9,13,12,13,2,8,13,12,10,1,9,8,12,0,1,9,2,12,13,2,7,6,12,9,2,8,9,12,13,2,8,13,12,9,2,9,9,12,13,2,8,13,12,9,2,8,9,12,13,2,8,13,12,9,2,8,12,12,13,2,8,13,12,13,3,7,13,12,13,2,7,16,12,13,2,9,13,12,13,3,7,13,12,9,2,8,9,12,9,2,8,9,12,9,2,8,9,12,10,2,8,13,12,9,2,8,12,12,9,2,8,9,12,9,2,8,9,12,12,1,9,12,12,9,2,8,9,12,9,2,8,9,12,9,2,8,9,12,9,1,9,9,12,9,2,8,12,12,9,1,9,9,12,13,1,10,13,12,13,5,2,13,12,13,1,10,13,12,13,2,8,6],"a18":[12,0,0,0,0,12,13,4,3,13,12,13,1,9,6,12,13,1,9,11,12,13,2,8,13,12,13,2,8,13,12,12,2,8,12,12,13,4,3,8,12,13,4,5,13,12,13,3,5,13,12,12,2,8,11,12,11,1,9,10,12,6,3,5,6,12,7,1,9,3,12,5,4,3,3,12,13,1,9,13,12,13,2,8,13,12,13,2,8,13,12,13,2,8,13,12,13,1,9,13,12,13,2,8,13,12,13,1,9,13,12,13,1,9,13,12,13,1,9,13,12,13,2,8,13,12,13,2,8,13,12,11,4,3,9,12,11,3,5,11,12,12,1,9,11,12,9,1,9,5,12,12,1,9,11,12,13,1,9,13,12,13,2,8,13,12,13,1,9,13,12,13,2,8,13,12,13,2,9,13,12,13,1,9,13,12,13,2,9,13,12,13,2,9,13,12,13,2,8,13,12,13,2,8,13,12,13,1,9,13,12,13,2,7,13,12,13,2,8,13,12,13,2,9,13,12,13,2,8,13,12,13,2,8,13,12,13,2,8,13,12,13,2,8,13,12,13,2,8,13,12,13,2,8,13,12,13,2,8,13,12,13,2,8,13,12,13,2,8,13,12,13,2,8,13,12,13,2,8,13,12,13,1,9,13,12,13,2,8,13,12,13,1,9,13,12,13,2,8,13,12,13,1,9,13,12,13,2,8,13,12,10,1,9,8,12,0,1,9,2,12,13,2,7,6,12,9,2,8,9,12,13,2,8,13,12,9,2,9,9,12,13,2,8,13,12,9,2,8,9,12,13,2,8,13,12,9,2,8,12,12,13,2,8,13,12,14,3,7,14,12,14,2,7,17,12,13,2,9,13,12,13,3,7,13,12,9,2,8,9,12,9,2,8,9,12,9,2,8,9,12,10,2,8,13,12,9,2,8,12,12,9,2,8,9,12,9,2,8,9,12,12,1,9,12,12,9,2,8,9,12,9,2,8,9,12,9,2,8,9,12,9,1,9,9,12,9,2,8,12,12,9,1,9,9,12,13,1,9,13,12,13,5,2,13,12,13,1,9,13,12,13,2,8,6],"m20":[13,0,0,0,0,13,14,5,3,14,13,14,1,10,7,13,14,2,9,13,13,14,2,9,14,13,14,2,9,14,13,13,2,9,13,13,14,5,4,8,13,14,5,5,14,13,14,3,5,14,13,13,1,10,12,13,12,1,10,10,13,7,4,5,7,13,8,2,9,3,13,5,5,4,3,13,14,2,9,14,13,14,2,9,14,13,14,2,9,14,13,14,2,9,14,13,14,2,9,14,13,14,2,9,14,13,14,2,9,14,13,14,2,9,14,13,14,1,10,14,13,14,2,9,14,13,14,2,9,14,13,11,5,4,9,13,12,4,6,12,13,13,2,9,12,13,10,2,9,5,13,13,2,9,12,13,14,2,10,14,13,14,2,9,14,13,14,1,10,14,13,14,2,9,14,13,14,2,9,14,13,14,2,9,14,13,14,2,9,14,13,14,2,9,14,13,14,2,8,14,13,14,2,9,14,13,14,1,10,14,13,14,3,7,14,13,14,2,9,14,13,14,2,9,14,13,14,2,9,14,13,14,2,9,14,13,14,2,9,14,13,14,2,9,14,13,14,2,9,14,13,14,2,9,14,13,14,2,9,14,13,14,2,9,14,13,14,2,9,14,13,14,2,9,14,13,14,2,9,14,13,14,2,9,14,13,14,2,9,14,13,14,1,10,14,13,14,2,10,14,13,14,2,9,14,13,14,1,10,14,13,11,1,10,9,13,0,2,9,2,13,14,2,8,7,13,10,2,9,10,13,14,2,9,14,13,10,2,9,10,13,14,2,9,14,13,10,2,9,10,13,14,2,9,14,13,10,2,9,14,13,14,2,9,14,13,15,3,7,15,13,15,3,7,19,13,14,2,9,14,13,14,3,7,14,13,10,2,9,10,13,10,2,9,10,13,10,2,9,10,13,11,2,9,15,13,10,2,9,14,13,10,2,9,10,13,10,2,9,10,13,14,2,10,14,13,10,2,9,10,13,10,2,9,10,13,10,2,9,10,13,10,2,9,10,13,10,2,9,14,13,10,2,9,10,13,14,1,10,14,13,14,5,2,14,13,14,1,10,14,13,14,2,9,6],"a20":[13,0,0,0,0,13,14,5,3,14,13,14,1,10,7,13,14,2,9,12,13,14,2,9,14,13,14,2,9,14,13,13,2,9,13,13,14,5,4,8,13,14,5,5,14,13,14,3,5,14,13,13,1,10,12,13,12,1,10,10,13,7,4,5,7,13,8,2,9,3,13,5,5,4,3,13,14,2,9,14,13,14,2,9,14,13,14,2,9,14,13,14,2,9,14,13,14,2,9,14,13,14,2,9,14,13,14,2,9,14,13,14,2,9,14,13,14,1,10,14,13,14,2,9,14,13,14,2,9,14,13,11,5,4,9,13,12,4,6,12,13,13,2,9,12,13,10,2,9,5,13,13,2,9,12,13,14,2,10,14,13,14,2,9,14,13,14,1,10,14,13,14,2,9,14,13,14,2,9,14,13,14,2,9,14,13,14,2,9,14,13,14,2,9,14,13,14,2,8,14,13,14,2,9,14,13,14,1,10,14,13,14,3,7,14,13,14,2,9,14,13,14,2,9,14,13,14,2,9,14,13,14,2,9,14,13,14,2,9,14,13,14,2,9,14,13,14,2,9,14,13,14,2,9,14,13,14,2,9,14,13,14,2,9,14,13,14,2,9,14,13,14,2,9,14,13,14,2,9,14,13,14,2,9,14,13,14,2,9,14,13,14,1,10,14,13,14,2,10,14,13,14,2,9,14,13,14,1,10,14,13,11,1,10,9,13,0,2,9,2,13,14,2,8,7,13,10,2,9,10,13,14,2,9,14,13,10,2,9,10,13,14,2,9,14,13,10,2,9,10,13,14,2,9,14,13,10,2,9,14,13,14,2,9,14,13,15,3,7,15,13,15,3,7,19,13,14,2,9,14,13,14,3,7,14,13,10,2,9,10,13,10,2,9,10,13,10,2,9,10,13,11,2,9,15,13,10,2,9,14,13,10,2,9,10,13,10,2,9,10,13,14,2,10,14,13,10,2,9,10,13,10,2,9,10,13,10,2,9,10,13,10,2,9,10,13,10,2,9,14,13,10,2,9,10,13,14,1,10,14,13,14,5,2,14,13,14,1,10,14,13,14,2,9,6],"m24":[16,0,0,0,0,16,17,6,3,17,16,17,2,11,8,16,17,2,12,15,16,17,2,11,17,16,17,2,11,17,16,15,2,12,15,16,17,6,4,10,16,17,6,6,17,16,17,4,6,17,16,15,2,12,14,16,14,2,12,12,16,8,4,6,8,16,9,2,11,3,16,7,6,4,4,16,17,2,11,17,16,17,2,11,17,16,17,2,11,17,16,17,2,11,17,16,17,2,11,17,16,17,3,10,17,16,17,2,11,17,16,17,2,11,17,16,17,2,11,17,16,17,2,11,17,16,17,2,11,17,16,14,6,4,11,16,15,4,6,15,16,15,2,11,14,16,12,2,11,6,16,15,2,11,14,16,17,2,12,17,16,17,2,11,17,16,17,2,11,17,16,17,2,11,17,16,17,2,11,17,16,17,2,11,17,16,17,2,11,17,16,17,2,11,17,16,17,2,11,17,16,17,2,11,17,16,17,2,12,17,16,17,3,9,17,16,17,2,11,17,16,17,2,11,17,16,17,2,11,17,16,17,2,11,17,16,17,2,11,17,16,17,2,11,17,16,17,2,11,17,16,17,2,11,17,16,17,2,11,17,16,17,2,11,17,16,17,2,11,17,16,17,2,11,17,16,17,2,11,17,16,17,2,11,17,16,17,2,11,17,16,17,2,11,17,16,17,2,12,17,16,17,2,11,17,16,17,2,11,17,16,14,2,11,12,16,0,2,11,2,16,17,3,9,8,16,12,2,11,12,16,17,2,11,17,16,12,2,11,12,16,17,2,11,17,16,12,2,11,12,16,17,2,11,17,16,12,2,11,17,16,17,2,11,17,16,18,3,9,18,16,18,3,9,23,16,17,2,11,17,16,17,3,9,17,16,12,2,11,12,16,12,2,11,12,16,12,2,11,12,16,13,2,11,18,16,12,2,11,17,16,12,2,11,12,16,12,3,10,12,16,16,2,12,16,16,12,2,11,12,16,12,2,11,12,16,12,2,11,12,16,12,2,11,12,16,12,2,11,17,16,12,2,11,12,16,17,2,12,17,16,17,7,2,17,16,17,2,12,17,16,17,2,11,8],"a24":[16,0,0,0,0,16,17,6,3,17,16,17,2,11,8,16,17,2,12,15,16,17,2,11,17,16,17,2,11,17,16,15,2,12,15,16,17,5,5,10,16,17,5,7,17,16,17,4,6,17,16,15,2,12,14,16,14,2,12,12,16,8,4,7,8,16,9,2,11,3,16,7,6,4,4,16,17,2,11,17,16,17,2,11,17,16,17,2,11,17,16,17,2,11,17,16,17,2,11,17,16,17,2,11,17,16,17,2,11,17,16,17,1,12,17,16,17,2,11,17,16,17,2,12,17,16,17,2,12,17,16,14,6,4,11,16,15,4,7,15,16,15,2,11,14,16,12,2,11,7,16,15,2,11,14,16,17,2,12,17,16,17,2,11,17,16,17,2,11,17,16,17,2,11,17,16,17,2,11,17,16,17,2,11,17,16,17,2,11,17,16,17,2,11,17,16,17,2,12,17,16,17,2,11,17,16,17,2,12,17,16,17,3,9,17,16,17,2,11,17,16,17,2,11,17,16,17,2,11,17,16,17,2,11,17,16,17,2,11,17,16,17,2,11,17,16,17,2,11,17,16,17,2,11,17,16,17,2,11,17,16,17,2,11,17,16,17,2,11,17,16,17,2,11,17,16,17,2,11,17,16,17,2,11,17,16,17,2,11,17,16,17,2,11,17,16,17,2,12,17,16,17,2,11,17,16,17,2,11,17,16,14,2,11,12,16,0,2,11,3,16,17,3,9,8,16,12,2,12,12,16,17,2,11,17,16,12,2,11,12,16,17,2,11,17,16,12,2,11,12,16,17,2,11,17,16,12,2,11,17,16,17,2,11,17,16,18,3,9,18,16,18,3,10,23,16,17,2,11,17,16,17,3,9,17,16,12,2,12,12,16,12,2,11,12,16,12,2,11,12,16,13,2,11,18,16,12,2,11,17,16,12,2,11,12,16,12,2,11,12,16,16,2,12,16,16,12,2,11,12,16,12,2,11,12,16,12,2,12,12,16,12,2,11,12,16,12,2,11,17,16,12,2,11,12,16,17,2,12,17,16,17,6,3,17,16,17,2,12,17,16,17,2,11,8],"m27":[18,0,0,0,0,18,19,7,4,19,18,19,2,13,9,18,19,2,13,17,18,19,3,11,19,18,19,3,12,19,18,17,3,12,17,18,19,6,5,11,18,19,6,6,19,18,19,5,6,19,18,17,3,12,15,18,15,3,12,12,18,8,5,7,8,18,11,2,13,4,18,8,6,5,5,18,19,2,13,19,18,19,3,11,19,18,19,3,11,19,18,19,3,11,19,18,19,2,12,19,18,19,3,12,19,18,19,2,12,19,18,19,3,11,19,18,19,3,11,19,18,19,3,11,19,18,19,3,12,19,18,16,6,5,13,18,17,5,7,17,18,17,2,13,15,18,14,2,13,8,18,17,2,13,15,18,19,2,13,19,18,19,3,11,19,18,19,2,13,19,18,19,3,11,19,18,19,3,12,19,18,19,2,12,19,18,19,3,12,19,18,19,3,12,19,18,19,3,12,19,18,19,3,11,19,18,19,3,12,19,18,19,4,9,19,18,19,3,12,19,18,19,3,12,19,18,19,3,11,19,18,19,3,11,19,18,19,3,11,19,18,19,3,11,19,18,19,3,11,19,18,19,3,11,19,18,19,3,11,19,18,19,3,11,19,18,19,3,11,19,18,19,3,11,19,18,19,3,11,19,18,19,2,13,19,18,19,3,11,19,18,19,2,13,19,18,19,3,12,19,18,19,2,13,19,18,19,2,12,19,18,15,2,13,12,18,0,2,13,2,18,19,3,11,9,18,14,3,11,14,18,19,3,11,19,18,14,3,12,14,18,19,3,11,19,18,14,3,11,14,18,19,3,12,19,18,14,3,11,19,18,19,3,11,19,18,20,4,10,20,18,20,3,10,25,18,19,3,12,19,18,19,4,10,19,18,14,3,11,14,18,14,3,11,14,18,14,3,11,14,18,14,3,11,19,18,14,3,11,19,18,14,3,11,14,18,14,3,11,14,18,19,2,13,19,18,14,3,11,14,18,14,3,11,14,18,14,3,12,14,18,14,2,13,14,18,14,3,11,19,18,14,2,13,14,18,19,3,12,19,18,19,8,2,19,18,19,3,12,19,18,19,3,11,9],"a27":[18,0,0,0,0,18,19,7,4,19,18,19,2,13,9,18,19,2,13,17,18,19,2,13,19,18,19,3,12,19,18,17,2,13,17,18,19,6,5,11,18,19,6,6,19,18,19,5,6,19,18,17,3,12,15,18,15,3,12,12,18,9,5,7,9,18,11,2,13,4,18,8,6,5,5,18,19,2,13,19,18,19,2,13,19,18,19,3,12,19,18,19,2,13,19,18,19,2,13,19,18,19,3,12,19,18,19,2,13,19,18,19,3,12,19,18,19,3,12,19,18,19,2,13,19,18,19,2,13,19,18,16,6,5,13,18,17,5,7,17,18,17,2,13,15,18,14,2,13,8,18,17,2,13,15,18,19,2,13,19,18,19,2,13,19,18,19,2,13,19,18,19,2,13,19,18,19,2,13,19,18,19,2,13,19,18,19,2,13,19,18,19,2,13,19,18,19,2,13,19,18,19,2,13,19,18,19,3,12,19,18,19,4,10,19,18,19,2,13,19,18,19,2,13,19,18,19,2,13,19,18,19,2,13,19,18,19,2,13,19,18,19,2,13,19,18,19,2,13,19,18,19,2,13,19,18,19,2,13,19,18,19,2,13,19,18,19,2,13,19,18,19,2,13,19,18,19,2,13,19,18,19,2,13,19,18,19,2,13,19,18,19,2,13,19,18,19,2,13,19,18,19,2,13,19,18,19,2,13,19,18,15,2,13,12,18,0,2,13,3,18,19,3,11,9,18,14,2,13,14,18,19,2,13,19,18,14,2,13,14,18,19,2,13,19,18,14,2,13,14,18,19,3,12,19,18,14,2,13,19,18,19,2,13,19,18,20,4,10,20,18,20,3,10,25,18,19,2,13,19,18,19,4,10,19,18,14,2,13,14,18,14,2,13,14,18,14,2,13,14,18,14,2,13,19,18,14,2,13,19,18,14,2,13,14,18,14,3,12,14,18,19,2,13,19,18,14,2,13,14,18,14,2,13,14,18,14,2,13,14,18,14,2,13,14,18,14,2,13,19,18,14,2,13,14,18,19,3,12,19,18,19,7,3,19,18,19,3,12,19,18,19,2,13,9],"m30":[20,0,0,0,0,20,21,8,4,21,20,21,2,15,10,20,21,3,14,18,20,21,3,13,21,20,21,3,13,21,20,19,3,14,19,20,21,7,5,12,20,21,7,7,21,20,21,5,7,21,20,19,2,15,17,20,18,2,15,15,20,10,6,8,10,20,12,2,15,4,20,8,7,5,5,20,21,2,15,21,20,21,3,13,21,20,21,3,13,21,20,21,3,13,21,20,21,2,14,21,20,21,4,12,21,20,21,3,13,21,20,21,3,13,21,20,21,2,14,21,20,21,3,14,21,20,21,3,13,21,20,18,7,5,15,20,19,6,8,19,20,19,2,15,17,20,15,2,15,8,20,19,2,15,17,20,21,3,14,21,20,21,3,13,21,20,21,2,15,21,20,21,3,13,21,20,21,3,14,21,20,21,3,13,21,20,21,3,14,21,20,21,3,14,21,20,21,3,14,21,20,21,3,13,21,20,21,2,15,21,20,21,4,11,21,20,21,3,14,21,20,21,3,14,21,20,21,3,13,21,20,21,3,13,21,20,21,3,13,21,20,21,3,13,21,20,21,3,13,21,20,21,3,13,21,20,21,3,13,21,20,21,3,13,21,20,21,3,13,21,20,21,3,13,21,20,21,3,13,21,20,21,2,15,21,20,21,3,13,21,20,21,2,15,21,20,21,3,14,21,20,21,2,15,21,20,21,2,14,21,20,17,2,15,14,20,0,2,15,3,20,21,4,11,10,20,15,3,13,15,20,21,3,13,21,20,15,3,14,15,20,21,3,13,21,20,15,3,13,15,20,21,3,13,21,20,15,3,13,21,20,21,3,13,21,20,22,4,11,22,20,22,4,11,28,20,21,3,14,21,20,21,4,11,21,20,15,3,13,15,20,15,3,13,15,20,15,3,13,15,20,16,3,13,22,20,15,3,13,21,20,15,3,13,15,20,15,3,13,15,20,20,3,14,20,20,15,3,13,15,20,15,3,13,15,20,15,3,14,15,20,15,2,15,15,20,15,3,13,21,20,15,2,15,15,20,21,2,15,21,20,21,8,3,21,20,21,2,15,21,20,21,3,13,10],"a30":[20,0,0,0,0,20,21,8,4,21,20,21,2,15,10,20,21,3,14,18,20,21,3,13,21,20,21,3,13,21,20,19,3,14,19,20,21,7,5,12,20,21,7,7,21,20,21,5,7,21,20,19,2,15,17,20,18,2,15,15,20,10,6,8,10,20,12,2,15,4,20,8,7,5,5,20,21,2,15,21,20,21,3,13,21,20,21,3,13,21,20,21,3,13,21,20,21,2,14,21,20,21,4,12,21,20,21,3,13,21,20,21,3,13,21,20,21,2,14,21,20,21,3,14,21,20,21,3,13,21,20,18,7,5,15,20,19,6,8,19,20,19,2,15,17,20,15,2,15,8,20,19,2,15,17,20,21,3,14,21,20,21,3,13,21,20,21,2,15,21,20,21,3,13,21,20,21,3,14,21,20,21,3,13,21,20,21,3,14,21,20,21,3,14,21,20,21,3,14,21,20,21,3,13,21,20,21,2,15,21,20,21,4,11,21,20,21,3,14,21,20,21,3,14,21,20,21,3,13,21,20,21,3,13,21,20,21,3,13,21,20,21,3,13,21,20,21,3,13,21,20,21,3,13,21,20,21,3,13,21,20,21,3,13,21,20,21,3,13,21,20,21,3,13,21,20,21,3,13,21,20,21,2,15,21,20,21,3,13,21,20,21,2,15,21,20,21,3,14,21,20,21,2,15,21,20,21,2,14,21,20,17,2,15,14,20,0,2,15,3,20,21,4,11,10,20,15,3,13,15,20,21,3,13,21,20,15,3,14,15,20,21,3,13,21,20,15,3,13,15,20,21,3,13,21,20,15,3,13,21,20,21,3,13,21,20,22,4,11,22,20,22,4,11,28,20,21,3,14,21,20,21,4,11,21,20,15,3,14,15,20,15,3,13,15,20,15,3,13,15,20,16,3,13,22,20,15,3,13,21,20,15,3,13,15,20,15,3,13,15,20,20,3,14,20,20,15,3,13,15,20,15,3,13,15,20,15,3,14,15,20,15,2,15,15,20,15,3,13,21,20,15,2,15,15,20,21,2,15,21,20,21,8,3,21,20,21,2,15,21,20,21,3,13,10],"m32":[21,0,0,0,0,21,22,8,4,22,21,22,3,15,11,21,22,3,15,18,21,22,3,14,22,21,22,3,15,22,21,20,3,15,20,21,22,7,6,13,21,22,7,8,22,21,22,5,8,22,21,21,3,15,19,21,19,3,15,15,21,10,6,8,10,21,13,3,15,4,21,9,7,6,5,21,22,3,15,22,21,22,3,14,22,21,22,3,14,22,21,22,3,14,22,21,22,2,15,22,21,22,4,13,22,21,22,3,14,22,21,22,3,14,22,21,22,3,14,22,21,22,3,15,22,21,22,3,14,22,21,18,7,6,14,21,20,6,9,20,21,21,3,15,19,21,16,3,15,9,21,21,3,15,19,21,22,3,15,22,21,22,3,14,22,21,22,3,15,22,21,22,3,14,22,21,22,3,15,22,21,22,3,14,22,21,22,3,15,22,21,22,3,15,22,21,22,3,14,22,21,22,3,14,22,21,22,3,15,22,21,22,5,11,22,21,22,3,15,22,21,22,3,15,22,21,22,3,14,22,21,22,3,14,22,21,22,3,14,22,21,22,3,14,22,21,22,3,14,22,21,22,3,14,22,21,22,3,14,22,21,22,3,14,22,21,22,3,14,22,21,22,3,14,22,21,22,3,14,22,21,22,3,15,22,21,22,3,14,22,21,22,3,15,22,21,22,3,15,22,21,22,3,15,22,21,22,3,14,22,21,18,3,15,15,21,0,3,15,3,21,22,4,12,11,21,16,3,14,16,21,22,3,14,22,21,16,3,15,16,21,22,3,14,22,21,16,3,14,16,21,22,4,13,22,21,16,3,14,22,21,22,3,14,22,21,24,5,12,24,21,24,4,11,30,21,22,3,15,22,21,22,5,12,22,21,16,3,14,16,21,16,3,14,16,21,16,3,14,16,21,17,3,14,23,21,16,3,14,22,21,16,3,14,16,21,16,3,14,16,21,21,3,15,21,21,16,3,14,16,21,16,3,14,16,21,16,3,15,16,21,16,3,15,16,21,16,3,14,22,21,16,3,15,16,21,22,3,15,22,21,22,9,3,22,21,22,3,15,22,21,22,3,14,10],"a32":[21,0,0,0,0,21,22,8,4,22,21,22,3,15,11,21,22,3,15,18,21,22,3,14,22,21,22,3,15,22,21,20,3,15,20,21,22,7,6,13,21,22,7,8,22,21,22,5,8,22,21,21,3,15,19,21,19,3,15,15,21,10,6,8,10,21,13,3,15,4,21,9,7,6,5,21,22,3,15,22,21,22,3,14,22,21,22,3,14,22,21,22,3,14,22,21,22,2,15,22,21,22,4,13,22,21,22,3,14,22,21,22,3,14,22,21,22,3,14,22,21,22,3,15,22,21,22,3,14,22,21,18,7,6,14,21,20,6,9,20,21,21,3,15,19,21,16,3,15,9,21,21,3,15,19,21,22,3,15,22,21,22,3,14,22,21,22,3,15,22,21,22,3,14,22,21,22,3,15,22,21,22,3,14,22,21,22,3,15,22,21,22,3,15,22,21,22,3,14,22,21,22,3,14,22,21,22,3,15,22,21,22,5,11,22,21,22,3,15,22,21,22,3,15,22,21,22,3,14,22,21,22,3,14,22,21,22,3,14,22,21,22,3,14,22,21,22,3,14,22,21,22,3,14,22,21,22,3,14,22,21,22,3,14,22,21,22,3,14,22,21,22,3,14,22,21,22,3,14,22,21,22,3,15,22,21,22,3,14,22,21,22,3,15,22,21,22,3,15,22,21,22,3,15,22,21,22,3,14,22,21,18,3,15,15,21,0,3,15,3,21,22,4,12,11,21,16,3,14,16,21,22,3,14,22,21,16,3,15,16,21,22,3,14,22,21,16,3,14,16,21,22,4,13,22,21,16,3,14,22,21,22,3,14,22,21,24,5,12,24,21,24,4,11,30,21,22,3,15,22,21,22,5,12,22,21,16,3,14,16,21,16,3,14,16,21,16,3,14,16,21,17,3,14,23,21,16,3,14,22,21,16,3,14,16,21,16,3,14,16,21,21,3,15,21,21,16,3,14,16,21,16,3,14,16,21,16,3,15,16,21,16,3,15,16,21,16,3,14,22,21,16,3,15,16,21,22,3,15,22,21,22,9,3,22,21,22,3,15,22,21,22,3,14,10],"m36":[23,0,0,0,0,23,25,9,5,25,23,25,3,17,12,23,25,3,17,21,23,25,4,15,25,23,25,4,15,25,23,22,4,16,22,23,25,8,6,15,23,25,8,8,25,23,25,6,8,25,23,23,3,17,21,23,21,3,17,17,23,11,7,9,11,23,15,3,17,5,23,10,8,6,6,23,25,3,17,25,23,25,4,15,25,23,25,4,15,25,23,25,4,15,25,23,25,3,16,25,23,25,4,15,25,23,25,3,16,25,23,25,4,15,25,23,25,3,16,25,23,25,4,15,25,23,25,4,16,25,23,21,8,6,17,23,22,7,9,22,23,23,3,17,21,23,18,3,17,10,23,23,3,17,21,23,25,3,17,25,23,25,4,15,25,23,25,3,17,25,23,25,4,15,25,23,25,4,16,25,23,25,3,16,25,23,25,4,16,25,23,25,4,16,25,23,25,4,15,25,23,25,4,15,25,23,25,3,17,25,23,25,5,13,25,23,25,4,16,25,23,25,4,16,25,23,25,4,15,25,23,25,4,15,25,23,25,4,15,25,23,25,4,15,25,23,25,4,15,25,23,25,4,15,25,23,25,4,15,25,23,25,4,15,25,23,25,4,15,25,23,25,4,15,25,23,25,4,15,25,23,25,3,17,25,23,25,4,15,25,23,25,3,17,25,23,25,4,15,25,23,25,3,17,25,23,25,3,16,25,23,20,3,17,16,23,0,3,17,3,23,25,5,13,12,23,18,4,15,18,23,25,4,15,25,23,18,4,16,18,23,25,4,15,25,23,18,4,15,18,23,25,3,15,25,23,18,4,15,25,23,25,4,15,25,23,27,5,13,27,23,27,4,14,34,23,25,4,16,25,23,25,5,13,25,23,18,3,16,18,23,18,4,15,18,23,18,4,15,18,23,19,4,15,26,23,18,4,15,25,23,18,4,15,18,23,18,4,15,18,23,24,3,17,24,23,18,4,15,18,23,18,4,15,18,23,18,3,17,18,23,18,3,17,18,23,18,4,15,25,23,18,3,17,18,23,25,3,17,25,23,25,10,3,25,23,25,3,17,25,23,25,4,15,11],"a36":[23,0,0,0,0,23,25,9,5,25,23,25,3,17,12,23,25,3,17,21,23,25,4,15,25,23,25,4,15,25,23,22,4,16,22,23,25,8,6,15,23,25,8,8,25,23,25,6,8,25,23,23,3,17,21,23,21,3,17,17,23,11,7,9,11,23,15,3,17,5,23,10,8,6,6,23,25,3,17,25,23,25,4,15,25,23,25,4,15,25,23,25,4,15,25,23,25,3,16,25,23,25,4,15,25,23,25,3,16,25,23,25,4,15,25,23,25,3,16,25,23,25,4,15,25,23,25,4,16,25,23,21,8,6,17,23,22,7,9,22,23,23,3,17,21,23,18,3,17,10,23,23,3,17,21,23,25,3,17,25,23,25,4,15,25,23,25,3,17,25,23,25,4,15,25,23,25,4,16,25,23,25,3,16,25,23,25,4,16,25,23,25,4,16,25,23,25,4,15,25,23,25,4,15,25,23,25,3,17,25,23,25,5,13,25,23,25,4,16,25,23,25,4,16,25,23,25,4,15,25,23,25,4,15,25,23,25,4,15,25,23,25,4,15,25,23,25,4,15,25,23,25,4,15,25,23,25,4,15,25,23,25,4,15,25,23,25,4,15,25,23,25,4,15,25,23,25,4,15,25,23,25,3,17,25,23,25,4,15,25,23,25,3,17,25,23,25,4,15,25,23,25,3,17,25,23,25,3,16,25,23,20,3,17,16,23,0,3,17,3,23,25,5,13,12,23,18,4,15,18,23,25,4,15,25,23,18,4,16,18,23,25,4,15,25,23,18,4,15,18,23,25,3,15,25,23,18,4,15,25,23,25,4,15,25,23,27,5,13,27,23,27,4,14,34,23,25,4,16,25,23,25,5,13,25,23,18,3,16,18,23,18,4,15,18,23,18,4,15,18,23,19,4,15,26,23,18,4,15,25,23,18,4,15,18,23,18,4,15,18,23,24,3,17,24,23,18,4,15,18,23,18,4,15,18,23,18,3,17,18,23,18,3,17,18,23,18,4,15,25,23,18,3,17,18,23,25,3,17,25,23,25,10,3,25,23,25,3,17,25,23,25,4,15,11],"m40":[26,0,0,0,0,26,28,10,5,28,26,28,3,19,14,26,28,3,19,24,26,28,4,18,28,26,28,4,17,28,26,25,4,18,25,26,28,9,7,16,26,28,9,10,28,26,28,7,10,28,26,26,4,19,23,26,24,4,19,20,26,13,7,11,13,26,16,3,19,5,26,12,9,7,7,26,28,3,19,28,26,28,4,18,28,26,28,4,18,28,26,28,4,18,28,26,28,4,18,28,26,28,5,16,28,26,28,3,19,28,26,28,4,18,28,26,28,4,18,28,26,28,4,18,28,26,28,4,18,28,26,24,9,7,19,26,25,7,11,25,26,26,3,19,23,26,20,3,19,11,26,26,3,19,23,26,28,3,19,28,26,28,4,18,28,26,28,3,19,28,26,28,4,18,28,26,28,4,18,28,26,28,3,19,28,26,28,4,18,28,26,28,4,18,28,26,28,4,18,28,26,28,4,18,28,26,28,4,19,28,26,28,6,14,28,26,28,4,18,28,26,28,4,18,28,26,28,4,18,28,26,28,4,18,28,26,28,4,18,28,26,28,4,18,28,26,28,4,18,28,26,28,4,18,28,26,28,4,18,28,26,28,4,18,28,26,28,4,18,28,26,28,4,18,28,26,28,4,18,28,26,28,3,19,28,26,28,4,18,28,26,28,3,19,28,26,28,4,18,28,26,28,3,19,28,26,28,4,18,28,26,22,3,19,18,26,0,3,19,4,26,28,5,15,13,26,20,4,18,20,26,28,4,18,28,26,20,4,18,20,26,28,4,18,28,26,20,4,18,20,26,28,4,17,28,26,20,4,18,28,26,28,4,18,28,26,29,6,15,29,26,29,5,15,37,26,28,4,18,28,26,28,6,15,28,26,20,4,19,20,26,20,4,18,20,26,20,4,18,20,26,22,4,18,30,26,20,4,18,28,26,20,4,18,20,26,20,4,18,20,26,27,3,19,27,26,20,4,18,20,26,20,4,18,20,26,20,4,18,20,26,20,3,19,20,26,20,4,18,28,26,20,3,19,20,26,28,4,19,28,26,28,11,4,28,26,28,4,19,28,26,28,4,18,13],"a40":[26,0,0,0,0,26,28,10,5,28,26,28,3,19,13,26,28,3,19,24,26,28,4,18,28,26,28,4,17,28,26,25,4,18,25,26,28,9,7,16,26,28,9,10,28,26,28,7,10,28,26,26,4,19,23,26,24,4,19,20,26,13,7,11,13,26,16,3,19,5,26,12,9,7,7,26,28,3,19,28,26,28,4,18,28,26,28,4,18,28,26,28,4,18,28,26,28,4,18,28,26,28,5,16,28,26,28,3,19,28,26,28,4,18,28,26,28,4,18,28,26,28,4,18,28,26,28,4,18,28,26,24,9,7,19,26,25,7,11,25,26,26,3,19,23,26,20,3,19,11,26,26,3,19,23,26,28,3,19,28,26,28,4,18,28,26,28,3,19,28,26,28,4,18,28,26,28,4,18,28,26,28,3,19,28,26,28,4,18,28,26,28,4,18,28,26,28,4,18,28,26,28,4,18,28,26,28,4,19,28,26,28,6,14,28,26,28,4,18,28,26,28,4,18,28,26,28,4,18,28,26,28,4,18,28,26,28,4,18,28,26,28,4,18,28,26,28,4,18,28,26,28,4,18,28,26,28,4,18,28,26,28,4,18,28,26,28,4,18,28,26,28,4,18,28,26,28,4,18,28,26,28,3,19,28,26,28,4,18,28,26,28,3,19,28,26,28,4,18,28,26,28,3,19,28,26,28,4,18,28,26,22,3,19,18,26,0,3,19,4,26,28,5,15,13,26,20,4,18,20,26,28,4,18,28,26,20,4,18,20,26,28,4,18,28,26,20,4,18,20,26,28,4,17,28,26,20,4,18,28,26,28,4,18,28,26,29,6,15,29,26,29,5,15,37,26,28,4,18,28,26,28,6,15,28,26,20,4,19,20,26,20,4,18,20,26,20,4,18,20,26,22,4,18,30,26,20,4,18,28,26,20,4,18,20,26,20,4,18,20,26,27,3,19,27,26,20,4,18,20,26,20,4,18,20,26,20,4,18,20,26,20,3,19,20,26,20,4,18,28,26,20,3,19,20,26,28,4,19,28,26,28,11,4,28,26,28,4,19,28,26,28,4,18,13],"m45":[29,0,0,0,0,29,32,12,6,32,29,32,4,21,15,29,32,4,22,27,29,32,5,20,32,29,32,5,20,32,29,28,5,21,28,29,32,11,8,18,29,32,11,11,32,29,32,8,11,32,29,29,4,22,26,29,27,4,22,22,29,14,9,12,14,29,18,4,21,6,29,13,11,8,8,29,32,4,21,32,29,32,5,20,32,29,32,5,20,32,29,32,5,20,32,29,32,4,21,32,29,32,6,19,32,29,32,4,21,32,29,32,5,20,32,29,32,4,21,32,29,32,5,20,32,29,32,5,20,32,29,27,11,8,22,29,29,9,12,29,29,29,4,21,26,29,23,4,21,13,29,29,4,21,26,29,32,4,22,32,29,32,5,20,32,29,32,4,21,32,29,32,5,20,32,29,32,5,21,32,29,32,4,21,32,29,32,5,21,32,29,32,5,21,32,29,32,5,20,32,29,32,5,20,32,29,32,4,22,32,29,32,7,16,32,29,32,5,21,32,29,32,5,21,32,29,32,5,20,32,29,32,5,20,32,29,32,5,20,32,29,32,5,20,32,29,32,5,20,32,29,32,5,20,32,29,32,5,20,32,29,32,5,20,32,29,32,5,20,32,29,32,5,20,32,29,32,5,20,32,29,32,4,21,32,29,32,5,20,32,29,32,4,21,32,29,32,5,21,32,29,32,4,21,32,29,32,4,21,32,29,25,4,21,20,29,0,4,21,4,29,32,6,17,15,29,23,5,19,23,29,32,5,20,32,29,23,5,21,23,29,32,5,20,32,29,23,5,20,23,29,32,5,20,32,29,23,5,20,32,29,32,5,20,32,29,34,7,17,34,29,34,6,17,43,29,32,5,21,32,29,32,7,17,32,29,23,4,21,23,29,23,5,20,23,29,23,5,20,23,29,24,5,20,33,29,23,5,20,32,29,23,5,20,23,29,23,4,21,23,29,30,4,22,30,29,23,5,20,23,29,23,5,20,23,29,23,4,21,23,29,23,4,21,23,29,23,5,20,32,29,23,4,21,23,29,32,4,22,32,29,32,13,4,32,29,32,4,22,32,29,32,5,20,15],"a45":[29,0,0,0,0,29,32,12,6,32,29,32,4,21,15,29,32,4,22,27,29,32,5,20,32,29,32,5,20,32,29,28,5,21,28,29,32,11,8,18,29,32,11,11,32,29,32,8,11,32,29,29,4,22,26,29,27,4,22,22,29,14,9,12,14,29,18,4,21,6,29,13,11,8,8,29,32,4,21,32,29,32,5,20,32,29,32,5,20,32,29,32,5,20,32,29,32,4,21,32,29,32,6,19,32,29,32,4,21,32,29,32,5,20,32,29,32,4,21,32,29,32,5,20,32,29,32,5,20,32,29,27,11,8,22,29,29,9,12,29,29,29,4,21,26,29,23,4,21,13,29,29,4,21,26,29,32,4,22,32,29,32,5,20,32,29,32,4,21,32,29,32,5,20,32,29,32,5,21,32,29,32,4,21,32,29,32,5,21,32,29,32,5,21,32,29,32,5,20,32,29,32,5,20,32,29,32,4,22,32,29,32,7,16,32,29,32,5,21,32,29,32,5,21,32,29,32,5,20,32,29,32,5,20,32,29,32,5,20,32,29,32,5,20,32,29,32,5,20,32,29,32,5,20,32,29,32,5,20,32,29,32,5,20,32,29,32,5,20,32,29,32,5,20,32,29,32,5,20,32,29,32,4,21,32,29,32,5,20,32,29,32,4,21,32,29,32,5,21,32,29,32,4,21,32,29,32,4,21,32,29,25,4,21,20,29,0,4,21,4,29,32,6,17,15,29,23,5,19,23,29,32,5,20,32,29,23,5,21,23,29,32,5,20,32,29,23,5,20,23,29,32,5,20,32,29,23,5,20,32,29,32,5,20,32,29,34,7,17,34,29,34,6,17,43,29,32,5,21,32,29,32,7,17,32,29,23,4,21,23,29,23,5,20,23,29,23,5,20,23,29,24,5,20,33,29,23,5,20,32,29,23,5,20,23,29,23,4,21,23,29,30,4,22,30,29,23,5,20,23,29,23,5,20,23,29,23,4,21,23,29,23,4,21,23,29,23,5,20,32,29,23,4,21,23,29,32,4,22,32,29,32,13,4,32,29,32,4,22,32,29,32,5,20,15],"m48":[31,0,0,0,0,31,34,12,7,34,31,34,4,23,16,31,34,4,23,29,31,34,5,21,34,31,34,5,22,34,31,30,5,22,30,31,34,11,9,19,31,34,11,11,34,31,34,9,11,34,31,31,4,22,28,31,28,4,22,22,31,15,9,13,15,31,19,4,23,6,31,13,11,9,8,31,34,4,23,34,31,34,5,21,34,31,34,5,21,34,31,34,5,21,34,31,34,4,22,34,31,34,6,20,34,31,34,4,22,34,31,34,5,21,34,31,34,4,22,34,31,34,5,21,34,31,34,5,22,34,31,28,11,9,23,31,30,9,13,30,31,31,4,23,28,31,25,4,23,14,31,31,4,23,28,31,34,4,22,34,31,34,5,21,34,31,34,4,23,34,31,34,5,21,34,31,34,5,22,34,31,34,4,22,34,31,34,5,22,34,31,34,5,22,34,31,34,5,22,34,31,34,5,21,34,31,34,4,22,34,31,34,7,17,34,31,34,5,22,34,31,34,5,22,34,31,34,5,21,34,31,34,5,21,34,31,34,5,21,34,31,34,5,21,34,31,34,5,21,34,31,34,5,21,34,31,34,5,21,34,31,34,5,21,34,31,34,5,21,34,31,34,5,21,34,31,34,5,21,34,31,34,4,23,34,31,34,5,21,34,31,34,4,23,34,31,34,5,21,34,31,34,4,23,34,31,34,5,21,34,31,27,4,23,22,31,0,4,23,4,31,34,7,18,16,31,25,5,21,25,31,34,5,21,34,31,25,5,22,25,31,34,5,21,34,31,25,5,21,25,31,34,5,20,34,31,25,5,21,34,31,34,5,21,34,31,35,7,18,35,31,35,6,18,44,31,34,5,22,34,31,34,7,18,34,31,25,5,21,25,31,25,5,21,25,31,25,5,21,25,31,25,5,21,34,31,25,5,21,34,31,25,5,21,25,31,25,4,22,25,31,33,4,22,33,31,25,5,21,25,31,25,5,21,25,31,25,4,22,25,31,25,4,23,25,31,25,5,21,34,31,25,4,23,25,31,34,4,22,34,31,34,13,4,34,31,34,4,22,34,31,34,5,21,16],"a48":[31,0,0,0,0,31,34,12,7,34,31,34,4,23,16,31,34,4,23,29,31,34,5,21,34,31,34,5,22,34,31,30,5,22,30,31,34,11,9,19,31,34,11,11,34,31,34,9,11,34,31,31,4,22,28,31,28,4,22,22,31,15,9,13,15,31,19,4,23,6,31,13,11,9,8,31,34,4,23,34,31,34,5,21,34,31,34,5,21,34,31,34,5,21,34,31,34,4,22,34,31,34,6,20,34,31,34,4,22,34,31,34,5,21,34,31,34,4,22,34,31,34,5,21,34,31,34,5,22,34,31,28,11,9,23,31,30,9,13,30,31,31,4,23,28,31,25,4,23,14,31,31,4,23,28,31,34,4,22,34,31,34,5,21,34,31,34,4,23,34,31,34,5,21,34,31,34,5,22,34,31,34,4,22,34,31,34,5,22,34,31,34,5,22,34,31,34,5,22,34,31,34,5,21,34,31,34,4,22,34,31,34,7,17,34,31,34,5,22,34,31,34,5,22,34,31,34,5,21,34,31,34,5,21,34,31,34,5,21,34,31,34,5,21,34,31,34,5,21,34,31,34,5,21,34,31,34,5,21,34,31,34,5,21,34,31,34,5,21,34,31,34,5,21,34,31,34,5,21,34,31,34,4,23,34,31,34,5,21,34,31,34,4,23,34,31,34,5,21,34,31,34,4,23,34,31,34,5,21,34,31,27,4,23,22,31,0,4,23,4,31,34,7,18,16,31,25,5,21,25,31,34,5,21,34,31,25,5,22,25,31,34,5,21,34,31,25,5,21,25,31,34,5,20,34,31,25,5,21,34,31,34,5,21,34,31,35,7,18,35,31,35,6,18,44,31,34,5,22,34,31,34,7,18,34,31,25,5,21,25,31,25,5,21,25,31,25,5,21,25,31,25,5,21,34,31,25,5,21,34,31,25,5,21,25,31,25,4,22,25,31,33,4,22,33,31,25,5,21,25,31,25,5,21,25,31,25,4,22,25,31,25,4,23,25,31,25,5,21,34,31,25,4,23,25,31,34,4,22,34,31,34,13,4,34,31,34,4,22,34,31,34,5,21,16],"m50":[33,0,0,0,0,33,35,13,7,35,33,35,5,23,17,33,35,5,23,30,33,35,5,23,35,33,35,5,22,35,33,32,5,23,32,33,35,12,9,20,33,35,12,12,35,33,35,9,12,35,33,32,5,23,29,33,29,5,23,23,33,16,10,14,16,33,19,5,23,6,33,14,12,9,8,33,35,5,23,35,33,35,5,23,35,33,35,5,23,35,33,35,5,23,35,33,35,5,23,35,33,35,6,22,35,33,35,5,23,35,33,35,5,23,35,33,35,5,23,35,33,35,5,23,35,33,35,5,23,35,33,29,12,9,23,33,31,10,14,31,33,32,5,23,29,33,26,5,23,15,33,32,5,23,29,33,35,5,24,35,33,35,5,23,35,33,35,4,24,35,33,35,5,23,35,33,35,5,23,35,33,35,5,23,35,33,35,5,23,35,33,35,5,23,35,33,35,5,23,35,33,35,5,23,35,33,35,5,23,35,33,35,7,19,35,33,35,5,23,35,33,35,5,23,35,33,35,5,23,35,33,35,5,23,35,33,35,5,23,35,33,35,5,23,35,33,35,5,23,35,33,35,5,23,35,33,35,5,23,35,33,35,5,23,35,33,35,5,23,35,33,35,5,23,35,33,35,5,23,35,33,35,5,23,35,33,35,5,23,35,33,35,5,23,35,33,35,5,24,35,33,35,5,23,35,33,35,4,24,35,33,28,5,23,23,33,0,5,23,5,33,35,7,19,17,33,26,5,23,26,33,35,5,23,35,33,26,5,23,26,33,35,5,23,35,33,26,5,23,26,33,35,6,22,35,33,26,5,23,36,33,35,5,23,35,33,37,7,18,37,33,37,7,19,47,33,35,5,23,35,33,35,7,18,35,33,26,5,24,26,33,26,5,23,26,33,26,5,23,26,33,27,5,23,37,33,26,5,23,36,33,26,5,23,26,33,26,5,23,26,33,34,5,24,34,33,26,5,23,26,33,26,5,23,26,33,26,5,23,26,33,26,5,23,26,33,26,5,23,36,33,26,5,23,26,33,35,5,23,35,33,35,14,5,35,33,35,5,23,35,33,35,5,23,16],"a50":[33,0,0,0,0,33,35,13,7,35,33,35,5,23,17,33,35,5,23,30,33,35,5,23,35,33,35,5,22,35,33,32,5,23,32,33,35,12,9,20,33,35,12,12,35,33,35,9,12,35,33,32,5,23,29,33,29,5,23,23,33,16,10,14,16,33,19,5,23,6,33,14,12,9,8,33,35,5,23,35,33,35,5,23,35,33,35,5,23,35,33,35,5,23,35,33,35,5,23,35,33,35,6,22,35,33,35,5,23,35,33,35,5,23,35,33,35,5,23,35,33,35,5,23,35,33,35,5,23,35,33,29,12,9,23,33,31,10,14,31,33,32,5,23,29,33,26,5,23,15,33,32,5,23,29,33,35,5,24,35,33,35,5,23,35,33,35,4,24,35,33,35,5,23,35,33,35,5,23,35,33,35,5,23,35,33,35,5,23,35,33,35,5,23,35,33,35,5,23,35,33,35,5,23,35,33,35,5,23,35,33,35,7,19,35,33,35,5,23,35,33,35,5,23,35,33,35,5,23,35,33,35,5,23,35,33,35,5,23,35,33,35,5,23,35,33,35,5,23,35,33,35,5,23,35,33,35,5,23,35,33,35,5,23,35,33,35,5,23,35,33,35,5,23,35,33,35,5,23,35,33,35,5,23,35,33,35,5,23,35,33,35,5,23,35,33,35,5,24,35,33,35,5,23,35,33,35,4,24,35,33,28,5,23,23,33,0,5,23,5,33,35,7,19,17,33,26,5,23,26,33,35,5,23,35,33,26,5,23,26,33,35,5,23,35,33,26,5,23,26,33,35,6,22,35,33,26,5,23,36,33,35,5,23,35,33,37,7,18,37,33,37,7,19,47,33,35,5,23,35,33,35,7,18,35,33,26,5,24,26,33,26,5,23,26,33,26,5,23,26,33,27,5,23,37,33,26,5,23,36,33,26,5,23,26,33,26,5,23,26,33,34,5,24,34,33,26,5,23,26,33,26,5,23,26,33,26,5,23,26,33,26,5,23,26,33,26,5,23,36,33,26,5,23,26,33,35,5,23,35,33,35,14,5,35,33,35,5,23,35,33,35,5,23,16],"m54":[35,0,0,0,0,35,38,14,7,38,35,38,5,25,18,35,38,5,25,33,35,38,5,25,38,35,38,5,25,38,35,34,5,25,34,35,38,13,10,22,35,38,13,13,38,35,38,10,13,38,35,34,5,25,30,35,32,5,25,25,35,17,10,15,17,35,21,5,25,7,35,15,13,10,9,35,38,5,25,38,35,38,5,25,38,35,38,5,25,38,35,38,5,25,38,35,38,5,25,38,35,38,7,22,38,35,38,5,25,38,35,38,5,25,38,35,38,5,25,38,35,38,5,25,38,35,38,5,24,38,35,32,13,10,26,35,34,10,15,34,35,34,5,25,30,35,28,5,25,16,35,34,5,25,30,35,38,5,25,38,35,38,5,25,38,35,38,5,25,38,35,38,5,25,38,35,38,5,25,38,35,38,5,25,38,35,38,5,25,38,35,38,5,25,38,35,38,5,25,38,35,38,5,25,38,35,38,5,25,38,35,38,8,19,38,35,38,5,25,38,35,38,5,25,38,35,38,5,25,38,35,38,5,25,38,35,38,5,25,38,35,38,5,25,38,35,38,5,25,38,35,38,5,25,38,35,38,5,25,38,35,38,5,25,38,35,38,5,25,38,35,38,5,25,38,35,38,5,25,38,35,38,5,25,38,35,38,5,25,38,35,38,5,25,38,35,38,5,26,38,35,38,5,25,38,35,38,5,25,38,35,30,5,25,24,35,0,5,25,5,35,38,7,21,18,35,28,5,24,28,35,38,5,25,38,35,28,5,25,28,35,38,5,25,38,35,28,5,25,28,35,38,6,23,38,35,28,5,25,38,35,38,5,25,38,35,40,8,20,40,35,40,8,20,50,35,38,5,25,38,35,38,8,20,38,35,28,5,25,28,35,28,5,25,28,35,28,5,25,28,35,29,5,25,39,35,28,5,25,38,35,28,5,25,28,35,28,5,25,28,35,37,5,25,37,35,28,5,25,28,35,28,5,25,28,35,28,5,25,28,35,28,5,25,28,35,28,5,25,38,35,28,5,25,28,35,38,5,25,38,35,38,15,5,38,35,38,5,25,38,35,38,5,25,17],"a54":[35,0,0,0,0,35,38,14,7,38,35,38,5,25,18,35,38,5,25,33,35,38,5,25,38,35,38,5,25,38,35,34,5,25,34,35,38,13,10,22,35,38,13,13,38,35,38,10,13,38,35,34,5,25,30,35,32,5,25,25,35,17,10,15,17,35,21,5,25,7,35,15,13,10,9,35,38,5,25,38,35,38,5,25,38,35,38,5,25,38,35,38,5,25,38,35,38,5,25,38,35,38,7,22,38,35,38,5,25,38,35,38,5,25,38,35,38,5,25,38,35,38,5,25,38,35,38,5,24,38,35,32,13,10,26,35,34,10,15,34,35,34,5,25,30,35,28,5,25,16,35,34,5,25,30,35,38,5,25,38,35,38,5,25,38,35,38,5,25,38,35,38,5,25,38,35,38,5,25,38,35,38,5,25,38,35,38,5,25,38,35,38,5,25,38,35,38,5,25,38,35,38,5,25,38,35,38,5,25,38,35,38,8,19,38,35,38,5,25,38,35,38,5,25,38,35,38,5,25,38,35,38,5,25,38,35,38,5,25,38,35,38,5,25,38,35,38,5,25,38,35,38,5,25,38,35,38,5,25,38,35,38,5,25,38,35,38,5,25,38,35,38,5,25,38,35,38,5,25,38,35,38,5,25,38,35,38,5,25,38,35,38,5,25,38,35,38,5,26,38,35,38,5,25,38,35,38,5,25,38,35,30,5,25,24,35,0,5,25,5,35,38,7,21,18,35,28,5,24,28,35,38,5,25,38,35,28,5,25,28,35,38,5,25,38,35,28,5,25,28,35,38,6,23,38,35,28,5,25,38,35,38,5,25,38,35,40,8,20,40,35,40,8,20,50,35,38,5,25,38,35,38,8,20,38,35,28,5,25,28,35,28,5,25,28,35,28,5,25,28,35,29,5,25,39,35,28,5,25,38,35,28,5,25,28,35,28,5,25,28,35,37,5,25,37,35,28,5,25,28,35,28,5,25,28,35,28,5,25,28,35,28,5,25,28,35,28,5,25,38,35,28,5,25,28,35,38,5,25,38,35,38,15,5,38,35,38,5,25,38,35,38,5,25,17],"m56":[36,0,0,0,0,36,39,14,8,39,36,39,5,26,18,36,39,6,26,33,36,39,6,25,39,36,39,6,25,39,36,35,6,26,35,36,39,13,10,23,36,39,13,13,39,36,39,10,13,39,36,36,6,26,32,36,32,6,26,26,36,18,11,15,18,36,22,5,26,7,36,15,13,10,9,36,39,5,26,39,36,39,6,25,39,36,39,6,25,39,36,39,6,25,39,36,39,6,25,39,36,39,7,23,39,36,39,6,25,39,36,39,6,25,39,36,39,6,25,39,36,39,6,25,39,36,39,6,25,39,36,33,13,10,27,36,35,11,15,35,36,36,5,26,32,36,29,5,26,16,36,36,5,26,32,36,39,6,26,39,36,39,6,25,39,36,39,5,26,39,36,39,6,25,39,36,39,6,25,39,36,39,6,25,39,36,39,6,25,39,36,39,6,25,39,36,39,6,25,39,36,39,6,25,39,36,39,6,26,39,36,39,8,20,39,36,39,6,25,39,36,39,6,25,39,36,39,6,25,39,36,39,6,25,39,36,39,6,25,39,36,39,6,25,39,36,39,6,25,39,36,39,6,25,39,36,39,6,25,39,36,39,6,25,39,36,39,6,25,39,36,39,6,25,39,36,39,6,25,39,36,39,5,26,39,36,39,6,25,39,36,39,5,26,39,36,39,6,25,39,36,39,5,26,39,36,39,5,26,39,36,31,5,26,25,36,0,5,26,5,36,39,8,21,18,36,29,6,24,29,36,39,6,25,39,36,29,6,25,29,36,39,6,25,39,36,29,6,25,29,36,39,7,23,39,36,29,6,25,40,36,39,6,25,39,36,41,8,21,41,36,41,8,20,52,36,39,6,25,39,36,39,8,21,39,36,29,5,26,29,36,29,6,25,29,36,29,6,25,29,36,30,6,25,41,36,29,6,25,40,36,29,6,25,29,36,29,5,26,29,36,38,6,26,38,36,29,6,25,29,36,29,6,25,29,36,29,5,26,29,36,29,5,26,29,36,29,6,25,40,36,29,5,26,29,36,39,6,26,39,36,39,16,5,39,36,39,6,26,39,36,39,6,25,18],"a56":[36,0,0,0,0,36,39,14,8,39,36,39,5,26,18,36,39,6,26,33,36,39,6,25,39,36,39,6,25,39,36,35,6,26,35,36,39,13,10,23,36,39,13,13,39,36,39,10,13,39,36,36,6,26,32,36,32,6,26,26,36,18,11,15,18,36,22,5,26,7,36,15,13,10,9,36,39,5,26,39,36,39,6,25,39,36,39,6,25,39,36,39,6,25,39,36,39,6,25,39,36,39,7,23,39,36,39,6,25,39,36,39,6,25,39,36,39,6,25,39,36,39,6,25,39,36,39,6,25,39,36,33,13,10,27,36,35,11,15,35,36,36,5,26,32,36,29,5,26,16,36,36,5,26,32,36,39,6,26,39,36,39,6,25,39,36,39,5,26,39,36,39,6,25,39,36,39,6,25,39,36,39,6,25,39,36,39,6,25,39,36,39,6,25,39,36,39,6,25,39,36,39,6,25,39,36,39,6,26,39,36,39,8,20,39,36,39,6,25,39,36,39,6,25,39,36,39,6,25,39,36,39,6,25,39,36,39,6,25,39,36,39,6,25,39,36,39,6,25,39,36,39,6,25,39,36,39,6,25,39,36,39,6,25,39,36,39,6,25,39,36,39,6,25,39,36,39,6,25,39,36,39,5,26,39,36,39,6,25,39,36,39,5,26,39,36,39,6,25,39,36,39,5,26,39,36,39,5,26,39,36,31,5,26,25,36,0,5,26,5,36,39,8,21,18,36,29,6,24,29,36,39,6,25,39,36,29,6,25,29,36,39,6,25,39,36,29,6,25,29,36,39,7,23,39,36,29,6,25,40,36,39,6,25,39,36,41,8,21,41,36,41,8,20,52,36,39,6,25,39,36,39,8,21,39,36,29,5,26,29,36,29,6,25,29,36,29,6,25,29,36,30,6,25,41,36,29,6,25,40,36,29,6,25,29,36,29,5,26,29,36,38,6,26,38,36,29,6,25,29,36,29,6,25,29,36,29,5,26,29,36,29,5,26,29,36,29,6,25,40,36,29,5,26,29,36,39,6,26,39,36,39,16,5,39,36,39,6,26,39,36,39,6,25,18],"m60":[39,0,0,0,0,39,42,15,8,42,39,42,6,28,19,39,42,5,28,36,39,42,6,27,42,39,42,6,27,42,39,37,6,27,37,39,42,14,11,24,39,42,14,14,42,39,42,11,14,42,39,38,6,27,34,39,34,6,27,27,39,19,11,16,19,39,24,6,27,8,39,17,14,11,10,39,42,6,27,42,39,42,6,27,42,39,42,6,27,42,39,42,6,27,42,39,42,6,27,42,39,42,7,24,42,39,42,5,28,42,39,42,6,27,42,39,42,6,27,42,39,42,6,26,42,39,42,6,27,42,39,35,14,11,28,39,37,11,16,37,39,38,6,27,34,39,31,6,27,17,39,38,6,27,34,39,42,5,28,42,39,42,6,27,42,39,42,5,29,42,39,42,6,27,42,39,42,6,27,42,39,42,5,28,42,39,42,6,27,42,39,42,6,27,42,39,42,6,27,42,39,42,6,27,42,39,42,6,27,42,39,42,9,21,42,39,42,6,27,42,39,42,6,27,42,39,42,6,27,42,39,42,6,27,42,39,42,6,27,42,39,42,6,27,42,39,42,6,27,42,39,42,6,27,42,39,42,6,27,42,39,42,6,27,42,39,42,6,27,42,39,42,6,27,42,39,42,6,27,42,39,42,6,27,42,39,42,6,26,42,39,42,6,27,42,39,42,6,27,42,39,42,6,27,42,39,42,6,27,42,39,33,6,27,27,39,0,6,27,5,39,42,8,23,20,39,31,6,26,31,39,42,6,27,42,39,31,6,27,31,39,42,6,27,42,39,31,6,27,31,39,42,7,25,42,39,31,6,27,42,39,42,6,27,42,39,45,9,23,45,39,45,8,22,56,39,42,6,27,42,39,42,9,23,42,39,31,6,27,31,39,31,6,27,31,39,31,6,27,31,39,31,6,27,42,39,31,6,27,42,39,31,6,27,31,39,31,6,26,31,39,41,5,28,41,39,31,6,27,31,39,31,6,27,31,39,31,6,27,31,39,31,6,27,31,39,31,6,26,42,39,31,6,27,31,39,42,6,27,42,39,42,17,5,42,39,42,6,27,42,39,42,6,27,19],"a60":[39,0,0,0,0,39,42,15,8,42,39,42,6,28,20,39,42,5,28,36,39,42,6,27,42,39,42,6,27,42,39,37,6,27,37,39,42,14,11,24,39,42,14,14,42,39,42,11,14,42,39,38,6,27,34,39,34,6,27,27,39,19,11,16,19,39,24,6,27,8,39,17,14,11,10,39,42,6,27,42,39,42,6,27,42,39,42,6,27,42,39,42,6,27,42,39,42,6,27,42,39,42,7,24,42,39,42,5,28,42,39,42,6,27,42,39,42,6,27,42,39,42,6,26,42,39,42,6,27,42,39,35,14,11,28,39,37,11,16,37,39,38,6,27,34,39,31,6,27,17,39,38,6,27,34,39,42,5,28,42,39,42,6,27,42,39,42,5,29,42,39,42,6,27,42,39,42,6,27,42,39,42,5,28,42,39,42,6,27,42,39,42,6,27,42,39,42,6,27,42,39,42,6,27,42,39,42,6,27,42,39,42,9,21,42,39,42,6,27,42,39,42,6,27,42,39,42,6,27,42,39,42,6,27,42,39,42,6,27,42,39,42,6,27,42,39,42,6,27,42,39,42,6,27,42,39,42,6,27,42,39,42,6,27,42,39,42,6,27,42,39,42,6,27,42,39,42,6,27,42,39,42,6,27,42,39,42,6,26,42,39,42,6,27,42,39,42,6,27,42,39,42,6,27,42,39,42,6,27,42,39,33,6,27,27,39,0,6,27,5,39,42,8,23,20,39,31,6,26,31,39,42,6,27,42,39,31,6,27,31,39,42,6,27,42,39,31,6,27,31,39,42,7,25,42,39,31,6,27,42,39,42,6,27,42,39,45,9,23,45,39,45,8,22,56,39,42,6,27,42,39,42,9,23,42,39,31,6,27,31,39,31,6,27,31,39,31,6,27,31,39,31,6,27,42,39,31,6,27,42,39,31,6,27,31,39,31,6,26,31,39,41,5,28,41,39,31,6,27,31,39,31,6,27,31,39,31,6,27,31,39,31,6,27,31,39,31,6,26,42,39,31,6,27,31,39,42,6,27,42,39,42,17,5,42,39,42,6,27,42,39,42,6,27,19],"m63":[41,0,0,0,0,41,44,16,8,44,41,44,6,29,21,41,44,6,29,37,41,44,6,28,44,41,44,6,28,44,41,40,6,29,40,41,44,15,11,25,41,44,15,15,44,41,44,11,15,44,41,40,6,29,36,41,37,6,29,30,41,20,12,17,20,41,25,6,28,8,41,18,15,11,11,41,44,6,28,44,41,44,6,28,44,41,44,6,28,44,41,44,6,28,44,41,44,5,29,44,41,44,8,25,44,41,44,6,28,44,41,44,6,28,44,41,44,6,28,44,41,44,6,29,44,41,44,6,29,44,41,37,15,11,30,41,40,12,17,40,41,40,6,28,36,41,32,6,28,18,41,40,6,28,36,41,44,6,29,44,41,44,6,28,44,41,44,6,28,44,41,44,6,28,44,41,44,6,29,44,41,44,6,28,44,41,44,6,29,44,41,44,6,29,44,41,44,6,28,44,41,44,6,28,44,41,44,6,29,44,41,44,9,23,44,41,44,6,29,44,41,44,6,29,44,41,44,6,28,44,41,44,6,28,44,41,44,6,28,44,41,44,6,28,44,41,44,6,28,44,41,44,6,28,44,41,44,6,28,44,41,44,6,28,44,41,44,6,28,44,41,44,6,28,44,41,44,6,28,44,41,44,6,28,44,41,44,6,28,44,41,44,6,28,44,41,44,6,29,44,41,44,6,28,44,41,44,6,28,44,41,35,6,28,28,41,0,6,28,6,41,44,9,23,21,41,32,6,29,32,41,44,6,28,44,41,32,6,29,32,41,44,6,28,44,41,32,6,28,32,41,44,7,26,44,41,32,6,28,44,41,44,6,28,44,41,46,9,23,46,41,46,9,23,58,41,44,6,29,44,41,44,9,23,44,41,32,6,29,32,41,32,6,28,32,41,32,6,28,32,41,34,6,28,46,41,32,6,28,44,41,32,6,28,32,41,32,6,28,32,41,42,6,29,42,41,32,6,28,32,41,32,6,28,32,41,32,6,29,32,41,32,6,28,32,41,32,6,28,44,41,32,6,28,32,41,44,6,29,44,41,44,17,6,44,41,44,6,29,44,41,44,6,28,20],"a63":[41,0,0,0,0,41,44,16,8,44,41,44,6,29,21,41,44,6,29,37,41,44,6,28,44,41,44,6,28,44,41,40,6,29,40,41,44,15,11,25,41,44,15,15,44,41,44,11,15,44,41,40,6,29,36,41,37,6,29,30,41,20,12,17,20,41,25,6,28,8,41,18,15,11,11,41,44,6,28,44,41,44,6,28,44,41,44,6,28,44,41,44,6,28,44,41,44,5,29,44,41,44,8,25,44,41,44,6,28,44,41,44,6,28,44,41,44,6,28,44,41,44,6,29,44,41,44,6,29,44,41,37,15,11,30,41,40,12,17,40,41,40,6,28,36,41,32,6,28,18,41,40,6,28,36,41,44,6,29,44,41,44,6,28,44,41,44,6,28,44,41,44,6,28,44,41,44,6,29,44,41,44,6,28,44,41,44,6,29,44,41,44,6,29,44,41,44,6,28,44,41,44,6,28,44,41,44,6,29,44,41,44,9,23,44,41,44,6,29,44,41,44,6,29,44,41,44,6,28,44,41,44,6,28,44,41,44,6,28,44,41,44,6,28,44,41,44,6,28,44,41,44,6,28,44,41,44,6,28,44,41,44,6,28,44,41,44,6,28,44,41,44,6,28,44,41,44,6,28,44,41,44,6,28,44,41,44,6,28,44,41,44,6,28,44,41,44,6,29,44,41,44,6,28,44,41,44,6,28,44,41,35,6,28,28,41,0,6,28,6,41,44,9,23,21,41,32,6,29,32,41,44,6,28,44,41,32,6,29,32,41,44,6,28,44,41,32,6,28,32,41,44,7,26,44,41,32,6,28,44,41,44,6,28,44,41,46,9,23,46,41,46,9,23,58,41,44,6,29,44,41,44,9,23,44,41,32,6,29,32,41,32,6,28,32,41,32,6,28,32,41,34,6,28,46,41,32,6,28,44,41,32,6,28,32,41,32,6,28,32,41,42,6,29,42,41,32,6,28,32,41,32,6,28,32,41,32,6,29,32,41,32,6,28,32,41,32,6,28,44,41,32,6,28,32,41,44,6,29,44,41,44,17,6,44,41,44,6,29,44,41,44,6,28,20],"m64":[42,0,0,0,0,42,45,16,9,45,42,45,6,30,21,42,45,6,30,38,42,45,7,28,45,42,45,7,29,45,42,40,7,29,40,42,45,15,12,26,42,45,15,15,45,42,45,12,15,45,42,41,6,30,37,42,38,6,30,30,42,21,12,17,21,42,25,6,30,8,42,18,15,12,11,42,45,6,30,45,42,45,7,28,45,42,45,7,28,45,42,45,7,28,45,42,45,6,29,45,42,45,8,26,45,42,45,6,29,45,42,45,7,28,45,42,45,6,29,45,42,45,7,29,45,42,45,7,29,45,42,38,15,12,31,42,40,12,18,40,42,41,6,30,37,42,33,6,30,18,42,41,6,30,37,42,45,6,30,45,42,45,7,28,45,42,45,6,30,45,42,45,7,28,45,42,45,7,29,45,42,45,6,29,45,42,45,7,29,45,42,45,7,29,45,42,45,7,28,45,42,45,7,28,45,42,45,6,30,45,42,45,9,24,45,42,45,7,29,45,42,45,7,29,45,42,45,7,28,45,42,45,7,28,45,42,45,7,28,45,42,45,7,28,45,42,45,7,28,45,42,45,7,28,45,42,45,7,28,45,42,45,7,28,45,42,45,7,28,45,42,45,7,28,45,42,45,7,28,45,42,45,6,30,45,42,45,7,28,45,42,45,6,30,45,42,45,7,29,45,42,45,6,30,45,42,45,6,29,45,42,36,6,30,29,42,0,6,30,6,42,45,9,24,21,42,33,7,28,33,42,45,7,28,45,42,33,7,29,33,42,45,7,28,45,42,33,7,28,33,42,45,8,27,45,42,33,7,28,45,42,45,7,28,45,42,47,9,23,47,42,47,9,23,59,42,45,7,29,45,42,45,9,23,45,42,33,6,30,33,42,33,7,28,33,42,33,7,28,33,42,34,7,28,46,42,33,7,28,45,42,33,7,28,33,42,33,6,29,33,42,43,6,30,43,42,33,7,28,33,42,33,7,28,33,42,33,6,30,33,42,33,6,30,33,42,33,7,28,45,42,33,6,30,33,42,45,6,30,45,42,45,18,6,45,42,45,6,30,45,42,45,7,28,20],"a64":[42,0,0,0,0,42,45,16,9,45,42,45,6,30,21,42,45,6,30,38,42,45,7,28,45,42,45,7,29,45,42,40,7,29,40,42,45,15,12,26,42,45,15,15,45,42,45,12,15,45,42,41,6,30,37,42,38,6,30,30,42,21,12,17,21,42,25,6,30,8,42,18,15,12,11,42,45,6,30,45,42,45,7,28,45,42,45,7,28,45,42,45,7,28,45,42,45,6,29,45,42,45,8,26,45,42,45,6,29,45,42,45,7,28,45,42,45,6,29,45,42,45,7,29,45,42,45,7,29,45,42,38,15,12,31,42,40,12,18,40,42,41,6,30,37,42,33,6,30,18,42,41,6,30,37,42,45,6,30,45,42,45,7,28,45,42,45,6,30,45,42,45,7,28,45,42,45,7,29,45,42,45,6,29,45,42,45,7,29,45,42,45,7,29,45,42,45,7,28,45,42,45,7,28,45,42,45,6,30,45,42,45,9,24,45,42,45,7,29,45,42,45,7,29,45,42,45,7,28,45,42,45,7,28,45,42,45,7,28,45,42,45,7,28,45,42,45,7,28,45,42,45,7,28,45,42,45,7,28,45,42,45,7,28,45,42,45,7,28,45,42,45,7,28,45,42,45,7,28,45,42,45,6,30,45,42,45,7,28,45,42,45,6,30,45,42,45,7,29,45,42,45,6,30,45,42,45,6,29,45,42,36,6,30,29,42,0,6,30,6,42,45,9,24,21,42,33,7,28,33,42,45,7,28,45,42,33,7,29,33,42,45,7,28,45,42,33,7,28,33,42,45,8,27,45,42,33,7,28,45,42,45,7,28,45,42,47,9,23,47,42,47,9,23,59,42,45,7,29,45,42,45,9,23,45,42,33,6,30,33,42,33,7,28,33,42,33,7,28,33,42,34,7,28,46,42,33,7,28,45,42,33,7,28,33,42,33,6,29,33,42,43,6,30,43,42,33,7,28,33,42,33,7,28,33,42,33,6,30,33,42,33,6,30,33,42,33,7,28,45,42,33,6,30,33,42,45,6,30,45,42,45,18,6,45,42,45,6,30,45,42,45,7,28,20],"m70":[46,0,0,0,0,46,49,18,10,49,46,49,7,32,23,46,49,6,33,41,46,49,7,31,49,46,49,7,32,49,46,43,7,32,43,46,49,16,13,28,46,49,16,16,49,46,49,13,16,49,46,44,7,32,39,46,40,7,32,32,46,22,13,19,22,46,28,7,32,9,46,20,16,13,12,46,49,7,32,49,46,49,7,31,49,46,49,7,31,49,46,49,7,31,49,46,49,6,32,49,46,49,9,28,49,46,49,6,32,49,46,49,7,31,49,46,49,7,31,49,46,49,7,31,49,46,49,7,32,49,46,41,16,13,33,46,44,13,19,44,46,44,7,32,39,46,36,7,32,20,46,44,7,32,39,46,49,6,32,49,46,49,7,31,49,46,49,6,33,49,46,49,7,31,49,46,49,7,32,49,46,49,6,32,49,46,49,7,32,49,46,49,7,32,49,46,49,7,31,49,46,49,7,31,49,46,49,7,32,49,46,49,10,25,49,46,49,7,32,49,46,49,7,32,49,46,49,7,31,49,46,49,7,31,49,46,49,7,31,49,46,49,7,31,49,46,49,7,31,49,46,49,7,31,49,46,49,7,31,49,46,49,7,31,49,46,49,7,31,49,46,49,7,31,49,46,49,7,31,49,46,49,7,32,49,46,49,7,31,49,46,49,7,32,49,46,49,7,31,49,46,49,7,32,49,46,49,7,31,49,46,39,7,32,31,46,0,7,32,6,46,49,10,26,23,46,36,7,30,36,46,49,7,31,49,46,36,7,32,36,46,49,7,31,49,46,36,7,31,36,46,49,8,29,49,46,36,7,31,49,46,49,7,31,49,46,52,10,26,52,46,52,9,26,65,46,49,7,32,49,46,49,10,26,49,46,36,7,32,36,46,36,7,31,36,46,36,7,31,36,46,37,7,31,50,46,36,7,31,49,46,36,7,31,36,46,36,7,31,36,46,47,6,32,47,46,36,7,31,36,46,36,7,31,36,46,36,6,33,36,46,36,7,32,36,46,36,7,31,49,46,36,7,32,36,46,49,7,32,49,46,49,20,6,49,46,49,7,32,49,46,49,7,31,22],"a70":[46,0,0,0,0,46,49,18,10,49,46,49,7,32,23,46,49,6,33,41,46,49,7,31,49,46,49,7,32,49,46,43,7,32,43,46,49,16,13,28,46,49,16,16,49,46,49,13,16,49,46,44,7,32,39,46,40,7,32,32,46,22,13,19,22,46,28,7,32,9,46,20,16,13,12,46,49,7,32,49,46,49,7,31,49,46,49,7,31,49,46,49,7,31,49,46,49,6,32,49,46,49,9,28,49,46,49,6,32,49,46,49,7,31,49,46,49,7,31,49,46,49,7,31,49,46,49,7,32,49,46,41,16,13,33,46,44,13,19,44,46,44,7,32,39,46,36,7,32,20,46,44,7,32,39,46,49,6,32,49,46,49,7,31,49,46,49,6,33,49,46,49,7,31,49,46,49,7,32,49,46,49,6,32,49,46,49,7,32,49,46,49,7,32,49,46,49,7,31,49,46,49,7,31,49,46,49,7,32,49,46,49,10,25,49,46,49,7,32,49,46,49,7,32,49,46,49,7,31,49,46,49,7,31,49,46,49,7,31,49,46,49,7,31,49,46,49,7,31,49,46,49,7,31,49,46,49,7,31,49,46,49,7,31,49,46,49,7,31,49,46,49,7,31,49,46,49,7,31,49,46,49,7,32,49,46,49,7,31,49,46,49,7,32,49,46,49,7,31,49,46,49,7,32,49,46,49,7,31,49,46,39,7,32,31,46,0,7,32,6,46,49,10,26,23,46,36,7,30,36,46,49,7,31,49,46,36,7,32,36,46,49,7,31,49,46,36,7,31,36,46,49,8,29,49,46,36,7,31,49,46,49,7,31,49,46,52,10,26,52,46,52,9,26,65,46,49,7,32,49,46,49,10,26,49,46,36,7,32,36,46,36,7,31,36,46,36,7,31,36,46,37,7,31,50,46,36,7,31,49,46,36,7,31,36,46,36,7,31,36,46,47,6,32,47,46,36,7,31,36,46,36,7,31,36,46,36,6,33,36,46,36,7,32,36,46,36,7,31,49,46,36,7,32,36,46,49,7,32,49,46,49,20,6,49,46,49,7,32,49,46,49,7,31,22],"m72":[47,0,0,0,0,47,51,19,10,51,47,51,7,33,24,47,51,7,34,44,47,51,7,33,51,47,51,7,33,51,47,45,7,34,45,47,51,17,13,29,47,51,17,17,51,47,51,13,17,51,47,46,7,33,41,47,42,7,33,33,47,24,14,20,24,47,28,7,33,9,47,20,17,13,12,47,51,7,33,51,47,51,7,33,51,47,51,7,33,51,47,51,7,33,51,47,51,7,33,51,47,51,9,30,51,47,51,7,33,51,47,51,7,33,51,47,51,7,33,51,47,51,7,34,51,47,51,7,33,51,47,42,17,13,34,47,45,14,20,45,47,46,7,33,41,47,37,7,33,20,47,46,7,33,41,47,51,7,34,51,47,51,7,33,51,47,51,7,33,51,47,51,7,33,51,47,51,7,33,51,47,51,7,33,51,47,51,7,33,51,47,51,7,33,51,47,51,7,33,51,47,51,7,33,51,47,51,7,33,51,47,51,11,26,51,47,51,7,33,51,47,51,7,33,51,47,51,7,33,51,47,51,7,33,51,47,51,7,33,51,47,51,7,33,51,47,51,7,33,51,47,51,7,33,51,47,51,7,33,51,47,51,7,33,51,47,51,7,33,51,47,51,7,33,51,47,51,7,33,51,47,51,7,33,51,47,51,7,33,51,47,51,7,33,51,47,51,7,34,51,47,51,7,33,51,47,51,6,34,51,47,40,7,33,32,47,0,7,33,7,47,51,10,27,24,47,37,7,33,37,47,51,7,33,51,47,37,7,33,37,47,51,7,33,51,47,37,7,33,37,47,51,8,31,51,47,37,7,33,51,47,51,7,33,51,47,53,11,26,53,47,53,10,27,67,47,51,7,33,51,47,51,11,26,51,47,37,7,34,37,47,37,7,33,37,47,37,7,33,37,47,39,7,33,53,47,37,7,33,51,47,37,7,33,37,47,37,7,33,37,47,49,7,34,49,47,37,7,33,37,47,37,7,33,37,47,37,7,34,37,47,37,7,33,37,47,37,7,33,51,47,37,7,33,37,47,51,7,33,51,47,51,20,7,51,47,51,7,33,51,47,51,7,33,23],"a72":[47,0,0,0,0,47,51,19,10,51,47,51,7,33,24,47,51,7,34,44,47,51,7,33,51,47,51,7,33,51,47,45,7,34,45,47,51,17,13,29,47,51,17,17,51,47,51,13,17,51,47,46,7,33,41,47,42,7,33,33,47,24,14,20,24,47,28,7,33,9,47,20,17,13,12,47,51,7,33,51,47,51,7,33,51,47,51,7,33,51,47,51,7,33,51,47,51,7,33,51,47,51,9,30,51,47,51,7,33,51,47,51,7,33,51,47,51,7,33,51,47,51,7,34,51,47,51,7,33,51,47,42,17,13,34,47,45,14,20,45,47,46,7,33,41,47,37,7,33,20,47,46,7,33,41,47,51,7,34,51,47,51,7,33,51,47,51,7,33,51,47,51,7,33,51,47,51,7,33,51,47,51,7,33,51,47,51,7,33,51,47,51,7,33,51,47,51,7,33,51,47,51,7,33,51,47,51,7,33,51,47,51,11,26,51,47,51,7,33,51,47,51,7,33,51,47,51,7,33,51,47,51,7,33,51,47,51,7,33,51,47,51,7,33,51,47,51,7,33,51,47,51,7,33,51,47,51,7,33,51,47,51,7,33,51,47,51,7,33,51,47,51,7,33,51,47,51,7,33,51,47,51,7,33,51,47,51,7,33,51,47,51,7,33,51,47,51,7,34,51,47,51,7,33,51,47,51,6,34,51,47,40,7,33,32,47,0,7,33,7,47,51,10,27,24,47,37,7,33,37,47,51,7,33,51,47,37,7,33,37,47,51,7,33,51,47,37,7,33,37,47,51,8,31,51,47,37,7,33,51,47,51,7,33,51,47,53,11,26,53,47,53,10,27,67,47,51,7,33,51,47,51,11,26,51,47,37,7,34,37,47,37,7,33,37,47,37,7,33,37,47,39,7,33,53,47,37,7,33,51,47,37,7,33,37,47,37,7,33,37,47,49,7,34,49,47,37,7,33,37,47,37,7,33,37,47,37,7,34,37,47,37,7,33,37,47,37,7,33,51,47,37,7,33,37,47,51,7,33,51,47,51,20,7,51,47,51,7,33,51,47,51,7,33,23],"m80":[52,0,0,0,0,52,56,21,11,56,52,56,7,38,26,52,56,7,38,48,52,56,8,36,56,52,56,8,36,56,52,50,8,37,50,52,56,19,14,32,52,56,19,18,56,52,56,15,18,56,52,51,7,37,45,52,46,7,37,37,52,25,15,22,25,52,31,8,36,10,52,22,19,14,13,52,56,8,36,56,52,56,8,36,56,52,56,8,36,56,52,56,8,36,56,52,56,8,36,56,52,56,10,32,56,52,56,7,37,56,52,56,8,36,56,52,56,7,37,56,52,56,8,36,56,52,56,8,36,56,52,47,19,14,38,52,50,15,21,50,52,51,8,36,45,52,41,8,36,23,52,51,8,36,45,52,56,7,37,56,52,56,8,36,56,52,56,7,38,56,52,56,8,36,56,52,56,8,36,56,52,56,7,37,56,52,56,8,36,56,52,56,8,36,56,52,56,8,36,56,52,56,8,36,56,52,56,7,37,56,52,56,12,28,56,52,56,8,36,56,52,56,8,36,56,52,56,8,36,56,52,56,8,36,56,52,56,8,36,56,52,56,8,36,56,52,56,8,36,56,52,56,8,36,56,52,56,8,36,56,52,56,8,36,56,52,56,8,36,56,52,56,8,36,56,52,56,8,36,56,52,56,8,36,56,52,56,8,36,56,52,56,8,36,56,52,56,8,36,56,52,56,8,36,56,52,56,8,36,56,52,44,8,36,35,52,0,8,36,7,52,56,11,30,26,52,41,8,35,41,52,56,8,36,56,52,41,8,36,41,52,56,8,36,56,52,41,8,36,41,52,56,9,33,56,52,41,8,36,56,52,56,8,36,56,52,59,12,29,59,52,59,11,30,74,52,56,8,36,56,52,56,12,29,56,52,41,8,36,41,52,41,8,36,41,52,41,8,36,41,52,42,8,36,57,52,41,8,36,56,52,41,8,36,41,52,41,8,36,41,52,54,7,37,54,52,41,8,36,41,52,41,8,36,41,52,41,7,37,41,52,41,8,36,41,52,41,8,36,56,52,41,8,36,41,52,56,7,37,56,52,56,22,7,56,52,56,7,37,56,52,56,8,36,25],"a80":[52,0,0,0,0,52,56,21,11,56,52,56,7,38,26,52,56,7,38,48,52,56,8,36,56,52,56,8,36,56,52,50,8,37,50,52,56,19,14,32,52,56,19,18,56,52,56,15,18,56,52,51,7,37,45,52,46,7,37,37,52,25,15,22,25,52,31,8,36,10,52,22,19,14,13,52,56,8,36,56,52,56,8,36,56,52,56,8,36,56,52,56,8,36,56,52,56,8,36,56,52,56,10,32,56,52,56,7,37,56,52,56,8,36,56,52,56,7,37,56,52,56,8,36,56,52,56,8,36,56,52,47,19,14,38,52,50,15,21,50,52,51,8,36,45,52,41,8,36,23,52,51,8,36,45,52,56,7,37,56,52,56,8,36,56,52,56,7,38,56,52,56,8,36,56,52,56,8,36,56,52,56,7,37,56,52,56,8,36,56,52,56,8,36,56,52,56,8,36,56,52,56,8,36,56,52,56,7,37,56,52,56,12,28,56,52,56,8,36,56,52,56,8,36,56,52,56,8,36,56,52,56,8,36,56,52,56,8,36,56,52,56,8,36,56,52,56,8,36,56,52,56,8,36,56,52,56,8,36,56,52,56,8,36,56,52,56,8,36,56,52,56,8,36,56,52,56,8,36,56,52,56,8,36,56,52,56,8,36,56,52,56,8,36,56,52,56,8,36,56,52,56,8,36,56,52,56,8,36,56,52,44,8,36,35,52,0,8,36,7,52,56,11,30,26,52,41,8,35,41,52,56,8,36,56,52,41,8,36,41,52,56,8,36,56,52,41,8,36,41,52,56,9,33,56,52,41,8,36,56,52,56,8,36,56,52,59,12,29,59,52,59,11,30,74,52,56,8,36,56,52,56,12,29,56,52,41,8,36,41,52,41,8,36,41,52,41,8,36,41,52,42,8,36,57,52,41,8,36,56,52,41,8,36,41,52,41,8,36,41,52,54,7,37,54,52,41,8,36,41,52,41,8,36,41,52,41,7,37,41,52,41,8,36,41,52,41,8,36,56,52,41,8,36,41,52,56,7,37,56,52,56,22,7,56,52,56,7,37,56,52,56,8,36,25],"m81":[53,0,0,0,0,53,57,20,11,57,53,57,7,37,26,53,57,7,37,48,53,57,8,35,57,53,57,8,35,57,53,50,8,36,50,53,57,19,14,33,53,57,19,18,57,53,57,15,18,57,53,51,7,37,45,53,47,7,37,37,53,26,15,21,26,53,33,8,36,11,53,23,19,14,14,53,57,8,36,57,53,57,8,35,57,53,57,8,35,57,53,57,8,35,57,53,57,7,36,57,53,57,10,32,57,53,57,7,36,57,53,57,8,35,57,53,57,7,36,57,53,57,8,35,57,53,57,8,36,57,53,48,19,14,39,53,51,15,21,51,53,51,8,36,45,53,41,8,36,22,53,51,8,36,45,53,57,7,37,57,53,57,8,35,57,53,57,7,37,57,53,57,8,35,57,53,57,8,36,57,53,57,7,36,57,53,57,8,36,57,53,57,8,36,57,53,57,8,36,57,53,57,8,35,57,53,57,7,37,57,53,57,12,28,57,53,57,8,36,57,53,57,8,36,57,53,57,8,35,57,53,57,8,35,57,53,57,8,35,57,53,57,8,35,57,53,57,8,35,57,53,57,8,35,57,53,57,8,35,57,53,57,8,35,57,53,57,8,35,57,53,57,8,35,57,53,57,8,35,57,53,57,8,36,57,53,57,8,35,57,53,57,8,36,57,53,57,8,36,57,53,57,8,36,57,53,57,8,35,57,53,45,8,36,36,53,0,8,36,7,53,57,11,30,26,53,41,8,35,41,53,57,8,35,57,53,41,8,36,41,53,57,8,35,57,53,41,8,35,41,53,57,9,33,57,53,41,8,35,56,53,57,8,35,57,53,60,12,29,60,53,60,11,29,75,53,57,8,36,57,53,57,12,29,57,53,41,8,36,41,53,41,8,35,41,53,41,8,35,41,53,43,8,35,58,53,41,8,35,56,53,41,8,35,41,53,41,8,35,41,53,54,7,37,54,53,41,8,35,41,53,41,8,35,41,53,41,7,37,41,53,41,8,36,41,53,41,8,35,56,53,41,8,36,41,53,57,7,37,57,53,57,22,7,57,53,57,7,37,57,53,57,8,35,26],"a81":[53,0,0,0,0,53,57,20,11,57,53,57,7,37,26,53,57,7,37,48,53,57,8,35,57,53,57,8,35,57,53,50,8,36,50,53,57,19,14,33,53,57,19,18,57,53,57,15,18,57,53,51,7,37,45,53,47,7,37,37,53,26,15,21,26,53,33,8,36,11,53,23,19,14,14,53,57,8,36,57,53,57,8,35,57,53,57,8,35,57,53,57,8,35,57,53,57,7,36,57,53,57,10,32,57,53,57,7,36,57,53,57,8,35,57,53,57,7,36,57,53,57,8,35,57,53,57,8,36,57,53,48,19,14,39,53,51,15,21,51,53,51,8,36,45,53,41,8,36,22,53,51,8,36,45,53,57,7,37,57,53,57,8,35,57,53,57,7,37,57,53,57,8,35,57,53,57,8,36,57,53,57,7,36,57,53,57,8,36,57,53,57,8,36,57,53,57,8,36,57,53,57,8,35,57,53,57,7,37,57,53,57,12,28,57,53,57,8,36,57,53,57,8,36,57,53,57,8,35,57,53,57,8,35,57,53,57,8,35,57,53,57,8,35,57,53,57,8,35,57,53,57,8,35,57,53,57,8,35,57,53,57,8,35,57,53,57,8,35,57,53,57,8,35,57,53,57,8,35,57,53,57,8,36,57,53,57,8,35,57,53,57,8,36,57,53,57,8,36,57,53,57,8,36,57,53,57,8,35,57,53,45,8,36,36,53,0,8,36,7,53,57,11,30,26,53,41,8,35,41,53,57,8,35,57,53,41,8,36,41,53,57,8,35,57,53,41,8,35,41,53,57,9,33,57,53,41,8,35,56,53,57,8,35,57,53,60,12,29,60,53,60,11,29,75,53,57,8,36,57,53,57,12,29,57,53,41,8,36,41,53,41,8,35,41,53,41,8,35,41,53,43,8,35,58,53,41,8,35,56,53,41,8,35,41,53,41,8,35,41,53,54,7,37,54,53,41,8,35,41,53,41,8,35,41,53,41,7,37,41,53,41,8,36,41,53,41,8,35,56,53,41,8,36,41,53,57,7,37,57,53,57,22,7,57,53,57,7,37,57,53,57,8,35,26],"m84":[55,0,0,0,0,55,59,22,11,59,55,59,8,39,28,55,59,8,39,50,55,59,9,38,59,55,59,9,37,59,55,53,9,38,53,55,59,20,15,34,55,59,20,20,59,55,59,15,20,59,55,53,9,39,47,55,49,9,39,39,55,27,16,23,27,55,33,8,39,11,55,24,20,15,14,55,59,8,39,59,55,59,9,38,59,55,59,9,38,59,55,59,9,38,59,55,59,8,39,59,55,59,10,35,59,55,59,8,39,59,55,59,9,38,59,55,59,9,38,59,55,59,9,38,59,55,59,9,38,59,55,49,20,15,39,55,52,16,23,52,55,53,8,39,47,55,43,8,39,24,55,53,8,39,47,55,59,8,39,59,55,59,9,38,59,55,59,8,39,59,55,59,9,38,59,55,59,9,39,59,55,59,8,39,59,55,59,9,39,59,55,59,9,39,59,55,59,9,37,59,55,59,9,38,59,55,59,9,39,59,55,59,12,31,59,55,59,9,39,59,55,59,9,39,59,55,59,9,38,59,55,59,9,38,59,55,59,9,38,59,55,59,9,38,59,55,59,9,38,59,55,59,9,38,59,55,59,9,38,59,55,59,9,38,59,55,59,9,38,59,55,59,9,38,59,55,59,9,38,59,55,59,8,39,59,55,59,9,37,59,55,59,8,39,59,55,59,9,38,59,55,59,8,39,59,55,59,8,39,59,55,47,8,39,38,55,0,8,39,8,55,59,12,31,27,55,43,9,38,43,55,59,9,38,59,55,43,9,39,43,55,59,9,38,59,55,43,9,38,43,55,59,10,35,59,55,43,9,38,59,55,59,9,38,59,55,62,12,31,62,55,62,12,31,78,55,59,9,39,59,55,59,12,31,59,55,43,8,39,43,55,43,9,38,43,55,43,9,38,43,55,45,9,38,61,55,43,9,38,59,55,43,9,38,43,55,43,8,38,43,55,56,8,39,56,55,43,9,38,43,55,43,9,38,43,55,43,8,39,43,55,43,8,39,43,55,43,9,37,59,55,43,8,39,43,55,59,9,39,59,55,59,24,8,59,55,59,9,39,59,55,59,9,38,27],"a84":[55,0,0,0,0,55,59,22,11,59,55,59,8,39,27,55,59,8,39,50,55,59,9,38,59,55,59,9,37,59,55,53,9,38,53,55,59,20,15,34,55,59,20,20,59,55,59,15,20,59,55,53,9,39,47,55,49,9,39,39,55,27,16,23,27,55,33,8,39,11,55,24,20,15,14,55,59,8,39,59,55,59,9,38,59,55,59,9,38,59,55,59,9,38,59,55,59,8,39,59,55,59,10,35,59,55,59,8,39,59,55,59,9,38,59,55,59,9,38,59,55,59,9,38,59,55,59,9,38,59,55,49,20,15,39,55,52,16,23,52,55,53,8,39,47,55,43,8,39,24,55,53,8,39,47,55,59,8,39,59,55,59,9,38,59,55,59,8,39,59,55,59,9,38,59,55,59,9,39,59,55,59,8,39,59,55,59,9,39,59,55,59,9,39,59,55,59,9,37,59,55,59,9,38,59,55,59,9,39,59,55,59,12,31,59,55,59,9,39,59,55,59,9,39,59,55,59,9,38,59,55,59,9,38,59,55,59,9,38,59,55,59,9,38,59,55,59,9,38,59,55,59,9,38,59,55,59,9,38,59,55,59,9,38,59,55,59,9,38,59,55,59,9,38,59,55,59,9,38,59,55,59,8,39,59,55,59,9,37,59,55,59,8,39,59,55,59,9,38,59,55,59,8,39,59,55,59,8,39,59,55,47,8,39,38,55,0,8,39,8,55,59,12,31,27,55,43,9,38,43,55,59,9,38,59,55,43,9,39,43,55,59,9,38,59,55,43,9,38,43,55,59,10,35,59,55,43,9,38,59,55,59,9,38,59,55,62,12,31,62,55,62,12,31,78,55,59,9,39,59,55,59,12,31,59,55,43,8,39,43,55,43,9,38,43,55,43,9,38,43,55,45,9,38,61,55,43,9,38,59,55,43,9,38,43,55,43,8,38,43,55,56,8,39,56,55,43,9,38,43,55,43,9,38,43,55,43,8,39,43,55,43,8,39,43,55,43,9,37,59,55,43,8,39,43,55,59,9,39,59,55,59,24,8,59,55,59,9,39,59,55,59,9,38,27],"m90":[59,0,0,0,0,59,63,23,12,63,59,63,8,41,29,59,63,9,40,53,59,63,9,39,63,59,63,9,40,63,59,56,9,40,56,59,63,21,16,36,59,63,21,20,63,59,63,17,20,63,59,57,9,40,51,59,53,9,40,42,59,29,17,24,29,59,36,8,41,12,59,25,21,16,15,59,63,8,41,63,59,63,9,39,63,59,63,9,39,63,59,63,9,39,63,59,63,8,40,63,59,63,11,36,63,59,63,9,39,63,59,63,9,39,63,59,63,9,39,63,59,63,9,40,63,59,63,9,40,63,59,53,21,16,43,59,56,17,24,56,59,57,8,41,51,59,46,8,41,25,59,57,8,41,51,59,63,9,40,63,59,63,9,39,63,59,63,8,41,63,59,63,9,39,63,59,63,9,40,63,59,63,9,39,63,59,63,9,40,63,59,63,9,40,63,59,63,9,40,63,59,63,9,39,63,59,63,9,40,63,59,63,13,32,63,59,63,9,40,63,59,63,9,40,63,59,63,9,39,63,59,63,9,39,63,59,63,9,39,63,59,63,9,39,63,59,63,9,39,63,59,63,9,39,63,59,63,9,39,63,59,63,9,39,63,59,63,9,39,63,59,63,9,39,63,59,63,9,39,63,59,63,8,41,63,59,63,9,39,63,59,63,8,41,63,59,63,9,40,63,59,63,8,41,63,59,63,9,39,63,59,50,8,41,40,59,0,8,41,8,59,63,12,33,29,59,46,9,39,46,59,63,9,39,63,59,46,9,40,46,59,63,9,39,63,59,46,9,39,46,59,63,11,36,63,59,46,9,39,63,59,63,9,39,63,59,67,13,32,67,59,67,13,32,84,59,63,9,40,63,59,63,13,32,63,59,46,8,41,46,59,46,9,39,46,59,46,9,39,46,59,48,9,39,65,59,46,9,39,63,59,46,9,39,46,59,46,9,39,46,59,60,9,40,60,59,46,9,39,46,59,46,9,39,46,59,46,8,41,46,59,46,8,41,46,59,46,9,39,63,59,46,8,41,46,59,63,9,40,63,59,63,25,8,63,59,63,9,40,63,59,63,9,39,28],"a90":[59,0,0,0,0,59,63,23,12,63,59,63,8,41,29,59,63,9,40,53,59,63,9,39,63,59,63,9,40,63,59,56,9,40,56,59,63,21,16,36,59,63,21,20,63,59,63,17,20,63,59,57,9,40,51,59,53,9,40,42,59,29,17,24,29,59,36,8,41,12,59,25,21,16,15,59,63,8,41,63,59,63,9,39,63,59,63,9,39,63,59,63,9,39,63,59,63,8,40,63,59,63,11,36,63,59,63,9,39,63,59,63,9,39,63,59,63,9,39,63,59,63,9,40,63,59,63,9,40,63,59,53,21,16,43,59,56,17,24,56,59,57,8,41,51,59,46,8,41,25,59,57,8,41,51,59,63,9,40,63,59,63,9,39,63,59,63,8,41,63,59,63,9,39,63,59,63,9,40,63,59,63,9,39,63,59,63,9,40,63,59,63,9,40,63,59,63,9,40,63,59,63,9,39,63,59,63,9,40,63,59,63,13,32,63,59,63,9,40,63,59,63,9,40,63,59,63,9,39,63,59,63,9,39,63,59,63,9,39,63,59,63,9,39,63,59,63,9,39,63,59,63,9,39,63,59,63,9,39,63,59,63,9,39,63,59,63,9,39,63,59,63,9,39,63,59,63,9,39,63,59,63,8,41,63,59,63,9,39,63,59,63,8,41,63,59,63,9,40,63,59,63,8,41,63,59,63,9,39,63,59,50,8,41,40,59,0,8,41,8,59,63,12,33,29,59,46,9,39,46,59,63,9,39,63,59,46,9,40,46,59,63,9,39,63,59,46,9,39,46,59,63,11,36,63,59,46,9,39,63,59,63,9,39,63,59,67,13,32,67,59,67,13,32,84,59,63,9,40,63,59,63,13,32,63,59,46,8,41,46,59,46,9,39,46,59,46,9,39,46,59,48,9,39,65,59,46,9,39,63,59,46,9,39,46,59,46,9,39,46,59,60,9,40,60,59,46,9,39,46,59,46,9,39,46,59,46,8,41,46,59,46,8,41,46,59,46,9,39,63,59,46,8,41,46,59,63,9,40,63,59,63,25,8,63,59,63,9,40,63,59,63,9,39,28],"m96":[62,0,0,0,0,62,67,24,13,67,62,67,9,43,32,62,67,9,44,56,62,67,10,42,67,62,67,10,42,67,62,60,10,43,60,62,67,22,17,39,62,67,22,22,67,62,67,18,22,67,62,61,9,43,54,62,56,9,43,45,62,31,18,26,31,62,38,9,43,12,62,27,22,17,16,62,67,9,43,67,62,67,10,42,67,62,67,10,42,67,62,67,10,42,67,62,67,9,43,67,62,67,12,39,67,62,67,9,43,67,62,67,10,42,67,62,67,9,43,67,62,67,10,43,67,62,67,10,42,67,62,56,22,17,45,62,60,18,26,60,62,61,9,43,54,62,49,9,43,27,62,61,9,43,54,62,67,9,44,67,62,67,10,42,67,62,67,9,43,67,62,67,10,42,67,62,67,10,43,67,62,67,9,43,67,62,67,10,43,67,62,67,10,43,67,62,67,10,43,67,62,67,10,42,67,62,67,9,43,67,62,67,14,34,67,62,67,10,43,67,62,67,10,43,67,62,67,10,42,67,62,67,10,42,67,62,67,10,42,67,62,67,10,42,67,62,67,10,42,67,62,67,10,42,67,62,67,10,42,67,62,67,10,42,67,62,67,10,42,67,62,67,10,42,67,62,67,10,42,67,62,67,9,43,67,62,67,10,42,67,62,67,9,43,67,62,67,10,43,67,62,67,9,43,67,62,67,10,42,67,62,53,9,43,43,62,0,9,43,9,62,67,13,35,31,62,49,10,42,49,62,67,10,42,67,62,49,10,43,49,62,67,10,42,67,62,49,10,42,49,62,67,11,39,67,62,49,10,42,67,62,67,10,42,67,62,71,14,34,71,62,71,13,35,89,62,67,10,43,67,62,67,14,34,67,62,49,9,44,49,62,49,10,42,49,62,49,10,42,49,62,51,10,42,69,62,49,10,42,67,62,49,10,42,49,62,49,9,43,49,62,64,9,44,64,62,49,10,42,49,62,49,10,42,49,62,49,9,44,49,62,49,9,43,49,62,49,10,42,67,62,49,9,43,49,62,67,9,43,67,62,67,26,9,67,62,67,9,43,67,62,67,10,42,30],"a96":[62,0,0,0,0,62,67,24,13,67,62,67,9,43,31,62,67,9,44,56,62,67,10,42,67,62,67,10,42,67,62,60,10,43,60,62,67,22,17,39,62,67,22,22,67,62,67,18,22,67,62,61,9,43,54,62,56,9,43,45,62,31,18,26,31,62,38,9,43,12,62,27,22,17,16,62,67,9,43,67,62,67,10,42,67,62,67,10,42,67,62,67,10,42,67,62,67,9,43,67,62,67,12,39,67,62,67,9,43,67,62,67,10,42,67,62,67,9,43,67,62,67,10,43,67,62,67,10,42,67,62,56,22,17,45,62,60,18,26,60,62,61,9,43,54,62,49,9,43,27,62,61,9,43,54,62,67,9,44,67,62,67,10,42,67,62,67,9,43,67,62,67,10,42,67,62,67,10,43,67,62,67,9,43,67,62,67,10,43,67,62,67,10,43,67,62,67,10,43,67,62,67,10,42,67,62,67,9,43,67,62,67,14,34,67,62,67,10,43,67,62,67,10,43,67,62,67,10,42,67,62,67,10,42,67,62,67,10,42,67,62,67,10,42,67,62,67,10,42,67,62,67,10,42,67,62,67,10,42,67,62,67,10,42,67,62,67,10,42,67,62,67,10,42,67,62,67,10,42,67,62,67,9,43,67,62,67,10,42,67,62,67,9,43,67,62,67,10,43,67,62,67,9,43,67,62,67,10,42,67,62,53,9,43,43,62,0,9,43,9,62,67,13,35,31,62,49,10,42,49,62,67,10,42,67,62,49,10,43,49,62,67,10,42,67,62,49,10,42,49,62,67,11,39,67,62,49,10,42,67,62,67,10,42,67,62,71,14,34,71,62,71,13,35,89,62,67,10,43,67,62,67,14,34,67,62,49,9,44,49,62,49,10,42,49,62,49,10,42,49,62,51,10,42,69,62,49,10,42,67,62,49,10,42,49,62,49,9,43,49,62,64,9,44,64,62,49,10,42,49,62,49,10,42,49,62,49,9,44,49,62,49,9,43,49,62,49,10,42,67,62,49,9,43,49,62,67,9,43,67,62,67,26,9,67,62,67,9,43,67,62,67,10,42,30],"m108":[70,0,0,0,0,70,76,27,14,76,70,76,10,49,35,70,76,10,49,64,70,76,11,48,76,70,76,11,47,76,70,68,11,48,68,70,76,25,19,43,70,76,25,25,76,70,76,19,25,76,70,68,11,48,60,70,63,11,48,50,70,35,20,29,35,70,43,10,49,14,70,30,25,19,18,70,76,10,49,76,70,76,11,48,76,70,76,11,48,76,70,76,11,48,76,70,76,11,48,76,70,76,13,43,76,70,76,10,49,76,70,76,11,48,76,70,76,11,48,76,70,76,11,48,76,70,76,11,48,76,70,63,25,19,51,70,67,20,29,67,70,68,10,49,60,70,55,10,49,30,70,68,10,49,60,70,76,10,49,76,70,76,11,48,76,70,76,10,49,76,70,76,11,48,76,70,76,11,48,76,70,76,10,49,76,70,76,11,48,76,70,76,11,48,76,70,76,11,47,76,70,76,11,48,76,70,76,11,48,76,70,76,15,39,76,70,76,11,48,76,70,76,11,48,76,70,76,11,48,76,70,76,11,48,76,70,76,11,48,76,70,76,11,48,76,70,76,11,48,76,70,76,11,48,76,70,76,11,48,76,70,76,11,48,76,70,76,11,48,76,70,76,11,48,76,70,76,11,48,76,70,76,10,49,76,70,76,11,47,76,70,76,10,49,76,70,76,11,48,76,70,76,10,49,76,70,76,10,49,76,70,60,10,49,48,70,0,10,49,10,70,76,15,39,35,70,55,11,48,55,70,76,11,48,76,70,55,11,48,55,70,76,11,48,76,70,55,11,48,55,70,76,12,44,76,70,55,11,48,76,70,76,11,48,76,70,80,15,38,80,70,80,15,39,101,70,76,11,48,76,70,76,15,39,76,70,55,10,50,55,70,55,11,48,55,70,55,11,48,55,70,57,11,48,78,70,55,11,48,76,70,55,11,48,55,70,55,10,48,55,70,72,10,49,72,70,55,11,48,55,70,55,11,48,55,70,55,10,50,55,70,55,10,49,55,70,55,11,47,76,70,55,10,49,55,70,76,11,48,76,70,76,30,10,76,70,76,11,48,76,70,76,11,48,35],"a108":[70,0,0,0,0,70,76,27,14,76,70,76,10,49,35,70,76,10,49,64,70,76,11,48,76,70,76,11,47,76,70,68,11,48,68,70,76,25,19,43,70,76,25,25,76,70,76,19,25,76,70,68,11,48,60,70,63,11,48,50,70,35,20,29,35,70,43,10,49,14,70,30,25,19,18,70,76,10,49,76,70,76,11,48,76,70,76,11,48,76,70,76,11,48,76,70,76,11,48,76,70,76,13,43,76,70,76,10,49,76,70,76,11,48,76,70,76,11,48,76,70,76,11,48,76,70,76,11,48,76,70,63,25,19,51,70,67,20,29,67,70,68,10,49,60,70,55,10,49,30,70,68,10,49,60,70,76,10,49,76,70,76,11,48,76,70,76,10,49,76,70,76,11,48,76,70,76,11,48,76,70,76,10,49,76,70,76,11,48,76,70,76,11,48,76,70,76,11,47,76,70,76,11,48,76,70,76,11,48,76,70,76,15,39,76,70,76,11,48,76,70,76,11,48,76,70,76,11,48,76,70,76,11,48,76,70,76,11,48,76,70,76,11,48,76,70,76,11,48,76,70,76,11,48,76,70,76,11,48,76,70,76,11,48,76,70,76,11,48,76,70,76,11,48,76,70,76,11,48,76,70,76,10,49,76,70,76,11,47,76,70,76,10,49,76,70,76,11,48,76,70,76,10,49,76,70,76,10,49,76,70,60,10,49,48,70,0,10,49,10,70,76,15,39,35,70,55,11,48,55,70,76,11,48,76,70,55,11,48,55,70,76,11,48,76,70,55,11,48,55,70,76,12,44,76,70,55,11,48,76,70,76,11,48,76,70,80,15,38,80,70,80,15,39,101,70,76,11,48,76,70,76,15,39,76,70,55,10,50,55,70,55,11,48,55,70,55,11,48,55,70,57,11,48,78,70,55,11,48,76,70,55,11,48,55,70,55,10,48,55,70,72,10,49,72,70,55,11,48,55,70,55,11,48,55,70,55,10,50,55,70,55,10,49,55,70,55,11,47,76,70,55,10,49,55,70,76,11,48,76,70,76,30,10,76,70,76,11,48,76,70,76,11,48,35]},"fonts":{}};
  if (typeof bwipjs_fonts == "object") {
    bwipjs_fonts.fontsets[0] = desc;
    bwipjs_fonts.names["OCR-A"] = 0;
  } else {
    module.exports = desc;
  }
})();
// fnt1-desc.js
// $ node genfont 1
(function() {
  var desc = {"name":"OCR-B","minsize":8,"maxsize":108,"minchar":32,"maxchar":126,"sizes":{"8":8,"9":9,"10":10,"12":12,"16":16,"18":18,"20":20,"24":24,"27":27,"30":30,"32":32,"36":36,"40":40,"45":45,"48":48,"50":50,"54":54,"56":56,"60":60,"63":63,"64":64,"70":70,"72":72,"80":80,"81":81,"84":84,"90":90,"96":96,"108":108},"metrics":{"m8":[5,0,0,0,0,5,6,2,1,7,5,6,1,3,2,5,6,0,4,6,5,6,1,4,6,5,6,1,4,6,5,6,1,4,6,5,6,2,1,4,5,6,2,3,6,5,6,1,3,6,5,5,0,5,4,5,5,0,5,5,5,2,1,3,3,5,3,0,5,1,5,1,2,2,1,5,6,1,3,6,5,6,1,4,6,5,6,0,3,6,5,6,0,4,6,5,6,0,4,6,5,6,0,5,6,5,6,0,4,6,5,6,1,4,6,5,6,1,4,6,5,6,1,3,6,5,6,1,4,6,5,3,2,2,3,5,3,1,3,5,5,5,0,5,5,5,4,0,5,3,5,5,0,5,5,5,6,1,3,7,5,6,1,3,6,5,5,0,5,5,5,5,1,3,5,5,5,1,3,5,5,5,1,3,5,5,5,1,4,5,5,5,1,3,5,5,5,1,3,5,5,5,1,3,5,5,5,1,3,5,5,5,1,3,5,5,5,1,4,5,5,5,1,4,5,5,5,1,3,5,5,5,1,3,5,5,5,1,3,5,5,5,1,3,5,5,5,1,3,5,5,5,1,3,5,5,5,1,3,5,5,5,0,5,5,5,5,1,3,5,5,5,0,5,5,5,5,1,3,5,5,5,0,5,5,5,5,0,5,5,5,5,1,4,5,5,6,1,4,6,5,6,1,3,6,5,6,0,4,6,5,6,0,5,4,5,-1,0,5,1,5,6,2,1,1,5,4,1,3,4,5,6,1,4,6,5,4,1,3,4,5,6,1,3,6,5,4,1,4,4,5,6,0,5,6,5,4,1,3,5,5,6,1,3,6,5,6,1,2,6,5,6,0,3,7,5,6,1,4,6,5,6,2,3,6,5,4,1,4,4,5,4,1,3,4,5,4,1,4,4,5,4,1,4,5,5,4,1,3,5,5,4,1,3,4,5,4,1,3,4,5,6,1,4,6,5,4,1,3,4,5,4,0,5,4,5,4,1,4,4,5,4,1,4,4,5,4,0,5,5,5,4,1,4,4,5,6,0,5,6,5,6,2,1,7,5,6,0,5,6,5,5,0,5,1],"a8":[5,0,0,0,0,5,6,2,1,7,5,6,1,3,3,5,6,0,4,6,5,6,1,4,6,5,6,1,4,6,5,6,1,4,6,5,6,2,1,4,5,6,2,3,6,5,6,1,3,6,5,5,0,5,4,5,5,0,5,5,5,2,1,3,3,5,3,0,5,1,5,2,2,2,2,5,6,1,3,6,5,6,1,4,6,5,6,0,3,6,5,6,0,4,6,5,6,0,4,6,5,6,0,5,6,5,6,0,4,6,5,6,1,4,6,5,6,1,4,6,5,6,1,3,6,5,6,1,4,6,5,3,2,2,3,5,3,1,3,5,5,5,0,5,5,5,4,0,5,3,5,5,0,5,5,5,6,1,3,7,5,6,1,3,6,5,5,0,5,5,5,5,1,3,5,5,5,1,3,5,5,5,1,3,5,5,5,1,4,5,5,5,1,3,5,5,5,1,3,5,5,5,1,3,5,5,5,1,3,5,5,5,1,3,5,5,5,1,4,5,5,5,1,4,5,5,5,1,3,5,5,5,1,3,5,5,5,1,3,5,5,5,1,3,5,5,5,1,3,5,5,5,1,3,5,5,5,1,3,5,5,5,0,5,5,5,5,1,3,5,5,5,0,5,5,5,5,1,3,5,5,5,0,5,5,5,5,0,5,5,5,5,1,4,5,5,6,1,4,6,5,6,1,3,6,5,6,0,4,6,5,6,0,5,4,5,-1,0,5,1,5,6,2,1,2,5,4,1,3,4,5,6,1,4,6,5,4,1,3,4,5,6,1,3,6,5,4,1,4,4,5,6,0,5,6,5,4,1,3,5,5,6,1,3,6,5,6,1,3,6,5,6,0,3,7,5,6,1,4,6,5,6,2,3,6,5,4,1,4,4,5,4,1,3,4,5,4,1,4,4,5,4,1,4,5,5,4,1,3,5,5,4,1,3,4,5,4,1,3,4,5,6,1,4,6,5,4,1,3,4,5,4,0,5,4,5,4,1,4,4,5,4,1,4,4,5,4,0,5,5,5,4,1,4,4,5,6,0,5,6,5,6,2,1,7,5,6,0,5,6,5,5,0,5,1],"m9":[6,0,0,0,0,6,7,2,1,8,6,6,1,4,3,6,6,1,4,6,6,7,1,4,7,6,6,1,4,6,6,6,1,4,6,6,7,2,1,4,6,6,2,3,6,6,6,1,3,6,6,6,0,5,5,6,6,0,5,5,6,3,1,4,4,6,3,0,5,1,6,2,2,2,2,6,6,1,4,6,6,6,1,4,6,6,7,1,3,7,6,6,1,4,6,6,6,0,5,6,6,7,0,5,7,6,6,1,4,6,6,7,1,4,7,6,6,1,5,6,6,6,1,4,6,6,6,1,4,6,6,4,2,2,4,6,4,1,3,6,6,6,0,5,6,6,4,0,5,2,6,6,0,5,6,6,6,1,4,7,6,6,1,4,6,6,6,0,5,6,6,6,1,4,6,6,6,1,4,6,6,6,1,4,6,6,6,1,4,6,6,6,2,4,6,6,6,1,4,6,6,6,1,4,6,6,6,0,5,6,6,6,1,3,6,6,6,1,5,6,6,6,1,5,6,6,6,1,4,6,6,6,1,4,6,6,6,1,4,6,6,6,1,4,6,6,6,1,4,6,6,6,1,4,6,6,6,1,4,6,6,6,0,5,6,6,6,1,4,6,6,6,0,5,6,6,6,1,4,6,6,6,1,4,6,6,6,0,5,6,6,6,1,4,6,6,6,2,4,6,6,6,1,4,6,6,6,0,4,6,6,7,0,5,5,6,-1,0,5,1,6,6,2,1,2,6,4,1,4,4,6,6,1,4,6,6,4,1,4,4,6,6,1,4,6,6,4,1,4,4,6,6,0,5,6,6,4,1,4,6,6,6,1,4,6,6,6,2,2,6,6,6,1,3,7,6,6,1,4,6,6,6,2,3,6,6,4,1,4,4,6,4,1,4,4,6,4,1,4,4,6,4,1,4,5,6,4,1,4,5,6,4,2,3,4,6,4,1,4,4,6,6,1,4,6,6,4,1,4,4,6,4,0,5,4,6,4,1,4,4,6,4,1,4,4,6,4,0,5,5,6,4,1,4,4,6,6,0,5,6,6,6,2,1,8,6,6,1,5,6,6,6,1,4,2],"a9":[6,0,0,0,0,6,7,2,1,8,6,6,1,4,3,6,6,1,4,6,6,7,1,4,7,6,6,1,4,6,6,6,1,4,6,6,7,2,1,4,6,6,2,3,6,6,6,1,3,6,6,6,0,5,5,6,6,0,5,5,6,3,1,4,4,6,3,0,5,1,6,2,2,2,2,6,6,1,4,6,6,6,1,4,6,6,7,1,3,7,6,6,1,4,6,6,6,0,5,6,6,7,0,5,7,6,6,1,4,6,6,7,1,4,7,6,6,1,5,6,6,6,1,4,6,6,6,1,4,6,6,4,2,2,4,6,4,1,3,6,6,6,0,5,6,6,4,0,5,2,6,6,0,5,6,6,6,1,4,7,6,6,1,4,6,6,6,0,5,6,6,6,1,4,6,6,6,1,4,6,6,6,1,4,6,6,6,1,4,6,6,6,2,4,6,6,6,1,4,6,6,6,1,4,6,6,6,0,5,6,6,6,1,3,6,6,6,1,5,6,6,6,1,5,6,6,6,1,4,6,6,6,1,4,6,6,6,1,4,6,6,6,1,4,6,6,6,1,4,6,6,6,1,4,6,6,6,1,4,6,6,6,0,5,6,6,6,1,4,6,6,6,0,5,6,6,6,1,4,6,6,6,1,4,6,6,6,0,5,6,6,6,1,4,6,6,6,2,4,6,6,6,1,4,6,6,6,0,4,6,6,7,0,5,5,6,-1,0,5,1,6,6,2,1,2,6,4,1,4,4,6,6,1,4,6,6,4,1,4,4,6,6,1,4,6,6,4,1,4,4,6,6,0,5,6,6,4,1,4,6,6,6,1,4,6,6,6,1,3,6,6,6,1,3,7,6,6,1,4,6,6,6,2,3,6,6,4,1,4,4,6,4,1,4,4,6,4,1,4,4,6,4,1,4,5,6,4,1,4,5,6,4,2,3,4,6,4,1,4,4,6,6,1,4,6,6,4,1,4,4,6,4,0,5,4,6,4,1,4,4,6,4,1,4,4,6,4,0,5,5,6,4,1,4,4,6,6,0,5,6,6,6,2,1,8,6,6,1,5,6,6,6,1,4,2],"m10":[7,0,0,0,0,7,7,3,1,8,7,7,1,4,3,7,7,0,5,7,7,7,1,5,7,7,7,1,5,7,7,7,1,5,7,7,7,2,2,4,7,7,2,3,7,7,7,1,3,7,7,6,1,5,5,7,7,1,5,7,7,2,1,4,4,7,4,1,5,1,7,2,2,2,2,7,7,1,4,7,7,7,1,5,7,7,7,1,3,7,7,7,0,5,7,7,7,0,5,7,7,7,1,5,7,7,7,1,4,7,7,7,1,5,7,7,7,0,6,7,7,7,1,4,7,7,7,1,5,7,7,5,2,2,5,7,5,0,4,7,7,7,1,5,7,7,5,1,5,3,7,7,1,5,7,7,7,1,4,8,7,7,1,4,7,7,7,1,5,7,7,7,1,4,7,7,7,1,4,7,7,7,1,4,7,7,7,1,5,7,7,7,2,4,7,7,7,1,4,7,7,7,1,4,7,7,7,1,5,7,7,7,1,3,7,7,7,1,5,7,7,7,1,5,7,7,7,1,4,7,7,7,1,4,7,7,7,1,4,7,7,7,1,4,7,7,7,1,4,7,7,7,1,4,7,7,7,1,4,7,7,7,1,5,7,7,7,1,4,7,7,7,0,6,7,7,7,1,4,7,7,7,1,5,7,7,7,1,5,7,7,7,1,5,7,7,7,2,4,7,7,7,1,4,7,7,7,1,4,7,7,7,1,5,4,7,-1,1,5,1,7,7,2,1,2,7,5,1,4,5,7,7,1,5,7,7,5,1,4,5,7,7,1,4,7,7,5,1,5,5,7,7,1,5,7,7,5,1,4,7,7,7,1,4,7,7,7,2,3,7,7,7,1,3,9,7,7,2,4,7,7,7,2,3,7,7,5,1,5,5,7,5,1,4,5,7,5,1,5,5,7,5,1,5,7,7,5,1,4,7,7,5,2,3,5,7,5,1,4,5,7,7,0,5,7,7,5,1,4,5,7,5,1,5,5,7,5,1,5,5,7,5,1,5,5,7,5,1,5,7,7,5,1,5,5,7,7,1,6,7,7,7,3,1,9,7,7,0,6,7,7,6,1,5,1],"a10":[7,0,0,0,0,7,7,3,1,8,7,7,1,4,3,7,7,0,5,7,7,7,1,5,7,7,7,1,5,7,7,7,1,5,7,7,7,2,2,4,7,7,2,3,7,7,7,1,3,7,7,6,1,5,5,7,7,1,5,7,7,2,1,4,4,7,4,1,5,1,7,2,2,3,2,7,7,1,4,7,7,7,1,5,7,7,7,1,3,7,7,7,0,5,7,7,7,0,5,7,7,7,1,5,7,7,7,1,4,7,7,7,1,5,7,7,7,0,6,7,7,7,1,4,7,7,7,1,5,7,7,5,2,2,5,7,5,1,3,7,7,7,1,5,7,7,5,1,5,3,7,7,1,5,7,7,7,1,4,8,7,7,1,4,7,7,7,1,5,7,7,7,1,4,7,7,7,1,4,7,7,7,1,4,7,7,7,1,5,7,7,7,2,4,7,7,7,1,5,7,7,7,1,4,7,7,7,1,5,7,7,7,1,3,7,7,7,1,5,7,7,7,1,5,7,7,7,1,4,7,7,7,1,4,7,7,7,1,4,7,7,7,1,4,7,7,7,1,4,7,7,7,1,4,7,7,7,1,4,7,7,7,1,5,7,7,7,1,4,7,7,7,0,6,7,7,7,1,4,7,7,7,1,5,7,7,7,1,5,7,7,7,1,5,7,7,7,2,4,7,7,7,1,4,7,7,7,1,4,7,7,7,1,5,4,7,-1,1,5,1,7,7,2,1,2,7,5,1,4,5,7,7,1,5,7,7,5,1,4,5,7,7,1,4,7,7,5,1,5,5,7,7,1,5,7,7,5,1,4,7,7,7,1,4,7,7,7,2,3,7,7,7,1,3,9,7,7,2,4,7,7,7,2,3,7,7,5,1,5,5,7,5,1,4,5,7,5,1,5,5,7,5,1,5,7,7,5,1,4,7,7,5,2,3,5,7,5,1,4,5,7,7,0,5,7,7,5,1,4,5,7,5,1,5,5,7,5,1,5,5,7,5,1,5,5,7,5,1,5,7,7,5,1,5,5,7,7,1,6,7,7,7,3,1,9,7,7,0,6,7,7,6,1,5,1],"m12":[8,0,0,0,0,8,9,3,1,10,8,8,2,4,3,8,8,1,5,8,8,9,1,6,9,8,8,1,6,8,8,8,1,6,8,8,9,3,2,5,8,8,3,4,8,8,8,1,4,8,8,8,0,7,7,8,8,0,7,7,8,3,1,5,5,8,4,1,6,1,8,2,3,3,2,8,8,1,5,8,8,8,1,6,8,8,9,1,4,9,8,8,1,5,8,8,8,0,6,8,8,9,0,7,9,8,8,1,5,8,8,9,1,6,9,8,8,1,6,8,8,8,1,5,8,8,8,1,6,8,8,6,3,2,6,8,6,1,4,8,8,8,1,6,8,8,6,1,6,4,8,8,1,6,8,8,8,2,4,9,8,8,1,5,8,8,8,1,6,8,8,8,1,5,8,8,8,2,4,8,8,8,2,4,8,8,8,2,5,8,8,8,2,4,8,8,8,1,5,8,8,8,2,4,8,8,8,1,5,8,8,8,1,4,8,8,8,1,6,8,8,8,2,5,8,8,8,1,5,8,8,8,1,5,8,8,8,1,5,8,8,8,2,4,8,8,8,1,5,8,8,8,2,4,8,8,8,2,4,8,8,8,0,7,8,8,8,1,5,8,8,8,1,6,8,8,8,1,5,8,8,8,1,6,8,8,8,1,5,8,8,8,1,6,8,8,8,2,5,8,8,8,1,5,8,8,8,1,5,8,8,9,1,6,6,8,-2,1,6,1,8,8,3,1,2,8,6,1,5,6,8,8,1,6,8,8,6,2,4,6,8,8,1,5,8,8,6,1,6,6,8,8,1,5,8,8,6,1,5,8,8,8,2,4,8,8,8,2,3,8,8,8,1,4,10,8,8,2,5,8,8,8,3,4,8,8,6,1,6,6,8,6,1,5,6,8,6,1,6,6,8,6,1,6,8,8,6,1,5,8,8,6,2,4,6,8,6,2,4,6,8,8,1,6,8,8,6,1,5,6,8,6,1,6,6,8,6,1,6,6,8,6,1,6,6,8,6,1,6,8,8,6,1,6,6,8,8,1,6,8,8,8,3,1,11,8,8,1,6,8,8,8,1,6,2],"a12":[8,0,0,0,0,8,9,3,2,10,8,8,1,5,4,8,8,1,6,8,8,9,1,6,9,8,8,1,6,8,8,8,1,6,8,8,9,3,2,5,8,8,2,5,8,8,8,1,5,8,8,8,0,7,7,8,8,0,7,7,8,3,1,5,5,8,5,1,6,2,8,3,3,3,3,8,8,1,5,8,8,8,1,6,8,8,9,1,4,9,8,8,1,6,8,8,9,0,7,9,8,9,0,7,9,8,9,1,6,9,8,9,1,6,9,8,9,1,6,9,8,8,1,6,8,8,8,1,6,8,8,6,2,3,6,8,6,1,5,8,8,8,1,6,8,8,6,1,6,4,8,8,1,6,8,8,8,1,5,9,8,8,1,6,8,8,8,1,6,8,8,8,1,6,8,8,8,1,5,8,8,8,1,6,8,8,8,1,6,8,8,8,2,4,8,8,8,1,5,8,8,8,1,6,8,8,8,1,5,8,8,8,1,5,8,8,8,1,6,8,8,8,1,6,8,8,8,1,6,8,8,8,1,6,8,8,8,1,6,8,8,8,1,6,8,8,8,1,6,8,8,8,1,6,8,8,8,1,6,8,8,8,0,7,8,8,8,1,6,8,8,8,1,6,8,8,8,1,6,8,8,8,1,6,8,8,8,1,5,8,8,8,1,6,8,8,8,2,5,8,8,8,1,5,8,8,8,1,5,8,8,9,1,6,6,8,-1,1,6,2,8,8,2,2,3,8,6,1,6,6,8,8,1,6,8,8,6,1,6,6,8,8,1,6,8,8,6,1,6,6,8,8,1,5,8,8,6,1,6,9,8,8,1,6,8,8,8,2,3,8,8,8,1,4,10,8,8,1,6,8,8,8,2,5,8,8,6,1,6,6,8,6,1,6,6,8,6,1,6,6,8,6,1,6,8,8,6,1,6,8,8,6,2,5,6,8,6,1,6,6,8,8,1,6,8,8,6,1,6,6,8,6,1,6,6,8,6,1,6,6,8,6,1,6,6,8,6,1,6,8,8,6,1,6,6,8,8,1,6,8,8,8,3,2,11,8,8,1,6,8,8,8,1,6,3],"m16":[10,0,0,0,0,10,12,4,2,13,10,11,2,6,5,10,11,1,7,11,10,12,2,6,12,10,11,2,6,11,10,11,2,6,11,10,12,4,3,7,10,11,4,4,11,10,11,2,4,11,10,10,2,7,8,10,10,2,7,9,10,3,2,6,6,10,6,1,8,1,10,3,3,4,3,10,11,2,6,11,10,11,2,6,11,10,12,1,5,12,10,11,1,7,11,10,11,1,7,11,10,12,1,8,12,10,11,2,6,11,10,12,2,6,12,10,11,2,7,11,10,11,2,6,11,10,11,2,6,11,10,8,4,3,8,10,8,2,5,11,10,11,1,8,11,10,8,1,8,5,10,11,1,8,11,10,11,2,6,12,10,11,2,6,11,10,11,1,8,11,10,11,2,6,11,10,11,2,6,11,10,11,2,6,11,10,11,2,6,11,10,11,3,5,11,10,11,2,7,11,10,11,2,6,11,10,11,1,7,11,10,11,2,5,11,10,11,2,7,11,10,11,2,7,11,10,11,2,6,11,10,11,2,6,11,10,11,2,6,11,10,11,2,6,11,10,11,2,6,11,10,11,2,6,11,10,11,2,6,11,10,11,1,7,11,10,11,2,6,11,10,11,1,8,11,10,11,2,6,11,10,11,1,8,11,10,11,1,7,11,10,11,1,8,11,10,11,3,6,11,10,11,2,6,11,10,11,1,6,11,10,12,1,8,7,10,-2,1,8,1,10,11,4,2,3,10,8,2,6,8,10,11,2,6,11,10,8,2,6,8,10,11,2,6,11,10,8,2,7,8,10,11,1,7,11,10,8,2,6,11,10,11,2,6,11,10,11,3,4,11,10,11,2,5,14,10,11,3,6,11,10,11,4,4,11,10,8,2,6,8,10,8,2,6,8,10,8,2,6,8,10,8,2,6,11,10,8,2,6,11,10,8,3,5,8,10,8,2,6,8,10,10,2,6,10,10,8,2,6,8,10,8,1,8,8,10,8,2,6,8,10,8,2,7,8,10,8,1,8,11,10,7,2,7,7,10,11,1,8,11,10,11,5,1,14,10,11,1,8,11,10,10,1,8,2],"a16":[10,0,0,0,0,10,12,4,3,13,10,11,1,7,5,10,11,1,8,11,10,12,1,8,12,10,11,1,8,11,10,11,1,8,11,10,12,4,3,7,10,11,3,5,11,10,11,2,5,11,10,10,1,8,8,10,10,1,8,9,10,3,2,6,6,10,6,1,8,2,10,3,3,4,3,10,11,2,6,11,10,11,1,8,11,10,12,1,6,12,10,11,1,8,11,10,11,1,8,11,10,12,1,8,12,10,11,2,6,11,10,12,1,8,12,10,11,2,7,11,10,11,1,8,11,10,11,1,8,11,10,8,4,3,8,10,8,2,5,11,10,11,1,8,11,10,8,1,8,5,10,11,1,8,11,10,11,2,6,12,10,11,1,8,11,10,11,1,8,11,10,11,1,8,11,10,11,2,6,11,10,11,2,6,11,10,11,2,6,11,10,11,3,6,11,10,11,1,8,11,10,11,2,6,11,10,11,1,7,11,10,11,1,6,11,10,11,1,8,11,10,11,2,7,11,10,11,1,8,11,10,11,1,8,11,10,11,1,8,11,10,11,2,7,11,10,11,1,7,11,10,11,2,6,11,10,11,2,6,11,10,11,1,7,11,10,11,1,8,11,10,11,1,8,11,10,11,1,8,11,10,11,1,8,11,10,11,1,7,11,10,11,1,8,11,10,11,3,6,11,10,11,2,6,11,10,11,1,7,11,10,12,1,8,7,10,-2,1,8,2,10,11,3,3,3,10,8,1,7,8,10,11,1,8,11,10,8,2,6,8,10,11,1,8,11,10,8,2,7,8,10,11,1,7,11,10,8,1,8,11,10,11,2,6,11,10,11,3,5,11,10,11,2,5,14,10,11,2,7,11,10,11,3,5,11,10,8,1,8,8,10,8,1,8,8,10,8,1,8,8,10,8,1,8,11,10,8,1,8,11,10,8,3,6,8,10,8,2,6,8,10,10,2,6,10,10,8,1,8,8,10,8,1,8,8,10,8,1,8,8,10,8,2,7,8,10,8,1,8,11,10,8,2,7,8,10,11,1,8,11,10,11,4,2,14,10,11,1,8,11,10,10,1,8,3],"m18":[12,0,0,0,0,12,13,5,2,14,12,12,3,6,5,12,12,2,8,12,12,13,2,8,13,12,12,2,8,12,12,12,2,9,12,12,13,4,3,7,12,12,4,6,12,12,12,2,6,12,12,11,1,10,9,12,11,1,10,10,12,4,2,7,7,12,7,1,9,2,12,3,4,4,3,12,12,2,7,12,12,12,2,8,12,12,13,2,6,13,12,12,2,8,12,12,13,1,9,13,12,13,2,9,13,12,13,2,8,13,12,13,2,8,13,12,13,1,10,13,12,12,2,9,12,12,12,2,8,12,12,9,4,3,9,12,9,3,5,12,12,12,1,9,11,12,9,1,9,6,12,12,1,9,11,12,13,2,7,14,12,13,2,8,13,12,12,1,9,12,12,12,2,8,12,12,12,2,7,12,12,12,3,7,12,12,12,2,8,12,12,12,3,7,12,12,12,2,7,12,12,12,2,8,12,12,12,2,8,12,12,12,2,6,12,12,12,2,9,12,12,12,2,8,12,12,12,2,8,12,12,12,2,8,12,12,12,2,8,12,12,12,2,8,12,12,12,2,8,12,12,12,2,8,12,12,12,2,8,12,12,12,1,10,12,12,12,2,8,12,12,12,1,9,12,12,12,2,8,12,12,12,2,8,12,12,12,2,8,12,12,12,2,8,12,12,12,3,7,12,12,12,2,7,12,12,12,2,7,12,12,13,1,9,8,12,-2,1,9,2,12,12,4,3,3,12,9,2,8,9,12,12,2,8,12,12,9,3,7,9,12,12,2,8,12,12,9,2,8,9,12,12,2,8,12,12,9,2,8,12,12,12,2,8,12,12,12,3,5,12,12,12,2,5,15,12,12,3,8,12,12,12,4,6,12,12,9,2,8,9,12,9,2,8,9,12,9,2,8,9,12,9,2,8,12,12,9,2,8,12,12,9,3,7,9,12,9,3,7,9,12,12,2,8,12,12,9,2,8,9,12,9,1,9,9,12,9,2,8,9,12,9,2,8,9,12,9,1,9,12,12,9,2,8,9,12,12,2,9,12,12,12,5,2,16,12,12,1,9,12,12,11,2,8,2],"a18":[12,0,0,0,0,12,13,4,3,15,12,12,3,6,6,12,12,2,8,12,12,13,2,8,13,12,12,2,8,12,12,12,2,8,12,12,13,4,3,7,12,12,4,6,12,12,12,2,6,12,12,11,1,10,9,12,11,1,10,10,12,4,2,7,7,12,7,1,9,2,12,4,4,4,4,12,12,2,7,12,12,12,2,8,12,12,13,2,6,13,12,12,2,8,12,12,13,1,9,13,12,13,2,9,13,12,13,2,8,13,12,13,2,8,13,12,13,1,10,13,12,12,2,9,12,12,12,2,8,12,12,9,4,3,9,12,9,3,5,12,12,12,1,9,11,12,9,1,9,6,12,12,1,9,11,12,13,2,7,14,12,13,2,8,13,12,12,1,9,12,12,12,2,8,12,12,12,2,7,12,12,12,3,7,12,12,12,2,8,12,12,12,3,7,12,12,12,2,7,12,12,12,2,8,12,12,12,2,8,12,12,12,2,6,12,12,12,2,8,12,12,12,2,8,12,12,12,2,8,12,12,12,2,8,12,12,12,2,8,12,12,12,2,8,12,12,12,2,8,12,12,12,2,8,12,12,12,2,8,12,12,12,1,10,12,12,12,2,8,12,12,12,1,9,12,12,12,2,8,12,12,12,2,8,12,12,12,2,8,12,12,12,2,8,12,12,12,3,7,12,12,12,2,7,12,12,12,2,7,12,12,13,1,9,8,12,-2,1,9,2,12,12,4,3,4,12,9,2,8,9,12,12,2,8,12,12,9,3,7,9,12,12,2,8,12,12,9,2,8,9,12,12,2,8,12,12,9,2,8,12,12,12,2,8,12,12,12,3,5,12,12,12,2,5,15,12,12,3,7,12,12,12,4,6,12,12,9,2,8,9,12,9,2,8,9,12,9,2,8,9,12,9,2,8,12,12,9,2,8,12,12,9,3,7,9,12,9,3,7,9,12,12,2,8,12,12,9,2,8,9,12,9,1,9,9,12,9,2,8,9,12,9,2,8,9,12,9,1,9,12,12,9,2,8,9,12,12,2,9,12,12,12,5,2,16,12,12,1,9,12,12,11,2,8,2],"m20":[13,0,0,0,0,13,14,5,3,15,13,14,3,7,6,13,14,1,10,14,13,14,2,9,14,13,14,2,9,14,13,14,2,10,14,13,14,5,3,8,13,14,5,6,14,13,14,3,6,14,13,12,2,10,10,13,13,2,10,12,13,5,2,8,8,13,8,2,10,2,13,4,4,4,4,13,14,3,7,14,13,14,2,9,14,13,14,2,6,14,13,14,2,9,14,13,14,2,9,14,13,14,1,10,14,13,14,1,9,14,13,14,2,9,14,13,14,1,10,14,13,14,2,10,14,13,14,2,9,14,13,9,5,4,9,13,9,2,7,13,13,13,2,10,12,13,10,2,10,6,13,13,2,10,12,13,15,2,8,16,13,15,2,9,15,13,14,2,9,14,13,14,2,9,14,13,14,3,7,14,13,14,3,8,14,13,14,3,9,14,13,14,4,7,14,13,14,2,9,14,13,14,2,9,14,13,14,3,8,14,13,14,2,7,14,13,14,2,9,14,13,14,2,9,14,13,14,2,9,14,13,14,2,9,14,13,14,2,9,14,13,14,2,9,14,13,14,2,9,14,13,14,3,8,14,13,14,3,8,14,13,14,2,10,14,13,14,2,9,14,13,14,2,10,14,13,14,2,9,14,13,14,2,9,14,13,14,2,10,14,13,14,2,9,14,13,14,4,8,14,13,14,3,7,14,13,14,1,8,14,13,14,2,10,8,13,-2,2,10,2,13,14,5,3,3,13,10,2,8,10,13,14,2,9,14,13,10,3,7,10,13,14,2,9,14,13,10,2,9,10,13,14,2,8,14,13,10,2,9,13,13,14,3,7,14,13,14,3,5,14,13,14,2,7,17,13,14,3,8,14,13,14,4,6,14,13,10,2,9,10,13,10,2,9,10,13,10,2,9,10,13,10,2,9,13,13,10,2,9,13,13,10,4,7,10,13,10,3,8,10,13,13,1,9,13,13,10,2,9,10,13,10,2,9,10,13,10,2,9,10,13,10,2,9,10,13,10,2,10,13,13,10,2,9,10,13,14,1,11,14,13,14,6,2,17,13,14,1,11,14,13,13,2,9,3],"a20":[13,0,0,0,0,13,14,5,3,15,13,14,3,7,6,13,14,1,10,14,13,14,2,9,14,13,14,2,9,14,13,14,2,10,14,13,14,5,3,8,13,14,5,6,14,13,14,3,6,14,13,12,2,10,10,13,12,2,10,11,13,5,2,8,8,13,8,2,10,2,13,4,4,4,4,13,14,3,7,14,13,14,2,9,14,13,14,2,6,14,13,14,2,9,14,13,14,2,9,14,13,14,1,10,14,13,14,2,8,14,13,14,2,9,14,13,14,1,10,14,13,14,2,10,14,13,14,2,9,14,13,9,5,4,9,13,9,2,7,13,13,13,2,10,12,13,10,2,10,6,13,13,2,10,12,13,15,2,8,16,13,15,2,9,15,13,14,2,9,14,13,14,2,9,14,13,14,3,7,14,13,14,3,8,14,13,14,3,8,14,13,14,4,7,14,13,14,2,9,14,13,14,2,9,14,13,14,3,8,14,13,14,2,7,14,13,14,2,9,14,13,14,2,9,14,13,14,2,9,14,13,14,2,9,14,13,14,2,9,14,13,14,2,9,14,13,14,2,9,14,13,14,3,8,14,13,14,3,8,14,13,14,2,10,14,13,14,2,9,14,13,14,2,10,14,13,14,2,9,14,13,14,2,9,14,13,14,2,10,14,13,14,2,9,14,13,14,4,7,14,13,14,3,7,14,13,14,2,7,14,13,14,2,10,8,13,-2,2,10,2,13,14,5,3,4,13,10,2,8,10,13,14,2,9,14,13,10,3,7,10,13,14,2,9,14,13,10,2,9,10,13,14,2,8,14,13,10,2,9,13,13,14,3,7,14,13,14,3,5,14,13,14,2,7,17,13,14,3,8,14,13,14,4,6,14,13,10,2,9,10,13,10,2,9,10,13,10,2,9,10,13,10,2,9,13,13,10,2,9,13,13,10,4,7,10,13,10,3,8,10,13,13,1,9,13,13,10,2,9,10,13,10,2,9,10,13,10,2,9,10,13,10,2,9,10,13,10,2,10,13,13,10,2,9,10,13,14,2,10,14,13,14,6,2,17,13,14,2,10,14,13,13,2,9,3],"m24":[16,0,0,0,0,16,17,6,3,19,16,17,3,9,7,16,17,2,11,17,16,17,3,10,17,16,17,3,10,17,16,17,3,10,17,16,17,6,4,9,16,17,6,7,17,16,17,3,7,17,16,15,2,12,12,16,14,2,12,12,16,5,3,9,9,16,9,2,12,2,16,4,5,5,4,16,17,3,9,17,16,17,3,10,17,16,17,3,7,17,16,17,3,10,17,16,16,2,11,16,16,17,2,12,17,16,16,2,10,16,16,17,3,10,17,16,16,2,11,16,16,17,3,10,17,16,17,3,10,17,16,12,6,4,12,16,12,3,7,16,16,16,2,12,15,16,12,2,12,7,16,16,2,12,15,16,17,3,9,19,16,17,2,11,17,16,17,2,11,17,16,17,2,11,17,16,17,3,9,17,16,17,4,8,17,16,17,3,10,17,16,17,4,8,17,16,17,3,10,17,16,17,3,9,17,16,17,3,10,17,16,17,3,7,17,16,17,3,11,17,16,17,3,10,17,16,17,2,11,17,16,17,3,10,17,16,17,3,10,17,16,17,3,10,17,16,17,2,11,17,16,17,3,9,17,16,17,3,9,17,16,17,2,12,17,16,17,3,10,17,16,17,2,11,17,16,17,2,11,17,16,17,2,11,17,16,17,3,10,17,16,17,2,11,17,16,17,5,9,17,16,17,3,9,17,16,17,2,9,17,16,17,2,12,10,16,-2,2,12,3,16,17,6,3,4,16,12,3,9,12,16,17,3,10,17,16,12,4,8,12,16,17,3,10,17,16,12,3,11,12,16,17,3,10,17,16,12,3,10,16,16,17,3,9,17,16,17,4,6,17,16,17,3,7,21,16,17,4,9,17,16,17,6,7,17,16,12,3,10,12,16,12,3,10,12,16,12,3,10,12,16,12,3,10,16,16,12,3,10,16,16,12,4,9,12,16,12,4,8,12,16,15,2,10,15,16,12,3,10,12,16,12,2,12,12,16,12,3,10,12,16,12,3,10,12,16,12,2,12,16,16,11,3,10,11,16,17,2,11,17,16,17,7,2,21,16,17,2,11,17,16,15,2,11,3],"a24":[16,0,0,0,0,16,17,6,3,19,16,17,3,9,7,16,17,2,11,17,16,17,2,12,17,16,17,2,12,17,16,17,2,11,17,16,17,6,4,9,16,17,5,8,17,16,17,3,8,17,16,15,2,12,12,16,15,2,12,13,16,5,3,9,9,16,9,2,12,3,16,5,5,5,5,16,17,3,9,17,16,17,2,12,17,16,17,3,7,17,16,17,3,10,17,16,17,2,11,17,16,17,2,12,17,16,17,2,11,17,16,17,2,12,17,16,17,2,12,17,16,17,2,11,17,16,17,2,12,17,16,12,6,4,12,16,12,3,7,16,16,16,2,12,15,16,12,2,12,8,16,16,2,12,15,16,17,3,9,19,16,17,2,11,17,16,17,2,11,17,16,17,2,12,17,16,17,3,9,17,16,17,3,10,17,16,17,3,10,17,16,17,4,8,17,16,17,2,11,17,16,17,3,10,17,16,17,3,10,17,16,17,2,9,17,16,17,2,12,17,16,17,3,10,17,16,17,2,11,17,16,17,2,11,17,16,17,2,11,17,16,17,3,10,17,16,17,2,11,17,16,17,3,10,17,16,17,3,10,17,16,17,2,12,17,16,17,2,11,17,16,17,2,11,17,16,17,2,11,17,16,17,2,11,17,16,17,3,10,17,16,17,2,11,17,16,17,4,10,17,16,17,3,9,17,16,17,2,10,17,16,17,2,12,10,16,-2,2,12,3,16,17,6,3,5,16,12,2,11,12,16,17,2,12,17,16,12,3,10,12,16,17,2,11,17,16,12,3,11,12,16,17,3,10,17,16,12,2,11,17,16,17,3,10,17,16,17,4,6,17,16,17,3,8,21,16,17,3,10,17,16,17,5,8,17,16,12,2,12,12,16,12,2,11,12,16,12,2,12,12,16,12,2,12,16,16,12,2,11,16,16,12,4,9,12,16,12,3,10,12,16,15,2,10,15,16,12,2,11,12,16,12,2,12,12,16,12,2,12,12,16,12,3,10,12,16,12,2,12,16,16,12,3,10,12,16,17,2,12,17,16,17,6,3,21,16,17,2,11,17,16,15,2,11,4],"m27":[18,0,0,0,0,18,19,7,3,21,18,19,4,9,8,18,19,2,12,19,18,19,3,11,19,18,19,3,11,19,18,19,3,11,19,18,19,7,4,10,18,19,6,7,19,18,19,4,7,19,18,16,3,12,13,18,16,3,12,14,18,6,3,10,10,18,11,2,13,3,18,5,6,6,5,18,19,4,10,19,18,19,3,11,19,18,19,3,8,19,18,19,3,11,19,18,18,2,12,18,18,19,2,13,19,18,18,3,10,18,18,19,3,11,19,18,18,3,12,18,18,19,3,11,19,18,19,3,11,19,18,13,6,5,13,18,13,4,8,17,18,17,2,13,16,18,13,2,13,8,18,17,2,13,16,18,20,3,10,22,18,20,3,12,20,18,19,2,13,19,18,19,3,11,19,18,19,4,9,19,18,19,4,10,19,18,19,4,11,19,18,19,5,9,19,18,19,3,11,19,18,19,3,11,19,18,19,4,10,19,18,19,3,9,19,18,19,3,12,19,18,19,3,12,19,18,19,3,12,19,18,19,3,11,19,18,19,3,11,19,18,19,3,12,19,18,19,3,12,19,18,19,4,10,19,18,19,4,10,19,18,19,3,12,19,18,19,3,11,19,18,19,2,13,19,18,19,3,12,19,18,19,3,12,19,18,19,3,12,19,18,19,3,12,19,18,19,5,9,19,18,19,4,10,19,18,19,3,9,19,18,19,2,13,11,18,-3,2,13,3,18,19,6,4,5,18,13,3,10,13,18,19,3,11,19,18,13,4,9,13,18,19,3,11,19,18,13,3,12,13,18,19,3,10,19,18,13,3,11,18,18,19,4,9,19,18,19,4,7,19,18,19,3,8,23,18,19,4,10,19,18,19,6,7,19,18,13,3,11,13,18,13,3,11,13,18,13,3,11,13,18,13,3,11,17,18,13,3,11,17,18,13,5,9,13,18,13,4,9,13,18,16,3,10,16,18,13,3,11,13,18,13,2,13,13,18,13,3,11,13,18,13,3,11,13,18,13,2,13,17,18,12,3,11,12,18,19,2,13,19,18,19,8,2,23,18,19,2,13,19,18,17,3,12,4],"a27":[18,0,0,0,0,18,19,7,3,21,18,19,4,9,8,18,19,2,13,19,18,19,2,13,19,18,19,2,13,19,18,19,2,13,19,18,19,7,4,10,18,19,6,7,19,18,19,4,7,19,18,16,3,12,13,18,16,3,12,14,18,6,3,10,10,18,11,2,13,3,18,5,6,6,5,18,19,4,10,19,18,19,2,13,19,18,19,3,8,19,18,19,3,11,19,18,19,2,13,19,18,19,2,13,19,18,19,3,11,19,18,19,2,13,19,18,19,2,13,19,18,19,2,12,19,18,19,2,13,19,18,13,6,5,13,18,13,4,8,17,18,17,2,13,16,18,13,2,13,8,18,17,2,13,16,18,20,3,11,22,18,20,2,13,20,18,19,2,13,19,18,19,2,13,19,18,19,3,11,19,18,19,4,10,19,18,19,3,12,19,18,19,5,9,19,18,19,3,11,19,18,19,3,11,19,18,19,4,10,19,18,19,3,9,19,18,19,3,12,19,18,19,3,12,19,18,19,2,13,19,18,19,3,12,19,18,19,2,13,19,18,19,3,12,19,18,19,2,13,19,18,19,3,11,19,18,19,3,11,19,18,19,2,13,19,18,19,3,12,19,18,19,2,13,19,18,19,2,13,19,18,19,3,12,19,18,19,3,12,19,18,19,3,12,19,18,19,5,9,19,18,19,4,10,19,18,19,3,9,19,18,19,2,13,11,18,-3,2,13,3,18,19,6,4,5,18,13,3,11,13,18,19,3,12,19,18,13,4,10,13,18,19,2,13,19,18,13,3,12,13,18,19,3,11,19,18,13,2,12,18,18,19,3,11,19,18,19,4,7,19,18,19,3,8,23,18,19,4,10,19,18,19,6,7,19,18,13,2,13,13,18,13,3,11,13,18,13,2,13,13,18,13,3,12,17,18,13,2,13,17,18,13,4,10,13,18,13,3,11,13,18,16,3,10,16,18,13,3,11,13,18,13,2,13,13,18,13,2,13,13,18,13,3,11,13,18,13,2,13,17,18,13,3,11,13,18,19,2,13,19,18,19,7,3,23,18,19,2,13,19,18,17,3,12,4],"m30":[20,0,0,0,0,20,21,8,4,23,20,21,4,11,9,20,21,2,14,21,20,21,3,14,21,20,21,3,14,21,20,21,3,13,21,20,21,7,5,11,20,21,7,8,21,20,21,5,8,21,20,18,2,15,14,20,19,2,15,17,20,6,4,11,11,20,12,2,15,3,20,5,6,6,5,20,21,4,11,21,20,21,3,14,21,20,21,3,9,21,20,21,3,13,21,20,21,2,14,21,20,21,3,14,21,20,21,3,12,21,20,21,3,14,21,20,21,2,15,21,20,21,3,14,21,20,21,3,14,21,20,15,7,6,15,20,15,3,10,20,20,19,2,15,18,20,14,2,15,8,20,19,2,15,18,20,21,4,11,23,20,21,3,14,21,20,21,3,14,21,20,21,3,14,21,20,21,4,11,21,20,21,4,12,21,20,21,4,13,21,20,21,5,11,21,20,21,3,13,21,20,21,4,12,21,20,21,4,11,21,20,21,3,10,21,20,21,3,14,21,20,21,4,13,21,20,21,3,14,21,20,21,3,13,21,20,21,3,13,21,20,21,4,13,21,20,21,3,14,21,20,21,4,12,21,20,21,4,12,21,20,21,2,15,21,20,21,3,13,21,20,21,2,15,21,20,21,3,13,21,20,21,3,13,21,20,21,3,13,21,20,21,3,13,21,20,21,5,11,21,20,21,4,11,21,20,21,3,11,21,20,21,2,15,12,20,-3,2,15,3,20,21,7,4,5,20,14,3,13,14,20,21,3,14,21,20,14,4,11,14,20,21,3,13,21,20,14,3,13,14,20,21,3,13,21,20,14,3,13,19,20,21,4,11,21,20,21,5,7,21,20,21,4,9,26,20,21,5,12,21,20,21,7,9,21,20,14,3,14,14,20,14,3,13,14,20,14,3,14,14,20,14,3,14,19,20,14,3,13,19,20,14,5,11,14,20,14,4,12,14,20,19,3,13,19,20,14,3,13,14,20,14,3,14,14,20,14,3,13,14,20,14,3,13,14,20,14,2,15,19,20,14,3,13,14,20,21,2,14,21,20,21,8,3,26,20,21,2,15,21,20,19,3,13,5],"a30":[20,0,0,0,0,20,21,8,4,23,20,21,4,11,9,20,21,2,14,21,20,21,3,14,21,20,21,3,14,21,20,21,3,14,21,20,21,7,5,11,20,21,7,8,21,20,21,5,8,21,20,18,2,15,14,20,18,2,15,16,20,6,4,11,11,20,12,2,15,3,20,6,6,6,6,20,21,4,11,21,20,21,3,14,21,20,21,3,9,21,20,21,3,13,21,20,21,2,14,21,20,21,3,14,21,20,21,3,12,21,20,21,3,14,21,20,21,2,15,21,20,21,3,14,21,20,21,3,14,21,20,15,7,6,15,20,15,3,10,20,20,19,2,15,18,20,14,2,15,8,20,19,2,15,18,20,21,4,11,23,20,21,3,14,21,20,21,3,14,21,20,21,3,14,21,20,21,4,11,21,20,21,4,12,21,20,21,4,13,21,20,21,5,11,21,20,21,3,13,21,20,21,4,12,21,20,21,4,11,21,20,21,3,10,21,20,21,3,14,21,20,21,4,13,21,20,21,3,14,21,20,21,3,13,21,20,21,3,13,21,20,21,4,13,21,20,21,3,14,21,20,21,4,12,21,20,21,4,12,21,20,21,2,15,21,20,21,3,13,21,20,21,2,15,21,20,21,3,13,21,20,21,3,13,21,20,21,3,13,21,20,21,3,13,21,20,21,5,11,21,20,21,4,11,21,20,21,3,11,21,20,21,2,15,12,20,-3,2,15,3,20,21,7,4,6,20,14,3,13,14,20,21,3,14,21,20,14,4,11,14,20,21,3,13,21,20,14,3,13,14,20,21,3,13,21,20,14,3,13,19,20,21,4,11,21,20,21,5,7,21,20,21,4,9,26,20,21,5,12,21,20,21,7,9,21,20,14,3,14,14,20,14,3,13,14,20,14,3,14,14,20,14,3,14,19,20,14,3,13,19,20,14,5,11,14,20,14,4,12,14,20,19,3,13,19,20,14,3,13,14,20,14,3,14,14,20,14,3,13,14,20,14,3,13,14,20,14,2,15,19,20,14,3,13,14,20,21,2,14,21,20,21,8,3,26,20,21,2,15,21,20,19,3,13,5],"m32":[21,0,0,0,0,21,23,8,4,25,21,22,5,11,9,21,22,3,14,22,21,23,3,14,23,21,22,3,14,22,21,22,3,14,22,21,23,8,5,12,21,22,7,9,22,21,22,4,9,22,21,19,3,15,15,21,19,3,15,17,21,7,4,12,12,21,13,3,15,4,21,6,7,7,6,21,22,4,12,22,21,22,3,14,22,21,23,4,9,23,21,22,3,14,22,21,22,3,14,22,21,23,3,15,23,21,22,3,13,22,21,23,3,14,23,21,22,3,15,22,21,22,3,14,22,21,22,3,14,22,21,15,7,6,15,21,15,4,10,21,21,21,3,15,20,21,15,3,15,9,21,21,3,15,20,21,23,4,12,25,21,23,3,14,23,21,22,3,15,22,21,22,3,14,22,21,22,4,12,22,21,22,5,12,22,21,22,4,13,22,21,22,6,11,22,21,22,3,13,22,21,22,4,13,22,21,22,4,13,22,21,22,4,10,22,21,22,4,14,22,21,22,4,14,22,21,22,3,14,22,21,22,4,13,22,21,22,3,14,22,21,22,4,13,22,21,22,3,15,22,21,22,4,13,22,21,22,4,12,22,21,22,3,15,22,21,22,4,13,22,21,22,3,15,22,21,22,3,14,22,21,22,3,14,22,21,22,4,13,22,21,22,3,14,22,21,22,6,12,22,21,22,4,12,22,21,22,3,12,22,21,23,3,15,13,21,-3,3,15,4,21,22,7,4,5,21,15,4,12,15,21,22,3,14,22,21,15,5,11,15,21,22,3,14,22,21,15,4,14,15,21,22,4,13,22,21,15,3,14,21,21,22,4,12,22,21,22,5,8,22,21,22,4,9,27,21,22,5,12,22,21,22,7,9,22,21,15,3,14,15,21,15,4,13,15,21,15,3,14,15,21,15,3,14,20,21,15,3,14,20,21,15,6,11,15,21,15,4,12,15,21,20,3,13,20,21,15,4,13,15,21,15,3,15,15,21,15,3,14,15,21,15,4,13,15,21,15,3,15,20,21,15,4,13,15,21,22,3,15,22,21,22,9,3,27,21,22,3,15,22,21,20,3,14,5],"a32":[21,0,0,0,0,21,23,8,4,25,21,22,5,11,10,21,22,3,14,22,21,23,3,14,23,21,22,3,14,22,21,22,3,14,22,21,23,8,5,12,21,22,7,9,22,21,22,4,9,22,21,19,3,15,15,21,19,3,15,17,21,7,4,12,12,21,13,3,15,4,21,6,7,7,6,21,22,4,12,22,21,22,3,14,22,21,23,4,9,23,21,22,3,14,22,21,22,3,14,22,21,23,3,15,23,21,22,3,13,22,21,23,3,14,23,21,22,3,15,22,21,22,3,14,22,21,22,3,14,22,21,15,7,6,15,21,15,4,10,21,21,21,3,15,20,21,15,3,15,9,21,21,3,15,20,21,23,4,12,25,21,23,3,14,23,21,22,3,15,22,21,22,3,14,22,21,22,4,12,22,21,22,5,12,22,21,22,4,13,22,21,22,6,11,22,21,22,3,13,22,21,22,4,13,22,21,22,4,12,22,21,22,4,10,22,21,22,4,14,22,21,22,4,14,22,21,22,3,14,22,21,22,4,13,22,21,22,3,14,22,21,22,4,13,22,21,22,3,15,22,21,22,4,13,22,21,22,4,12,22,21,22,3,15,22,21,22,4,13,22,21,22,3,15,22,21,22,3,14,22,21,22,3,14,22,21,22,4,13,22,21,22,3,14,22,21,22,6,11,22,21,22,4,12,22,21,22,3,12,22,21,23,3,15,13,21,-3,3,15,4,21,22,7,4,6,21,15,4,12,15,21,22,3,14,22,21,15,5,11,15,21,22,3,14,22,21,15,4,14,15,21,22,4,13,22,21,15,3,14,21,21,22,4,12,22,21,22,5,8,22,21,22,4,9,27,21,22,5,12,22,21,22,7,9,22,21,15,3,14,15,21,15,4,13,15,21,15,3,14,15,21,15,3,14,20,21,15,3,14,20,21,15,6,11,15,21,15,4,12,15,21,20,3,13,20,21,15,4,13,15,21,15,3,15,15,21,15,3,14,15,21,15,4,13,15,21,15,3,15,20,21,15,4,13,15,21,22,3,15,22,21,22,9,3,27,21,22,3,15,22,21,20,3,14,5],"m36":[23,0,0,0,0,23,25,9,5,27,23,25,5,13,10,23,25,3,16,25,23,25,4,15,25,23,25,4,15,25,23,25,4,15,25,23,25,9,6,13,23,25,8,9,25,23,25,6,9,25,23,22,3,17,18,23,22,3,17,19,23,7,5,13,13,23,14,3,17,4,23,6,8,8,6,23,25,5,13,25,23,25,4,15,25,23,25,4,10,25,23,25,3,15,25,23,24,3,16,24,23,25,3,17,25,23,24,4,14,24,23,25,4,15,25,23,24,4,16,24,23,25,4,15,25,23,25,4,15,25,23,17,8,7,17,23,17,5,11,24,23,23,3,17,21,23,17,3,17,10,23,23,3,17,21,23,26,5,13,28,23,26,4,16,26,23,25,3,17,25,23,25,4,15,25,23,25,5,13,25,23,25,5,13,25,23,25,5,14,25,23,25,7,12,25,23,25,4,15,25,23,25,4,14,25,23,25,5,13,25,23,25,4,11,25,23,25,4,16,25,23,25,4,15,25,23,25,4,15,25,23,25,4,15,25,23,25,4,15,25,23,25,4,15,25,23,25,4,16,25,23,25,5,13,25,23,25,5,13,25,23,25,3,17,25,23,25,4,15,25,23,25,3,17,25,23,25,4,15,25,23,25,4,15,25,23,25,4,15,25,23,25,4,15,25,23,25,7,13,25,23,25,5,13,25,23,25,3,13,25,23,25,3,17,14,23,-4,3,17,4,23,25,8,5,6,23,17,4,14,17,23,25,4,15,25,23,17,5,13,17,23,25,4,15,25,23,17,4,15,17,23,25,4,15,25,23,17,4,14,23,23,25,5,13,25,23,25,6,9,25,23,25,5,11,31,23,25,5,13,25,23,25,8,10,25,23,17,4,15,17,23,17,4,15,17,23,17,4,15,17,23,17,4,15,23,23,17,4,15,23,23,17,6,13,17,23,17,5,13,17,23,22,4,14,22,23,17,4,15,17,23,17,3,17,17,23,17,4,15,17,23,17,4,15,17,23,17,3,17,23,23,17,4,15,17,23,25,3,17,25,23,25,10,3,31,23,25,3,17,25,23,23,4,16,6],"a36":[23,0,0,0,0,23,25,9,5,27,23,25,5,13,11,23,25,3,16,25,23,25,4,15,25,23,25,4,15,25,23,25,4,15,25,23,25,9,6,13,23,25,8,9,25,23,25,6,9,25,23,22,3,17,18,23,22,3,17,19,23,7,5,13,13,23,14,3,17,4,23,7,8,8,7,23,25,5,13,25,23,25,4,15,25,23,25,4,10,25,23,25,3,15,25,23,24,3,16,24,23,25,3,17,25,23,24,4,14,24,23,25,4,15,25,23,24,4,16,24,23,25,4,15,25,23,25,4,15,25,23,17,8,7,17,23,17,5,11,24,23,23,3,17,21,23,17,3,17,10,23,23,3,17,21,23,26,5,13,28,23,26,4,16,26,23,25,3,17,25,23,25,4,15,25,23,25,5,13,25,23,25,5,13,25,23,25,5,14,25,23,25,7,12,25,23,25,4,15,25,23,25,4,14,25,23,25,5,13,25,23,25,4,11,25,23,25,4,16,25,23,25,4,15,25,23,25,4,15,25,23,25,4,15,25,23,25,4,15,25,23,25,4,15,25,23,25,4,16,25,23,25,5,13,25,23,25,5,13,25,23,25,3,17,25,23,25,4,15,25,23,25,3,17,25,23,25,4,15,25,23,25,4,15,25,23,25,4,15,25,23,25,4,15,25,23,25,7,13,25,23,25,5,13,25,23,25,3,13,25,23,25,3,17,14,23,-4,3,17,4,23,25,8,5,7,23,17,4,14,17,23,25,4,15,25,23,17,5,13,17,23,25,4,15,25,23,17,4,15,17,23,25,4,15,25,23,17,4,14,23,23,25,5,13,25,23,25,6,9,25,23,25,5,11,31,23,25,5,13,25,23,25,8,10,25,23,17,4,15,17,23,17,4,15,17,23,17,4,15,17,23,17,4,15,23,23,17,4,15,23,23,17,6,13,17,23,17,5,13,17,23,22,4,14,22,23,17,4,15,17,23,17,3,17,17,23,17,4,15,17,23,17,4,15,17,23,17,3,17,23,23,17,4,15,17,23,25,3,17,25,23,25,10,3,31,23,25,3,17,25,23,23,4,16,6],"m40":[26,0,0,0,0,26,28,10,5,31,26,28,6,14,12,26,28,3,19,28,26,28,4,18,28,26,28,4,18,28,26,28,4,18,28,26,28,10,6,15,26,28,9,11,28,26,28,6,11,28,26,24,3,20,19,26,25,3,20,22,26,8,5,15,15,26,16,3,19,5,26,7,9,9,7,26,28,6,14,28,26,28,4,18,28,26,28,5,12,28,26,28,4,17,28,26,28,4,18,28,26,28,3,19,28,26,28,5,16,28,26,28,4,18,28,26,28,3,19,28,26,28,4,18,28,26,28,4,18,28,26,19,9,7,19,26,19,5,12,26,26,26,3,19,24,26,19,3,19,11,26,26,3,19,24,26,30,5,15,33,26,30,4,18,31,26,29,4,18,29,26,29,4,19,29,26,29,6,15,30,26,29,6,16,29,26,29,6,17,29,26,29,8,14,29,26,29,4,18,30,26,30,5,16,30,26,29,5,16,29,26,30,5,13,31,26,30,5,18,30,26,30,5,18,30,26,30,4,18,30,26,30,5,17,30,26,29,4,18,30,26,29,5,17,29,26,29,4,19,29,26,29,6,15,29,26,29,6,15,30,26,29,3,20,29,26,30,5,17,31,26,30,4,19,30,26,30,4,18,30,26,30,4,18,30,26,30,4,18,30,26,29,5,17,29,26,28,7,15,28,26,28,6,14,28,26,28,4,15,28,26,28,3,19,15,26,-3,3,19,5,26,28,9,5,7,26,19,4,17,19,26,28,4,18,28,26,19,6,15,19,26,28,4,18,28,26,19,4,17,19,26,28,5,16,28,26,19,4,17,26,26,28,6,15,28,26,28,6,10,28,26,28,4,12,35,26,28,6,16,28,26,28,9,11,28,26,19,4,18,19,26,19,4,18,19,26,19,4,18,19,26,19,4,18,26,26,19,4,18,26,26,19,7,15,19,26,19,6,15,19,26,25,4,16,25,26,19,4,18,19,26,19,4,18,19,26,19,4,18,19,26,19,5,16,19,26,19,3,19,26,26,19,5,16,19,26,28,4,19,28,26,28,11,4,34,26,28,3,20,28,26,25,4,18,6],"a40":[26,0,0,0,0,26,28,10,5,31,26,28,6,14,12,26,28,3,19,28,26,28,4,18,28,26,28,4,18,28,26,28,4,18,28,26,28,10,6,15,26,28,9,11,28,26,28,6,11,28,26,24,3,20,19,26,25,3,20,22,26,8,5,15,15,26,16,3,19,5,26,8,9,9,8,26,28,6,14,28,26,28,4,18,28,26,28,5,12,28,26,28,4,17,28,26,28,4,18,28,26,28,3,19,28,26,28,5,16,28,26,28,4,18,28,26,28,3,19,28,26,28,4,18,28,26,28,4,18,28,26,19,9,7,19,26,19,5,12,26,26,26,3,19,24,26,19,3,19,11,26,26,3,19,24,26,30,5,15,33,26,30,4,18,31,26,29,4,18,29,26,29,4,19,29,26,29,6,15,30,26,29,6,16,29,26,29,6,17,29,26,29,8,14,29,26,29,4,18,30,26,29,5,16,29,26,29,5,16,29,26,29,5,13,30,26,29,5,18,29,26,29,5,18,29,26,29,4,18,29,26,29,5,17,29,26,29,4,18,30,26,29,5,17,29,26,29,4,19,29,26,29,6,15,29,26,29,6,15,30,26,29,3,20,29,26,29,5,17,30,26,29,4,19,29,26,29,4,18,29,26,29,4,18,29,26,29,4,18,29,26,29,5,17,29,26,28,7,15,28,26,28,6,14,28,26,28,4,15,28,26,28,3,19,15,26,-3,3,19,5,26,28,9,5,7,26,19,4,17,19,26,28,4,18,28,26,19,6,15,19,26,28,4,18,28,26,19,4,17,19,26,28,5,16,28,26,19,4,17,26,26,28,6,15,28,26,28,6,10,28,26,28,4,12,35,26,28,6,16,28,26,28,9,11,28,26,19,4,18,19,26,19,4,18,19,26,19,4,18,19,26,19,4,18,26,26,19,4,18,26,26,19,7,15,19,26,19,6,15,19,26,25,4,16,25,26,19,4,18,19,26,19,4,18,19,26,19,4,18,19,26,19,5,16,19,26,19,3,19,26,26,19,5,16,19,26,28,4,19,28,26,28,11,4,34,26,28,3,20,28,26,25,4,18,6],"m45":[29,0,0,0,0,29,32,12,6,35,29,31,7,16,13,29,31,4,20,32,29,32,5,20,32,29,31,5,20,32,29,31,5,20,32,29,32,11,7,17,29,31,10,12,32,29,31,7,12,32,29,27,4,22,21,29,27,4,22,24,29,9,6,16,16,29,18,4,21,5,29,8,10,10,8,29,31,7,16,32,29,31,5,20,32,29,32,5,13,32,29,31,5,19,31,29,31,4,20,32,29,32,4,22,32,29,31,5,18,32,29,32,5,20,33,29,31,4,22,31,29,31,5,19,32,29,31,5,20,31,29,21,10,8,21,29,21,6,13,29,29,29,4,21,27,29,22,4,21,13,29,29,4,21,27,29,33,6,16,36,29,33,5,20,34,29,32,4,21,32,29,32,4,21,32,29,33,6,17,34,29,32,7,16,32,29,32,6,18,32,29,32,8,15,32,29,33,5,18,34,29,34,6,17,34,29,32,6,16,32,29,34,5,14,35,29,34,5,20,34,29,34,6,19,34,29,34,4,20,34,29,34,5,19,34,29,33,5,19,34,29,32,6,18,32,29,33,5,20,33,29,32,6,17,32,29,33,6,17,34,29,32,4,20,32,29,34,5,19,35,29,34,4,21,34,29,34,4,20,34,29,34,5,19,34,29,34,4,20,34,29,32,5,19,32,29,31,9,16,32,29,31,7,16,32,29,31,5,16,32,29,32,4,21,18,29,-5,4,21,5,29,31,10,6,8,29,22,5,18,23,29,31,5,20,32,29,22,7,16,23,29,31,5,19,32,29,22,5,20,23,29,31,5,18,32,29,22,5,19,30,29,31,6,17,32,29,31,7,11,32,29,31,6,13,38,29,31,7,17,32,29,31,10,12,31,29,22,5,20,23,29,22,5,19,23,29,22,5,20,23,29,22,5,20,29,29,22,5,19,29,29,22,8,16,23,29,22,6,17,23,29,28,5,17,29,29,22,5,19,23,29,22,4,21,23,29,22,5,20,23,29,22,5,19,23,29,22,4,21,29,29,21,5,19,21,29,31,5,21,32,29,31,13,4,38,29,31,4,21,32,29,29,5,20,7],"a45":[29,0,0,0,0,29,32,12,6,35,29,31,7,16,13,29,31,4,20,32,29,32,5,20,32,29,31,5,20,32,29,31,5,20,32,29,32,11,7,17,29,31,10,12,32,29,31,7,12,32,29,27,4,22,21,29,27,4,22,24,29,9,6,16,16,29,18,4,21,5,29,8,10,10,8,29,31,7,16,32,29,31,5,20,32,29,32,5,13,32,29,31,5,19,31,29,31,4,20,32,29,32,4,22,32,29,31,5,18,32,29,32,5,20,33,29,31,4,22,31,29,31,5,19,32,29,31,5,20,31,29,21,10,8,21,29,21,6,13,29,29,29,4,21,27,29,22,4,21,13,29,29,4,21,27,29,33,6,16,36,29,33,5,20,34,29,32,4,21,32,29,32,4,21,32,29,33,6,17,34,29,32,7,16,32,29,32,6,18,32,29,32,8,15,32,29,33,5,18,34,29,33,6,17,33,29,32,6,16,32,29,33,5,14,34,29,33,5,20,33,29,33,6,19,33,29,33,4,20,33,29,33,5,19,33,29,33,5,19,34,29,32,6,18,32,29,33,5,20,33,29,32,6,17,32,29,33,6,17,34,29,32,4,20,32,29,33,5,19,34,29,33,4,21,33,29,33,4,20,33,29,33,5,19,33,29,33,4,20,33,29,32,5,19,32,29,31,9,16,32,29,31,7,16,32,29,31,5,16,32,29,32,4,21,18,29,-5,4,21,5,29,31,10,6,8,29,22,5,18,23,29,31,5,20,32,29,22,7,16,23,29,31,5,19,32,29,22,5,20,23,29,31,5,18,32,29,22,5,19,30,29,31,6,17,32,29,31,7,11,32,29,31,6,13,38,29,31,7,17,32,29,31,10,12,31,29,22,5,20,23,29,22,5,19,23,29,22,5,20,23,29,22,5,20,29,29,22,5,19,29,29,22,8,16,23,29,22,6,17,23,29,28,5,17,29,29,22,5,19,23,29,22,4,21,23,29,22,5,20,23,29,22,5,19,23,29,22,4,21,29,29,21,5,19,21,29,31,5,21,32,29,31,13,4,38,29,31,4,21,32,29,29,5,20,7],"m48":[31,0,0,0,0,31,34,12,6,37,31,33,7,17,14,31,33,4,21,34,31,34,5,21,34,31,33,5,21,34,31,33,5,21,34,31,34,12,7,18,31,33,11,12,34,31,33,8,12,34,31,29,4,22,23,31,28,4,22,24,31,9,6,17,17,31,20,4,22,6,31,9,10,10,9,31,33,7,17,34,31,33,5,21,34,31,34,6,13,34,31,33,5,20,33,31,32,4,21,33,31,34,4,22,34,31,32,5,19,33,31,34,5,21,35,31,32,4,22,32,31,33,5,20,34,31,33,5,21,33,31,23,11,9,23,31,23,6,15,31,31,31,4,22,29,31,23,4,22,14,31,31,4,22,29,31,35,6,18,38,31,36,5,21,37,31,34,5,21,35,31,34,5,21,34,31,35,7,17,36,31,34,7,18,34,31,34,7,19,34,31,34,9,16,35,31,35,5,20,36,31,36,6,18,37,31,34,6,18,34,31,36,6,15,37,31,36,5,21,36,31,36,6,20,36,31,36,5,21,36,31,36,5,20,36,31,35,5,21,36,31,34,6,20,35,31,35,5,22,36,31,34,6,18,35,31,35,6,18,36,31,34,4,22,35,31,36,5,20,37,31,36,4,22,36,31,36,5,21,36,31,36,5,21,36,31,36,5,20,36,31,34,5,20,34,31,33,9,17,34,31,33,7,17,34,31,33,5,17,34,31,34,4,22,19,31,-4,4,22,6,31,33,11,7,8,31,23,5,19,24,31,33,5,21,34,31,23,7,17,24,31,33,5,20,34,31,23,5,21,24,31,33,6,19,34,31,23,5,20,31,31,33,7,17,34,31,33,8,12,34,31,33,6,13,41,31,33,7,18,34,31,33,11,13,33,31,23,5,21,24,31,23,5,20,24,31,23,5,21,24,31,23,5,21,31,31,23,5,20,31,31,23,9,16,24,31,23,7,17,24,31,29,5,19,30,31,23,5,20,24,31,23,4,22,24,31,23,5,21,24,31,23,6,19,24,31,23,4,22,31,31,22,6,19,22,31,33,4,22,34,31,33,13,4,41,31,33,4,22,34,31,31,5,21,8],"a48":[31,0,0,0,0,31,34,12,6,37,31,33,7,17,14,31,33,4,21,34,31,34,5,21,34,31,33,5,21,34,31,33,5,21,34,31,34,12,7,18,31,33,11,12,34,31,33,8,12,34,31,29,4,22,23,31,28,4,22,24,31,9,6,17,17,31,20,4,22,6,31,9,10,10,9,31,33,7,17,34,31,33,5,21,34,31,34,6,13,34,31,33,5,20,33,31,32,4,21,33,31,34,4,22,34,31,32,5,19,33,31,34,5,21,35,31,32,4,22,32,31,33,5,20,34,31,33,5,21,33,31,23,11,9,23,31,23,6,15,31,31,31,4,22,29,31,23,4,22,14,31,31,4,22,29,31,35,6,18,38,31,36,5,21,37,31,34,5,21,35,31,34,5,21,34,31,35,7,17,36,31,34,7,18,34,31,34,7,19,34,31,34,9,16,35,31,35,5,20,36,31,35,6,18,36,31,34,6,18,34,31,35,6,15,36,31,35,5,21,36,31,35,6,20,35,31,35,5,21,36,31,35,5,20,36,31,35,5,21,36,31,34,6,20,35,31,35,5,22,36,31,34,6,18,35,31,35,6,18,36,31,34,4,22,35,31,35,5,20,36,31,35,4,22,36,31,35,5,21,36,31,35,5,21,36,31,35,5,20,36,31,34,5,20,34,31,33,9,17,34,31,33,7,17,34,31,33,5,17,34,31,34,4,22,19,31,-4,4,22,6,31,33,11,7,9,31,23,5,19,24,31,33,5,21,34,31,23,7,17,24,31,33,5,20,34,31,23,5,21,24,31,33,6,19,34,31,23,5,20,31,31,33,7,17,34,31,33,8,12,34,31,33,6,13,41,31,33,7,18,34,31,33,11,13,33,31,23,5,21,24,31,23,5,20,24,31,23,5,21,24,31,23,5,21,31,31,23,5,20,31,31,23,9,16,24,31,23,7,17,24,31,29,5,19,30,31,23,5,20,24,31,23,4,22,24,31,23,5,21,24,31,23,6,19,24,31,23,4,22,31,31,22,6,19,22,31,33,4,22,34,31,33,13,4,41,31,33,4,22,34,31,31,5,21,8],"m50":[33,0,0,0,0,33,35,13,6,38,33,35,7,18,14,33,35,4,22,36,33,35,5,22,35,33,35,5,22,36,33,35,5,22,36,33,35,12,8,18,33,35,11,13,36,33,35,7,13,36,33,30,5,22,24,33,31,5,22,27,33,10,7,17,18,33,20,4,24,6,33,9,11,11,9,33,35,7,18,36,33,35,5,22,36,33,35,6,14,35,33,35,5,20,35,33,35,4,22,36,33,35,4,23,35,33,35,6,19,36,33,35,5,22,36,33,35,5,23,35,33,35,5,21,36,33,35,5,22,35,33,24,11,9,24,33,24,6,15,32,33,32,4,24,30,33,24,4,24,14,33,32,4,24,30,33,37,6,20,40,33,37,5,23,38,33,35,5,22,36,33,35,5,21,35,33,36,7,18,37,33,35,7,18,35,33,35,7,20,35,33,35,9,17,36,33,36,5,22,37,33,37,6,19,38,33,35,7,18,35,33,37,6,15,38,33,37,6,22,37,33,37,6,21,37,33,37,5,22,37,33,37,6,20,37,33,36,5,21,37,33,35,6,21,35,33,36,5,23,37,33,35,7,18,35,33,36,7,18,37,33,35,5,22,36,33,37,6,20,38,33,37,4,24,37,33,37,5,22,37,33,37,5,22,37,33,37,6,20,37,33,35,6,20,35,33,35,9,17,36,33,35,7,18,36,33,35,5,17,36,33,35,4,24,19,33,-5,4,24,6,33,35,11,7,8,33,24,6,19,25,33,35,5,22,36,33,24,7,18,25,33,35,5,21,36,33,24,6,22,25,33,35,6,20,37,33,24,5,20,33,33,35,7,18,37,33,35,8,12,36,33,35,6,14,43,33,35,8,18,36,33,35,11,13,35,33,24,5,22,26,33,24,6,20,26,33,24,5,22,25,33,24,5,22,32,33,24,5,21,32,33,24,9,17,26,33,24,7,18,25,33,31,5,19,32,33,24,6,20,25,33,24,5,22,25,33,24,5,21,25,33,24,6,20,25,33,24,4,24,32,33,24,6,20,24,33,35,5,23,36,33,35,14,4,43,33,35,4,23,36,33,33,5,22,9],"a50":[33,0,0,0,0,33,35,13,6,38,33,35,7,18,15,33,35,4,22,36,33,35,5,22,35,33,35,5,22,36,33,35,5,22,36,33,35,12,8,18,33,35,11,13,36,33,35,7,13,36,33,30,5,22,24,33,31,5,22,27,33,10,7,17,18,33,20,4,24,6,33,9,11,11,9,33,35,7,18,36,33,35,5,22,36,33,35,6,14,35,33,35,5,20,35,33,35,4,22,36,33,35,4,23,35,33,35,6,19,36,33,35,5,22,36,33,35,5,23,35,33,35,5,21,36,33,35,5,22,35,33,24,11,9,24,33,24,6,15,32,33,32,4,24,30,33,24,4,24,14,33,32,4,24,30,33,37,6,20,40,33,37,5,23,38,33,35,5,22,36,33,35,5,21,35,33,36,7,18,37,33,35,7,18,35,33,35,7,20,35,33,35,9,17,36,33,36,5,22,37,33,36,6,19,37,33,35,7,18,35,33,36,6,15,37,33,36,6,22,37,33,36,6,21,36,33,36,5,22,37,33,36,6,20,37,33,36,5,21,37,33,35,6,21,35,33,36,5,23,37,33,35,7,18,35,33,36,7,18,37,33,35,5,22,36,33,36,6,20,37,33,36,4,24,37,33,36,5,22,37,33,36,5,22,37,33,36,6,20,37,33,35,6,20,35,33,35,9,17,36,33,35,7,18,36,33,35,5,17,36,33,35,4,24,19,33,-5,4,24,6,33,35,11,7,9,33,24,6,19,25,33,35,5,22,36,33,24,7,18,25,33,35,5,21,36,33,24,6,22,25,33,35,6,20,37,33,24,5,20,33,33,35,7,18,36,33,35,8,12,36,33,35,6,14,43,33,35,8,18,36,33,35,11,13,35,33,24,5,22,25,33,24,6,20,25,33,24,5,22,25,33,24,5,22,32,33,24,5,21,32,33,24,9,17,25,33,24,7,18,25,33,31,5,19,32,33,24,6,20,25,33,24,5,22,25,33,24,5,21,25,33,24,6,20,25,33,24,4,24,32,33,24,6,20,24,33,35,5,23,36,33,35,14,4,43,33,35,4,23,36,33,33,5,22,9],"m54":[35,0,0,0,0,35,38,14,7,42,35,37,8,20,16,35,37,5,24,38,35,38,6,23,39,35,37,5,24,38,35,37,5,24,38,35,38,13,8,20,35,37,12,14,39,35,37,9,14,39,35,32,5,25,25,35,33,5,25,29,35,10,7,19,19,35,21,5,25,6,35,10,11,12,10,35,37,8,19,39,35,37,5,24,38,35,38,6,16,39,35,37,6,22,37,35,37,5,24,38,35,38,5,25,39,35,37,7,21,38,35,38,5,24,39,35,37,5,25,38,35,37,5,25,38,35,37,5,24,38,35,26,12,10,26,35,26,7,16,35,35,34,5,25,31,35,26,5,25,15,35,34,5,25,31,35,40,7,20,44,35,40,5,25,41,35,38,5,24,39,35,38,5,25,38,35,39,8,19,40,35,38,8,20,38,35,38,8,22,38,35,38,10,19,39,35,38,6,23,39,35,40,7,21,41,35,38,8,19,38,35,40,6,18,41,35,40,6,24,40,35,40,7,23,40,35,40,5,24,40,35,40,6,23,40,35,39,6,23,40,35,38,7,22,39,35,39,5,25,40,35,38,7,21,39,35,39,7,21,40,35,38,5,25,39,35,40,6,23,41,35,40,5,25,40,35,40,5,24,40,35,40,6,23,40,35,40,6,23,40,35,38,6,22,38,35,37,10,19,38,35,37,8,19,39,35,37,6,19,38,35,38,5,25,21,35,-5,5,25,6,35,37,12,7,9,35,26,6,22,27,35,37,6,23,38,35,26,8,19,27,35,37,5,24,38,35,26,6,23,27,35,37,7,21,38,35,26,6,22,35,35,37,7,20,38,35,37,9,13,38,35,37,7,15,46,35,37,8,20,38,35,37,12,15,37,35,26,5,24,27,35,26,6,23,27,35,26,5,24,27,35,26,6,23,35,35,26,5,24,35,35,26,10,19,27,35,26,8,20,27,35,34,6,21,35,35,26,6,23,27,35,26,5,25,27,35,26,5,24,27,35,26,7,21,27,35,26,5,25,35,35,26,7,21,26,35,37,5,25,38,35,37,15,5,46,35,37,5,25,38,35,35,6,23,9],"a54":[35,0,0,0,0,35,38,14,7,42,35,37,8,20,16,35,37,5,24,38,35,38,6,23,39,35,37,5,24,38,35,37,5,24,38,35,38,13,8,20,35,37,12,14,39,35,37,9,14,39,35,32,5,25,25,35,33,5,25,29,35,10,7,19,19,35,21,5,25,6,35,10,11,12,10,35,37,8,19,39,35,37,5,24,38,35,38,6,16,39,35,37,6,22,37,35,37,5,24,38,35,38,5,25,39,35,37,7,21,38,35,38,5,24,39,35,37,5,25,38,35,37,5,25,38,35,37,5,24,38,35,26,12,10,26,35,26,7,16,35,35,34,5,25,31,35,26,5,25,15,35,34,5,25,31,35,40,7,20,44,35,40,5,25,41,35,38,5,24,39,35,38,5,25,38,35,39,8,19,40,35,38,8,20,38,35,38,8,22,38,35,38,10,19,39,35,38,6,23,39,35,39,7,21,40,35,38,8,19,38,35,39,6,18,40,35,39,6,24,40,35,39,7,23,39,35,39,5,24,40,35,39,6,23,40,35,39,6,23,40,35,38,7,22,39,35,39,5,25,40,35,38,7,21,39,35,39,7,21,40,35,38,5,25,39,35,39,6,23,40,35,39,5,25,40,35,39,5,24,40,35,39,6,23,40,35,39,6,23,40,35,38,6,22,38,35,37,10,19,38,35,37,8,19,39,35,37,6,19,38,35,38,5,25,21,35,-5,5,25,6,35,37,12,7,10,35,26,6,22,27,35,37,6,23,38,35,26,8,19,27,35,37,5,24,38,35,26,6,23,27,35,37,7,21,38,35,26,6,22,35,35,37,7,20,38,35,37,9,13,38,35,37,7,15,46,35,37,8,20,38,35,37,12,15,37,35,26,5,24,27,35,26,6,23,27,35,26,5,24,27,35,26,6,23,35,35,26,5,24,35,35,26,10,19,27,35,26,8,20,27,35,34,6,21,35,35,26,6,23,27,35,26,5,25,27,35,26,5,24,27,35,26,7,21,27,35,26,5,25,35,35,26,7,21,26,35,37,5,25,38,35,37,15,5,46,35,37,5,25,38,35,35,6,23,9],"m56":[36,0,0,0,0,36,39,14,7,43,36,39,8,20,16,36,39,5,25,40,36,39,6,24,40,36,39,6,24,40,36,39,6,24,40,36,39,14,9,20,36,39,13,15,40,36,39,8,15,40,36,33,5,25,26,36,34,5,25,29,36,11,7,20,20,36,22,5,26,6,36,10,12,12,10,36,39,8,20,40,36,39,6,24,40,36,39,7,16,40,36,39,6,23,39,36,38,5,25,39,36,39,5,26,40,36,38,6,22,39,36,39,6,24,40,36,38,5,26,39,36,39,6,25,40,36,39,6,24,40,36,27,13,10,27,36,27,7,17,36,36,36,5,26,33,36,27,5,26,16,36,36,5,26,33,36,41,7,21,45,36,42,6,25,43,36,40,5,26,41,36,40,6,25,40,36,41,8,20,42,36,40,8,21,40,36,40,8,22,40,36,40,10,19,41,36,41,6,23,42,36,41,7,22,43,36,40,7,21,40,36,41,6,18,42,36,41,6,25,41,36,41,7,24,41,36,41,6,24,41,36,41,6,24,41,36,41,6,24,42,36,40,7,23,41,36,41,6,25,42,36,40,8,21,41,36,41,8,21,42,36,40,5,25,41,36,41,6,24,42,36,41,5,26,41,36,41,6,24,41,36,41,6,24,41,36,41,6,23,41,36,40,6,23,40,36,39,10,20,40,36,39,8,20,40,36,39,6,20,40,36,39,5,26,21,36,-5,5,26,7,36,39,13,8,9,36,27,6,23,28,36,39,6,24,40,36,27,8,20,28,36,39,6,24,40,36,27,6,24,28,36,39,7,22,40,36,27,6,23,37,36,39,8,20,40,36,39,9,14,41,36,39,7,15,48,36,39,9,21,40,36,39,13,15,39,36,27,6,24,28,36,27,6,24,28,36,27,6,24,28,36,27,6,24,36,36,27,6,24,36,36,27,10,20,28,36,27,8,21,28,36,35,5,22,36,36,27,6,24,28,36,27,5,26,28,36,27,6,24,28,36,27,7,22,28,36,27,5,26,36,36,26,7,22,26,36,39,5,26,40,36,39,15,5,48,36,39,5,26,40,36,36,6,24,9],"a56":[36,0,0,0,0,36,39,14,7,43,36,39,8,20,17,36,39,5,25,40,36,39,6,24,40,36,39,6,24,40,36,39,6,24,40,36,39,14,9,20,36,39,13,15,40,36,39,8,15,40,36,33,5,25,26,36,34,5,25,29,36,11,7,20,20,36,22,5,26,6,36,10,12,12,10,36,39,8,20,40,36,39,6,24,40,36,39,7,16,40,36,39,6,23,39,36,38,5,25,39,36,39,5,26,40,36,38,6,22,39,36,39,6,24,40,36,38,5,26,39,36,39,6,25,40,36,39,6,24,40,36,27,13,10,27,36,27,7,17,36,36,36,5,26,33,36,27,5,26,16,36,36,5,26,33,36,41,7,21,45,36,42,6,25,43,36,40,5,26,41,36,40,6,25,40,36,41,8,20,42,36,40,8,21,40,36,40,8,22,40,36,40,10,19,41,36,41,6,23,42,36,40,7,22,42,36,40,7,21,40,36,40,6,18,41,36,40,6,25,41,36,40,7,24,40,36,40,6,24,41,36,40,6,24,41,36,41,6,24,42,36,40,7,23,41,36,41,6,25,42,36,40,8,21,41,36,41,8,21,42,36,40,5,25,41,36,40,6,24,41,36,40,5,26,41,36,40,6,24,41,36,40,6,24,41,36,40,6,23,41,36,40,6,23,40,36,39,10,20,40,36,39,8,20,40,36,39,6,20,40,36,39,5,26,21,36,-5,5,26,7,36,39,13,8,10,36,27,6,23,28,36,39,6,24,40,36,27,8,20,28,36,39,6,24,40,36,27,6,24,28,36,39,7,22,40,36,27,6,23,37,36,39,8,20,40,36,39,9,14,41,36,39,7,15,48,36,39,9,21,40,36,39,13,15,39,36,27,6,24,28,36,27,6,24,28,36,27,6,24,28,36,27,6,24,36,36,27,6,24,36,36,27,10,20,28,36,27,8,21,28,36,35,5,22,36,36,27,6,24,28,36,27,5,26,28,36,27,6,24,28,36,27,7,22,28,36,27,5,26,36,36,26,7,22,26,36,39,5,26,40,36,39,15,5,48,36,39,5,26,40,36,36,6,24,9],"m60":[39,0,0,0,0,39,42,16,8,46,39,42,9,22,17,39,42,6,27,43,39,42,6,27,44,39,42,6,27,43,39,42,6,27,43,39,42,15,9,22,39,42,14,16,43,39,42,9,16,43,39,36,6,27,28,39,36,6,27,31,39,12,8,22,22,39,24,6,28,7,39,10,13,13,11,39,42,9,21,43,39,42,6,27,43,39,42,7,17,43,39,42,6,25,42,39,41,6,26,42,39,42,5,28,43,39,41,7,24,42,39,42,6,27,43,39,41,6,27,42,39,42,6,27,43,39,42,6,27,43,39,29,14,11,30,39,29,8,18,39,39,38,6,28,35,39,29,6,28,17,39,38,6,28,35,39,44,7,22,48,39,44,6,26,45,39,43,6,27,44,39,43,6,27,43,39,44,9,21,45,39,43,9,22,43,39,43,9,24,43,39,43,11,20,44,39,44,7,25,45,39,45,8,23,46,39,43,9,21,43,39,45,7,19,46,39,45,7,26,45,39,45,8,26,45,39,45,6,27,45,39,45,7,25,45,39,44,6,27,45,39,43,8,25,44,39,44,6,27,45,39,43,8,23,44,39,44,8,23,45,39,43,6,27,44,39,45,7,25,46,39,45,6,28,45,39,45,6,27,45,39,45,7,26,45,39,45,7,25,45,39,43,7,25,43,39,42,11,21,43,39,42,9,21,43,39,42,6,21,43,39,42,6,28,23,39,-6,6,28,7,39,42,14,8,10,39,29,7,24,30,39,42,7,26,43,39,29,9,21,30,39,42,6,26,43,39,29,7,26,30,39,42,8,23,44,39,29,6,25,39,39,42,8,22,43,39,42,10,15,43,39,42,7,18,52,39,42,9,22,43,39,42,14,16,42,39,29,6,27,30,39,29,7,25,30,39,29,6,27,30,39,29,7,26,39,39,29,6,26,39,39,29,11,21,30,39,29,9,22,30,39,37,6,23,38,39,29,7,25,30,39,29,6,27,30,39,29,6,27,30,39,29,7,25,30,39,29,6,28,39,39,28,7,25,28,39,42,5,28,43,39,42,17,5,51,39,42,5,28,43,39,39,6,27,10],"a60":[39,0,0,0,0,39,42,16,8,46,39,42,9,22,18,39,42,6,27,43,39,42,6,27,44,39,42,6,27,43,39,42,6,27,43,39,42,15,9,22,39,42,14,16,43,39,42,9,16,43,39,36,6,27,28,39,36,6,27,31,39,12,8,22,22,39,24,6,28,7,39,10,13,13,11,39,42,9,21,43,39,42,6,27,43,39,42,7,17,43,39,42,6,25,42,39,41,6,26,42,39,42,5,28,43,39,41,7,24,42,39,42,6,27,43,39,41,6,27,42,39,42,6,27,43,39,42,6,27,43,39,29,14,11,30,39,29,8,18,39,39,38,6,28,35,39,29,6,28,17,39,38,6,28,35,39,44,7,22,48,39,44,6,26,45,39,43,6,27,44,39,43,6,27,43,39,44,9,21,45,39,43,9,22,43,39,43,9,24,43,39,43,11,20,44,39,44,7,25,45,39,44,8,23,45,39,43,9,21,43,39,44,7,19,45,39,44,7,26,45,39,44,8,26,44,39,44,6,27,45,39,44,7,25,45,39,44,6,27,45,39,43,8,25,44,39,44,6,27,45,39,43,8,23,44,39,44,8,23,45,39,43,6,27,44,39,44,7,25,45,39,44,6,28,45,39,44,6,27,45,39,44,7,26,45,39,44,7,25,45,39,43,7,25,43,39,42,11,21,43,39,42,9,21,43,39,42,6,21,43,39,42,6,28,23,39,-6,6,28,7,39,42,14,8,11,39,29,7,24,30,39,42,7,26,43,39,29,9,21,30,39,42,6,26,43,39,29,7,26,30,39,42,8,23,44,39,29,6,25,39,39,42,8,22,43,39,42,10,15,43,39,42,7,18,52,39,42,9,22,43,39,42,14,16,42,39,29,6,27,30,39,29,7,25,30,39,29,6,27,30,39,29,7,26,39,39,29,6,26,39,39,29,11,21,30,39,29,9,22,30,39,37,6,23,38,39,29,7,25,30,39,29,6,27,30,39,29,6,27,30,39,29,7,25,30,39,29,6,28,39,39,28,7,25,28,39,42,5,28,43,39,42,17,5,51,39,42,5,28,43,39,38,6,27,9],"m63":[41,0,0,0,0,41,44,16,8,48,41,44,9,23,18,41,44,5,30,45,41,44,6,29,45,41,44,6,29,45,41,44,6,29,45,41,44,15,10,23,41,44,14,17,45,41,44,10,17,45,41,37,5,30,29,41,38,5,30,32,41,13,8,23,23,41,25,6,29,7,41,10,13,14,11,41,44,9,22,45,41,44,6,29,45,41,44,8,18,45,41,44,7,26,44,41,43,6,28,44,41,44,6,29,45,41,43,7,25,44,41,44,6,29,45,41,43,6,29,44,41,44,6,29,45,41,44,6,29,46,41,30,14,12,31,41,30,9,19,41,41,40,6,29,37,41,31,6,29,19,41,40,6,29,37,41,46,8,24,50,41,46,6,29,47,41,45,6,28,46,41,45,6,29,45,41,46,9,23,47,41,45,9,24,45,41,45,9,25,45,41,45,12,22,46,41,45,7,27,46,41,47,8,25,48,41,45,9,22,45,41,47,7,21,48,41,47,7,28,47,41,47,8,27,47,41,47,6,29,47,41,47,7,27,47,41,46,6,28,47,41,45,8,27,47,41,46,6,29,47,41,45,9,24,47,41,46,9,24,47,41,45,5,30,46,41,47,7,27,48,41,47,6,29,47,41,47,6,28,47,41,47,7,26,47,41,47,7,26,47,41,45,7,26,45,41,44,12,22,45,41,44,9,22,45,41,44,7,22,45,41,44,6,29,24,41,-6,6,29,7,41,44,14,9,11,41,31,7,25,32,41,44,7,28,45,41,30,9,23,31,41,44,6,28,45,41,31,7,26,32,41,44,8,25,45,41,30,6,27,41,41,44,9,23,46,41,44,10,16,45,41,44,8,18,54,41,44,10,24,46,41,44,14,18,44,41,30,6,29,32,41,30,7,27,32,41,31,6,29,32,41,30,7,28,40,41,30,6,28,40,41,30,11,23,32,41,31,9,24,32,41,39,7,25,40,41,30,7,27,31,41,30,6,28,32,41,30,6,28,32,41,30,8,25,32,41,30,6,29,40,41,30,8,25,30,41,44,6,29,45,41,44,17,6,53,41,44,6,29,45,41,41,7,27,11],"a63":[41,0,0,0,0,41,44,16,8,48,41,44,9,23,19,41,44,5,30,45,41,44,6,29,45,41,44,6,29,45,41,44,6,29,45,41,44,15,10,23,41,44,14,17,45,41,44,10,17,45,41,37,5,30,29,41,38,5,30,32,41,13,8,23,23,41,25,6,29,7,41,10,13,14,11,41,44,9,22,45,41,44,6,29,45,41,44,8,18,45,41,44,7,26,44,41,43,6,28,44,41,44,6,29,45,41,43,7,25,44,41,44,6,29,45,41,43,6,29,44,41,44,6,29,45,41,44,6,29,46,41,30,14,12,31,41,30,9,19,41,41,40,6,29,37,41,31,6,29,19,41,40,6,29,37,41,46,8,24,50,41,46,6,29,47,41,45,6,28,46,41,45,6,29,45,41,46,9,23,47,41,45,9,24,45,41,45,9,25,45,41,45,12,22,46,41,45,7,27,46,41,46,8,25,47,41,45,9,22,45,41,46,7,21,47,41,46,7,28,47,41,46,8,27,46,41,46,6,29,47,41,46,7,27,47,41,46,6,28,47,41,45,8,27,47,41,46,6,29,47,41,45,9,24,47,41,46,9,24,47,41,45,5,30,46,41,46,7,27,47,41,46,6,29,47,41,46,6,28,47,41,46,7,26,47,41,46,7,26,47,41,45,7,26,45,41,44,12,22,45,41,44,9,22,45,41,44,7,22,45,41,44,6,29,24,41,-6,6,29,7,41,44,14,9,11,41,31,7,25,32,41,44,7,28,45,41,30,9,23,31,41,44,6,28,45,41,31,7,26,32,41,44,8,25,45,41,30,6,27,41,41,44,9,23,46,41,44,10,16,45,41,44,8,18,54,41,44,10,24,46,41,44,14,18,44,41,30,6,29,32,41,30,7,27,32,41,31,6,29,32,41,30,7,28,40,41,30,6,28,40,41,30,11,23,32,41,31,9,24,32,41,39,7,25,40,41,30,7,27,31,41,30,6,28,32,41,30,6,28,32,41,30,8,25,32,41,30,6,29,40,41,30,8,25,30,41,44,6,29,45,41,44,17,6,53,41,44,6,29,45,41,41,7,27,11],"m64":[42,0,0,0,0,42,45,17,8,49,42,44,9,24,18,42,44,6,30,45,42,45,7,29,46,42,44,7,29,45,42,44,7,28,45,42,45,16,10,23,42,44,15,17,46,42,44,10,17,46,42,38,6,30,30,42,39,6,30,34,42,12,9,23,23,42,25,6,30,7,42,10,14,14,11,42,44,9,23,46,42,44,7,29,45,42,45,8,19,46,42,44,7,27,44,42,44,6,29,45,42,45,6,30,47,42,44,7,26,45,42,45,7,29,46,42,44,6,30,45,42,44,7,29,45,42,44,7,29,45,42,30,15,12,31,42,30,8,20,41,42,41,6,30,38,42,32,6,30,20,42,41,6,30,38,42,47,8,24,51,42,48,6,29,49,42,46,6,30,47,42,46,7,29,46,42,47,9,24,48,42,46,9,25,46,42,46,9,26,46,42,46,12,22,47,42,47,7,27,48,42,47,8,26,48,42,46,9,24,46,42,47,7,21,48,42,47,7,29,48,42,47,8,28,47,42,47,7,29,48,42,47,7,28,48,42,47,7,28,48,42,46,8,28,47,42,47,7,29,48,42,46,9,25,47,42,47,9,25,48,42,46,6,30,47,42,47,7,28,48,42,47,6,30,48,42,47,7,28,48,42,47,7,28,48,42,47,7,28,48,42,46,8,26,46,42,44,12,23,45,42,44,9,23,46,42,44,7,23,45,42,45,6,30,24,42,-7,6,30,7,42,44,15,9,11,42,32,7,26,33,42,44,7,29,45,42,31,10,23,32,42,44,7,28,45,42,32,7,27,33,42,44,8,26,45,42,31,7,27,42,42,44,9,24,46,42,44,10,16,46,42,44,8,18,55,42,44,10,25,45,42,44,15,19,44,42,31,7,29,33,42,31,7,28,33,42,32,7,29,33,42,31,7,29,42,42,31,7,28,42,42,31,12,23,32,42,32,9,25,33,42,40,6,26,41,42,31,7,28,32,42,31,6,30,32,42,31,7,28,32,42,31,8,26,32,42,31,6,30,42,42,30,8,26,30,42,44,6,30,45,42,44,18,6,54,42,44,6,30,45,42,42,7,28,11],"a64":[42,0,0,0,0,42,45,17,8,49,42,44,9,24,19,42,44,6,30,45,42,45,7,29,46,42,44,7,29,45,42,44,7,28,45,42,45,16,10,23,42,44,15,17,46,42,44,10,17,46,42,38,6,30,30,42,39,6,30,34,42,12,9,23,23,42,25,6,30,7,42,10,14,14,11,42,44,9,23,46,42,44,7,29,45,42,45,8,19,46,42,44,7,27,44,42,44,6,29,45,42,45,6,30,47,42,44,7,26,45,42,45,7,29,46,42,44,6,30,45,42,44,7,29,45,42,44,7,29,45,42,30,15,12,31,42,30,8,20,41,42,41,6,30,38,42,32,6,30,20,42,41,6,30,38,42,47,8,24,51,42,48,6,29,49,42,46,6,30,47,42,46,7,29,46,42,47,9,24,48,42,46,9,25,46,42,46,9,26,46,42,46,12,22,47,42,47,7,27,48,42,46,8,26,47,42,46,9,24,46,42,46,7,21,47,42,46,7,29,48,42,46,8,28,46,42,46,7,29,48,42,46,7,28,48,42,47,7,28,48,42,46,8,28,47,42,47,7,29,48,42,46,9,25,47,42,47,9,25,48,42,46,6,30,47,42,46,7,28,47,42,46,6,30,48,42,46,7,28,48,42,46,7,28,48,42,46,7,28,48,42,46,8,26,46,42,44,12,23,45,42,44,9,23,46,42,44,7,23,45,42,45,6,30,24,42,-7,6,30,7,42,44,15,9,11,42,32,7,26,33,42,44,7,29,45,42,31,10,23,32,42,44,7,28,45,42,32,7,27,33,42,44,8,26,45,42,31,7,27,42,42,44,9,24,45,42,44,10,16,46,42,44,8,18,55,42,44,10,25,45,42,44,15,19,44,42,31,7,29,32,42,31,7,28,32,42,32,7,29,33,42,31,7,29,42,42,31,7,28,42,42,31,12,23,32,42,32,9,25,33,42,40,6,26,41,42,31,7,28,32,42,31,6,30,32,42,31,7,28,32,42,31,8,26,32,42,31,6,30,42,42,30,8,26,30,42,44,6,30,45,42,44,18,6,54,42,44,6,30,45,42,42,7,28,11],"m70":[46,0,0,0,0,46,49,18,9,54,46,49,10,27,20,46,49,7,32,50,46,49,7,31,50,46,49,7,31,50,46,49,7,31,50,46,49,17,11,25,46,49,16,18,50,46,49,11,18,50,46,41,7,32,32,46,42,7,32,36,46,13,10,25,25,46,28,7,32,8,46,11,15,15,12,46,49,10,25,50,46,49,7,31,50,46,49,8,20,50,46,49,8,29,49,46,47,7,31,48,46,49,7,32,50,46,47,8,28,48,46,49,7,31,50,46,47,7,32,49,46,49,7,31,50,46,49,7,31,50,46,33,16,13,34,46,33,10,21,45,46,44,7,32,40,46,34,7,32,20,46,44,7,32,40,46,51,9,26,56,46,52,7,31,53,46,50,7,31,51,46,50,7,31,50,46,51,10,25,52,46,50,10,26,50,46,50,10,28,50,46,50,13,23,52,46,50,7,30,51,46,52,9,27,53,46,50,10,24,50,46,52,8,22,53,46,52,8,30,52,46,52,9,29,52,46,52,7,31,52,46,52,8,29,52,46,51,7,30,52,46,50,9,29,52,46,51,7,31,52,46,50,9,27,52,46,51,9,27,52,46,50,6,32,51,46,52,8,29,53,46,52,6,32,52,46,52,7,31,52,46,52,8,29,52,46,52,8,28,52,46,50,8,29,50,46,49,13,25,50,46,49,10,25,50,46,49,7,25,50,46,49,7,32,26,46,-7,7,32,8,46,49,16,10,12,46,34,8,28,35,46,49,8,30,50,46,34,11,24,35,46,49,7,31,50,46,34,8,30,35,46,49,8,28,51,46,34,7,30,46,46,49,10,26,50,46,49,11,18,50,46,49,9,20,61,46,49,11,26,50,46,49,16,19,49,46,34,7,31,35,46,34,8,29,35,46,34,7,31,35,46,34,8,30,46,46,34,7,31,46,46,34,13,24,35,46,34,10,26,35,46,43,8,27,44,46,34,8,29,35,46,34,7,32,35,46,34,7,31,35,46,34,9,28,35,46,34,7,32,46,46,33,9,28,33,46,49,7,32,50,46,49,20,6,59,46,49,6,32,50,46,45,8,30,11],"a70":[46,0,0,0,0,46,49,18,9,54,46,49,10,27,21,46,49,7,32,50,46,49,7,31,50,46,49,7,31,50,46,49,7,31,50,46,49,17,11,25,46,49,16,18,50,46,49,11,18,50,46,41,7,32,32,46,42,7,32,36,46,13,10,25,25,46,28,7,32,8,46,11,15,15,12,46,49,10,25,50,46,49,7,31,50,46,49,8,20,50,46,49,8,29,49,46,47,7,31,48,46,49,7,32,50,46,47,8,28,48,46,49,7,31,50,46,47,7,32,49,46,49,7,31,50,46,49,7,31,50,46,33,16,13,34,46,33,10,21,45,46,44,7,32,40,46,34,7,32,20,46,44,7,32,40,46,51,9,26,56,46,52,7,31,53,46,50,7,31,51,46,50,7,31,50,46,51,10,25,52,46,50,10,26,50,46,50,10,28,50,46,50,13,23,52,46,50,7,30,51,46,51,9,27,52,46,50,10,24,50,46,51,8,22,52,46,51,8,30,52,46,51,9,29,51,46,51,7,31,52,46,51,8,29,52,46,51,7,30,52,46,50,9,29,52,46,51,7,31,52,46,50,9,27,52,46,51,9,27,52,46,50,6,32,51,46,51,8,29,52,46,51,6,32,52,46,51,7,31,52,46,51,8,29,52,46,51,8,28,52,46,50,8,29,50,46,49,13,25,50,46,49,10,25,50,46,49,7,25,50,46,49,7,32,26,46,-7,7,32,8,46,49,16,10,12,46,34,8,28,35,46,49,8,30,50,46,34,11,24,35,46,49,7,31,50,46,34,8,30,35,46,49,8,28,51,46,34,7,30,46,46,49,10,26,50,46,49,11,18,50,46,49,9,20,61,46,49,11,26,50,46,49,16,19,49,46,34,7,31,35,46,34,8,29,35,46,34,7,31,35,46,34,8,30,46,46,34,7,31,46,46,34,13,24,35,46,34,10,26,35,46,43,8,27,44,46,34,8,29,35,46,34,7,32,35,46,34,7,31,35,46,34,9,28,35,46,34,7,32,46,46,33,9,28,33,46,49,7,32,50,46,49,20,6,59,46,49,6,32,50,46,45,8,30,11],"m72":[47,0,0,0,0,47,50,19,9,55,47,50,10,27,21,47,50,7,33,52,47,50,8,31,51,47,50,7,32,51,47,50,7,31,51,47,50,18,11,25,47,50,17,19,51,47,50,11,19,51,47,43,7,32,34,47,42,7,32,36,47,14,10,26,26,47,28,7,33,8,47,12,16,16,13,47,50,11,25,51,47,50,7,32,51,47,50,9,20,51,47,50,8,30,50,47,49,7,32,50,47,50,7,33,51,47,49,9,28,50,47,50,7,32,51,47,49,6,33,50,47,50,7,32,51,47,50,7,32,51,47,35,17,13,36,47,35,10,21,47,47,46,7,33,42,47,35,7,33,21,47,46,7,33,42,47,52,9,27,57,47,53,7,32,54,47,53,7,32,54,47,51,7,32,51,47,52,10,26,53,47,51,10,27,51,47,51,10,28,51,47,51,13,24,52,47,52,8,30,53,47,53,9,28,54,47,51,10,26,51,47,53,8,23,54,47,53,8,31,53,47,53,9,30,53,47,53,7,32,53,47,53,8,30,53,47,52,7,31,53,47,51,9,30,52,47,52,7,32,54,47,51,10,27,52,47,52,10,26,53,47,51,7,32,52,47,53,8,30,54,47,53,7,32,53,47,53,7,32,53,47,53,8,30,53,47,53,8,30,53,47,51,8,30,51,47,50,14,25,51,47,50,11,25,51,47,50,8,25,51,47,50,7,33,27,47,-7,7,33,8,47,50,17,10,12,47,35,8,29,36,47,50,8,31,51,47,35,11,25,36,47,50,7,31,51,47,35,8,31,36,47,50,9,28,51,47,35,8,30,47,47,50,10,26,51,47,50,12,18,52,47,50,10,20,62,47,50,11,27,51,47,50,16,19,50,47,35,7,32,36,47,35,8,30,36,47,35,7,32,36,47,35,8,31,47,47,35,7,31,47,47,35,13,25,36,47,35,10,27,36,47,44,8,27,45,47,35,8,30,36,47,35,7,33,36,47,35,7,32,36,47,35,9,29,36,47,35,7,33,47,47,33,9,30,33,47,50,6,33,51,47,50,20,6,61,47,50,7,33,51,47,47,8,31,12],"a72":[47,0,0,0,0,47,50,19,9,55,47,50,10,27,21,47,50,7,33,52,47,50,8,31,51,47,50,7,32,51,47,50,7,31,51,47,50,18,11,25,47,50,17,19,51,47,50,11,19,51,47,43,7,32,34,47,42,7,32,36,47,14,10,26,26,47,28,7,33,8,47,12,16,16,13,47,50,11,25,51,47,50,7,32,51,47,50,9,20,51,47,50,8,30,50,47,49,7,32,50,47,50,7,33,51,47,49,9,28,50,47,50,7,32,51,47,49,6,33,50,47,50,7,32,51,47,50,7,32,51,47,35,17,13,36,47,35,10,21,47,47,46,7,33,42,47,35,7,33,21,47,46,7,33,42,47,52,9,27,57,47,53,7,32,54,47,52,7,32,53,47,51,7,32,51,47,52,10,26,53,47,51,10,27,51,47,51,10,28,51,47,51,13,24,52,47,52,8,30,53,47,52,9,28,53,47,51,10,26,51,47,52,8,23,53,47,52,8,31,53,47,52,9,30,52,47,52,7,32,53,47,52,8,30,53,47,52,7,31,53,47,51,9,30,52,47,52,7,32,54,47,51,10,27,52,47,52,10,26,53,47,51,7,32,52,47,52,8,30,53,47,52,7,32,53,47,52,7,32,53,47,52,8,30,53,47,52,8,30,53,47,51,8,30,51,47,50,14,25,51,47,50,11,25,51,47,50,8,25,51,47,50,7,33,27,47,-7,7,33,8,47,50,17,10,13,47,35,8,29,36,47,50,8,31,51,47,35,11,25,36,47,50,7,31,51,47,35,8,31,36,47,50,9,28,51,47,35,8,30,47,47,50,10,26,51,47,50,12,18,52,47,50,10,20,62,47,50,11,27,51,47,50,16,19,50,47,35,7,32,36,47,35,8,30,36,47,35,7,32,36,47,35,8,31,47,47,35,7,31,47,47,35,13,25,36,47,35,10,27,36,47,44,8,27,45,47,35,8,30,36,47,35,7,33,36,47,35,7,32,36,47,35,9,29,36,47,35,7,33,47,47,33,9,30,33,47,50,6,33,51,47,50,20,6,61,47,50,7,33,51,47,46,8,31,11],"m80":[52,0,0,0,0,52,56,21,11,61,52,55,12,30,23,52,55,7,37,56,52,56,8,36,57,52,55,8,36,56,52,55,8,36,56,52,56,20,13,29,52,55,18,21,57,52,55,13,21,57,52,47,8,37,36,52,48,8,37,41,52,16,11,28,29,52,32,8,37,9,52,13,17,18,14,52,55,12,28,57,52,55,8,36,56,52,56,10,23,57,52,55,9,33,55,52,54,8,35,55,52,56,8,37,57,52,54,9,32,55,52,56,8,36,57,52,54,7,37,56,52,55,8,35,56,52,55,8,36,56,52,38,19,15,39,52,38,11,24,52,52,51,8,37,47,52,39,8,37,23,52,51,8,37,47,52,59,10,29,65,52,59,8,35,61,52,59,8,36,61,52,57,8,36,57,52,58,12,29,59,52,57,12,30,57,52,57,11,32,57,52,57,15,27,58,52,58,9,34,59,52,59,10,32,61,52,57,12,29,57,52,59,9,26,60,52,59,9,35,60,52,59,10,34,59,52,59,8,36,60,52,59,9,34,60,52,59,8,36,60,52,57,10,34,59,52,58,8,36,59,52,57,11,31,59,52,58,11,30,59,52,57,8,37,58,52,59,9,34,60,52,59,8,37,60,52,59,8,36,60,52,59,9,34,60,52,59,10,33,60,52,57,10,33,57,52,55,15,28,56,52,55,12,28,57,52,55,9,28,56,52,56,8,37,30,52,-8,8,37,9,52,55,19,11,13,52,39,9,32,40,52,55,9,35,56,52,39,12,29,40,52,55,8,35,56,52,39,9,34,40,52,55,10,31,57,52,39,8,34,53,52,55,11,30,56,52,55,13,20,56,52,55,10,23,68,52,55,12,30,56,52,55,18,22,55,52,39,8,36,40,52,39,9,34,40,52,39,8,36,40,52,39,9,35,52,52,39,8,35,52,52,39,15,28,40,52,39,11,30,40,52,49,9,31,50,52,39,9,34,40,52,39,8,36,40,52,39,8,36,40,52,39,10,32,40,52,39,8,37,52,52,38,10,33,38,52,55,7,37,56,52,55,23,7,68,52,55,8,37,56,52,52,9,35,13],"a80":[52,0,0,0,0,52,56,21,11,61,52,55,12,30,24,52,55,7,37,56,52,56,8,36,57,52,55,8,36,56,52,55,8,36,56,52,56,20,13,29,52,55,18,21,57,52,55,13,21,57,52,47,8,37,36,52,48,8,37,41,52,16,11,28,29,52,32,8,37,9,52,13,17,18,14,52,55,12,28,57,52,55,8,36,56,52,56,10,23,57,52,55,9,33,55,52,54,8,35,55,52,56,8,37,57,52,54,9,32,55,52,56,8,36,57,52,54,7,37,56,52,55,8,35,56,52,55,8,36,56,52,38,19,15,39,52,38,11,24,52,52,51,8,37,47,52,39,8,37,23,52,51,8,37,47,52,59,10,29,65,52,59,8,35,61,52,58,8,36,60,52,57,8,36,57,52,58,12,29,59,52,57,12,30,57,52,57,11,32,57,52,57,15,27,58,52,58,9,34,59,52,58,10,32,60,52,57,12,29,57,52,58,9,26,59,52,58,9,35,60,52,58,10,34,58,52,58,8,36,60,52,58,9,34,60,52,59,8,36,60,52,57,10,34,59,52,58,8,36,59,52,57,11,31,59,52,58,11,30,59,52,57,8,37,59,52,58,9,34,59,52,58,8,37,60,52,58,8,36,60,52,58,9,34,60,52,58,10,33,60,52,57,10,33,57,52,55,15,28,56,52,55,12,28,57,52,55,9,28,56,52,56,8,37,30,52,-8,8,37,9,52,55,19,11,14,52,39,9,32,40,52,55,9,35,56,52,39,12,29,40,52,55,8,35,56,52,39,9,34,40,52,55,10,31,57,52,39,8,34,53,52,55,11,30,56,52,55,13,20,56,52,55,10,23,68,52,55,12,30,56,52,55,18,22,55,52,39,8,36,40,52,39,9,34,40,52,39,8,36,40,52,39,9,35,52,52,39,8,35,52,52,39,15,28,40,52,39,11,30,40,52,49,9,31,50,52,39,9,34,40,52,39,8,36,40,52,39,8,36,40,52,39,10,32,40,52,39,8,37,52,52,38,10,33,38,52,55,7,37,56,52,55,23,7,68,52,55,8,37,56,52,52,9,35,13],"m81":[53,0,0,0,0,53,57,21,10,62,53,56,12,29,23,53,56,8,36,57,53,57,8,36,59,53,56,8,36,57,53,56,8,35,57,53,57,20,13,29,53,56,18,21,58,53,56,12,21,58,53,48,7,37,37,53,48,7,37,41,53,16,11,28,29,53,32,8,36,9,53,13,17,17,14,53,56,12,28,58,53,56,8,36,57,53,57,9,23,58,53,56,9,33,56,53,55,8,35,56,53,57,7,37,58,53,55,10,31,56,53,57,8,36,58,53,55,8,36,56,53,56,8,35,57,53,56,8,36,57,53,39,19,15,40,53,39,11,24,52,53,51,8,36,47,53,39,8,36,23,53,51,8,36,47,53,60,10,30,66,53,60,8,36,62,53,59,8,36,61,53,58,8,36,58,53,59,12,28,60,53,58,12,30,58,53,58,11,32,58,53,58,15,27,59,53,59,9,34,60,53,60,10,32,61,53,58,11,29,58,53,60,9,26,61,53,60,9,35,60,53,60,10,34,60,53,60,8,36,60,53,60,9,34,60,53,59,8,35,60,53,58,10,34,59,53,59,8,36,60,53,58,11,31,59,53,59,11,30,60,53,58,7,37,59,53,60,9,34,61,53,60,8,36,60,53,60,8,36,60,53,60,9,34,60,53,60,9,33,60,53,58,10,32,58,53,56,15,28,57,53,56,12,28,58,53,56,9,28,57,53,56,8,36,30,53,-8,8,36,9,53,56,19,11,14,53,39,9,32,40,53,56,9,35,57,53,39,12,28,40,53,56,8,35,57,53,39,9,34,40,53,56,10,31,57,53,39,8,34,53,53,56,11,30,57,53,56,13,20,57,53,56,10,23,69,53,56,12,30,58,53,56,18,22,57,53,39,8,36,40,53,39,9,34,40,53,39,8,36,40,53,39,9,35,52,53,39,8,35,52,53,39,15,27,40,53,39,11,30,40,53,49,9,31,50,53,39,9,34,40,53,39,8,36,41,53,39,8,35,41,53,39,10,32,41,53,39,8,36,52,53,38,10,32,38,53,56,7,37,57,53,56,22,7,69,53,56,7,37,57,53,52,9,34,13],"a81":[53,0,0,0,0,53,57,21,10,62,53,56,12,29,24,53,56,8,36,57,53,57,8,36,59,53,56,8,36,57,53,56,8,35,57,53,57,20,13,29,53,56,18,21,58,53,56,12,21,58,53,48,7,37,37,53,48,7,37,41,53,16,11,28,29,53,32,8,36,9,53,13,17,17,14,53,56,12,28,58,53,56,8,36,57,53,57,9,23,58,53,56,9,33,56,53,55,8,35,56,53,57,7,37,58,53,55,10,31,56,53,57,8,36,58,53,55,8,36,56,53,56,8,35,57,53,56,8,36,57,53,39,19,15,40,53,39,11,24,52,53,51,8,36,47,53,39,8,36,23,53,51,8,36,47,53,60,10,30,66,53,60,8,36,62,53,58,8,36,60,53,58,8,36,58,53,59,12,28,60,53,58,12,30,58,53,58,11,32,58,53,58,15,27,59,53,59,9,34,60,53,59,10,32,60,53,58,11,29,58,53,59,9,26,60,53,59,9,35,60,53,59,10,34,59,53,59,8,36,60,53,59,9,34,60,53,59,8,35,60,53,58,10,34,59,53,59,8,36,60,53,58,11,31,59,53,59,11,30,60,53,58,7,37,59,53,59,9,34,60,53,59,8,36,60,53,59,8,36,60,53,59,9,34,60,53,59,9,33,60,53,58,10,32,58,53,56,15,28,57,53,56,12,28,58,53,56,9,28,57,53,56,8,36,30,53,-8,8,36,9,53,56,19,11,14,53,39,9,32,40,53,56,9,35,57,53,39,12,28,40,53,56,8,35,57,53,39,9,34,40,53,56,10,31,57,53,39,8,34,53,53,56,11,30,58,53,56,13,20,57,53,56,10,23,69,53,56,12,30,58,53,56,18,22,57,53,39,8,36,41,53,39,9,34,41,53,39,8,36,40,53,39,9,35,52,53,39,8,35,52,53,39,15,27,41,53,39,11,30,40,53,49,9,31,50,53,39,9,34,40,53,39,8,36,41,53,39,8,35,41,53,39,10,32,41,53,39,8,36,52,53,38,10,32,38,53,56,7,37,57,53,56,22,7,69,53,56,7,37,57,53,52,9,34,13],"m84":[55,0,0,0,0,55,59,22,11,65,55,58,12,31,24,55,58,8,39,60,55,59,9,38,60,55,58,9,38,59,55,58,9,38,59,55,59,21,13,30,55,58,19,23,60,55,58,13,23,60,55,50,8,40,39,55,51,8,40,44,55,16,12,29,30,55,34,8,39,10,55,14,18,18,15,55,58,13,29,60,55,58,9,38,59,55,59,10,25,60,55,58,10,35,58,55,58,8,38,59,55,59,8,39,61,55,58,10,34,59,55,59,9,38,60,55,58,8,39,59,55,58,9,38,59,55,58,9,38,59,55,41,20,16,42,55,41,11,26,55,55,53,8,39,49,55,42,8,39,26,55,53,8,39,49,55,62,10,31,68,55,62,8,37,64,55,61,9,37,62,55,60,9,38,60,55,62,12,31,63,55,60,12,33,60,55,60,12,34,60,55,60,16,29,61,55,61,9,36,62,55,62,11,34,63,55,60,13,30,60,55,62,10,27,63,55,62,10,38,62,55,62,11,37,62,55,62,9,38,62,55,62,10,36,62,55,62,9,38,63,55,60,11,36,61,55,62,9,39,63,55,60,12,33,61,55,62,12,32,63,55,60,8,40,62,55,62,10,36,63,55,62,8,39,62,55,62,9,38,62,55,62,10,35,62,55,62,10,36,62,55,60,10,35,60,55,58,16,30,59,55,58,13,29,60,55,58,9,30,59,55,58,8,39,31,55,-8,8,39,10,55,58,20,12,14,55,42,10,34,43,55,58,9,38,59,55,41,13,30,42,55,58,9,37,59,55,42,10,36,43,55,58,10,34,59,55,41,9,36,55,55,58,12,32,60,55,58,14,22,60,55,58,11,24,72,55,58,13,32,59,55,58,19,24,59,55,41,9,38,43,55,41,10,36,43,55,42,9,38,43,55,41,9,38,55,55,41,9,37,55,55,41,15,31,42,55,42,12,32,43,55,52,9,34,53,55,41,10,36,42,55,41,8,39,42,55,41,9,38,42,55,41,11,33,42,55,41,8,39,55,55,40,11,34,40,55,58,8,39,59,55,58,24,8,71,55,58,8,39,59,55,55,9,37,15],"a84":[55,0,0,0,0,55,59,22,11,65,55,58,12,31,25,55,58,8,39,60,55,59,9,38,60,55,58,9,38,59,55,58,9,38,59,55,59,21,13,30,55,58,19,23,60,55,58,13,23,60,55,50,8,40,39,55,51,8,40,44,55,16,12,29,30,55,34,8,39,10,55,14,18,18,15,55,58,13,29,60,55,58,9,38,59,55,59,10,25,60,55,58,10,35,58,55,58,8,38,59,55,59,8,39,61,55,58,10,34,59,55,59,9,38,60,55,58,8,39,59,55,58,9,38,59,55,58,9,38,59,55,41,20,16,42,55,41,11,26,55,55,53,8,39,49,55,42,8,39,26,55,53,8,39,49,55,62,10,31,68,55,62,8,37,64,55,60,9,37,61,55,60,9,38,60,55,62,12,31,63,55,60,12,33,60,55,60,12,34,60,55,60,16,29,61,55,61,9,36,62,55,61,11,34,62,55,60,13,30,60,55,61,10,27,62,55,61,10,38,62,55,61,11,37,61,55,61,9,38,62,55,61,10,36,62,55,62,9,38,63,55,60,11,36,61,55,62,9,39,63,55,60,12,33,61,55,62,12,32,63,55,60,8,40,61,55,61,10,36,62,55,61,8,39,62,55,61,9,38,62,55,61,10,35,62,55,61,10,36,62,55,60,10,35,60,55,58,16,30,59,55,58,13,29,60,55,58,9,30,59,55,58,8,39,31,55,-8,8,39,10,55,58,20,12,15,55,42,10,34,43,55,58,9,38,59,55,41,13,30,42,55,58,9,37,59,55,42,10,36,43,55,58,10,34,59,55,41,9,36,55,55,58,12,32,59,55,58,14,22,60,55,58,11,24,72,55,58,13,32,59,55,58,19,24,59,55,41,9,38,42,55,41,10,36,42,55,42,9,38,43,55,41,9,38,55,55,41,9,37,55,55,41,15,31,42,55,42,12,32,43,55,52,9,34,53,55,41,10,36,42,55,41,8,39,42,55,41,9,38,42,55,41,11,33,42,55,41,8,39,55,55,40,11,34,40,55,58,8,39,59,55,58,24,8,71,55,58,8,39,59,55,55,9,37,15],"m90":[59,0,0,0,0,59,63,23,12,69,59,62,13,33,26,59,62,9,40,63,59,63,9,40,65,59,62,9,40,63,59,62,9,40,63,59,63,22,14,32,59,62,20,23,64,59,62,14,23,64,59,53,9,40,41,59,54,9,40,46,59,17,12,31,32,59,36,8,41,10,59,15,19,19,16,59,62,13,31,64,59,62,9,40,63,59,63,11,25,64,59,62,10,38,62,59,61,9,39,62,59,63,9,41,65,59,61,11,35,62,59,63,9,40,64,59,61,9,40,63,59,62,9,40,63,59,62,9,40,63,59,44,21,17,45,59,44,12,27,59,59,57,8,41,52,59,44,8,41,26,59,57,8,41,52,59,66,11,34,72,59,66,9,41,68,59,66,9,41,68,59,64,9,40,64,59,65,13,33,66,59,64,13,34,64,59,64,13,36,64,59,64,17,31,65,59,65,10,38,66,59,66,11,36,68,59,64,13,32,64,59,66,11,29,67,59,66,10,40,67,59,66,11,39,66,59,66,9,41,67,59,66,10,38,67,59,66,9,40,67,59,64,11,39,66,59,65,9,42,66,59,64,12,36,66,59,65,11,36,66,59,64,8,42,65,59,66,10,38,67,59,66,9,41,67,59,66,9,40,67,59,66,10,39,67,59,66,10,38,67,59,64,11,37,64,59,62,17,31,63,59,62,13,31,64,59,62,10,31,63,59,63,8,41,34,59,-9,8,41,10,59,62,21,12,15,59,44,10,36,45,59,62,10,39,63,59,43,13,32,44,59,62,9,39,63,59,44,10,38,45,59,62,11,35,64,59,43,9,38,58,59,62,12,33,64,59,62,14,23,64,59,62,12,26,77,59,62,14,33,64,59,62,20,25,63,59,43,9,40,45,59,43,10,38,45,59,44,9,40,45,59,43,10,39,58,59,43,9,39,58,59,43,16,31,45,59,44,12,34,45,59,56,10,34,57,59,43,10,38,44,59,43,9,40,45,59,43,9,39,45,59,43,11,35,45,59,43,8,41,58,59,42,11,36,42,59,62,9,41,63,59,62,25,8,76,59,62,8,41,63,59,58,10,38,15],"a90":[59,0,0,0,0,59,63,23,12,69,59,62,13,33,26,59,62,9,40,63,59,63,9,40,65,59,62,9,40,63,59,62,9,40,63,59,63,22,14,32,59,62,20,23,64,59,62,14,23,64,59,53,9,40,41,59,54,9,40,46,59,17,12,31,32,59,36,8,41,10,59,15,19,19,16,59,62,13,31,64,59,62,9,40,63,59,63,11,25,64,59,62,10,38,62,59,61,9,39,62,59,63,9,41,65,59,61,11,35,62,59,63,9,40,64,59,61,9,40,63,59,62,9,40,63,59,62,9,40,63,59,44,21,17,45,59,44,12,27,59,59,57,8,41,52,59,44,8,41,26,59,57,8,41,52,59,66,11,34,72,59,66,9,41,68,59,65,9,41,67,59,64,9,40,64,59,65,13,33,66,59,64,13,34,64,59,64,13,36,64,59,64,17,31,65,59,65,10,38,66,59,65,11,36,67,59,64,13,32,64,59,65,11,29,66,59,65,10,40,67,59,65,11,39,65,59,65,9,41,67,59,65,10,38,67,59,66,9,40,67,59,64,11,39,66,59,65,9,42,66,59,64,12,36,66,59,65,11,36,66,59,64,8,42,65,59,65,10,38,66,59,65,9,41,67,59,65,9,40,67,59,65,10,39,67,59,65,10,38,67,59,64,11,37,64,59,62,17,31,63,59,62,13,31,64,59,62,10,31,63,59,63,8,41,34,59,-9,8,41,10,59,62,21,12,16,59,44,10,36,45,59,62,10,39,63,59,43,13,32,44,59,62,9,39,63,59,44,10,38,45,59,62,11,35,64,59,43,9,38,58,59,62,12,33,64,59,62,14,23,64,59,62,12,26,77,59,62,14,33,64,59,62,20,25,63,59,43,9,40,45,59,43,10,38,45,59,44,9,40,45,59,43,10,39,58,59,43,9,39,58,59,43,16,31,45,59,44,12,34,45,59,56,10,34,57,59,43,10,38,44,59,43,9,40,45,59,43,9,39,45,59,43,11,35,45,59,43,8,41,58,59,42,11,36,42,59,62,9,41,63,59,62,25,8,76,59,62,8,41,63,59,58,10,38,15],"m96":[62,0,0,0,0,62,67,25,12,73,62,67,14,34,28,62,67,9,43,68,62,67,10,41,68,62,67,10,41,68,62,67,10,41,68,62,67,23,15,34,62,67,22,24,68,62,67,15,24,68,62,57,10,42,44,62,58,10,42,49,62,18,13,33,34,62,38,9,43,11,62,16,20,21,17,62,67,14,33,68,62,67,10,41,68,62,67,12,26,68,62,67,10,40,67,62,66,10,41,67,62,67,9,43,69,62,66,11,37,67,62,67,10,41,68,62,66,9,43,67,62,67,10,42,68,62,67,10,41,68,62,47,22,18,48,62,47,13,29,62,62,61,9,43,56,62,47,9,43,28,62,61,9,43,56,62,71,12,35,78,62,71,10,41,73,62,70,10,42,72,62,68,10,41,68,62,70,14,34,71,62,68,14,35,68,62,68,13,37,68,62,68,18,32,69,62,70,10,40,71,62,70,12,37,72,62,68,14,34,68,62,70,11,30,72,62,70,11,41,71,62,70,12,40,70,62,70,10,41,71,62,70,11,39,71,62,70,10,41,71,62,68,12,39,69,62,70,10,42,72,62,68,13,37,69,62,70,12,37,71,62,68,10,42,70,62,70,11,39,71,62,70,9,43,71,62,70,10,41,71,62,70,11,40,71,62,70,12,38,71,62,68,11,39,68,62,67,18,33,68,62,67,14,33,68,62,67,10,33,68,62,67,9,43,36,62,-9,9,43,11,62,67,22,13,16,62,47,11,37,48,62,67,11,40,68,62,46,14,34,48,62,67,10,40,68,62,47,11,41,48,62,67,12,37,69,62,46,10,39,62,62,67,13,35,70,62,67,15,24,68,62,67,13,26,83,62,67,15,35,69,62,67,21,25,68,62,46,10,41,49,62,46,11,39,49,62,47,10,41,48,62,46,11,40,62,62,46,10,40,62,62,46,17,33,48,62,47,12,37,48,62,60,10,36,61,62,46,11,39,47,62,46,9,43,48,62,46,10,41,48,62,46,12,38,48,62,46,9,43,62,62,46,12,38,46,62,67,9,43,68,62,67,27,8,81,62,67,9,43,68,62,63,10,41,17],"a96":[62,0,0,0,0,62,67,25,12,73,62,67,14,34,28,62,67,9,43,68,62,67,10,41,68,62,67,10,41,68,62,67,10,41,68,62,67,23,15,34,62,67,22,24,68,62,67,15,24,68,62,57,10,42,44,62,58,10,42,49,62,18,13,33,34,62,38,9,43,11,62,16,20,21,17,62,67,14,33,68,62,67,10,41,68,62,67,12,26,68,62,67,10,40,67,62,66,10,41,67,62,67,9,43,69,62,66,11,37,67,62,67,10,41,68,62,66,9,43,67,62,67,10,42,68,62,67,10,41,68,62,47,22,18,48,62,47,13,29,62,62,61,9,43,56,62,47,9,43,28,62,61,9,43,56,62,71,12,35,78,62,71,10,41,73,62,69,10,42,71,62,68,10,41,68,62,70,14,34,71,62,68,14,35,68,62,68,13,37,68,62,68,18,32,69,62,70,10,40,71,62,69,12,37,71,62,68,14,34,68,62,69,11,30,71,62,69,11,41,71,62,69,12,40,69,62,69,10,41,71,62,69,11,39,71,62,70,10,41,71,62,68,12,39,69,62,70,10,42,72,62,68,13,37,69,62,70,12,37,71,62,68,10,42,70,62,69,11,39,70,62,69,9,43,71,62,69,10,41,71,62,69,11,40,71,62,69,12,38,71,62,68,11,39,68,62,67,18,33,68,62,67,14,33,68,62,67,10,33,68,62,67,9,43,36,62,-9,9,43,11,62,67,22,13,17,62,47,11,37,48,62,67,11,40,68,62,46,14,34,48,62,67,10,40,68,62,47,11,41,48,62,67,12,37,69,62,46,10,39,62,62,67,13,35,69,62,67,15,24,68,62,67,13,26,83,62,67,15,35,69,62,67,21,25,68,62,46,10,41,48,62,46,11,39,48,62,47,10,41,48,62,46,11,40,62,62,46,10,40,62,62,46,17,33,48,62,47,12,37,48,62,60,10,36,61,62,46,11,39,47,62,46,9,43,48,62,46,10,41,48,62,46,12,38,48,62,46,9,43,62,62,46,12,38,46,62,67,9,43,68,62,67,27,8,81,62,67,9,43,68,62,63,10,41,17],"m108":[70,0,0,0,0,70,75,28,14,82,70,75,15,40,31,70,75,10,49,77,70,75,11,48,76,70,75,11,48,76,70,75,11,48,76,70,75,26,17,38,70,75,24,28,77,70,75,17,28,77,70,64,10,50,50,70,65,10,50,56,70,20,15,37,38,70,43,10,49,12,70,18,23,23,19,70,75,16,37,77,70,75,11,48,76,70,75,13,31,77,70,75,12,45,75,70,74,10,48,75,70,75,10,49,76,70,74,13,42,75,70,75,11,48,76,70,74,11,49,75,70,75,11,48,76,70,75,11,48,76,70,52,25,20,53,70,52,15,32,70,70,68,10,49,62,70,53,10,49,32,70,68,10,49,62,70,80,14,40,87,70,80,11,49,82,70,79,11,49,80,70,77,11,50,77,70,79,16,39,80,70,77,16,41,77,70,77,15,44,77,70,77,20,37,78,70,78,12,46,80,70,79,14,43,81,70,77,16,38,77,70,79,13,35,81,70,79,12,48,80,70,79,14,47,79,70,79,11,49,80,70,79,12,47,80,70,79,11,49,80,70,77,14,46,78,70,79,11,50,80,70,77,15,43,78,70,79,14,43,81,70,77,10,50,79,70,79,12,47,80,70,79,10,50,80,70,79,11,49,80,70,79,12,46,80,70,79,12,46,80,70,77,13,44,77,70,75,20,38,76,70,75,16,37,77,70,75,12,38,76,70,75,10,49,40,70,-10,10,49,13,70,75,25,15,18,70,53,12,43,55,70,75,12,47,77,70,52,16,39,54,70,75,11,47,76,70,53,12,45,54,70,75,13,43,77,70,52,11,45,70,70,75,15,40,77,70,75,17,27,77,70,75,14,30,93,70,75,16,40,77,70,75,24,30,76,70,52,11,48,54,70,52,12,46,54,70,53,11,48,55,70,52,12,47,70,70,52,11,47,70,70,52,19,38,54,70,53,14,42,55,70,67,12,42,68,70,52,12,46,54,70,52,11,48,54,70,52,11,48,54,70,52,14,42,54,70,52,10,49,70,70,51,14,42,51,70,75,10,49,76,70,75,30,10,91,70,75,10,49,76,70,70,12,46,18],"a108":[70,0,0,0,0,70,75,28,14,82,70,75,15,40,32,70,75,10,49,77,70,75,11,48,76,70,75,11,48,76,70,75,11,48,76,70,75,26,17,38,70,75,24,28,77,70,75,17,28,77,70,64,10,50,50,70,65,10,50,56,70,20,15,37,38,70,43,10,49,12,70,18,23,23,19,70,75,16,37,77,70,75,11,48,76,70,75,13,31,77,70,75,12,45,75,70,74,10,48,75,70,75,10,49,76,70,74,13,42,75,70,75,11,48,76,70,74,11,49,75,70,75,11,48,76,70,75,11,48,76,70,52,25,20,53,70,52,15,32,70,70,68,10,49,62,70,53,10,49,32,70,68,10,49,62,70,80,14,40,87,70,80,11,49,82,70,78,11,49,79,70,77,11,50,77,70,79,16,39,80,70,77,16,41,77,70,77,15,44,77,70,77,20,37,78,70,78,12,46,80,70,78,14,43,80,70,77,16,38,77,70,78,13,35,80,70,78,12,48,80,70,78,14,47,78,70,78,11,49,80,70,78,12,47,80,70,79,11,49,80,70,77,14,46,78,70,79,11,50,80,70,77,15,43,78,70,79,14,43,81,70,77,10,50,79,70,78,12,47,79,70,78,10,50,80,70,78,11,49,80,70,78,12,46,80,70,78,12,46,80,70,77,13,44,77,70,75,20,38,76,70,75,16,37,77,70,75,12,38,76,70,75,10,49,40,70,-10,10,49,13,70,75,25,15,19,70,53,12,43,55,70,75,12,47,77,70,52,16,39,54,70,75,11,47,76,70,53,12,45,54,70,75,13,43,77,70,52,11,45,70,70,75,15,40,77,70,75,17,27,77,70,75,14,30,93,70,75,16,40,77,70,75,24,30,76,70,52,11,48,54,70,52,12,46,54,70,53,11,48,55,70,52,12,47,70,70,52,11,47,70,70,52,19,38,54,70,53,14,42,55,70,67,12,42,68,70,52,12,46,54,70,52,11,48,54,70,52,11,48,54,70,52,14,42,54,70,52,10,49,70,70,51,14,42,51,70,75,10,49,76,70,75,30,10,91,70,75,10,49,76,70,70,12,46,18]},"fonts":{}};
  if (typeof bwipjs_fonts == "object") {
    bwipjs_fonts.fontsets[1] = desc;
    bwipjs_fonts.names["OCR-B"] = 1;
  } else {
    module.exports = desc;
  }
})();
// fnt2-desc.js
(function() {
  var desc = { name:"Symbol", minsize:10, maxsize:90, minchar:48, maxchar:57,
    sizes:{10:10,20:20,30:30,40:40,50:50,60:60,70:70,80:80,90:90}, fonts:{},
    metrics:{
a10:[4,4,0,3,2,4,4,1,3,2,4,4,1,3,2,4,4,2,2,2,30,30,0,30,30],
a20:[8,8,1,6,6,8,8,1,6,6,8,8,1,7,6,8,8,1,7,6,60,60,0,60,60],
a30:[12,12,1,10,10,12,12,1,10,10,12,12,1,10,10,12,12,2,10,10,90,90,0,90,90],
a40:[14,16,0,14,14,14,16,0,14,14,14,16,0,14,14,14,16,0,14,14,120,120,0,120,120],
a50:[18,20,0,17,18,18,20,1,17,18,18,20,1,17,18,18,20,2,16,18,150,150,0,150,150],
a60:[22,24,1,20,22,22,24,1,20,22,22,24,1,21,22,22,24,1,21,22,180,180,0,180,180],
a70:[26,28,1,24,26,26,28,1,24,26,26,28,1,24,26,26,28,2,24,26,210,210,0,210,210],
a80:[28,32,1,26,30,28,32,1,26,30,28,32,1,26,30,28,32,1,26,30,240,240,0,240,240],
a90:[32,36,0,31,34,32,36,1,31,34,32,36,1,31,34,32,36,2,30,34,272,272,1,270,270],
m10:[4,4,1,2,2,4,4,1,2,2,4,4,2,2,2,4,4,2,2,2,30,30,0,30,30],
m20:[8,8,1,6,6,8,8,1,6,6,8,8,1,6,6,8,8,1,6,6,60,60,0,60,60],
m30:[12,12,2,8,10,12,12,2,8,10,12,12,2,8,10,12,12,2,8,10,90,90,0,90,90],
m40:[14,16,1,12,14,14,16,1,12,14,14,16,1,12,14,14,16,1,12,14,120,120,0,120,120],
m50:[18,20,1,16,18,18,20,1,16,18,18,20,1,16,18,18,20,1,16,18,150,150,0,150,150],
m60:[22,24,1,20,22,22,24,1,20,22,22,24,1,20,22,22,24,1,20,22,180,180,0,180,180],
m70:[26,28,2,22,26,26,28,2,22,26,26,28,2,22,26,26,28,2,22,26,210,210,0,210,210],
m80:[28,32,1,26,30,28,32,1,26,30,28,32,1,26,30,28,32,1,26,30,240,240,0,240,240],
m90:[32,36,1,30,34,32,36,1,30,34,32,36,1,30,34,32,36,1,30,34,272,272,1,270,270],
    }
  };
  if (typeof bwipjs_fonts == "object") {
    bwipjs_fonts.fontsets[2] = desc;
    bwipjs_fonts.names.SYMBOL = 2;
  } else {
    module.exports = desc;
  }
})();
// fnt3-desc.js
// $ node genfont Inconsolata fonts/Inconsolata.otf 3 108
(function() {
  var desc = {"name":"Inconsolata","minsize":8,"maxsize":108,"minchar":32,"maxchar":126,"sizes":{"8":8,"9":9,"10":10,"12":12,"16":16,"18":18,"20":20,"24":24,"27":27,"30":30,"32":32,"36":36,"40":40,"45":45,"48":48,"50":50,"54":54,"56":56,"60":60,"63":63,"64":64,"70":70,"72":72,"80":80,"81":81,"84":84,"90":90,"96":96,"108":108},"metrics":{"m8":[4,0,0,0,0,4,6,1,2,6,4,6,0,4,3,4,6,0,5,6,4,6,0,4,7,4,6,0,5,6,4,6,0,5,6,4,6,1,2,3,4,6,1,3,8,4,6,0,4,8,4,5,0,4,5,4,5,0,4,5,4,1,1,2,3,4,3,0,4,1,4,1,1,2,1,4,6,0,4,7,4,6,0,4,6,4,6,0,3,6,4,6,0,4,6,4,6,0,4,6,4,6,0,4,6,4,6,0,4,6,4,6,0,4,6,4,6,0,4,6,4,6,0,4,6,4,6,0,4,6,4,4,1,2,4,4,4,1,2,6,4,5,0,4,5,4,4,0,4,3,4,5,0,5,5,4,6,0,4,6,4,6,0,5,6,4,6,0,5,6,4,6,0,4,6,4,6,0,5,6,4,6,0,4,6,4,6,0,4,6,4,6,0,4,6,4,6,0,4,6,4,6,0,4,6,4,6,0,4,6,4,6,0,4,6,4,6,0,5,7,4,6,0,4,6,4,6,0,4,6,4,6,0,4,6,4,6,0,5,6,4,6,0,4,6,4,6,0,5,8,4,6,0,4,6,4,6,0,4,6,4,6,0,4,6,4,6,0,4,6,4,6,0,5,6,4,6,0,5,6,4,6,0,5,6,4,6,0,5,6,4,6,0,4,6,4,6,1,3,7,4,6,0,4,7,4,6,0,4,7,4,6,0,4,4,4,0,0,5,1,4,6,1,2,3,4,4,0,4,4,4,6,0,4,6,4,4,0,4,4,4,6,0,4,6,4,4,0,4,4,4,6,0,5,6,4,5,0,5,7,4,6,0,4,6,4,6,0,4,6,4,6,0,4,8,4,6,0,5,7,4,6,0,4,6,4,4,0,5,4,4,4,0,4,4,4,4,0,4,4,4,4,0,4,6,4,4,0,4,6,4,4,0,4,4,4,4,0,4,4,4,6,0,4,6,4,4,0,4,4,4,4,0,4,4,4,4,0,5,4,4,4,0,4,4,4,4,0,4,6,4,4,0,4,4,4,6,0,4,8,4,6,1,2,8,4,6,0,4,8,4,4,0,5,2],"a8":[4,0,0,0,0,4,6,1,2,6,4,6,0,4,3,4,6,0,5,6,4,6,0,4,7,4,6,0,5,6,4,6,0,5,6,4,6,1,2,3,4,6,1,3,8,4,6,0,4,8,4,5,0,4,5,4,5,0,4,5,4,1,1,2,3,4,3,0,4,1,4,1,1,2,1,4,6,0,4,7,4,6,0,4,6,4,6,0,3,6,4,6,0,4,6,4,6,0,4,6,4,6,0,4,6,4,6,0,4,6,4,6,0,4,6,4,6,0,4,6,4,6,0,4,6,4,6,0,4,6,4,4,1,2,4,4,4,1,2,6,4,5,0,4,5,4,4,0,4,3,4,5,0,5,5,4,6,0,4,6,4,6,0,5,6,4,6,0,5,6,4,6,0,4,6,4,6,0,5,6,4,6,0,4,6,4,6,0,4,6,4,6,0,4,6,4,6,0,4,6,4,6,0,4,6,4,6,0,4,6,4,6,0,4,6,4,6,0,5,7,4,6,0,4,6,4,6,0,4,6,4,6,0,4,6,4,6,0,5,6,4,6,0,4,6,4,6,0,5,8,4,6,0,4,6,4,6,0,4,6,4,6,0,4,6,4,6,0,4,6,4,6,0,5,6,4,6,0,5,6,4,6,0,5,6,4,6,0,5,6,4,6,0,4,6,4,6,1,3,7,4,6,0,4,7,4,6,0,4,7,4,6,0,4,4,4,0,0,5,1,4,6,1,2,3,4,4,0,4,4,4,6,0,4,6,4,4,0,4,4,4,6,0,4,6,4,4,0,4,4,4,6,0,5,6,4,5,0,5,7,4,6,0,4,6,4,6,0,4,6,4,6,0,4,8,4,6,0,5,7,4,6,0,4,6,4,4,0,5,4,4,4,0,4,4,4,4,0,4,4,4,4,0,4,6,4,4,0,4,6,4,4,0,4,4,4,4,0,4,4,4,6,0,4,6,4,4,0,4,4,4,4,0,4,4,4,4,0,5,4,4,4,0,4,4,4,4,0,4,6,4,4,0,4,4,4,6,0,4,8,4,6,1,2,8,4,6,0,4,8,4,4,0,5,2],"m9":[5,0,0,0,0,5,7,1,2,7,5,7,1,3,4,5,7,0,5,7,5,7,0,5,8,5,7,0,5,7,5,7,0,5,7,5,7,1,3,4,5,7,1,4,9,5,7,0,4,9,5,6,0,5,5,5,6,0,5,5,5,1,1,2,3,5,4,0,5,2,5,2,1,2,2,5,7,0,5,8,5,7,0,5,7,5,7,0,3,7,5,7,0,5,7,5,7,0,5,7,5,7,0,5,7,5,7,0,5,7,5,7,0,5,7,5,7,0,5,7,5,7,0,5,7,5,7,0,5,7,5,5,1,2,5,5,5,1,2,7,5,6,0,5,6,5,5,0,5,4,5,6,0,5,6,5,7,0,5,7,5,7,0,5,7,5,7,0,5,7,5,7,0,5,7,5,7,0,5,7,5,7,0,5,7,5,7,0,5,7,5,7,0,5,7,5,7,0,5,7,5,7,0,5,7,5,7,0,4,7,5,7,0,5,7,5,7,0,5,8,5,7,0,5,7,5,7,0,5,7,5,7,0,5,7,5,7,0,5,7,5,7,0,5,7,5,7,0,5,9,5,7,0,5,7,5,7,0,5,7,5,7,0,5,7,5,7,0,5,7,5,7,0,5,7,5,7,0,5,7,5,7,0,5,7,5,7,0,5,7,5,7,0,5,7,5,7,1,3,8,5,7,0,5,8,5,7,0,4,8,5,7,0,4,4,5,0,0,5,1,5,7,1,3,3,5,5,0,5,5,5,7,0,5,7,5,5,0,5,5,5,7,0,5,7,5,5,0,5,5,5,7,0,5,7,5,5,0,5,7,5,7,0,5,7,5,7,0,4,7,5,7,0,4,9,5,7,0,5,8,5,7,0,5,7,5,5,0,5,5,5,5,0,5,5,5,5,0,5,5,5,5,0,5,7,5,5,0,5,7,5,5,0,5,5,5,5,0,5,5,5,6,0,5,6,5,5,0,5,5,5,5,0,5,5,5,5,0,5,5,5,5,0,5,5,5,5,0,5,7,5,5,0,5,5,5,7,0,4,9,5,7,2,1,9,5,7,0,5,9,5,5,0,5,2],"a9":[5,0,0,0,0,5,7,1,2,7,5,7,1,3,4,5,7,0,5,7,5,7,0,5,8,5,7,0,5,7,5,7,0,5,7,5,7,1,3,4,5,7,1,4,9,5,7,0,4,9,5,6,0,5,5,5,6,0,5,5,5,1,1,2,3,5,4,0,5,2,5,2,1,2,2,5,7,0,5,8,5,7,0,5,7,5,7,0,3,7,5,7,0,5,7,5,7,0,5,7,5,7,0,5,7,5,7,0,5,7,5,7,0,5,7,5,7,0,5,7,5,7,0,5,7,5,7,0,5,7,5,5,1,2,5,5,5,1,2,7,5,6,0,5,6,5,5,0,5,4,5,6,0,5,6,5,7,0,5,7,5,7,0,5,7,5,7,0,5,7,5,7,0,5,7,5,7,0,5,7,5,7,0,5,7,5,7,0,5,7,5,7,0,5,7,5,7,0,5,7,5,7,0,5,7,5,7,0,4,7,5,7,0,5,7,5,7,0,5,8,5,7,0,5,7,5,7,0,5,7,5,7,0,5,7,5,7,0,5,7,5,7,0,5,7,5,7,0,5,9,5,7,0,5,7,5,7,0,5,7,5,7,0,5,7,5,7,0,5,7,5,7,0,5,7,5,7,0,5,7,5,7,0,5,7,5,7,0,5,7,5,7,0,5,7,5,7,1,3,8,5,7,0,5,8,5,7,0,4,8,5,7,0,4,4,5,0,0,5,1,5,7,1,3,3,5,5,0,5,5,5,7,0,5,7,5,5,0,5,5,5,7,0,5,7,5,5,0,5,5,5,7,0,5,7,5,5,0,5,7,5,7,0,5,7,5,7,0,4,7,5,7,0,4,9,5,7,0,5,8,5,7,0,5,7,5,5,0,5,5,5,5,0,5,5,5,5,0,5,5,5,5,0,5,7,5,5,0,5,7,5,5,0,5,5,5,5,0,5,5,5,6,0,5,6,5,5,0,5,5,5,5,0,5,5,5,5,0,5,5,5,5,0,5,5,5,5,0,5,7,5,5,0,5,5,5,7,0,4,9,5,7,2,1,9,5,7,0,5,9,5,5,0,5,2],"m10":[5,0,0,0,0,5,8,1,3,8,5,8,1,4,4,5,7,0,6,7,5,8,0,5,9,5,7,0,6,7,5,7,0,6,7,5,8,2,2,4,5,8,1,4,10,5,8,0,4,10,5,6,0,5,5,5,6,0,5,5,5,2,1,3,4,5,4,0,5,1,5,2,1,3,2,5,8,0,5,9,5,7,0,5,7,5,7,0,4,7,5,7,0,5,7,5,7,0,5,7,5,7,0,5,7,5,7,0,5,7,5,7,0,5,7,5,7,0,5,7,5,7,0,5,7,5,7,0,5,7,5,5,1,3,5,5,5,1,3,7,5,7,0,5,7,5,5,0,5,4,5,7,0,6,7,5,8,0,5,8,5,7,0,6,7,5,7,0,6,7,5,7,0,5,7,5,7,0,6,7,5,7,0,5,7,5,7,0,5,7,5,7,0,5,7,5,7,0,5,7,5,7,0,5,7,5,7,0,5,7,5,7,0,5,7,5,7,0,6,8,5,7,0,5,7,5,7,0,5,7,5,7,0,5,7,5,7,0,6,7,5,7,0,5,7,5,7,0,6,9,5,7,0,6,7,5,7,0,5,7,5,7,0,5,7,5,7,0,5,7,5,7,0,6,7,5,7,0,6,7,5,7,0,6,7,5,7,0,6,7,5,7,0,5,7,5,8,1,4,10,5,8,0,5,9,5,8,0,4,10,5,7,1,4,4,5,0,0,6,1,5,8,1,3,4,5,5,0,5,5,5,8,0,5,8,5,5,0,5,5,5,8,0,5,8,5,5,0,5,5,5,8,0,6,8,5,6,0,6,8,5,8,0,5,8,5,8,1,4,8,5,8,0,5,10,5,8,0,6,9,5,8,0,5,8,5,5,0,6,5,5,5,0,5,5,5,5,0,5,5,5,5,0,5,7,5,5,0,5,7,5,5,1,4,5,5,5,0,5,5,5,7,0,5,7,5,5,0,5,5,5,5,0,5,5,5,5,0,6,5,5,5,0,5,5,5,5,0,5,7,5,5,0,5,5,5,7,0,5,9,5,8,2,2,10,5,7,1,4,9,5,5,0,6,2],"a10":[5,0,0,0,0,5,8,1,3,8,5,8,1,4,4,5,7,0,6,7,5,8,0,5,9,5,7,0,6,7,5,7,0,6,7,5,8,2,2,4,5,8,1,4,10,5,8,0,4,10,5,6,0,5,5,5,6,0,5,5,5,2,1,3,4,5,4,0,5,1,5,2,1,3,2,5,8,0,5,9,5,7,0,5,7,5,7,0,4,7,5,7,0,5,7,5,7,0,5,7,5,7,0,5,7,5,7,0,5,7,5,7,0,5,7,5,7,0,5,7,5,7,0,5,7,5,7,0,5,7,5,5,1,3,5,5,5,1,3,7,5,7,0,5,7,5,5,0,5,4,5,7,0,6,7,5,8,0,5,8,5,7,0,6,7,5,7,0,6,7,5,7,0,5,7,5,7,0,6,7,5,7,0,5,7,5,7,0,5,7,5,7,0,5,7,5,7,0,5,7,5,7,0,5,7,5,7,0,5,7,5,7,0,5,7,5,7,0,6,8,5,7,0,5,7,5,7,0,5,7,5,7,0,5,7,5,7,0,6,7,5,7,0,5,7,5,7,0,6,9,5,7,0,6,7,5,7,0,5,7,5,7,0,5,7,5,7,0,5,7,5,7,0,6,7,5,7,0,6,7,5,7,0,6,7,5,7,0,6,7,5,7,0,5,7,5,8,1,4,10,5,8,0,5,9,5,8,0,4,10,5,7,1,4,4,5,0,0,6,1,5,8,1,3,4,5,5,0,5,5,5,8,0,5,8,5,5,0,5,5,5,8,0,5,8,5,5,0,5,5,5,8,0,6,8,5,6,0,6,8,5,8,0,5,8,5,8,1,4,8,5,8,0,5,10,5,8,0,6,9,5,8,0,5,8,5,5,0,6,5,5,5,0,5,5,5,5,0,5,5,5,5,0,5,7,5,5,0,5,7,5,5,1,4,5,5,5,0,5,5,5,7,0,5,7,5,5,0,5,5,5,5,0,5,5,5,5,0,6,5,5,5,0,5,5,5,5,0,5,7,5,5,0,5,5,5,7,0,5,9,5,8,2,2,10,5,7,1,4,9,5,5,0,6,2],"m12":[6,0,0,0,0,6,9,2,2,9,6,9,1,5,4,6,9,0,7,9,6,9,0,6,10,6,9,0,7,9,6,9,0,7,10,6,9,2,3,4,6,9,1,5,12,6,9,0,5,12,6,7,0,6,6,6,7,0,6,6,6,2,2,2,5,6,5,0,6,2,6,2,2,2,2,6,9,0,6,10,6,9,0,6,9,6,9,1,3,9,6,9,0,6,9,6,9,0,6,9,6,9,0,6,9,6,9,0,6,9,6,9,0,6,9,6,9,1,5,9,6,9,0,6,9,6,9,0,6,9,6,6,2,2,6,6,6,2,2,9,6,8,0,6,8,6,6,0,6,4,6,8,0,7,8,6,9,0,6,9,6,9,0,7,9,6,9,0,7,9,6,9,0,6,9,6,9,0,7,9,6,9,0,6,9,6,9,0,6,9,6,9,1,5,9,6,9,0,6,9,6,9,0,6,9,6,9,1,5,9,6,9,0,6,9,6,9,0,7,10,6,9,0,6,9,6,9,0,6,9,6,9,0,6,9,6,9,0,7,9,6,9,0,6,9,6,9,0,7,12,6,9,0,7,9,6,9,0,6,9,6,9,0,6,9,6,9,0,6,9,6,9,0,7,9,6,9,0,7,9,6,9,0,7,9,6,9,0,7,9,6,9,0,6,9,6,9,1,5,11,6,9,0,6,10,6,9,1,4,11,6,9,1,5,5,6,0,0,7,2,6,9,1,4,4,6,6,0,6,6,6,9,0,6,9,6,6,0,6,6,6,9,0,6,9,6,6,0,6,6,6,9,0,7,9,6,7,0,7,10,6,9,0,6,9,6,9,1,5,9,6,9,0,5,12,6,9,0,7,10,6,9,1,5,9,6,6,0,7,6,6,6,0,6,6,6,6,0,6,6,6,6,0,6,9,6,6,0,6,9,6,6,1,5,6,6,6,0,6,6,6,8,0,6,8,6,6,0,6,6,6,6,0,6,6,6,6,0,7,6,6,6,0,6,6,6,6,0,6,9,6,6,0,6,6,6,9,0,6,12,6,9,2,2,11,6,9,1,5,12,6,6,0,7,2],"a12":[6,0,0,0,0,6,9,2,2,9,6,9,1,5,4,6,9,0,7,9,6,9,0,6,10,6,9,0,7,9,6,9,0,7,10,6,9,2,3,4,6,9,1,5,12,6,9,0,5,12,6,7,0,6,6,6,7,0,6,6,6,2,2,2,5,6,5,0,6,2,6,2,2,2,2,6,9,0,6,10,6,9,0,6,9,6,9,1,3,9,6,9,0,6,9,6,9,0,6,9,6,9,0,6,9,6,9,0,6,9,6,9,0,6,9,6,9,1,5,9,6,9,0,6,9,6,9,0,6,9,6,6,2,2,6,6,6,2,2,9,6,8,0,6,8,6,6,0,6,4,6,8,0,7,8,6,9,0,6,9,6,9,0,7,9,6,9,0,7,9,6,9,0,6,9,6,9,0,7,9,6,9,0,6,9,6,9,0,6,9,6,9,1,5,9,6,9,0,6,9,6,9,0,6,9,6,9,1,5,9,6,9,0,6,9,6,9,0,7,10,6,9,0,6,9,6,9,0,6,9,6,9,0,6,9,6,9,0,7,9,6,9,0,6,9,6,9,0,7,12,6,9,0,7,9,6,9,0,6,9,6,9,0,6,9,6,9,0,6,9,6,9,0,7,9,6,9,0,7,9,6,9,0,7,9,6,9,0,7,9,6,9,0,6,9,6,9,1,5,11,6,9,0,6,10,6,9,1,4,11,6,9,1,5,5,6,0,0,7,2,6,9,1,4,4,6,6,0,6,6,6,9,0,6,9,6,6,0,6,6,6,9,0,6,9,6,6,0,6,6,6,9,0,7,9,6,7,0,7,10,6,9,0,6,9,6,9,1,5,9,6,9,0,5,12,6,9,0,7,10,6,9,1,5,9,6,6,0,7,6,6,6,0,6,6,6,6,0,6,6,6,6,0,6,9,6,6,0,6,9,6,6,1,5,6,6,6,0,6,6,6,8,0,6,8,6,6,0,6,6,6,6,0,6,6,6,6,0,7,6,6,6,0,6,6,6,6,0,6,9,6,6,0,6,6,6,9,0,6,12,6,9,2,2,11,6,9,1,5,12,6,6,0,7,2],"m16":[9,0,0,0,0,9,12,3,3,12,9,12,1,6,5,9,11,0,9,11,9,12,1,7,13,9,11,0,9,11,9,11,0,9,12,9,12,3,3,5,9,12,2,6,15,9,12,1,6,16,9,9,0,8,8,9,10,0,8,9,9,2,2,4,5,9,6,1,7,2,9,2,3,3,2,9,12,1,7,13,9,11,0,8,11,9,11,1,5,11,9,11,1,7,11,9,11,1,7,11,9,11,0,8,11,9,11,1,7,11,9,11,1,7,11,9,11,1,7,11,9,11,1,7,11,9,11,1,7,11,9,8,3,3,8,9,8,2,4,11,9,10,0,8,10,9,8,0,8,6,9,10,0,9,10,9,12,1,7,12,9,11,0,9,11,9,11,0,9,11,9,11,0,8,11,9,11,0,9,11,9,11,0,8,11,9,11,1,7,11,9,11,1,7,11,9,11,0,8,11,9,11,0,8,11,9,11,1,6,11,9,11,0,8,11,9,11,0,9,12,9,11,1,7,11,9,11,0,8,11,9,11,0,8,11,9,11,0,9,11,9,11,1,7,11,9,11,0,9,14,9,11,0,9,11,9,11,0,8,11,9,11,0,8,11,9,11,0,8,11,9,11,0,9,11,9,11,0,9,11,9,11,0,9,11,9,11,0,9,11,9,11,0,8,11,9,12,2,6,14,9,12,1,7,13,9,12,1,6,14,9,11,1,6,6,9,0,0,9,2,9,12,2,4,5,9,8,0,8,8,9,12,1,7,12,9,8,0,8,8,9,12,0,8,12,9,8,0,8,8,9,12,1,8,12,9,9,0,9,12,9,12,1,7,12,9,12,1,6,12,9,12,0,7,16,9,12,1,8,13,9,12,1,7,12,9,8,0,9,8,9,8,1,7,8,9,8,0,8,8,9,8,1,7,11,9,8,0,8,11,9,8,1,7,8,9,8,0,8,8,9,11,1,7,11,9,8,1,7,8,9,8,0,8,8,9,8,0,9,8,9,8,0,8,8,9,8,0,8,11,9,8,0,8,8,9,12,0,8,15,9,12,3,2,15,9,12,1,7,15,9,8,0,9,3],"a16":[9,0,0,0,0,9,12,3,3,12,9,12,1,6,5,9,11,0,9,11,9,12,1,7,13,9,11,0,9,11,9,11,0,9,12,9,12,3,3,5,9,12,2,6,15,9,12,1,6,16,9,9,0,8,8,9,10,0,8,9,9,2,2,4,5,9,6,1,7,2,9,2,3,3,2,9,12,1,7,13,9,11,0,8,11,9,11,1,5,11,9,11,1,7,11,9,11,1,7,11,9,11,0,8,11,9,11,1,7,11,9,11,1,7,11,9,11,1,7,11,9,11,1,7,11,9,11,1,7,11,9,8,3,3,8,9,8,2,4,11,9,10,0,8,10,9,8,0,8,6,9,10,0,9,10,9,12,1,7,12,9,11,0,9,11,9,11,0,9,11,9,11,0,8,11,9,11,0,9,11,9,11,0,8,11,9,11,1,7,11,9,11,1,7,11,9,11,0,8,11,9,11,0,8,11,9,11,1,6,11,9,11,0,8,11,9,11,0,9,12,9,11,1,7,11,9,11,0,8,11,9,11,0,8,11,9,11,0,9,11,9,11,1,7,11,9,11,0,9,14,9,11,0,9,11,9,11,0,8,11,9,11,0,8,11,9,11,0,8,11,9,11,0,9,11,9,11,0,9,11,9,11,0,9,11,9,11,0,9,11,9,11,0,8,11,9,12,2,6,14,9,12,1,7,13,9,12,1,6,14,9,11,1,6,6,9,0,0,9,2,9,12,2,4,5,9,8,0,8,8,9,12,1,7,12,9,8,0,8,8,9,12,0,8,12,9,8,0,8,8,9,12,1,8,12,9,9,0,9,12,9,12,1,7,12,9,12,1,6,12,9,12,0,7,16,9,12,1,8,13,9,12,1,7,12,9,8,0,9,8,9,8,1,7,8,9,8,0,8,8,9,8,1,7,11,9,8,0,8,11,9,8,1,7,8,9,8,0,8,8,9,11,1,7,11,9,8,1,7,8,9,8,0,8,8,9,8,0,9,8,9,8,0,8,8,9,8,0,8,11,9,8,0,8,8,9,12,0,8,15,9,12,3,2,15,9,12,1,7,15,9,8,0,9,3],"m18":[10,0,0,0,0,10,13,3,3,13,10,13,2,6,6,10,13,0,10,13,10,13,1,8,14,10,13,0,10,13,10,13,0,10,14,10,13,4,3,6,10,13,2,7,17,10,13,1,7,17,10,11,0,9,9,10,11,0,9,9,10,2,3,3,6,10,7,1,8,2,10,3,3,3,3,10,13,1,8,14,10,13,0,9,13,10,13,1,5,13,10,13,1,8,13,10,13,1,8,13,10,13,0,9,13,10,13,1,8,13,10,13,1,8,13,10,13,1,8,13,10,13,1,8,13,10,13,1,8,13,10,9,3,3,9,10,9,3,3,13,10,11,0,9,11,10,9,0,9,6,10,11,0,10,11,10,13,1,8,13,10,13,0,10,13,10,13,0,10,13,10,13,0,9,13,10,13,0,10,13,10,13,1,8,13,10,13,1,8,13,10,13,1,8,13,10,13,0,9,13,10,13,1,8,13,10,13,1,7,13,10,13,0,9,13,10,13,0,10,14,10,13,1,8,13,10,13,0,9,13,10,13,0,9,13,10,13,0,10,13,10,13,1,8,13,10,13,0,10,17,10,13,1,9,13,10,13,0,9,13,10,13,0,9,13,10,13,0,9,13,10,13,0,10,13,10,13,0,10,13,10,13,0,10,13,10,13,0,10,13,10,13,0,9,13,10,13,2,7,15,10,13,1,8,14,10,13,1,7,15,10,13,1,7,7,10,0,0,10,2,10,14,2,5,6,10,9,0,9,9,10,13,1,8,13,10,9,1,8,9,10,13,0,9,13,10,9,0,9,9,10,13,1,9,13,10,10,0,10,14,10,13,1,8,13,10,13,1,7,13,10,13,0,8,17,10,13,1,9,14,10,13,1,8,13,10,9,0,10,9,10,9,1,8,9,10,9,0,9,9,10,9,1,8,13,10,9,0,9,13,10,9,1,8,9,10,9,1,8,9,10,12,1,8,12,10,9,1,8,9,10,9,0,9,9,10,9,0,10,9,10,9,0,9,9,10,9,0,9,13,10,9,0,9,9,10,13,1,7,17,10,13,4,2,16,10,13,1,8,17,10,9,0,10,3],"a18":[10,0,0,0,0,10,13,3,3,13,10,13,2,6,6,10,13,0,10,13,10,13,1,8,14,10,13,0,10,13,10,13,0,10,14,10,13,4,3,6,10,13,2,7,17,10,13,1,7,17,10,11,0,9,9,10,11,0,9,9,10,2,3,3,6,10,7,1,8,2,10,3,3,3,3,10,13,1,8,14,10,13,0,9,13,10,13,1,5,13,10,13,1,8,13,10,13,1,8,13,10,13,0,9,13,10,13,1,8,13,10,13,1,8,13,10,13,1,8,13,10,13,1,8,13,10,13,1,8,13,10,9,3,3,9,10,9,3,3,13,10,11,0,9,11,10,9,0,9,6,10,11,0,10,11,10,13,1,8,13,10,13,0,10,13,10,13,0,10,13,10,13,0,9,13,10,13,0,10,13,10,13,1,8,13,10,13,1,8,13,10,13,1,8,13,10,13,0,9,13,10,13,1,8,13,10,13,1,7,13,10,13,0,9,13,10,13,0,10,14,10,13,1,8,13,10,13,0,9,13,10,13,0,9,13,10,13,0,10,13,10,13,1,8,13,10,13,0,10,17,10,13,1,9,13,10,13,0,9,13,10,13,0,9,13,10,13,0,9,13,10,13,0,10,13,10,13,0,10,13,10,13,0,10,13,10,13,0,10,13,10,13,0,9,13,10,13,2,7,15,10,13,1,8,14,10,13,1,7,15,10,13,1,7,7,10,0,0,10,2,10,14,2,5,6,10,9,0,9,9,10,13,1,8,13,10,9,1,8,9,10,13,0,9,13,10,9,0,9,9,10,13,1,9,13,10,10,0,10,14,10,13,1,8,13,10,13,1,7,13,10,13,0,8,17,10,13,1,9,14,10,13,1,8,13,10,9,0,10,9,10,9,1,8,9,10,9,0,9,9,10,9,1,8,13,10,9,0,9,13,10,9,1,8,9,10,9,1,8,9,10,12,1,8,12,10,9,1,8,9,10,9,0,9,9,10,9,0,10,9,10,9,0,9,9,10,9,0,9,13,10,9,0,9,9,10,13,1,7,17,10,13,4,2,16,10,13,1,8,17,10,9,0,10,3],"m20":[11,0,0,0,0,11,15,3,4,15,11,15,2,7,7,11,14,0,11,14,11,15,1,9,17,11,14,0,11,14,11,14,0,11,15,11,15,4,3,7,11,15,2,8,19,11,15,1,7,19,11,12,0,10,10,11,12,0,10,10,11,3,3,4,7,11,8,1,9,2,11,3,3,4,3,11,15,1,9,16,11,14,1,9,14,11,14,1,6,14,11,14,1,9,14,11,14,1,9,14,11,14,1,9,14,11,14,1,9,14,11,14,1,9,14,11,14,1,9,14,11,14,1,9,14,11,14,1,9,14,11,10,3,4,10,11,10,3,4,14,11,13,0,10,12,11,10,0,10,7,11,13,0,11,12,11,15,1,9,15,11,14,0,11,14,11,14,0,11,14,11,14,1,9,14,11,14,0,11,14,11,14,1,9,14,11,14,1,9,14,11,14,1,9,14,11,14,0,10,14,11,14,1,9,14,11,14,1,8,14,11,14,1,9,14,11,14,0,11,15,11,14,1,9,14,11,14,0,10,14,11,14,1,9,14,11,14,0,11,14,11,14,1,9,14,11,14,0,11,18,11,14,1,10,14,11,14,1,9,14,11,14,0,10,14,11,14,1,9,14,11,14,0,11,14,11,14,0,11,14,11,14,0,11,14,11,14,0,11,14,11,14,1,9,14,11,15,2,7,18,11,15,1,9,16,11,15,1,7,18,11,14,2,7,7,11,0,0,11,2,11,15,3,4,6,11,10,1,9,10,11,15,1,9,15,11,10,1,9,10,11,15,0,10,15,11,10,1,9,10,11,15,1,10,15,11,11,0,11,15,11,15,1,9,15,11,15,2,7,15,11,15,1,8,19,11,15,1,10,16,11,15,1,9,15,11,10,0,11,10,11,10,1,9,10,11,10,0,10,10,11,10,1,9,14,11,10,0,10,14,11,10,2,8,10,11,10,1,9,10,11,13,1,9,13,11,10,1,9,10,11,10,0,10,10,11,10,0,11,10,11,10,1,9,10,11,10,0,10,14,11,10,1,9,10,11,14,1,8,18,11,15,4,3,19,11,14,1,9,18,11,10,0,11,4],"a20":[11,0,0,0,0,11,15,3,4,15,11,15,2,7,7,11,14,0,11,14,11,15,1,9,17,11,14,0,11,14,11,14,0,11,15,11,15,4,3,7,11,15,2,8,19,11,15,1,7,19,11,12,0,10,10,11,12,0,10,10,11,3,3,4,7,11,8,1,9,2,11,3,3,4,3,11,15,1,9,16,11,14,1,9,14,11,14,1,6,14,11,14,1,9,14,11,14,1,9,14,11,14,1,9,14,11,14,1,9,14,11,14,1,9,14,11,14,1,9,14,11,14,1,9,14,11,14,1,9,14,11,10,3,4,10,11,10,3,4,14,11,13,0,10,12,11,10,0,10,7,11,13,0,11,12,11,15,1,9,15,11,14,0,11,14,11,14,0,11,14,11,14,1,9,14,11,14,0,11,14,11,14,1,9,14,11,14,1,9,14,11,14,1,9,14,11,14,0,10,14,11,14,1,9,14,11,14,1,8,14,11,14,1,9,14,11,14,0,11,15,11,14,1,9,14,11,14,0,10,14,11,14,1,9,14,11,14,0,11,14,11,14,1,9,14,11,14,0,11,18,11,14,1,10,14,11,14,1,9,14,11,14,0,10,14,11,14,1,9,14,11,14,0,11,14,11,14,0,11,14,11,14,0,11,14,11,14,0,11,14,11,14,1,9,14,11,15,2,7,18,11,15,1,9,16,11,15,1,7,18,11,14,2,7,7,11,0,0,11,2,11,15,3,4,6,11,10,1,9,10,11,15,1,9,15,11,10,1,9,10,11,15,0,10,15,11,10,1,9,10,11,15,1,10,15,11,11,0,11,15,11,15,1,9,15,11,15,2,7,15,11,15,1,8,19,11,15,1,10,16,11,15,1,9,15,11,10,0,11,10,11,10,1,9,10,11,10,0,10,10,11,10,1,9,14,11,10,0,10,14,11,10,2,8,10,11,10,1,9,10,11,13,1,9,13,11,10,1,9,10,11,10,0,10,10,11,10,0,11,10,11,10,1,9,10,11,10,0,10,14,11,10,1,9,10,11,14,1,8,18,11,15,4,3,19,11,14,1,9,18,11,10,0,11,4],"m24":[13,0,0,0,0,13,18,4,4,18,13,18,2,9,8,13,17,0,13,17,13,17,1,11,19,13,17,0,13,17,13,17,0,13,18,13,18,5,4,8,13,18,3,9,23,13,18,1,9,23,13,14,1,11,12,13,14,1,11,12,13,3,4,4,8,13,9,1,11,2,13,4,4,4,4,13,18,1,11,20,13,17,1,11,17,13,17,2,6,17,13,17,1,11,17,13,17,1,10,17,13,17,1,11,17,13,17,1,11,17,13,17,1,11,17,13,17,2,10,17,13,17,1,11,17,13,17,1,11,17,13,12,4,4,12,13,11,4,4,16,13,15,0,12,14,13,12,1,11,8,13,15,1,12,14,13,18,1,11,18,13,17,0,13,17,13,17,0,13,17,13,17,1,11,17,13,17,1,12,17,13,17,1,11,17,13,17,1,11,17,13,17,2,10,17,13,17,0,12,17,13,17,1,11,17,13,17,2,9,17,13,17,1,11,17,13,17,1,12,18,13,17,1,11,17,13,17,1,11,17,13,17,1,11,17,13,17,0,13,17,13,17,1,11,17,13,17,0,13,22,13,17,1,12,17,13,17,1,11,17,13,17,0,12,17,13,17,1,11,17,13,17,0,13,17,13,17,0,13,17,13,17,1,12,17,13,17,0,13,17,13,17,1,11,17,13,18,3,8,21,13,18,1,11,20,13,18,2,8,21,13,17,2,9,9,13,0,0,13,3,13,18,3,6,7,13,12,1,11,12,13,18,1,11,18,13,12,1,11,12,13,18,1,11,18,13,12,1,11,12,13,18,1,12,18,13,13,0,13,18,13,18,1,11,18,13,17,2,9,17,13,17,1,9,22,13,18,1,12,19,13,18,2,9,18,13,12,0,13,12,13,12,1,11,12,13,12,1,11,12,13,12,1,11,17,13,12,1,11,17,13,12,2,10,12,13,12,1,11,12,13,16,1,11,16,13,12,1,11,12,13,12,1,11,12,13,12,0,13,12,13,12,1,11,12,13,12,0,12,17,13,12,1,11,12,13,17,1,10,22,13,17,5,3,21,13,17,2,10,22,13,12,1,12,4],"a24":[13,0,0,0,0,13,18,4,4,18,13,18,2,9,8,13,17,0,13,17,13,17,1,11,19,13,17,0,13,17,13,17,0,13,18,13,18,5,4,8,13,18,3,9,23,13,18,1,9,23,13,14,1,11,12,13,14,1,11,12,13,3,4,4,8,13,9,1,11,2,13,4,4,4,4,13,18,1,11,20,13,17,1,11,17,13,17,2,6,17,13,17,1,11,17,13,17,1,10,17,13,17,1,11,17,13,17,1,11,17,13,17,1,11,17,13,17,2,10,17,13,17,1,11,17,13,17,1,11,17,13,12,4,4,12,13,11,4,4,16,13,15,0,12,14,13,12,1,11,8,13,15,1,12,14,13,18,1,11,18,13,17,0,13,17,13,17,0,13,17,13,17,1,11,17,13,17,1,12,17,13,17,1,11,17,13,17,1,11,17,13,17,2,10,17,13,17,0,12,17,13,17,1,11,17,13,17,2,9,17,13,17,1,11,17,13,17,1,12,18,13,17,1,11,17,13,17,1,11,17,13,17,1,11,17,13,17,0,13,17,13,17,1,11,17,13,17,0,13,22,13,17,1,12,17,13,17,1,11,17,13,17,0,12,17,13,17,1,11,17,13,17,0,13,17,13,17,0,13,17,13,17,1,12,17,13,17,0,13,17,13,17,1,11,17,13,18,3,8,21,13,18,1,11,20,13,18,2,8,21,13,17,2,9,9,13,0,0,13,3,13,18,3,6,7,13,12,1,11,12,13,18,1,11,18,13,12,1,11,12,13,18,1,11,18,13,12,1,11,12,13,18,1,12,18,13,13,0,13,18,13,18,1,11,18,13,17,2,9,17,13,17,1,9,22,13,18,1,12,19,13,18,2,9,18,13,12,0,13,12,13,12,1,11,12,13,12,1,11,12,13,12,1,11,17,13,12,1,11,17,13,12,2,10,12,13,12,1,11,12,13,16,1,11,16,13,12,1,11,12,13,12,1,11,12,13,12,0,13,12,13,12,1,11,12,13,12,0,12,17,13,12,1,11,12,13,17,1,10,22,13,17,5,3,21,13,17,2,10,22,13,12,1,12,4],"m27":[15,0,0,0,0,15,20,5,4,20,15,20,3,9,9,15,19,0,14,19,15,19,1,13,21,15,19,0,14,19,15,19,1,13,20,15,20,5,5,9,15,20,3,10,26,15,20,2,9,26,15,16,1,13,13,15,16,1,12,13,15,3,4,5,8,15,11,2,11,3,15,4,5,4,4,15,20,1,12,22,15,19,1,12,19,15,19,2,7,19,15,19,2,11,19,15,19,1,12,19,15,19,1,12,19,15,19,1,12,19,15,19,1,12,19,15,19,2,11,19,15,19,1,12,19,15,19,1,12,19,15,13,5,4,13,15,13,4,5,18,15,17,0,14,16,15,13,1,12,9,15,17,1,13,16,15,20,1,12,20,15,19,0,14,19,15,19,0,14,19,15,19,1,13,19,15,19,1,13,19,15,19,1,13,19,15,19,1,12,19,15,19,2,11,19,15,19,1,13,19,15,19,1,12,19,15,19,2,10,19,15,19,1,13,19,15,19,1,13,20,15,19,1,12,19,15,19,1,13,19,15,19,1,12,19,15,19,0,14,19,15,19,1,12,19,15,19,0,14,24,15,19,1,13,19,15,19,1,12,19,15,19,0,14,19,15,19,1,12,19,15,19,0,14,19,15,19,0,14,19,15,19,1,13,19,15,19,0,14,19,15,19,1,13,19,15,20,3,9,23,15,20,1,12,22,15,20,2,9,23,15,19,2,10,10,15,0,0,14,3,15,20,3,7,8,15,14,1,12,14,15,20,1,13,20,15,14,1,13,14,15,20,1,12,20,15,14,1,12,14,15,20,1,13,20,15,14,1,13,19,15,20,1,12,20,15,20,2,10,20,15,20,1,10,26,15,20,1,13,21,15,20,2,11,20,15,14,1,13,14,15,14,1,12,14,15,14,1,13,14,15,14,1,13,19,15,14,1,12,19,15,14,2,12,14,15,14,1,12,14,15,18,1,12,18,15,14,1,12,14,15,14,1,13,14,15,14,0,14,14,15,14,1,12,14,15,14,0,14,19,15,14,1,12,14,15,19,1,11,24,15,20,6,3,25,15,19,2,11,24,15,14,1,13,5],"a27":[15,0,0,0,0,15,20,5,4,20,15,20,3,9,9,15,19,0,14,19,15,19,1,13,21,15,19,0,14,19,15,19,1,13,20,15,20,5,5,9,15,20,3,10,26,15,20,2,9,26,15,16,1,13,13,15,16,1,12,13,15,3,4,5,8,15,11,2,11,3,15,4,5,4,4,15,20,1,12,22,15,19,1,12,19,15,19,2,7,19,15,19,2,11,19,15,19,1,12,19,15,19,1,12,19,15,19,1,12,19,15,19,1,12,19,15,19,2,11,19,15,19,1,12,19,15,19,1,12,19,15,13,5,4,13,15,13,4,5,18,15,17,0,14,16,15,13,1,12,9,15,17,1,13,16,15,20,1,12,20,15,19,0,14,19,15,19,0,14,19,15,19,1,13,19,15,19,1,13,19,15,19,1,13,19,15,19,1,12,19,15,19,2,11,19,15,19,1,13,19,15,19,1,12,19,15,19,2,10,19,15,19,1,13,19,15,19,1,13,20,15,19,1,12,19,15,19,1,13,19,15,19,1,12,19,15,19,0,14,19,15,19,1,12,19,15,19,0,14,24,15,19,1,13,19,15,19,1,12,19,15,19,0,14,19,15,19,1,12,19,15,19,0,14,19,15,19,0,14,19,15,19,1,13,19,15,19,0,14,19,15,19,1,13,19,15,20,3,9,23,15,20,1,12,22,15,20,2,9,23,15,19,2,10,10,15,0,0,14,3,15,20,3,7,8,15,14,1,12,14,15,20,1,13,20,15,14,1,13,14,15,20,1,12,20,15,14,1,12,14,15,20,1,13,20,15,14,1,13,19,15,20,1,12,20,15,20,2,10,20,15,20,1,10,26,15,20,1,13,21,15,20,2,11,20,15,14,1,13,14,15,14,1,12,14,15,14,1,13,14,15,14,1,13,19,15,14,1,12,19,15,14,2,12,14,15,14,1,12,14,15,18,1,12,18,15,14,1,12,14,15,14,1,13,14,15,14,0,14,14,15,14,1,12,14,15,14,0,14,19,15,14,1,12,14,15,19,1,11,24,15,20,6,3,25,15,19,2,11,24,15,14,1,13,5],"m30":[16,0,0,0,0,16,22,5,5,22,16,22,3,10,9,16,21,0,16,21,16,22,1,14,24,16,21,0,16,21,16,21,1,15,22,16,22,6,5,9,16,22,4,10,28,16,22,2,10,28,16,17,1,14,14,16,17,1,14,14,16,4,5,5,10,16,12,2,12,3,16,4,5,5,4,16,22,2,13,24,16,21,1,14,21,16,21,2,8,21,16,21,2,13,21,16,21,2,12,21,16,21,1,14,21,16,21,2,13,21,16,21,2,13,21,16,21,2,12,21,16,21,1,14,21,16,21,2,13,21,16,15,5,5,15,16,14,5,5,20,16,19,1,14,18,16,15,1,14,10,16,19,1,15,18,16,22,2,13,22,16,21,1,15,21,16,21,0,16,21,16,21,1,14,21,16,21,1,15,21,16,21,1,14,21,16,21,1,14,21,16,21,2,13,21,16,21,1,14,21,16,21,1,14,21,16,21,2,12,21,16,21,1,14,21,16,21,1,15,22,16,21,2,13,21,16,21,1,14,21,16,21,1,14,21,16,21,0,16,21,16,21,1,14,21,16,21,0,16,27,16,21,1,15,21,16,21,1,14,21,16,21,1,14,21,16,21,1,14,21,16,21,0,16,21,16,21,0,16,21,16,21,1,15,21,16,21,0,16,21,16,21,1,14,21,16,22,4,10,26,16,22,2,13,24,16,22,2,10,26,16,21,3,11,10,16,0,1,15,3,16,22,4,7,9,16,15,1,14,15,16,22,1,14,22,16,15,1,14,15,16,22,1,14,22,16,15,1,14,15,16,22,2,14,22,16,16,1,15,22,16,22,2,13,22,16,22,3,10,22,16,22,1,12,28,16,22,2,14,23,16,22,2,12,22,16,15,1,15,15,16,15,2,12,15,16,15,1,14,15,16,15,1,14,21,16,15,1,14,21,16,15,3,12,15,16,15,1,14,15,16,20,2,13,20,16,15,2,13,15,16,15,1,14,15,16,15,0,16,15,16,15,1,14,15,16,15,0,15,21,16,15,1,14,15,16,21,1,13,27,16,22,7,3,27,16,21,2,13,27,16,15,1,15,5],"a30":[16,0,0,0,0,16,22,5,5,22,16,22,3,10,9,16,21,0,16,21,16,22,1,14,24,16,21,0,16,21,16,21,1,15,22,16,22,6,5,9,16,22,4,10,28,16,22,2,10,28,16,17,1,14,14,16,17,1,14,14,16,4,5,5,10,16,12,2,12,3,16,4,5,5,4,16,22,2,13,24,16,21,1,14,21,16,21,2,8,21,16,21,2,13,21,16,21,2,12,21,16,21,1,14,21,16,21,2,13,21,16,21,2,13,21,16,21,2,12,21,16,21,1,14,21,16,21,2,13,21,16,15,5,5,15,16,14,5,5,20,16,19,1,14,18,16,15,1,14,10,16,19,1,15,18,16,22,2,13,22,16,21,1,15,21,16,21,0,16,21,16,21,1,14,21,16,21,1,15,21,16,21,1,14,21,16,21,1,14,21,16,21,2,13,21,16,21,1,14,21,16,21,1,14,21,16,21,2,12,21,16,21,1,14,21,16,21,1,15,22,16,21,2,13,21,16,21,1,14,21,16,21,1,14,21,16,21,0,16,21,16,21,1,14,21,16,21,0,16,27,16,21,1,15,21,16,21,1,14,21,16,21,1,14,21,16,21,1,14,21,16,21,0,16,21,16,21,0,16,21,16,21,1,15,21,16,21,0,16,21,16,21,1,14,21,16,22,4,10,26,16,22,2,13,24,16,22,2,10,26,16,21,3,11,10,16,0,1,15,3,16,22,4,7,9,16,15,1,14,15,16,22,1,14,22,16,15,1,14,15,16,22,1,14,22,16,15,1,14,15,16,22,2,14,22,16,16,1,15,22,16,22,2,13,22,16,22,3,10,22,16,22,1,12,28,16,22,2,14,23,16,22,2,12,22,16,15,1,15,15,16,15,2,12,15,16,15,1,14,15,16,15,1,14,21,16,15,1,14,21,16,15,3,12,15,16,15,1,14,15,16,20,2,13,20,16,15,2,13,15,16,15,1,14,15,16,15,0,16,15,16,15,1,14,15,16,15,0,15,21,16,15,1,14,15,16,21,1,13,27,16,22,7,3,27,16,21,2,13,27,16,15,1,15,5],"m32":[17,0,0,0,0,17,23,6,5,23,17,23,3,11,9,17,22,0,17,22,17,23,2,14,25,17,22,0,17,22,17,22,1,16,23,17,23,7,5,9,17,24,4,11,30,17,23,2,11,30,17,18,1,15,15,17,19,1,15,16,17,4,5,6,10,17,12,2,13,3,17,4,6,5,4,17,23,2,14,25,17,22,1,15,22,17,22,3,8,22,17,22,2,14,22,17,22,2,13,22,17,22,1,15,22,17,22,2,14,22,17,22,2,14,22,17,22,2,13,22,17,22,2,14,22,17,22,2,14,22,17,16,6,5,16,17,15,5,6,21,17,20,1,15,19,17,16,1,15,11,17,20,1,16,19,17,23,2,13,23,17,22,1,16,22,17,22,0,17,22,17,22,1,15,22,17,22,1,16,22,17,22,1,15,22,17,22,2,14,22,17,22,2,14,22,17,22,1,15,22,17,22,1,15,22,17,22,2,12,22,17,22,1,15,22,17,22,1,16,23,17,22,2,14,22,17,22,1,15,22,17,22,1,15,22,17,22,1,16,22,17,22,2,14,22,17,22,1,16,28,17,22,1,16,22,17,22,1,15,22,17,22,1,15,22,17,22,1,15,22,17,22,0,17,22,17,22,0,17,22,17,22,1,16,22,17,22,0,17,22,17,22,1,15,22,17,23,4,11,27,17,23,2,14,25,17,23,2,11,27,17,22,3,11,11,17,0,1,16,3,17,24,4,7,10,17,16,1,14,16,17,23,2,14,23,17,16,1,15,16,17,23,1,15,23,17,16,1,15,16,17,23,2,15,23,17,17,1,16,23,17,23,2,13,23,17,23,3,11,23,17,23,1,13,30,17,23,2,15,24,17,23,2,13,23,17,16,1,16,16,17,16,2,13,16,17,16,1,15,16,17,16,2,14,22,17,16,1,15,22,17,16,3,13,16,17,16,1,15,16,17,21,2,14,21,17,16,2,14,16,17,16,1,15,16,17,16,0,17,16,17,16,1,15,16,17,16,0,16,22,17,16,1,15,16,17,23,1,14,29,17,23,7,3,29,17,23,3,13,29,17,16,1,16,6],"a32":[17,0,0,0,0,17,23,6,5,23,17,23,3,11,9,17,22,0,17,22,17,23,2,14,25,17,22,0,17,22,17,22,1,16,23,17,23,7,5,9,17,24,4,11,30,17,23,2,11,30,17,18,1,15,15,17,19,1,15,16,17,4,5,6,10,17,12,2,13,3,17,4,6,5,4,17,23,2,14,25,17,22,1,15,22,17,22,3,8,22,17,22,2,14,22,17,22,2,13,22,17,22,1,15,22,17,22,2,14,22,17,22,2,14,22,17,22,2,13,22,17,22,2,14,22,17,22,2,14,22,17,16,6,5,16,17,15,5,6,21,17,20,1,15,19,17,16,1,15,11,17,20,1,16,19,17,23,2,13,23,17,22,1,16,22,17,22,0,17,22,17,22,1,15,22,17,22,1,16,22,17,22,1,15,22,17,22,2,14,22,17,22,2,14,22,17,22,1,15,22,17,22,1,15,22,17,22,2,12,22,17,22,1,15,22,17,22,1,16,23,17,22,2,14,22,17,22,1,15,22,17,22,1,15,22,17,22,1,16,22,17,22,2,14,22,17,22,1,16,28,17,22,1,16,22,17,22,1,15,22,17,22,1,15,22,17,22,1,15,22,17,22,0,17,22,17,22,0,17,22,17,22,1,16,22,17,22,0,17,22,17,22,1,15,22,17,23,4,11,27,17,23,2,14,25,17,23,2,11,27,17,22,3,11,11,17,0,1,16,3,17,24,4,7,10,17,16,1,14,16,17,23,2,14,23,17,16,1,15,16,17,23,1,15,23,17,16,1,15,16,17,23,2,15,23,17,17,1,16,23,17,23,2,13,23,17,23,3,11,23,17,23,1,13,30,17,23,2,15,24,17,23,2,13,23,17,16,1,16,16,17,16,2,13,16,17,16,1,15,16,17,16,2,14,22,17,16,1,15,22,17,16,3,13,16,17,16,1,15,16,17,21,2,14,21,17,16,2,14,16,17,16,1,15,16,17,16,0,17,16,17,16,1,15,16,17,16,0,16,22,17,16,1,15,16,17,23,1,14,29,17,23,7,3,29,17,23,3,13,29,17,16,1,16,6],"m36":[19,0,0,0,0,19,26,7,5,26,19,26,4,12,11,19,25,0,19,25,19,26,2,16,28,19,25,1,18,25,19,25,1,18,26,19,26,8,5,11,19,26,5,12,33,19,26,2,13,33,19,21,1,17,17,19,21,1,17,17,19,4,6,6,11,19,14,2,15,4,19,5,6,6,5,19,26,2,15,28,19,25,1,17,25,19,25,3,9,25,19,25,2,16,25,19,25,2,15,25,19,25,1,17,25,19,25,2,16,25,19,25,2,16,25,19,25,3,14,25,19,25,2,16,25,19,25,2,15,25,19,17,6,6,17,19,17,6,6,24,19,22,1,17,21,19,18,1,17,12,19,22,1,18,21,19,26,2,15,26,19,25,1,18,25,19,25,0,19,25,19,25,1,17,25,19,25,1,18,25,19,25,2,16,25,19,25,2,16,25,19,25,3,14,25,19,25,1,17,25,19,25,2,16,25,19,25,3,13,25,19,25,1,17,25,19,25,1,18,26,19,25,2,16,25,19,25,1,17,25,19,25,1,17,25,19,25,1,18,25,19,25,2,16,25,19,25,1,18,32,19,25,2,17,25,19,25,1,17,25,19,25,1,17,25,19,25,1,17,25,19,25,0,19,25,19,25,0,19,25,19,25,1,18,25,19,25,1,18,25,19,25,1,17,25,19,26,5,12,30,19,26,2,15,28,19,26,3,12,30,19,25,3,13,12,19,0,1,18,4,19,27,5,8,11,19,18,1,16,18,19,26,2,16,26,19,18,2,16,18,19,26,1,17,26,19,18,1,17,18,19,26,2,17,26,19,19,1,18,26,19,26,2,15,26,19,26,3,13,26,19,26,1,14,33,19,26,2,17,27,19,26,3,14,26,19,18,1,18,18,19,18,2,15,18,19,18,1,17,18,19,18,2,16,25,19,18,1,17,25,19,18,3,15,18,19,18,2,16,18,19,23,2,16,23,19,18,2,16,18,19,18,1,17,18,19,18,0,19,18,19,18,1,17,18,19,18,0,18,25,19,18,1,17,18,19,25,2,14,32,19,26,8,3,32,19,25,3,15,32,19,18,1,18,6],"a36":[19,0,0,0,0,19,26,7,5,26,19,26,4,12,11,19,25,0,19,25,19,26,2,16,28,19,25,1,18,25,19,25,1,18,26,19,26,8,5,11,19,26,5,12,33,19,26,2,13,33,19,21,1,17,17,19,21,1,17,17,19,4,6,6,11,19,14,2,15,4,19,5,6,6,5,19,26,2,15,28,19,25,1,17,25,19,25,3,9,25,19,25,2,16,25,19,25,2,15,25,19,25,1,17,25,19,25,2,16,25,19,25,2,16,25,19,25,3,14,25,19,25,2,16,25,19,25,2,15,25,19,17,6,6,17,19,17,6,6,24,19,22,1,17,21,19,18,1,17,12,19,22,1,18,21,19,26,2,15,26,19,25,1,18,25,19,25,0,19,25,19,25,1,17,25,19,25,1,18,25,19,25,2,16,25,19,25,2,16,25,19,25,3,14,25,19,25,1,17,25,19,25,2,16,25,19,25,3,13,25,19,25,1,17,25,19,25,1,18,26,19,25,2,16,25,19,25,1,17,25,19,25,1,17,25,19,25,1,18,25,19,25,2,16,25,19,25,1,18,32,19,25,2,17,25,19,25,1,17,25,19,25,1,17,25,19,25,1,17,25,19,25,0,19,25,19,25,0,19,25,19,25,1,18,25,19,25,1,18,25,19,25,1,17,25,19,26,5,12,30,19,26,2,15,28,19,26,3,12,30,19,25,3,13,12,19,0,1,18,4,19,27,5,8,11,19,18,1,16,18,19,26,2,16,26,19,18,2,16,18,19,26,1,17,26,19,18,1,17,18,19,26,2,17,26,19,19,1,18,26,19,26,2,15,26,19,26,3,13,26,19,26,1,14,33,19,26,2,17,27,19,26,3,14,26,19,18,1,18,18,19,18,2,15,18,19,18,1,17,18,19,18,2,16,25,19,18,1,17,25,19,18,3,15,18,19,18,2,16,18,19,23,2,16,23,19,18,2,16,18,19,18,1,17,18,19,18,0,19,18,19,18,1,17,18,19,18,0,18,25,19,18,1,17,18,19,25,2,14,32,19,26,8,3,32,19,25,3,15,32,19,18,1,18,6],"m40":[22,0,0,0,0,22,29,7,6,30,22,29,4,14,12,22,28,1,20,28,22,29,2,18,32,22,28,1,20,29,22,28,1,20,29,22,29,8,6,12,22,29,5,14,37,22,29,3,13,37,22,23,1,19,19,22,23,1,19,19,22,5,7,7,13,22,15,3,16,3,22,5,7,6,6,22,29,2,17,31,22,28,2,18,29,22,27,3,10,28,22,28,3,16,28,22,27,2,17,28,22,27,2,18,27,22,27,2,18,28,22,28,2,17,29,22,27,3,16,27,22,28,2,18,29,22,28,2,17,29,22,19,7,6,20,22,19,7,7,27,22,25,1,19,23,22,19,1,19,12,22,25,1,20,23,22,29,2,17,30,22,28,1,20,29,22,28,0,21,29,22,27,2,18,27,22,28,1,20,29,22,27,2,18,28,22,27,2,17,27,22,27,3,16,27,22,28,1,19,29,22,27,2,18,28,22,27,3,15,28,22,27,2,18,28,22,28,1,20,29,22,27,2,17,28,22,27,1,19,28,22,27,2,18,28,22,28,1,20,29,22,27,2,18,27,22,28,1,20,35,22,27,2,19,27,22,28,2,18,29,22,27,1,19,28,22,27,2,18,28,22,27,1,20,28,22,27,0,22,28,22,27,1,20,27,22,27,1,20,27,22,27,2,18,28,22,29,5,14,33,22,29,2,17,31,22,29,3,13,33,22,27,4,14,13,22,0,1,20,4,22,29,6,8,11,22,21,2,17,22,22,29,2,18,30,22,21,2,18,22,22,29,1,19,30,22,21,2,18,22,22,29,2,19,29,22,21,1,20,29,22,29,2,17,29,22,29,4,14,29,22,29,2,15,37,22,29,2,19,30,22,29,3,16,29,22,21,1,20,22,22,21,2,17,21,22,21,1,19,22,22,21,2,18,29,22,21,1,19,29,22,21,4,16,22,22,21,2,17,22,22,26,2,18,27,22,20,2,17,21,22,20,1,19,21,22,20,0,21,21,22,20,2,18,20,22,20,0,20,28,22,20,2,18,21,22,28,2,16,35,22,29,9,4,36,22,28,3,17,35,22,21,1,20,8],"a40":[22,0,0,0,0,22,29,7,6,30,22,29,4,14,12,22,28,1,20,28,22,29,2,18,32,22,28,1,20,29,22,28,1,20,29,22,29,8,6,12,22,29,5,14,37,22,29,3,13,37,22,23,1,19,19,22,23,1,19,19,22,5,7,7,13,22,15,3,16,3,22,5,7,6,6,22,29,2,17,31,22,28,2,18,29,22,27,3,10,28,22,28,3,16,28,22,27,2,17,28,22,27,2,18,27,22,27,2,18,28,22,28,2,17,29,22,27,3,16,27,22,28,2,18,29,22,28,2,17,29,22,19,7,6,20,22,19,7,7,27,22,25,1,19,23,22,19,1,19,12,22,25,1,20,23,22,29,2,17,30,22,28,1,20,29,22,28,0,21,29,22,27,2,18,27,22,28,1,20,29,22,27,2,18,28,22,27,2,17,27,22,27,3,16,27,22,28,1,19,29,22,27,2,18,28,22,27,3,15,28,22,27,2,18,28,22,28,1,20,29,22,27,2,17,28,22,27,1,19,28,22,27,2,18,28,22,28,1,20,29,22,27,2,18,27,22,28,1,20,35,22,27,2,19,27,22,28,2,18,29,22,27,1,19,28,22,27,2,18,28,22,27,1,20,28,22,27,0,22,28,22,27,1,20,27,22,27,1,20,27,22,27,2,18,28,22,29,5,14,33,22,29,2,17,31,22,29,3,13,33,22,27,4,14,13,22,0,1,20,4,22,29,6,8,11,22,21,2,17,22,22,29,2,18,30,22,21,2,18,22,22,29,1,19,30,22,21,2,18,22,22,29,2,19,29,22,21,1,20,29,22,29,2,17,29,22,29,4,14,29,22,29,2,15,37,22,29,2,19,30,22,29,3,16,29,22,21,1,20,22,22,21,2,17,21,22,21,1,19,22,22,21,2,18,29,22,21,1,19,29,22,21,4,16,22,22,21,2,17,22,22,26,2,18,27,22,20,2,17,21,22,20,1,19,21,22,20,0,21,21,22,20,2,18,20,22,20,0,20,28,22,20,2,18,21,22,28,2,16,35,22,29,9,4,36,22,28,3,17,35,22,21,1,20,8],"m45":[24,0,0,0,0,24,33,8,7,34,24,33,5,14,14,24,31,1,22,31,24,32,2,20,35,24,31,1,22,32,24,31,1,22,32,24,33,9,7,14,24,33,6,15,42,24,33,3,15,42,24,26,1,21,21,24,26,2,20,21,24,5,7,8,14,24,17,3,18,4,24,5,8,7,6,24,33,3,18,36,24,31,2,20,32,24,31,4,11,32,24,31,3,18,31,24,31,3,18,32,24,31,2,20,31,24,31,3,19,32,24,31,3,18,32,24,31,3,18,31,24,31,2,19,32,24,31,3,18,32,24,21,8,7,22,24,21,7,8,30,24,28,1,21,26,24,22,2,20,14,24,28,1,22,26,24,33,3,18,34,24,31,1,22,32,24,31,0,23,32,24,31,2,20,31,24,31,1,22,32,24,31,2,20,32,24,31,2,19,31,24,31,3,18,31,24,31,1,21,32,24,31,2,20,32,24,31,3,17,32,24,31,2,20,32,24,31,2,21,32,24,31,3,18,32,24,31,1,21,32,24,31,2,20,32,24,31,1,22,32,24,31,2,20,31,24,31,1,22,39,24,31,2,21,31,24,31,2,20,32,24,31,1,21,32,24,31,2,20,32,24,31,1,22,32,24,31,0,24,32,24,31,1,22,31,24,31,1,22,31,24,31,2,20,32,24,33,6,14,38,24,33,3,18,36,24,33,3,15,38,24,31,4,16,15,24,0,1,22,4,24,33,6,10,13,24,23,2,19,24,24,33,2,20,34,24,23,2,20,24,24,33,2,20,34,24,23,2,20,24,24,33,2,21,33,24,23,1,22,32,24,33,3,18,33,24,32,4,15,32,24,32,2,17,41,24,33,3,20,34,24,33,3,18,33,24,23,1,22,24,24,23,3,18,23,24,23,1,21,24,24,23,2,20,32,24,23,2,19,32,24,23,4,18,24,24,23,2,19,24,24,29,3,19,30,24,23,2,19,24,24,23,1,21,24,24,23,0,23,24,24,23,2,20,23,24,23,1,22,32,24,23,2,20,24,24,32,2,18,40,24,32,10,4,40,24,32,4,18,40,24,23,1,22,8],"a45":[24,0,0,0,0,24,33,8,7,34,24,33,5,14,14,24,31,1,22,31,24,32,2,20,35,24,31,1,22,32,24,31,1,22,32,24,33,9,7,14,24,33,6,15,42,24,33,3,15,42,24,26,1,21,21,24,26,2,20,21,24,5,7,8,14,24,17,3,18,4,24,5,8,7,6,24,33,3,18,36,24,31,2,20,32,24,31,4,11,32,24,31,3,18,31,24,31,3,18,32,24,31,2,20,31,24,31,3,19,32,24,31,3,18,32,24,31,3,18,31,24,31,2,19,32,24,31,3,18,32,24,21,8,7,22,24,21,7,8,30,24,28,1,21,26,24,22,2,20,14,24,28,1,22,26,24,33,3,18,34,24,31,1,22,32,24,31,0,23,32,24,31,2,20,31,24,31,1,22,32,24,31,2,20,32,24,31,2,19,31,24,31,3,18,31,24,31,1,21,32,24,31,2,20,32,24,31,3,17,32,24,31,2,20,32,24,31,2,21,32,24,31,3,18,32,24,31,1,21,32,24,31,2,20,32,24,31,1,22,32,24,31,2,20,31,24,31,1,22,39,24,31,2,21,31,24,31,2,20,32,24,31,1,21,32,24,31,2,20,32,24,31,1,22,32,24,31,0,24,32,24,31,1,22,31,24,31,1,22,31,24,31,2,20,32,24,33,6,14,38,24,33,3,18,36,24,33,3,15,38,24,31,4,16,15,24,0,1,22,4,24,33,6,10,13,24,23,2,19,24,24,33,2,20,34,24,23,2,20,24,24,33,2,20,34,24,23,2,20,24,24,33,2,21,33,24,23,1,22,32,24,33,3,18,33,24,32,4,15,32,24,32,2,17,41,24,33,3,20,34,24,33,3,18,33,24,23,1,22,24,24,23,3,18,23,24,23,1,21,24,24,23,2,20,32,24,23,2,19,32,24,23,4,18,24,24,23,2,19,24,24,29,3,19,30,24,23,2,19,24,24,23,1,21,24,24,23,0,23,24,24,23,2,20,23,24,23,1,22,32,24,23,2,20,24,24,32,2,18,40,24,32,10,4,40,24,32,4,18,40,24,23,1,22,8],"m48":[26,0,0,0,0,26,35,9,7,36,26,35,5,16,14,26,33,1,24,33,26,34,3,21,37,26,33,1,24,34,26,33,1,24,34,26,35,10,7,14,26,35,7,16,44,26,35,3,16,45,26,27,2,22,22,26,28,2,22,23,26,6,8,8,15,26,18,3,20,4,26,6,9,7,7,26,35,3,20,38,26,33,2,22,34,26,33,4,12,34,26,33,3,20,33,26,33,3,19,34,26,33,2,22,33,26,33,3,21,34,26,33,3,20,34,26,33,4,19,33,26,33,3,20,34,26,33,3,20,34,26,22,9,7,23,26,22,8,8,31,26,30,1,23,28,26,23,2,22,15,26,30,2,23,28,26,35,3,20,36,26,33,1,24,34,26,33,0,26,34,26,33,2,22,33,26,33,2,23,34,26,33,2,22,34,26,33,3,20,33,26,33,3,20,33,26,33,1,23,34,26,33,2,22,34,26,33,4,17,34,26,33,2,22,34,26,33,2,23,34,26,33,3,20,34,26,33,2,22,34,26,33,2,22,34,26,33,1,24,34,26,33,3,21,33,26,33,1,24,41,26,33,2,23,33,26,33,2,22,34,26,33,1,23,34,26,33,2,22,34,26,33,1,24,34,26,33,0,26,34,26,33,2,23,33,26,33,1,24,33,26,33,2,22,34,26,35,6,16,40,26,35,3,20,38,26,35,4,16,40,26,33,4,17,16,26,0,1,24,5,26,35,7,10,13,26,25,2,21,26,26,35,3,21,36,26,25,2,22,26,26,35,2,22,36,26,25,2,22,26,26,35,3,23,35,26,25,1,24,34,26,35,3,20,35,26,34,5,16,34,26,34,2,18,44,26,35,3,22,36,26,35,4,18,35,26,25,1,24,26,26,25,3,20,25,26,25,2,22,26,26,25,3,21,34,26,25,2,21,34,26,25,5,19,26,26,25,2,21,26,26,31,3,21,32,26,24,3,20,25,26,24,2,22,25,26,24,0,25,25,26,24,2,22,24,26,24,1,23,34,26,24,2,22,25,26,34,2,20,43,26,34,11,4,42,26,34,4,20,43,26,25,2,23,9],"a48":[26,0,0,0,0,26,35,9,7,36,26,35,5,16,14,26,33,1,24,33,26,34,3,21,37,26,33,1,24,34,26,33,1,24,34,26,35,10,7,14,26,35,7,16,44,26,35,3,16,45,26,27,2,22,22,26,28,2,22,23,26,6,8,8,15,26,18,3,20,4,26,6,9,7,7,26,35,3,20,38,26,33,2,22,34,26,33,4,12,34,26,33,3,20,33,26,33,3,19,34,26,33,2,22,33,26,33,3,21,34,26,33,3,20,34,26,33,4,19,33,26,33,3,20,34,26,33,3,20,34,26,22,9,7,23,26,22,8,8,31,26,30,1,23,28,26,23,2,22,15,26,30,2,23,28,26,35,3,20,36,26,33,1,24,34,26,33,0,26,34,26,33,2,22,33,26,33,2,23,34,26,33,2,22,34,26,33,3,20,33,26,33,3,20,33,26,33,1,23,34,26,33,2,22,34,26,33,4,17,34,26,33,2,22,34,26,33,2,23,34,26,33,3,20,34,26,33,2,22,34,26,33,2,22,34,26,33,1,24,34,26,33,3,21,33,26,33,1,24,41,26,33,2,23,33,26,33,2,22,34,26,33,1,23,34,26,33,2,22,34,26,33,1,24,34,26,33,0,26,34,26,33,2,23,33,26,33,1,24,33,26,33,2,22,34,26,35,6,16,40,26,35,3,20,38,26,35,4,16,40,26,33,4,17,16,26,0,1,24,5,26,35,7,10,13,26,25,2,21,26,26,35,3,21,36,26,25,2,22,26,26,35,2,22,36,26,25,2,22,26,26,35,3,23,35,26,25,1,24,34,26,35,3,20,35,26,34,5,16,34,26,34,2,18,44,26,35,3,22,36,26,35,4,18,35,26,25,1,24,26,26,25,3,20,25,26,25,2,22,26,26,25,3,21,34,26,25,2,21,34,26,25,5,19,26,26,25,2,21,26,26,31,3,21,32,26,24,3,20,25,26,24,2,22,25,26,24,0,25,25,26,24,2,22,24,26,24,1,23,34,26,24,2,22,25,26,34,2,20,43,26,34,11,4,42,26,34,4,20,43,26,25,2,23,9],"m50":[27,0,0,0,0,27,37,9,8,38,27,36,6,16,14,27,34,1,25,34,27,36,3,22,39,27,35,1,26,36,27,34,1,25,35,27,36,11,7,14,27,36,7,16,46,27,36,3,17,46,27,28,2,23,23,27,29,2,23,24,27,6,8,9,16,27,19,3,21,4,27,6,9,8,7,27,36,3,21,39,27,34,2,23,35,27,34,4,13,35,27,34,3,21,34,27,34,3,20,35,27,34,2,23,34,27,34,3,22,35,27,35,3,21,36,27,34,4,20,34,27,35,3,21,36,27,34,3,21,35,27,23,9,8,24,27,23,8,9,33,27,31,1,24,29,27,24,2,23,15,27,31,2,24,29,27,36,3,21,37,27,35,1,25,36,27,35,0,27,36,27,34,2,23,34,27,34,2,24,35,27,34,2,23,35,27,34,3,21,34,27,34,4,20,34,27,34,1,24,35,27,34,2,23,35,27,34,4,18,35,27,34,2,23,35,27,34,2,24,35,27,34,3,21,35,27,34,2,23,35,27,34,2,23,35,27,34,1,25,35,27,34,3,22,34,27,34,1,25,43,27,34,3,23,34,27,34,2,23,35,27,34,1,24,35,27,34,2,23,35,27,34,1,25,35,27,34,0,27,35,27,34,2,24,34,27,34,1,25,34,27,34,2,23,35,27,37,7,16,42,27,36,3,21,39,27,37,4,16,42,27,34,5,17,16,27,-1,1,25,4,27,37,7,11,14,27,26,2,22,27,27,36,3,22,37,27,26,3,22,27,27,36,2,23,37,27,26,2,23,27,27,37,3,24,37,27,26,2,24,36,27,36,3,21,36,27,36,5,17,36,27,36,2,19,46,27,36,3,23,37,27,36,4,19,36,27,26,2,24,27,27,26,3,21,26,27,26,2,23,27,27,26,3,22,36,27,26,2,22,36,27,26,5,20,27,27,26,3,21,27,27,32,3,21,33,27,25,3,21,26,27,25,2,23,26,27,25,0,26,26,27,25,2,23,25,27,25,1,24,35,27,25,2,23,26,27,35,2,21,44,27,36,11,5,45,27,35,4,21,44,27,26,2,24,9],"a50":[27,0,0,0,0,27,37,9,8,38,27,36,6,16,14,27,34,1,25,34,27,36,3,22,39,27,35,1,26,36,27,34,1,25,35,27,36,11,7,14,27,36,7,16,46,27,36,3,17,46,27,28,2,23,23,27,29,2,23,24,27,6,8,9,16,27,19,3,21,4,27,6,9,8,7,27,36,3,21,39,27,34,2,23,35,27,34,4,13,35,27,34,3,21,34,27,34,3,20,35,27,34,2,23,34,27,34,3,22,35,27,35,3,21,36,27,34,4,20,34,27,35,3,21,36,27,34,3,21,35,27,23,9,8,24,27,23,8,9,33,27,31,1,24,29,27,24,2,23,15,27,31,2,24,29,27,36,3,21,37,27,35,1,25,36,27,35,0,27,36,27,34,2,23,34,27,34,2,24,35,27,34,2,23,35,27,34,3,21,34,27,34,4,20,34,27,34,1,24,35,27,34,2,23,35,27,34,4,18,35,27,34,2,23,35,27,34,2,24,35,27,34,3,21,35,27,34,2,23,35,27,34,2,23,35,27,34,1,25,35,27,34,3,22,34,27,34,1,25,43,27,34,3,23,34,27,34,2,23,35,27,34,1,24,35,27,34,2,23,35,27,34,1,25,35,27,34,0,27,35,27,34,2,24,34,27,34,1,25,34,27,34,2,23,35,27,37,7,16,42,27,36,3,21,39,27,37,4,16,42,27,34,5,17,16,27,-1,1,25,4,27,37,7,11,14,27,26,2,22,27,27,36,3,22,37,27,26,3,22,27,27,36,2,23,37,27,26,2,23,27,27,37,3,24,37,27,26,2,24,36,27,36,3,21,36,27,36,5,17,36,27,36,2,19,46,27,36,3,23,37,27,36,4,19,36,27,26,2,24,27,27,26,3,21,26,27,26,2,23,27,27,26,3,22,36,27,26,2,22,36,27,26,5,20,27,27,26,3,21,27,27,32,3,21,33,27,25,3,21,26,27,25,2,23,26,27,25,0,26,26,27,25,2,23,25,27,25,1,24,35,27,25,2,23,26,27,35,2,21,44,27,36,11,5,45,27,35,4,21,44,27,26,2,24,9],"m54":[29,0,0,0,0,29,39,10,8,40,29,39,6,18,16,29,37,1,27,37,29,38,3,24,41,29,37,1,28,38,29,37,2,26,38,29,39,12,7,16,29,39,8,17,50,29,39,4,18,50,29,31,2,25,25,29,31,2,25,25,29,6,9,9,16,29,21,4,22,5,29,6,10,8,7,29,39,3,23,42,29,37,2,25,38,29,37,5,13,38,29,37,4,22,37,29,37,4,21,38,29,37,2,25,37,29,37,3,23,38,29,37,3,23,38,29,37,4,22,37,29,37,3,23,38,29,37,3,23,38,29,25,10,8,26,29,25,9,9,35,29,33,1,26,31,29,26,2,25,17,29,33,2,26,31,29,39,3,23,40,29,37,1,27,38,29,37,0,29,38,29,37,2,25,37,29,37,2,26,38,29,37,3,24,38,29,37,3,23,37,29,37,4,22,37,29,37,2,25,38,29,37,3,24,38,29,37,4,20,38,29,37,2,25,38,29,37,2,26,38,29,37,3,23,38,29,37,2,25,38,29,37,2,25,38,29,37,1,27,38,29,37,3,24,37,29,37,1,27,46,29,37,3,25,37,29,37,2,25,38,29,37,1,26,38,29,37,2,25,38,29,37,1,27,38,29,37,0,29,38,29,37,2,26,37,29,37,1,27,37,29,37,2,25,38,29,40,7,18,46,29,39,3,23,42,29,40,4,18,46,29,37,5,19,18,29,-1,1,27,4,29,40,8,11,16,29,28,2,24,29,29,39,3,24,40,29,28,3,24,29,29,39,2,25,40,29,28,2,24,29,29,40,3,26,40,29,28,2,26,39,29,39,4,22,39,29,39,5,19,39,29,39,2,21,50,29,39,3,25,40,29,39,4,21,39,29,28,2,26,29,29,28,3,23,28,29,28,2,25,29,29,28,3,24,38,29,28,2,24,38,29,28,5,22,29,29,28,3,23,29,29,35,4,22,36,29,27,3,23,28,29,27,2,25,28,29,27,0,28,28,29,27,2,25,27,29,27,1,26,38,29,27,2,25,28,29,38,3,21,48,29,39,12,5,48,29,38,5,22,48,29,28,2,26,10],"a54":[29,0,0,0,0,29,39,10,8,40,29,39,6,18,16,29,37,1,27,37,29,38,3,24,41,29,37,1,28,38,29,37,2,26,38,29,39,12,7,16,29,39,8,17,50,29,39,4,18,50,29,31,2,25,25,29,31,2,25,25,29,6,9,9,16,29,21,4,22,5,29,6,10,8,7,29,39,3,23,42,29,37,2,25,38,29,37,5,13,38,29,37,4,22,37,29,37,4,21,38,29,37,2,25,37,29,37,3,23,38,29,37,3,23,38,29,37,4,22,37,29,37,3,23,38,29,37,3,23,38,29,25,10,8,26,29,25,9,9,35,29,33,1,26,31,29,26,2,25,17,29,33,2,26,31,29,39,3,23,40,29,37,1,27,38,29,37,0,29,38,29,37,2,25,37,29,37,2,26,38,29,37,3,24,38,29,37,3,23,37,29,37,4,22,37,29,37,2,25,38,29,37,3,24,38,29,37,4,20,38,29,37,2,25,38,29,37,2,26,38,29,37,3,23,38,29,37,2,25,38,29,37,2,25,38,29,37,1,27,38,29,37,3,24,37,29,37,1,27,46,29,37,3,25,37,29,37,2,25,38,29,37,1,26,38,29,37,2,25,38,29,37,1,27,38,29,37,0,29,38,29,37,2,26,37,29,37,1,27,37,29,37,2,25,38,29,40,7,18,46,29,39,3,23,42,29,40,4,18,46,29,37,5,19,18,29,-1,1,27,4,29,40,8,11,16,29,28,2,24,29,29,39,3,24,40,29,28,3,24,29,29,39,2,25,40,29,28,2,24,29,29,40,3,26,40,29,28,2,26,39,29,39,4,22,39,29,39,5,19,39,29,39,2,21,50,29,39,3,25,40,29,39,4,21,39,29,28,2,26,29,29,28,3,23,28,29,28,2,25,29,29,28,3,24,38,29,28,2,24,38,29,28,5,22,29,29,28,3,23,29,29,35,4,22,36,29,27,3,23,28,29,27,2,25,28,29,27,0,28,28,29,27,2,25,27,29,27,1,26,38,29,27,2,25,28,29,38,3,21,48,29,39,12,5,48,29,38,5,22,48,29,28,2,26,10],"m56":[30,0,0,0,0,30,41,10,9,42,30,41,6,18,17,30,38,1,28,38,30,40,3,25,43,30,39,1,29,40,30,39,2,28,40,30,41,12,8,17,30,41,8,18,52,30,41,4,19,52,30,32,2,26,26,30,32,2,26,26,30,7,9,10,18,30,21,4,22,5,30,7,10,8,8,30,41,3,24,44,30,38,3,25,39,30,38,5,14,39,30,38,4,23,38,30,38,4,22,39,30,38,2,26,38,30,38,3,24,39,30,39,4,23,40,30,38,4,23,38,30,39,3,24,40,30,39,4,23,40,30,26,10,8,27,30,26,9,10,37,30,35,2,26,33,30,27,2,26,17,30,35,2,27,33,30,41,3,24,42,30,39,1,28,40,30,39,0,30,40,30,38,2,26,38,30,38,2,27,39,30,38,3,25,39,30,38,3,24,38,30,38,4,23,38,30,39,2,26,40,30,38,3,25,39,30,38,4,21,39,30,38,2,26,39,30,38,2,27,39,30,38,3,24,39,30,38,2,26,39,30,38,3,25,39,30,39,1,28,40,30,38,3,25,38,30,39,1,28,49,30,38,3,26,38,30,38,3,25,39,30,38,1,27,39,30,38,3,25,39,30,38,1,28,39,30,38,1,29,39,30,38,2,27,38,30,38,1,28,38,30,38,3,25,39,30,41,7,19,47,30,41,3,24,44,30,41,4,19,47,30,38,5,20,18,30,-1,2,27,4,30,41,8,12,16,30,29,2,25,30,30,41,3,25,42,30,29,3,25,30,30,41,2,26,42,30,29,2,25,30,30,41,3,27,41,30,29,2,27,40,30,41,4,23,41,30,40,6,18,40,30,40,2,22,51,30,41,3,26,42,30,41,4,22,41,30,29,2,27,30,30,29,4,23,29,30,29,2,26,30,30,29,3,25,40,30,29,2,25,40,30,29,5,23,30,30,29,3,24,30,30,36,4,23,37,30,28,3,24,29,30,28,2,26,29,30,28,0,30,29,30,28,2,26,28,30,28,1,27,39,30,28,3,25,29,30,39,3,22,49,30,40,13,5,50,30,39,5,23,49,30,29,2,27,10],"a56":[30,0,0,0,0,30,41,10,9,42,30,41,6,18,17,30,38,1,28,38,30,40,3,25,43,30,39,1,29,40,30,39,2,28,40,30,41,12,8,17,30,41,8,18,52,30,41,4,19,52,30,32,2,26,26,30,32,2,26,26,30,7,9,10,18,30,21,4,22,5,30,7,10,8,8,30,41,3,24,44,30,38,3,25,39,30,38,5,14,39,30,38,4,23,38,30,38,4,22,39,30,38,2,26,38,30,38,3,24,39,30,39,4,23,40,30,38,4,23,38,30,39,3,24,40,30,39,4,23,40,30,26,10,8,27,30,26,9,10,37,30,35,2,26,33,30,27,2,26,17,30,35,2,27,33,30,41,3,24,42,30,39,1,28,40,30,39,0,30,40,30,38,2,26,38,30,38,2,27,39,30,38,3,25,39,30,38,3,24,38,30,38,4,23,38,30,39,2,26,40,30,38,3,25,39,30,38,4,21,39,30,38,2,26,39,30,38,2,27,39,30,38,3,24,39,30,38,2,26,39,30,38,3,25,39,30,39,1,28,40,30,38,3,25,38,30,39,1,28,49,30,38,3,26,38,30,38,3,25,39,30,38,1,27,39,30,38,3,25,39,30,38,1,28,39,30,38,1,29,39,30,38,2,27,38,30,38,1,28,38,30,38,3,25,39,30,41,7,19,47,30,41,3,24,44,30,41,4,19,47,30,38,5,20,18,30,-1,2,27,4,30,41,8,12,16,30,29,2,25,30,30,41,3,25,42,30,29,3,25,30,30,41,2,26,42,30,29,2,25,30,30,41,3,27,41,30,29,2,27,40,30,41,4,23,41,30,40,6,18,40,30,40,2,22,51,30,41,3,26,42,30,41,4,22,41,30,29,2,27,30,30,29,4,23,29,30,29,2,26,30,30,29,3,25,40,30,29,2,25,40,30,29,5,23,30,30,29,3,24,30,30,36,4,23,37,30,28,3,24,29,30,28,2,26,29,30,28,0,30,29,30,28,2,26,28,30,28,1,27,39,30,28,3,25,29,30,39,3,22,49,30,40,13,5,50,30,39,5,23,49,30,29,2,27,10],"m60":[32,0,0,0,0,32,44,11,9,45,32,44,7,19,18,32,41,1,30,41,32,43,3,27,47,32,42,1,31,43,32,41,2,30,42,32,44,13,8,18,32,44,8,20,56,32,44,4,20,56,32,34,2,28,28,32,34,2,28,28,32,7,10,10,18,32,23,4,24,5,32,7,11,9,8,32,44,4,25,47,32,41,3,27,42,32,41,5,15,42,32,41,4,25,41,32,41,4,24,42,32,41,3,27,41,32,41,4,25,42,32,41,4,25,42,32,41,5,23,41,32,41,3,26,42,32,41,4,25,42,32,28,11,9,29,32,28,10,10,39,32,37,2,28,34,32,29,2,28,18,32,37,2,29,34,32,44,4,25,45,32,41,2,29,42,32,42,0,32,43,32,41,3,27,41,32,41,2,29,42,32,41,3,27,42,32,41,3,26,41,32,41,4,25,41,32,41,2,28,42,32,41,3,27,42,32,41,5,22,42,32,41,3,27,42,32,41,2,29,42,32,41,4,25,42,32,41,2,28,42,32,41,3,27,42,32,41,1,30,42,32,41,3,27,41,32,41,1,30,51,32,41,3,28,41,32,41,3,27,42,32,41,2,28,42,32,41,3,27,42,32,41,1,30,42,32,41,1,31,42,32,41,2,29,41,32,41,1,30,41,32,41,3,27,42,32,44,8,20,50,32,44,4,25,47,32,44,5,19,50,32,41,6,21,19,32,-1,2,29,5,32,44,9,12,17,32,31,3,26,32,32,44,3,27,45,32,31,3,27,32,32,44,2,28,45,32,31,3,26,32,32,44,4,28,44,32,31,2,29,43,32,44,4,25,44,32,43,6,20,43,32,43,3,22,55,32,44,4,27,45,32,44,5,23,44,32,31,2,29,32,32,31,4,24,31,32,31,2,28,32,32,31,3,27,42,32,31,2,27,42,32,31,6,24,32,32,31,3,26,32,32,39,4,25,40,32,30,4,25,31,32,30,2,28,31,32,30,0,32,31,32,30,3,27,30,32,30,1,29,42,32,30,3,27,31,32,42,3,24,53,32,43,14,5,53,32,42,5,24,53,32,31,2,29,11],"a60":[32,0,0,0,0,32,44,11,9,45,32,44,7,19,18,32,41,1,30,41,32,43,3,27,47,32,42,1,31,43,32,41,2,30,42,32,44,13,8,18,32,44,8,20,56,32,44,4,20,56,32,34,2,28,28,32,34,2,28,28,32,7,10,10,18,32,23,4,24,5,32,7,11,9,8,32,44,4,25,47,32,41,3,27,42,32,41,5,15,42,32,41,4,25,41,32,41,4,24,42,32,41,3,27,41,32,41,4,25,42,32,41,4,25,42,32,41,5,23,41,32,41,3,26,42,32,41,4,25,42,32,28,11,9,29,32,28,10,10,39,32,37,2,28,34,32,29,2,28,18,32,37,2,29,34,32,44,4,25,45,32,41,2,29,42,32,42,0,32,43,32,41,3,27,41,32,41,2,29,42,32,41,3,27,42,32,41,3,26,41,32,41,4,25,41,32,41,2,28,42,32,41,3,27,42,32,41,5,22,42,32,41,3,27,42,32,41,2,29,42,32,41,4,25,42,32,41,2,28,42,32,41,3,27,42,32,41,1,30,42,32,41,3,27,41,32,41,1,30,51,32,41,3,28,41,32,41,3,27,42,32,41,2,28,42,32,41,3,27,42,32,41,1,30,42,32,41,1,31,42,32,41,2,29,41,32,41,1,30,41,32,41,3,27,42,32,44,8,20,50,32,44,4,25,47,32,44,5,19,50,32,41,6,21,19,32,-1,2,29,5,32,44,9,12,17,32,31,3,26,32,32,44,3,27,45,32,31,3,27,32,32,44,2,28,45,32,31,3,26,32,32,44,4,28,44,32,31,2,29,43,32,44,4,25,44,32,43,6,20,43,32,43,3,22,55,32,44,4,27,45,32,44,5,23,44,32,31,2,29,32,32,31,4,24,31,32,31,2,28,32,32,31,3,27,42,32,31,2,27,42,32,31,6,24,32,32,31,3,26,32,32,39,4,25,40,32,30,4,25,31,32,30,2,28,31,32,30,0,32,31,32,30,3,27,30,32,30,1,29,42,32,30,3,27,31,32,42,3,24,53,32,43,14,5,53,32,42,5,24,53,32,31,2,29,11],"m63":[34,0,0,0,0,34,46,12,8,47,34,46,7,20,19,34,43,1,31,43,34,45,3,28,49,34,44,1,32,45,34,43,2,30,44,34,46,13,9,19,34,46,9,20,58,34,46,4,21,58,34,36,2,29,29,34,36,2,29,29,34,7,10,11,19,34,24,4,25,5,34,7,11,9,8,34,46,4,26,49,34,43,3,27,44,34,43,5,15,44,34,43,4,26,43,34,43,4,25,44,34,43,3,28,43,34,43,4,26,44,34,43,4,26,44,34,43,5,24,43,34,43,4,26,44,34,43,4,26,44,34,29,11,9,30,34,29,10,11,41,34,39,2,29,36,34,30,2,29,19,34,39,2,30,36,34,46,4,25,47,34,43,2,29,44,34,44,0,33,45,34,43,3,28,43,34,43,2,30,44,34,43,3,28,44,34,43,3,27,43,34,43,5,25,43,34,43,2,29,44,34,43,3,27,44,34,43,5,22,44,34,43,3,28,44,34,43,2,30,44,34,43,4,26,44,34,43,2,29,44,34,43,3,28,44,34,43,1,31,44,34,43,3,28,43,34,43,1,31,54,34,43,3,28,43,34,43,3,28,44,34,43,2,29,44,34,43,3,28,44,34,43,1,31,44,34,43,1,32,44,34,43,2,30,43,34,43,1,31,43,34,43,3,28,44,34,46,8,20,53,34,46,4,26,49,34,46,5,20,53,34,43,6,21,20,34,-1,2,30,5,34,46,9,13,18,34,32,3,26,33,34,46,3,28,47,34,32,3,28,33,34,46,2,28,47,34,32,3,27,33,34,46,4,29,46,34,33,2,30,45,34,46,4,25,46,34,45,6,21,45,34,45,3,23,57,34,46,4,28,47,34,46,5,24,46,34,32,2,30,33,34,32,4,25,32,34,32,2,29,33,34,32,3,28,44,34,32,2,28,44,34,32,6,25,33,34,32,3,27,33,34,41,4,26,42,34,32,4,26,33,34,32,2,29,33,34,32,1,31,33,34,32,3,28,32,34,32,1,30,44,34,32,3,27,33,34,44,3,25,56,34,45,14,5,56,34,44,6,24,56,34,32,2,30,11],"a63":[34,0,0,0,0,34,46,12,8,47,34,46,7,20,19,34,43,1,31,43,34,45,3,28,49,34,44,1,32,45,34,43,2,30,44,34,46,13,9,19,34,46,9,20,58,34,46,4,21,58,34,36,2,29,29,34,36,2,29,29,34,7,10,11,19,34,24,4,25,5,34,7,11,9,8,34,46,4,26,49,34,43,3,27,44,34,43,5,15,44,34,43,4,26,43,34,43,4,25,44,34,43,3,28,43,34,43,4,26,44,34,43,4,26,44,34,43,5,24,43,34,43,4,26,44,34,43,4,26,44,34,29,11,9,30,34,29,10,11,41,34,39,2,29,36,34,30,2,29,19,34,39,2,30,36,34,46,4,25,47,34,43,2,29,44,34,44,0,33,45,34,43,3,28,43,34,43,2,30,44,34,43,3,28,44,34,43,3,27,43,34,43,5,25,43,34,43,2,29,44,34,43,3,27,44,34,43,5,22,44,34,43,3,28,44,34,43,2,30,44,34,43,4,26,44,34,43,2,29,44,34,43,3,28,44,34,43,1,31,44,34,43,3,28,43,34,43,1,31,54,34,43,3,28,43,34,43,3,28,44,34,43,2,29,44,34,43,3,28,44,34,43,1,31,44,34,43,1,32,44,34,43,2,30,43,34,43,1,31,43,34,43,3,28,44,34,46,8,20,53,34,46,4,26,49,34,46,5,20,53,34,43,6,21,20,34,-1,2,30,5,34,46,9,13,18,34,32,3,26,33,34,46,3,28,47,34,32,3,28,33,34,46,2,28,47,34,32,3,27,33,34,46,4,29,46,34,33,2,30,45,34,46,4,25,46,34,45,6,21,45,34,45,3,23,57,34,46,4,28,47,34,46,5,24,46,34,32,2,30,33,34,32,4,25,32,34,32,2,29,33,34,32,3,28,44,34,32,2,28,44,34,32,6,25,33,34,32,3,27,33,34,41,4,26,42,34,32,4,26,33,34,32,2,29,33,34,32,1,31,33,34,32,3,28,32,34,32,1,30,44,34,32,3,27,33,34,44,3,25,56,34,45,14,5,56,34,44,6,24,56,34,32,2,30,11],"m64":[35,0,0,0,0,35,47,12,9,48,35,47,7,21,19,35,44,1,32,44,35,45,4,28,49,35,44,1,33,45,35,44,2,32,45,35,47,14,9,19,35,47,9,21,59,35,46,4,22,59,35,36,2,30,29,35,37,3,29,30,35,8,11,11,20,35,24,4,26,5,35,8,12,9,9,35,46,4,27,49,35,44,3,29,45,35,44,6,15,45,35,44,5,26,44,35,44,4,26,45,35,44,3,29,44,35,44,4,27,45,35,44,4,27,45,35,44,5,25,44,35,44,4,27,45,35,44,4,27,45,35,30,12,9,31,35,30,11,11,42,35,40,2,30,37,35,31,3,29,20,35,40,2,31,37,35,46,4,26,47,35,44,2,31,45,35,44,0,34,45,35,44,3,29,44,35,44,2,31,45,35,44,3,29,45,35,44,4,27,44,35,44,5,26,44,35,44,2,30,45,35,44,3,28,45,35,44,5,23,45,35,44,3,29,45,35,44,2,31,45,35,44,4,27,45,35,44,2,30,45,35,44,3,29,45,35,44,2,31,45,35,44,4,28,44,35,44,2,31,55,35,44,3,30,44,35,44,3,29,45,35,44,2,30,45,35,44,3,29,45,35,44,1,33,45,35,44,1,33,45,35,44,2,31,44,35,44,1,32,44,35,44,3,29,45,35,47,9,20,54,35,46,4,27,49,35,47,5,21,54,35,44,6,22,21,35,-1,2,31,5,35,47,9,13,18,35,33,3,27,34,35,46,4,28,47,35,33,3,29,34,35,46,3,28,47,35,33,3,28,34,35,47,4,30,47,35,33,2,31,45,35,46,4,26,46,35,46,6,22,46,35,46,3,24,59,35,46,4,30,47,35,46,5,25,46,35,33,2,31,34,35,33,4,26,33,35,33,2,30,34,35,33,4,28,45,35,33,2,29,45,35,33,6,26,34,35,33,3,28,34,35,41,4,27,42,35,32,4,27,33,35,32,2,30,33,35,32,1,33,33,35,32,3,29,32,35,32,1,31,45,35,32,3,29,33,35,45,3,26,57,35,46,14,6,57,35,45,6,25,57,35,33,2,31,11],"a64":[35,0,0,0,0,35,47,12,9,48,35,47,7,21,19,35,44,1,32,44,35,45,4,28,49,35,44,1,33,45,35,44,2,32,45,35,47,14,9,19,35,47,9,21,59,35,46,4,22,59,35,36,2,30,29,35,37,3,29,30,35,8,11,11,20,35,24,4,26,5,35,8,12,9,9,35,46,4,27,49,35,44,3,29,45,35,44,6,15,45,35,44,5,26,44,35,44,4,26,45,35,44,3,29,44,35,44,4,27,45,35,44,4,27,45,35,44,5,25,44,35,44,4,27,45,35,44,4,27,45,35,30,12,9,31,35,30,11,11,42,35,40,2,30,37,35,31,3,29,20,35,40,2,31,37,35,46,4,26,47,35,44,2,31,45,35,44,0,34,45,35,44,3,29,44,35,44,2,31,45,35,44,3,29,45,35,44,4,27,44,35,44,5,26,44,35,44,2,30,45,35,44,3,28,45,35,44,5,23,45,35,44,3,29,45,35,44,2,31,45,35,44,4,27,45,35,44,2,30,45,35,44,3,29,45,35,44,2,31,45,35,44,4,28,44,35,44,2,31,55,35,44,3,30,44,35,44,3,29,45,35,44,2,30,45,35,44,3,29,45,35,44,1,33,45,35,44,1,33,45,35,44,2,31,44,35,44,1,32,44,35,44,3,29,45,35,47,9,20,54,35,46,4,27,49,35,47,5,21,54,35,44,6,22,21,35,-1,2,31,5,35,47,9,13,18,35,33,3,27,34,35,46,4,28,47,35,33,3,29,34,35,46,3,28,47,35,33,3,28,34,35,47,4,30,47,35,33,2,31,45,35,46,4,26,46,35,46,6,22,46,35,46,3,24,59,35,46,4,30,47,35,46,5,25,46,35,33,2,31,34,35,33,4,26,33,35,33,2,30,34,35,33,4,28,45,35,33,2,29,45,35,33,6,26,34,35,33,3,28,34,35,41,4,27,42,35,32,4,27,33,35,32,2,30,33,35,32,1,33,33,35,32,3,29,32,35,32,1,31,45,35,32,3,29,33,35,45,3,26,57,35,46,14,6,57,35,45,6,25,57,35,33,2,31,11],"m70":[38,0,0,0,0,38,51,13,10,52,38,51,8,22,20,38,48,1,35,48,38,50,4,31,54,38,48,2,35,49,38,48,2,35,49,38,51,15,10,20,38,51,10,23,65,38,51,5,23,65,38,40,3,32,32,38,40,3,32,33,38,8,12,12,21,38,27,5,28,6,38,8,13,10,9,38,51,4,29,55,38,48,3,31,49,38,48,6,17,49,38,48,5,29,48,38,48,5,28,49,38,48,3,32,48,38,48,4,30,49,38,48,5,29,49,38,48,5,28,48,38,48,4,30,49,38,48,5,29,49,38,32,13,10,33,38,32,12,12,45,38,43,2,33,40,38,34,3,32,22,38,43,2,34,40,38,51,4,29,52,38,48,2,34,49,38,48,1,36,49,38,48,3,32,48,38,48,3,33,49,38,48,4,31,49,38,48,4,30,48,38,48,5,28,48,38,48,2,33,49,38,48,4,30,49,38,48,6,25,49,38,48,3,32,49,38,48,3,33,49,38,48,4,30,49,38,48,3,32,49,38,48,3,32,49,38,48,2,34,49,38,48,4,31,48,38,48,2,34,60,38,48,4,32,48,38,48,3,32,49,38,48,2,33,49,38,48,3,32,49,38,48,1,36,49,38,48,1,36,49,38,48,3,33,48,38,48,2,34,48,38,48,3,32,49,38,51,9,23,58,38,51,4,29,55,38,51,6,22,58,38,48,7,24,23,38,-1,2,34,6,38,51,10,14,19,38,36,3,30,37,38,51,4,31,52,38,36,4,31,37,38,51,3,31,52,38,36,3,31,37,38,51,4,33,51,38,36,2,34,50,38,51,5,28,51,38,50,7,23,50,38,50,3,27,64,38,51,4,33,52,38,51,5,27,51,38,36,2,34,37,38,36,5,28,36,38,36,3,32,37,38,36,4,31,49,38,36,3,31,49,38,36,7,28,37,38,36,4,30,37,38,45,5,29,46,38,35,4,30,36,38,35,2,33,36,38,35,1,36,36,38,35,3,32,35,38,35,1,34,49,38,35,3,31,36,38,49,3,28,62,38,50,16,6,62,38,49,6,28,62,38,36,3,33,12],"a70":[38,0,0,0,0,38,51,13,10,52,38,51,8,22,20,38,48,1,35,48,38,50,4,31,54,38,48,2,35,49,38,48,2,35,49,38,51,15,10,20,38,51,10,23,65,38,51,5,23,65,38,40,3,32,32,38,40,3,32,33,38,8,12,12,21,38,27,5,28,6,38,8,13,10,9,38,51,4,29,55,38,48,3,31,49,38,48,6,17,49,38,48,5,29,48,38,48,5,28,49,38,48,3,32,48,38,48,4,30,49,38,48,5,29,49,38,48,5,28,48,38,48,4,30,49,38,48,5,29,49,38,32,13,10,33,38,32,12,12,45,38,43,2,33,40,38,34,3,32,22,38,43,2,34,40,38,51,4,29,52,38,48,2,34,49,38,48,1,36,49,38,48,3,32,48,38,48,3,33,49,38,48,4,31,49,38,48,4,30,48,38,48,5,28,48,38,48,2,33,49,38,48,4,30,49,38,48,6,25,49,38,48,3,32,49,38,48,3,33,49,38,48,4,30,49,38,48,3,32,49,38,48,3,32,49,38,48,2,34,49,38,48,4,31,48,38,48,2,34,60,38,48,4,32,48,38,48,3,32,49,38,48,2,33,49,38,48,3,32,49,38,48,1,36,49,38,48,1,36,49,38,48,3,33,48,38,48,2,34,48,38,48,3,32,49,38,51,9,23,58,38,51,4,29,55,38,51,6,22,58,38,48,7,24,23,38,-1,2,34,6,38,51,10,14,19,38,36,3,30,37,38,51,4,31,52,38,36,4,31,37,38,51,3,31,52,38,36,3,31,37,38,51,4,33,51,38,36,2,34,50,38,51,5,28,51,38,50,7,23,50,38,50,3,27,64,38,51,4,33,52,38,51,5,27,51,38,36,2,34,37,38,36,5,28,36,38,36,3,32,37,38,36,4,31,49,38,36,3,31,49,38,36,7,28,37,38,36,4,30,37,38,45,5,29,46,38,35,4,30,36,38,35,2,33,36,38,35,1,36,36,38,35,3,32,35,38,35,1,34,49,38,35,3,31,36,38,49,3,28,62,38,50,16,6,62,38,49,6,28,62,38,36,3,33,12],"m72":[39,0,0,0,0,39,52,14,10,53,39,52,8,23,21,39,49,1,36,49,39,51,4,32,55,39,50,2,36,51,39,49,2,36,51,39,52,16,9,21,39,52,10,24,66,39,52,5,24,66,39,41,3,33,33,39,41,3,33,33,39,8,12,12,22,39,27,5,29,6,39,9,13,11,10,39,52,4,30,56,39,49,3,32,50,39,49,6,18,50,39,49,5,30,49,39,49,5,28,50,39,49,3,33,49,39,49,4,31,50,39,49,5,30,50,39,49,6,28,49,39,49,4,31,50,39,49,5,29,50,39,33,13,11,34,39,33,12,12,47,39,44,2,34,41,39,35,3,33,22,39,44,3,34,41,39,52,4,30,54,39,50,2,35,51,39,50,1,37,51,39,49,3,33,49,39,49,3,34,50,39,49,4,32,50,39,49,4,31,49,39,49,5,29,49,39,49,2,34,50,39,49,4,31,50,39,49,6,26,50,39,49,3,33,50,39,49,3,34,50,39,49,5,30,50,39,49,3,33,50,39,49,3,33,50,39,49,2,35,50,39,49,4,32,49,39,49,2,35,61,39,49,4,33,49,39,49,3,33,50,39,49,2,34,50,39,49,3,33,50,39,49,1,37,50,39,49,1,37,50,39,49,3,34,49,39,49,2,35,49,39,49,3,33,50,39,53,10,23,60,39,52,4,30,56,39,53,6,23,60,39,49,7,25,23,39,-1,2,35,6,39,53,10,15,20,39,37,3,31,38,39,52,4,32,53,39,37,4,32,38,39,52,3,32,53,39,37,3,32,38,39,53,4,34,53,39,37,2,35,51,39,52,5,29,52,39,51,7,24,51,39,51,3,27,65,39,52,5,33,53,39,52,6,27,52,39,37,2,35,38,39,37,5,29,37,39,37,3,33,39,39,37,4,32,50,39,37,3,32,50,39,37,7,29,38,39,37,4,31,38,39,47,5,30,48,39,36,4,31,37,39,36,3,33,37,39,36,1,37,37,39,36,3,33,36,39,36,1,35,50,39,36,3,32,37,39,50,4,28,63,39,51,16,6,63,39,50,7,28,63,39,37,3,34,12],"a72":[39,0,0,0,0,39,52,14,10,53,39,52,8,23,21,39,49,1,36,49,39,51,4,32,55,39,50,2,36,51,39,49,2,36,51,39,52,16,9,21,39,52,10,24,66,39,52,5,24,66,39,41,3,33,33,39,41,3,33,33,39,8,12,12,22,39,27,5,29,6,39,9,13,11,10,39,52,4,30,56,39,49,3,32,50,39,49,6,18,50,39,49,5,30,49,39,49,5,28,50,39,49,3,33,49,39,49,4,31,50,39,49,5,30,50,39,49,6,28,49,39,49,4,31,50,39,49,5,29,50,39,33,13,11,34,39,33,12,12,47,39,44,2,34,41,39,35,3,33,22,39,44,3,34,41,39,52,4,30,54,39,50,2,35,51,39,50,1,37,51,39,49,3,33,49,39,49,3,34,50,39,49,4,32,50,39,49,4,31,49,39,49,5,29,49,39,49,2,34,50,39,49,4,31,50,39,49,6,26,50,39,49,3,33,50,39,49,3,34,50,39,49,5,30,50,39,49,3,33,50,39,49,3,33,50,39,49,2,35,50,39,49,4,32,49,39,49,2,35,61,39,49,4,33,49,39,49,3,33,50,39,49,2,34,50,39,49,3,33,50,39,49,1,37,50,39,49,1,37,50,39,49,3,34,49,39,49,2,35,49,39,49,3,33,50,39,53,10,23,60,39,52,4,30,56,39,53,6,23,60,39,49,7,25,23,39,-1,2,35,6,39,53,10,15,20,39,37,3,31,38,39,52,4,32,53,39,37,4,32,38,39,52,3,32,53,39,37,3,32,38,39,53,4,34,53,39,37,2,35,51,39,52,5,29,52,39,51,7,24,51,39,51,3,27,65,39,52,5,33,53,39,52,6,27,52,39,37,2,35,38,39,37,5,29,37,39,37,3,33,39,39,37,4,32,50,39,37,3,32,50,39,37,7,29,38,39,37,4,31,38,39,47,5,30,48,39,36,4,31,37,39,36,3,33,37,39,36,1,37,37,39,36,3,33,36,39,36,1,35,50,39,36,3,32,37,39,50,4,28,63,39,51,16,6,63,39,50,7,28,63,39,37,3,34,12],"m80":[43,0,0,0,0,43,58,15,11,60,43,58,9,25,23,43,55,2,39,55,43,57,5,34,62,43,55,2,39,56,43,55,3,38,57,43,58,17,10,23,43,58,11,25,73,43,58,6,25,74,43,45,3,36,36,43,46,3,36,38,43,9,13,13,24,43,30,5,32,6,43,9,15,11,11,43,58,5,32,62,43,55,4,35,57,43,54,7,19,55,43,55,6,32,55,43,54,5,31,55,43,54,4,35,54,43,54,5,33,56,43,55,5,33,56,43,54,6,31,54,43,55,5,33,56,43,55,5,32,56,43,37,15,11,39,43,37,13,13,52,43,49,2,37,45,43,38,3,36,24,43,49,3,37,45,43,58,5,32,60,43,55,2,38,57,43,55,1,40,56,43,54,4,35,54,43,55,3,37,56,43,54,4,35,55,43,54,4,34,54,43,54,6,31,54,43,55,3,36,56,43,54,4,34,55,43,54,6,29,55,43,54,4,35,56,43,55,3,37,56,43,54,5,33,55,43,54,3,36,55,43,54,4,35,55,43,55,2,38,56,43,54,4,35,54,43,55,2,38,68,43,54,4,36,54,43,55,4,35,56,43,54,2,38,55,43,54,4,35,56,43,54,2,39,55,43,54,1,41,55,43,54,3,37,54,43,54,2,38,54,43,54,4,36,55,43,58,11,25,66,43,58,5,32,62,43,58,6,26,66,43,54,7,28,25,43,-1,2,38,6,43,58,11,16,22,43,41,4,33,43,43,58,5,34,60,43,41,4,35,43,43,58,3,35,59,43,41,4,34,42,43,58,5,36,58,43,41,3,38,56,43,58,5,32,58,43,57,8,26,57,43,57,4,29,73,43,58,5,36,59,43,58,6,30,58,43,41,3,37,42,43,41,5,32,41,43,41,3,36,43,43,41,5,35,56,43,41,3,35,56,43,41,8,31,42,43,41,4,34,42,43,52,5,33,53,43,40,5,33,42,43,40,3,36,41,43,40,1,40,41,43,40,3,36,40,43,40,1,39,56,43,40,4,35,41,43,56,4,31,70,43,57,18,6,71,43,56,7,31,70,43,41,3,38,14],"a80":[43,0,0,0,0,43,58,15,11,60,43,58,9,25,23,43,55,2,39,55,43,57,5,34,62,43,55,2,39,56,43,55,3,38,57,43,58,17,10,23,43,58,11,25,73,43,58,6,25,74,43,45,3,36,36,43,46,3,36,38,43,9,13,13,24,43,30,5,32,6,43,9,15,11,11,43,58,5,32,62,43,55,4,35,57,43,54,7,19,55,43,55,6,32,55,43,54,5,31,55,43,54,4,35,54,43,54,5,33,56,43,55,5,33,56,43,54,6,31,54,43,55,5,33,56,43,55,5,32,56,43,37,15,11,39,43,37,13,13,52,43,49,2,37,45,43,38,3,36,24,43,49,3,37,45,43,58,5,32,60,43,55,2,38,57,43,55,1,40,56,43,54,4,35,54,43,55,3,37,56,43,54,4,35,55,43,54,4,34,54,43,54,6,31,54,43,55,3,36,56,43,54,4,34,55,43,54,6,29,55,43,54,4,35,56,43,55,3,37,56,43,54,5,33,55,43,54,3,36,55,43,54,4,35,55,43,55,2,38,56,43,54,4,35,54,43,55,2,38,68,43,54,4,36,54,43,55,4,35,56,43,54,2,38,55,43,54,4,35,56,43,54,2,39,55,43,54,1,41,55,43,54,3,37,54,43,54,2,38,54,43,54,4,36,55,43,58,11,25,66,43,58,5,32,62,43,58,6,26,66,43,54,7,28,25,43,-1,2,38,6,43,58,11,16,22,43,41,4,33,43,43,58,5,34,60,43,41,4,35,43,43,58,3,35,59,43,41,4,34,42,43,58,5,36,58,43,41,3,38,56,43,58,5,32,58,43,57,8,26,57,43,57,4,29,73,43,58,5,36,59,43,58,6,30,58,43,41,3,37,42,43,41,5,32,41,43,41,3,36,43,43,41,5,35,56,43,41,3,35,56,43,41,8,31,42,43,41,4,34,42,43,52,5,33,53,43,40,5,33,42,43,40,3,36,41,43,40,1,40,41,43,40,3,36,40,43,40,1,39,56,43,40,4,35,41,43,56,4,31,70,43,57,18,6,71,43,56,7,31,70,43,41,3,38,14],"m81":[44,0,0,0,0,44,59,15,12,61,44,59,9,26,24,44,55,2,40,55,44,57,5,36,62,44,56,2,41,57,44,56,3,39,58,44,59,18,10,24,44,59,12,26,75,44,59,6,26,75,44,46,3,38,37,44,46,3,37,37,44,9,14,13,24,44,31,6,32,7,44,10,15,11,12,44,59,5,34,63,44,55,4,36,57,44,55,7,20,56,44,55,6,33,55,44,55,6,32,56,44,55,4,36,55,44,55,5,34,57,44,56,5,34,57,44,55,6,32,55,44,56,5,34,57,44,56,5,34,57,44,37,15,11,39,44,37,14,13,52,44,50,2,39,46,44,39,3,37,25,44,50,3,38,46,44,59,5,33,61,44,56,2,39,58,44,56,1,42,57,44,55,4,36,55,44,55,3,38,56,44,55,4,37,56,44,55,5,34,55,44,55,6,33,55,44,56,3,38,57,44,55,4,36,56,44,55,7,29,56,44,55,4,36,57,44,55,3,39,56,44,55,5,34,56,44,55,3,38,56,44,55,4,36,56,44,56,2,40,57,44,55,5,35,55,44,56,2,40,70,44,55,4,37,55,44,55,4,36,56,44,55,2,39,56,44,55,4,36,57,44,55,2,40,56,44,55,1,42,56,44,55,3,38,55,44,55,2,40,55,44,55,4,37,56,44,59,11,26,67,44,59,5,34,63,44,59,7,26,67,44,55,8,28,26,44,-1,2,39,7,44,59,12,16,22,44,41,4,34,43,44,59,5,35,61,44,41,4,36,43,44,59,3,37,60,44,41,4,35,42,44,59,5,38,59,44,42,3,39,58,44,59,6,32,59,44,58,8,27,58,44,58,4,30,74,44,59,5,37,60,44,59,6,31,59,44,41,3,39,42,44,41,5,33,41,44,41,3,38,43,44,41,5,36,56,44,41,3,36,56,44,41,8,33,42,44,41,4,35,42,44,52,6,33,53,44,40,5,34,42,44,40,3,37,41,44,40,1,41,41,44,40,4,36,40,44,40,1,40,56,44,40,4,36,41,44,56,4,32,71,44,58,18,7,72,44,56,8,32,71,44,41,3,39,13],"a81":[44,0,0,0,0,44,59,15,12,61,44,59,9,26,24,44,55,2,40,55,44,57,5,36,62,44,56,2,41,57,44,56,3,39,58,44,59,18,10,24,44,59,12,26,75,44,59,6,26,75,44,46,3,38,37,44,46,3,37,37,44,9,14,13,24,44,31,6,32,7,44,10,15,11,12,44,59,5,34,63,44,55,4,36,57,44,55,7,20,56,44,55,6,33,55,44,55,6,32,56,44,55,4,36,55,44,55,5,34,57,44,56,5,34,57,44,55,6,32,55,44,56,5,34,57,44,56,5,34,57,44,37,15,11,39,44,37,14,13,52,44,50,2,39,46,44,39,3,37,25,44,50,3,38,46,44,59,5,33,61,44,56,2,39,58,44,56,1,42,57,44,55,4,36,55,44,55,3,38,56,44,55,4,37,56,44,55,5,34,55,44,55,6,33,55,44,56,3,38,57,44,55,4,36,56,44,55,7,29,56,44,55,4,36,57,44,55,3,39,56,44,55,5,34,56,44,55,3,38,56,44,55,4,36,56,44,56,2,40,57,44,55,5,35,55,44,56,2,40,70,44,55,4,37,55,44,55,4,36,56,44,55,2,39,56,44,55,4,36,57,44,55,2,40,56,44,55,1,42,56,44,55,3,38,55,44,55,2,40,55,44,55,4,37,56,44,59,11,26,67,44,59,5,34,63,44,59,7,26,67,44,55,8,28,26,44,-1,2,39,7,44,59,12,16,22,44,41,4,34,43,44,59,5,35,61,44,41,4,36,43,44,59,3,37,60,44,41,4,35,42,44,59,5,38,59,44,42,3,39,58,44,59,6,32,59,44,58,8,27,58,44,58,4,30,74,44,59,5,37,60,44,59,6,31,59,44,41,3,39,42,44,41,5,33,41,44,41,3,38,43,44,41,5,36,56,44,41,3,36,56,44,41,8,33,42,44,41,4,35,42,44,52,6,33,53,44,40,5,34,42,44,40,3,37,41,44,40,1,41,41,44,40,4,36,40,44,40,1,40,56,44,40,4,36,41,44,56,4,32,71,44,58,18,7,72,44,56,8,32,71,44,41,3,39,13],"m84":[45,0,0,0,0,45,61,16,11,63,45,61,9,27,24,45,57,2,41,56,45,59,5,36,64,45,58,2,41,59,45,58,3,40,60,45,61,18,11,24,45,61,12,26,77,45,61,6,27,77,45,47,3,38,38,45,48,3,38,39,45,10,14,14,26,45,32,6,33,7,45,10,15,12,12,45,61,5,34,65,45,57,4,36,59,45,57,7,20,58,45,57,6,34,57,45,57,6,32,58,45,57,4,37,57,45,57,5,35,59,45,58,5,35,59,45,57,6,33,57,45,58,5,35,59,45,58,6,33,59,45,39,15,12,41,45,39,14,14,55,45,52,3,38,48,45,40,3,38,25,45,52,3,39,48,45,61,5,34,63,45,58,2,40,60,45,58,1,42,59,45,57,4,37,57,45,57,3,39,58,45,57,4,37,58,45,57,5,35,57,45,57,6,33,57,45,58,3,38,59,45,57,4,36,58,45,57,7,29,58,45,57,4,37,59,45,57,3,39,58,45,57,5,35,58,45,57,3,38,58,45,57,4,37,58,45,58,2,40,59,45,57,5,36,57,45,58,2,40,72,45,57,4,38,57,45,57,4,37,58,45,57,2,40,58,45,57,4,37,59,45,57,2,41,58,45,57,1,43,58,45,57,3,39,57,45,57,2,40,57,45,57,4,38,58,45,61,11,27,70,45,61,5,34,65,45,61,7,26,70,45,57,8,28,26,45,-1,3,39,7,45,61,12,17,23,45,43,4,35,45,45,61,5,36,63,45,43,4,37,45,45,61,3,37,62,45,43,4,36,44,45,61,5,38,61,45,43,3,40,59,45,61,6,33,61,45,60,8,28,60,45,60,4,31,76,45,61,5,38,62,45,61,6,32,61,45,43,3,39,44,45,43,6,33,43,45,43,3,38,45,45,43,5,37,59,45,43,3,37,59,45,43,8,33,44,45,43,4,36,44,45,54,6,34,55,45,42,5,35,44,45,42,3,38,43,45,42,1,42,43,45,42,4,37,42,45,42,1,41,58,45,42,4,36,43,45,59,4,33,74,45,60,19,7,74,45,59,8,32,74,45,43,3,40,14],"a84":[45,0,0,0,0,45,61,16,11,63,45,61,9,27,24,45,57,2,41,56,45,59,5,36,64,45,58,2,41,59,45,58,3,40,60,45,61,18,11,24,45,61,12,26,77,45,61,6,27,77,45,47,3,38,38,45,48,3,38,39,45,10,14,14,26,45,32,6,33,7,45,10,15,12,12,45,61,5,34,65,45,57,4,36,59,45,57,7,20,58,45,57,6,34,57,45,57,6,32,58,45,57,4,37,57,45,57,5,35,59,45,58,5,35,59,45,57,6,33,57,45,58,5,35,59,45,58,6,33,59,45,39,15,12,41,45,39,14,14,55,45,52,3,38,48,45,40,3,38,25,45,52,3,39,48,45,61,5,34,63,45,58,2,40,60,45,58,1,42,59,45,57,4,37,57,45,57,3,39,58,45,57,4,37,58,45,57,5,35,57,45,57,6,33,57,45,58,3,38,59,45,57,4,36,58,45,57,7,29,58,45,57,4,37,59,45,57,3,39,58,45,57,5,35,58,45,57,3,38,58,45,57,4,37,58,45,58,2,40,59,45,57,5,36,57,45,58,2,40,72,45,57,4,38,57,45,57,4,37,58,45,57,2,40,58,45,57,4,37,59,45,57,2,41,58,45,57,1,43,58,45,57,3,39,57,45,57,2,40,57,45,57,4,38,58,45,61,11,27,70,45,61,5,34,65,45,61,7,26,70,45,57,8,28,26,45,-1,3,39,7,45,61,12,17,23,45,43,4,35,45,45,61,5,36,63,45,43,4,37,45,45,61,3,37,62,45,43,4,36,44,45,61,5,38,61,45,43,3,40,59,45,61,6,33,61,45,60,8,28,60,45,60,4,31,76,45,61,5,38,62,45,61,6,32,61,45,43,3,39,44,45,43,6,33,43,45,43,3,38,45,45,43,5,37,59,45,43,3,37,59,45,43,8,33,44,45,43,4,36,44,45,54,6,34,55,45,42,5,35,44,45,42,3,38,43,45,42,1,42,43,45,42,4,37,42,45,42,1,41,58,45,42,4,36,43,45,59,4,33,74,45,60,19,7,74,45,59,8,32,74,45,43,3,40,14],"m90":[49,0,0,0,0,49,65,17,12,67,49,65,10,28,26,49,61,2,44,60,49,64,5,39,69,49,62,2,44,64,49,62,3,43,64,49,65,19,12,26,49,65,13,28,82,49,65,6,29,83,49,51,3,41,41,49,51,4,40,41,49,10,15,15,27,49,34,6,35,7,49,11,17,12,13,49,65,6,36,70,49,61,4,39,63,49,61,8,21,62,49,61,6,36,61,49,61,6,35,63,49,61,4,40,61,49,61,6,37,63,49,62,6,36,64,49,61,7,35,61,49,62,5,37,64,49,62,6,36,64,49,42,17,12,44,49,42,15,15,59,49,56,3,41,52,49,43,4,40,27,49,56,3,42,52,49,65,6,36,67,49,62,2,43,64,49,62,1,45,63,49,61,4,40,61,49,62,3,42,64,49,61,5,39,62,49,61,5,37,61,49,61,7,35,61,49,62,3,41,63,49,61,5,38,62,49,61,7,32,62,49,61,4,40,63,49,62,4,41,63,49,61,6,36,62,49,61,3,41,62,49,61,4,40,62,49,62,2,43,63,49,61,5,39,61,49,62,2,43,77,49,61,5,40,61,49,61,4,40,63,49,61,2,42,62,49,61,4,40,63,49,61,2,44,62,49,61,1,46,62,49,61,3,42,61,49,61,2,43,61,49,61,4,40,62,49,66,12,28,75,49,65,6,36,70,49,66,7,29,75,49,61,8,31,28,49,-1,3,42,7,49,66,13,18,25,49,46,4,38,48,49,65,5,39,67,49,46,5,39,48,49,65,4,39,67,49,46,4,39,48,49,66,5,41,66,49,46,3,43,63,49,65,6,36,65,49,64,9,29,64,49,64,4,33,82,49,65,6,40,66,49,65,7,34,65,49,46,3,42,47,49,46,6,36,46,49,46,3,41,48,49,46,5,39,63,49,46,4,38,63,49,46,9,35,47,49,46,5,37,48,49,58,6,37,59,49,45,5,37,47,49,45,3,41,46,49,45,1,45,46,49,45,4,40,45,49,45,2,43,62,49,45,4,39,46,49,63,4,35,79,49,64,20,7,79,49,63,8,35,79,49,46,3,43,15],"a90":[49,0,0,0,0,49,65,17,12,67,49,65,10,28,26,49,61,2,44,60,49,64,5,39,69,49,62,2,44,64,49,62,3,43,64,49,65,19,12,26,49,65,13,28,82,49,65,6,29,83,49,51,3,41,41,49,51,4,40,41,49,10,15,15,27,49,34,6,35,7,49,11,17,12,13,49,65,6,36,70,49,61,4,39,63,49,61,8,21,62,49,61,6,36,61,49,61,6,35,63,49,61,4,40,61,49,61,6,37,63,49,62,6,36,64,49,61,7,35,61,49,62,5,37,64,49,62,6,36,64,49,42,17,12,44,49,42,15,15,59,49,56,3,41,52,49,43,4,40,27,49,56,3,42,52,49,65,6,36,67,49,62,2,43,64,49,62,1,45,63,49,61,4,40,61,49,62,3,42,64,49,61,5,39,62,49,61,5,37,61,49,61,7,35,61,49,62,3,41,63,49,61,5,38,62,49,61,7,32,62,49,61,4,40,63,49,62,4,41,63,49,61,6,36,62,49,61,3,41,62,49,61,4,40,62,49,62,2,43,63,49,61,5,39,61,49,62,2,43,77,49,61,5,40,61,49,61,4,40,63,49,61,2,42,62,49,61,4,40,63,49,61,2,44,62,49,61,1,46,62,49,61,3,42,61,49,61,2,43,61,49,61,4,40,62,49,66,12,28,75,49,65,6,36,70,49,66,7,29,75,49,61,8,31,28,49,-1,3,42,7,49,66,13,18,25,49,46,4,38,48,49,65,5,39,67,49,46,5,39,48,49,65,4,39,67,49,46,4,39,48,49,66,5,41,66,49,46,3,43,63,49,65,6,36,65,49,64,9,29,64,49,64,4,33,82,49,65,6,40,66,49,65,7,34,65,49,46,3,42,47,49,46,6,36,46,49,46,3,41,48,49,46,5,39,63,49,46,4,38,63,49,46,9,35,47,49,46,5,37,48,49,58,6,37,59,49,45,5,37,47,49,45,3,41,46,49,45,1,45,46,49,45,4,40,45,49,45,2,43,62,49,45,4,39,46,49,63,4,35,79,49,64,20,7,79,49,63,8,35,79,49,46,3,43,15],"m96":[52,0,0,0,0,52,70,18,13,72,52,70,11,30,28,52,65,2,47,64,52,68,6,41,73,52,66,2,47,68,52,66,3,46,68,52,70,20,13,28,52,70,14,30,88,52,69,7,30,88,52,54,4,43,44,52,55,4,43,45,52,11,16,15,29,52,36,7,37,7,52,11,18,13,13,52,69,6,39,74,52,65,5,41,67,52,65,8,23,66,52,65,7,38,65,52,65,7,37,67,52,65,4,42,65,52,65,6,40,67,52,66,6,39,68,52,65,7,37,65,52,66,6,39,68,52,66,6,39,68,52,44,18,13,46,52,44,16,15,62,52,59,3,44,55,52,46,4,43,29,52,59,3,45,55,52,69,6,39,71,52,66,3,45,68,52,66,1,49,67,52,65,4,43,65,52,66,4,44,68,52,65,5,42,66,52,65,5,40,65,52,65,7,38,65,52,66,3,44,68,52,65,5,41,66,52,65,8,33,66,52,65,4,43,67,52,66,4,44,67,52,65,6,39,66,52,65,4,43,66,52,65,5,42,66,52,66,2,46,68,52,65,5,42,65,52,66,2,46,82,52,65,5,43,65,52,65,5,42,67,52,65,3,44,66,52,65,5,42,67,52,65,2,47,66,52,65,1,49,66,52,65,4,44,65,52,65,2,46,65,52,65,5,42,66,52,70,13,30,80,52,69,6,39,74,52,70,8,30,80,52,65,9,32,30,52,-1,3,45,8,52,70,14,19,26,52,49,4,41,51,52,69,6,41,71,52,49,5,42,51,52,69,4,42,71,52,49,4,42,51,52,70,6,44,70,52,49,3,46,67,52,69,7,37,69,52,68,10,31,68,52,68,4,36,87,52,69,6,43,70,52,69,7,36,69,52,49,3,45,50,52,49,6,38,49,52,49,4,43,51,52,49,6,41,67,52,49,4,41,67,52,49,10,37,50,52,49,5,40,51,52,62,7,39,63,52,48,6,39,50,52,48,3,44,49,52,48,1,48,49,52,48,4,43,48,52,48,2,46,67,52,48,5,41,49,52,67,5,37,84,52,68,21,8,84,52,67,9,37,84,52,49,4,45,16],"a96":[52,0,0,0,0,52,70,18,13,72,52,70,11,30,28,52,65,2,47,64,52,68,6,41,73,52,66,2,47,68,52,66,3,46,68,52,70,20,13,28,52,70,14,30,88,52,69,7,30,88,52,54,4,43,44,52,55,4,43,45,52,11,16,15,29,52,36,7,37,7,52,11,18,13,13,52,69,6,39,74,52,65,5,41,67,52,65,8,23,66,52,65,7,38,65,52,65,7,37,67,52,65,4,42,65,52,65,6,40,67,52,66,6,39,68,52,65,7,37,65,52,66,6,39,68,52,66,6,39,68,52,44,18,13,46,52,44,16,15,62,52,59,3,44,55,52,46,4,43,29,52,59,3,45,55,52,69,6,39,71,52,66,3,45,68,52,66,1,49,67,52,65,4,43,65,52,66,4,44,68,52,65,5,42,66,52,65,5,40,65,52,65,7,38,65,52,66,3,44,68,52,65,5,41,66,52,65,8,33,66,52,65,4,43,67,52,66,4,44,67,52,65,6,39,66,52,65,4,43,66,52,65,5,42,66,52,66,2,46,68,52,65,5,42,65,52,66,2,46,82,52,65,5,43,65,52,65,5,42,67,52,65,3,44,66,52,65,5,42,67,52,65,2,47,66,52,65,1,49,66,52,65,4,44,65,52,65,2,46,65,52,65,5,42,66,52,70,13,30,80,52,69,6,39,74,52,70,8,30,80,52,65,9,32,30,52,-1,3,45,8,52,70,14,19,26,52,49,4,41,51,52,69,6,41,71,52,49,5,42,51,52,69,4,42,71,52,49,4,42,51,52,70,6,44,70,52,49,3,46,67,52,69,7,37,69,52,68,10,31,68,52,68,4,36,87,52,69,6,43,70,52,69,7,36,69,52,49,3,45,50,52,49,6,38,49,52,49,4,43,51,52,49,6,41,67,52,49,4,41,67,52,49,10,37,50,52,49,5,40,51,52,62,7,39,63,52,48,6,39,50,52,48,3,44,49,52,48,1,48,49,52,48,4,43,48,52,48,2,46,67,52,48,5,41,49,52,67,5,37,84,52,68,21,8,84,52,67,9,37,84,52,49,4,45,16],"m108":[58,0,0,0,0,58,78,20,15,80,58,78,12,34,31,58,74,2,53,73,58,76,6,47,82,58,74,3,53,76,58,74,4,51,76,58,78,23,14,31,58,78,15,34,99,58,78,8,34,99,58,61,4,49,49,58,61,5,47,49,58,12,18,17,32,58,41,8,42,9,58,13,20,14,15,58,78,7,43,84,58,74,5,47,76,58,73,10,25,74,58,74,8,43,74,58,73,7,42,75,58,73,5,47,73,58,73,7,44,75,58,74,7,44,76,58,73,8,42,73,58,74,6,45,76,58,74,7,43,76,58,50,20,14,52,58,50,18,17,70,58,67,3,50,62,58,52,5,47,33,58,67,4,50,62,58,78,7,43,80,58,74,3,51,76,58,74,1,55,75,58,73,5,48,73,58,74,4,50,76,58,73,6,47,74,58,73,6,45,73,58,73,8,42,73,58,74,4,49,76,58,73,6,46,74,58,73,9,38,74,58,73,5,48,75,58,74,4,50,75,58,73,7,44,74,58,73,4,49,74,58,73,5,47,74,58,74,3,51,76,58,73,6,46,73,58,74,3,51,92,58,73,6,48,73,58,73,5,47,75,58,73,3,50,74,58,73,5,47,75,58,73,2,53,74,58,73,1,55,74,58,73,4,50,73,58,73,3,51,73,58,73,5,48,74,58,79,14,34,90,58,78,7,43,84,58,79,9,34,90,58,73,10,37,34,58,-2,3,51,8,58,79,15,22,30,58,55,5,45,57,58,78,6,47,80,58,55,6,47,57,58,78,5,47,80,58,55,5,46,57,58,79,7,49,79,58,56,4,51,77,58,78,7,43,78,58,77,11,35,77,58,77,5,39,98,58,78,7,48,79,58,78,8,41,78,58,55,4,50,56,58,55,7,43,55,58,55,4,49,57,58,55,6,47,75,58,55,4,47,75,58,55,11,42,56,58,55,6,45,57,58,70,7,44,72,58,54,7,44,56,58,54,4,49,55,58,54,1,54,55,58,54,5,47,54,58,54,2,51,75,58,54,5,47,55,58,75,5,42,94,58,77,24,9,95,58,75,10,42,94,58,55,4,51,18],"a108":[58,0,0,0,0,58,78,20,15,80,58,78,12,34,31,58,74,2,53,73,58,76,6,47,82,58,74,3,53,76,58,74,4,51,76,58,78,23,14,31,58,78,15,34,99,58,78,8,34,99,58,61,4,49,49,58,61,5,47,49,58,12,18,17,32,58,41,8,42,9,58,13,20,14,15,58,78,7,43,84,58,74,5,47,76,58,73,10,25,74,58,74,8,43,74,58,73,7,42,75,58,73,5,47,73,58,73,7,44,75,58,74,7,44,76,58,73,8,42,73,58,74,6,45,76,58,74,7,43,76,58,50,20,14,52,58,50,18,17,70,58,67,3,50,62,58,52,5,47,33,58,67,4,50,62,58,78,7,43,80,58,74,3,51,76,58,74,1,55,75,58,73,5,48,73,58,74,4,50,76,58,73,6,47,74,58,73,6,45,73,58,73,8,42,73,58,74,4,49,76,58,73,6,46,74,58,73,9,38,74,58,73,5,48,75,58,74,4,50,75,58,73,7,44,74,58,73,4,49,74,58,73,5,47,74,58,74,3,51,76,58,73,6,46,73,58,74,3,51,92,58,73,6,48,73,58,73,5,47,75,58,73,3,50,74,58,73,5,47,75,58,73,2,53,74,58,73,1,55,74,58,73,4,50,73,58,73,3,51,73,58,73,5,48,74,58,79,14,34,90,58,78,7,43,84,58,79,9,34,90,58,73,10,37,34,58,-2,3,51,8,58,79,15,22,30,58,55,5,45,57,58,78,6,47,80,58,55,6,47,57,58,78,5,47,80,58,55,5,46,57,58,79,7,49,79,58,56,4,51,77,58,78,7,43,78,58,77,11,35,77,58,77,5,39,98,58,78,7,48,79,58,78,8,41,78,58,55,4,50,56,58,55,7,43,55,58,55,4,49,57,58,55,6,47,75,58,55,4,47,75,58,55,11,42,56,58,55,6,45,57,58,70,7,44,72,58,54,7,44,56,58,54,4,49,55,58,54,1,54,55,58,54,5,47,54,58,54,2,51,75,58,54,5,47,55,58,75,5,42,94,58,77,24,9,95,58,75,10,42,94,58,55,4,51,18]},"fonts":{}};
  if (typeof bwipjs_fonts == "object") {
    bwipjs_fonts.fontsets[3] = desc;
    bwipjs_fonts.names["INCONSOLATA"] = 3;
  } else {
    module.exports = desc;
  }
})();
if (bwipjs_fonts.onready) { bwipjs_fonts.onready.call(null); }
