import { useState, useEffect } from 'react';
import { useOktaAuth } from '@okta/okta-react';
import { Box, Text } from '@okta/odyssey-react';
import InfoBox from '../InfoBox';
import Spinner from '../Spinner';
import AddAttributeButton from '../AddAttributeButton';
import RemoveButton from '../RemoveAttributeButton';
import { 
  addEmail, 
  addPhone, 
  deleteEmail, 
  deletePhone, 
  getEmails, 
  getPhones 
} from '../../api/MyAccountAPI';
import { capitalizeFirstLetter } from '../../util';

const IdentifiersSection = () => {
  const { oktaAuth } = useOktaAuth();
  const [emails, setEmails] = useState();
  const [phones, setPhones] = useState();

  useEffect(() => {
    const fetchEmails = async () => {
      const emails = await getEmails(oktaAuth);
      setEmails(emails.map(email => {
        if (email.roles.includes('PRIMARY')) {
          email.label = 'Primary email';
          email.selectorId='primary-email';
        } else {
          email.label = 'Secondary email';
          email.selectorId='secondary-email';
        }
        email.selectorId = email.label
          .split(' ')
          .map(str => str.toLowerCase())
          .join('-');
        email.selectorHint = email.label
          .split(' ')
          .map(capitalizeFirstLetter)
          .join('');
        return email;
      }));
    };
    if (!emails) {
      fetchEmails();
    }
  }, [oktaAuth, emails]);

  useEffect(() => {
    const fetchPhones = async () => {
      const phones = await getPhones(oktaAuth);
      setPhones(phones);
    };
    if (!phones) {
      fetchPhones();
    }
  }, [oktaAuth, phones]);

  const finishEmailTransaction = () => {
    // re-fetch phones list
    setEmails(null);
  };

  const finishPhoneTransaction = () => {
    // re-fetch phones list
    setPhones(null);
  };

  const startUpdateEmailTransaction = async (emailObj, email) => {
    return addEmail(oktaAuth, {
      profile: {
        email: email
      },
      sendEmail: true,
      role: emailObj.roles[0]
    });
  };

  const startAddSecondaryEmailTransaction = async (email) => {
    return addEmail(oktaAuth, {
      profile: {
        email
      },
      sendEmail: true,
      role: 'SECONDARY'
    });
  };

  const handleRemoveEmail = async (emailId) => {
    await deleteEmail(oktaAuth, emailId);
  };

  const startAddPhoneTransaction = async phone => {
    return addPhone(oktaAuth, phone);
  };

  const handleRemovePhone = async (phoneId) => {
    await deletePhone(oktaAuth, phoneId);
  };

  const startEmailVerificationTransaction = async (email) => {
    return email.challenge();
  };

  const startPhoneVerificationTransaction = async (phone) => {
    await phone.challenge({ data: { method: 'SMS' } }); // no response
    return phone;
  };

  return (
    <Box display="flex" className="pure-g" padding="m">
      <Box 
        className="pure-u-1 pure-u-sm-1-2" 
        paddingRight="s" 
        display="flex" 
        flexDirection="column"
        marginTop="s" 
      >
        {emails ? (
          <Box>
            {emails.map(email => (
                <Box key={email.id} display="flex" flexDirection="column" paddingBottom="s">
                  <Box display="flex" alignItems="center" justifyContent="flex-start">
                    <Text as="strong">{email.label}</Text>
                    {email.status === 'VERIFIED' ? (
                      <Box marginLeft="s">
                        <AddAttributeButton 
                          heading={`Edit ${email.label}`}
                          initInputLabel="Email"
                          selectorHint={`edit${email.selectorHint}`}
                          onStartTransaction={startUpdateEmailTransaction.bind(null, email)} 
                          onFinishTransaction={finishEmailTransaction}
                        >
                          Edit
                        </AddAttributeButton>
                      </Box>
                    ) : (
                      <>
                      <Box marginLeft="s">
                        <AddAttributeButton 
                          heading="Verify Email"
                          initInputLabel="Verification Code"
                          selectorHint={`verify${email.selectorHint}`}
                          autoStartTransaction
                          onStartTransaction={startEmailVerificationTransaction.bind(null, email)} 
                          onFinishTransaction={finishEmailTransaction}
                        >
                          Verify
                        </AddAttributeButton>
                      </Box>
                      <Box marginLeft="s">
                        <RemoveButton 
                          heading="Are you sure you want to remove this email?" 
                          description={email.profile.email}
                          onStartTransaction={handleRemoveEmail.bind(null, email.id)}
                          onFinishTransaction={finishEmailTransaction}
                        >
                          Remove
                        </RemoveButton>
                      </Box>
                      </>
                    )}
                  </Box>
                  <Box paddingTop="s">
                    <Text id={email.selectorId}>{email.profile.email}</Text>
                  </Box>
                </Box>
              ))}
            {!emails.some(email => email.roles.includes('SECONDARY')) && (
              <AddAttributeButton 
                heading="Add secondary email" 
                initInputLabel="Email"
                selectorHint="addSecondaryEmail"
                onStartTransaction={startAddSecondaryEmailTransaction} 
                onFinishTransaction={finishEmailTransaction}
              >
                Add Secondary Email
              </AddAttributeButton>
            )}
          </Box>
        ) : (
          <Spinner />
        )}
        <Box id="phone-section">
          <Text as="strong">Phone number</Text>
          {phones ? phones.map(phone => (
            <Box key={phone.id} display="flex" alignItems="center" paddingTop="s">
              <Box className='phone-number'>
                <Text>{phone.profile.phoneNumber}</Text>
              </Box>
              {phone.status === 'UNVERIFIED' && (
                <Box marginLeft="s">
                  <AddAttributeButton 
                    heading="Verify Phone Number"
                    initInputLabel="Verification Code"
                    autoStartTransaction
                    onStartTransaction={startPhoneVerificationTransaction.bind(null, phone)} 
                    onFinishTransaction={finishPhoneTransaction}
                  >
                    Verify
                  </AddAttributeButton>
                </Box>
              )}
              <Box marginLeft="s">
                <RemoveButton 
                  selectorHint="removePhoneNumber"
                  heading="Are you sure you want to remove this phone number?" 
                  description={phone.profile.phoneNumber}
                  onStartTransaction={handleRemovePhone.bind(null, phone.id)}
                  onFinishTransaction={finishPhoneTransaction}
                >
                  Remove
                </RemoveButton>
              </Box>
            </Box>
          )) : (
            <Spinner />
          )}
          <Box paddingTop="s" paddingBottom="s">
            <AddAttributeButton 
              heading="Add Phone Number"
              initInputLabel="Phone Number"
              selectorHint="addPhoneNumber"
              onStartTransaction={startAddPhoneTransaction} 
              onFinishTransaction={finishPhoneTransaction}
            >
              Add phone number
            </AddAttributeButton>
          </Box>
        </Box>
      </Box>
      <Box className="pure-u-1 pure-u-sm-1-2" marginTop="s">
        <InfoBox 
          id="identifiers-tip"
          heading="Tip" 
          icon="information-circle-filled" 
          renderInfo={() => (
            <Text as="p">User identifiers are separated because changes require verification.</Text>
          )} 
        />
      </Box>
    </Box>
  );
};

export default IdentifiersSection;
