# v5.0.0

The main theme of the v5.0.0 release is an improved developer/user experience with OpenLayers. Toward this end, we have reworked the library as a set of ES Modules, completely removing any dependency on the Closure Compiler, and improving compatibility with mainstream module bundlers.

See the [hosted examples](https://openlayers.org/en/latest/examples/), [API docs](https://openlayers.org/en/latest/apidoc/), and [bundle tutorial](https://openlayers.org/en/latest/doc/tutorials/bundle.html) for the new syntax, but basic usage looks like this:

```js
import 'ol/ol.css';
import {Map, View} from 'ol';
import TileLayer from 'ol/layer/Tile';
import OSM from 'ol/source/OSM';

const map = new Map({
  target: 'map',
  layers: [
    new TileLayer({
      source: new OSM()
    })
  ],
  view: new View({
    center: [0, 0],
    zoom: 0
  })
});
```

See below for specific notes on upgrading from 4.x releases.  We'll be working on updating the website with improved documentation and examples.  Take a look at the issue tracker if you're interested in contributing.

### Upgrade Notes

#### Renamed `ol/source/TileUTFGrid` to `ol/source/UTFGrid`

The module name is now `ol/source/UTFGrid` (`ol.source.UTFGrid` in the full build).

#### Renaming of the `defaultDataProjection` in the options and property of the `ol/format/Feature` class and its subclasses

The `defaultDataProjection` option is now named `dataProjection`. The protected property available on the class is also renamed.

#### `transition` option of `ol/source/VectorTile` is ignored

The `transition` option to get an opacity transition to fade in tiles has been  disabled for `ol/source/VectorTile`. Vector tiles are now always rendered without an opacity transition.

#### `ol/style/Fill` with `CanvasGradient` or `CanvasPattern`

The origin for gradients and patterns has changed from the top-left corner of the extent of the geometry being filled to 512 css pixel increments from map coordinate `[0, 0]`. This allows repeat patterns to be aligned properly with vector tiles. For seamless repeat patterns, width and height of the pattern image must be a factor of two (2, 4, 8, ..., 512).

#### Removal of the renderer option for maps

The `renderer` option has been removed from the `Map` constructor.  The purpose of this change is to avoid bundling code in your application that you do not need.  Previously, code for both the Canvas and WebGL renderers was included in all applications - even though most people only use one renderer.  The `Map` constructor now gives you a Canvas (2D) based renderer.  If you want to try the WebGL renderer, you can import the constructor from `ol/WebGLMap`.

Old code:
```js
import Map from 'ol/Map';

const canvasMap = new Map({
  renderer: ['canvas']
  // other options...
});

const webglMap = new Map({
  renderer: ['webgl']
  // other options...
});
```

New code:
```js
import Map from 'ol/Map';
import WebGLMap from 'ol/WebGLMap';

const canvasMap = new Map({
  // options...
});

const webglMap = new WebGLMap({
  // options...
});
```

#### Removal of ol.FeatureStyleFunction

The signature of the vector style function passed to the feature has changed. The function now always takes the `feature` and the `resolution` as arguments, the `feature` is no longer bound to `this`.

Old code:
```js
feature.setStyle(function(resolution) {
  var text = this.get('name');
  ...
});
```

New code:
```js
feature.setStyle(function(feature, resolution) {
  var text = feature.get('name');
  ...
});
```

#### Changed behavior of the `Draw` interaction

For better drawing experience, two changes were made to the behavior of the Draw interaction:

 1. On long press, the current vertex can be dragged to its desired position.
 2. On touch move (e.g. when panning the map on a mobile device), no draw cursor is shown, and the geometry being drawn is not updated. But because of 1., the draw cursor will appear on long press. Mouse moves are not affected by this change.

#### Changes in proj4 integration

Because relying on a globally available proj4 is not practical with ES modules, we have made a change to the way we integrate proj4:

 * The `setProj4()` function from the `ol/proj` module was removed.
 * A new `ol/proj/proj4` module with a `register()` function was added. Regardless of whether the application imports `proj4` or uses a global `proj4`, this function needs to be called with the proj4 instance as argument whenever projection definitions were added to proj4's registry with (`proj4.defs`).

It is also recommended to no longer use a global `proj4`. Instead,

    npm install proj4

and import it:

```js
import proj4 from 'proj4';
```

Applications can be updated by importing the `register` function from the `ol/proj/proj4` module

```js
import {register} from 'ol/proj/proj4'
```

and calling it before using projections, and any time the proj4 registry was changed by `proj4.defs()` calls:

```js
register(proj4);
```

#### Removal of logos

The map and sources no longer accept a `logo` option.  Instead, if you wish to append a logo to your map, add the desired markup directly in your HTML.  In addition, you can use the `attributions` property of a source to display arbitrary markup per-source with the attribution control.

#### Replacement of `ol/Sphere` constructor with `ol/sphere` functions

The `ol/Sphere` constructor has been removed.  If you were using the `getGeodesicArea` method, use the `getArea` function instead.  If you were using the `haversineDistance` method, use the `getDistance` function instead.

Examples before:
```js
// using ol@4
import Sphere from 'ol/sphere';

var sphere = new Sphere(Sphere.DEFAULT_RADIUS);
var area = sphere.getGeodesicArea(polygon);
var distance = sphere.haversineDistance(g1, g2);
```

Examples after:
```js
// using ol@5
import {circular as circularPolygon} from 'ol/geom/Polygon';
import {getArea, getDistance} from 'ol/sphere';

var area = getArea(polygon);
var distance = getDistance(g1, g2);
var circle = circularPolygon(center, radius);
```

#### New signature for the `circular` function for creating polygons

The `circular` function exported from `ol/geom/Polygon` no longer requires a `Sphere` as the first argument.

Example before:
```js
// using ol@4
import Polygon from 'ol/geom/polygon';
import Sphere from 'ol/sphere';

var poly = Polygon.circular(new Sphere(Sphere.DEFAULT_RADIUS), center, radius);
```

Example after:
```js
// using ol@5
import {circular as circularPolygon} from 'ol/geom/Polygon';

var poly = circularPolygon(center, radius);
```

#### Removal of optional this arguments.

The optional this (i.e. opt_this) arguments were removed from the following methods. Please use closures, the es6 arrow function or the bind method to achieve this effect (Bind is explained here: https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Global_Objects/Function/bind).

* Collection#forEach
* geom/LineString#forEachSegment
* Observable#on, #once, #un
* Map#forEachLayerAtPixel
* source/TileUTFGrid#forDataAtCoordinateAndResolution
* source/Vector#forEachFeature, #forEachFeatureInExtent, #forEachFeatureIntersectingExtent


#### `Map#forEachLayerAtPixel` parameters have changed
 
If you are using the layer filter, please note that you now have to pass in the layer filter via an `AtPixelOptions` object. If you are not using the layer filter the usage has not changed.

Old syntax:
```
    map.forEachLayerAtPixel(pixel, callback, callbackThis, layerFilterFn, layerFilterThis);
```

New syntax:

```
    map.forEachLayerAtPixel(pixel, callback, {
        layerFilter: layerFilterFn
    });
```

To bind a function to a this, please use the bind method of the function (See: https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Global_Objects/Function/bind).

This change is due to the introduction of the `hitTolerance` parameter which can be passed in via this `AtPixelOptions` object, too.

### New Features and Fixes

 * [#8314](https://github.com/openlayers/openlayers/pull/8314) - Accept hitTolerance in map.forEachLayerAtPixel() ([@simonseyock](https://github.com/simonseyock))
 * [#7750](https://github.com/openlayers/openlayers/pull/7750) - Don't assume fill for canvas polygon hit detection. ([@schmidtk](https://github.com/schmidtk))
 * [#8305](https://github.com/openlayers/openlayers/pull/8305) - Update quickstart and tutorials for v5 ([@ahocevar](https://github.com/ahocevar))
 * [#8301](https://github.com/openlayers/openlayers/pull/8301) - Do not render vectors in image mode ([@ahocevar](https://github.com/ahocevar))
 * [#8302](https://github.com/openlayers/openlayers/pull/8302) - Ignore node_modules everywhere ([@ahocevar](https://github.com/ahocevar))
 * [#8299](https://github.com/openlayers/openlayers/pull/8299) - Make classes in ol/ available as named exports from ol ([@ahocevar](https://github.com/ahocevar))
 * [#8298](https://github.com/openlayers/openlayers/pull/8298) - Add README for ol package ([@ahocevar](https://github.com/ahocevar))
 * [#8295](https://github.com/openlayers/openlayers/pull/8295) - Prepare for building/publishing v5 ([@ahocevar](https://github.com/ahocevar))
 * [#8296](https://github.com/openlayers/openlayers/pull/8296) - Fix style documentation ([@fredj](https://github.com/fredj))
 * [#8294](https://github.com/openlayers/openlayers/pull/8294) - Remove non-functional type checks from testing ([@ahocevar](https://github.com/ahocevar))
 * [#8292](https://github.com/openlayers/openlayers/pull/8292) - Remove now unneeded single quotes in examples ([@fredj](https://github.com/fredj))
 * [#8291](https://github.com/openlayers/openlayers/pull/8291) - Graticule code cleanup ([@fredj](https://github.com/fredj))
 * [#8287](https://github.com/openlayers/openlayers/pull/8287) - Rename ol/source/TileUTFGrid to ol/source/UTFGrid ([@fredj](https://github.com/fredj))
 * [#8286](https://github.com/openlayers/openlayers/pull/8286) - Fix class name property in Overlay documentation ([@fredj](https://github.com/fredj))
 * [#8281](https://github.com/openlayers/openlayers/pull/8281) - Rename default data projection ([@gberaudo](https://github.com/gberaudo))
 * [#8283](https://github.com/openlayers/openlayers/pull/8283) - Remove references to tutorials/custom-builds.html ([@fredj](https://github.com/fredj))
 * [#8280](https://github.com/openlayers/openlayers/pull/8280) - Add missing property in VectorTile options, remove params from createRenderer ([@fredj](https://github.com/fredj))
 * [#8277](https://github.com/openlayers/openlayers/pull/8277) - Fix code block in comment ([@ahocevar](https://github.com/ahocevar))
 * [#8273](https://github.com/openlayers/openlayers/pull/8273) - Remove the externs/ directory ([@ahocevar](https://github.com/ahocevar))
 * [#8246](https://github.com/openlayers/openlayers/pull/8246) - Don't go back to not moving when dragging ([@ahocevar](https://github.com/ahocevar))
 * [#8258](https://github.com/openlayers/openlayers/pull/8258) - Use Webpack 4 for tests (and examples) ([@ahocevar](https://github.com/ahocevar))
 * [#8250](https://github.com/openlayers/openlayers/pull/8250) - Fix various jsdoc types ([@fredj](https://github.com/fredj))
 * [#8245](https://github.com/openlayers/openlayers/pull/8245) - Fix vector tile docs and types ([@ahocevar](https://github.com/ahocevar))
 * [#8233](https://github.com/openlayers/openlayers/pull/8233) - Use rollup instead of webpack for the full build ([@ahocevar](https://github.com/ahocevar))
 * [#8231](https://github.com/openlayers/openlayers/pull/8231) - Fix JSTS usage with OL classes injected ([@bjornharrtell](https://github.com/bjornharrtell))
 * [#8229](https://github.com/openlayers/openlayers/pull/8229) - Support layers instead of renderers for ol/source/Raster ([@ahocevar](https://github.com/ahocevar))
 * [#8227](https://github.com/openlayers/openlayers/pull/8227) - Better vectortile experience ([@ahocevar](https://github.com/ahocevar))
 * [#8225](https://github.com/openlayers/openlayers/pull/8225) - renderMode: 'image' fixes for ol/layer/Vector ([@ahocevar](https://github.com/ahocevar))
 * [#8226](https://github.com/openlayers/openlayers/pull/8226) - Accept renderers as sources for the Raster source ([@ahocevar](https://github.com/ahocevar))
 * [#8210](https://github.com/openlayers/openlayers/pull/8210) - Align patterns and gradients to a grid ([@ahocevar](https://github.com/ahocevar))
 * [#8201](https://github.com/openlayers/openlayers/pull/8201) - Remove babelrc, transform using babel-core ([@ahocevar](https://github.com/ahocevar))
 * [#8203](https://github.com/openlayers/openlayers/pull/8203) - Type checking complete ([@ahocevar](https://github.com/ahocevar))
 * [#8204](https://github.com/openlayers/openlayers/pull/8204) - Document ol/control~defaults in the right place ([@ahocevar](https://github.com/ahocevar))
 * [#8200](https://github.com/openlayers/openlayers/pull/8200) - Set textAlign to the canvas context ([@fredj](https://github.com/fredj))
 * [#8194](https://github.com/openlayers/openlayers/pull/8194) - miscellaneous code cleanup ([@fredj](https://github.com/fredj))
 * [#8189](https://github.com/openlayers/openlayers/pull/8189) - More apidoc fixes ([@ahocevar](https://github.com/ahocevar))
 * [#8191](https://github.com/openlayers/openlayers/pull/8191) - Do not scale text outline width ([@ahocevar](https://github.com/ahocevar))
 * [#8187](https://github.com/openlayers/openlayers/pull/8187) - Hide attribution control if there are no attributions ([@tschaub](https://github.com/tschaub))
 * [#8186](https://github.com/openlayers/openlayers/pull/8186) - Updates to get the examples building ([@tschaub](https://github.com/tschaub))
 * [#8180](https://github.com/openlayers/openlayers/pull/8180) - Fix some JSDoc issues ([@ahocevar](https://github.com/ahocevar))
 * [#8175](https://github.com/openlayers/openlayers/pull/8175) - More typing fixes ([@fredj](https://github.com/fredj))
 * [#8177](https://github.com/openlayers/openlayers/pull/8177) - Remove documentation related to the closure compiler ([@gberaudo](https://github.com/gberaudo))
 * [#8174](https://github.com/openlayers/openlayers/pull/8174) - Fix typo in ol/style/Image~Options typedef ([@fredj](https://github.com/fredj))
 * [#8172](https://github.com/openlayers/openlayers/pull/8172) - Ignore 'this' type in function typedef ([@fredj](https://github.com/fredj))
 * [#8173](https://github.com/openlayers/openlayers/pull/8173) - Fix typing in ol/xml module ([@fredj](https://github.com/fredj))
 * [#8169](https://github.com/openlayers/openlayers/pull/8169) - Shorter module paths for default exports ([@fredj](https://github.com/fredj))
 * [#8171](https://github.com/openlayers/openlayers/pull/8171) - Remove wrong override tags ([@fredj](https://github.com/fredj))
 * [#8170](https://github.com/openlayers/openlayers/pull/8170) - Basic apidocs ([@ahocevar](https://github.com/ahocevar))
 * [#8167](https://github.com/openlayers/openlayers/pull/8167) - Remove externs/oli.js ([@tschaub](https://github.com/tschaub))
 * [#8166](https://github.com/openlayers/openlayers/pull/8166) - Updated type annotations ([@tschaub](https://github.com/tschaub))
 * [#8165](https://github.com/openlayers/openlayers/pull/8165) - Update linter config ([@tschaub](https://github.com/tschaub))
 * [#8163](https://github.com/openlayers/openlayers/pull/8163) - Fix type syntax ([@ahocevar](https://github.com/ahocevar))
 * [#8161](https://github.com/openlayers/openlayers/pull/8161) - Fix skipping of texts with background ([@ahocevar](https://github.com/ahocevar))
 * [#8156](https://github.com/openlayers/openlayers/pull/8156) -  fix assert 36 on error documentation #8147 ([@mainakae](https://github.com/mainakae))
 * [#8143](https://github.com/openlayers/openlayers/pull/8143) - Shorter module paths for default exports ([@openlayers](https://github.com/openlayers))
 * [#8137](https://github.com/openlayers/openlayers/pull/8137) - Avoid `this` at top level of modules ([@tschaub](https://github.com/tschaub))
 * [#8138](https://github.com/openlayers/openlayers/pull/8138) - Update tasks to use native async/await ([@tschaub](https://github.com/tschaub))
 * [#8139](https://github.com/openlayers/openlayers/pull/8139) - Add @module annotation for ol/control/util ([@tschaub](https://github.com/tschaub))
 * [#8135](https://github.com/openlayers/openlayers/pull/8135) - Break the circular dependency: overview map, map, controls, overview map ([@tschaub](https://github.com/tschaub))
 * [#8132](https://github.com/openlayers/openlayers/pull/8132) - Add minZoom to the createResolutionConstraint function return type ([@fredj](https://github.com/fredj))
 * [#8129](https://github.com/openlayers/openlayers/pull/8129) - Fix some types ([@ahocevar](https://github.com/ahocevar))
 * [#8127](https://github.com/openlayers/openlayers/pull/8127) - Move RasterOperationType into ol/source/Raster ([@fredj](https://github.com/fredj))
 * [#8123](https://github.com/openlayers/openlayers/pull/8123) - More Jsdoc module fixes ([@fredj](https://github.com/fredj))
 * [#8125](https://github.com/openlayers/openlayers/pull/8125) - Shorter module paths for default exports ([@ahocevar](https://github.com/ahocevar))
 * [#8122](https://github.com/openlayers/openlayers/pull/8122) - Remove unused 'format' options from Cluster source constructor ([@fredj](https://github.com/fredj))
 * [#8121](https://github.com/openlayers/openlayers/pull/8121) - More Jsdoc module fixes ([@fredj](https://github.com/fredj))
 * [#8117](https://github.com/openlayers/openlayers/pull/8117) - Re-export style constructors from ol/style ([@tschaub](https://github.com/tschaub))
 * [#8115](https://github.com/openlayers/openlayers/pull/8115) - Re-export layer constructors from ol/layer ([@tschaub](https://github.com/tschaub))
 * [#8114](https://github.com/openlayers/openlayers/pull/8114) - Re-export interaction constructors from ol/interaction ([@tschaub](https://github.com/tschaub))
 * [#8116](https://github.com/openlayers/openlayers/pull/8116) - Re-export source constructors from ol/source ([@tschaub](https://github.com/tschaub))
 * [#8113](https://github.com/openlayers/openlayers/pull/8113) - Re-export geometry constructors from ol/geom ([@tschaub](https://github.com/tschaub))
 * [#8112](https://github.com/openlayers/openlayers/pull/8112) - Re-export format constructors from ol/format ([@tschaub](https://github.com/tschaub))
 * [#8111](https://github.com/openlayers/openlayers/pull/8111) - Re-export control constructors ([@tschaub](https://github.com/tschaub))
 * [#8118](https://github.com/openlayers/openlayers/pull/8118) - Use list for cloak entries ([@tschaub](https://github.com/tschaub))
 * [#8109](https://github.com/openlayers/openlayers/pull/8109) - Make any falsey undefinedHTML retain the mouse position ([@tschaub](https://github.com/tschaub))
 * [#8103](https://github.com/openlayers/openlayers/pull/8103) - Fix some jsdoc path types ([@ahocevar](https://github.com/ahocevar))
 * [#7784](https://github.com/openlayers/openlayers/pull/7784) - Adds option to retain MousePosition when pointer leaves viewport ([@pedros007](https://github.com/pedros007))
 * [#8101](https://github.com/openlayers/openlayers/pull/8101) - More Jsdoc module fixes ([@fredj](https://github.com/fredj))
 * [#7908](https://github.com/openlayers/openlayers/pull/7908) - ScaleLine fix for non-degree based units ([@hyperknot](https://github.com/hyperknot))
 * [#8099](https://github.com/openlayers/openlayers/pull/8099) - More Jsdoc module fixes ([@openlayers](https://github.com/openlayers))
 * [#8098](https://github.com/openlayers/openlayers/pull/8098) - Remove externs/olx.js and src/ol/typedefs.js ([@tschaub](https://github.com/tschaub))
 * [#8096](https://github.com/openlayers/openlayers/pull/8096) - Fix rotation default ([@ahocevar](https://github.com/ahocevar))
 * [#8081](https://github.com/openlayers/openlayers/pull/8081) - Move olx.render.* and related typedefs to ol/render/* ([@pfirpfel](https://github.com/pfirpfel))
 * [#8087](https://github.com/openlayers/openlayers/pull/8087) - Move various typedefs out of typdefs.js ([@pfirpfel](https://github.com/pfirpfel))
 * [#8052](https://github.com/openlayers/openlayers/pull/8052) - More Jsdoc module fixes ([@fredj](https://github.com/fredj))
 * [#8048](https://github.com/openlayers/openlayers/pull/8048) - Remove all comments from shaders ([@fredj](https://github.com/fredj))
 * [#8089](https://github.com/openlayers/openlayers/pull/8089) - Allow unprefixed WMTS tileMatrix identifiers ([@fredj](https://github.com/fredj))
 * [#8088](https://github.com/openlayers/openlayers/pull/8088) - Correct type for TextPlacement (jsdoc) ([@camptocamp](https://github.com/camptocamp))
 * [#8083](https://github.com/openlayers/openlayers/pull/8083) - Move structs typedefs to ol/structs/* ([@pfirpfel](https://github.com/pfirpfel))
 * [#8077](https://github.com/openlayers/openlayers/pull/8077) - Add hasFeature method to ol.source.Vector ([@icholy](https://github.com/icholy))
 * [#8080](https://github.com/openlayers/openlayers/pull/8080) - #8079 Mark functions setRotateWithView and setSnapToPixel of ol/style/Image with @api ([@iagafonov](https://github.com/iagafonov))
 * [#8084](https://github.com/openlayers/openlayers/pull/8084) - Move ol.Reproj* typedefs to ol/reproj ([@pfirpfel](https://github.com/pfirpfel))
 * [#8082](https://github.com/openlayers/openlayers/pull/8082) - Remove unused typedefs ([@pfirpfel](https://github.com/pfirpfel))
 * [#8076](https://github.com/openlayers/openlayers/pull/8076) - Move olx.style.* and ol.Style* to ol/style/* ([@pfirpfel](https://github.com/pfirpfel))
 * [#8075](https://github.com/openlayers/openlayers/pull/8075) - Move olx.tilegrid.* to ol/tilegrid/* ([@pfirpfel](https://github.com/pfirpfel))
 * [#8073](https://github.com/openlayers/openlayers/pull/8073) - Remove dependency on deprecated sinon@5 ([@tschaub](https://github.com/tschaub))
 * [#7994](https://github.com/openlayers/openlayers/pull/7994) - Explicit renderer dependencies ([@tschaub](https://github.com/tschaub))
 * [#7863](https://github.com/openlayers/openlayers/pull/7863) - Add note about null geom in VectorSource.forEachFeature ([@fredj](https://github.com/fredj))
 * [#8055](https://github.com/openlayers/openlayers/pull/8055) - Mark optional property with square brackets ([@fredj](https://github.com/fredj))
 * [#8054](https://github.com/openlayers/openlayers/pull/8054) - Generate correct includes list for `build-index` script on windows hosts. ([@lilliputten](https://github.com/lilliputten))
 * [#8053](https://github.com/openlayers/openlayers/pull/8053) - Don't import from 'functions' multiple times ([@fredj](https://github.com/fredj))
 * [#8041](https://github.com/openlayers/openlayers/pull/8041) - Remove olx typedefs ol/source ([@fgravin](https://github.com/fgravin))
 * [#8026](https://github.com/openlayers/openlayers/pull/8026) - [doc] dispatchChangeEvent -> changed ([@nyurik](https://github.com/nyurik))
 * [#8003](https://github.com/openlayers/openlayers/pull/8003) - Add an example showing Chaikins smoothing algorithm ([@marcjansen](https://github.com/marcjansen))
 * [#8044](https://github.com/openlayers/openlayers/pull/8044) - Rename constructor options typedef to 'Options' ([@fredj](https://github.com/fredj))
 * [#8042](https://github.com/openlayers/openlayers/pull/8042) - Only process loaded tiles ([@ahocevar](https://github.com/ahocevar))
 * [#8036](https://github.com/openlayers/openlayers/pull/8036) - Move the temporary arrays to the module level ([@fredj](https://github.com/fredj))
 * [#8032](https://github.com/openlayers/openlayers/pull/8032) - Start removing externs for `ol/source` ([@romanzoller](https://github.com/romanzoller))
 * [#8038](https://github.com/openlayers/openlayers/pull/8038) - Fix ZoomToExtent default tipLabel value ([@fredj](https://github.com/fredj))
 * [#8034](https://github.com/openlayers/openlayers/pull/8034) - More Jsdoc module fixes ([@fredj](https://github.com/fredj))
 * [#8029](https://github.com/openlayers/openlayers/pull/8029) - More Jsdoc module fixes ([@fredj](https://github.com/fredj))
 * [#8030](https://github.com/openlayers/openlayers/pull/8030) - Set `user-select: text` for .ol-selectable ([@walkermatt](https://github.com/walkermatt))
 * [#8025](https://github.com/openlayers/openlayers/pull/8025) - Move olx.layer.* to ol/layer/* ([@romanzoller](https://github.com/romanzoller))
 * [#8023](https://github.com/openlayers/openlayers/pull/8023) -  Module type changes for ol.webgl ([@fredj](https://github.com/fredj))
 * [#8021](https://github.com/openlayers/openlayers/pull/8021) - Move olx.ViewState to ol/View ([@fredj](https://github.com/fredj))
 * [#8012](https://github.com/openlayers/openlayers/pull/8012) - More Jsdoc module fixes ([@fredj](https://github.com/fredj))
 * [#8015](https://github.com/openlayers/openlayers/pull/8015) - Remove some typedefs from src/ol/typedefs.js ([@fredj](https://github.com/fredj))
 * [#8016](https://github.com/openlayers/openlayers/pull/8016) - Move olx.layer.BaseOptions to ol/layer/Base ([@romanzoller](https://github.com/romanzoller))
 * [#8005](https://github.com/openlayers/openlayers/pull/8005) - Add vector tile selection example ([@chrismayer](https://github.com/chrismayer))
 * [#7999](https://github.com/openlayers/openlayers/pull/7999) - More Jsdoc module fixes ([@fredj](https://github.com/fredj))
 * [#7998](https://github.com/openlayers/openlayers/pull/7998) - More Jsdoc module fixes ([@fredj](https://github.com/fredj))
 * [#7992](https://github.com/openlayers/openlayers/pull/7992) - Fix freeze on export PDF example ([@trenneman](https://github.com/trenneman))
 * [#7997](https://github.com/openlayers/openlayers/pull/7997) - Jsdoc module ([@fredj](https://github.com/fredj))
 * [#7986](https://github.com/openlayers/openlayers/pull/7986) - Use module syntax for type annotations ([@fredj](https://github.com/fredj))
 * [#7996](https://github.com/openlayers/openlayers/pull/7996) - 4.6.5 ([@openlayers](https://github.com/openlayers))
 * [#7993](https://github.com/openlayers/openlayers/pull/7993) - Module types ([@tschaub](https://github.com/tschaub))
 * [#7989](https://github.com/openlayers/openlayers/pull/7989) - Redistribute olx types for ol/interaction ([@ahocevar](https://github.com/ahocevar))
 * [#7991](https://github.com/openlayers/openlayers/pull/7991) - Fix some type paths ([@ahocevar](https://github.com/ahocevar))
 * [#7985](https://github.com/openlayers/openlayers/pull/7985) - Use path type instead of ol.tilegrid ([@fredj](https://github.com/fredj))
 * [#7983](https://github.com/openlayers/openlayers/pull/7983) - Use path type instead of ol.Extent ([@ahocevar](https://github.com/ahocevar))
 * [#7980](https://github.com/openlayers/openlayers/pull/7980) - Use module syntax for type annotations ([@fredj](https://github.com/fredj))
 * [#7979](https://github.com/openlayers/openlayers/pull/7979) - Mark optional property with square brackets ([@fredj](https://github.com/fredj))
 * [#7975](https://github.com/openlayers/openlayers/pull/7975) - Mark optional property with square brackets ([@fredj](https://github.com/fredj))
 * [#7960](https://github.com/openlayers/openlayers/pull/7960) - Fix memory leak in CanvasImageRenderer ([@PetroDE](https://github.com/PetroDE))
 * [#7974](https://github.com/openlayers/openlayers/pull/7974) - Fix spelling ([@fredj](https://github.com/fredj))
 * [#7973](https://github.com/openlayers/openlayers/pull/7973) - Remove setAttributeNS function from ol/xml ([@fredj](https://github.com/fredj))
 * [#7971](https://github.com/openlayers/openlayers/pull/7971) - Use module syntax for type annotations ([@tschaub](https://github.com/tschaub))
 * [#7968](https://github.com/openlayers/openlayers/pull/7968) - More types in the right places ([@tschaub](https://github.com/tschaub))
 * [#7966](https://github.com/openlayers/openlayers/pull/7966) - Remove namespaces from olx.js ([@tschaub](https://github.com/tschaub))
 * [#7965](https://github.com/openlayers/openlayers/pull/7965) - Redistribute types from externs/olx.js ([@tschaub](https://github.com/tschaub))
 * [#7608](https://github.com/openlayers/openlayers/pull/7608) - Facilitate the removal of externs/olx.js ([@tschaub](https://github.com/tschaub))
 * [#7964](https://github.com/openlayers/openlayers/pull/7964) - Coordinate type typo ([@tschaub](https://github.com/tschaub))
 * [#7961](https://github.com/openlayers/openlayers/pull/7961) - Correct type for style ([@tschaub](https://github.com/tschaub))
 * [#7958](https://github.com/openlayers/openlayers/pull/7958) - miscellaneous fixes to the types annotation ([@fredj](https://github.com/fredj))
 * [#7953](https://github.com/openlayers/openlayers/pull/7953) - Get rid of externs/olx.js and src/ol/typedefs.js typedefs for ol/ and ol/events/ ([@ahocevar](https://github.com/ahocevar))
 * [#7956](https://github.com/openlayers/openlayers/pull/7956) - Make sure namespaces are populated after they were created ([@ahocevar](https://github.com/ahocevar))
 * [#7952](https://github.com/openlayers/openlayers/pull/7952) - Fix typo in jsdoc type ([@fredj](https://github.com/fredj))
 * [#7948](https://github.com/openlayers/openlayers/pull/7948) - Better type definition ([@fredj](https://github.com/fredj))
 * [#7946](https://github.com/openlayers/openlayers/pull/7946) - New typedef syntax and module path types for some modules ([@ahocevar](https://github.com/ahocevar))
 * [#7944](https://github.com/openlayers/openlayers/pull/7944) - Make wrap work with points in the vector-labels example ([@ahocevar](https://github.com/ahocevar))
 * [#7933](https://github.com/openlayers/openlayers/pull/7933) - Better code indentation ([@fredj](https://github.com/fredj))
 * [#7930](https://github.com/openlayers/openlayers/pull/7930) - Use listen function instead of on ([@openlayers](https://github.com/openlayers))
 * [#7929](https://github.com/openlayers/openlayers/pull/7929) - Remove unused devDependencies ([@openlayers](https://github.com/openlayers))
 * [#7927](https://github.com/openlayers/openlayers/pull/7927) - Remove unused deps, unused tasks, and outdated info ([@tschaub](https://github.com/tschaub))
 * [#7923](https://github.com/openlayers/openlayers/pull/7923) - Fix exports and use of constants ([@ahocevar](https://github.com/ahocevar))
 * [#7914](https://github.com/openlayers/openlayers/pull/7914) - Some easy window cleanups ([@gberaudo](https://github.com/gberaudo))
 * [#7915](https://github.com/openlayers/openlayers/pull/7915) - Untie feature loader and XML code ([@gberaudo](https://github.com/gberaudo))
 * [#7918](https://github.com/openlayers/openlayers/pull/7918) - Move the nullFunction to ol/functions ([@fredj](https://github.com/fredj))
 * [#7910](https://github.com/openlayers/openlayers/pull/7910) - Better code indentation ([@fredj](https://github.com/fredj))
 * [#7904](https://github.com/openlayers/openlayers/pull/7904) - Add focus condition, e.g. for wheel zoom ([@ahocevar](https://github.com/ahocevar))
 * [#7902](https://github.com/openlayers/openlayers/pull/7902) - Add a unicode symbol for symbol font detection ([@ahocevar](https://github.com/ahocevar))
 * [#7899](https://github.com/openlayers/openlayers/pull/7899) - Use nomnom@1.8.1 ([@tschaub](https://github.com/tschaub))
 * [#7897](https://github.com/openlayers/openlayers/pull/7897) - Better variables scoping ([@fredj](https://github.com/fredj))
 * [#7894](https://github.com/openlayers/openlayers/pull/7894) - More named exports, fewer static members ([@tschaub](https://github.com/tschaub))
 * [#7881](https://github.com/openlayers/openlayers/pull/7881) - Better variables scoping ([@fredj](https://github.com/fredj))
 * [#7888](https://github.com/openlayers/openlayers/pull/7888) - Rework away static members from pointer related classes ([@bjornharrtell](https://github.com/bjornharrtell))
 * [#7880](https://github.com/openlayers/openlayers/pull/7880) - Named exports from ol/style/Style ([@tschaub](https://github.com/tschaub))
 * [#7872](https://github.com/openlayers/openlayers/pull/7872) - More named export ([@fredj](https://github.com/fredj))
 * [#7874](https://github.com/openlayers/openlayers/pull/7874) - Cleaning up exports ([@tschaub](https://github.com/tschaub))
 * [#7871](https://github.com/openlayers/openlayers/pull/7871) - Memory optimization ([@fredj](https://github.com/fredj))
 * [#7870](https://github.com/openlayers/openlayers/pull/7870) - Use exported function from ol/webgl ([@fredj](https://github.com/fredj))
 * [#7868](https://github.com/openlayers/openlayers/pull/7868) - Named export ([@fredj](https://github.com/fredj))
 * [#7861](https://github.com/openlayers/openlayers/pull/7861) - Named exports from ol/structs/PriorityQueue ([@fredj](https://github.com/fredj))
 * [#7860](https://github.com/openlayers/openlayers/pull/7860) - Don't export private constant from ol/webgl ([@openlayers](https://github.com/openlayers))
 * [#7855](https://github.com/openlayers/openlayers/pull/7855) - More named exports ([@tschaub](https://github.com/tschaub))
 * [#7851](https://github.com/openlayers/openlayers/pull/7851) - Replacing static functions with named exports ([@tschaub](https://github.com/tschaub))
 * [#7852](https://github.com/openlayers/openlayers/pull/7852) - Rename extent ([@nicholas-l](https://github.com/nicholas-l))
 * [#7840](https://github.com/openlayers/openlayers/pull/7840) - More export renaming ([@raiyni](https://github.com/raiyni))
 * [#7843](https://github.com/openlayers/openlayers/pull/7843) - Better code indentation ([@fredj](https://github.com/fredj))
 * [#7842](https://github.com/openlayers/openlayers/pull/7842) - Allow setting the label cache size ([@ahocevar](https://github.com/ahocevar))
 * [#7791](https://github.com/openlayers/openlayers/pull/7791) - Named exports from ol/tilecoord ([@marcjansen](https://github.com/marcjansen))
 * [#7835](https://github.com/openlayers/openlayers/pull/7835) - Add and use a constant for the collapsed CSS class ([@marcjansen](https://github.com/marcjansen))
 * [#7834](https://github.com/openlayers/openlayers/pull/7834) - Named exports from ol/transform ([@marcjansen](https://github.com/marcjansen))
 * [#7831](https://github.com/openlayers/openlayers/pull/7831) - Add new maxTilesLoading option to ol/PluggableMap ([@fredj](https://github.com/fredj))
 * [#7833](https://github.com/openlayers/openlayers/pull/7833) - Fix createEmptyTexture function call in ol/webgl/Context ([@fredj](https://github.com/fredj))
 * [#7832](https://github.com/openlayers/openlayers/pull/7832) - Webgl Layer and Context named exports ([@fgravin](https://github.com/fgravin))
 * [#7826](https://github.com/openlayers/openlayers/pull/7826) - Better code indentation and modules renaming ([@fredj](https://github.com/fredj))
 * [#7828](https://github.com/openlayers/openlayers/pull/7828) - Refactor constraints static methods to module functions ([@bjornharrtell](https://github.com/bjornharrtell))
 * [#7827](https://github.com/openlayers/openlayers/pull/7827) - Named exports from ol/render/canvas ([@tschaub](https://github.com/tschaub))
 * [#7823](https://github.com/openlayers/openlayers/pull/7823) - Replace static members with named exports ([@fredj](https://github.com/fredj))
 * [#7820](https://github.com/openlayers/openlayers/pull/7820) - ol/geom/flat/* exports ([@raiyni](https://github.com/raiyni))
 * [#7824](https://github.com/openlayers/openlayers/pull/7824) - Fix code indentation ([@fredj](https://github.com/fredj))
 * [#7821](https://github.com/openlayers/openlayers/pull/7821) - Add new canvas instruction array in ol/render/canvas/Instruction ([@fredj](https://github.com/fredj))
 * [#7817](https://github.com/openlayers/openlayers/pull/7817) - Remove more private static members ([@tschaub](https://github.com/tschaub))
 * [#7814](https://github.com/openlayers/openlayers/pull/7814) - Remove more private static members ([@tschaub](https://github.com/tschaub))
 * [#7816](https://github.com/openlayers/openlayers/pull/7816) - Use toString instead of asString in asColorLike ([@openlayers](https://github.com/openlayers))
 * [#7811](https://github.com/openlayers/openlayers/pull/7811) - Remove ol.FeatureStyleFunction support ([@fredj](https://github.com/fredj))
 * [#7815](https://github.com/openlayers/openlayers/pull/7815) -  Remove private statics from webgl related modules ([@bjornharrtell](https://github.com/bjornharrtell))
 * [#7803](https://github.com/openlayers/openlayers/pull/7803) - Remove private static members from constructors ([@tschaub](https://github.com/tschaub))
 * [#7808](https://github.com/openlayers/openlayers/pull/7808) - Make sure the requested image size matches the desired ratio ([@ahocevar](https://github.com/ahocevar))
 * [#7807](https://github.com/openlayers/openlayers/pull/7807) - Run typecheck after tests, because of its current verbosity ([@ahocevar](https://github.com/ahocevar))
 * [#7805](https://github.com/openlayers/openlayers/pull/7805) - Remove private static members from constructors ([@fredj](https://github.com/fredj))
 * [#7806](https://github.com/openlayers/openlayers/pull/7806) - Fix rotate import ([@ahocevar](https://github.com/ahocevar))
 * [#7800](https://github.com/openlayers/openlayers/pull/7800) - More named exports ([@tschaub](https://github.com/tschaub))
 * [#7741](https://github.com/openlayers/openlayers/pull/7741) - Type checking and full build creation ([@ahocevar](https://github.com/ahocevar))
 * [#7797](https://github.com/openlayers/openlayers/pull/7797) - Better default export names for WebGL modules ([@tschaub](https://github.com/tschaub))
 * [#7794](https://github.com/openlayers/openlayers/pull/7794) - Call animation callback in a timeout ([@tschaub](https://github.com/tschaub))
 * [#7793](https://github.com/openlayers/openlayers/pull/7793) - More renaming of default exports ([@tschaub](https://github.com/tschaub))
 * [#7799](https://github.com/openlayers/openlayers/pull/7799) - Use tmpSize to reduce garbage generation ([@fredj](https://github.com/fredj))
 * [#7788](https://github.com/openlayers/openlayers/pull/7788) - Fix shaders generation ([@fredj](https://github.com/fredj))
 * [#7789](https://github.com/openlayers/openlayers/pull/7789) - Use STATIC_DRAW as default WebGL buffer usage ([@fredj](https://github.com/fredj))
 * [#7787](https://github.com/openlayers/openlayers/pull/7787) - More renaming ([@fredj](https://github.com/fredj))
 * [#7782](https://github.com/openlayers/openlayers/pull/7782) - Named exports from specific proj modules ([@tschaub](https://github.com/tschaub))
 * [#7780](https://github.com/openlayers/openlayers/pull/7780) - More named exports ([@tschaub](https://github.com/tschaub))
 * [#7783](https://github.com/openlayers/openlayers/pull/7783) - Named exports from ol/xml ([@raiyni](https://github.com/raiyni))
 * [#7774](https://github.com/openlayers/openlayers/pull/7774) - Pass destination extent to avoid garbage generation ([@fredj](https://github.com/fredj))
 * [#7771](https://github.com/openlayers/openlayers/pull/7771) - Named exports from loadingstrategy ([@marcjansen](https://github.com/marcjansen))
 * [#7772](https://github.com/openlayers/openlayers/pull/7772) - Named exports from filter module ([@marcjansen](https://github.com/marcjansen))
 * [#7770](https://github.com/openlayers/openlayers/pull/7770) - Rename _ol_render_Box_ to RenderBox ([@marcjansen](https://github.com/marcjansen))
 * [#7763](https://github.com/openlayers/openlayers/pull/7763) - Use the temporary extent in VectorTileLayer ([@fredj](https://github.com/fredj))
 * [#7765](https://github.com/openlayers/openlayers/pull/7765) - Named exports from events condition ([@marcjansen](https://github.com/marcjansen))
 * [#7767](https://github.com/openlayers/openlayers/pull/7767) - Transformation functions: Fix spelling and use the symbol × ([@marcjansen](https://github.com/marcjansen))
 * [#7764](https://github.com/openlayers/openlayers/pull/7764) - Named exports from ol/has ([@tschaub](https://github.com/tschaub))
 * [#7761](https://github.com/openlayers/openlayers/pull/7761) - Rename _ol_render_canvas_Instruction_ to CanvasInstruction ([@fredj](https://github.com/fredj))
 * [#7749](https://github.com/openlayers/openlayers/pull/7749) - Check if the sketch feature is defined in finishDrawing ([@fredj](https://github.com/fredj))
 * [#7745](https://github.com/openlayers/openlayers/pull/7745) - More named exports ([@fredj](https://github.com/fredj))
 * [#7743](https://github.com/openlayers/openlayers/pull/7743) - Set enableHighAccuracy in geolocation example ([@fredj](https://github.com/fredj))
 * [#7738](https://github.com/openlayers/openlayers/pull/7738) - Named exports from ol/events ([@tschaub](https://github.com/tschaub))
 * [#7724](https://github.com/openlayers/openlayers/pull/7724) - Update prism library ([@fredj](https://github.com/fredj))
 * [#7727](https://github.com/openlayers/openlayers/pull/7727) - Deactivate transition effect for semi transparent layer example ([@fredj](https://github.com/fredj))
 * [#7726](https://github.com/openlayers/openlayers/pull/7726) - Named exports from ol/string ([@fredj](https://github.com/fredj))
 * [#7725](https://github.com/openlayers/openlayers/pull/7725) - Remove extra comment ([@fredj](https://github.com/fredj))
 * [#7721](https://github.com/openlayers/openlayers/pull/7721) - More renaming ([@fredj](https://github.com/fredj))
 * [#7716](https://github.com/openlayers/openlayers/pull/7716) - Better variables scoping ([@fredj](https://github.com/fredj))
 * [#7698](https://github.com/openlayers/openlayers/pull/7698) - Fix for loop in color test ([@fredj](https://github.com/fredj))
 * [#7703](https://github.com/openlayers/openlayers/pull/7703) - Improved drawing experience on touch devices ([@ahocevar](https://github.com/ahocevar))
 * [#7713](https://github.com/openlayers/openlayers/pull/7713) - Better variables scoping ([@fredj](https://github.com/fredj))
 * [#7711](https://github.com/openlayers/openlayers/pull/7711) - Do not use (I)CLA anymore ([@bartvde](https://github.com/bartvde))
 * [#7709](https://github.com/openlayers/openlayers/pull/7709) - More named export ([@fredj](https://github.com/fredj))
 * [#7705](https://github.com/openlayers/openlayers/pull/7705) - More renaming ([@fredj](https://github.com/fredj))
 * [#7700](https://github.com/openlayers/openlayers/pull/7700) - Fix ScaleLine control for Projections in degrees ([@pedros007](https://github.com/pedros007))
 * [#7697](https://github.com/openlayers/openlayers/pull/7697) - Use const and let ([@tschaub](https://github.com/tschaub))
 * [#7695](https://github.com/openlayers/openlayers/pull/7695) - More renaming ([@tschaub](https://github.com/tschaub))
 * [#7693](https://github.com/openlayers/openlayers/pull/7693) - More import renaming ([@fredj](https://github.com/fredj))
 * [#7678](https://github.com/openlayers/openlayers/pull/7678) - Add jsDelivr hits badge ([@LukasDrgon](https://github.com/LukasDrgon))
 * [#7686](https://github.com/openlayers/openlayers/pull/7686) - More import renaming ([@tschaub](https://github.com/tschaub))
 * [#7669](https://github.com/openlayers/openlayers/pull/7669) - Use declutter tree only for text and image replays ([@ahocevar](https://github.com/ahocevar))
 * [#7683](https://github.com/openlayers/openlayers/pull/7683) - Move IGCZ enum to src/ol/format/IGC.js ([@fredj](https://github.com/fredj))
 * [#7682](https://github.com/openlayers/openlayers/pull/7682) - Add new condition to MouseWheelZoom interaction ([@fredj](https://github.com/fredj))
 * [#7676](https://github.com/openlayers/openlayers/pull/7676) - Check fonts against two reference fonts ([@ahocevar](https://github.com/ahocevar))
 * [#7668](https://github.com/openlayers/openlayers/pull/7668) - Hit detect text background ([@ahocevar](https://github.com/ahocevar))
 * [#7661](https://github.com/openlayers/openlayers/pull/7661) - Background fill stroke ([@ahocevar](https://github.com/ahocevar))
 * [#7654](https://github.com/openlayers/openlayers/pull/7654) - Add named export for transformWithOptions function ([@fredj](https://github.com/fredj))
 * [#7652](https://github.com/openlayers/openlayers/pull/7652) - More renaming ([@fredj](https://github.com/fredj))
 * [#7649](https://github.com/openlayers/openlayers/pull/7649) - Named Map export for ol/Map ([@raiyni](https://github.com/raiyni))
 * [#7646](https://github.com/openlayers/openlayers/pull/7646) - Named export for Polyline format and SimpleGeometry ([@openlayers](https://github.com/openlayers))
 * [#7650](https://github.com/openlayers/openlayers/pull/7650) -  Rename _ol_PluggableMap_ to PluggableMap ([@fredj](https://github.com/fredj))
 * [#7642](https://github.com/openlayers/openlayers/pull/7642) - Don't store private function into GPX ([@fredj](https://github.com/fredj))
 * [#7643](https://github.com/openlayers/openlayers/pull/7643) - Replace ol/Sphere with ol/sphere ([@tschaub](https://github.com/tschaub))
 * [#7644](https://github.com/openlayers/openlayers/pull/7644) - Named exports from ol/math module ([@raiyni](https://github.com/raiyni))
 * [#7636](https://github.com/openlayers/openlayers/pull/7636) - Named exports from the ol/array module ([@marcjansen](https://github.com/marcjansen))
 * [#7614](https://github.com/openlayers/openlayers/pull/7614) - Remove opt_this from the API ([@KlausBenndorf](https://github.com/KlausBenndorf))
 * [#7640](https://github.com/openlayers/openlayers/pull/7640) - Remove unneeded and already skipped test ([@openlayers](https://github.com/openlayers))
 * [#7637](https://github.com/openlayers/openlayers/pull/7637) - Named exports from the ol/color module ([@marcjansen](https://github.com/marcjansen))
 * [#7639](https://github.com/openlayers/openlayers/pull/7639) - Don't store private function into GeoJSON ([@fredj](https://github.com/fredj))
 * [#7635](https://github.com/openlayers/openlayers/pull/7635) - More module renaming ([@fredj](https://github.com/fredj))
 * [#7632](https://github.com/openlayers/openlayers/pull/7632) - Named exports from ol/css module ([@marcjansen](https://github.com/marcjansen))
 * [#7634](https://github.com/openlayers/openlayers/pull/7634) - Rename _ol_layer_Tile_ to TileLayer ([@marcjansen](https://github.com/marcjansen))
 * [#7633](https://github.com/openlayers/openlayers/pull/7633) - Import assert function directly ([@marcjansen](https://github.com/marcjansen))
 * [#7631](https://github.com/openlayers/openlayers/pull/7631) - More named export and rename ([@fredj](https://github.com/fredj))
 * [#7627](https://github.com/openlayers/openlayers/pull/7627) - Nicer import name ([@fredj](https://github.com/fredj))
 * [#7622](https://github.com/openlayers/openlayers/pull/7622) - Named export ([@fredj](https://github.com/fredj))
 * [#7624](https://github.com/openlayers/openlayers/pull/7624) - Remove eslint-plugin-openlayers-internal rules ([@fredj](https://github.com/fredj))
 * [#7625](https://github.com/openlayers/openlayers/pull/7625) - Named export for ol/control and ol/interaction ([@fredj](https://github.com/fredj))
 * [#7623](https://github.com/openlayers/openlayers/pull/7623) - Named export for ol/tileurlfunction ([@fredj](https://github.com/fredj))
 * [#7621](https://github.com/openlayers/openlayers/pull/7621) - Replace ZeroClipboard with Clipboard.js ([@fredj](https://github.com/fredj))
 * [#7620](https://github.com/openlayers/openlayers/pull/7620) - Rename variables in format modules ([@tschaub](https://github.com/tschaub))
 * [#7619](https://github.com/openlayers/openlayers/pull/7619) - More variable renaming ([@fredj](https://github.com/fredj))
 * [#7612](https://github.com/openlayers/openlayers/pull/7612) - Better import name for _ol_ext_* import ([@fredj](https://github.com/fredj))
 * [#7611](https://github.com/openlayers/openlayers/pull/7611) - Change olx.SphereMetricOptions to SphereMetricOptions ([@fredj](https://github.com/fredj))
 * [#7615](https://github.com/openlayers/openlayers/pull/7615) - Remove logos ([@tschaub](https://github.com/tschaub))
 * [#7610](https://github.com/openlayers/openlayers/pull/7610) - More variable renaming ([@fredj](https://github.com/fredj))
 * [#7609](https://github.com/openlayers/openlayers/pull/7609) - Redistribute a few types and docs ([@tschaub](https://github.com/tschaub))
 * [#7598](https://github.com/openlayers/openlayers/pull/7598) - Modular proj4 integration ([@ahocevar](https://github.com/ahocevar))
 * [#7607](https://github.com/openlayers/openlayers/pull/7607) - Rename interaction variables ([@tschaub](https://github.com/tschaub))
 * [#7599](https://github.com/openlayers/openlayers/pull/7599) - Rename event variables ([@marcjansen](https://github.com/marcjansen))
 * [#7605](https://github.com/openlayers/openlayers/pull/7605) - Better import and class name in ol/format/filter/ ([@fredj](https://github.com/fredj))
 * [#7604](https://github.com/openlayers/openlayers/pull/7604) - Named exports from ol/functions ([@fredj](https://github.com/fredj))
 * [#7603](https://github.com/openlayers/openlayers/pull/7603) - Simplify default export ([@fredj](https://github.com/fredj))
 * [#7602](https://github.com/openlayers/openlayers/pull/7602) - Remove now unused DEVICE_ORIENTATION ([@fredj](https://github.com/fredj))
 * [#7597](https://github.com/openlayers/openlayers/pull/7597) - More es6 named export ([@fredj](https://github.com/fredj))
 * [#7600](https://github.com/openlayers/openlayers/pull/7600) - Simplify default export of ol/events/KeyCode ([@marcjansen](https://github.com/marcjansen))
 * [#7594](https://github.com/openlayers/openlayers/pull/7594) - Rename control variables ([@tschaub](https://github.com/tschaub))
 * [#7593](https://github.com/openlayers/openlayers/pull/7593) - Simplify default export ([@tschaub](https://github.com/tschaub))
 * [#7591](https://github.com/openlayers/openlayers/pull/7591) - Rename geometry variables ([@tschaub](https://github.com/tschaub))
 * [#7586](https://github.com/openlayers/openlayers/pull/7586) - Named exports from ol/dom ([@fredj](https://github.com/fredj))
 * [#7581](https://github.com/openlayers/openlayers/pull/7581) - Named exports from ol/proj ([@tschaub](https://github.com/tschaub))
 * [#7585](https://github.com/openlayers/openlayers/pull/7585) - Remove leftovers olx.animation typedefs ([@fredj](https://github.com/fredj))
 * [#7584](https://github.com/openlayers/openlayers/pull/7584) - Named exports from ol/easing ([@fredj](https://github.com/fredj))
 * [#7582](https://github.com/openlayers/openlayers/pull/7582) - Remove default export from ol/extent ([@ahocevar](https://github.com/ahocevar))
 * [#7580](https://github.com/openlayers/openlayers/pull/7580) - No default export from lowercase modules ([@tschaub](https://github.com/tschaub))
 * [#7575](https://github.com/openlayers/openlayers/pull/7575) - Add named exports for extent ([@ahocevar](https://github.com/ahocevar))
 * [#7579](https://github.com/openlayers/openlayers/pull/7579) - VectorImageTile: call changed() before removing listeners on dispose ([@mbleichner](https://github.com/mbleichner))
 * [#7578](https://github.com/openlayers/openlayers/pull/7578) - Remove deprecated functionality from 4.x ([@tschaub](https://github.com/tschaub))
 * [#7577](https://github.com/openlayers/openlayers/pull/7577) - Redistribute things in ol/index.js ([@tschaub](https://github.com/tschaub))
 * [#7576](https://github.com/openlayers/openlayers/pull/7576) - Named exports from ol/index.js ([@tschaub](https://github.com/tschaub))
 * [#7570](https://github.com/openlayers/openlayers/pull/7570) - Transform sources to ES modules ([@tschaub](https://github.com/tschaub))
 * [#7569](https://github.com/openlayers/openlayers/pull/7569) - Remove unnecessary require ([@tschaub](https://github.com/tschaub))
 * [#7568](https://github.com/openlayers/openlayers/pull/7568) - Remove unneeded configs from test ([@ahocevar](https://github.com/ahocevar))
 * [#7566](https://github.com/openlayers/openlayers/pull/7566) - Proper projection construction ([@tschaub](https://github.com/tschaub))
 * [#7565](https://github.com/openlayers/openlayers/pull/7565) - Release v4.6.4 ([@openlayers](https://github.com/openlayers))
 * [#7563](https://github.com/openlayers/openlayers/pull/7563) - Use full precision for the declutter box ([@ahocevar](https://github.com/ahocevar))
 * [#7559](https://github.com/openlayers/openlayers/pull/7559) - Handle skipping and unskipping features with renderMode: 'image' ([@ahocevar](https://github.com/ahocevar))
 * [#7562](https://github.com/openlayers/openlayers/pull/7562) - Remove unused, confusing projection code ([@ahocevar](https://github.com/ahocevar))
 * [#7555](https://github.com/openlayers/openlayers/pull/7555) - Release v4.6.3 ([@openlayers](https://github.com/openlayers))
 * [#7554](https://github.com/openlayers/openlayers/pull/7554) - Only compose image vector frame when the replay group has changed ([@ahocevar](https://github.com/ahocevar))
 * [#7550](https://github.com/openlayers/openlayers/pull/7550) - Update examples to d3.js v4 ([@fredj](https://github.com/fredj))
 * [#7548](https://github.com/openlayers/openlayers/pull/7548) - Fix pull request link ([@openlayers](https://github.com/openlayers))
 * [#7547](https://github.com/openlayers/openlayers/pull/7547) - Release v4.6.2 ([@openlayers](https://github.com/openlayers))
 * [#7546](https://github.com/openlayers/openlayers/pull/7546) - Do not request features for wrapped extent ([@ahocevar](https://github.com/ahocevar))
 * [#7545](https://github.com/openlayers/openlayers/pull/7545) - Release v4.6.1 ([@openlayers](https://github.com/openlayers))
 * [#7543](https://github.com/openlayers/openlayers/pull/7543) - Donut polygon labels do not get a chance to get rendered ([@ahocevar](https://github.com/ahocevar))
 * [#7542](https://github.com/openlayers/openlayers/pull/7542) - Still respect deprecated exceedLength option ([@ahocevar](https://github.com/ahocevar))
 * [#7541](https://github.com/openlayers/openlayers/pull/7541) - Fix case of vectorrendertype.js ([@ahocevar](https://github.com/ahocevar))


<details>
 <summary>Dependency Updates</summary>

 * [#8315](https://github.com/openlayers/openlayers/pull/8315) - Update eslint to the latest version 🚀 ([@openlayers](https://github.com/openlayers))
 * [#8309](https://github.com/openlayers/openlayers/pull/8309) - Update eslint to the latest version 🚀 ([@openlayers](https://github.com/openlayers))
 * [#8311](https://github.com/openlayers/openlayers/pull/8311) - Update webpack to the latest version 🚀 ([@openlayers](https://github.com/openlayers))
 * [#8310](https://github.com/openlayers/openlayers/pull/8310) - Update rollup to the latest version 🚀 ([@openlayers](https://github.com/openlayers))
 * [#8297](https://github.com/openlayers/openlayers/pull/8297) - Update rollup to the latest version 🚀 ([@openlayers](https://github.com/openlayers))
 * [#8293](https://github.com/openlayers/openlayers/pull/8293) - Update rollup to the latest version 🚀 ([@openlayers](https://github.com/openlayers))
 * [#8285](https://github.com/openlayers/openlayers/pull/8285) - Update google-closure-compiler to the latest version 🚀 ([@openlayers](https://github.com/openlayers))
 * [#8275](https://github.com/openlayers/openlayers/pull/8275) - Update serve to the latest version 🚀 ([@openlayers](https://github.com/openlayers))
 * [#8271](https://github.com/openlayers/openlayers/pull/8271) - chore(package): update google-closure-compiler to version 20180610.0.1 ([@openlayers](https://github.com/openlayers))
 * [#8270](https://github.com/openlayers/openlayers/pull/8270) - chore(package): update rollup to version 0.60.7 ([@openlayers](https://github.com/openlayers))
 * [#8266](https://github.com/openlayers/openlayers/pull/8266) - chore(package): update rollup to version 0.60.4 ([@openlayers](https://github.com/openlayers))
 * [#8261](https://github.com/openlayers/openlayers/pull/8261) - Update rollup to the latest version 🚀 ([@openlayers](https://github.com/openlayers))
 * [#8260](https://github.com/openlayers/openlayers/pull/8260) - Update sinon to the latest version 🚀 ([@openlayers](https://github.com/openlayers))
 * [#8252](https://github.com/openlayers/openlayers/pull/8252) - Update rollup to the latest version 🚀 ([@openlayers](https://github.com/openlayers))
 * [#8248](https://github.com/openlayers/openlayers/pull/8248) - Update serve to the latest version 🚀 ([@openlayers](https://github.com/openlayers))
 * [#8247](https://github.com/openlayers/openlayers/pull/8247) - Update rollup to the latest version 🚀 ([@openlayers](https://github.com/openlayers))
 * [#8241](https://github.com/openlayers/openlayers/pull/8241) - Update recast to the latest version 🚀 ([@openlayers](https://github.com/openlayers))
 * [#8230](https://github.com/openlayers/openlayers/pull/8230) - Update serve to the latest version 🚀 ([@openlayers](https://github.com/openlayers))
 * [#8208](https://github.com/openlayers/openlayers/pull/8208) - Update marked to the latest version 🚀 ([@openlayers](https://github.com/openlayers))
 * [#8205](https://github.com/openlayers/openlayers/pull/8205) - Update mocha to the latest version 🚀 ([@openlayers](https://github.com/openlayers))
 * [#8179](https://github.com/openlayers/openlayers/pull/8179) - Update google-closure-compiler to the latest version 🚀 ([@openlayers](https://github.com/openlayers))
 * [#8146](https://github.com/openlayers/openlayers/pull/8146) - Update coveralls to the latest version 🚀 ([@openlayers](https://github.com/openlayers))
 * [#8145](https://github.com/openlayers/openlayers/pull/8145) - Update fs-extra to the latest version 🚀 ([@openlayers](https://github.com/openlayers))
 * [#8140](https://github.com/openlayers/openlayers/pull/8140) - Update sinon to the latest version 🚀 ([@openlayers](https://github.com/openlayers))
 * [#8131](https://github.com/openlayers/openlayers/pull/8131) - Update babel-plugin-jsdoc-closure to the latest version 🚀 ([@openlayers](https://github.com/openlayers))
 * [#8130](https://github.com/openlayers/openlayers/pull/8130) - Update babel-plugin-jsdoc-closure to the latest version 🚀 ([@openlayers](https://github.com/openlayers))
 * [#8128](https://github.com/openlayers/openlayers/pull/8128) - Update babel-plugin-jsdoc-closure to the latest version 🚀 ([@openlayers](https://github.com/openlayers))
 * [#8124](https://github.com/openlayers/openlayers/pull/8124) - Update babel-plugin-jsdoc-closure to the latest version 🚀 ([@openlayers](https://github.com/openlayers))
 * [#8102](https://github.com/openlayers/openlayers/pull/8102) - Update babel-plugin-jsdoc-closure to the latest version 🚀 ([@openlayers](https://github.com/openlayers))
 * [#8095](https://github.com/openlayers/openlayers/pull/8095) - Update babel-plugin-jsdoc-closure to the latest version 🚀 ([@openlayers](https://github.com/openlayers))
 * [#8094](https://github.com/openlayers/openlayers/pull/8094) - Update babel-plugin-jsdoc-closure to the latest version 🚀 ([@openlayers](https://github.com/openlayers))
 * [#8093](https://github.com/openlayers/openlayers/pull/8093) - chore(package): update serve to version 6.5.5 ([@openlayers](https://github.com/openlayers))
 * [#8090](https://github.com/openlayers/openlayers/pull/8090) - chore(package): update url-polyfill to version 1.0.13 ([@openlayers](https://github.com/openlayers))
 * [#8091](https://github.com/openlayers/openlayers/pull/8091) - chore(package): update karma-firefox-launcher to version 1.1.0 ([@openlayers](https://github.com/openlayers))
 * [#8085](https://github.com/openlayers/openlayers/pull/8085) - Update mocha to the latest version 🚀 ([@openlayers](https://github.com/openlayers))
 * [#8071](https://github.com/openlayers/openlayers/pull/8071) - Update mocha to the latest version 🚀 ([@openlayers](https://github.com/openlayers))
 * [#8060](https://github.com/openlayers/openlayers/pull/8060) - Update google-closure-compiler to the latest version 🚀 ([@openlayers](https://github.com/openlayers))
 * [#8039](https://github.com/openlayers/openlayers/pull/8039) - Update google-closure-compiler to the latest version 🚀 ([@openlayers](https://github.com/openlayers))
 * [#8028](https://github.com/openlayers/openlayers/pull/8028) - Update marked to the latest version 🚀 ([@openlayers](https://github.com/openlayers))
 * [#8019](https://github.com/openlayers/openlayers/pull/8019) - Update mocha to the latest version 🚀 ([@openlayers](https://github.com/openlayers))
 * [#8017](https://github.com/openlayers/openlayers/pull/8017) - Update marked to the latest version 🚀 ([@openlayers](https://github.com/openlayers))
 * [#8018](https://github.com/openlayers/openlayers/pull/8018) - Update recast to the latest version 🚀 ([@openlayers](https://github.com/openlayers))
 * [#8014](https://github.com/openlayers/openlayers/pull/8014) - Update recast to the latest version 🚀 ([@openlayers](https://github.com/openlayers))
 * [#8009](https://github.com/openlayers/openlayers/pull/8009) - chore(package): update sinon to version 5.0.0 ([@openlayers](https://github.com/openlayers))
 * [#8008](https://github.com/openlayers/openlayers/pull/8008) - Update eslint to the latest version 🚀 ([@openlayers](https://github.com/openlayers))
 * [#8000](https://github.com/openlayers/openlayers/pull/8000) - Update karma-webpack to the latest version 🚀 ([@openlayers](https://github.com/openlayers))
 * [#7990](https://github.com/openlayers/openlayers/pull/7990) - Update eslint to the latest version 🚀 ([@openlayers](https://github.com/openlayers))
 * [#7984](https://github.com/openlayers/openlayers/pull/7984) - Update sinon to the latest version 🚀 ([@openlayers](https://github.com/openlayers))
 * [#7981](https://github.com/openlayers/openlayers/pull/7981) - Update sinon to the latest version 🚀 ([@openlayers](https://github.com/openlayers))
 * [#7977](https://github.com/openlayers/openlayers/pull/7977) - Update sinon to the latest version 🚀 ([@openlayers](https://github.com/openlayers))
 * [#7972](https://github.com/openlayers/openlayers/pull/7972) - Update sinon to the latest version 🚀 ([@openlayers](https://github.com/openlayers))
 * [#7963](https://github.com/openlayers/openlayers/pull/7963) - chore(package): update url-polyfill to version 1.0.11 ([@openlayers](https://github.com/openlayers))
 * [#7962](https://github.com/openlayers/openlayers/pull/7962) - chore(package): update serve to version 6.5.2 ([@openlayers](https://github.com/openlayers))
 * [#7959](https://github.com/openlayers/openlayers/pull/7959) - Update babel-plugin-jsdoc-closure to the latest version 🚀 ([@openlayers](https://github.com/openlayers))
 * [#7957](https://github.com/openlayers/openlayers/pull/7957) - Update recast to the latest version 🚀 ([@openlayers](https://github.com/openlayers))
 * [#7950](https://github.com/openlayers/openlayers/pull/7950) - Update mocha to the latest version 🚀 ([@openlayers](https://github.com/openlayers))
 * [#7945](https://github.com/openlayers/openlayers/pull/7945) - Update babel-plugin-jsdoc-closure to the latest version 🚀 ([@openlayers](https://github.com/openlayers))
 * [#7943](https://github.com/openlayers/openlayers/pull/7943) - Update mocha to the latest version 🚀 ([@openlayers](https://github.com/openlayers))
 * [#7940](https://github.com/openlayers/openlayers/pull/7940) - Update babel-plugin-jsdoc-closure to the latest version 🚀 ([@openlayers](https://github.com/openlayers))
 * [#7938](https://github.com/openlayers/openlayers/pull/7938) - Update mocha to the latest version 🚀 ([@openlayers](https://github.com/openlayers))
 * [#7925](https://github.com/openlayers/openlayers/pull/7925) - Update eslint to the latest version 🚀 ([@openlayers](https://github.com/openlayers))
 * [#7921](https://github.com/openlayers/openlayers/pull/7921) - Update clean-css-cli to the latest version 🚀 ([@openlayers](https://github.com/openlayers))
 * [#7916](https://github.com/openlayers/openlayers/pull/7916) - chore(package): update html-webpack-plugin to version 3.0.1 ([@openlayers](https://github.com/openlayers))
 * [#7905](https://github.com/openlayers/openlayers/pull/7905) - Update karma-webpack to the latest version 🚀 ([@openlayers](https://github.com/openlayers))
 * [#7903](https://github.com/openlayers/openlayers/pull/7903) - Update marked to the latest version 🚀 ([@openlayers](https://github.com/openlayers))
 * [#7901](https://github.com/openlayers/openlayers/pull/7901) - Update karma-webpack to the latest version 🚀 ([@openlayers](https://github.com/openlayers))
 * [#7898](https://github.com/openlayers/openlayers/pull/7898) - chore(package): update karma-webpack to version 2.0.11 ([@openlayers](https://github.com/openlayers))
 * [#7893](https://github.com/openlayers/openlayers/pull/7893) - Update sinon to the latest version 🚀 ([@openlayers](https://github.com/openlayers))
 * [#7891](https://github.com/openlayers/openlayers/pull/7891) - chore(package): update recast to version 0.14.4 ([@openlayers](https://github.com/openlayers))
 * [#7886](https://github.com/openlayers/openlayers/pull/7886) - Update sinon to the latest version 🚀 ([@openlayers](https://github.com/openlayers))
 * [#7882](https://github.com/openlayers/openlayers/pull/7882) - Update sinon to the latest version 🚀 ([@openlayers](https://github.com/openlayers))
 * [#7876](https://github.com/openlayers/openlayers/pull/7876) - Update recast to the latest version 🚀 ([@openlayers](https://github.com/openlayers))
 * [#7877](https://github.com/openlayers/openlayers/pull/7877) - Update webpack-merge to the latest version 🚀 ([@openlayers](https://github.com/openlayers))
 * [#7865](https://github.com/openlayers/openlayers/pull/7865) - Update marked to the latest version 🚀 ([@openlayers](https://github.com/openlayers))
 * [#7866](https://github.com/openlayers/openlayers/pull/7866) - Update eslint to the latest version 🚀 ([@openlayers](https://github.com/openlayers))
 * [#7867](https://github.com/openlayers/openlayers/pull/7867) - Update recast to the latest version 🚀 ([@openlayers](https://github.com/openlayers))
 * [#7858](https://github.com/openlayers/openlayers/pull/7858) - Update recast to the latest version 🚀 ([@openlayers](https://github.com/openlayers))
 * [#7850](https://github.com/openlayers/openlayers/pull/7850) - Update marked to the latest version 🚀 ([@openlayers](https://github.com/openlayers))
 * [#7849](https://github.com/openlayers/openlayers/pull/7849) - Update recast to the latest version 🚀 ([@openlayers](https://github.com/openlayers))
 * [#7848](https://github.com/openlayers/openlayers/pull/7848) - Update nomnom to the latest version 🚀 ([@openlayers](https://github.com/openlayers))
 * [#7847](https://github.com/openlayers/openlayers/pull/7847) - Update eslint to the latest version 🚀 ([@openlayers](https://github.com/openlayers))
 * [#7845](https://github.com/openlayers/openlayers/pull/7845) - chore(package): update marked to version 0.3.14 ([@openlayers](https://github.com/openlayers))
 * [#7841](https://github.com/openlayers/openlayers/pull/7841) - Greenkeeper/babel plugin jsdoc closure 1.2.0 ([@openlayers](https://github.com/openlayers))
 * [#7822](https://github.com/openlayers/openlayers/pull/7822) - Update mocha to the latest version 🚀 ([@openlayers](https://github.com/openlayers))
 * [#7813](https://github.com/openlayers/openlayers/pull/7813) - Update recast to the latest version 🚀 ([@openlayers](https://github.com/openlayers))
 * [#7809](https://github.com/openlayers/openlayers/pull/7809) - Update webpack to the latest version 🚀 ([@openlayers](https://github.com/openlayers))
 * [#7812](https://github.com/openlayers/openlayers/pull/7812) - chore(package): update sinon to version 4.3.0 ([@openlayers](https://github.com/openlayers))
 * [#7777](https://github.com/openlayers/openlayers/pull/7777) - Update babel-minify-webpack-plugin to the latest version 🚀 ([@openlayers](https://github.com/openlayers))
 * [#7759](https://github.com/openlayers/openlayers/pull/7759) - Update eslint to the latest version 🚀 ([@openlayers](https://github.com/openlayers))
 * [#7746](https://github.com/openlayers/openlayers/pull/7746) - Update sinon to the latest version 🚀 ([@openlayers](https://github.com/openlayers))
 * [#7735](https://github.com/openlayers/openlayers/pull/7735) - Update sinon to the latest version 🚀 ([@openlayers](https://github.com/openlayers))
 * [#7729](https://github.com/openlayers/openlayers/pull/7729) - Update jquery to the latest version 🚀 ([@openlayers](https://github.com/openlayers))
 * [#7732](https://github.com/openlayers/openlayers/pull/7732) - Update sinon to the latest version 🚀 ([@openlayers](https://github.com/openlayers))
 * [#7731](https://github.com/openlayers/openlayers/pull/7731) - Update webpack-dev-server to the latest version 🚀 ([@openlayers](https://github.com/openlayers))
 * [#7728](https://github.com/openlayers/openlayers/pull/7728) - Update eslint to the latest version 🚀 ([@openlayers](https://github.com/openlayers))
 * [#7720](https://github.com/openlayers/openlayers/pull/7720) - Update mocha to the latest version 🚀 ([@openlayers](https://github.com/openlayers))
 * [#7712](https://github.com/openlayers/openlayers/pull/7712) - Update sinon to the latest version 🚀 ([@openlayers](https://github.com/openlayers))
 * [#7702](https://github.com/openlayers/openlayers/pull/7702) - Update webpack-dev-server to the latest version 🚀 ([@openlayers](https://github.com/openlayers))
 * [#7701](https://github.com/openlayers/openlayers/pull/7701) - Update sinon to the latest version 🚀 ([@openlayers](https://github.com/openlayers))
 * [#7691](https://github.com/openlayers/openlayers/pull/7691) - Update webpack-dev-server to the latest version 🚀 ([@openlayers](https://github.com/openlayers))
 * [#7688](https://github.com/openlayers/openlayers/pull/7688) - Update marked to the latest version 🚀 ([@openlayers](https://github.com/openlayers))
 * [#7685](https://github.com/openlayers/openlayers/pull/7685) - Update sinon to the latest version 🚀 ([@openlayers](https://github.com/openlayers))
 * [#7681](https://github.com/openlayers/openlayers/pull/7681) - Update eslint to the latest version 🚀 ([@openlayers](https://github.com/openlayers))
 * [#7680](https://github.com/openlayers/openlayers/pull/7680) - Update webpack-dev-server to the latest version 🚀 ([@openlayers](https://github.com/openlayers))
 * [#7667](https://github.com/openlayers/openlayers/pull/7667) - Update mocha to the latest version 🚀 ([@openlayers](https://github.com/openlayers))
 * [#7656](https://github.com/openlayers/openlayers/pull/7656) - Update eslint to the latest version 🚀 ([@openlayers](https://github.com/openlayers))
 * [#7655](https://github.com/openlayers/openlayers/pull/7655) - Update marked to the latest version 🚀 ([@openlayers](https://github.com/openlayers))
 * [#7651](https://github.com/openlayers/openlayers/pull/7651) - Update karma to the latest version 🚀 ([@openlayers](https://github.com/openlayers))
 * [#7647](https://github.com/openlayers/openlayers/pull/7647) - Update rbush to the latest version 🚀 ([@openlayers](https://github.com/openlayers))
 * [#7596](https://github.com/openlayers/openlayers/pull/7596) - Update karma-webpack to the latest version 🚀 ([@openlayers](https://github.com/openlayers))
 * [#7589](https://github.com/openlayers/openlayers/pull/7589) - chore(package): update karma-webpack to version 2.0.8 ([@openlayers](https://github.com/openlayers))
 * [#7571](https://github.com/openlayers/openlayers/pull/7571) - Update eslint to the latest version 🚀 ([@openlayers](https://github.com/openlayers))
 * [#7567](https://github.com/openlayers/openlayers/pull/7567) - Update fs-extra to the latest version 🚀 ([@openlayers](https://github.com/openlayers))
 * [#7558](https://github.com/openlayers/openlayers/pull/7558) - Update eslint to the latest version 🚀 ([@openlayers](https://github.com/openlayers))
 * [#7552](https://github.com/openlayers/openlayers/pull/7552) - Update jscodeshift to the latest version 🚀 ([@openlayers](https://github.com/openlayers))
 * [#7551](https://github.com/openlayers/openlayers/pull/7551) - Update sinon to the latest version 🚀 ([@openlayers](https://github.com/openlayers))

</details>
