/*
 * Paper.js - The Swiss Army Knife of Vector Graphics Scripting.
 * http://paperjs.org/
 *
 * Copyright (c) 2011 - 2016, Juerg Lehni & Jonathan Puckey
 * http://scratchdisk.com/ & https://puckey.studio/
 *
 * Distributed under the MIT license. See LICENSE file for details.
 *
 * All rights reserved.
 */

QUnit.module('Path Boolean Operations');

function testOperations(path1, path2, results) {
    compareBoolean(function() { return path1.unite(path2); }, results[0]);
    compareBoolean(function() { return path2.unite(path1); }, results[0]);
    compareBoolean(function() { return path1.subtract(path2); }, results[1]);
    compareBoolean(function() { return path2.subtract(path1); }, results[2]);
    compareBoolean(function() { return path1.intersect(path2); }, results[3]);
    compareBoolean(function() { return path2.intersect(path1); }, results[3]);
    compareBoolean(function() { return path1.exclude(path2); }, results[4]);
    compareBoolean(function() { return path2.exclude(path1); }, results[4]);
}

test('Boolean operations without crossings', function() {
    var path1 = new Path.Rectangle({
        point: [0, 0],
        size: [200, 200]
    });

    var path2 = new Path.Rectangle({
        point: [50, 50],
        size: [100, 100]
    });

    var path3 = new Path.Rectangle({
        point: [250, 50],
        size: [100, 100]
    });

    testOperations(path1, path2, [
        'M0,200v-200h200v200z', // path1.unite(path2);
        'M0,200v-200h200v200zM150,150v-100h-100v100z', // path1.subtract(path2);
        '', // path2.subtract(path1);
        'M50,150v-100h100v100z', // path1.intersect(path2);
        'M0,200v-200h200v200zM150,150v-100h-100v100z' // path1.exclude(path2);
    ]);

    testOperations(path1, path3, [
        'M0,200v-200h200v200zM250,150v-100h100v100z', // path1.unite(path3);
        'M0,200v-200h200v200z', // path1.subtract(path3);
        'M350,150v-100h-100v100z', // path3.subtract(path1);
        '', // path1.intersect(path3);
        'M0,200v-200h200v200zM250,150v-100h100v100z' // path1.exclude(path3);
    ]);
});

test('frame.intersect(rect)', function() {
    var frame = new CompoundPath();
    frame.addChild(new Path.Rectangle(new Point(140, 10), [100, 300]));
    frame.addChild(new Path.Rectangle(new Point(150, 80), [50, 80]));
    var rect = new Path.Rectangle(new Point(50, 50), [100, 150]);

    compareBoolean(function() { return frame.intersect(rect); },
        'M140,50l10,0l0,150l-10,0z');
});

test('PathItem#resolveCrossings()', function() {
    var paths = [
        'M100,300l0,-50l50,-50l-50,0l150,0l-150,0l50,0l-50,0l100,0l-100,0l0,-100l200,0l0,200z',
        'M50,300l0,-150l50,25l0,-75l200,0l0,200z M100,200l50,0l-50,-25z',
        'M330.1,388.5l-65,65c0,0 -49.1,-14.5 -36.6,-36.6c12.5,-22.1 92.4,25.1 92.4,25.1c0,0 -33.3,-73.3 -23.2,-85.9c10,-12.8 32.4,32.4 32.4,32.4z',
        'M570,290l5.8176000300452415,33.58556812220928l-28.17314339506561,-14.439003967264455l31.189735425395614,-4.568209255479985c-5.7225406635552645e-9,-3.907138079739525e-8 -59.366611385062015,8.695139599513823 -59.366611385062015,8.695139599513823z',
        'M228.26666666666668,222.72h55.46666666666667c3.05499999999995,0 5.546666666666624,2.4916666666666742 5.546666666666624,5.546666666666681v55.46666666666667c0,3.05499999999995 -2.4916666666666742,5.546666666666624 -5.546666666666624,5.546666666666624h-55.46666666666667c-3.055000000000007,0 -5.546666666666681,-2.4916666666666742 -5.546666666666681,-5.546666666666624v-55.46666666666667c0,-3.055000000000007 2.4916666666666742,-5.546666666666681 5.546666666666681,-5.546666666666681zM283.73333399705655,289.2799999999998c-2.212411231994338e-7,1.1368683772161603e-13 2.212409526691772e-7,0 0,0z'
    ];
    var results = [
        'M100,300l0,-50l50,-50l-50,0l0,-100l200,0l0,200z',
        'M50,300l0,-150l50,25l0,-75l200,0l0,200z M100,200l50,0l-50,-25z',
        'M291.85631,426.74369l-26.75631,26.75631c0,0 -49.1,-14.5 -36.6,-36.6c7.48773,-13.23831 39.16013,-1.61018 63.35631,9.84369z M330.1,388.5l-22.09831,22.09831c-8.06306,-21.54667 -15.93643,-47.46883 -10.30169,-54.49831c10,-12.8 32.4,32.4 32.4,32.4z M320.9,442c0,0 -12.84682,-7.58911 -29.04369,-15.25631l16.14539,-16.14539c6.38959,17.07471 12.89831,31.40169 12.89831,31.40169z',
        'M570,290l5.8176,33.58557l-28.17314,-14.439c-14.32289,2.0978 -28.17688,4.12693 -28.17688,4.12693z',
        'M228.26666666666668,222.72h55.46666666666667c3.05499999999995,0 5.546666666666624,2.4916666666666742 5.546666666666624,5.546666666666681v55.46666666666667c0,3.05499999999995 -2.4916666666666742,5.546666666666624 -5.546666666666624,5.546666666666624h-55.46666666666667c-3.055000000000007,0 -5.546666666666681,-2.4916666666666742 -5.546666666666681,-5.546666666666624v-55.46666666666667c0,-3.055000000000007 2.4916666666666742,-5.546666666666681 5.546666666666681,-5.546666666666681z'
    ];
    for (var i = 0; i < paths.length; i++) {
        var path = PathItem.create(paths[i]),
            result = PathItem.create(results[i]);
        path.fillRule = 'evenodd';
        compareBoolean(path.resolveCrossings(), result, 'path.resolveCrossings(); // Test ' + (i + 1));
    }
});

test('#541', function() {
    var shape0 = new Path.Rectangle({
        insert: false,
        point: [304, 226],
        size: [328, 328]
    });
    var shape1 = new Path({
        insert: false,
        segments: [
            [213.5, 239],
            [476.5, 279],
            [716, 233.5],
            [469, 74]
        ],
        closed: true
    });
    var res1 = shape0.exclude(shape1);
    var shape2 = new Path.Rectangle({
        insert: false,
        point: [174, 128],
        size: [309, 251]
    });
    var res2 = res1.exclude(shape2);
    // shape3
    var shape3 = new Path.Rectangle({
        insert: false,
        point: [318, 148],
        size: [302, 302]
    });
    // exclude res2 & shape3
    compareBoolean(function() { return res2.exclude(shape3); },
        'M304,554l0,-175l14,0l0,71l302,0l0,-198.262l12,-2.27975l0,304.54175z M318,379l165,0l0,-101.23486l137,-26.02714l0,-25.738l-137,0l0,-78l-128.58788,0l-36.41212,23.51468l0,54.48532l165,0l0,51.76514l-6.5,1.23486l-158.5,-24.10646z M174,379l0,-251l211.38182,0l-30.9697,20l-36.41212,0l0,23.51468l-104.5,67.48532l90.5,13.76426l0,-26.76426l14,0l0,28.89354l-14,-2.12928l0,126.23574z M385.38182,128l83.61818,-54l114.59561,74l-100.59561,0l0,-20z M583.59561,148l36.40439,23.5081l0,-23.5081z M620,171.5081l96,61.9919l-84,15.95825l0,-23.45825l-12,0z');
});

test('#609', function() {
    // path1 and path2 are half circles, applying unite should result in a circle
    var path1 = new Path();
    path1.moveTo(new Point(100, 100));
    path1.arcTo(new Point(100, 200));
    path1.closePath();

    var path2 = new Path();
    path2.moveTo(new Point(100, 200));
    path2.arcTo(new Point(100, 100));
    path2.closePath();

    compareBoolean(function() { return path1.unite(path2); },
        'M150,150c0,27.61424 -22.38576,50 -50,50c-27.61424,0 -50,-22.38576 -50,-50c0,-27.61424 22.38576,-50 50,-50c27.61424,0 50,22.38576 50,50z');
});

test('#610', function() {
    var square = new Path.Rectangle({
        position: [140, 0],
        size: 300
    });

    // Make a ring using subtraction of two circles:
    var inner = new Path.Circle({
        center: [0, 0],
        radius: 100
    });

    var outer = new Path.Circle({
        center: [0, 0],
        radius: 132
    });

    var ring = outer.subtract(inner);

    compareBoolean(function() { return ring.subtract(square); },
        'M-132,0c0,-69.53737 53.7698,-126.51614 122,-131.62689l0,32.12064c-50.53323,5.01724 -90,47.65277 -90,99.50625c0,51.85348 39.46677,94.489 90,99.50625l0,32.12064c-68.2302,-5.11075 -122,-62.08951 -122,-131.62689z');
});

test('#719', function() {
    var radius = 50;
    var circle = new Path.Circle([0, 0], radius);
    var arc = new Path.Arc([0, -radius], [radius, 0], [0, radius]);
    arc.closed = true;
    arc.pivot = arc.bounds.leftCenter;

    var result = circle.subtract(arc);
    // Rotate the arc by 180 to receive the expected shape to compare against
    var expected = arc.rotate(180);

    compareBoolean(result, expected);
});

test('#757 (path1.intersect(pat2, { trace: false }))', function() {
    var rect = new Path.Rectangle({
        from: [100, 250],
        to: [350, 350]
    });

    var line = new Path({
        segments: [
            [100, 200],
            [150, 400],
            [200, 200],
            [250, 400],
            [300, 200],
            [350, 400]
        ]
    });

    var res = line.intersect(rect, { trace: false });

    var children = res.removeChildren();
    var first = children[0];
    for (var i = 1; i < children.length; i++) {
        first.join(children[i]);
    }
    first.insertAbove(res);
    res.remove();
    res = first;
    compareBoolean(res,
        'M112.5,250l25,100l25,0l25,-100l25,0l25,100l25,0l25,-100l25,0l25,100');
});

test('#784', function() {
    var path1 = PathItem.create('M495.9,1693.5c-42.2-203.5-64.5-304.9-78-299.9 c-1.7,0.6-0.3,6.7,5.3,22.5l209.4-74.8l75.8,303.9L495.9,1693.5z');
    var path2 = PathItem.create('M632.6,1341.2l-209.4,74.9c95.4,267,135.6,201-60.1-144.5l202.9-85.7 L632.6,1341.2z');
    compareBoolean(function() { return path1.unite(path2); },
        'M495.9,1693.5c-17.51923,-84.48253 -31.60874,-151.36838 -43.06274,-200.34989c-9.02339,-21.58227 -18.9863,-47.24083 -29.63726,-77.05011c-5.6,-15.8 -7,-21.9 -5.3,-22.5c3.68921,-1.36638 8.03561,5.21313 13.26571,19.65076l6.17555,-2.20892c0.00094,0.00191 0.00189,0.00383 0.00283,0.00574l195.25591,-69.74757l75.8,303.9z M632.6,1341.2l-195.25874,69.84183c-19.60056,-39.73292 -44.12819,-86.27851 -74.24126,-139.44183l202.9,-85.7z');

    var path1 = new Path('M330.1,388.5l-65,65c0,0-49.1-14.5-36.6-36.6 c12.5-22.1,92.4,25.1,92.4,25.1s-33.3-73.3-23.2-85.9C307.7,343.3,330.1,388.5,330.1,388.5z');
    var path2 = new Path('M395.1,453.4c0,15.2-33.8,65-65,65s-65-65-65-65l65-65 C330.1,388.5,395.1,438.2,395.1,453.4z');
    compareBoolean(function() { return path1.unite(path2); },
        'M265.13434,453.46566l-0.03434,0.03434c0,0 -49.1,-14.5 -36.6,-36.6c7.48073,-13.22593 39.10093,-1.6319 63.28843,9.81157l16.18604,-16.18604c-8.05354,-21.53223 -15.90287,-47.40397 -10.27447,-54.42553c9.77623,-12.51358 31.40373,30.40618 32.36674,32.33326l0.03326,-0.03326c0,0.1 65,49.8 65,65c0,15.2 -33.8,65 -65,65c-30.62393,0 -63.75273,-62.62185 -64.96566,-64.93434z');
});


test('#784#issuecomment-144653463', function() {
    var path1 = new Path({
        segments: [
            [400, 300],
            [396.4240965757225, 386.760212367686],
            [363.8902430603039, 336.3464406833805]
        ],
        closed: true
    });
    var path2 = new Path({
        segments: [
            [380.15716053320796, 361.5533174872367],
            [368.9579765078272, 389.3845783631412],
            [352.11749924000907, 372.22000125020173]
        ],
        closed: true
    });
    var path3 = new Path({
        segments: [
            [381.9248139754118, 360.88087710036456],
            [352.11749931845384, 372.22000145641056],
            [353.8723170322086, 346.9400510828104]
        ],
        closed: true
    });
    compareBoolean(function() { return path1.unite(path2).unite(path3); },
        'M400,300l-3.5759,86.76021l-16.26693,-25.2069l0,0l-11.19918,27.83126l-16.84048,-17.16458l1.75482,-25.27995l24.8115,12.3302l-14.79357,-22.92381z M352.1175,372.22z');
});

test('#784#issuecomment-144993215', function() {
    var path1 = new Path({
        segments: [
            [428.65986693122585, 123.24312916360232],
            [448.9732353341095, 290.23336023178985, -1.297313778199964, -0.24666929481787747, -0.06896642337790126, -0.004714867204086204],
            [448.9732339473677, 290.2333601369859, 0.22704183013848933, 0.04316939284507271, 0.24127512029406262, 0.016494695478172616],
            [375.34013306877415, 150.7568708363977]
        ],
        closed: true
    });
    compareBoolean(function() { return path1.unite(); },
        'M428.65987,123.24313c0,0 18.24445,159.97772 20.21157,166.76806c-3.05664,-6.18082 -73.53131,-139.25432 -73.53131,-139.25432z M448.97323,290.23336c0,0 0,0 0,0c0.22704,0.04317 -0.06896,-0.00471 0,0c-0.02659,-0.00506 -0.06063,-0.08007 -0.1018,-0.22217c0.07286,0.14733 0.10741,0.22256 0.1018,0.22217z',
        null, { tolerance: 1e-3 });
});

test('#784#issuecomment-168605018', function() {
    var path1 = new CompoundPath();
    path1.setChildren([
        new Path({
            segments: [
                [401.77542835664826, 286.9803609495646],
                [410.6261525310172, 207.97354059345616],
                [460.3783408790767, 174.43669899386418],
            ],
            closed: true
        }), new Path({
            segments: [
                [410.6261524612045, 207.9735406405153],
                [410.6261525310172, 207.97354059345614, -0.0005059167983745283, -0.0007686158121771314]
            ],
            closed: true
        })
    ], true);
    var path2 = new Path({
        segments: [
            [410.6261524612047, 207.97354064051552, 0, 0, 0.19904749518872222, 0.2952886437272184],
            [409.163896522797, 207.2586618457598, 1.6828473498011363, 0.6114523237241087],
            [460.3783408790765, 174.43669899386396]
        ],
        closed: true
    });
    compareBoolean(function() { return path1.unite(path2); },
        'M401.77543,286.98036l8.85044,-79.00432c-0.16499,-0.13413 -0.57872,-0.39645 -1.46198,-0.71738l51.21444,-32.82196z M410.62615,207.97354c0,0 0,0 0,0z');
});

test('#854', function() {
    var p = new Path({
        segments:[
            [110, 60],
            [99.09349390368303, 54.102707245527334, 0, 0, 0.2109804080034046, -0.3885775729842338],
            [99.76298270272372, 52.95998195277986, -0.2353176636361775, 0.3732505490504639,],
            [107.23497309089953, 78.84996109436266, 0, 0, -9.106249727572695, -5.544212301393825],
            [100.0593186152821, 52.32387941069251, -5.08710486250942, 9.369281629969153, 0, 0],
            ],
        closed:true
    });
    p.translate(100, 100);
    p.scale(4);
    compareBoolean(function() {
            p.windingRule = 'evenodd';
            return p.unite();
        },
        'M228.55086,143.23924l-38.88978,-21.02822l27.82967,96.42806c-34.09062,-20.75559 -47.33495,-62.77314 -32.21096,-98.79709l-0.35495,-0.19193c0.64433,-1.18671 1.31713,-2.35551 2.01826,-3.50647c0,0 0,0 0,0c-0.58827,1.22495 -1.1426,2.45812 -1.66331,3.6984l4.38129,2.36902l-2.05829,-7.13185c-0.22261,0.3531 -0.44251,0.70791 -0.6597,1.06443c0.58173,-1.21134 1.19664,-2.41465 1.84504,-3.60884z');

    compareBoolean(function() {
            p.windingRule = 'nonzero';
            return p.unite();
        },
        'M228.55086,143.23924l-38.88978,-21.02822l27.82967,96.42806c-34.09062,-20.75559 -47.33495,-62.77314 -32.21096,-98.79709l-0.35495,-0.19193c0.64433,-1.18671 1.31713,-2.35551 2.01826,-3.50647c0.58173,-1.21134 1.19664,-2.41465 1.84504,-3.60884z');
});

test('#859', function() {
    var p1 = new Path({
        segments: [
            [230, 360],
            [326.04999999999995, 361.95, 0, 0, 7.100000000000023, 5.300000000000011],
            [347.74999999999994, 377.3, -7.300000000000011, -5, 0, 0],
            [260, 400],
        ],
        closed: true
    });
    var p2 = new Path({
        segments: [
            [329.8373529833907, 360.99927475751736, 0, 0, -0.5084518552435497, 1.7136677994218417],
            [327.9816245617005, 366.0401044369074, 0.7293957729680756, -1.645975116174725, 0, 0],
            [300, 380]
        ],
        closed: true
    });
    compareBoolean(function() { return p1.subtract(p2); },
        'M230,360l96.05,1.95c0.3523,0.26298 0.70509,0.52535 1.05835,0.78713l-27.10835,17.26287l27.98162,-13.9599c0.29878,-0.67424 0.57885,-1.35439 0.84036,-2.04026c6.22144,4.55915 12.57473,8.94859 18.92802,13.30016l-87.75,22.7z');
});

test('#839', function() {
    var p1 = new Path({
        segments: [
            [522, 352],
            [500, 400],
            [480, 400],
            [448, 448],
            [100, 448],
            [100, 352]
        ],
        closed: true
    });
    var p2 = new Path.Rectangle(250, 300, 100, 200);
    compareBoolean(function() { return p1.subtract(p2); },
        'M522,352l-22,48l-20,0l-32,48l-98,0l0,-96z M250,448l-150,0l0,-96l150,0z');
});

test('#865', function() {
    function executeTest(offset) {
        var p1 = new Path({
            segments:[
                [300, 100],
                [300, 350],
                [256.00000000000006, 350],
                [256.00000000000006, 250],
                [230, 224],
                [256.00000000000006, 200],
                [256.00000000000006, 100]
            ],
            closed:true
        });
        var p2 = new Path({
            segments:[
                [256, 150],
                [256, 300],
                [200, 300],
                [200, 150]
            ],
            closed:true
        });

        p1.position = p1.position.add([offset, 0]);

        compareBoolean(p1.subtract(p2),
                'M300,100l0,250l-44,0l0,-250z',
                'p1.subtract(p2); // with offset = ' + offset);
    }
    // Attempt different x-offsets since that produced different version of the
    // issue:
    executeTest(0);
    executeTest(-0.0000001);
    executeTest( 0.0000001);
    executeTest(-0.000000001);
    executeTest( 0.000000001);
});

test('#870', function() {
    var path1 = new Path.Rectangle({
        point: [50, 50],
        size: [150, 50]
    });

    var path2 = new Path.Rectangle({
        point: [70, 50],
        size: [150, 50]
    });

    compareBoolean(function() { return path1.intersect(path2); },
        'M70,50l130,0l0,50l-130,0z');

    var path1 = new Path.Rectangle({
        point: [50, 150],
        size: [50, 100]
    });

    var path2 = new Path.Rectangle({
        point: [50, 175],
        size: [50, 100]
    });

    compareBoolean(function() { return path1.intersect(path2); },
        'M50,250l0,-75l50,0l0,75z');
});

test('#875', function() {
    var p1 = new Path({
        segments: [
            [158.7, 389.3, 0, 0, -4.95, 4.95],
            [158.7, 407.2, -4.95, -4.95, 4.95, 4.95],
            [176.6, 407.2, -4.95, 4.95, 4.95, -4.95],
            [176.6, 389.3, 4.95, 4.95, -4.95, -4.95],
            [158.7, 389.3, 4.95, -4.95, 0, 0]
        ],
        closed: true
    });
    var p2 = new Path.Circle(260, 320, 100);

    compareBoolean(function() { return p1.subtract(p2); },
        'M158.7,407.2c4.95,4.95 12.95,4.95 17.9,0c4.95,-4.95 4.95,-12.95 0,-17.9c-4.95,-4.95 -12.95,-4.95 -17.9,0c-4.95,4.95 -4.95,12.95 0,17.9z');
});

test('#877', function() {
    var cp = new CompoundPath({
        children: [
            new Path.Circle(100, 60, 50),
            new Path.Circle(100, 60, 30),
            new Path({
                segments: [
                    [120, 140],
                    [150, 140],
                    [150, 190],
                    [120, 190]
                ],
                closed: true
            }),
        ],
        fillRule: 'evenodd'
    });

    var p = new Path({
        segments: [
            [135, 200],
            [135, 120],
            [170, 120],
            [170, 200]
        ],
        closed: true
    });
    compareBoolean(function() { return cp.subtract(p); },
        'M50,60c0,-27.61424 22.38576,-50 50,-50c27.61424,0 50,22.38576 50,50c0,27.61424 -22.38576,50 -50,50c-27.61424,0 -50,-22.38576 -50,-50z M100,90c16.56854,0 30,-13.43146 30,-30c0,-16.56854 -13.43146,-30 -30,-30c-16.56854,0 -30,13.43146 -30,30c0,16.56854 13.43146,30 30,30z M120,140l15,0l0,50l-15,0z');
});

test('#878', function() {
    var p1 = new Path({ segments:[ [431.90000000000003, 479.99999999999994], [128.00000000000003, 370], [80.00000000000003, 479.99999999999994], ], closed:true });
    var p2 = new Path({ segments:[ [400, 480, 10, 0, 0, 0], [140, 480, 0, 0, -10, 0], [300, 300]], closed:true });
    compareBoolean(function() { return p1.unite(p2); },
        'M431.9,480l-35.62956,-12.89652c-19.34292,-41.22209 -96.27044,-167.10348 -96.27044,-167.10348c0,0 -48.3459,51.18977 -91.9573,98.97235l-80.0427,-28.97235l-48,110z');

    compareBoolean(function() { return p1.subtract(p2); },
        'M431.9,480l-35.62956,-12.89652c3.78718,8.07094 5.3669,12.89652 3.72956,12.89652z M208.0427,398.97235l-80.0427,-28.97235l-48,110l60,0c-4.6672,0 29.87455,-39.20895 68.0427,-81.02765z');
});

test('#885', function() {
    var p1 = new Path.Rectangle(100, 100, 100, 100);
    var p2 = p1.clone();
    var empty = new Path();
    compareBoolean(function() { return p1.unite(p2); }, p1);
    compareBoolean(function() { return p1.intersect(p2); }, p1);
    compareBoolean(function() { return p1.subtract(p2); }, empty);
    compareBoolean(function() { return p1.exclude(p2); }, empty);
});

test('#889', function() {
    var cp = new CompoundPath([
        new Path({ segments: [ [340.26, 358.4], [576, 396.8], [345.78, 396.8] ], closed: true }),
        new Path({ segments: [ [691.2, 685.76], [672, 550.4, 0, 0, 10, 0], [729.6, 608, 0, -20, 0, 0] ], closed: true })
    ]);
    var p = new Path({ segments: [ [739, 418], [637, 704], [205, 704], [204.30709922574619, 356.553500194953] ], closed: true });
    compareBoolean(function() { return cp.subtract(p); },
        'M340.26,358.4l235.74,38.4l-21.47738,0l-212.24889,-24.39148z M691.2,685.76l-13.57151,-95.67911l11.09506,-31.10967c17.43794,12.2938 40.87645,34.99446 40.87645,49.02878z');
});

test('#890', function() {
    var cp = new CompoundPath([
        new Path({ segments: [ [676, 396.8], [445.78, 396.8], [426, 260] ], closed: true }),
        new Path({ segments: [ [672, 550.4, 0, 0, 31.74000000000001, 0], [633.5999999999999, 732.8, 0, 26.519999999999982] ], closed: true })
    ]);
    var p = new Path({ segments: [ [732, 418, 0, 0, 0, 52.45802606694065], [676.3444534355956, 552.3444534355972, 34.385686865893945, -34.38568686589349], [630, 704], [400, 704], [542, 228] ], closed: true });
    compareBoolean(function() { return cp.subtract(p); },
        'M491.6437,396.8l-45.8637,0l-19.78,-136.8l91.51486,50.07693z M676.40276,552.28613c19.70425,20.76653 -42.80276,205.5493 -42.80276,180.51387l27.46296,-130.44906l15.28149,-50.00648c0.01944,-0.01944 0.03887,-0.03888 0.0583,-0.05833z');

    var cp = new CompoundPath([
        new Path({segments:[[576, 396.8], [345.78, 396.8],  [249.6, 243.20000000000002]], closed:true}),
        new Path({segments:[[624.84, 550.46], [672, 550.4], [729.6, 608]], closed:true})
    ]);
    var p = new Path({segments:[[323.0000000000002, 228.0000000000001, 0, 0, 51.95879291356505, 0], [410.9999999999998, 704.0000000000002], [-21.000000000000227, 703.9999999999998], [-21.692900774253758, 356.5535001949528]], closed:true});
    compareBoolean(function() { return cp.subtract(p); },
        'M576,396.8l-199.38152,0c-6.06278,-38.62456 -12.80947,-74.58601 -20.1433,-103.3058z M255.78013,253.06971l-6.18013,-9.86971l14.43275,6.79188z M624.84,550.46l47.16,-0.06l57.6,57.6z');
});

test('#899', function() {
    var path = new Path({segments:[[349.31714952528665, 176.97192499841802, 2.0667935742653185, -10.850610233997372, 0, 0], [308.23418424120047, 394.39737504104323, 0, 0, -0.3852007263769224, 2.0386172397749647], [273.60284923271973, 415.7138856106701, 9.194683617471242, 1.7373535056759692, 0, 0], [340.6967045975081, 152.3115389894092, -4.006231967752797, -3.6687868382265094, 5.694026309937783, 5.214418174126138], [349.72123243229134, 170.85618880187394, -0.36146393999291604, -6.612268801346318, 0.1110858449889065, 2.0320961120115726], [349.31714952528654, 176.97192499841861, 0.3852007263769224, -2.038617239775249, -2.0667935742656027, 10.850610233997315], [333.4126425432937, 153.58289639999975, -10.850610233997315, -2.0667935742653754, 10.850610233997372, 2.0667935742653754]], closed:true});
    compareBoolean(function() { return path.unite(); },
        'M349.31715,176.97192l-41.08297,217.42545c-0.3852,2.03862 -25.43665,23.05386 -34.63134,21.31651c0,0 45.72613,-193.57408 61.79225,-248.85827c-4.42287,-6.99195 -6.99542,-14.22758 -1.98246,-13.27272c1.9865,0.37838 3.84798,1.03847 5.55095,1.93102c0.85247,-2.33202 1.44527,-3.46596 1.73311,-3.20238c5.69403,5.21442 8.66306,11.93238 9.02453,18.54465c0.10579,1.93513 -0.00703,3.88198 -0.35068,5.82386c-0.01706,0.09726 -0.03486,0.19455 -0.0534,0.29188z');
});

test('#936', function() {
    var p1 = new Path({
        segments:[
            [437.02857142857147, 121.64285714285714],
            [390.3571428571429, 74.97142857142856],
            [390.3571428571429, 437.02857142857147],
            [437.02857142857147, 390.3571428571429],
            [74.97142857142862, 390.3571428571429],
            [121.6428571428572, 437.02857142857147],
            [121.64285714285714, 74.97142857142858],
            [74.97142857142859, 121.64285714285714]
        ],
        closed:true
    });
    compareBoolean(function() { return p1.unite(); },
        'M390.35714,121.64286l0,268.71429l-268.71429,0l0,-268.71429z M121.64286,437.02857l-46.67143,-46.67143l46.67143,0z M390.35714,121.64286l0,-46.67143l46.67143,46.67143z M437.02857,390.35714l-46.67143,46.67143l0,-46.67143z M74.97143,121.64286l46.67143,-46.67143l0,46.67143z');

    var p2 = new Path({
        segments:[
            [384.0884708724184, 194.49096034834812, -66.07592496222571, -39.741588860234515],
            [364.2895784370851, 164.00333956484477, 0, 0, 9.430248075502885, 76.52772155228502],
            [317.5090396516519, 384.0884708724184, 39.74158886023454, -66.07592496222564],
            [347.9966604351552, 364.2895784370851, 0, 0, -76.52772155228439, 9.430248075502845],
            [127.91152912758163, 317.50903965165196, 66.0759249622259, 39.74158886023463],
            [147.71042156291495, 347.9966604351553, 0, 0, -9.43024807550288, -76.52772155228509],
            [194.49096034834812, 127.91152912758156, -39.74158886023451, 66.07592496222576],
            [164.00333956484474, 147.7104215629149, 0, 0, 76.52772155228499, -9.430248075502845]
        ],
        closed:true
    });
    compareBoolean(function() { return p2.unite(); },
        'M366.1932,184.4397c3.98163,62.97187 -9.38253,125.8453 -38.6329,181.75349c-62.97187,3.98163 -125.8453,-9.38253 -181.75349,-38.6329c-3.98163,-62.97187 9.38253,-125.8453 38.6329,-181.75349c62.97187,-3.98163 125.8453,9.38253 181.75349,38.6329z M164.00334,147.71042l30.48762,-19.79889c-3.53519,5.87774 -6.88622,11.84573 -10.05126,17.89528c-6.81387,0.43083 -13.62889,1.06475 -20.43636,1.90362z M147.71042,347.99666l-19.79889,-30.48762c5.87774,3.53519 11.84573,6.88622 17.89528,10.05126c0.43083,6.81387 1.06475,13.62889 1.90362,20.43636z M366.1932,184.4397c-0.43083,-6.81387 -1.06475,-13.62889 -1.90362,-20.43636l19.79889,30.48762c-5.87774,-3.53519 -11.84573,-6.88622 -17.89528,-10.05126z M347.99666,364.28958l-30.48762,19.79889c3.53519,-5.87774 6.88622,-11.84573 10.05126,-17.89528c6.81387,-0.43083 13.62889,-1.06475 20.43636,-1.90362z');
});

test('#923', function() {
    var p1 = new Path.Circle({center: [100, 100], radius: 20});
    var p2 = new Path.Circle({center: [150, 100], radius: 20});
    var cp = new CompoundPath([p1, p2]);
    var p3 = new Path.Circle({center: [100, 100], radius: 20});
    cp.strokeColor = 'red';
    p3.strokeColor = 'green';
    compareBoolean(function() { return cp.unite(p3); },
        'M80,100c0,-11.04569 8.95431,-20 20,-20c11.04569,0 20,8.95431 20,20c0,11.04569 -8.95431,20 -20,20c-11.04569,0 -20,-8.95431 -20,-20z M130,100c0,-11.04569 8.95431,-20 20,-20c11.04569,0 20,8.95431 20,20c0,11.04569 -8.95431,20 -20,20c-11.04569,0 -20,-8.95431 -20,-20z');
});

test('#958', function() {
    var p1 = new Path.Rectangle(100, 200, 200, 20);
    var p2 = new Path.Rectangle(140, 100, 20, 20);
    var cp = new CompoundPath({children: [p1, p2]});
    var bounds = new Path.Rectangle(cp.bounds);
    compareBoolean(function() { return bounds.intersect(cp); },
        'M100,220l0,-20l200,0l0,20z M140,100l20,0l0,20l-20,0z');
});

test('#968', function() {
    var p1 = new paper.Path({
        segments: [
            [352, 280, 0, -26.5, 0, 0],
            [352, 356, 0, 0, 0, 2.1999999999999886],
            [348, 360, 2.1999999999999886, 0, -72, 0]
        ],
        closed: true
    });
    var p2 = new paper.Path({
        segments: [
            [352, 344],
            [352, 356, 0, 0, 0, 2.1999999999999886],
            [348, 360, 2.1999999999999886, 0, 0, 0],
            [232, 360, 0, 0, -2.1999999999999886, 0]
        ],
        closed: true
    });
    compareBoolean(function() { return p1.subtract(p2); },
        'M352,280l0,64c0,0 -13.69105,1.79261 -31.82528,4.17778c-15.66463,-26.96617 31.82528,-89.12564 31.82528,-68.17778z');
});

test('#973', function() {
    var path = new Path.Ellipse(100, 100, 150, 110);
    path.segments[1].point.y += 60;
    path.segments[3].point.y -= 60;

    var resolved = path.resolveCrossings();
    var orientation = resolved.children.map(function(child) {
        return child.isClockwise();
    });
    equals(orientation, [true, false, true],
            'children orientation after calling path.resolveCrossings()');
});

test('#1036', function() {
    var line1 = new Path([
        [[305.10732,101.34786],[0,0],[62.9214,0]],
        [[499.20274,169.42611],[-29.38716,-68.57004],[5.78922,13.50818]],
        [[497.75426,221.57115],[2.90601,-13.5614],[-9.75434,45.52027]],
        [[416.63976,331.65512],[31.9259,-30.40562],[-21.77284,20.73604]],
        [[350.00999,391.04252],[23.92578,-18.22917],[-23.33885,17.78198]],
        [[277.58633,431.59977],[27.45996,-10.67887],[-1.72805,0.67202]],
        [[251.51381,437.39367],[0,5.7145],[0,0]]
    ]);

    var line2 = new Path([
        [[547.00236,88.31161],[0,0],[-1.36563,0]],
        [[544.10541,85.41466],[1.29555,0.43185],[-9.83725,-3.27908]],
        [[509.34205,82.51771],[10.32634,-1.29079],[-10.20055,1.27507]],
        [[444.16075,97.00245],[7.10741,-4.06138],[-4.93514,2.82008]],
        [[431.12449,105.69328],[4.27047,-2.13524],[-14.94798,7.47399]],
        [[407.94892,175.22],[1.27008,-13.33587],[-3.16966,33.28138]],
        [[399.25808,279.51008],[-5.61644,-33.69865],[1.73417,10.40499]],
        [[415.19129,307.03107],[-5.98792,-8.16534],[2.74694,3.74583]],
        [[432.57297,328.75817],[-3.89061,-3.29206],[2.9716,2.51443]],
        [[442.71228,334.55206],[-3.01275,-2.46498],[2.39275,1.95771]],
        [[448.50617,341.79443],[-2.37502,-1.97918],[39.75954,33.13295]],
        [[578.86877,378.00626],[-51.65429,10.33086],[11.28627,-2.25725]],
        [[612.18365,366.41848],[-10.6547,4.26188],[3.10697,-1.24279]],
        [[617.97755,362.07306],[-3.19904,0],[0,0]]
    ]);
    compareBoolean(function() { return line1.intersect(line2); },
        'M424.54226,112.15158c32.89387,9.15202 61.28089,26.0555 74.66048,57.27453c5.78922,13.50818 1.45753,38.58364 -1.44848,52.14504c-8.75233,40.8442 -41.40003,72.54068 -71.07836,100.58668c-4.48065,-5.55963 -9.68976,-12.67924 -11.48461,-15.12676c-5.98792,-8.16534 -14.19904,-17.116 -15.93321,-27.52099c-5.61644,-33.69865 5.52118,-71.0087 8.69084,-104.29008c1.06648,-11.19805 6.14308,-47.34273 16.59334,-63.06842z');
});

test('#1054', function() {
    var p1 = new Path({
        segments: [
            [310, 170],
            [200, 170],
            [200, 350],
            [190, 350],
            [190, 160],
            [320, 160],
            [320, 350],
            [310, 350]
        ],
        closed: true
    });
    var p2 = new Path({
        segments: [
            [195, 165],
            [315, 165],
            [315, 350],
            [195, 350]
        ],
        closed: true
    });
    compareBoolean(function() { return p1.unite(p2); },
        'M190,350l0,-190l130,0l0,190z');


    var cp = new CompoundPath();
    cp.addChild(new Path({
        segments: [
            [150, 100],
            [150, 150],
            [100, 150],
            [100, 100]
        ],
        closed: true
    }));
    cp.addChild(new Path({
        segments: [
            [200, 150],
            [200, 100],
            [220, 100],
            [220, 150]
        ],
        closed: true
    }));
    var p = new Path({
        segments: [
            [200, 100],
            [200, 150],
            [120, 150],
            [120, 100]
        ],
        closed: true
    });
    compareBoolean(function() { return cp.unite(p); },
        'M100,150l0,-50l120,0l0,50z');
});

test('#1059', function() {
    var p1 = new paper.Path({
        segments: [
            [428.48408954611784, 189.03444010371618, 0, 0, -21.46171522996096, 0],
            [369.18465116477694, 213.5984519858447, 16.376007921419017, -16.376007921419017, 0, 0],
            [209.1846511647769, 213.5984519858447, 0, 0, 16.376007921419017, -16.376007921419017],
            [268.48408954611784, 189.03444010371618, -21.46171522996096, 0, 0, 0]
        ],
        closed: true
    });
    var p2 = new paper.Path({
        segments: [
            [369.18465470031083, 213.59844845031083, 0, 0, -32.75201584283798, 32.752015842838006],
            [369.18465470031083, 332.1973252129927, -32.752015842838034, -32.75201584283809, 0, 0],
            [209.1846547003108, 332.1973252129927],
            [209.1846547003108, 213.59844845031083, -32.75201584283798, 32.752015842838006, 0, 0]
        ],
        closed: true
    });
    compareBoolean(function() { return p1.unite(p2); },
        'M428.48409,189.03444c-21.46172,0 -42.92343,8.188 -59.29943,24.56401c-32.75202,32.75202 -32.75202,85.84686 0,118.59888l-160,0c0,0 -32.75202,-85.84686 0,-118.59888l0,0c16.37601,-16.37601 37.83772,-24.56401 59.29944,-24.56401z');
});

test('#1075', function() {
    var p1 = new paper.Path({
        segments: [
            [150, 120],
            [150, 85],
            [178, 85],
            [178, 110],
            [315, 110],
            [315, 85],
            [342, 85],
            [342, 120],
        ],
        closed: true
    });
    var p2 = new paper.Path({
        segments: [
            [350, 60],
            [350, 125],
            [315, 125],
            [315, 85],
            [178, 85],
            [178, 125],
            [140, 125],
            [140, 60]
        ],
        closed: true
    });

    compareBoolean(function() { return p1.unite(p2); },
        'M140,125l0,-65l210,0l0,65l-35,0l0,-5l-137,0l0,5z M315,85l-137,0l0,25l137,0z');
});

test('#1091', function() {
    var data = [
        'M91.24228100000002,45.622281000000015v396.89171899999997c0,25.195925999999986 -20.426355000000015,45.622281000000044 -45.62000000000001,45.622281000000044c-25.198207000000007,0 -45.620000000000005,-20.42407400000002 -45.620000000000005,-45.622281000000044v-396.889438c0,-25.19820700000001 20.421792999999997,-45.62228100000001 45.620000000000005,-45.62228100000001c25.193644999999997,0 45.617719000000015,20.424074000000015 45.62000000000001,45.62000000000001z0',
        'M223.67029800000003,488.13628100000005h-178.05029800000003c-25.198207000000004,0 -45.620000000000005,-20.426355000000058 -45.620000000000005,-45.622281000000044v0c0,25.198207000000025 20.421793,45.622281000000044 45.620000000000005,45.622281000000044c25.193645000000004,0 45.620000000000005,-20.426355000000058 45.620000000000005,-45.622281000000044v-45.620000000000005h132.430298v0c-25.193645000000004,0 -45.620000000000005,20.42407400000002 -45.620000000000005,45.620000000000005c0.0022810000000106356,25.198207000000025 20.426355,45.622281000000044 45.620000000000005,45.622281000000044',
        'M225.369643,488.13628100000005h-1.6993449999999939c1.0013589999999795,0 2.0004369999999767,-0.006843000000003485 2.9995150000000024,-0.01824800000002824c-0.4311089999999922,0.011404999999967913 -0.8644990000000234,0.01824800000002824 -1.3001700000000085,0.01824800000002824z',
        'M45.62228100000001,488.13628100000005c-25.198207000000007,0 -45.620000000000005,-20.426355000000058 -45.620000000000005,-45.622281000000044c0,-25.193644999999947 20.421792999999997,-45.620000000000005 45.620000000000005,-45.620000000000005h45.62000000000001v45.620000000000005c-0.0022809999999964248,25.198207000000025 -20.426355000000015,45.622281000000044 -45.62000000000001,45.622281000000044z',
        'M223.67029800000003,91.242281h-132.430298v-45.620000000000005c0,-25.195926000000004 -20.424074000000005,-45.62228100000001 -45.620000000000005,-45.62228100000001h179.667527c0.41742300000001364,0 0.8325649999999882,0.004561999999999999 1.2477069999999912,0.015967000000000002c-0.9511770000000013,-0.011405000000000002 -1.9137589999999989,-0.015967000000000002 -2.867217000000011,-0.015967000000000002c-25.193645000000004,0 -45.620000000000005,20.424074 -45.620000000000005,45.62228100000001c0.0045619999999928496,25.195926 20.42863600000004,45.620000000000005 45.622281000000044,45.620000000000005z',
        'M91.24000000000001,91.242281h-45.620000000000005c-25.198207000000004,0 -45.620000000000005,-20.424074000000005 -45.620000000000005,-45.620000000000005c0,-25.195926000000004 20.421793,-45.62228100000001 45.620000000000005,-45.62228100000001v0c25.193645000000004,0 45.620000000000005,20.424074 45.620000000000005,45.62228100000001z',
        'M226.66981300000003,488.118033c24.59146100000001,-0.6888619999999719 44.31982999999997,-20.841497000000004 44.31982999999997,-45.604033000000015c0,-25.193644999999947 -20.42407400000002,-45.620000000000005 -45.620000000000005,-45.620000000000005h-1.6993449999999939c80.626507,0 139.143281,-64.27173699999997 139.143281,-152.827v0c0,25.198207000000025 20.42407400000002,45.620000000000005 45.622281000000044,45.620000000000005c25.193644999999947,0 45.620000000000005,-20.42179299999998 45.620000000000005,-45.620000000000005c0,138.080335 -97.54012200000005,242.46573800000004 -227.38604700000002,244.05103300000002z',
        'M223.67029800000003,488.13628100000005v0c-25.193645000000004,0 -45.620000000000005,-20.426355000000058 -45.620000000000005,-45.622281000000044c0,-25.193644999999947 20.426355,-45.620000000000005 45.620000000000005,-45.620000000000005v0h1.6993450000000223c25.193645000000004,0 45.620000000000005,20.42407400000002 45.620000000000005,45.620000000000005c0,24.76253600000001 -19.726088000000004,44.91517100000004 -44.319829999999996,45.604033000000015c-0.9967969999999866,0.011404999999967913 -1.9958750000000407,0.01824800000002824 -2.999515000000031,0.01824800000002824z',
        'M454.05586000000005,244.06928100000002c0,-25.198206999999996 -20.42407400000002,-45.617718999999994 -45.620000000000005,-45.617718999999994c-25.198207000000025,0 -45.622281000000044,20.421793000000008 -45.622281000000044,45.617718999999994v0c0,-88.55298200000001 -58.516774,-152.827 -139.143281,-152.827v0h1.617229000000009c25.198206999999996,0 45.62228099999999,-20.424074000000005 45.62228099999999,-45.620000000000005c0,-24.778503 -19.755740999999972,-44.942543 -44.372292999999985,-45.606314000000005c129.918917,1.5123030000000002 227.51834500000004,105.922797 227.51834500000004,244.053314z',
        'M225.28980800000002,91.242281h-1.617229000000009c-25.193645000000004,0 -45.620000000000005,-20.424074000000005 -45.620000000000005,-45.620000000000005c0,-25.195926000000004 20.426355,-45.62228100000001 45.620000000000005,-45.62228100000001c0.9534580000000119,0 1.9160400000000095,0.006843000000000001 2.867217000000011,0.015967000000000002c24.618833000000024,0.6637710000000001 44.37229300000001,20.827811 44.37229300000001,45.60631400000001c0,25.195926 -20.426355,45.620000000000005 -45.622281000000015,45.620000000000005z',
        'M408.43586000000005,289.687c-25.198207000000025,0 -45.622281000000044,-20.42179299999998 -45.622281000000044,-45.620000000000005v0v0c0,-25.198206999999996 20.42407400000002,-45.617718999999994 45.622281000000044,-45.617718999999994c25.193644999999947,0 45.620000000000005,20.421793000000008 45.620000000000005,45.617718999999994v0v0c0,25.198207000000025 -20.42407400000002,45.620000000000005 -45.620000000000005,45.620000000000005z'
    ];
    var res = new CompoundPath();

    for (var i = 0; i < data.length; i++) {
        var path = PathItem.create(data[i]);
        path.remove();
        var prev = res;
        res = res.unite(path);
        prev.remove();
    };
    compareBoolean(function() { return res; },
        'M91.24228,396.894h132.42802c-25.19365,0 -45.62,20.42407 -45.62,45.62c0,-25.19364 20.42635,-45.62 45.62,-45.62c80.62581,0 139.14228,-64.27063 139.14328,-152.82472l0,-0.00228c-0.001,-88.55097 -58.51636,-152.82351 -139.141,-152.82472l-0.00228,0c-25.1926,-0.00123 -45.61772,-20.42483 -45.61772,-45.62c0,-25.1955 20.42566,-45.62158 45.61871,-45.62228h1.61624c0.4166,0 0.83093,0.00454 1.24526,0.0159c0.00234,0.00002 0.00467,0.00004 0.00701,0.00007c0.00058,0.00002 0.00116,0.00003 0.00173,0.00005c129.91593,1.5144 227.51285,105.92259 227.51433,244.05012c0,0.00029 0,0.00057 0,0.00086c0,0.00012 0,0.00024 0,0.00036l0,0.00192c-0.00107,138.0792 -97.54084,242.46347 -227.38605,244.04875c-0.43111,0.0114 -0.8645,0.01825 -1.30017,0.01825h-1.69934c-12.59632,0 -24.00091,-5.10618 -32.25663,-13.36168c8.2555,8.25572 19.65987,13.36168 32.25663,13.36168l-178.04574,0c-0.00076,0 -0.00152,0 -0.00228,0c-0.00076,0 -0.00152,0 -0.00228,0h0c-25.19716,-0.00123 -45.61772,-20.42483 -45.61772,-45.62228v-396.88944c0,-25.19821 20.42179,-45.62228 45.62,-45.62228c14.89455,0 28.12203,7.13863 36.44812,18.18156c-8.3258,-11.04405 -21.55413,-18.18384 -36.4504,-18.18384h178.04802c-25.19365,0 -45.62,20.42407 -45.62,45.62228c0.00456,25.19593 20.42864,45.62 45.62228,45.62l-132.42802,0zM45.62,488.13628c-25.19821,0 -45.62,-20.42407 -45.62,-45.62228c0,25.19593 20.42179,45.62228 45.62,45.62228zM226.51682,0.01575c-0.93686,-0.01114 -1.88465,-0.01567 -2.82377,-0.01575c0.93909,0.0001 1.88688,0.0068 2.82377,0.01575zM362.81358,244.06928c0.00123,25.19716 20.42483,45.61772 45.62228,45.61772c-25.19745,0 -45.62105,-20.42056 -45.62228,-45.61772z');
});

test('#1109', function() {
    var segments = [
        [274.616046379547, 215.30654471841746],
        [296.47810618021146, 202.68940186127458],
        [297.0266776087829, 160.32607960213838],
        [260.5870762798793, 181.99677727655697],
        [260.5870762798793, 207.20554804067655],
    ];

    var p1 = new CompoundPath([
        new Path.Rectangle(200, 100, 200, 200),
        new Path({ segments: segments, closed: true })
    ]);

    var p2 = new paper.CompoundPath([
        new Path.Rectangle(210, 110, 180, 180),
        new Path({ segments: segments, closed: true })
    ]);

    // Rotate segments to produce different sequence that describes same shape.
    segments.push(segments.shift());

    var p3 = new paper.CompoundPath([
        new Path.Rectangle(210, 110, 180, 180),
        new Path({ segments: segments, closed: true })
    ]);

    // Add a little spike, see #issuecomment-236026445
    segments.splice(3, 0,
        [260.5870762798793, 200],
        [265.5870762798793, 201],
        [260.5870762798793, 202]);
    var p4 = new paper.CompoundPath([
        new Path.Rectangle(210, 110, 180, 180),
        new Path({ segments: segments, closed: true })
    ]);

    compareBoolean(function() { return p1.unite(p2); },
        'M200,300v-200h200v200zM274.61605,215.30654l21.86206,-12.61714l0.54857,-42.36332l-36.4396,21.6707v25.20877z');
    compareBoolean(function() { return p1.unite(p3); },
        'M200,300v-200h200v200zM274.61605,215.30654l21.86206,-12.61714l0.54857,-42.36332l-36.4396,21.6707v25.20877z');
    compareBoolean(function() { return p1.unite(p4); },
        'M200,300v-200h200v200zM265.58708,201l-5,1v5.20555l14.02897,8.101l21.86206,-12.61714l0.54857,-42.36332l-36.4396,21.6707v18.00322z');
});

test('#1123', function() {
    var p1 = new paper.Path({
        segments: [
            [29.562573408343212, 132.54192488999095, 0, 0, -1.417935581083043, 0.174100597711913],
            [2.5662795205536213, 124.83546947214427, 13.887634830511104, -1.705187144644185, 0, 0],
            [18.447017946814753, 122.88555997766193, 0, 0, 13.88763483051116, -1.7051871446442135],
            [45.443311834604344, 130.5920153955086, -1.417935581083043, 0.17410059771188457, 0, 0]
        ],
        closed: true
    });
    var p2 = new paper.Path({
        segments: [
            [5.684341886080802e-14, 132.07000000000002, 12.910000000000025, 27.075000000000017, 21.125, -21.119999999999976],
            [44.59500000000003, 130.54000000000005, 0, 2.3300000000000125, 4.816669792192272, -4.501887430254385],
            [56.887403180177614, 116.74936916538223, -3.66267533741933, 4.517228939026069, 0.002528361414420033, 50]
        ],
        closed: true
    });
    compareBoolean(function() { return p1.intersect(p2); },
        'M29.56257,132.54192c-1.1465,0.14077 -9.00925,-6.09934 -19.31903,-7.59213c2.44054,-1.04008 4.85728,-1.65337 7.21093,-1.94237c13.88763,-1.70519 25.57836,7.88056 26.99629,7.70646zM44.75533,130.3897c0.31974,0.14538 0.55147,0.21908 0.68798,0.20232l-0.99255,0.12187c0.09534,-0.01171 0.14423,-0.06753 0.14423,-0.17388c0.05357,-0.05007 0.10701,-0.10017 0.16033,-0.1503z');
});

test('#1123', function() {
    var p1 = new Path.Rectangle(100, 100, 100, 100);
    var p2 = new Path.Rectangle(120, 120, 60, 60);
    compareBoolean(function() { return p1.subtract(p2).subtract(p2); },
        'M100,200v-100h100v100zM180,180v-60h-60v60z');
});

test('#1221', function() {
    var rect1 = new Path.Rectangle({
        point: [100, 100],
        size: [200, 200]
    });

    var circle = new Path.Circle({
        center: [100, 100],
        radius: 100
    });

    compareBoolean(function() { return rect1.subtract(circle, { trace: false }); },
        'M200,100h100v200h-200v-100');
    compareBoolean(function() { return rect1.subtract(circle, { trace: true }); },
        'M100,300v-100c55.22847,0 100,-44.77153 100,-100h100v200z');


    var blob = PathItem.create("M534,273C171.7,111,60.5,117.1,30,158c-40.5,54.3,31.5,210.2,111,222c60.8,9,88-71.9,159-66c81.6,6.8,99.6,118.3,179,128c33.8,4.1,83.1-9.7,150-90")
    var rect2 = new Path.Rectangle({
        point: [150, 100],
        size: [300, 300]
    });

    compareBoolean(function() { return blob.subtract(rect2, { trace: false }); },
        'M534,273c-29.65069,-13.2581 -57.61955,-25.39031 -84,-36.46967M150,138.13156c-71.67127,-11.53613 -105.25987,0.10217 -120,19.86844c-40.5,54.3 31.5,210.2 111,222c3.08303,0.45637 6.07967,0.68158 9,0.69867M409.85616,400c18.87105,20.95032 39.82014,38.41763 69.14384,42c33.8,4.1 83.1,-9.7 150,-90');
    compareBoolean(function() { return blob.subtract(rect2, { trace: true }); },
        'M629,352c-66.9,80.3 -116.2,94.1 -150,90c-29.3237,-3.58237 -50.27279,-21.04968 -69.14384,-42h40.14384v-163.46967c26.38045,11.07937 54.34931,23.21157 84,36.46967M141,380c-79.5,-11.8 -151.5,-167.7 -111,-222c14.74013,-19.76627 48.32873,-31.40457 120,-19.86844v242.56712c-2.92033,-0.01709 -5.91697,-0.24231 -9,-0.69867z');

    var rect3 = new Path.Rectangle({
        point: [150, 100],
        size: [300, 150]
    });

    compareBoolean(function() { return blob.subtract(rect3, { trace: false }); },
        'M534,273c-29.65069,-13.2581 -57.61955,-25.39031 -84,-36.46967M150,138.13156c-71.67127,-11.53613 -105.25987,0.10217 -120,19.86844c-40.5,54.3 31.5,210.2 111,222c60.8,9 88,-71.9 159,-66c81.6,6.8 99.6,118.3 179,128c33.8,4.1 83.1,-9.7 150,-90');
    compareBoolean(function() { return blob.subtract(rect3, { trace: true }); },
        'M629,352c-66.9,80.3 -116.2,94.1 -150,90c-79.4,-9.7 -97.4,-121.2 -179,-128c-71,-5.9 -98.2,75 -159,66c-79.5,-11.8 -151.5,-167.7 -111,-222c14.74013,-19.76627 48.32873,-31.40457 120,-19.86844v111.86844h300v-13.46967c26.38045,11.07937 54.34931,23.21157 84,36.46967');


    var rect4 = new Path.Rectangle({
        point: [200, 200],
        size: [400, 200]
    });

    var line = new Path.Line({
        from: [400, 300],
        to: [400, 600]
    });

    var division = line.divide(rect4, { trace: false });
    equals(function() { return division.children.length; }, 2);
    compareBoolean(function() { return division.children[0]; }, 'M400,300v100');
    compareBoolean(function() { return division.children[1]; }, 'M400,400v200');
});

test('#1239 / #1073', function() {
    var p1 = new Path([[890.91, 7.52, 0, 0, 85.40999999999997, 94.02], [1024, 351.78, 0, -127.03999999999996, 0, 127.14999999999998], [890.69, 696.28, 85.54999999999995, -94.03999999999996, 0, 0], [843.44, 653.28, 0, 0, 75.20000000000005, -82.66999999999996], [960, 351.78, 0, 111.75, 0, -111.63], [843.65, 50.51999999999998, 75.07000000000005, 82.63999999999999, 0, 0], true]);
    var p2 = new Path([[960, 352, -0.05999999999994543, 111.67999999999995, 0, 0], [1024, 352, 0, 0, -0.05999999999994543, 127.07], [890.69, 696.28, 85.5, -93.98000000000002, 0, 0], [843.44, 653.28, 0, 0, 75.14999999999998, -82.61000000000001], true]);
    project.activeLayer.scale(0.25);
    compareBoolean(function() { return p1.unite(p2); },
        'M923.0175,265.805c21.3525,23.505 33.2725,54.305 33.2725,86.065c0,0.01833 0,0.03667 -0.00001,0.055h0.00001c-0.00005,0.10258 -0.00022,0.20515 -0.00052,0.3077c-0.06338,22.18242 -5.9393,43.88534 -16.78017,62.94682c-4.63138,8.14369 -10.16899,15.8051 -16.54682,22.81548l-11.8125,-10.75c8.97181,-9.86302 16.01692,-21.11585 20.93099,-33.22212c5.34364,-13.16533 8.16725,-27.34044 8.20856,-41.83592c0.0003,-0.10564 0.00044,-0.21129 0.00044,-0.31697c0,-27.9075 -10.32,-54.655 -29.0875,-75.315z');
});

test('#1261', function() {
    var p1 = PathItem.create("M933.13,1023.97l-516.19,-171.71l607.1,-101.57z");
    var p2 = PathItem.create("M933.13,1023.97l-516.19,-171.71l67.33,-11.27l337.49,112.27c16.7,-50.22 71.03,-77.36 121.34,-60.63l45.23,-135.96l35.71,-5.98z");

    compareBoolean(function() { return p1.unite(p2); },
        'M933.13,1023.97l-516.19,-171.71l67.33,-11.27l0.0109,0.00363l539.7591,-90.30363z');
});

test('#1321', function() {
    var path = PathItem.create('M24,38l2,1l-2,1l-2,-1z M26,39l2,1l-2,1l-2,-1z M28,40l2,1l-2,1l-2,-1z')
    compareBoolean(function() { return path.unite(); },
        'M24,38l6,3l-2,1l-6,-3z');
})

test('Selected edge-cases from @hari\'s boolean-test suite', function() {
    var g = PathItem.create('M316.6,266.4Q332.6,266.4,343.8,272.8Q355,279.2,362,289.8Q369,300.4,372.2,313.6Q375.4,326.8,375.4,340.4Q375.4,354.8,372,369.2Q368.6,383.6,361.4,395Q354.2,406.4,342.4,413.4Q330.6,420.4,313.8,420.4Q297,420.4,285.8,413.4Q274.6,406.4,267.8,395Q261,383.6,258.2,369.6Q255.4,355.6,255.4,341.6Q255.4,326.8,258.8,313.2Q262.2,299.6,269.6,289.2Q277,278.8,288.6,272.6Q300.2,266.4,316.6,266.4Z M315,236.4Q288.2,236.4,269.8,246.6Q251.4,256.8,240.2,272.6Q229,288.4,224.2,307.8Q219.4,327.2,219.4,345.6Q219.4,366.8,225.2,385.8Q231,404.8,242.6,419Q254.2,433.2,271.4,441.6Q288.6,450,311.8,450Q331.8,450,349.6,441Q367.4,432,376.2,412.8L377,412.8L377,426.4Q377,443.6,373.6,458Q370.2,472.4,362.6,482.6Q355,492.8,343.4,498.6Q331.8,504.4,315,504.4Q306.6,504.4,297.4,502.6Q288.2,500.8,280.4,496.8Q272.6,492.8,267.2,486.4Q261.8,480,261.4,470.8L227.4,470.8Q228.2,487.6,236.2,499.2Q244.2,510.8,256.4,518Q268.6,525.2,283.6,528.4Q298.6,531.6,313,531.6Q362.6,531.6,385.8,506.4Q409,481.2,409,430.4L409,241.2L377,241.2L377,270.8L376.6,270.8Q367.4,253.6,351,245Q334.6,236.4,315,236.4Z');
    var u = PathItem.create('M253,316.74Q242.25,316.74,232.77,318.39Q218.77,320.83,208.21,328.52Q197.65,336.21,191.32,349.4Q185,362.6,183.59,382.95Q182.01,405.69,189.83,423.08Q197.64,440.46,216.05,452.56L215.99,453.36L183.27,451.09L181.06,483.01L387.37,497.31L389.72,463.39L273.2,455.32Q259.23,454.35,247.72,449.74Q236.21,445.14,227.96,436.95Q219.7,428.76,215.7,417.05Q211.7,405.35,212.78,389.78Q214.14,370.23,226.09,359.83Q236.68,350.61,252.94,350.61Q255.02,350.61,257.19,350.76L396.85,360.44L399.2,326.52L263.53,317.12Q258.12,316.74,253,316.74Z');
    compareBoolean(function() { return g.unite(u); },
        'M315,236.4c-17.86667,0 -32.93333,3.4 -45.2,10.2c-12.26667,6.8 -22.13333,15.46667 -29.6,26c-7.46667,10.53333 -12.8,22.26667 -16,35.2c-1.13596,4.59118 -2.0703,9.14036 -2.80301,13.64753c-4.76983,1.78645 -9.1655,4.14394 -13.18699,7.07247c-7.04,5.12667 -12.67,12.08667 -16.89,20.88c-4.21333,8.8 -6.79,19.98333 -7.73,33.55c-1.05333,15.16 1.02667,28.53667 6.24,40.13c5.20667,11.58667 13.94667,21.41333 26.22,29.48l-0.06,0.8l-32.72,-2.27l-2.21,31.92l48.97248,3.39444c1.46464,4.69217 3.52048,8.95735 6.16752,12.79556c5.33333,7.73333 12.06667,14 20.2,18.8c8.13333,4.8 17.2,8.26667 27.2,10.4c10,2.13333 19.8,3.2 29.4,3.2c33.06667,0 57.33333,-8.4 72.8,-25.2c15.46667,-16.8 23.2,-42.13333 23.2,-76l0,-189.2l-32,0l0,29.6l-0.4,0c-6.13333,-11.46667 -14.66667,-20.06667 -25.6,-25.8c-10.93333,-5.73333 -22.93333,-8.6 -36,-8.6z M220.76108,365.62343c0.96573,6.92045 2.44537,13.64598 4.43892,20.17657c3.86667,12.66667 9.66667,23.73333 17.4,33.2c7.73333,9.46667 17.33333,17 28.8,22.6c11.46667,5.6 24.93333,8.4 40.4,8.4c13.33333,0 25.93333,-3 37.8,-9c11.86667,-6 20.73333,-15.4 26.6,-28.2l0.8,0l0,13.6c0,11.46667 -1.13333,22 -3.4,31.6c-0.33765,1.43005 -0.70637,2.82903 -1.10615,4.19694l-99.29385,-6.87694c-9.31333,-0.64667 -17.80667,-2.50667 -25.48,-5.58c-7.67333,-3.06667 -14.26,-7.33 -19.76,-12.79c-5.50667,-5.46 -9.59333,-12.09333 -12.26,-19.9c-2.66667,-7.8 -3.64,-16.89 -2.92,-27.27c0.67923,-9.76393 3.33959,-17.81612 7.98108,-24.15657z M350.00561,494.72016c-2.08166,1.42552 -4.28353,2.7188 -6.60561,3.87984c-7.73333,3.86667 -17.2,5.8 -28.4,5.8c-5.6,0 -11.46667,-0.6 -17.6,-1.8c-6.13333,-1.2 -11.8,-3.13333 -17,-5.8c-4.03048,-2.06691 -7.58035,-4.61444 -10.6496,-7.64259z M316.6,266.4c10.66667,0 19.73333,2.13333 27.2,6.4c7.46667,4.26667 13.53333,9.93333 18.2,17c4.66667,7.06667 8.06667,15 10.2,23.8c0.89905,3.70858 1.60866,7.44084 2.12882,11.19678l-110.79882,-7.67678c-1.88359,-0.1323 -3.74081,-0.23006 -5.57167,-0.29327c0.26064,-1.21594 0.5412,-2.42485 0.84167,-3.62673c2.26667,-9.06667 5.86667,-17.06667 10.8,-24c4.93333,-6.93333 11.26667,-12.46667 19,-16.6c7.73333,-4.13333 17.06667,-6.2 28,-6.2z M374.00366,358.85649c-0.5217,3.44784 -1.18958,6.89567 -2.00366,10.34351c-2.26667,9.6 -5.8,18.2 -10.6,25.8c-4.8,7.6 -11.13333,13.73333 -19,18.4c-7.86667,4.66667 -17.4,7 -28.6,7c-11.2,0 -20.53333,-2.33333 -28,-7c-7.46667,-4.66667 -13.46667,-10.8 -18,-18.4c-4.53333,-7.6 -7.73333,-16.06667 -9.6,-25.4c-1.26174,-6.30869 -2.09705,-12.61738 -2.50594,-18.92608c0.49503,0.02266 0.99368,0.05136 1.49594,0.08608z');
    compareBoolean(function() { return g.subtract(u); },
        'M315,236.4c-17.86667,0 -32.93333,3.4 -45.2,10.2c-12.26667,6.8 -22.13333,15.46667 -29.6,26c-7.46667,10.53333 -12.8,22.26667 -16,35.2c-1.13596,4.59118 -2.0703,9.14036 -2.80301,13.64753c3.5802,-1.34089 7.3712,-2.36007 11.37301,-3.05753c6.32,-1.1 13.06333,-1.65 20.23,-1.65c1.63071,0 3.28349,0.02891 4.95833,0.08673c0.26064,-1.21594 0.5412,-2.42485 0.84167,-3.62673c2.26667,-9.06667 5.86667,-17.06667 10.8,-24c4.93333,-6.93333 11.26667,-12.46667 19,-16.6c7.73333,-4.13333 17.06667,-6.2 28,-6.2c10.66667,0 19.73333,2.13333 27.2,6.4c7.46667,4.26667 13.53333,9.93333 18.2,17c4.66667,7.06667 8.06667,15 10.2,23.8c0.89905,3.70858 1.60866,7.44084 2.12882,11.19678l24.87118,1.72322l-2.35,33.92l-22.84634,-1.58351c-0.5217,3.44784 -1.18958,6.89567 -2.00366,10.34351c-2.26667,9.6 -5.8,18.2 -10.6,25.8c-4.8,7.6 -11.13333,13.73333 -19,18.4c-7.86667,4.66667 -17.4,7 -28.6,7c-11.2,0 -20.53333,-2.33333 -28,-7c-7.46667,-4.66667 -13.46667,-10.8 -18,-18.4c-4.53333,-7.6 -7.73333,-16.06667 -9.6,-25.4c-1.26174,-6.30869 -2.09705,-12.61738 -2.50594,-18.92608c-0.9308,-0.04262 -1.84882,-0.06392 -2.75406,-0.06392c-10.84,0 -19.79,3.07333 -26.85,9.22c-1.99843,1.73922 -3.77474,3.67036 -5.32892,5.79343c0.96573,6.92045 2.44537,13.64598 4.43892,20.17657c3.86667,12.66667 9.66667,23.73333 17.4,33.2c7.73333,9.46667 17.33333,17 28.8,22.6c11.46667,5.6 24.93333,8.4 40.4,8.4c13.33333,0 25.93333,-3 37.8,-9c11.86667,-6 20.73333,-15.4 26.6,-28.2l0.8,0l0,13.6c0,11.46667 -1.13333,22 -3.4,31.6c-0.33765,1.43005 -0.70637,2.82903 -1.10615,4.19694l17.22615,1.19306l-2.35,33.92l-37.36439,-2.58984c-2.08166,1.42552 -4.28353,2.7188 -6.60561,3.87984c-7.73333,3.86667 -17.2,5.8 -28.4,5.8c-5.6,0 -11.46667,-0.6 -17.6,-1.8c-6.13333,-1.2 -11.8,-3.13333 -17,-5.8c-4.03048,-2.06691 -7.58035,-4.61444 -10.6496,-7.64259l-39.71792,-2.75297c1.46464,4.69217 3.52048,8.95735 6.16752,12.79556c5.33333,7.73333 12.06667,14 20.2,18.8c8.13333,4.8 17.2,8.26667 27.2,10.4c10,2.13333 19.8,3.2 29.4,3.2c33.06667,0 57.33333,-8.4 72.8,-25.2c15.46667,-16.8 23.2,-42.13333 23.2,-76l0,-189.2l-32,0l0,29.6l-0.4,0c-6.13333,-11.46667 -14.66667,-20.06667 -25.6,-25.8c-10.93333,-5.73333 -22.93333,-8.6 -36,-8.6z');
    compareBoolean(function() { return g.intersect(u); },
        'M221.39699,321.44753c-1.33133,8.1895 -1.99699,16.24033 -1.99699,24.15247c0,6.84657 0.45369,13.52104 1.36108,20.02343c1.55418,-2.12307 3.33048,-4.05421 5.32892,-5.79343c7.06,-6.14667 16.01,-9.22 26.85,-9.22c0.90523,0 1.82325,0.02131 2.75406,0.06393c-0.19604,-3.02464 -0.29406,-6.04928 -0.29406,-9.07393c0,-8.55874 0.85278,-16.81649 2.55833,-24.77327c-1.67484,-0.05782 -3.32762,-0.08673 -4.95833,-0.08673c-7.16667,0 -13.91,0.55 -20.23,1.65c-4.0018,0.69746 -7.79281,1.71663 -11.37301,3.05753z M372.49385,462.19694c-2.28398,7.81499 -5.58193,14.61601 -9.89385,20.40306c-3.5453,4.75816 -7.74343,8.79822 -12.59439,12.12016l37.36439,2.58984l2.35,-33.92z M269.7504,489.15741c-0.8906,-0.87867 -1.74073,-1.79781 -2.5504,-2.75741c-3.6,-4.26667 -5.53333,-9.46667 -5.8,-15.6l-34,0c0.26863,5.64121 1.14612,10.84269 2.63248,15.60444z M374.32882,324.79678c0.71412,5.15644 1.07118,10.35751 1.07118,15.60322c0,6.15216 -0.46545,12.30433 -1.39634,18.45649l22.84634,1.58351l2.35,-33.92z');

    var t = PathItem.create('M283.8,165.2L283.8,227.2L248.6,227.2L248.6,257.2L283.8,257.2L283.8,388.8Q283.8,403.2,286.6,412Q289.4,420.8,295.2,425.6Q301,430.4,310.4,432.2Q319.8,434,333,434L359,434L359,404L343.4,404Q335.4,404,330.4,403.4Q325.4,402.8,322.6,400.8Q319.8,398.8,318.8,395.2Q317.8,391.6,317.8,385.6L317.8,257.2L359,257.2L359,227.2L317.8,227.2L317.8,165.2Z');
    var k = PathItem.create('M402.62,200.96L193.81,395.8L217,420.66L274.33,367.18L317.8,370.38L287.14,495.82L316.61,527.41L352.79,373.29L463.72,381.94L463.72,381.94L432.61,348.6L302.11,341.25L425.82,225.82L402.62,200.96Z');
    compareBoolean(function() { return t.unite(k); },
        'M283.8,165.2l0,62l-35.2,0l0,30l35.2,0l0,54.63059l-89.99,83.96941l23.19,24.86l57.33,-53.48l9.47,0.69712l0,20.92288c0,9.6 0.93333,17.33333 2.8,23.2c1.86667,5.86667 4.73333,10.4 8.6,13.6c2.25382,1.86523 4.91535,3.39071 7.98458,4.57643l-16.04458,65.64357l29.47,31.59l21.9282,-93.41l20.4618,0l0,-30l-13.41924,0l7.20924,-30.71l110.93,8.65l-31.11,-33.34l-114.81,-6.46631l0,-15.52355l108.02,-100.79014l-23.2,-24.86l-43.62,40.70169l0,-14.46169l-41.2,0l0,-62z M317.8,280.10529l0,-22.90529l24.5476,0z');
    compareBoolean(function() { return t.subtract(k); },
        'M283.8,165.2l0,62l-35.2,0l0,30l35.2,0l0,54.63059l34,-31.7253l0,-22.90529l24.5476,0l16.6524,-15.53831l0,-14.46169l-41.2,0l0,-62z M283.8,367.87712l0,20.92288c0,9.6 0.93333,17.33333 2.8,23.2c1.86667,5.86667 4.73333,10.4 8.6,13.6c2.25382,1.86523 4.91535,3.39071 7.98458,4.57643l14.61542,-59.79643z M338.5382,434l20.4618,0l0,-30l-13.41924,0z M317.8,342.13369l0,-15.52355l-15.69,14.63986z');
    compareBoolean(function() { return t.intersect(k); },
        'M283.8,311.83059l0,56.04653l34,2.50288l-14.61542,59.79643c2.19636,0.84851 4.6015,1.52303 7.21542,2.02357c6.26667,1.2 13.8,1.8 22.6,1.8l5.5382,0l7.04256,-30l-2.18076,0c-5.33333,0 -9.66667,-0.2 -13,-0.6c-3.33333,-0.4 -5.93333,-1.26667 -7.8,-2.6c-1.86667,-1.33333 -3.13333,-3.2 -3.8,-5.6c-0.66667,-2.4 -1,-5.6 -1,-9.6l0,-43.46631l-15.69,-0.88369l15.69,-14.63986l0,-46.50485z M342.3476,257.2l16.6524,0l0,-15.53831z');

    var s = PathItem.create('M239.6,157.4L239.6,443L273.6,443L273.6,326.2Q273.6,312.2,277.4,300.4Q281.2,288.6,288.8,279.8Q296.4,271,307.8,266.2Q319.2,261.4,334.8,261.4Q354.4,261.4,365.6,272.6Q376.8,283.8,376.8,303L376.8,443L410.8,443L410.8,307Q410.8,290.2,407.4,276.4Q404,262.6,395.6,252.6Q387.2,242.6,373.6,237Q360,231.4,339.6,231.4Q330.4,231.4,320.6,233.4Q310.8,235.4,301.8,239.6Q292.8,243.8,285.6,250.4Q278.4,257,274.4,266.6L273.6,266.6L273.6,157.4Z');
    var h = PathItem.create('M255.73,157.4Q241.22,157.4,228.27,163.84Q215.15,170.35,205.35,186.39Q197.63,199.02,197.37,210.58Q197.1,222.14,201.68,232.67Q206.26,243.2,214.59,252.98Q222.91,262.75,232.38,271.81Q241.16,280.46,250.04,288.93Q258.93,297.41,265.14,305.66Q271.35,313.9,273.62,322.08Q275.88,330.27,271.3,337.78Q267.13,344.6,261.11,346.78Q256.67,348.39,251.68,348.39Q249.91,348.39,248.06,348.19Q241.04,347.41,233.75,344.37Q226.47,341.33,219.99,337.37Q212.82,332.99,206.76,327.18Q200.7,321.37,196.98,314.41Q193.26,307.45,192.95,299.52Q192.64,291.6,197.17,282.65L168.15,264.92Q159.38,281.59,158.61,295.89Q157.84,310.18,163.72,322.44Q169.59,334.7,180.65,344.98Q191.72,355.25,206.74,364.43Q218.34,371.52,231.89,375.81Q244.73,379.88,257.1,379.88Q257.79,379.88,258.48,379.86Q271.51,379.62,283.32,373.72Q295.14,367.81,303.27,354.5Q313.7,337.43,311,322.66Q308.31,307.89,298.74,294.78Q289.17,281.67,275.92,269.59Q262.68,257.51,252.07,246.11Q241.46,234.7,236.47,223.69Q231.48,212.67,238.36,201.41Q243.36,193.22,250.69,190.43Q255.91,188.44,261.46,188.44Q263.71,188.44,266,188.77Q273.99,189.89,282.12,193.46Q290.26,197.02,297.08,201.19Q305.96,206.61,313.28,213.43Q320.6,220.24,324.86,228.24Q329.12,236.23,329.15,245.39Q329.18,254.54,323.68,264.31L352.69,282.04Q362.02,265.23,362.86,250.05Q363.7,234.86,357.74,221.6Q351.77,208.35,340.34,197.15Q328.9,185.94,314.22,176.97Q300.91,168.84,285.86,163.16Q270.82,157.49,256.1,157.4Q255.92,157.4,255.73,157.4Z');
    compareBoolean(function() { return s.unite(h); },
        'M239.6,157.4l0,2.11816c-3.87125,1.05014 -7.64792,2.49075 -11.33,4.32184c-8.74667,4.34 -16.38667,11.85667 -22.92,22.55c-5.14667,8.42 -7.80667,16.48333 -7.98,24.19c-0.18,7.70667 1.25667,15.07 4.31,22.09c3.05333,7.02 7.35667,13.79 12.91,20.31c5.54667,6.51333 11.47667,12.79 17.79,18.83c2.40106,2.36551 4.80772,4.72092 7.22,7.06623l0,67.58233c-1.93529,-0.57307 -3.88529,-1.26926 -5.85,-2.08856c-4.85333,-2.02667 -9.44,-4.36 -13.76,-7c-4.78,-2.92 -9.19,-6.31667 -13.23,-10.19c-4.04,-3.87333 -7.3,-8.13 -9.78,-12.77c-2.48,-4.64 -3.82333,-9.60333 -4.03,-14.89c-0.20667,-5.28 1.2,-10.90333 4.22,-16.87l-29.02,-17.73c-5.84667,11.11333 -9.02667,21.43667 -9.54,30.97c-0.51333,9.52667 1.19,18.37667 5.11,26.55c3.91333,8.17333 9.55667,15.68667 16.93,22.54c7.38,6.84667 16.07667,13.33 26.09,19.45c7.73333,4.72667 16.11667,8.52 25.15,11.38c2.58428,0.81916 5.15428,1.51467 7.71,2.08652l0,65.10348l34,0l0,-65.44295c3.30609,-0.97242 6.54609,-2.25143 9.72,-3.83705c7.88,-3.94 14.53,-10.34667 19.95,-19.22c6.95333,-11.38 9.53,-21.99333 7.73,-31.84c-1.79333,-9.84667 -5.88,-19.14 -12.26,-27.88c-3.36811,-4.614 -7.0781,-9.13232 -11.12995,-13.55494c0.38857,-0.4814 0.78522,-0.95642 1.18995,-1.42506c5.06667,-5.86667 11.4,-10.4 19,-13.6c5.08764,-2.14216 10.80266,-3.56732 17.14507,-4.27546c-0.394,0.79205 -0.81569,1.5872 -1.26507,2.38546l29.01,17.73c2.73672,-4.93079 4.92559,-9.7564 6.5666,-14.47683c2.30112,1.43007 4.41559,3.10902 6.3434,5.03683c7.46667,7.46667 11.2,17.6 11.2,30.4l0,140l34,0l0,-136c0,-11.2 -1.13333,-21.4 -3.4,-30.6c-2.26667,-9.2 -6.2,-17.13333 -11.8,-23.8c-5.6,-6.66667 -12.93333,-11.86667 -22,-15.6c-3.63009,-1.49474 -7.62353,-2.69025 -11.98032,-3.58654c-0.87044,-4.05782 -2.16367,-7.99564 -3.87968,-11.81346c-3.98,-8.83333 -9.78,-16.98333 -17.4,-24.45c-7.62667,-7.47333 -16.33333,-14.2 -26.12,-20.18c-8.87333,-5.42 -18.32667,-10.02333 -28.36,-13.81c-4.10454,-1.54739 -8.19121,-2.78309 -12.26,-3.70709l0,-2.05291z M239.6,199.51295l0,30.19548c-1.1819,-2.01601 -2.22524,-4.02215 -3.13,-6.01844c-3.32667,-7.34667 -2.69667,-14.77333 1.89,-22.28c0.40204,-0.65853 0.81537,-1.29088 1.24,-1.89705z M273.6,332.6753l0,-6.4753c0,-1.31917 0.0253,-2.62368 0.07591,-3.91355c0.94573,3.55949 0.92043,7.02244 -0.07591,10.38885z M326.73614,232.33536c-2.02539,0.28825 -4.07078,0.64313 -6.13614,1.06464c-6.53333,1.33333 -12.8,3.4 -18.8,6.2c-6,2.8 -11.4,6.4 -16.2,10.8c-4.8,4.4 -8.53333,9.8 -11.2,16.2l-0.8,0l0,-76.21948c2.82707,0.80022 5.66707,1.82671 8.52,3.07948c5.42667,2.37333 10.41333,4.95 14.96,7.73c5.92,3.61333 11.32,7.69333 16.2,12.24c4.88,4.54 8.74,9.47667 11.58,14.81c0.71467,1.34042 1.34005,2.70554 1.87614,4.09536z');
    compareBoolean(function() { return s.subtract(h); },
        'M239.6,157.4l0,2.11816c5.20563,-1.41211 10.5823,-2.11816 16.13,-2.11816z M239.6,199.51295l0,30.19548c3.16376,5.39652 7.32042,10.86371 12.47,16.40156c7.07333,7.6 15.02333,15.42667 23.85,23.48c4.17006,3.80184 8.06674,7.68019 11.69005,11.63506c0.38857,-0.4814 0.78522,-0.95642 1.18995,-1.42506c5.06667,-5.86667 11.4,-10.4 19,-13.6c5.08764,-2.14216 10.80266,-3.56732 17.14507,-4.27546c2.82084,-5.67063 4.22248,-11.18214 4.20493,-16.53454c-0.01497,-4.56996 -0.81959,-8.92151 -2.41386,-13.05464c-2.02539,0.28825 -4.07078,0.64313 -6.13614,1.06464c-6.53333,1.33333 -12.8,3.4 -18.8,6.2c-6,2.8 -11.4,6.4 -16.2,10.8c-4.8,4.4 -8.53333,9.8 -11.2,16.2l-0.8,0l0,-76.21948c-2.5438,-0.72004 -5.07714,-1.25688 -7.6,-1.61052c-1.52667,-0.22 -3.04,-0.33 -4.54,-0.33c-3.7,0 -7.29,0.66333 -10.77,1.99c-4.29728,1.63566 -7.99395,4.66332 -11.09,9.08295z M239.6,278.87623l0,67.58233c2.85194,0.84451 5.67194,1.42165 8.46,1.73144c1.23333,0.13333 2.44,0.2 3.62,0.2c3.32667,0 6.47,-0.53667 9.43,-1.61c4.01333,-1.45333 7.41,-4.45333 10.19,-9c1.02221,-1.67616 1.78888,-3.37773 2.3,-5.1047l0,-6.4753c0,-1.31917 0.0253,-2.62368 0.07591,-3.91355c-0.01827,-0.06878 -0.03691,-0.1376 -0.05591,-0.20645c-1.51333,-5.45333 -4.34,-10.92667 -8.48,-16.42c-4.14,-5.5 -9.17333,-11.07667 -15.1,-16.73c-3.4916,-3.33038 -6.9716,-6.68164 -10.44,-10.05377z M239.6,377.89652l0,65.10348l34,0l0,-65.44295c-4.89512,1.4398 -9.93512,2.20745 -15.12,2.30295c-0.46,0.01333 -0.92,0.02 -1.38,0.02c-5.75698,0 -11.59032,-0.66116 -17.5,-1.98348z M359.2566,267.56317c2.30112,1.43007 4.41559,3.10902 6.3434,5.03683c7.46667,7.46667 11.2,17.6 11.2,30.4l0,140l34,0l0,-136c0,-11.2 -1.13333,-21.4 -3.4,-30.6c-2.26667,-9.2 -6.2,-17.13333 -11.8,-23.8c-5.6,-6.66667 -12.93333,-11.86667 -22,-15.6c-3.63009,-1.49474 -7.62353,-2.69025 -11.98032,-3.58654c1.14503,5.33787 1.55847,10.88338 1.24032,16.63654c-0.31361,5.66733 -1.51474,11.50505 -3.6034,17.51317z M273.6,159.45291l0,-2.05291l-17.5,0c5.79612,0.03544 11.62946,0.71974 17.5,2.05291z');
    compareBoolean(function() { return s.intersect(h); },
        'M239.6,159.51816l0,39.99479c3.09605,-4.41964 6.79272,-7.44729 11.09,-9.08295c3.48,-1.32667 7.07,-1.99 10.77,-1.99c1.5,0 3.01333,0.11 4.54,0.33c2.52286,0.35364 5.0562,0.89048 7.6,1.61052l0,-30.92761c-5.87054,-1.33317 -11.70388,-2.01747 -17.5,-2.05291l-0.37,0c-5.5477,0 -10.92437,0.70605 -16.13,2.11816z M239.6,229.70844l0,49.1678c3.4684,3.37213 6.9484,6.72338 10.44,10.05377c5.92667,5.65333 10.96,11.23 15.1,16.73c4.14,5.49333 6.96667,10.96667 8.48,16.42c0.019,0.06885 0.03764,0.13767 0.05591,0.20645c0.30745,-7.83617 1.54881,-15.13165 3.72409,-21.88645c2.33097,-7.23827 5.73432,-13.62991 10.21005,-19.17494c-3.62331,-3.95487 -7.51999,-7.83322 -11.69005,-11.63506c-8.82667,-8.05333 -16.77667,-15.88 -23.85,-23.48c-5.14958,-5.53786 -9.30624,-11.00505 -12.47,-16.40156z M239.6,346.45856l0,31.43796c5.90968,1.32232 11.74302,1.98348 17.5,1.98348c0.46,0 0.92,-0.00667 1.38,-0.02c5.18488,-0.0955 10.22488,-0.86315 15.12,-2.30295l0,-44.88175c-0.51112,1.72697 -1.27779,3.42854 -2.3,5.1047c-2.78,4.54667 -6.17667,7.54667 -10.19,9c-2.96,1.07333 -6.10333,1.61 -9.43,1.61c-1.18,0 -2.38667,-0.06667 -3.62,-0.2c-2.78806,-0.30978 -5.60806,-0.88693 -8.46,-1.73144z M324.94507,261.92454c3.13199,-0.34969 6.41696,-0.52454 9.85493,-0.52454c9.69299,0 17.84519,2.05439 24.4566,6.16317c2.08866,-6.00812 3.2898,-11.84584 3.6034,-17.51317c0.31815,-5.75315 -0.09529,-11.29867 -1.24032,-16.63654c-6.52492,-1.34231 -13.86481,-2.01346 -22.01968,-2.01346c-4.19442,0 -8.48237,0.31179 -12.86386,0.93536c1.59427,4.13313 2.39889,8.48468 2.41386,13.05464c0.01755,5.3524 -1.38409,10.86391 -4.20493,16.53454z');

    var c1 = PathItem.create('M318,196.4Q293.6,196.4,275.2,205Q256.8,213.6,244.6,228.6Q232.4,243.6,226.4,263.8Q220.4,284,220.4,307.2Q220.4,330.4,226.6,349.8Q232.8,369.2,245,383.2Q257.2,397.2,275.4,404.8Q293.6,412.4,317.2,412.4Q356.8,412.4,379.8,391.6Q402.8,370.8,408.4,332.4L373.6,332.4Q370.4,356.4,356.2,369.6Q342,382.8,316.8,382.8Q300.8,382.8,289.2,376.4Q277.6,370,270.4,359.4Q263.2,348.8,259.8,335.2Q256.4,321.6,256.4,307.2Q256.4,291.6,259.6,277Q262.8,262.4,270.2,251.2Q277.6,240,290,233.2Q302.4,226.4,320.8,226.4Q342.4,226.4,355.2,237.2Q368,248,372,267.6L407.2,267.6Q405.2,249.2,397.6,235.8Q390,222.4,378.2,213.6Q366.4,204.8,351,200.6Q335.6,196.4,318,196.4Z');
    var c2 = PathItem.create('M292.67,129.19Q280.2,129.19,269.66,132.55Q254.98,137.23,243.94,146.96Q232.9,156.69,225.61,170.89Q218.33,185.1,214.7,202.32Q209.67,226.2,214.3,245.97Q218.92,265.75,231.08,280.78Q243.24,295.81,261.77,305.85Q280.3,315.88,303.01,320.66Q318.85,324,333.51,324Q339.85,324,345.97,323.38Q366.23,321.31,382.44,312.26Q398.66,303.2,409.85,286.96Q421.03,270.72,425.9,247.63Q434.06,208.88,418.45,182.08Q402.84,155.29,366.42,141.89L359.24,175.95Q382.07,184.03,392.06,200.64Q402.05,217.26,396.85,241.92Q393.55,257.57,384.9,267.6Q376.25,277.64,364.39,282.5Q352.53,287.36,338.53,287.88Q336.42,287.96,334.32,287.96Q322.4,287.96,310.43,285.44Q295.16,282.22,281.53,276.08Q267.91,269.94,258.47,260.39Q249.04,250.84,244.94,237.3Q240.84,223.77,244.64,205.76Q249.09,184.63,262.29,174.33Q275.34,164.15,295.02,164.15Q295.26,164.15,295.5,164.16L302.76,129.71Q297.57,129.19,292.67,129.19Z');
    compareBoolean(function() { return c1.unite(c2); },
        'M318,196.4c-16.26667,0 -30.53333,2.86667 -42.8,8.6c-12.26667,5.73333 -22.46667,13.6 -30.6,23.6c-0.41596,0.51143 -0.82652,1.02739 -1.23167,1.54789c-1.04296,-7.29746 -0.61907,-15.42676 1.27167,-24.38789c2.96667,-14.08667 8.85,-24.56333 17.65,-31.43c8.7,-6.78667 19.61,-10.18 32.73,-10.18c0.16,0 0.32,0.00333 0.48,0.01l7.26,-34.45c-3.46,-0.34667 -6.82333,-0.52 -10.09,-0.52c-8.31333,0 -15.98333,1.12 -23.01,3.36c-9.78667,3.12 -18.36,7.92333 -25.72,14.41c-7.36,6.48667 -13.47,14.46333 -18.33,23.93c-4.85333,9.47333 -8.49,19.95 -10.91,31.43c-3.35333,15.92 -3.48667,30.47 -0.4,43.65c2.14848,9.19846 5.51991,17.62649 10.11429,25.28409c-2.67619,11.31292 -4.01429,23.29489 -4.01429,35.94591c0,15.46667 2.06667,29.66667 6.2,42.6c4.13333,12.93333 10.26667,24.06667 18.4,33.4c8.13333,9.33333 18.26667,16.53333 30.4,21.6c12.13333,5.06667 26.06667,7.6 41.8,7.6c26.4,0 47.26667,-6.93333 62.6,-20.8c15.33333,-13.86667 24.86667,-33.6 28.6,-59.2l-34.8,0c-2.13333,16 -7.93333,28.4 -17.4,37.2c-9.46667,8.8 -22.6,13.2 -39.4,13.2c-10.66667,0 -19.86667,-2.13333 -27.6,-6.4c-7.73333,-4.26667 -14,-9.93333 -18.8,-17c-4.8,-7.06667 -8.33333,-15.13333 -10.6,-24.2c-2.26667,-9.06667 -3.4,-18.4 -3.4,-28c0,-1.47564 0.02147,-2.94456 0.06442,-4.40678c1.72272,1.0548 3.49125,2.07373 5.30558,3.05678c12.35333,6.68667 26.1,11.62333 41.24,14.81c10.56,2.22667 20.72667,3.34 30.5,3.34c4.22667,0 8.38,-0.20667 12.46,-0.62c13.50667,-1.38 25.66333,-5.08667 36.47,-11.12c10.81333,-6.04 19.95,-14.47333 27.41,-25.3c7.45333,-10.82667 12.80333,-23.93667 16.05,-39.33c5.44,-25.83333 2.95667,-47.68333 -7.45,-65.55c-10.40667,-17.86 -27.75,-31.25667 -52.03,-40.19l-7.18,34.06c15.22,5.38667 26.16,13.61667 32.82,24.69c5.88091,9.78386 7.81382,21.65737 5.79873,35.62055c-0.0857,-0.154 -0.17194,-0.30752 -0.25873,-0.46055c-5.06667,-8.93333 -11.53333,-16.33333 -19.4,-22.2c-7.86667,-5.86667 -16.93333,-10.2 -27.2,-13c-10.26667,-2.8 -21.26667,-4.2 -33,-4.2z M263.24177,264.77742c1.88233,-4.87963 4.20174,-9.40544 6.95823,-13.57742c4.93333,-7.46667 11.53333,-13.46667 19.8,-18c8.26667,-4.53333 18.53333,-6.8 30.8,-6.8c14.4,0 25.86667,3.6 34.4,10.8c8.53333,7.2 14.13333,17.33333 16.8,30.4l12.9,0c-5.76667,6.69333 -12.60333,11.66 -20.51,14.9c-7.90667,3.24 -16.52667,5.03333 -25.86,5.38c-1.40667,0.05333 -2.81,0.08 -4.21,0.08c-7.94667,0 -15.91,-0.84 -23.89,-2.52c-10.18,-2.14667 -19.81333,-5.26667 -28.9,-9.36c-6.90089,-3.11097 -12.99696,-6.8785 -18.28823,-11.30258z');
    compareBoolean(function() { return c1.subtract(c2); },
        'M318,196.4c-16.26667,0 -30.53333,2.86667 -42.8,8.6c-12.26667,5.73333 -22.46667,13.6 -30.6,23.6c-0.41596,0.51143 -0.82652,1.02739 -1.23167,1.54789c0.35446,2.48012 0.87835,4.86415 1.57167,7.15211c2.73333,9.02667 7.24333,16.72333 13.53,23.09c1.51034,1.52794 3.10093,2.99041 4.77177,4.38742c1.88233,-4.87963 4.20174,-9.40544 6.95823,-13.57742c4.93333,-7.46667 11.53333,-13.46667 19.8,-18c8.26667,-4.53333 18.53333,-6.8 30.8,-6.8c14.4,0 25.86667,3.6 34.4,10.8c8.53333,7.2 14.13333,17.33333 16.8,30.4l12.9,0c5.76667,-6.68667 9.75,-15.24667 11.95,-25.68c0.40553,-1.92316 0.74178,-3.80964 1.00873,-5.65945c-0.0857,-0.154 -0.17194,-0.30752 -0.25873,-0.46055c-5.06667,-8.93333 -11.53333,-16.33333 -19.4,-22.2c-7.86667,-5.86667 -16.93333,-10.2 -27.2,-13c-10.26667,-2.8 -21.26667,-4.2 -33,-4.2z M224.41429,271.25409c-2.67619,11.31292 -4.01429,23.29489 -4.01429,35.94591c0,15.46667 2.06667,29.66667 6.2,42.6c4.13333,12.93333 10.26667,24.06667 18.4,33.4c8.13333,9.33333 18.26667,16.53333 30.4,21.6c12.13333,5.06667 26.06667,7.6 41.8,7.6c26.4,0 47.26667,-6.93333 62.6,-20.8c15.33333,-13.86667 24.86667,-33.6 28.6,-59.2l-34.8,0c-2.13333,16 -7.93333,28.4 -17.4,37.2c-9.46667,8.8 -22.6,13.2 -39.4,13.2c-10.66667,0 -19.86667,-2.13333 -27.6,-6.4c-7.73333,-4.26667 -14,-9.93333 -18.8,-17c-4.8,-7.06667 -8.33333,-15.13333 -10.6,-24.2c-2.26667,-9.06667 -3.4,-18.4 -3.4,-28c0,-1.47564 0.02147,-2.94456 0.06442,-4.40678c-10.0069,-6.12711 -18.46838,-13.46486 -25.38442,-22.01322c-2.4518,-3.03047 -4.6737,-6.20578 -6.66571,-9.52591z');
    compareBoolean(function() { return c1.intersect(c2); },
        'M243.36833,230.14789c-7.51679,9.6568 -13.1729,20.87417 -16.96833,33.65211c-0.72818,2.45155 -1.39009,4.93625 -1.98571,7.45409c1.992,3.32013 4.21391,6.49544 6.66571,9.52591c6.91604,8.54837 15.37752,15.88611 25.38442,22.01322c0.25975,-8.84319 1.30494,-17.44093 3.13558,-25.79322c0.94134,-4.29486 2.15526,-8.36905 3.64177,-12.22258c-1.67084,-1.39701 -3.26143,-2.85948 -4.77177,-4.38742c-6.28667,-6.36667 -10.79667,-14.06333 -13.53,-23.09c-0.69332,-2.28796 -1.21721,-4.67199 -1.57167,-7.15211z M384.9,267.6l22.3,0c-1.31049,-12.05654 -4.42425,-22.50303 -9.34127,-31.33945c-0.26695,1.84981 -0.6032,3.7363 -1.00873,5.65945c-2.2,10.43333 -6.18333,18.99333 -11.95,25.68z');

    var a1 = PathItem.create('M356.6,301.2L356.6,336Q356.6,343.2,353,351.4Q349.4,359.6,341.8,366.6Q334.2,373.6,322.2,378.2Q310.2,382.8,293.4,382.8Q285.8,382.8,278.4,381.2Q271,379.6,265.6,376Q260.2,372.4,256.8,366.6Q253.4,360.8,253.4,352.4Q253.4,339.6,259.2,332.2Q265,324.8,274.2,320.8Q283.4,316.8,295,315Q306.6,313.2,318,311.8Q329.4,310.4,339.8,308.2Q350.2,306,356.6,301.2Z M313,196.4Q295.8,196.4,280.2,200Q264.6,203.6,252.6,211.8Q240.6,220,233.4,233.2Q226.2,246.4,225.4,265.6L259.4,265.6Q260.6,242.8,275,234.6Q289.4,226.4,311,226.4Q319,226.4,327.2,227.6Q335.4,228.8,342,232.4Q348.6,236,352.8,242.8Q357,249.6,357,260.8Q357,270.4,351.4,275.4Q345.8,280.4,336.2,283Q326.6,285.6,314,287Q301.4,288.4,287.4,291.2Q274.2,293.6,261.6,297.2Q249,300.8,239.2,307.8Q229.4,314.8,223.4,326Q217.4,337.2,217.4,354.8Q217.4,370.4,222.8,381.4Q228.2,392.4,237.6,399.4Q247,406.4,259.4,409.4Q271.8,412.4,285.8,412.4Q307.4,412.4,326,405Q344.6,397.6,358.6,380.8Q358.6,397.6,366.4,405Q374.2,412.4,387.4,412.4Q403,412.4,411.8,407.2L411.8,380.8Q405.8,382.8,401.4,382.8Q393.4,382.8,392,377.2Q390.6,371.6,390.6,359.6L390.6,253.2Q390.6,235.2,383.8,224Q377,212.8,365.8,206.6Q354.6,200.4,340.8,198.4Q327,196.4,313,196.4Z');
    var a2 = PathItem.create('M318.28,246.42L351.45,256.96Q358.31,259.14,365.03,265.05Q371.76,270.96,376.13,280.33Q380.5,289.69,381.25,302.52Q382,315.35,376.91,331.36Q374.61,338.6,370.85,345.17Q367.08,351.74,362.02,355.8Q356.95,359.85,350.39,361.34Q347.98,361.88,345.36,361.88Q340.89,361.88,335.83,360.28Q323.63,356.4,318.33,348.63Q313.04,340.87,312.01,330.89Q310.99,320.91,312.78,309.31Q314.58,297.71,316.7,286.42Q318.81,275.13,319.87,264.55Q320.92,253.97,318.28,246.42Z M262.73,196Q257.55,196,252.94,197.12Q240.21,200.21,230.91,209.01Q221.61,217.81,215.52,230.35Q209.44,242.9,205.2,256.24Q199.99,272.63,198.7,288.59Q197.41,304.55,201.59,318.47Q205.77,332.39,216.17,343.25Q226.57,354.11,244.63,360.68L254.92,328.28Q233.56,320.23,230.1,304.02Q226.65,287.82,233.19,267.23Q235.61,259.61,239.23,252.16Q242.86,244.7,248.29,239.5Q253.72,234.3,261.47,232.36Q264.3,231.65,267.51,231.65Q273.11,231.65,279.9,233.81Q289.05,236.71,292.12,243.56Q295.19,250.42,294.76,260.35Q294.33,270.29,291.85,282.72Q289.37,295.15,287.8,309.34Q286.09,322.65,285.7,335.75Q285.32,348.85,289.02,360.31Q292.73,371.77,301.59,380.88Q310.44,389.99,327.22,395.32Q337.96,398.73,347.26,398.73Q350.84,398.73,354.2,398.22Q366.32,396.41,375.84,389.57Q385.36,382.73,391.97,371.82Q398.59,360.91,402.82,347.57Q409.36,326.98,407.94,307.01Q406.52,287.05,394.75,268.62L394.75,268.62Q402.67,271.13,408.98,271.13Q415.42,271.13,420.18,268.51Q429.59,263.32,433.59,250.74Q438.31,235.87,436.02,225.91L410.86,217.91L410.86,217.91Q410.95,224.24,409.62,228.43Q407.3,235.71,401.95,235.71Q401.69,235.71,401.43,235.69Q395.67,235.33,384.23,231.7L282.83,199.48Q271.88,196,262.73,196Z');
    compareBoolean(function() { return a1.unite(a2); },
        'M313,196.4c-10.54499,0 -20.63894,1.01484 -30.28184,3.04453c-7.2566,-2.29635 -13.91932,-3.44453 -19.98816,-3.44453c-3.45333,0 -6.71667,0.37333 -9.79,1.12c-8.48667,2.06 -15.83,6.02333 -22.03,11.89c-6.2,5.86667 -11.33,12.98 -15.39,21.34c-4.05333,8.36667 -7.49333,16.99667 -10.32,25.89c-3.47333,10.92667 -5.64,21.71 -6.5,32.35c-0.86,10.64 0.10333,20.6 2.89,29.88c2.78667,9.28 7.64667,17.54 14.58,24.78c0.56619,0.59124 1.14941,1.17293 1.74966,1.7451c-0.34644,3.08353 -0.51966,6.35183 -0.51966,9.8049c0,10.4 1.8,19.26667 5.4,26.6c3.6,7.33333 8.53333,13.33333 14.8,18c6.26667,4.66667 13.53333,8 21.8,10c8.26667,2 17.06667,3 26.4,3c14.4,0 27.8,-2.46667 40.2,-7.4c4.64254,-1.84703 9.07016,-4.13328 13.28283,-6.85874c2.74193,0.39249 5.40098,0.58874 7.97717,0.58874c2.38667,0 4.7,-0.17 6.94,-0.51c2.36408,-0.35305 4.65397,-0.84964 6.86967,-1.48975c1.27956,3.35579 3.05634,6.11238 5.33033,8.26975c5.2,4.93333 12.2,7.4 21,7.4c10.4,0 18.53333,-1.73333 24.4,-5.2l0,-26.4c-4,1.33333 -7.46667,2 -10.4,2c-5.33333,0 -8.46667,-1.86667 -9.4,-5.6c-0.29733,-1.18932 -0.5473,-2.59514 -0.74991,-4.21747c0.24278,-0.38358 0.48275,-0.77109 0.71991,-1.16253c4.41333,-7.27333 8.03,-15.35667 10.85,-24.25c4.36,-13.72667 6.06667,-27.24667 5.12,-40.56c-0.94667,-13.30667 -5.34333,-26.10333 -13.19,-38.39c5.28,1.67333 10.02333,2.51 14.23,2.51c4.29333,0 8.02667,-0.87333 11.2,-2.62c6.27333,-3.46 10.74333,-9.38333 13.41,-17.77c3.14667,-9.91333 3.95667,-18.19 2.43,-24.83l-25.16,-8c0.06,4.22 -0.35333,7.72667 -1.24,10.52c-1.54667,4.85333 -4.10333,7.28 -7.67,7.28c-0.17333,0 -0.34667,-0.00667 -0.52,-0.02c-3.22245,-0.2014 -7.77824,-1.17041 -13.66735,-2.90702c-1.03676,-3.21178 -2.35764,-6.13944 -3.96265,-8.78298c-4.53333,-7.46667 -10.53333,-13.26667 -18,-17.4c-7.46667,-4.13333 -15.8,-6.86667 -25,-8.2c-9.2,-1.33333 -18.46667,-2 -27.8,-2z M233.72204,265.6l25.67796,0c0.8,-15.2 6,-25.53333 15.6,-31c0.88297,-0.5028 1.78624,-0.98248 2.70981,-1.43904c0.71944,0.19707 1.4495,0.41341 2.19019,0.64904c6.1,1.93333 10.17333,5.18333 12.22,9.75c2.04667,4.57333 2.92667,10.17 2.64,16.79c-0.28667,6.62667 -1.25667,14.08333 -2.91,22.37c-0.51367,2.57457 -0.99806,5.20577 -1.45318,7.8936c-0.99354,0.19006 -1.99248,0.38553 -2.99682,0.5864c-8.8,1.6 -17.4,3.6 -25.8,6c-8.4,2.4 -15.86667,5.93333 -22.4,10.6c-2.02434,1.44596 -3.92707,3.02632 -5.70819,4.74109c-1.60844,-2.62254 -2.73904,-5.4629 -3.39181,-8.52109c-2.3,-10.8 -1.27,-23.06333 3.09,-36.79c0.17276,-0.54398 0.35011,-1.08732 0.53204,-1.63z M356.9816,259.33651c0.01227,0.4814 0.0184,0.96924 0.0184,1.46349c0,6.4 -1.86667,11.26667 -5.6,14.6c-3.73333,3.33333 -8.8,5.86667 -15.2,7.6c-5.71842,1.54874 -12.23519,2.77814 -19.55031,3.6882c0.01675,-0.08941 0.03352,-0.17881 0.05031,-0.2682c1.40667,-7.52667 2.46333,-14.81667 3.17,-21.87c0.7,-7.05333 0.17,-13.09667 -1.59,-18.13l33.17,10.54c1.85151,0.58838 3.69538,1.38055 5.5316,2.37651z M356.6,301.2l0,34.8c0,4.8 -1.2,9.93333 -3.6,15.4c-1.60966,3.66645 -3.81909,7.15297 -6.62829,10.45955c-0.33459,0.01363 -0.67182,0.02045 -1.01171,0.02045c-2.98,0 -6.15667,-0.53333 -9.53,-1.6c-8.13333,-2.58667 -13.96667,-6.47 -17.5,-11.65c-3.52667,-5.17333 -5.63333,-11.08667 -6.32,-17.74c-0.58502,-5.72404 -0.47676,-11.84777 0.32479,-18.37119c1.8925,-0.2478 3.7809,-0.4874 5.66521,-0.71881c7.6,-0.93333 14.86667,-2.13333 21.8,-3.6c6.93333,-1.46667 12.53333,-3.8 16.8,-7z M303.12737,382.37884c-3.09597,0.28078 -6.33842,0.42116 -9.72737,0.42116c-5.06667,0 -10.06667,-0.53333 -15,-1.6c-4.93333,-1.06667 -9.2,-2.8 -12.8,-5.2c-3.6,-2.4 -6.53333,-5.53333 -8.8,-9.4c-2.26667,-3.86667 -3.4,-8.6 -3.4,-14.2c0,-8.53333 1.93333,-15.26667 5.8,-20.2c3.86667,-4.93333 8.86667,-8.73333 15,-11.4c3.92731,-1.70753 8.18263,-3.11438 12.76596,-4.22055c-0.65483,6.42723 -1.07682,12.81741 -1.26596,19.17055c-0.25333,8.73333 0.85333,16.92 3.32,24.56c2.47333,7.64 6.66333,14.49667 12.57,20.57c0.49393,0.50844 1.00639,1.00805 1.53737,1.49884z');
    compareBoolean(function() { return a1.subtract(a2); },
        'M313,196.4c-10.54499,0 -20.63894,1.01484 -30.28184,3.04453c0.03727,0.01179 0.07455,0.02362 0.11184,0.03547l101.4,32.22c1.22652,0.38918 2.40407,0.75018 3.53265,1.08298c-1.03676,-3.21178 -2.35764,-6.13944 -3.96265,-8.78298c-4.53333,-7.46667 -10.53333,-13.26667 -18,-17.4c-7.46667,-4.13333 -15.8,-6.86667 -25,-8.2c-9.2,-1.33333 -18.46667,-2 -27.8,-2z M233.72204,265.6l25.67796,0c0.8,-15.2 6,-25.53333 15.6,-31c0.88297,-0.5028 1.78624,-0.98248 2.70981,-1.43904c-3.67741,-1.00731 -7.07735,-1.51096 -10.19981,-1.51096c-2.14,0 -4.15333,0.23667 -6.04,0.71c-5.16667,1.29333 -9.56,3.67333 -13.18,7.14c-3.62,3.46667 -6.64,7.68667 -9.06,12.66c-2.15491,4.43482 -3.99089,8.91482 -5.50796,13.44z M356.9816,259.33651c0.01227,0.4814 0.0184,0.96924 0.0184,1.46349c0,6.4 -1.86667,11.26667 -5.6,14.6c-3.73333,3.33333 -8.8,5.86667 -15.2,7.6c-5.71842,1.54874 -12.23519,2.77814 -19.55031,3.6882c-1.39405,7.43971 -2.68394,14.98031 -3.86969,22.6218c-0.16668,1.08014 -0.31508,2.14974 -0.44521,3.20881c1.8925,-0.2478 3.7809,-0.4874 5.66521,-0.71881c7.6,-0.93333 14.86667,-2.13333 21.8,-3.6c6.93333,-1.46667 12.53333,-3.8 16.8,-7l0,34.8c0,4.8 -1.2,9.93333 -3.6,15.4c-1.60966,3.66645 -3.81909,7.15297 -6.62829,10.45955c1.38484,-0.05642 2.72427,-0.22961 4.01829,-0.51955c4.37333,-0.99333 8.25,-2.84 11.63,-5.54c3.37333,-2.70667 6.31667,-6.25 8.83,-10.63c2.50667,-4.38 4.52667,-8.98333 6.06,-13.81c3.39333,-10.67333 4.84,-20.28667 4.34,-28.84c-0.5,-8.55333 -2.20667,-15.95 -5.12,-22.19c-2.91333,-6.24667 -6.61333,-11.34 -11.1,-15.28c-2.66627,-2.34489 -5.34907,-4.24939 -8.0484,-5.71349z M290.39682,290.6136c-0.99354,0.19006 -1.99248,0.38553 -2.99682,0.5864c-8.8,1.6 -17.4,3.6 -25.8,6c-8.4,2.4 -15.86667,5.93333 -22.4,10.6c-2.02434,1.44596 -3.92707,3.02632 -5.70819,4.74109c4.07526,6.64466 11.21799,11.89096 21.42819,15.73891l-10.29,32.4c-11.05679,-4.02232 -19.96023,-9.25062 -26.71034,-15.6849c-0.34644,3.08353 -0.51966,6.35183 -0.51966,9.8049c0,10.4 1.8,19.26667 5.4,26.6c3.6,7.33333 8.53333,13.33333 14.8,18c6.26667,4.66667 13.53333,8 21.8,10c8.26667,2 17.06667,3 26.4,3c14.4,0 27.8,-2.46667 40.2,-7.4c4.64254,-1.84703 9.07016,-4.13328 13.28283,-6.85874c-3.85696,-0.55211 -7.87791,-1.49253 -12.06283,-2.82126c-10.25015,-3.25586 -18.28103,-7.56958 -24.09263,-12.94116c-3.09597,0.28078 -6.33842,0.42116 -9.72737,0.42116c-5.06667,0 -10.06667,-0.53333 -15,-1.6c-4.93333,-1.06667 -9.2,-2.8 -12.8,-5.2c-3.6,-2.4 -6.53333,-5.53333 -8.8,-9.4c-2.26667,-3.86667 -3.4,-8.6 -3.4,-14.2c0,-8.53333 1.93333,-15.26667 5.8,-20.2c3.86667,-4.93333 8.86667,-8.73333 15,-11.4c3.92731,-1.70753 8.18263,-3.11438 12.76596,-4.22055c0.24533,-2.40795 0.52335,-4.8211 0.83404,-7.23945c0.72148,-6.52089 1.58709,-12.76302 2.59682,-18.7264z M361.06967,396.73025c1.27956,3.35579 3.05634,6.11238 5.33033,8.26975c5.2,4.93333 12.2,7.4 21,7.4c10.4,0 18.53333,-1.73333 24.4,-5.2l0,-26.4c-4,1.33333 -7.46667,2 -10.4,2c-5.33333,0 -8.46667,-1.86667 -9.4,-5.6c-0.29733,-1.18932 -0.5473,-2.59514 -0.74991,-4.21747c-4.2683,6.74372 -9.40499,12.27288 -15.41009,16.58747c-4.48973,3.22582 -9.41317,5.61256 -14.77033,7.16025z');
    compareBoolean(function() { return a1.intersect(a2); },
        'M282.71816,199.44453c-0.84283,0.1774 -1.68222,0.36256 -2.51816,0.55547c-10.4,2.4 -19.6,6.33333 -27.6,11.8c-8,5.46667 -14.4,12.6 -19.2,21.4c-4.8,8.8 -7.46667,19.6 -8,32.4l8.32204,0c1.51707,-4.52518 3.35305,-9.00518 5.50796,-13.44c2.42,-4.97333 5.44,-9.19333 9.06,-12.66c3.62,-3.46667 8.01333,-5.84667 13.18,-7.14c1.88667,-0.47333 3.9,-0.71 6.04,-0.71c3.12246,0 6.5224,0.50365 10.19981,1.51096c9.11791,-4.50731 20.21464,-6.76096 33.29019,-6.76096c5.33333,0 10.73333,0.4 16.2,1.2c5.46667,0.8 10.4,2.4 14.8,4.8c4.4,2.4 8,5.86667 10.8,10.4c2.61465,4.23325 4.00852,9.74542 4.1816,16.53651c2.69933,1.4641 5.38213,3.3686 8.0484,5.71349c4.48667,3.94 8.18667,9.03333 11.1,15.28c2.91333,6.24 4.62,13.63667 5.12,22.19c0.5,8.55333 -0.94667,18.16667 -4.34,28.84c-1.53333,4.82667 -3.55333,9.43 -6.06,13.81c-2.51333,4.38 -5.45667,7.92333 -8.83,10.63c-3.38,2.7 -7.25667,4.54667 -11.63,5.54c-1.29402,0.28995 -2.63345,0.46313 -4.01829,0.51955c-1.37931,1.62353 -2.90321,3.20368 -4.57171,4.74045c-5.06667,4.66667 -11.6,8.53333 -19.6,11.6c-5.57932,2.13874 -11.93686,3.53169 -19.07263,4.17884c5.8116,5.37159 13.84248,9.68531 24.09263,12.94116c4.18492,1.32873 8.20587,2.26915 12.06283,2.82126c7.03917,-4.55411 13.47822,-10.33452 19.31717,-17.34126c0,6.30217 0.82322,11.61226 2.46967,15.93025c5.35715,-1.54768 10.2806,-3.93443 14.77033,-7.16025c6.0051,-4.31459 11.14179,-9.84374 15.41009,-16.58747c-0.43339,-3.47023 -0.65009,-7.93108 -0.65009,-13.38253l0,-106.4c0,-7.75146 -0.94578,-14.55713 -2.83735,-20.41702c-1.12858,-0.3328 -2.30613,-0.69379 -3.53265,-1.08298l-101.4,-32.22c-0.0373,-0.01185 -0.07458,-0.02368 -0.11184,-0.03547z M316.64969,286.6882c-0.87189,0.10847 -1.75512,0.2124 -2.64969,0.3118c-7.49609,0.8329 -15.36381,2.03743 -23.60318,3.6136c-1.00974,5.96338 -1.87535,12.20551 -2.59682,18.7264c-0.3107,2.41835 -0.58871,4.8315 -0.83404,7.23945c2.57452,-0.62135 5.25253,-1.14784 8.03404,-1.57945c5.81597,-0.90248 11.59423,-1.72954 17.33479,-2.48119c0.13013,-1.05907 0.27853,-2.12867 0.44521,-3.20881c1.18575,-7.64149 2.47565,-15.18209 3.86969,-22.6218z M233.49181,312.54109c-3.96726,3.81947 -7.3312,8.30577 -10.09181,13.45891c-2.82282,5.26926 -4.6496,11.60096 -5.48034,18.9951c6.75011,6.43428 15.65356,11.66258 26.71034,15.6849l10.29,-32.4c-10.2102,-3.84795 -17.35294,-9.09425 -21.42819,-15.73891z');

    var m = PathItem.create('M280.2,194.4Q236.6,194.4,213.4,229.6L212.6,229.6L212.6,199.2L180.6,199.2L180.6,406L214.6,406L214.6,277.2Q214.6,271.2,217.6,262.2Q220.6,253.2,227.2,244.8Q233.8,236.4,244.4,230.4Q255,224.4,269.8,224.4Q281.4,224.4,288.8,227.8Q296.2,231.2,300.6,237.4Q305,243.6,306.8,252Q308.6,260.4,308.6,270.4L308.6,406L342.6,406L342.6,277.2Q342.6,253.2,357,238.8Q371.4,224.4,396.6,224.4Q409,224.4,416.8,228Q424.6,231.6,429,237.8Q433.4,244,435,252.4Q436.6,260.8,436.6,270.4L436.6,406L470.6,406L470.6,254.4Q470.6,238.4,465.6,227Q460.6,215.6,451.6,208.4Q442.6,201.2,430,197.8Q417.4,194.4,401.8,194.4Q381.4,194.4,364.4,203.6Q347.4,212.8,337,229.6Q330.6,210.4,315,202.4Q299.4,194.4,280.2,194.4Z').reduce();
    var s = PathItem.create('M374.87,186.92Q363.85,186.92,351.66,188.55Q338.18,190.36,324.95,195.56Q311.73,200.76,301.75,209.16Q291.78,217.56,286.33,229.59Q280.88,241.62,282.95,257.08Q285.61,276.91,297,286.68Q308.4,296.45,324.09,300.61Q339.78,304.76,357.68,305.79Q375.58,306.82,391,309Q406.43,311.17,417.24,316.58Q428.05,321.99,429.81,335.07Q431.08,344.59,427.15,351.37Q423.21,358.15,416.3,362.31Q409.39,366.46,400.83,368.82Q392.27,371.18,384.34,372.24Q377.44,373.17,370.68,373.17Q367.35,373.17,364.05,372.94Q354.07,372.26,345.73,368.74Q337.39,365.21,331.6,358.12Q325.8,351.03,323.92,339.98L290.22,344.49Q293.56,363.42,302.48,375.74Q311.4,388.07,324.38,394.6Q337.36,401.13,353.3,402.63Q359.61,403.22,366.09,403.22Q375.99,403.22,386.29,401.84Q401.75,399.77,417.01,394.7Q432.27,389.63,443.75,380.42Q455.23,371.22,461.32,357.89Q467.4,344.57,464.9,325.93Q462.94,311.26,455.86,302.12Q448.79,292.98,438.59,287.69Q428.4,282.39,415.78,280.05Q403.15,277.7,390.09,276.63Q377.83,275.45,365.59,274.46Q353.35,273.48,343.33,270.99Q333.31,268.5,326.4,263.57Q319.48,258.65,318.31,249.92Q317.25,242,320.55,236.51Q323.85,231.02,329.8,227.2Q335.74,223.38,343.31,221.15Q350.88,218.92,358.42,217.92Q363.55,217.23,368.69,217.23Q371.9,217.23,375.11,217.5Q383.48,218.19,390.75,221.25Q398.02,224.31,403.25,230.27Q408.49,236.23,410.61,246.03L444.31,241.52Q440.63,223.04,432.21,211.46Q423.8,199.87,411.52,194.05Q399.24,188.23,384.17,187.22Q379.63,186.92,374.87,186.92Z').reduce();
    compareBoolean(function() { return m.unite(s); },
        'M280.2,194.4c-29.06667,0 -51.33333,11.73333 -66.8,35.2l-0.8,0l0,-30.4l-32,0l0,206.8l34,0l0,-128.8c0,-4 1,-9 3,-15c2,-6 5.2,-11.8 9.6,-17.4c4.4,-5.6 10.13333,-10.4 17.2,-14.4c7.06667,-4 15.53333,-6 25.4,-6c7.05125,0 12.93858,0.94223 17.66197,2.82669c-0.39231,0.77574 -0.76963,1.56351 -1.13197,2.36331c-3.63333,8.02 -4.76,17.18333 -3.38,27.49c1.77333,13.22 6.45667,23.08667 14.05,29.6c3.55396,3.04581 7.42063,5.68269 11.6,7.91066l0,47.43958l-18.38,2.45976c2.22667,12.62 6.31333,23.03667 12.26,31.25c1.90161,2.62857 3.94161,5.05945 6.12,7.29262l0,22.96738l34,0l0,-5.01536c3.448,0.75011 7.01467,1.29856 10.7,1.64536c4.20667,0.39333 8.47,0.59 12.79,0.59c6.6,0 13.33333,-0.46 20.2,-1.38c10.30667,-1.38 20.54667,-3.76 30.72,-7.14c7.15288,-2.37648 13.68288,-5.43517 19.59,-9.17606l0,20.47606l34,0l0,-151.6c0,-10.66667 -1.66667,-19.8 -5,-27.4c-3.33333,-7.6 -8,-13.8 -14,-18.6c-6,-4.8 -13.2,-8.33333 -21.6,-10.6c-5.21033,-1.40596 -10.8054,-2.37588 -16.78522,-2.90976c-0.55865,-0.28944 -1.12357,-0.56953 -1.69478,-0.84024c-8.18667,-3.88 -17.30333,-6.15667 -27.35,-6.83c-3.02667,-0.2 -6.12667,-0.3 -9.3,-0.3c-7.34667,0 -15.08333,0.54333 -23.21,1.63c-8.98667,1.20667 -17.89,3.54333 -26.71,7.01c-4.27626,1.68204 -8.29826,3.61519 -12.066,5.79946c-9.85075,-4.63964 -20.74541,-6.95946 -32.684,-6.95946z M342.6,367.24173l0,-62.92768c4.88868,0.68679 9.91535,1.17877 15.08,1.47596c11.93333,0.68667 23.04,1.75667 33.32,3.21c10.28667,1.44667 19.03333,3.97333 26.24,7.58c7.20667,3.60667 11.39667,9.77 12.57,18.49c0.84667,6.34667 -0.04,11.78 -2.66,16.3c-2.62667,4.52 -6.24333,8.16667 -10.85,10.94c-4.60667,2.76667 -9.76333,4.93667 -15.47,6.51c-5.70667,1.57333 -11.20333,2.71333 -16.49,3.42c-4.6,0.62 -9.15333,0.93 -13.66,0.93c-2.22,0 -4.43,-0.07667 -6.63,-0.23c-6.65333,-0.45333 -12.76,-1.85333 -18.32,-4.2c-1.07512,-0.45505 -2.11845,-0.95448 -3.13,-1.49827z M342.86436,270.87254c1.1245,-13.08243 5.83638,-23.77328 14.13564,-32.07254c9.59972,-9.59972 22.79924,-14.39972 39.59854,-14.4c2.43969,1.64031 4.65684,3.59698 6.65146,5.87c3.49333,3.97333 5.94667,9.22667 7.36,15.76l21.48194,-2.87488c1.30195,2.8203 2.27131,5.90193 2.90806,9.24488c1.06667,5.6 1.6,11.6 1.6,18l0,16.29433c-6.27884,-3.02014 -13.21884,-5.23492 -20.82,-6.64433c-8.42,-1.56667 -16.98333,-2.70667 -25.69,-3.42c-8.17333,-0.78667 -16.34,-1.51 -24.5,-2.17c-8.16,-0.65333 -15.58,-1.81 -22.26,-3.47c-0.15578,-0.03871 -0.31099,-0.07786 -0.46564,-0.11746z M342.95383,221.25607c-2.13807,2.6046 -4.12268,5.38591 -5.95383,8.34393c-0.61219,-1.83657 -1.28751,-3.59627 -2.02597,-5.27912c2.48698,-1.19031 5.14691,-2.21192 7.9798,-3.0648z');
    compareBoolean(function() { return m.subtract(s); },
        'M280.2,194.4c-29.06667,0 -51.33333,11.73333 -66.8,35.2l-0.8,0l0,-30.4l-32,0l0,206.8l34,0l0,-128.8c0,-4 1,-9 3,-15c2,-6 5.2,-11.8 9.6,-17.4c4.4,-5.6 10.13333,-10.4 17.2,-14.4c7.06667,-4 15.53333,-6 25.4,-6c7.05125,0 12.93858,0.94223 17.66197,2.82669c3.54153,-7.00293 8.30421,-13.02516 14.28803,-18.06669c3.42512,-2.88286 7.13645,-5.48304 11.134,-7.80054c-9.85075,-4.63964 -20.74541,-6.95946 -32.684,-6.95946z M308.6,294.59066l0,47.43958l15.32,-2.05024c1.25333,7.36667 3.81333,13.41333 7.68,18.14c3.11361,3.81269 6.78027,6.85327 11,9.12173l0,-62.92768c-6.407,-0.90009 -12.577,-2.13477 -18.51,-3.70404c-5.56863,-1.47645 -10.73196,-3.4829 -15.49,-6.01934z M308.6,383.03262l0,22.96738l34,0l0,-5.01536c-6.49431,-1.41283 -12.56765,-3.54104 -18.22,-6.38464c-5.88619,-2.96123 -11.14619,-6.81703 -15.78,-11.56738z M342.86436,270.87254c1.1245,-13.08243 5.83638,-23.77328 14.13564,-32.07254c9.59972,-9.59972 22.79924,-14.39972 39.59854,-14.4c-1.82499,-1.22702 -3.7745,-2.27702 -5.84854,-3.15c-4.84667,-2.04 -10.06,-3.29 -15.64,-3.75c-2.14,-0.18 -4.28,-0.27 -6.42,-0.27c-3.42667,0 -6.85,0.23 -10.27,0.69c-5.02667,0.66667 -10.06333,1.74333 -15.11,3.23c-0.11902,0.03506 -0.23775,0.07042 -0.35617,0.10607c-2.13807,2.6046 -4.12268,5.38591 -5.95383,8.34393c-0.61219,-1.83657 -1.28751,-3.59627 -2.02597,-5.27912c-1.817,0.86965 -3.54168,1.82936 -5.17403,2.87912c-3.96667,2.54667 -7.05,5.65 -9.25,9.31c-2.2,3.66 -2.94667,8.13 -2.24,13.41c0.78,5.82 3.47667,10.37 8.09,13.65c4.49924,3.21002 9.98736,5.6442 16.46436,7.30254z M432.09194,243.15512c1.30195,2.8203 2.27131,5.90193 2.90806,9.24488c1.06667,5.6 1.6,11.6 1.6,18l0,16.29433c0.67088,0.3227 1.33421,0.65459 1.99,0.99567c6.8,3.52667 12.55667,8.33667 17.27,14.43c4.72,6.09333 7.73333,14.03 9.04,23.81c1.66667,12.42667 0.47333,23.08 -3.58,31.96c-4.06,8.88667 -9.91667,16.39667 -17.57,22.53c-2.27227,1.82296 -4.6556,3.52427 -7.15,5.10394l0,20.47606l34,0l0,-151.6c0,-10.66667 -1.66667,-19.8 -5,-27.4c-3.33333,-7.6 -8,-13.8 -14,-18.6c-6,-4.8 -13.2,-8.33333 -21.6,-10.6c-5.21033,-1.40596 -10.8054,-2.37588 -16.78522,-2.90976c7.44801,3.85895 13.77975,9.3822 18.99522,16.56976c5.61333,7.72 9.64667,17.74 12.1,30.06z');
    compareBoolean(function() { return m.intersect(s); },
        'M287.46197,227.22669c0.4569,0.18229 0.90291,0.37339 1.33803,0.57331c4.93333,2.26667 8.86667,5.46667 11.8,9.6c2.93333,4.13333 5,9 6.2,14.6c1.2,5.6 1.8,11.73333 1.8,18.4l0,24.19066c4.75804,2.53645 9.92137,4.54289 15.49,6.01934c5.933,1.56928 12.103,2.80396 18.51,3.70404l0,-27.11404c0,-2.1679 0.08812,-4.27705 0.26436,-6.32746c-6.477,-1.65834 -11.96512,-4.09252 -16.46436,-7.30254c-4.61333,-3.28 -7.31,-7.83 -8.09,-13.65c-0.70667,-5.28 0.04,-9.75 2.24,-13.41c2.2,-3.66 5.28333,-6.76333 9.25,-9.31c1.63236,-1.04976 3.35703,-2.00947 5.17403,-2.87912c-4.40823,-10.04582 -11.06624,-17.35278 -19.97403,-21.92088c-0.6999,-0.35892 -1.40523,-0.70577 -2.116,-1.04054c-3.99755,2.3175 -7.70888,4.91767 -11.134,7.80054c-5.98382,5.04153 -10.7465,11.06376 -14.28803,18.06669z M308.6,342.03024l0,41.00238c4.63381,4.75035 9.89381,8.60614 15.78,11.56738c5.65235,2.8436 11.72569,4.97181 18.22,6.38464l0,-33.74291c-4.21973,-2.26846 -7.88639,-5.30904 -11,-9.12173c-3.86667,-4.72667 -6.42667,-10.77333 -7.68,-18.14z M396.59854,224.4c0.00049,0 0.00097,0 0.00146,0c8.26667,0 15,1.2 20.2,3.6c5.2,2.4 9.26667,5.66667 12.2,9.8c1.18226,1.66591 2.21291,3.45095 3.09194,5.35512l12.21806,-1.63512c-2.45333,-12.32 -6.48667,-22.34 -12.1,-30.06c-5.21547,-7.18756 -11.54722,-12.71081 -18.99522,-16.56976c-3.66074,-0.32683 -7.46566,-0.49024 -11.41478,-0.49024c-13.6,0 -26.06667,3.06667 -37.4,9.2c-8.3401,4.51347 -15.48882,10.39882 -21.44617,17.65607c0.11842,-0.03565 0.23714,-0.07101 0.35617,-0.10607c5.04667,-1.48667 10.08333,-2.56333 15.11,-3.23c3.42,-0.46 6.84333,-0.69 10.27,-0.69c2.14,0 4.28,0.09 6.42,0.27c5.58,0.46 10.79333,1.71 15.64,3.75c2.07404,0.87298 4.02355,1.92298 5.84854,3.15z M436.6,286.69433l0,98.82961c2.4944,-1.57967 4.87773,-3.28098 7.15,-5.10394c7.65333,-6.13333 13.51,-13.64333 17.57,-22.53c4.05333,-8.88 5.24667,-19.53333 3.58,-31.96c-1.30667,-9.78 -4.32,-17.71667 -9.04,-23.81c-4.71333,-6.09333 -10.47,-10.90333 -17.27,-14.43c-0.65579,-0.34109 -1.31912,-0.67298 -1.99,-0.99567z');

    var t = PathItem.create('M283.8,165.2L283.8,227.2L248.6,227.2L248.6,257.2L283.8,257.2L283.8,388.8Q283.8,403.2,286.6,412Q289.4,420.8,295.2,425.6Q301,430.4,310.4,432.2Q319.8,434,333,434L359,434L359,404L343.4,404Q335.4,404,330.4,403.4Q325.4,402.8,322.6,400.8Q319.8,398.8,318.8,395.2Q317.8,391.6,317.8,385.6L317.8,257.2L359,257.2L359,227.2L317.8,227.2L317.8,165.2Z');
    var r = PathItem.create('M192.38,277.21L176.71,307.39L258.34,349.8Q276.09,359.03,287.91,369.68Q299.73,380.33,305.26,393.11Q310.79,405.9,309.12,421.26Q307.45,436.62,298.04,454.72L329.99,471.32Q330.82,469.82,331.6,468.32Q343.27,445.86,341.07,425.7Q338.72,404.19,322.08,381.57L322.45,380.86L361.14,400.96L375.89,372.56L192.38,277.21Z');
    compareBoolean(function() { return t.unite(r); },
        'M283.8,165.2l0,62l-35.2,0l0,30l35.2,0l0,67.51094l-91.42,-47.50094l-15.67,30.18l81.63,42.41c9.85867,5.12651 18.34534,10.58156 25.46,16.36515l0,22.63485c0,9.6 0.93333,17.33333 2.8,23.2c1.86667,5.86667 4.73333,10.4 8.6,13.6c3.19226,2.64187 7.20242,4.60215 12.03049,5.88084c-1.88234,7.32825 -4.94584,15.07464 -9.19049,23.23916l31.95,16.6c0.55333,-1 1.09,-2 1.61,-3c6.19684,-11.92639 9.46051,-23.36639 9.79102,-34.32l17.60898,0l0,-30l-15.6,0c-3.10687,0 -5.87439,-0.06787 -8.30256,-0.20361c-3.16081,-7.31727 -7.49996,-14.72606 -13.01744,-22.22639l0.37,-0.71l38.69,20.1l14.75,-28.4l-58.09,-30.183l0,-85.177l41.2,0l0,-30l-41.2,0l0,-62z');
    compareBoolean(function() { return t.subtract(r); },
        'M283.8,165.2l0,62l-35.2,0l0,30l35.2,0l0,67.51094l34,17.66607l0,-85.177l41.2,0l0,-30l-41.2,0l0,-62z M283.8,366.16515l0,22.63485c0,9.6 0.93333,17.33333 2.8,23.2c1.86667,5.86667 4.73333,10.4 8.6,13.6c3.19226,2.64187 7.20242,4.60215 12.03049,5.88084c0.89964,-3.50246 1.52948,-6.9094 1.88951,-10.22084c1.11333,-10.24 -0.17333,-19.62333 -3.86,-28.15c-3.68667,-8.52 -9.47,-16.33 -17.35,-23.43c-1.31496,-1.1848 -2.68496,-2.35641 -4.11,-3.51485z M341.39102,434l17.60898,0l0,-30l-15.6,0c-3.10687,0 -5.87439,-0.06787 -8.30256,-0.20361c3.19425,7.39468 5.1851,14.69588 5.97256,21.90361c0.29845,2.73492 0.40546,5.50159 0.32102,8.3z');
    compareBoolean(function() { return t.intersect(r); },
        'M283.8,324.71094l0,41.45422c1.42504,1.15843 2.79504,2.33005 4.11,3.51485c7.88,7.1 13.66333,14.91 17.35,23.43c3.68667,8.52667 4.97333,17.91 3.86,28.15c-0.36003,3.31143 -0.98987,6.71838 -1.88951,10.22084c1.02,0.27014 2.0765,0.50986 3.16951,0.71916c6.26667,1.2 13.8,1.8 22.6,1.8l8.39102,0c0.08444,-2.79841 -0.02257,-5.56508 -0.32102,-8.3c-0.78746,-7.20773 -2.77831,-14.50893 -5.97256,-21.90361c-1.74009,-0.09727 -3.3059,-0.2294 -4.69744,-0.39639c-3.33333,-0.4 -5.93333,-1.26667 -7.8,-2.6c-1.86667,-1.33333 -3.13333,-3.2 -3.8,-5.6c-0.66667,-2.4 -1,-5.6 -1,-9.6l0,-43.223z');
});

test('Selected test-cases provided by @iconexperience', function() {
    // Test all of test cases in one batch.
    // Read more in #784
    var paths = [[
        [[276.60479999999995, 265.6776, 2.6464000000000283, 5.646400000000028, 1.2208000000000538, -0.8512000000000057], [279.9632, 262.924, -1.0127999999999702, 1.0160000000000196, 0.019200000000012096, 3.302400000000034], [294.4, 269.31440000000003, -7.240000000000009, 0, 7.240000000000009, 0], [308.8384, 262.924, -0.020800000000008367, 3.302400000000034, 1.0112000000000307, 1.0160000000000196], [312.1968, 265.6776, -1.2224000000000501, -0.8512000000000057, -2.6464000000000283, 5.646400000000028], [294.4, 273.3128, 5.590400000000045, 0, -5.590399999999988, 0], true],
        [[294.4, 273.3128, 0, 0, -7.240000000000009, 0], [279.9808, 266.9064, 0.014400000000023283, 3.3023999999999774, 0, 0], [279.91999999999996, 253.31279999999998, 0, 0, 9.652800000000013, 0], [308.88, 253.31279999999998, -9.652800000000013, 0, 0, 0], [308.82079999999996, 266.9064, 0, 0, -0.014865740466348143, 3.068288832262283], [295.93610344888236, 273.2670371105283, 6.683812628997487, -0.3953268817066373, 5.682599074647953, -0.3268447810676207], [312.1968, 265.6776, -2.4122653499520084, 5.146166079897625, 8.40640000000002, 5.8559999999999945], [294.4, 307.1864, 16, 0.001599999999996271, 0, 0], true]
    ],
        [
            [[512, 563.0515712, 0, 0, 0, 0], [462.4499456000001, 563.0515712, 0, 0, -3.031499856000039, 0], [456.94438400000007, 557.549450576, 0, 3.031499855999982, 0, 0], true],
            [[462.4499456000001, 563.0515712, -3.031499856000039, 0, -3.0303527841790014, 0], [456.94438506256495, 557.5494505759262, 0.0018713092763391614, 3.0299308178812225, 0,0], [567.0556160000001, 564], true]
        ],
        [
            [[506.49443840000004, 600, 0, 0, 0, 0], [567.0556160000001, 623.6127488, 0, 0, 0, 3.031499856000096], [561.5500544, 629.1148694240001, 3.031499855999982, 0, 3.031499855999982, 0], [567.0556160000001, 634.620431024, 0, -3.031499855999982, 0, 0], true],
            [[561.5500544, 629.1148694240001, 3.031499855999982, 0, 3.031499855999982, 0], [567.0556160000001, 634.620431024, 0, -3.0280588800000032, 0, 0], [567.0556160000001, 656.6461184, 0, 0, 0, 3.0280588800000032], true]
        ],
        [
            [[400.81546878700505, 400.81546878700505, 79.97937495065997, -79.9793749506602, -79.97937495066014, 79.97937495065997], [111.18453121299513, 400.81546878700505, 79.97937495066012, 79.97937495065997, -79.97937495066009, -79.9793749506602], [111.18453121299507, 111.18453121299507, -79.97937495066003, 79.97937495066017, 79.97937495066009, -79.97937495066007], [400.815468787005, 111.18453121299503, -79.97937495066009, -79.97937495066003, 79.97937495066003, 79.97937495066007], true],
            [[400.815468787005, 400.815468787005, 79.97937495065997, -79.97937495066014, -79.97937495066014, 79.97937495065992], [111.18453121299507, 400.815468787005, 79.97937495066014, 79.97937495065992, -79.97937495066003, -79.9793749506602], [111.18453121299507, 111.18453121299504, -79.97937495066003, 79.97937495066014, 79.97937495066009, -79.97937495066006], [400.81546878700493, 111.184531212995, -79.97937495066009, -79.97937495066002, 79.97937495066003, 79.97937495066004], true]
        ],
        [
            [[512, 96, -125.89361307431193, 0, 168, 0], [736, 337.78, 0, -120.88999999999999, 0, 21.79000000000002], [729.34, 402.7, 4.319999999999936, -21.340000000000032, -25.649999999999977, 126.62999999999994], [512, 640, 95.85000000000002, 0, 0, 0], [288.04, 332.0799999999999, 100.000575152844760396, -0.000053268689669039304, 0.41767160812742077, -28.392122573051836], [299.43514767699014, 248.78534185388617, -7.421136238632187, 26.615225151165646, 23.67264213867054, -84.8997620019185], true],
            [[512, 96, -125.89322163652065, 0, 168, 0], [736, 337.78, 0, -120.88999999999999, 0, 151.11], [512, 640, 112, 0, -112, 0], [288, 337.78, 0, 151.11, 0, -1.9002097681526493], [288.04172546304807, 332.07984018287976, -0.02788946906741785, 1.8997402808149104, 0.4168153242491144, -28.39210955294908], [299.4351476773096, 248.7853418527407, -7.42079681324509, 26.615144268685953, 23.671675930689275, -84.89992191301803], true]
        ]];

    var results = [
        ['M276.6048,265.6776c1.22072,-0.85114 2.34544,-1.73748 3.35819,-2.75339l-0.04299,-9.61141c9.6528,0 19.3072,0 28.96,0l-0.04199,9.64131c0.0002,-0.01003 0.00032,-0.02007 0.00039,-0.03011c1.0112,1.016 2.136,1.9024 3.3584,2.7536c8.4064,5.856 -1.7968,41.5104 -17.7968,41.5088l0,-33.8736c-0.54424,0 -1.08806,-0.01754 -1.62844,-0.05139c-5.65571,-0.34382 -13.76286,-2.45483 -16.16676,-7.58381z M305.01293,271.38867c-2.95531,1.12961 -6.25621,1.70701 -8.98571,1.87282c-0.03036,0.0019 -0.06073,0.00375 -0.09111,0.00555c2.75482,-0.15845 6.09324,-0.73747 9.07682,-1.87836z'],
        ['M462.44995,563.05157c-3.0315,0 -5.50556,-2.47062 -5.50556,-5.50212l0,0l110.11123,6.45055c0,0 -107.63717,-0.94843 -104.60567,-0.94843z'],
        ['M506.49444,600l60.56118,23.61275c0,3.0315 -2.47406,5.50212 -5.50556,5.50212c3.0315,0 5.50556,2.47406 5.50556,5.50556l0,22.02569c0,2.39831 -1.552,-16.27299 -3.70373,-24.14296z'],
        ['M400.81547,400.81547c-79.97937,79.97937 -209.65156,79.97937 -289.63094,0c-79.97937,-79.97937 -79.97937,-209.65156 0,-289.63094c79.97937,-79.97937 209.65156,-79.97937 289.63094,0c79.97937,79.97937 79.97937,209.65156 0,289.63094z'],
        ['M551.12365,634.03909c-12.92155,3.89332 -26.02251,5.96091 -39.12365,5.96091c-112,0 -224,-151.11 -224,-302.22c0,-1.90016 0.01384,-3.80031 0.04172,-5.7c-0.00057,0 -0.00115,0 -0.00172,0c0.41767,-28.39212 3.97401,-56.67943 11.39515,-83.29466c23.67168,-84.89992 86.67163,-152.78534 212.56485,-152.78534c168,0 224,120.89 224,241.78c0,133.43422 -87.33052,266.86844 -184.87635,296.25909z']
    ];

    for (var i = 0; i < paths.length; i++) {
        var entry = paths[i],
            result = results[i],
            path1 = PathItem.create(entry[0]),
            path2 = PathItem.create(entry[1]);
        compareBoolean(path1.unite(path2), result[0], 'path1.unite(path2); // Test ' + (i + 1));
    }
});

test('Isolated edge-cases from @iconexperience\'s boolean-test suite', function() {
    // Test all of @iconexperience's isolated cases in one batch.
    // Read more in #784
    var paths = [[
        [[240, 270], [250.53466323186618, 221.0057178821544], [237.67824477332854, 193.79795691566554], [320, 240], true],
        [[263.31216874462405, 248.04647709671602], [230, 250], [237.6782447781314, 193.79795691339606], true]
    ], [
        [[450, 230], [362.46932779553646, 264.4394368330295], [329.00715661950534, 214.6369961279195], true],
        [[362.46932779553646, 264.4394368330295, 0, 0, 0.5029830904762775, -0.33795344231873514], [331.7750437531172, 267.3071389516512, 0.33347583635435285, -3.453534396532916, 0, 0], [329.00715661950534, 214.6369961279195], true ]
    ], [
        [[211.76470809030513, 391.5530064242439, 0, 0, -1.1306561612996688, -11.284564044000547], [206.28516988159765, 416.3206370377553, 9.29930425192211, -12.510751934521268, 0, 0], [142.17355611649077, 367.3087985311291], true],
        [[152.06363005021922, 397.5347520600572], [218.65631922872535, 332.74836297575644, 0, 0, -9.581012775680733, 10.097912103353224], [211.76470809030513, 391.55300642424385, -1.8914123817368989, -18.8773252964819, 0, 0], true ]
    ], [
        [[138.3141655808707, 456.06810826237086, 0, 0, 1.6246548956081597, -1.188772469512287], [103.70383447788026, 471.4426853592628, 15.914191732656704, 1.2415901505934812, 0, 0], [147.64340975918196, 340.17263448268653], true],
        [[102.88355437423151, 307.6462971693936], [311.9175162439421, 379.75248280363354, 0, 0, -68.45789375971557, 0.5465997368099806], [138.3141655808706, 456.06810826237097, 33.643258081480184, -24.617030422937262, 0, 0], true ]
    ], [
        [[260.5630488345416, 240.11728335016855], [230.82205831331396, 236.1805613638536], [300.041224753361, 200], [300, 300], true],
        [[290.30323645974596, 244.0538990495128], [320, 280], [230.82205831199582, 236.18056137381197], true ]
    ], [
        [[266.2339624919457, 249.28592706878334], [236.42561910533144, 245.88969337410168], [320, 240], [280, 300], true],
        [[296.03993000559, 252.68189006496056], [320, 300], [236.42561910512714, 245.8896933758947], true ]
    ], [
        [[109.2693308633721, 249.31916869409733], [157.6368865573791, 284.8238100271277], [150, 329.3972954455561], true],
        [[109.26933086337212, 249.3191686940973], [157.1468245594682, 224.11329044817836], [161.745373658654, 283.93680878488584, 0, 0, -4.753441100107182, 0.365390282896783], [157.63688655737909, 284.82381002712776, -0.5599416549340219, 0.7628019369743697, 0, 0], true ]
    ], [
        [[249.53400844979302, 269.5706278725207, 0, 0, -5.202299838213264, 1.3434437726778015], [243.02912232136828, 272.45353289318496, -0.02459817048722357, 0.029126503110205704, 0, 0], [197.1892333604494, 233.7404268430289], true],
        [[243.02912232136822, 272.453532893185, 0, 0, 0.44566136595048533, -0.527704170852985], [242.66927041668393, 276.2408467361064, -0.16957999946740188, -3.971488536288973, 0, 0], [197.18923336044946, 233.74042684302884], true ]
    ], [
        [[326.356206851053, 259.5165810014779, 0, 0, 2.309307036608857, -4.052538199979125], [327.9512902239682, 255.73918757869137, -0.3651148630259513, -1.1113992742978098, 0, 0], [384.95409529898944, 237.01272999645926], [350, 300], true],
        [[327.9512902239682, 255.73918757869126], [320, 250], [384.95409529898944, 237.01272999645937], true ]
    ], [
        [[231.45710796909526, 304.56978671182316, 0, 0, 6.965195106563414, -10.698308411775713], [230.61969090209206, 265.2654116388295, 4.522412663592235, 13.984636258635192, 0, 0], [287.7087867546365, 246.8036908431129], true],
        [[267.0542186716473, 253.4830551491999], [230.61969090209206, 265.26541163882956, 0, 0, 2.0144954400552137, 6.22941515281218], [229.4516168394906, 239.90168380046367, -4.821444118085253, 13.349082266661071, 0, 0], true ]
    ], [ // 10
        [[534.4141795042503, 349.4135521611639, 0, 0, 0.9952674258736351, -5.064332094235169], [518.8256109308445, 384.2317022309883, 13.426353275139377, -9.450213278660101, 0, 0], [487.8821067425946, 340.2688405948006], [529.4025393039343, 348.4286400051758, 0, 0, -0.93900751246872, -2.1407484665217567], true],
        [[475.5403232814915, 337.8433726698194], [538.3223483008236, 152.20174200302918], [576.6459309452254, 198.36777944721226, 0, 0, -12.4717091442418, 10.353077772467884], [534.4141795042503, 349.41355216116364, 9.869907229687897, -50.22216808367796, 0, 0], true ]
    ], [
        [[100.10301277238074, 299.8360064274938], [192.98476439737567, 381.03840618337944], [191.58826071683484, 405.5453629318501, 0, 0, 1.76117811362775, 0.1003589202450712], [158.80881838193744, 396.40041814383164, 11.345470204261431, 9.97219009394314, 0, 0], true],
        [[195.00176127688619, 345.64254151310115], [481.252611684097, 441.78388187198607], [448.747388315903, 492.21611812801393, 0, 0, -122.96965958186621, -79.25796175143017], [191.5882607168351, 405.54536293185015, 35.244503935513166, 2.0083717439883912, 0, 0], true ]
    ], [
        [[201.4099970302107, 456.883477028569], [226.29027430947582, 456.0825533500981, 0, 0, -0.2832959470065646, -8.800441169464932], [219.2838668275147, 483.73556312076016, 10.02977284964885, -11.976298343562064, 0, 0], [200, 500], true],
        [[166.32133819059692, 458.01301981440685], [204.6120898437701, 320.4482218772343], [244.3462523224993, 365.40593903584846, 0, 0, -18.086870972805855, 15.985390615581537], [226.29027430947582, 456.08255335009807, -0.8931629157136456, -27.74564118385956, 0, 0], true ]
    ], [
        [[134.30251416753708, 172.0081092040946, 0, 0, 97.01301228604092, -2.2561165647916255], [223.30038363130024, 161.29265138604404, 7.832155210448377, -11.227084549308643, 0, 0], [135.69748583246292, 231.99189079590533], true],
        [[248.04309709886002, 178.55347910623016], [223.30038363130373, 161.29265138603904, 0, 0, -6.854532836619114, 9.825701564243673], [240.2776561840474, 149.3557271403692, -14.092989377677696, 3.748174738020822, 0, 0], true ]
    ], [
        [[517.5047187481108, 202.91541735696418, 6.513140197398911, -18.244597886454756, 0, 0], [545.7583411018326, 213.0016798561104], [557.8431595513716, 185.5433980031361, -8.326516730755202, -3.664629555367924, 0, 0], [545.7583424843119, 213.00167993223172, 0, 0, -1.1368683772161603e-13, 0], true],
        [[550.3546243594556, 242.6474912770306, 0, 0, 11.17102286717818, -1.7319541248482437], [541.1620576122966, 183.35586693936358, -11.180538230113825, 1.733429385644115, 0, 0], true ]
    ], [
        [[290.1253327484206, 287.90652074109494], [342.6480156335393, 281.88796157138086], [340.20645047153664, 251.98747653472907], [370.8816252545162, 292.03012917196776], true],
        [[345.08958029318296, 311.78843842619824], [300.0082518321038, 300.98900280805503], [340.20645030689445, 251.98747654817316], true ]
    ], [
        [[371.53493854231004, 257.22260275739484, 0, 0, -5.3406979774814545, -1.8674091727692712], [360.1944182950855, 257.04852958152196, 1.4950314859081004, -0.8166674734584944, 0, 0], [430.4144838319712, 235.2349421258068, -7.8035474909103755, -18.555544116219835, 0, 0], true],
        [[360.1944182950853, 257.0485295815221, 0, 0, 0.09209791202033557, -0.05030888635417341], [358.8995515281498, 260.3188854078006, 0.4901759953118585, -3.4265416599628793, 0, 0], [300, 230], true ]
    ], [
        [[409.2490311439771, 253.4403128346141, 0, 0, -4.356295913055419, 1.322648861578557], [403.6254675741819, 256.25364724519244, 0.0777100557439212, -0.09457550986331853, 0, 0], [357.2674295740252, 218.16254181654907], true],
        [[403.625467574182, 256.2536472451924, 0, 0, 0.27347672342176566, -0.33282952001209765], [403.0346093549582, 261.0014812561057, -0.21747734382410044, -4.397037948600143, 0, 0], [357.26742957402513, 218.16254181654915], true ]
    ], [
        [[292.9399983081827, 321.04496242335836], [301.41124169557446, 297.40574481030876], [271.4382902635834, 296.1319728479751], [319.88255402047366, 273.7665574434494], true],
        [[331.38418270096264, 298.67951674085646], [350, 345.76013009266507], [271.4382902810318, 296.1319724373992], true ]
    ], [
        [[349.2988989052917, 302.360122358839], [306.8847250793194, 310.8417788081535], [317.34865046978905, 286.0848942283935], [295.03238113332606, 289.30770347326694], true],
        [[306.8847250793191, 310.8417788081534], [308.40196713650136, 308.29790890842884], [297.432392604982, 249.3091944403423], [330.2439214313423, 255.57562328972034], true ]
    ], [ // 20
        [[150, 290], [180, 260], [220, 300], true],
        [[180, 260], [230, 270], [170, 310], [220, 300], true ]
    ], [
        [[299.7385339651986, 102.20556080028058, -8.846854364457954, -13.558854632211663, 0, 0], [324.85949771228013, 85.81468459475798], [331.26017128225755, 56.500458549069435], [349.988176980811, 69.4187741812245, 0, 0, 9.063133638672639, 13.890328297113285], true],
        [[318.4609036471463, 115.11938678905685], [340.9228398583517, 111.1463404606569], [308.7912208820021, 60.47525341924158], [331.2601712818937, 56.50045854899], true ]
    ], [
        [[388.9213889025063, 223.3215931867822, 0, 0, 120.27613015930797, -2.2436869719875574], [597.6181955110758, 242.94850207232983], [628.0354673841999, 243.03685701873312, 0, 0, -0.031041118475286567, 10.686285016774008], [419.2789142850017, 364.27077547853787, 0, 0], true],
        [[568.0357205119291, 242.86257201242702], [576.294245114903, 263.4873716612576, -0, 0, 0, 0], [619.5200435607698, 221.87571045051882], [628.0354673841999, 243.03685701876972, 0.030302631585300333, -10.432051860541975, 0, 0], true ]
    ], [
        [[250, 130], [274.2339539863471, 194.10449080567912, 0, 0, 0.10568701547992987, -0.053016796189552906], [272.11154618911314, 196.81853259861245, 1.370678992642354, -2.513969427633242, 0, 0], true],
        [[250, 130], [282.64261453718956, 194.05831848298737, 0, 0, -6.1766919512210166, -1.0441192545144133], [274.23395398634705, 194.10449080567915, 0.11866362575074163, -0.05952637826862883, 0, 0], true ]
    ], [
        [[386.4093435542647, 270.73026596496675, 0, 0, 77.41690321854611, 15.777187877165375], [528.635149558586, 289.6368596320347, -12.835362144855708, 5.740740187064205, 0, 0], [553.1321511797777, 344.4081743101702], true],
        [[528.6351495585858, 289.6368596320348, 0, 0, -0.31548183780626005, 0.14110231126664985], [522.9057367288372, 295.33394762583623, 2.3982010141742194, -4.106159119428241, 0, 0], [574.716309478071, 325.5938972523037], [553.1321511797779, 344.40817431017007], true ]
    ], [
        [[525, 345], [557.3179687742295, 363.04379418599376], [587.1223034990716, 366.54087021669767], [500, 400], true],
        [[527.5311085637297, 359.5487685170587], [533.6060904205631, 381.4080959083297], [581.0430044030195, 344.66929908778826], [587.1223034989287, 366.54087021791537], true ]
    ], [
        [[250, 150], [234.74625221124325, 182.05024235296503], [205.60227766791786, 189.2546311484904], [205.60227766791786, 189.4546311484904], [205.61255586021304, 189.25209037995552], [263.84900039629053, 174.85604471078497], true],
        [[225.84313834895474, 153.1048226009985], [180, 190], [243.48255333255608, 210.4533277169673], true ]
    ], [
        [[598.5486959306086, 408.11025390080914], [556.2683544908633, 467.2163769525839], [552.429221746517, 437.22810161381835], true],
        [[556.2683546136653, 467.2163769491477], [518.5330018076562, 379.67700053006695], [554.5900896267148, 407.2398529865706], true ]
    ], [
        [[570, 290], [575.8176000300452, 323.5855681222093], [547.6444566349796, 309.1465641549448], [578.8341920603752, 304.57835489946484, 0, 0, -5.7225406635552645e-9, -3.907138079739525e-8], [519.4675806753132, 313.27349449897866], true],
        [[575.8176000300631, 323.58556812217466], [560, 300], [522.421811145802, 296.21971842903645], true ]
    ], [
        [[87.34388521634615, 0, -3.9382371300724373, 0, 2.1330446378301247, 0], [93.61538461538461, 0, -2.9185159236061224, 0, 2.2538650858334677, 0], [100.17943209134619, 0, -2.11714516843503, 0, 8.39247295673077, 0], [100.17943209134613, 0, -4.131239081724743, 0, -2.1880158253205195, 0], [93.61538461538458, 0, 2.1880158253205053, 0, -2.0904997996794776, 0], true],
        [[163.8269230769231, 0], [160.90144230769232, 0, 1.6090144230769283, 0, 0, 0], [2.9254807692307696, 0, 0, 0, -1.6090144230769234, 0], [0, 0], true ]
    ], [
        'M419.20344565830516,384.34607379092466c5.844713611412715,3.313058909545532 8.737629565616828,3.6008009941920136 9.663847696961,3.5457774486854987c0.4249875971154893,-0.23215482613579752 3.134401138218834,-1.9001699951828073 6.912521889816844,-9.027614775318739c33.358079343370036,-62.930193110079074 15.048818377810562,-263.53585410399967 14.89110904614597,-264.2035689431943l58.39331120497843,-13.792070421750992c0.38461989956749676,1.6284160973869177 24.314056080847877,221.98511298279283 -20.271816008415897,306.09659649109955c-7.779727674038611,14.676497403570863 -19.843728834889646,29.87171953249475 -38.60350857895327,37.0189975138764c-20.692089248499826,7.88346749760683 -41.830098370105475,3.183730282850888 -60.57334789311045,-7.440826188961864z',
        'M450.6709242911992,114.66066752097116c-5.7317866225195075,-24.267422495749997 -2.0221968032280984,26.377626042706837 32.186077599051714,22.94975715794014l-5.982390225081872,-59.70101345199127c34.20913357233178,-3.427954978623788 37.91990661944476,47.22024138695528 32.18962383106816,22.95918587255207z'
    ], [
        'M224.25080140147438,238.4376691273944c2.4512375314882604,34.92470808870988 9.866473254030097,54.13389818714606 17.201166692275166,64.07401871051215c4.079846680080152,5.529088305912239 8.23969086685824,8.377993223423289 12.35501848826604,9.971618948612331c7.68049930207755,2.9742082274120776 19.93298576498958,3.2254331896798476 38.31910020350554,-2.955571037063976c83.83623127989932,-28.183883094396606 198.0851750273717,-154.030084369221 199.08078586916076,-155.64409852867644l51.06598126433914,31.500246943637762c-5.409577386928277,8.769625774145538 -129.41511742843056,146.85627742712396 -231.0275670523032,181.01613119694778c-24.82193278389599,8.344583732809383 -52.91091773805937,12.178082640769901 -79.10498961275593,2.0346504217118877c-15.357074444304459,-5.946896857197203 -28.532715560643766,-16.156819377946817 -38.967559150968384,-30.29832424159372c-16.970737553128487,-22.99907665791602 -25.951185000357953,-55.26903399719629 -28.774696789006782,-95.49782608154422z',
        'M491.2068726546696,153.8836372207983c2.67966717656293,-4.344087653601775 -5.988630526531892,6.5447012048449835 -4.103367938301176,19.82205168468778l59.40415261280532,-8.434847500372314c1.8912589260252162,13.319580925081453 -6.821144106809129,24.30583609819925 -4.234803410140444,20.11304275928262z'
    ]];
    var results = [
        ['M240,270l4.48298,-20.84932l-14.48298,0.84932l7.67824,-56.20204l82.32176,46.20204zM237.67824,193.79796z', 'M244.48298,249.15068l6.05168,-28.14496l12.77751,27.04076z'],
        ['M450,230l-87.53067,34.43944l-0.01593,-0.02371h0c-0.91969,-0.32388 -30.35274,-0.48069 -30.67835,2.89141l-2.76789,-52.67014z', 'M362.46933,264.43944l-0.01593,-0.02371c0.02221,0.00782 0.02779,0.01574 0.01593,0.02371z'],
        ['M211.76471,391.55301c-1.13066,-11.28456 3.81977,12.25688 -5.47954,24.76763l-28.00902,-21.41225l-26.21252,2.62636l13.04584,-12.69198l-22.93592,-17.53397l30.159,10.50681l46.32376,-45.06725c-9.58101,10.09791 -8.78302,39.92732 -6.89161,58.80464z', 'M178.27615,394.90839l-13.16668,-10.06562l7.22309,-7.02716l39.43215,13.7374z'],
        ['M138.31417,456.06811c1.62465,-1.18877 -18.69614,16.61617 -34.61033,15.37458l22.34488,-66.7556l-23.16516,-97.04078l209.03396,72.10619c-68.45789,0.5466 -139.96009,51.6986 -173.60335,76.31563z', 'M126.04871,404.68708l21.5947,-64.51445l-9.32924,115.89547z'],
        ['M300.04122,200l-0.02301,55.81328l19.98178,24.18672l-19.98771,-9.82138l-0.01229,29.82138l-29.04124,-44.09743l-40.1367,-19.722z', 'M300.01822,255.81328l-0.00592,14.36534l-29.05353,-14.27606l-10.39571,-15.78528l29.74019,3.93662z'],
        ['M320,240l-17.26682,25.90022l17.26682,34.09978l-27.9401,-18.08984l-12.0599,18.08984l-8.52877,-31.42l-35.04561,-22.69031z', 'M302.73318,265.90022l-10.67329,16.00993l-20.58867,-13.33015l-5.23726,-19.29408l29.80597,3.39596z'],
        ['M157.63421,284.83942l-7.63421,44.55788l-40.73067,-80.07813l47.87749,-25.20588l4.59855,59.82352c-2.48454,0.19098 -3.64723,0.63304 -4.11116,0.90261z', 'M157.63421,284.83942c-0.00547,0.00318 -0.01083,0.00633 -0.01611,0.00945c0.0062,-0.0081 0.01246,-0.01645 0.01878,-0.02506zM157.61811,284.84887c-0.25231,0.3297 -0.39376,0.23354 0,0z'],
        ['M249.53401,269.57063c-5.12674,1.32393 -6.49012,2.86652 -6.50509,2.88317c-0.17123,0.40034 -0.45801,1.48334 -0.35964,3.78705l-45.48004,-42.50042zM243.02893,272.45376c0.00005,-0.00006 0.00012,-0.00014 0.00019,-0.00023l-0.00007,-0.00006c-0.00004,0.0001 -0.00008,0.00019 -0.00012,0.00029z', 'M243.02891,272.4538c-0.00017,0.00019 -0.00017,0.00018 0.00001,-0.00003c0,0.00001 -0.00001,0.00002 -0.00001,0.00003zM243.02912,272.45353l-0.00007,-0.00006c0.12387,-0.28946 0.18715,-0.22146 0.00007,0.00006z'],
        ['M326.35621,259.51658c1.15567,-2.02805 1.46269,-3.25061 1.52678,-3.82669l-7.88299,-5.68989l64.9541,-12.98727l-34.9541,62.98727z', 'M327.88299,255.68989c0.00623,-0.05602 0.01017,-0.10593 0.01237,-0.14987c0.01299,0.05713 0.03113,0.12366 0.05593,0.19917zM327.89536,255.54001c-0.0732,-0.32201 0.01728,-0.34576 0,0z'],
        ['M231.45711,304.56979c6.9652,-10.69831 3.685,-25.31974 -0.83742,-39.30438c2.0145,6.22942 -5.98952,-12.01465 -1.16807,-25.36373l37.6026,13.58137l20.65457,-6.67936z', ''],
        ['M534.41418,349.41355c0.99527,-5.06433 -2.16222,25.36794 -15.58857,34.81815l-30.9435,-43.96286l-12.34178,-2.42547l62.78203,-185.64163l38.32358,46.16604c-12.47171,10.35308 -32.36184,100.8236 -42.23175,151.04577z', 'M534.41418,349.41355l-5.01164,-0.98491c-0.93901,-2.14075 5.01164,0.98491 5.01164,0.98491z'],
        ['M100.10301,299.83601l92.88175,81.2024l2.017,-35.39586l286.25085,96.14134l-32.50522,50.43224c-122.96966,-79.25796 -221.91462,-84.66238 -257.15913,-86.67076c1.76118,0.10036 -21.43397,0.82725 -32.77944,-9.14494z', ''],
        ['M219.28387,483.73556l-19.28387,16.26444l1.41,-43.11652l-35.08866,1.12954l38.29075,-137.5648l39.73416,44.95772c-18.08687,15.98539 -18.94914,62.93097 -18.05598,90.67661c-0.2833,-8.80044 3.02337,15.67671 -7.00641,27.65301z', ''],
        ['M134.30251,172.00811c69.89455,-1.62545 89.33749,-12.29848 91.81689,-13.93979c-0.39444,0.32626 -1.16469,1.04523 -2.11233,2.26401c0.67085,-0.74847 1.44456,-1.56236 2.30914,-2.40044c-0.04649,0.03433 -0.11164,0.08005 -0.19681,0.13643c0.13147,-0.10874 0.22118,-0.17386 0.26181,-0.19932c3.45306,-3.33445 8.33582,-7.03437 13.89644,-8.51327l7.76544,29.19775l-24.74271,-17.26083c-3.03845,4.3555 -2.81641,3.24058 -0.39992,0.32276l-87.20297,70.37648z', 'M224.00707,160.33233c-0.22634,0.29111 -0.46281,0.61073 -0.70669,0.96032l-0.39992,0.32276c0.32907,-0.39735 0.69884,-0.82812 1.10661,-1.28308zM226.38122,157.869c0.04659,-0.0292 0.02861,-0.00624 -0.065,0.06289c0.02161,-0.02095 0.04328,-0.04191 0.065,-0.06289z'],
        ['M517.50472,202.91542l28.24931,10.08472c0.00241,0.00682 0.00483,0.01365 0.00724,0.02047l-0.00293,-0.01893c0,0 0,0 0,0c0,0 0,0 0,0l-4.59628,-29.64581c-5.58903,0.86652 -0.50164,15.25023 4.58932,29.63676c-0.44214,-0.57333 -21.80103,-28.13268 -28.24666,-10.07721zM545.75834,213.00168l12.08482,-27.45828c-8.32546,-3.66417 -12.08387,27.45042 -12.08482,27.45828c0,0 0,0 0,0zM550.35462,242.64749c5.58313,-0.86561 0.49657,-15.24328 -4.59335,-29.62688z', 'M545.75403,213.00014l0.00431,0.00154l0,0c-0.00007,-0.00009 -0.00241,-0.00314 -0.00697,-0.00905c0.00089,0.0025 0.00177,0.00501 0.00266,0.00751z'],
        ['M290.12533,287.90652l22.75403,-2.60738l27.32709,-33.31167l0,0l30.67517,40.04265l-27.52018,-1.40525l1.72814,21.16356l-45.08133,-10.79944l9.90334,-12.07215z', 'M312.87936,285.29914l29.76865,-3.41118l0.71343,8.73692l-33.44985,-1.70803z'],
        ['M371.53494,257.2226c-5.3407,-1.86741 -9.84549,-0.99074 -11.34052,-0.17407l-0.02947,-0.01324c0.0301,-0.00458 0.04264,0.00337 0.03269,0.01116c0.51449,-0.33235 62.44572,-40.28995 70.21684,-21.8115zM360.16495,257.03529c-0.16487,0.02506 -0.85657,0.42566 -1.2654,3.2836l-58.89955,-30.31889z', 'M360.19442,257.04853c0,0 0.00108,-0.0007 0.00322,-0.00208c-0.00089,0.0007 -0.00197,0.0014 -0.00322,0.00208z'],
        ['M409.24903,253.44031c-4.3563,1.32265 -5.54585,2.71876 -5.62356,2.81333l-0.01222,-0.01004c-0.1837,0.27203 -0.74035,1.48841 -0.57864,4.75787l-45.76718,-42.83894z', 'M403.62547,256.25365l-0.01222,-0.01004c0.06335,-0.09382 0.08235,-0.07531 0.01222,0.01004z'],
        ['M292.94,321.04496l3.31075,-9.23873l-24.81246,-15.67426l0,0l0,0l0,0l0,0l48.44426,-22.36542l-13.5895,23.84665l25.09113,1.06631l18.61582,47.08061l-52.31238,-33.04622z', 'M296.25075,311.80623l5.16049,-14.40049l0,0l4.88182,0.20746l-8.60544,15.1007zM301.41124,297.40574l-29.97295,-1.27377l0,0l0,0z'],
        ['M349.2989,302.36012l-42.41417,8.48166l1.38218,-3.27015l0,0l-2.93573,-15.78682l-10.2988,-2.47711l9.58086,-1.38362l-7.18085,-38.61489l32.81153,6.26643l-16.1911,38.30696z', 'M304.61324,287.92408l12.73541,-1.83919l-3.29583,7.79769l-8.72164,-2.09777zM308.40197,308.29791l-1.51724,2.54387l1.38218,-3.27015z'],
        ['M150,290l30,-30l50,10l-24,16l14,14l-50,10l21.17647,-14.11765z', 'M191.17647,295.88235l14.82353,-9.88235l14,14z'],
        ['M324.40827,86.1091l-24.66974,16.09646c-4.37001,-6.69756 10.04945,-11.38915 24.66974,-16.09646zM324.40827,86.1091l0.45123,-0.29442l-16.06828,-25.33943l22.46895,-3.97479l18.72801,12.91832c4.53222,6.94617 -10.24683,11.75354 -25.05014,16.51976l15.98481,25.2078l-22.46194,3.97305l6.36357,-29.14428c-0.13874,0.04467 -0.27748,0.08933 -0.4162,0.134z', 'M324.8595,85.81469l0,0l0,0zM324.93803,85.93854c-0.03785,0.01219 -0.07571,0.02438 -0.11356,0.03656l0.03503,-0.16042z'],
        ['M388.92139,223.32159c120.27613,-2.24369 208.69681,19.62691 208.69681,19.62691l0.0116,0.00003l0,0l21.89025,-21.07283c0,0 8.54573,10.72909 8.51542,21.16115c-0.03104,10.68629 -208.75655,121.23392 -208.75655,121.23392z', 'M576.29425,263.48737l-8.25852,-20.6248l29.59407,0.08596z'],
        ['M272.11155,196.81853l-22.11155,-66.81853l32.64261,64.05832c-6.17669,-1.04412 -8.29,-0.01335 -8.40866,0.04617l-0.00118,-0.00313c-0.09777,0.03858 -0.90866,0.4932 -2.12123,2.71717z', 'M274.23395,194.10449l-0.00118,-0.00313c0.01275,-0.00503 0.01337,-0.00299 0.00118,0.00313z'],
        ['M386.40934,270.73027c77.4169,15.77719 129.39044,24.64733 142.22581,18.90659l5.47489,12.24096l0,0l40.60627,23.71608l-21.58416,18.81428z', 'M522.90574,295.33395c2.3982,-4.10616 5.41393,-5.55599 5.72941,-5.69709l5.47489,12.24096z'],
        ['M525,345l32.31797,18.04379l23.72504,-18.37449l6.0793,21.87157l-87.1223,33.45913z', 'M557.31797,363.04379l0,0l0,0zM557.31797,363.04379l-23.71188,18.3643l-6.07498,-21.85933z'],
        ['M250,150l-15.25375,32.05024l29.10275,-7.1942zM225.84314,153.10482l-45.84314,36.89518l63.48255,20.45333z', 'M205.60228,189.25463v0.2l0.01028,-0.20254z'],
        ['M598.5487,408.11025l-42.28034,59.10612l0,0l0,0l0,0l0,0l-37.73535,-87.53938l36.05709,27.56285l0.78705,28.12706z', 'M556.26835,467.21638l0,0l-3.83913,-29.98827l2.94792,-1.86119z'],
        ['M570,290l5.8176,33.58557l-28.17314,-14.439c-14.32289,2.0978 -28.17688,4.12693 -28.17688,4.12693l19.08162,-8.78834l-16.12739,-8.26544l27.9654,2.81326z', 'M538.5492,304.48516l11.83801,-5.45218l9.61279,0.96702l15.8176,23.58557z'],
        ['', ''],
        ['M419.20345,384.34607c5.84471,3.31306 8.73763,3.6008 9.66385,3.54578c0.42499,-0.23215 3.1344,-1.90017 6.91252,-9.02761c31.69382,-59.79056 16.74714,-243.86183 15.04159,-262.65257c-3.42808,-8.00965 -2.79156,-12.73288 -0.15048,-1.551l29.19538,-6.89574l-2.9917,-29.85552c34.20913,-3.42795 37.91991,47.22024 32.18962,22.95919c0.38462,1.62842 24.31406,221.98511 -20.27182,306.0966c-7.77973,14.6765 -19.84373,29.87172 -38.60351,37.019c-20.69209,7.88347 -41.8301,3.18373 -60.57335,-7.44083z', 'M450.8214,116.21167c-0.08956,-0.98671 -0.14261,-1.51768 -0.15048,-1.551l29.19538,-6.89574l2.99069,29.84549c-18.44587,1.84838 -28.02391,-12.02551 -32.0356,-21.39876z'],
        ['M224.2508,238.43767c2.45124,34.92471 9.86647,54.1339 17.20117,64.07402c4.07985,5.52909 8.23969,8.37799 12.35502,9.97162c7.6805,2.97421 19.93299,3.22543 38.3191,-2.95557c81.13187,-27.27474 190.74601,-146.01269 198.63572,-155.10754c0.92148,-1.70426 1.39237,-2.07227 0.44507,-0.53656l25.35355,15.63944l29.94723,-4.25223c1.89126,13.31958 -6.82114,24.30584 -4.2348,20.11304c-5.40958,8.76963 -129.41512,146.85628 -231.02757,181.01613c-24.82193,8.34458 -52.91092,12.17808 -79.10499,2.03465c-15.35707,-5.9469 -28.53272,-16.15682 -38.96756,-30.29832c-16.97074,-22.99908 -25.95119,-55.26903 -28.7747,-95.49783z', 'M490.76181,154.42019c0.26299,-0.30316 0.41295,-0.48449 0.44507,-0.53656l25.35355,15.63944l-29.45692,4.18261c-1.21879,-8.58358 1.97318,-16.16889 3.6583,-19.28549z']
    ];
    for (var i = 0; i < paths.length; i++) {
        var entry = paths[i],
            result = results[i],
            path1 = PathItem.create(entry[0]),
            path2 = PathItem.create(entry[1]);
        compareBoolean(path1.unite(path2), result[0], 'path1.unite(path2); // Test ' + (i + 1));
        compareBoolean(path1.intersect(path2), result[1], 'path1.intersect(path2); // Test ' + (i + 1));
    }
});

test('#1506', function () {
    var path1 = new Path('M250,175c27.61424,0 50,22.38576 50,50c0,27.61424 -22.38576,50 -50,50c-9.10718,0 -17.64567,-2.43486 -25,-6.68911c14.94503,-8.64524 25,-24.80383 25,-43.31089c0,-18.50706 -10.05497,-34.66565 -25,-43.31089c7.35433,-4.25425 15.89282,-6.68911 25,-6.68911z');
    var path2 = new Path('M250,225c0,-27.61424 22.38576,-50 50,-50c27.61424,0 50,22.38576 50,50c0,27.61424 -22.38576,50 -50,50c-27.61424,0 -50,-22.38576 -50,-50z');
    var result = 'M250,175c9.10718,0 17.64567,2.43486 25,6.68911c-14.94503,8.64523 -25,24.80383 -25,43.31089c0,18.50706 10.05497,34.66566 25,43.31089c-7.35433,4.25425 -15.89282,6.68911 -25,6.68911c-9.10718,0 -17.64567,-2.43486 -25,-6.68911c14.94503,-8.64524 25,-24.80383 25,-43.31089c0,-18.50706 -10.05497,-34.66565 -25,-43.31089c7.35433,-4.25425 15.89282,-6.68911 25,-6.68911z';
    compareBoolean(path1.subtract(path2), result);
});

test('#1513', function () {
    var path1  = PathItem.create('M100,200v-100h200v100z');
    var path2  = PathItem.create('M200,100c55.22847,0 100,44.77153 100,100h-200c0,-55.22847 44.77153,-100 100,-100z');
    var result = 'M100,100h200v100c0,-55.22847 -44.77153,-100 -100,-100c-55.22847,0 -100,44.77153 -100,100z';
    compareBoolean(path1.subtract(path2), result);
});