---
title: DropDownList
page_title: Configuration, methods and events of Kendo UI DropDownList
description: "Learn how to control your DropDown UI widget's behavior to suit your needs: open, close, enable, disable the widget. Events data and code examples available."
res_type: api
component: dropdownlist
---

# kendo.ui.DropDownList

Represents the Kendo UI DropDownList widget. Inherits from [Widget](/api/javascript/ui/widget).

## Configuration

### animation `Boolean|Object`

Configures the opening and closing animations of the suggestion popup. Setting the `animation` option to `false` will disable the opening and closing animations. As a result the suggestion popup will open and close instantly.

`animation:true` is not a valid configuration.

#### Example - disable open and close animations

    <input id="dropdownlist" />
    <script>
    $("#dropdownlist").kendoDropDownList({
      dataSource: {
        data: ["One", "Two"]
      },
      animation: false
    });
    </script>

#### Example - configure the animation

    <input id="dropdownlist" />
    <script>
    $("#dropdownlist").kendoDropDownList({
      dataSource: ["One", "Two"],
      animation: {
       close: {
         effects: "fadeOut zoom:out",
         duration: 300
       },
       open: {
         effects: "fadeIn zoom:in",
         duration: 300
       }
      }
    });
    </script>

### animation.close `Object`

#### Example - configure the close animation

    <input id="dropdownlist" />
    <script>
    $("#dropdownlist").kendoDropDownList({
      dataSource: ["One", "Two"],
      animation: {
       close: {
         effects: "zoom:out",
         duration: 300
       }
      }
    });
    </script>

### animation.close.effects `String`

The effect(s) to use when playing the close animation. Multiple effects should be separated with a space.

[Complete list of available animations](/api/javascript/effects/common)

### animation.close.duration `Number` *(default: 100)*

The duration of the close animation in milliseconds.

### animation.open `Object`

The animation played when the suggestion popup is opened.

#### Example - configure the open animation

    <input id="dropdownlist" />
    <script>
    $("#dropdownlist").kendoDropDownList({
      animation: {
       open: {
         effects: "zoom:in",
         duration: 300
       }
      }
    });
    </script>

### animation.open.effects `String`

The effect(s) to use when playing the open animation. Multiple effects should be separated with a space.

[Complete list of available animations](/api/javascript/effects/common)

### animation.open.duration `Number` *(default: 200)*

The duration of the open animation in milliseconds.

### autoBind `Boolean`*(default: true)*

Controls whether to bind the widget to the data source on initialization.

#### Example

    <input id="dropdownlist" />
    <script>
    $("#dropdownlist").kendoDropDownList({
        autoBind: false
    });
    </script>

### autoWidth `Boolean`

If set to `true`, the widget automatically adjusts the width of the popup element and does not wrap up the item label.

> Note: Virtualized list doesn't support the auto-width functionality.

#### Example - enable autoWidth

    <input id="dropdownlist" style="width: 100px;" />
    <script>
    $("#dropdownlist").kendoDropDownList({
      autoWidth: true,
      dataSource: {
        data: ["Short item", "An item with really, really long text"]
      }
    });
    </script>

### cascadeFrom `String`

Use it to set the Id of the parent DropDownList widget.
[Help topic showing how cascading functionality works](/web/dropdownlist/cascading)

#### Example

    <input id="parent" />
    <input id="child" />
    <script>
    $("#parent").kendoDropDownList({
        dataTextField: "parentName",
        dataValueField: "parentId",
        dataSource: [
            { parentName: "Parent1", parentId: 1 },
            { parentName: "Parent2", parentId: 2 }
        ]
    });

    $("#child").kendoDropDownList({
        cascadeFrom: "parent",
        dataTextField: "childName",
        dataValueField: "childId",
        dataSource: [
            { childName: "Child1", childId: 1, parentId: 1 },
            { childName: "Child2", childId: 2, parentId: 2 },
            { childName: "Child3", childId: 3, parentId: 1 },
            { childName: "Child4", childId: 4, parentId: 2 }
        ]
    });
    </script>

### cascadeFromField `String`

Defines the field to be used to filter the data source. If not defined the [parent's dataValueField option will be used](/api/javascript/ui/dropdownlist/configuration/datavaluefield).
[Help topic showing how cascading functionality works](/web/dropdownlist/cascading)

#### Example

    <input id="parent" />
    <input id="child" />
    <script>
    $("#parent").kendoDropDownList({
        dataTextField: "name",
        dataValueField: "id",
        dataSource: [
            { name: "Parent1", id: 1 },
            { name: "Parent2", id: 2 }
        ]
    });

    $("#child").kendoDropDownList({
        cascadeFrom: "parent",
        cascadeFromField: "parentId",
        dataTextField: "name",
        dataValueField: "id",
        dataSource: [
            { name: "Child1", id: 1, parentId: 1 },
            { name: "Child2", id: 2, parentId: 2 },
            { name: "Child3", id: 3, parentId: 1 },
            { name: "Child4", id: 4, parentId: 2 }
        ]
    });
    </script>

### cascadeFromParentField `String`

Defines the parent field to be used to retain value from. This value will be used further to filter the dataSource. If not defined the value from the [parent's dataValueField will be used](/api/javascript/ui/dropdownlist/configuration/datavaluefield).

#### Example

    <input id="parent" />
    <input id="child" />
    <script>
    $("#parent").kendoDropDownList({
        dataTextField: "name",
        dataValueField: "id",
        dataSource: [
            { name: "Parent1", id: 123, cascadeId: 1 },
            { name: "Parent2", id: 234, cascadeId: 2 }
        ]
    });

    $("#child").kendoDropDownList({
        cascadeFrom: "parent",
        cascadeFromField: "parentId",
        cascadeFromParentField: "cascadeId",
        dataTextField: "name",
        dataValueField: "id",
        dataSource: [
            { name: "Child1", id: 1, parentId: 1 },
            { name: "Child2", id: 2, parentId: 2 },
            { name: "Child3", id: 3, parentId: 1 },
            { name: "Child4", id: 4, parentId: 2 }
        ]
    });
    </script>

### dataSource `Object|Array|kendo.data.DataSource`

The data source of the widget which is used to display a list of values. Can be a JavaScript object which represents a valid data source configuration, a JavaScript array or an existing [kendo.data.DataSource](/api/javascript/data/datasource)
instance.

If the `dataSource` option is set to a JavaScript object or array the widget will initialize a new [kendo.data.DataSource](/api/javascript/data/datasource) instance using that value as data source configuration.

If the `dataSource` option is an existing [kendo.data.DataSource](/api/javascript/data/datasource) instance the widget will use that instance and will **not** initialize a new one.

#### Example - set dataSource as a JavaScript object

    <input id="dropdownlist" />
    <script>
    $("#dropdownlist").kendoDropDownList({
      dataSource: {
        data: ["One", "Two"]
      }
    });
    </script>

#### Example - set dataSource as a JavaScript array

    <input id="dropdownlist" />
    <script>
    var data = ["One", "Two"];
    $("#dropdownlist").kendoDropDownList({
      dataSource: data
    });
    </script>

#### Example - set dataSource as an existing kendo.data.DataSource instance

    <input id="dropdownlist" />
    <script>
    var dataSource = new kendo.data.DataSource({
      transport: {
        read: {
          url: "https://demos.telerik.com/kendo-ui/service/products",
          dataType: "jsonp"
        }
      }
    });
    $("#dropdownlist").kendoDropDownList({
      dataSource: dataSource,
      dataTextField: "ProductName",
      dataValueField: "ProductID"
    });
    </script>

### dataTextField `String`*(default: "")*

The field of the data item that provides the text content of the list items. The widget will filter the data source based on this field.

> When `dataTextField` is defined, the `dataValueField` option also should be set.

#### Example - set the dataTextField

    <input id="dropdownlist" />
    <script>
      $("#dropdownlist").kendoDropDownList({
        dataSource: [
          { Name: "Parent1", Id: 1 },
          { Name: "Parent2", Id: 2 }
        ],
        dataTextField: "Name",
        dataValueField: "Id"
      });
    </script>

### dataValueField `String`*(default: "")*

The field of the data item that provides the value of the widget.

> When `dataValueField` is defined, the `dataTextField` option also should be set.

#### Example - set the dataValueField

    <input id="dropdownlist" />
    <script>
    $("#dropdownlist").kendoDropDownList({
        dataSource: [
            { Name: "Parent1", Id: 1 },
            { Name: "Parent2", Id: 2 }
        ],
        dataTextField: "Name",
        dataValueField: "Id"
    });
    </script>

### delay `Number`*(default: 500)*

 Specifies the delay in milliseconds before the search-text typed by the end user is sent for filtering.

#### Example - set the delay

    <input id="dropdownlist" />
    <script>
    $("#dropdownlist").kendoDropDownList({
        filter: true,
        delay: 1000 // wait 1 second before filtering with the user input
    });
    </script>

### enable `Boolean`*(default: true)*

If set to `false` the widget will be disabled and will not allow user input. The widget is enabled by default and allows user input.

#### Example - disable the widget

    <input id="dropdownlist" />
    <script>
    $("#dropdownlist").kendoDropDownList({
      enable: false
    });
    </script>

### enforceMinLength `Boolean` *(default: false)*

If set to `true` the widget will not show all items when the text of the search input cleared. By default the widget shows all items when the text of the search input is cleared. Works in conjunction with [minLength](/api/javascript/ui/dropdownlist#configuration-minLength).

#### Example - enforce minLength

    <input id="dropdownlist" />
    <script>
    $("#dropdownlist").kendoDropDownList({
        filter: "startswith",
        autoBind: false,
        minLength: 3,
        enforceMinLength: true,
        dataTextField: "ProductName",
        dataValueField: "ProductID",
        dataSource: {
            type: "odata",
            serverFiltering: true,
            transport: {
                read: {
                    url: "//demos.telerik.com/kendo-ui/service/Northwind.svc/Products",
                }
            }
        }
    });
    </script>

### fillMode `String`*(default: "solid")*

Sets a value controlling how the color is applied. Can also be set to the following string values:

- null
- "solid"
- "flat"
- "outline"

#### Example - sets the fillMode

    <select id="dropdown">
        <option>Item1</option>
        <option>Item2</option>
    </select>
    <script>
      $("#dropdown").kendoDropDownList({
        filter: "contains",
        fillMode: "flat"
      });
    </script>

### filter `String`*(default: "none")*

The filtering method used to determine the suggestions for the current value. Filtration is turned off by default, and can be performed over `string` values only (either the widget's data has to be an array of strings, or over the field, configured in the [`dataTextField`](/api/javascript/ui/dropdownlist#configuration-dataTextField) option).
The supported filter values are `startswith`, `endswith` and `contains`.

#### Example - set the filter

    <input id="dropdownlist" />
    <script>
    $("#dropdownlist").kendoDropDownList({
      dataSource: ["Chai", "Chang", "Tofu"],
      filter: "contains"
    });
    </script>

### filterTitle `String`

When filtering is enabled, allows custom title to be defined for the filter input element.

#### Example - set the filter

    <input id="dropdownlist" />
    <script>
    $("#dropdownlist").kendoDropDownList({
      dataSource: ["Chai", "Chang", "Tofu"],
      filter: "contains",
      filterTitle: "custom title"
    });
    </script>

### fixedGroupTemplate `String|Function`

The [template](/api/javascript/kendo/methods/template) used to render the fixed header group. By default the widget displays only the value of the current group.

    <input id="customers" style="width: 400px" />
    <script>
        $(document).ready(function() {
            $("#customers").kendoDropDownList({
                dataTextField: "ContactName",
                dataValueField: "CustomerID",
                fixedGroupTemplate: "Fixed group: #: data #",
                height: 400,
                dataSource: {
                    type: "odata",
                    transport: {
                        read: "https://demos.telerik.com/kendo-ui/service/Northwind.svc/Customers"
                    },
                    group: { field: "Country" }
                }
            });
        });
    </script>

### footerTemplate `String|Function`

The [template](/api/javascript/kendo/methods/template) used to render the footer template. The footer template receives the widget itself as a part of the data argument. Use the widget fields directly in the template.

#### Parameters

##### instance `Object`

The widget instance.

#### Example - specify footerTemplate as a string

    <input id="dropdownlist" />
    <script>
    $("#dropdownlist").kendoDropDownList({
      dataSource: [
        { id: 1, name: "Apples" },
        { id: 2, name: "Oranges" }
      ],
      dataTextField: "name",
      dataValueField: "id",
      footerTemplate: 'Total <strong>#: instance.dataSource.total() #</strong> items found'
    });
    </script>

### groupTemplate `String|Function`

The [template](/api/javascript/kendo/methods/template) used to render the groups. By default the widget displays only the value of the group.

    <input id="customers" style="width: 400px" />
    <script>
        $(document).ready(function() {
            $("#customers").kendoDropDownList({
                dataTextField: "ContactName",
                dataValueField: "CustomerID",
                groupTemplate: "Group: #: data #",
                height: 400,
                dataSource: {
                    type: "odata",
                    transport: {
                        read: "https://demos.telerik.com/kendo-ui/service/Northwind.svc/Customers"
                    },
                    group: { field: "Country" }
                }
            });
        });
    </script>

### height `Number`*(default: 200)*

The height of the suggestion popup in pixels. The default value is 200 pixels.

#### Example - set the height

    <input id="dropdownlist" />
    <script>
    $("#dropdownlist").kendoDropDownList({
      dataTextField: "ProductName",
      dataValueField: "ProductID",
	  height: 500,
      dataSource: {
        transport: {
          read: {
            dataType: "jsonp",
            url: "//demos.telerik.com/kendo-ui/service/Products",
          }
        }
      }
    });
    </script>

### ignoreCase `Boolean`*(default: true)*

If set to `false` case-sensitive search will be performed to find suggestions. The widget performs case-insensitive searching by default.

#### Example - disable case-insensitive suggestions

    <input id="dropdownlist" />
    <script>
    $("#dropdownlist").kendoDropDownList({
      ignoreCase: false
    });
    </script>

### index `Number`*(default: 0)*

The index of the initially selected item. The index is `0` based.

#### Example - select second item

    <input id="dropdownlist" />
    <script>
    var items = [{ text: "Item 1", value: "1" }, { text: "Item 2", value: "2" }];
    $("#dropdownlist").kendoDropDownList({
        dataTextField: "text",
        dataValueField: "value",
        dataSource: items,
        index: 1
    });
    </script>

### messages `Object`

The text messages displayed in the widget. Use this option to customize or localize the messages.

#### Example - customize DropDownList messages

    <input id="dropdownlist" />
    <script>
      var items = [{ text: "Item 1", value: "1" }, { text: "Item 2", value: "2" }];
      $("#dropdownlist").kendoDropDownList({
        dataTextField: "text",
        dataValueField: "value",
        dataSource: items,
        index: 1,
        messages: {
          noData: "There is no data!"
        }
      });
    </script>

### messages.noData `String` *(default: "No data found.")*

The text message shown in the noDataTemplate when no data is available in the widget drop-down.

#### Example - customize noData message

    <input id="dropdownlist" />
    <script>
      $("#dropdownlist").kendoDropDownList({
        dataSource: [],
        messages: {
          noData: "There is no data!"
        }
      });
    </script>

### minLength `Number`*(default: 1)*

The minimum number of characters the user must type before a filter is performed. Set to higher value than `1` if the search could match a lot of items.

> Widget will initiate a request when input value is cleared. If you would like to prevent this behavior please check the [filtering](/api/javascript/ui/dropdownlist/events/filtering) event for more details.

#### Example - set minLength

    <input id="dropdownlist" />
    <script>
    $("#dropdownlist").kendoDropDownList({
      dataSource: ["Chai", "Chang", "Tofu"],
      filter: "contains",
      minLength: 3
    });
    </script>

### noDataTemplate `String|Function|Boolean` *(default: true)*

The [template](/api/javascript/kendo/methods/template) used to render the "no data" template, which will be displayed if no results are found or the underlying data source is empty.
The noData template receives the widget itself as a part of the data argument. The template will be evaluated on every widget data bound.

> **Important** The popup will open when 'noDataTemplate' is defined

#### Example - specify noDataTemplate as a string

    <input id="dropdownlist" />
    <script>
    $("#dropdownlist").kendoDropDownList({
      dataSource: [],
      dataTextField: "name",
      dataValueField: "id",
      noDataTemplate: 'No Data!'
    });
    </script>

### popup `Object`

The options that will be used for the popup initialization. For more details about the available options
refer to [Popup](/api/javascript/ui/popup) documentation.

#### Example - append the popup to a specific element

    <div id="container">
        <input id="dropdownlist" />
    </div>
    <script>
    $("#dropdownlist").kendoDropDownList({
      dataSource: [
        { id: 1, name: "Apples" },
        { id: 2, name: "Oranges" }
      ],
      dataTextField: "name",
      dataValueField: "id",
      popup: {
        appendTo: $("#container")
      }
    });
    </script>


### popup.appendTo `String`

Defines a jQuery selector that will be used to find a container element, where the popup will be appended to.

#### Example - append the popup to a specific element

    <div id="container">
        <input id="dropdownlist" />
    </div>
    <script>
    $("#dropdownlist").kendoDropDownList({
      dataSource: [
        { id: 1, name: "Apples" },
        { id: 2, name: "Oranges" }
      ],
      dataTextField: "name",
      dataValueField: "id",
      popup: {
        appendTo: $("#container")
      }
    });
    </script>

### popup.origin `String`

Specifies how to position the popup element based on anchor point. The value is
space separated "y" plus "x" position.

The available "y" positions are:
- "bottom"
- "center"
- "top"

The available "x" positions are:
- "left"
- "center"
- "right"

#### Example - append the popup to a specific element

    <div id="container">
        <input id="dropdownlist" />
    </div>
    <script>
    $("#dropdownlist").kendoDropDownList({
      dataSource: [
        { id: 1, name: "Apples" },
        { id: 2, name: "Oranges" }
      ],
      dataTextField: "name",
      dataValueField: "id",
      popup: {
        origin: "top left"
      }
    });
    </script>

### popup.position `String`

Specifies which point of the popup element to attach to the anchor's origin point. The value is
space separated "y" plus "x" position.

The available "y" positions are:
- "bottom"
- "center"
- "top"

The available "x" positions are:
- "left"
- "center"
- "right"

#### Example - append the popup to a specific element

  <div id="container">
      <input id="dropdownlist" />
  </div>
  <script>
  $("#dropdownlist").kendoDropDownList({
    dataSource: [
      { id: 1, name: "Apples" },
      { id: 2, name: "Oranges" }
    ],
    dataTextField: "name",
    dataValueField: "id",
    popup: {
      position: "top center"
    }
  });
  </script>
  <style>
    #container{
      position: absolute;
      top: 50%;
      left: 50%;
      margin-top: -50px;
      margin-left: -50px;
      width: 100px;
      height: 100px;
    }
  </style>

### optionLabel `String | Object`*(default: "")*

 Define the text of the default empty item. If the value is an object, then the widget will use it as a valid data item.
 Note that the optionLabel will not be available if the widget is empty.

> * If `optionLabel` is an object, it needs to have at least `dataValueField` and `dataTextField` properties. Otherwise, widget will show `undefined`.
Note that the very same optionLabel object will be passed to the [valueTemplate](/api/javascript/ui/dropdownlist/configuration/valuetemplate). **You need to ensure that all required by the valueTemplate properties are present
in the optionLabel object**.
> * Since Q1 2015 (2015.1.318), the option label is rendered as a separate header template. The benefits of this change are:
- the widget's value will be empty string even when `dataValueField` and `dataTextField` options are equal or not defined
- the widget will not throw an exception when a custom item template is used and `optionLabel` is string
- option label has a separate template, that gives more freedom for customization
> * [Not relevant after Q1 2015] Widget's value will be equal to the `optionLabel` if the `dataValueField` and `dataTextField` options are equal or not defined

#### Example - specify optionLabel as a string

    <input id="dropdownlist" />
    <script>
    $("#dropdownlist").kendoDropDownList({
        dataSource: ["Apples", "Oranges"],
        optionLabel: "Select a fruit..."
    });
    </script>

#### Example - specify optionLabel as an object

    <input id="dropdownlist" />
    <script>
    $("#dropdownlist").kendoDropDownList({
        dataSource: [
            { productName: "Product 1", productId: 1 },
            { productName: "Product 2", productId: 2 }
        ],
        dataTextField: "productName",
        dataValueField: "productId",
        optionLabel: {
            productName: "Select a product...",
            productId: ""
        }
    });
    </script>

### optionLabelTemplate `String|Function`

The [template](/api/javascript/kendo/methods/template) used to render the option label. Use optionLabelTemplate if you want to customize the markup of the optionLabel.

> * [optionLabel](/api/javascript/ui/dropdownlist/configuration/optionlabel) has to be defined for the optionLabelTemplate to take effect.
> * Define the [optionLabel](/api/javascript/ui/dropdownlist/configuration/optionlabel) as **object** if complex template structure is used

#### Example - specify the optionLabelTemplate as a string
    <input id="dropdownlist" />
    <script>
    $("#dropdownlist").kendoDropDownList({
        dataSource: [
          { productName: "Product 1", productId: 1 },
          { productName: "Product 2", productId: 2 },
          { productName: "Product 3", productId: 3 },
          { productName: "Product 4", productId: 4 }
        ],
        dataTextField: "productName",
        dataValueField: "productId",
        optionLabel: "-- Please select --",
        optionLabelTemplate:'<span style="color:red">-- Please select --</span>'
    });
    </script>

### rounded `String`*(default: "medium")*

Sets a value controlling the border radius. Can also be set to the following string values:

- null
- "small"
- "medium"
- "large"
- "full"

#### Example - sets the rounded value

    <select id="dropdown">
        <option>Item1</option>
        <option>Item2</option>
    </select>
    <script>
      $("#dropdown").kendoDropDownList({
        filter: "contains",
        rounded: "large"
      });
    </script>

### size `String`*(default: "medium")*

Sets a value controlling size of the component. Can also be set to the following string values:

- "small"
- "medium"
- "large"
- null

#### Example - sets a size

    <select id="dropdown">
        <option>Item1</option>
        <option>Item2</option>
    </select>
    <script>
      $("#dropdown").kendoDropDownList({
        filter: "contains",
        size: "large"
      });
    </script>

### headerTemplate `String|Function`

Specifies a static HTML content, which will be rendered as a header of the popup element.

> * The header content **must be wrapped** with a HTML tag if it contains more than one element. This is applicable also when header content is just a string/text.
> * Widget does not pass model data to the header template. Use the header template only with static HTML (only presentation elements and not functionality like data bound inputs or fields).

#### Example - specify headerTemplate as a string

    <input id="dropdownlist" />
    <script>
    $("#dropdownlist").kendoDropDownList({
      dataSource: [
        { id: 1, name: "Apples" },
        { id: 2, name: "Oranges" }
      ],
      dataTextField: "name",
      dataValueField: "id",
      headerTemplate: '<div><h2>Fruits</h2></div>'
    });
    </script>

### template `String|Function`

The [template](/api/javascript/kendo/methods/template) used to render the items. By default the widget displays only the text of the data item (configured via `dataTextField`).

#### Example - specify template as a function

    <input id="dropdownlist" />
    <script id="template" type="text/x-kendo-template">
      <span>
        <img src="/img/#: id #.png" alt="#: name #" />
        #: name #
      </span>
    </script>
    <script>
    $("#dropdownlist").kendoDropDownList({
      dataSource: [
        { id: 1, name: "Apples" },
        { id: 2, name: "Oranges" }
      ],
      dataTextField: "name",
      dataValueField: "id",
      template: kendo.template($("#template").html())
    });
    </script>

#### Example - specify template as a string

    <input id="dropdownlist" />
    <script>
    $("#dropdownlist").kendoDropDownList({
      dataSource: [
        { id: 1, name: "Apples" },
        { id: 2, name: "Oranges" }
      ],
      dataTextField: "name",
      dataValueField: "id",
      template: '<span><img src="/img/#: id #.png" alt="#: name #" />#: name #</span>'
    });
    </script>

### valueTemplate `String|Function`

The [valueTemplate](/api/javascript/kendo/methods/template) used to render the selected value. By default the widget displays only the text of the data item (configured via `dataTextField`).

#### Example - specify valueTemplate as a function

    <input id="dropdownlist" />
    <script id="valueTemplate" type="text/x-kendo-template">
        <strong>#: name #</strong>
    </script>
    <script>
    $("#dropdownlist").kendoDropDownList({
      dataSource: [
        { id: 1, name: "Apples" },
        { id: 2, name: "Oranges" }
      ],
      dataTextField: "name",
      dataValueField: "id",
      valueTemplate: kendo.template($("#valueTemplate").html())
    });
    </script>

#### Example - specify template as a string

    <input id="dropdownlist" />
    <script>
    $("#dropdownlist").kendoDropDownList({
      dataSource: [
        { id: 1, name: "Apples" },
        { id: 2, name: "Oranges" }
      ],
      dataTextField: "name",
      dataValueField: "id",
      valueTemplate: '<strong>#: name #</strong>'
    });
    </script>

### text `String`*(default: "")*

The text of the widget used when the `autoBind` is set to `false`.

#### Example - specify text of the widget

    <input id="dropdownlist" />
    <script>
    $("#dropdownlist").kendoDropDownList({
         autoBind: false,
         text: "Chai"
    });
    </script>

### value `String`*(default: "")*

The value of the widget.

#### Example - specify value of the widget

    <input id="dropdownlist" />
    <script>
    $("#dropdownlist").kendoDropDownList({
         dataSource: ["Item1", "Item2"],
         value: "Item1"
    });
    </script>

### valuePrimitive `Boolean`*(default: false)*

Specifies the [value binding](/framework/mvvm/bindings/value) behavior for the widget when the initial model value is null. If set to `true`, the View-Model field will be updated with the primitive value of the selected item's field (defined in the dataValueField option).if set to `false`, the View-Model field will be updated with the selected item - the entire non-primitive object.


#### Example - specify that the View-Model field should be updated with the selected item value

    <div id="example">
      <div>Change the value of the dropdowns and observe the logged value in the console.</div>
      <br/>

      <select id="dropdownPrimitive" data-bind="value: selectedProductId, source: products" >
      </select>

      <select id="dropdown" data-bind="value: selectedProduct, source: products" >
      </select>
    </div>
    <script>
    $("#dropdownPrimitive").kendoDropDownList({
      valuePrimitive: true,
      dataTextField: "name",
      dataValueField: "id",
      optionLabel: "Select product..."
    });

    $("#dropdown").kendoDropDownList({
      valuePrimitive: false,
      dataTextField: "name",
      dataValueField: "id",
      optionLabel: "Select product..."
    });

    var viewModel = kendo.observable({
      selectedProductId: null,
      selectedProduct: null,
      products: [
        { id: 1, name: "Coffee" },
        { id: 2, name: "Tea" },
        { id: 3, name: "Juice" }
      ]
    });

    viewModel.bind("change", function(ev) {
      if (ev.field === "selectedProduct") {
        console.log("value: " + JSON.stringify(this.get(ev.field)));
      } else if (ev.field === "selectedProductId") {
        console.log("value: " + this.get(ev.field));
      }
    });

    kendo.bind($("#example"), viewModel);


### virtual `Boolean|Object`*(default: false)*

Enables the virtualization feature of the widget. The configuration can be set on an object, which contains two properties - `itemHeight` and `valueMapper`.

For detailed information, refer to the [article on virtualization]({% slug virtualization_kendoui_combobox_widget %}).

### virtual.itemHeight `Number`*(default: null)*

Specifies the height of the virtual item. All items in the virtualized list **must** have the same height.
If the developer does not specify one, the framework will automatically set `itemHeight` based on the current theme and font size.

### virtual.mapValueTo `String`*(default: "index")*

The changes introduced with the Kendo UI R3 2016 release enable you to determine if the `valueMapper` must resolve a *value to an `index`* or a *value to a `dataItem`*. This is configured through the `mapValueTo` option that accepts two possible values - `"index"` or `"dataItem"`. By default, the `mapValueTo` is set to `"index"`, which does not affect the current behavior of the virtualization process.

For more information, refer to the [article on virtualization]({% slug virtualization_kendoui_combobox_widget %}#value-mapping).

### virtual.valueMapper `Function`*(default: null)*

The widget calls the `valueMapper` function when the widget receives a value, that is not fetched from the remote server yet.
The widget will pass the selected value(s) in the `valueMapper` function. In turn, the valueMapper implementation should return the **respective data item(s) index/indices**.
> **Important**
>
> As of the Kendo UI R3 2016 release, the implementation of the `valueMapper` function is optional. It is required only if the widget contains an initial value or if the `value` method is used.

#### Example - DropDownList widget with a virtualized list

    <input id="orders" style="width: 400px" />
    <script>
        $(document).ready(function() {
            $("#orders").kendoDropDownList({
                template: '<span class="order-id">#= OrderID #</span> #= ShipName #, #= ShipCountry #',
                dataTextField: "ShipName",
                dataValueField: "OrderID",
                filter: "contains",
                virtual: {
                    itemHeight: 26,
                    valueMapper: function(options) {
                        $.ajax({
                            url: "https://demos.telerik.com/kendo-ui/service/Orders/ValueMapper",
                            type: "GET",
                            dataType: "jsonp",
                            data: convertValues(options.value),
                            success: function (data) {
                                //the **data** is either index or array of indices.
                                //Example:
                                // 10258 -> 10 (index in the Orders collection)
                                // [10258, 10261] -> [10, 14] (indices in the Orders collection)

                                options.success(data);
                            }
                        })
                    }
                },
                height: 520,
                dataSource: {
                    type: "odata",
                    transport: {
                        read: "https://demos.telerik.com/kendo-ui/service/Northwind.svc/Orders"
                    },
                    schema: {
                        model: {
                            fields: {
                                OrderID: { type: "number" },
                                Freight: { type: "number" },
                                ShipName: { type: "string" },
                                OrderDate: { type: "date" },
                                ShipCity: { type: "string" }
                            }
                        }
                    },
                    pageSize: 80,
                    serverPaging: true,
                    serverFiltering: true
                }
            });
        });

        function convertValues(value) {
            var data = {};

            value = $.isArray(value) ? value : [value];

            for (var idx = 0; idx < value.length; idx++) {
                data["values[" + idx + "]"] = value[idx];
            }

            return data;
        }
    </script>

#### Example - DropDownList widget with declarative virtualization config

    <div class="demo-section k-header">
        <h4>Search for shipping name</h4>
        <input id="orders" style="width: 400px"
               data-role="dropdownlist"
               data-bind="value: order, source: source"
               data-text-field="ShipName"
               data-value-field="OrderID"
               data-filter="contains"
               data-virtual="{itemHeight:26,valueMapper:orderValueMapper}"
               data-height="520"
               />
    </div>

    <script>
        $(document).ready(function() {
            var model = kendo.observable({
                    order: "10548",
              source: new kendo.data.DataSource({
                type: "odata",
                transport: {
                  read: "https://demos.telerik.com/kendo-ui/service/Northwind.svc/Orders"
                },
                schema: {
                  model: {
                    fields: {
                      OrderID: { type: "number" },
                      Freight: { type: "number" },
                      ShipName: { type: "string" },
                      OrderDate: { type: "date" },
                      ShipCity: { type: "string" }
                    }
                  }
                },
                pageSize: 80,
                serverPaging: true,
                serverFiltering: true
              })
            });

            kendo.bind($(document.body), model);
        });

        function orderValueMapper(options) {
            $.ajax({
              url: "https://demos.telerik.com/kendo-ui/service/Orders/ValueMapper",
              type: "GET",
              dataType: "jsonp",
              data: convertValues(options.value),
              success: function (data) {
                options.success(data);
              }
            })
        }

        function convertValues(value) {
            var data = {};

            value = $.isArray(value) ? value : [value];

            for (var idx = 0; idx < value.length; idx++) {
                data["values[" + idx + "]"] = value[idx];
            }

            return data;
        }
    </script>

## Fields

### dataSource `kendo.data.DataSource`

The [data source](/api/javascript/data/datasource) of the widget. configured via the [datasource](/api/javascript/ui/dropdownlist/configuration/datasource) option.

> * Changes of the data source will be reflected in the widget.
> * Assigning a new data source would have no effect. Use the [setDataSource](/api/javascript/ui/dropdownlist/methods/setdatasource) method instead.

#### Example - add a data item to the data source
    <input id="dropdownlist" />
    <script>
    $("#dropdownlist").kendoDropDownList({
      dataSource: [
        { name: "Apples" },
        { name: "Oranges" }
      ],
      dataTextField: "name",
      dataValueField: "name"
    });
    var dropdownlist = $("#dropdownlist").data("kendoDropDownList");
    dropdownlist.dataSource.add({ name: "Appricot" });
    dropdownlist.search("A");
    </script>

### span `jQuery`
A jQuery object of the span element which holds the selected text.

#### Example - modify span element

    <input id="dropdownlist" />
    <script>
    $("#dropdownlist").kendoDropDownList({
        dataSource: [
          { name: "Apples" },
          { name: "Oranges" }
        ],
        dataTextField: "name",
        dataValueField: "name"
    });

    var dropdownlist = $("#dropdownlist").data("kendoDropDownList");

    var span = dropdownlist.span;

    span.css("background-color", "red");
    </script>

### filterInput `jQuery`
A jQuery object of the filter input element, where the user types.

> Available only if the filtering is enabled.

#### Example - get input element

    <input id="dropdownlist" />
    <script>
    $("#dropdownlist").kendoDropDownList({
      filter:"startswith",
      dataSource:["Chai", "Chang","Tofu"]
    });

    var dropdownlist = $("#dropdownlist").data("kendoDropDownList");

    var filterInput = dropdownlist.filterInput;
    </script>

### options `Object`
An object, which holds the information about the configuration options of the widget.

#### Example - get options of the widget

    <input id="dropdownlist" />
    <script>
    $("#dropdownlist").kendoDropDownList();

    var dropdownlist = $("#dropdownlist").data("kendoDropDownList");

    var options = dropdownlist.options;
    console.log(options) //open the console to see the options object
    </script>

### list `jQuery`
A jQuery object of the drop-down list element.

#### Example - get list element

    <input id="dropdownlist" />
    <script>
    $("#dropdownlist").kendoDropDownList();

    var dropdownlist = $("#dropdownlist").data("kendoDropDownList");

    var list = dropdownlist.list;
    </script>

### ul `jQuery`
A jQuery object of the `ul` element, which holds the available options.

#### Example - get ul element

    <input id="dropdownlist" />
    <script>
    $("#dropdownlist").kendoDropDownList();

    var dropdownlist = $("#dropdownlist").data("kendoDropDownList");

    var ul = dropdownlist.ul;
    </script>

## Methods

### close

Closes the widget popup.

#### Example - close the suggestion popup

    <input id="dropdownlist" />
    <script>
    $("#dropdownlist").kendoDropDownList({
      dataSource: [ "Apples", "Oranges" ]
    });
    var dropdownlist = $("#dropdownlist").data("kendoDropDownList");
    // Search for items starting with "A" - will open the suggestion popup and show "Apples"
    dropdownlist.search("A");
    // Close the suggestion popup
    dropdownlist.close();
    </script>

### dataItem

Returns the data item at the specified index. If the index is not specified, the selected index will be used.

#### Parameters

##### index `jQuery|Number` *(optional)*

The zero-based index of the data record.

#### Returns

`Object` The raw data record. Returns *undefined* if no data.

#### Example

    <input id="dropdownlist" />
    <script>

    $("#dropdownlist").kendoDropDownList({
      dataSource: [
        { id: 1, name: "Apples" },
        { id: 2, name: "Oranges" }
      ],
      dataTextField: "name",
      dataValueField: "id",
      index: 1
    });

    var dropdownlist = $("#dropdownlist").data("kendoDropDownList");

    // get the dataItem corresponding to the selectedIndex.
    var dataItem = dropdownlist.dataItem();

    // get the dataItem corresponding to the passed index.
    var dataItem = dropdownlist.dataItem(0);
    </script>

### destroy

Prepares the **DropDownList** for safe removal from DOM. Detaches all event handlers and removes jQuery.data attributes to avoid memory leaks. Calls destroy method of any child Kendo widgets.

> This method does not remove the DropDownList element from DOM.

#### Example

    <input id="dropdownlist" />
    <script>
    $("#dropdownlist").kendoDropDownList();
    var dropdownlist = $("#dropdownlist").data("kendoDropDownList");
    dropdownlist.destroy();
    </script>

### focus

Focuses the widget.

#### Example - focus the widget

    <input id="dropdownlist" />
    <script>
    $("#dropdownlist").kendoDropDownList();
    var dropdownlist = $("#dropdownlist").data("kendoDropDownList");
    dropdownlist.focus();
    </script>

### items

Obtains an Array of the DOM elements, which correspond to the data items from the Kendo UI DataSource [view](/api/javascript/data/datasource/methods/view).

#### Returns

`Array` The currently rendered dropdown list items (`<li>` elements).

### enable

Enables or disables the widget.

#### Parameters

##### enable `Boolean`

If set to `true` the widget will be enabled. If set to `false` the widget will be disabled.

#### Example - enable the widget

    <select id="dropdownlist">
        <option>Item1</option>
        <option>Item2</option>
    </select>
    <script>
    $("#dropdownlist").kendoDropDownList({
      enable: false
    });
    var dropdownlist = $("#dropdownlist").data("kendoDropDownList");
    dropdownlist.enable(true);
    </script>

### open

Opens the popup.

#### Example

    <input id="dropdownlist" />
    <script>
    $("#dropdownlist").kendoDropDownList({
      dataSource: [
        { id: 1, name: "Apples" },
        { id: 2, name: "Oranges" }
      ],
      dataTextField: "name",
      dataValueField: "id",
      index: 1
    });

    var dropdownlist = $("#dropdownlist").data("kendoDropDownList");
    dropdownlist.open();
    </script>

### readonly

Controls whether the widget is editable or readonly.

#### Example

    <input id="dropdownlist" />
    <script>
    $("#dropdownlist").kendoDropDownList({
      dataSource: [
        { id: 1, name: "Apples" },
        { id: 2, name: "Oranges" }
      ],
      dataTextField: "name",
      dataValueField: "id",
      index: 1
    });

    // get a reference to the dropdownlist widget
    var dropdownlist = $("#dropdownlist").data("kendoDropDownList");

    // makes dropdownlist readonly
    dropdownlist.readonly();

    // makes dropdownlist editable
    dropdownlist.readonly(false);
    </script>

#### Parameters

##### readonly `Boolean`

The argument, which defines whether the DropDownList should be readonly or editable.

### refresh

Refresh the popup by rendering all items again.

#### Example - refresh the popup items

    <input id="dropdownlist" />
    <script>
    $("#dropdownlist").kendoDropDownList({
      dataSource: [
        { id: 1, name: "Apples" },
        { id: 2, name: "Oranges" }
      ],
      dataTextField: "name",
      dataValueField: "id",
      index: 1
    });

    var dropdownlist = $("#dropdownlist").data("kendoDropDownList");

    dropdownlist.refresh();
    </script>

### search

If the DropDownList filter is enabled, the method searches and filters the results available in the drop-down. If no filter is configured for the widget it selects the first item, which starts with the provided value.

#### Parameters

##### word `String`

The search value.

#### Example - search the widget

    <input id="dropdownlist" />
    <script>
    $("#dropdownlist").kendoDropDownList({
      dataSource: [
        { id: 1, name: "Apples" },
        { id: 2, name: "Oranges" }
      ],
      dataTextField: "name",
      dataValueField: "id"
    });

    var dropdownlist = $("#dropdownlist").data("kendoDropDownList");

    dropdownlist.search("Oranges");
    </script>

### select

Gets or sets the selected item. Selects the item provided as an argument and updates the value and text of the widget.

> * If the widget is not bound (e.g. `autoBind` is set to `false`), the `select` method will **not** pre-fetch the data before continuing with the selection and value setting (unlike the [value](/api/javascript/ui/dropdownlist/methods/value) method), and no item will be selected.
> * The numeric argument indicates the item index in the dropdown, not in the dataSource. If an [`optionLabel`](/api/javascript/ui/dropdownlist/configuration/optionlabel) is used, the dropdown item index can be obtained by incrementing the respective dataSource item index by 1.
> * When **virtualization** is enabled, the method **does not support** selection with a *function predicate*. The predicate function looks only
in the current datasource view, which represents only the active range/page. Hence it will not work properly.
> * This method **does not trigger** [change](/api/javascript/ui/dropdownlist/events/change) event.
This could affect [MVVM value binding](/framework/mvvm/bindings/value). The model bound to the widget will not be updated.
You can overcome this behavior trigerring the `change` event manually using [trigger("change")](/api/javascript/observable/methods/trigger) method.

    <input id="dropdownlist" />
    <script>
    $("#dropdownlist").kendoDropDownList({
      dataSource: [ "Apples", "Oranges" ]
    });

    var dropdownlist = $("#dropdownlist").data("kendoDropDownList");
    dropdownlist.select(1);
    dropdownlist.trigger("change");
    </script>

#### Parameters

##### li `jQuery | Number | Function`

A string, DOM element or jQuery object which represents the item to be selected. A string is treated as a jQuery selector.
A number representing the index of the item or function predicate which returns the correct data item.

#### Returns

`Number` The index of the selected item, if called with no parameters. If a custom value is entered, the returned selected index is `-1`.

`undefined` If called with a parameter as a setter.

#### Example - select item based on jQuery object

    <input id="dropdownlist" />
    <script>
    $("#dropdownlist").kendoDropDownList({
      dataSource: [
        { id: 1, name: "Apples" },
        { id: 2, name: "Oranges" }
      ],
      dataTextField: "name",
      dataValueField: "id"
    });

    var dropdownlist = $("#dropdownlist").data("kendoDropDownList");

    dropdownlist.select(dropdownlist.ul.children().eq(0));
    </script>

#### Example - select item based on index

    <input id="dropdownlist" />
    <script>
    $("#dropdownlist").kendoDropDownList({
      dataSource: [
        { id: 1, name: "Apples" },
        { id: 2, name: "Oranges" }
      ],
      dataTextField: "name",
      dataValueField: "id"
    });

    var dropdownlist = $("#dropdownlist").data("kendoDropDownList");

    dropdownlist.select(1);
    </script>

#### Example - select item based on function predicate

    <input id="dropdownlist" />
    <script>
    $("#dropdownlist").kendoDropDownList({
      dataSource: [
        { id: 1, name: "Apples" },
        { id: 2, name: "Oranges" }
      ],
      dataTextField: "name",
      dataValueField: "id"
    });

    var dropdownlist = $("#dropdownlist").data("kendoDropDownList");

    dropdownlist.select(function(dataItem) {
        return dataItem.name === "Apples";
    });
    </script>

#### Example - get selected index of the widget

    <input id="dropdownlist" />
    <script>
    $("#dropdownlist").kendoDropDownList({
      dataSource: [
        { id: 1, name: "Apples" },
        { id: 2, name: "Oranges" }
      ],
      dataTextField: "name",
      dataValueField: "id",
      index: 1
    });

    var dropdownlist = $("#dropdownlist").data("kendoDropDownList");

    var selectedIndex = dropdownlist.select();
    </script>

### setDataSource

Sets the dataSource of an existing DropDownList and rebinds it.

#### Parameters

##### dataSource `kendo.data.DataSource`

#### Example

    <input id="dropdownlist" />
    <script>
    $("#dropdownlist").kendoDropDownList({
      dataSource: [ "Apples", "Oranges" ]
    });
    var dataSource = new kendo.data.DataSource({
      data: [ "Bananas", "Cherries" ]
    });
    var dropdownlist = $("#dropdownlist").data("kendoDropDownList");
    dropdownlist.setDataSource(dataSource);
    </script>

### text

Gets or sets the text of the DropDownList.

#### Parameters

##### text `String`

The text to set.

#### Returns

`String` The text of the DropDownList.

#### Example - set text of the widget

    <input id="dropdownlist" />
    <script>
    $("#dropdownlist").kendoDropDownList({
      dataSource: [ "Apples", "Oranges" ]
    });

    var dropdownlist = $("#dropdownlist").data("kendoDropDownList");

    dropdownlist.text("Apples");
    </script>

### toggle

Opens or closes the widget popup.

#### Parameters

##### toggle `Boolean`

Defines the whether to open/close the drop-down list.

#### Example - set text of the widget

    <input id="dropdownlist" />
    <script>
    $("#dropdownlist").kendoDropDownList({
      dataSource: [ "Apples", "Oranges" ]
    });

    var dropdownlist = $("#dropdownlist").data("kendoDropDownList");

    dropdownlist.toggle();
    </script>

### value

Gets or sets the value of the DropDownList. The value will not be set if there is no item with such value. If value is undefined, text of the data item is used.

> * If the widget is not bound (e.g. `autoBind` is set to `false`), the `value` method will pre-fetch the data before continuing with the value setting.
**This does not apply when MVVM binding is used.**
> * The widget will **clear the applied filter** if a new value is set. Thus it ensures that the original/whole data set is available for selection.
> * This method **does not trigger** [change](/api/javascript/ui/dropdownlist/events/change) event.
This could affect [MVVM value binding](/framework/mvvm/bindings/value). The model bound to the widget will not be updated.
You can overcome this behavior trigerring the `change` event manually using [trigger("change")](/api/javascript/observable/methods/trigger) method.

    <input id="dropdownlist" />
    <script>
    $("#dropdownlist").kendoDropDownList({
      dataSource: [ "Apples", "Oranges" ]
    });

    var dropdownlist = $("#dropdownlist").data("kendoDropDownList");
    dropdownlist.value("Apples");
    dropdownlist.trigger("change");
    </script>

#### Parameters

##### value `String`

The value to set.

#### Returns

`String` The value of the DropDownList.

#### Example - set value of the widget

    <input id="dropdownlist" />
    <script>
    $("#dropdownlist").kendoDropDownList({
      dataSource: [ "Apples", "Oranges" ]
    });

    var dropdownlist = $("#dropdownlist").data("kendoDropDownList");

    dropdownlist.value("Oranges");
    </script>

#### Example - set value of the widget bound to remote data

    <input id="products" style="width: 100%" />
    <script>
        $("#products").kendoDropDownList({
            dataTextField: "ProductName",
            dataValueField: "ProductID",
            dataSource: {
                transport: {
                    read: {
                        dataType: "jsonp",
                        url: "https://demos.telerik.com/kendo-ui/service/Products",
                    }
               }
            }
        });

        var dropdownlist = $("#products").data("kendoDropDownList");
        dropdownlist.value(2);
    </script>

## Events

### change

Fired when the value of the widget is changed by the user. As of 2015 Q3 SP1 cascading widget will trigger change event when its value is changed due to parent update.

The event handler function context (available via the `this` keyword) will be set to the widget instance.

> The event is not fired when the value of the widget is changed programmatically. If you need to handle changes made by API, wire the [cascade](/api/javascript/ui/dropdownlist/events/cascade) event.

#### Event Data

##### e.sender `kendo.ui.DropDownList`

The widget instance which fired the event.

#### Example - subscribe to the "change" event during initialization

    <input id="dropdownlist" />
    <script>
    $("#dropdownlist").kendoDropDownList({
      dataSource: [ "Apples", "Oranges" ],
      change: function(e) {
        var value = this.value();
        // Use the value of the widget
      }
    });
    </script>

#### Example - subscribe to the "change" event after initialization

    <input id="dropdownlist" />
    <script>
    function dropdownlist_change(e) {
      var value = this.value();
      // Use the value of the widget
    }
    $("#dropdownlist").kendoDropDownList({
      dataSource: [ "Apples", "Oranges" ]
    });

    var dropdownlist = $("#dropdownlist").data("kendoDropDownList");
    dropdownlist.bind("change", dropdownlist_change);
    </script>

### close

Fired when the popup of the widget is closed.

The event handler function context (available via the `this` keyword) will be set to the widget instance.

#### Event Data

##### e.sender `kendo.ui.DropDownList`

The widget instance which fired the event.

#### Example - subscribe to the "close" event during initialization

    <input id="dropdownlist" />
    <script>
    $("#dropdownlist").kendoDropDownList({
      dataSource: [ "Apples", "Oranges" ],
      close: function(e) {
        // handle the event
      }
    });
    </script>

#### Example - subscribe to the "close" event after initialization

    <input id="dropdownlist" />
    <script>
    function dropdownlist_close(e) {
      // handle the event
    }
    $("#dropdownlist").kendoDropDownList({
      dataSource: [ "Apples", "Oranges" ]
    });

    var dropdownlist = $("#dropdownlist").data("kendoDropDownList");
    dropdownlist.bind("close", dropdownlist_close);
    </script>

### dataBound

Fired when the widget is bound to data from its data source.

The event handler function context (available via the `this` keyword) will be set to the widget instance.

#### Event Data

##### e.sender `kendo.ui.DropDownList`

The widget instance which fired the event.

#### Example - subscribe to the "dataBound" event during initialization

    <input id="dropdownlist" />
    <script>
    $("#dropdownlist").kendoDropDownList({
      dataSource: [ "Apples", "Oranges" ],
      dataBound: function(e) {
          // handle the event
      }
    });
    </script>

#### Example - subscribe to the "dataBound" event after initialization

    <input id="dropdownlist" />
    <script>
    function dropdownlist_dataBound(e) {
      // handle the event
    }
    $("#dropdownlist").kendoDropDownList({
      dataSource: [ "Apples", "Oranges" ]
    });
    var dropdownlist = $("#dropdownlist").data("kendoDropDownList");
    dropdownlist.bind("dataBound", dropdownlist_dataBound);
    </script>

### filtering

Fired when the widget is about to filter the data source.

The event handler function context (available via the `this` keyword) will be set to the widget instance.

#### Event Data

##### e.sender `kendo.ui.DropDownList`

The widget instance which fired the event.

##### e.filter `Object`

The filter descriptor that will be used to filter the data source.

> The data source filters the data items client-side unless the [data source serverFiltering](/api/javascript/data/datasource/configuration/serverfiltering) option is set to `true`.

#### Example - subscribe to the "filtering" event during initialization

    <input id="dropdownlist" />
    <script>
    $("#dropdownlist").kendoDropDownList({
      dataSource: [ "Apples", "Oranges" ],
      filter: "startswith",
      filtering: function(e) {
          //get filter descriptor
          var filter = e.filter;

          // handle the event
      }
    });
    </script>

#### Example - subscribe to the "filtering" event after initialization

    <input id="dropdownlist" />
    <script>
    function dropdownlist_filtering(e) {
      //get filter descriptor
      var filter = e.filter;

      // handle the event
    }
    $("#dropdownlist").kendoDropDownList({
      dataSource: [ "Apples", "Oranges" ],
      filter: "startswith"
    });
    var dropdownlist = $("#dropdownlist").data("kendoDropDownList");
    dropdownlist.bind("filtering", dropdownlist_filtering);
    </script>

#### Example - prevent filtering event when filter value is empty

    <input id="dropdownlist" />
    <script>
    $("#dropdownlist").kendoDropDownList({
      dataSource: [ "Apples", "Oranges" ],
      filter: "startswith",
      filtering: function(e) {
          var filter = e.filter;

          if (!filter.value) {
            //prevent filtering if the filter does not value
            e.preventDefault();
          }
      }
    });
    </script>

### open

Fired when the popup of the widget is opened by the user.

The event handler function context (available via the `this` keyword) will be set to the widget instance.

#### Event Data

##### e.sender `kendo.ui.DropDownList`

The widget instance which fired the event.

#### Example - subscribe to the "open" event during initialization

    <input id="dropdownlist" />
    <script>
    $("#dropdownlist").kendoDropDownList({
      dataSource: [ "Apples", "Oranges" ],
      open: function(e) {
        // handle the event
      }
    });
    </script>

#### Example - subscribe to the "open" event after initialization

    <input id="dropdownlist" />
    <script>
    function dropdownlist_open(e) {
      // handle the event
    }
    $("#dropdownlist").kendoDropDownList({
      dataSource: [ "Apples", "Oranges" ]
    });
    var dropdownlist = $("#dropdownlist").data("kendoDropDownList");
    dropdownlist.bind("open", dropdownlist_open);
    </script>

### select

Fired when an item from the popup is selected by the user either with mouse/tap or with keyboard navigation.

> * The event is not fired when an item is selected programmatically.
> * Since version Q1 2015 (2015.1.318), the [option label has been moved outside the item list DOM collection]({% slug breakingchanges2015_kendoui %}#kendo-ui-2015-q1). As a result, `jQuery.index()` can no longer be used to reliably detect if the option label is the selected dropdown item. A more appropriate approach would be to check if the selected dataItem value is an empty string, and/or check if the selected dateItem's text is equal to the `optionLabel` string.
>   *  `e.sender.dataItem(e.item)[e.sender.options.dataValueField] == ""`
>   *  `e.sender.dataItem(e.item)[e.sender.options.dataTextField] == e.sender.options.optionLabel`

#### Event Data

##### e.dataItem `Object`

The data item instance of the selected item.

##### e.item `jQuery`

The jQuery object which represents the selected item.

##### e.preventDefault `Function`

If invoked prevents the select action. The widget will retain the previous selected item.

##### e.sender `kendo.ui.DropDownList`

The widget instance which fired the event.

#### Example - subscribe to the "select" event during initialization

    <input id="dropdownlist" />
    <script>
    $("#dropdownlist").kendoDropDownList({
      dataSource: [ "Apples", "Oranges" ],
      select: function(e) {
        var item = e.item;
        var text = item.text();
        // Use the selected item or its text
      }
    });
    </script>

#### Example - subscribe to the "select" event after initialization

    <input id="dropdownlist" />
    <script>
    function dropdownlist_select(e) {
      var item = e.item;
      var text = item.text();
      // Use the selected item or its text
    }
    $("#dropdownlist").kendoDropDownList({
      dataSource: [ "Apples", "Oranges" ]
    });
    var dropdownlist = $("#dropdownlist").data("kendoDropDownList");
    dropdownlist.bind("select", dropdownlist_select);
    </script>

#### Example - prevent the item selection

    <input id="dropdownlist" />
    <script>
    $("#dropdownlist").kendoDropDownList({
      dataSource: [ "Apples", "Oranges" ],
      select: function(e) {
        //call preventDefault() to prevent the selection
        e.preventDefault();
      }
    });
    </script>

### cascade

Fired when the value of the widget is changed via API or user interaction.

#### Event Data

##### e.sender `kendo.ui.DropDownList`

The widget instance which fired the event.

#### Example - subscribe to the "cascade" event during initialization

    <input id="dropdownlist" />
    <script>
    $("#dropdownlist").kendoDropDownList({
      dataSource: [ "Apples", "Oranges" ],
      cascade: function() {
        // Handle the event
      }
    });
    </script>

#### Example - subscribe to the "cascade" event after initialization

    <input id="dropdownlist" />
    <script>
    function dropdownlist_cascade(e) {
        // Handle the event
    }
    $("#dropdownlist").kendoDropDownList({
      dataSource: [ "Apples", "Oranges" ]
    });

    var dropdownlist = $("#dropdownlist").data("kendoDropDownList");
    dropdownlist.bind("cascade", dropdownlist_cascade);
    </script>
