---
title: Inline
page_title: Inline Editing
description: "Define commands and set the edit mode to configure the Telerik UI Grid HtmlHelper for {{ site.framework }} for inline editing."
slug: inlineediting_grid_aspnetcore
position: 2
---

# Inline Editing

You can define commands and set the edit mode to configure the Telerik UI Grid for {{ site.framework }} for inline editing.

For runnable examples, refer to the [demos on implementing the editing approaches in the Grid](https://demos.telerik.com/{{ site.platform }}/grid/editing).

> Defining a Schema.Model.Id is mandatory for the proper execution of the Update, Create and Destroy of the Grid.

## Setting the Inline Edit Mode

{% if site.mvc %}
1. Create a new ASP.NET MVC application. If you have installed the [Telerik UI for ASP.NET MVC Visual Studio Extensions]({% slug overview_visualstudio_aspnetmvc %}), create a Telerik UI for ASP.NET MVC application. Name the application `KendoGridAjaxEditing`. If you decided not to use the Telerik UI for ASP.NET MVC Visual Studio Extensions, follow the steps from the [introductory article]({% slug gettingstarted_aspnetmvc %}) to add Telerik UI for ASP.NET MVC to the application.
1. Add a new `Entity Framework Data Model`. Right-click the `~/Models` folder in the solution explorer and pick **Add new item**. Choose **Data** > **ADO.NET Entity Data Model** in the **Add New Item** dialog. Name the model `Northwind.edmx` and click **Next**. This starts the **Entity Data Model Wizard**.

    ![A new entity data model](../images/grid-entity-data-model.png)

1. Pick the **Generate from database** option and click **Next**. Configure a connection to the Northwind database. Click **Next**.

    ![Choosing the connection](../images/grid-entity-data-model.png)

1. Choose the **Products** table from the **Which database objects do you want to include in your model?**. Leave all other options as they are set by default. Click **Finish**.

    ![Choosing the Products table](../images/grid-database-objects.png)
{% else %}

1. Add a new class to the `~/Models` folder. The following example uses the `ProductViewModel` name.

            public class ProductViewModel

            {
                public int ProductID { get; set; }
                // The ProductName property is required.
                [Required]
                public string ProductName { get; set; }
                // Use the Integer editor template for the UnitsInStock property.
                [UIHint("Integer")]
                public short? UnitsInStock { get; set; }
            }
{% endif %}

1. Open `HomeController.cs` and add a new action method which will return the **Products** as JSON. The Grid will make Ajax requests to this action.

            public ActionResult Products_Read([DataSourceRequest]DataSourceRequest request)
            {
			    // ToDataSourceResult works with IEnumerable and IQueryable.
                using (var northwind = new NorthwindEntities())
                {
                    IQueryable<Product> products = northwind.Products;
                    DataSourceResult result = products.ToDataSourceResult(request);
                    return Json(result);
                }
            }

1. Add a new action method to `HomeController.cs`. It will be responsible for saving the new data items. Name the method `Products_Create`.

            public ActionResult Products_Create([DataSourceRequest]DataSourceRequest request, ProductViewModel product)
            {
                if (ModelState.IsValid)
                {
                    using (var northwind = new NorthwindEntities())
                    {
                        // Create a new Product entity and set its properties from the posted ProductViewModel.
                        var entity = new Product
                        {
                            ProductName = product.ProductName,
                            UnitsInStock = product.UnitsInStock
                        };
                        // Add the entity.
                        northwind.Products.Add(entity);
                        // Insert the entity in the database.
                        northwind.SaveChanges();
                        // Get the ProductID generated by the database.
                        product.ProductID = entity.ProductID;
                    }
                }
                // Return the inserted product. The Grid needs the generated ProductID. Also return any validation errors.
                return Json(new[] { product }.ToDataSourceResult(request, ModelState));
            }

1. Add a new action method to `HomeController.cs`. It will be responsible for saving the updated data items. Name the method `Products_Update`.

            public ActionResult Products_Update([DataSourceRequest]DataSourceRequest request, ProductViewModel product)
            {
                if (ModelState.IsValid)
                {
                    using (var northwind = new NorthwindEntities())
                    {
                        // Create a new Product entity and set its properties from the posted ProductViewModel.
                        var entity = new Product
                        {
                            ProductID = product.ProductID,
                            ProductName = product.ProductName,
                            UnitsInStock = product.UnitsInStock
                        };
                        // Attach the entity.
                        northwind.Products.Attach(entity);
                        // Change its state to Modified so Entity Framework can update the existing product instead of creating a new one.
                        northwind.Entry(entity).State = EntityState.Modified;
                        // Or use ObjectStateManager if using a previous version of Entity Framework.
                        // northwind.ObjectStateManager.ChangeObjectState(entity, EntityState.Modified);
                        // Update the entity in the database.
                        northwind.SaveChanges();
                    }
                }
                // Return the updated product. Also return any validation errors.
                return Json(new[] { product }.ToDataSourceResult(request, ModelState));
            }

1. Add a new action method to `HomeController.cs`. It will be responsible for saving the deleted data items. Name the method `Products_Destroy`.

            public ActionResult Products_Destroy([DataSourceRequest]DataSourceRequest request, ProductViewModel product)
            {
                if (ModelState.IsValid)
                {
                    using (var northwind = new NorthwindEntities())
                    {
                        // Create a new Product entity and set its properties from the posted ProductViewModel.
                        var entity = new Product
                        {
                            ProductID = product.ProductID,
                                      ProductName = product.ProductName,
                                      UnitsInStock = product.UnitsInStock
                        };
                        // Attach the entity.
                        northwind.Products.Attach(entity);
                        // Delete the entity.
                        northwind.Products.Remove(entity);
                        // Or use DeleteObject if using a previous versoin of Entity Framework.
                        // northwind.Products.DeleteObject(entity);
                        // Delete the entity in the database.
                        northwind.SaveChanges();
                    }
                }
                // Return the removed product. Also return any validation errors.
                return Json(new[] { product }.ToDataSourceResult(request, ModelState));
            }

1. In the view, configure the Grid to use the action methods that were created in the previous steps. The `Create`, `Update`, and `Destroy` action methods have to return a collection with the modified or deleted records which will enable the DataSource to apply the changes accordingly. The `Create` method has to return a collection of the created records with the assigned ID field.

          @(Html.Kendo().Grid<KendoGridAjaxEditing.Models.ProductViewModel>()
                .Name("grid")
                .Columns(columns =>
                {
                    columns.Bound(product => product.ProductID).Width(100);
                    columns.Bound(product => product.ProductName);
                    columns.Bound(product => product.UnitsInStock).Width(250);
                    columns.Command(commands =>
                    {
                        commands.Edit(); // The "edit" command will edit and update data items.
                        commands.Destroy(); // The "destroy" command removes data items.
                    }).Title("Commands").Width(200);
                })
                .ToolBar(toolbar => toolbar.Create()) // The "create" command adds new data items.
                .Editable(editable => editable.Mode(GridEditMode.InLine)) // Use the inline edit mode.
                .DataSource(dataSource =>
                    dataSource.Ajax()
                      .Model(model =>
                      {
                          model.Id(product => product.ProductID); // Specify the property which is the unique identifier of the model.
                          model.Field(product => product.ProductID).Editable(false); // Make the ProductID property not editable.
                      })
                      .Create(create => create.Action("Products_Create", "Home")) // Action invoked when the user saves a new data item.
                      .Read(read => read.Action("Products_Read", "Home"))  // Action invoked when the Grid needs data.
                      .Update(update => update.Action("Products_Update", "Home"))  // Action invoked when the user saves an updated data item.
                      .Destroy(destroy => destroy.Action("Products_Destroy", "Home")) // Action invoked when the user removes a data item.
                )
                .Pageable()
          )		  

## Handling ModelState Errors

When editing is performed, server validation is often needed. This section demonstrates how to use the `AddModelError` method with the Telerik UI Grid for {{ site.framework }}.

1. Perform all steps from the previous section.
1. Add some validation code to the `Products_Update` method. For example, check the length of the `ProductName` property.

            public ActionResult Products_Update([DataSourceRequest]DataSourceRequest request, ProductViewModel product)
            {
                if (product.ProductName.Length < 3)
                {
                    ModelState.AddModelError("ProductName", "ProductName should be at least three characters long.");
                }
                if (ModelState.IsValid)
                {
                    using (var northwind = new NorthwindEntities())
                    {
                        // Create a new Product entity and set its properties from the posted ProductViewModel.
                        var entity = new Product
                        {
                            ProductID = product.ProductID,
                                      ProductName = product.ProductName,
                                      UnitsInStock = product.UnitsInStock
                        };

                        // Attach the entity.
                        northwind.Products.Attach(entity);
                        // Change its state to Modified so Entity Framework can update the existing product instead of creating a new one.
                        northwind.Entry(entity).State = EntityState.Modified;

                        // Or use ObjectStateManager if using a previous version of Entity Framework.
                        // northwind.ObjectStateManager.ChangeObjectState(entity, EntityState.Modified);

                        // Update the entity in the database.
                        northwind.SaveChanges();
                    }
                }
                // Return the updated product. Also return any validation errors.
                return Json(new[] { product }.ToDataSourceResult(request, ModelState));
            }

1. Subscribe to the [`DataSource.Error()`](https://docs.telerik.com/{{ site.platform }}/api/Kendo.Mvc.UI.Fluent/DataSourceEventBuilder#errorsystemstring) event handler. It is fired when model state errors or other unexpected problem occur when making the Ajax request. In the event handler, display the errors and call the [`cancelChanges`](https://docs.telerik.com/kendo-ui/api/javascript/ui/grid#methods-cancelChanges) method of the Grid.

          @(Html.Kendo().Grid<KendoGridAjaxEditing.Models.ProductViewModel>()
                .Name("grid")
                .Columns(columns =>
                {
                    columns.Bound(product => product.ProductID).Width(100);
                    columns.Bound(product => product.ProductName);
                    columns.Bound(product => product.UnitsInStock).Width(250);
                    columns.Command(commands =>
                    {
                        commands.Edit();
                        commands.Destroy();
                    }).Title("Commands").Width(200);
                })
                .ToolBar(toolbar => toolbar.Create())
                .Editable(editable => editable.Mode(GridEditMode.InLine))
                .DataSource(dataSource =>
                    dataSource.Ajax()
                      .Events(events => events.Error("grid_error")) // Handle the "error" event.
                      .Model(model =>
                      {
                          model.Id(product => product.ProductID);
                          model.Field(product => product.ProductID).Editable(false);
                      })
                      .Create(create => create.Action("Products_Create", "Home"))
                      .Read(read => read.Action("Products_Read", "Home"))
                      .Update(update => update.Action("Products_Update", "Home"))
                      .Destroy(destroy => destroy.Action("Products_Destroy", "Home"))
                )
                .Pageable()
          )
          <script>
          function grid_error(e) {
              if (e.errors) {
                  var message = "There are some errors:\n";
                  // Create a message containing all errors.
                  $.each(e.errors, function (key, value) {
                      if ('errors' in value) {
                          $.each(value.errors, function () {
                              message += this + "\n";
                          });
                      }
                  });
                  // Display the message.
                  alert(message);
                  // Cancel the changes.
                  var grid = $("#grid").data("kendoGrid");
                  grid.cancelChanges();
              }
          }
          </script>
          
## Runnable Samples

To check and download runnable samples demonstrating Inline editing, you can go to these resources:

* [Inline Editing Demo](https://demos.telerik.com/{{ site.platform }}/grid/editing-inline)
* [Inline Editing Sample](https://github.com/telerik/ui-for-aspnet-mvc-examples/tree/master/grid/grid-inline-editing-with-nullable-boolean)

## See Also

* [Editing Approaches by the Grid HtmlHelper for {{ site.framework }} (Demos)](https://demos.telerik.com/{{ site.platform }}/grid/editing)
* [Server-Side API](/api/grid)
