---
title: Show base64 Image in Grid
description: An example on how to show base64-encoded images in the Kendo UI Grid.
type: how-to
page_title: Render base64-Encoded Images | Kendo UI Grid for jQuery
slug: grid-show-base64-image
tags: grid, image, base64, encoded, cell, template, show, display
ticketid: 1149817
res_type: kb
---

## Environment

<table>
 <tr>
  <td>Product</td>
  <td>Progress Kendo UI Grid</td>
 </tr>
 </tr>
</table>


## Description

How can I display a base64-encoded image in the Grid?

## Solution

1. Specify a field in the server-side model that will hold the base64-encoded string.
1. When the Grid reads the data, display the image by using [`column.template`](https://docs.telerik.com/kendo-ui/api/javascript/ui/grid/configuration/columns.template).


```dojo
<div id="grid"></div>
<script>

    $(function () {
        $("#grid").kendoGrid({
            columns: [{
                field: "name",
                template: '<img src="' + imageData + '">'
            }],
            dataSource: [{ name: "Jane Doe" }]
        });
    });

    var imageData = "data:image/png;base64,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";
</script>
```
