'use strict';

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.splat = splat;
exports.noop = noop;
exports.uid = uid;
exports.merge = merge;
exports.getArrayType = getArrayType;
exports.getGLTypeFromArrayType = getGLTypeFromArrayType;
exports.getGLTypeFromArray = getGLTypeFromArray;
exports.getArrayTypeFromGLType = getArrayTypeFromGLType;
/**
 * Wraps the argument in an array if it is not one.
 * @param {object} a - The object to wrap.
 * @return {Array} array
 **/
function splat(a) {
  return a ? Array.isArray(a) ? a : [a] : [];
}

/**
* Provides a standard noop function.
**/
function noop() {}

var uidCounters = {};

/**
 * Returns a UID.
 * @param {String} id= - Identifier base name
 * @return {number} uid
 **/
function uid() {
  var id = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : 'id';

  uidCounters[id] = uidCounters[id] || 1;
  var count = uidCounters[id]++;
  return id + '-' + count;
}

/**
 * Merge multiple objects into one.
 * @param {...object} objects - The objects to merge.
 * @return {object} object
 **/
function merge(objects) {
  var mix = {};
  for (var i = 0, l = arguments.length; i < l; i++) {
    var object = arguments[i];
    if (!object || object.constructor.name !== 'Object') {
      /* eslint-disable no-continue */
      continue;
    }
    for (var key in object) {
      var op = object[key];
      var mp = mix[key];
      if (mp && op.constructor.name === 'Object' && mp.constructor.name === 'Object') {
        mix[key] = merge(mp, op);
      } else {
        mix[key] = detach(op);
      }
    }
  }
  return mix;
}

/**
 * Internal function for duplicating an object.
 * @param {object} elem - The object to recursively duplicate.
 * @return {object} object
 **/
function detach(elem) {
  var t = elem.constructor.name;
  var ans = void 0;
  if (t === 'Object') {
    ans = {};
    for (var p in elem) {
      ans[p] = detach(elem[p]);
    }
  } else if (t === 'Array') {
    ans = [];
    for (var i = 0, l = elem.length; i < l; i++) {
      ans[i] = detach(elem[i]);
    }
  } else {
    ans = elem;
  }

  return ans;
}

// TYPED ARRAYS

function getArrayType(array) {
  // Sorted in some order of likelihood to reduce amount of comparisons
  if (array instanceof Float32Array) {
    return Float32Array;
  } else if (array instanceof Uint16Array) {
    return Uint16Array;
  } else if (array instanceof Uint32Array) {
    return Uint32Array;
  } else if (array instanceof Uint8Array) {
    return Uint8Array;
  } else if (array instanceof Uint8ClampedArray) {
    return Uint8ClampedArray;
  } else if (array instanceof Int8Array) {
    return Int8Array;
  } else if (array instanceof Int16Array) {
    return Int16Array;
  } else if (array instanceof Int32Array) {
    return Int32Array;
  }
  throw new Error('Failed to deduce type from array');
}

function getGLTypeFromArrayType(arrayType) {
  // Sorted in some order of likelihood to reduce amount of comparisons
  switch (arrayType) {
    case Float32Array:
      return 'FLOAT';
    case Uint16Array:
      return 'UNSIGNED_SHORT';
    case Uint32Array:
      return 'UNSIGNED_INT';
    case Uint8Array:
      return 'UNSIGNED_BYTE';
    case Uint8ClampedArray:
      return 'UNSIGNED_BYTE';
    case Int8Array:
      return 'BYTE';
    case Int16Array:
      return 'SHORT';
    case Int32Array:
      return 'INT';
    default:
      throw new Error('Failed to deduce type from array');
  }
}

function getGLTypeFromArray(array) {
  return getGLTypeFromArrayType(getArrayType(array));
}

/* eslint-disable complexity */
function getArrayTypeFromGLType(glTypeString) {
  var clamped = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : false;

  // Sorted in some order of likelihood to reduce amount of comparisons
  switch (glTypeString) {
    case 'FLOAT':
      return Float32Array;
    case 'UNSIGNED_SHORT':
    case 'UNSIGNED_SHORT_5_6_5':
    case 'UNSIGNED_SHORT_4_4_4_4':
    case 'UNSIGNED_SHORT_5_5_5_1':
      return Uint16Array;
    case 'UNSIGNED_INT':
      return Uint32Array;
    case 'UNSIGNED_BYTE':
      return clamped ? Uint8ClampedArray : Uint8Array;
    case 'BYTE':
      return Int8Array;
    case 'SHORT':
      return Int16Array;
    case 'INT':
      return Int32Array;
    default:
      throw new Error('Failed to deduce type from array');
  }
}
/* eslint-enable complexity */
//# sourceMappingURL=data:application/json;base64,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