'use strict';

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getMaxAttributes = getMaxAttributes;
exports.hasDivisor = hasDivisor;
exports.isEnabled = isEnabled;
exports.getBuffer = getBuffer;
exports.getGeneric = getGeneric;
exports.getSize = getSize;
exports.getType = getType;
exports.isNormalized = isNormalized;
exports.isInteger = isInteger;
exports.getStride = getStride;
exports.getOffset = getOffset;
exports.enable = enable;
exports.disable = disable;
exports.setDivisor = setDivisor;
exports.getDivisor = getDivisor;
exports.setBuffer = setBuffer;
exports.setGeneric = setGeneric;
exports.setGenericValues = setGenericValues;

var _webglTypes = require('./webgl-types');

var _webglChecks = require('./webgl-checks');

var _buffer = require('./buffer');

var _buffer2 = _interopRequireDefault(_buffer);

var _context = require('./context');

var _utils = require('../utils');

var _assert = require('assert');

var _assert2 = _interopRequireDefault(_assert);

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

/**
 * Methods for manipulating the vertex attributes array, which is where
 * vertex data is staged for shader execution.
 *
 * Vertex attributes are stored in "arrays" with indices from 0 and up.
 * During shader execution, these indices (or 'locations') are matched to
 * the indices assigned to shader attributes during WebGLProgram linking.
 *
 * Note: The global state contains an implicit vertex attribute array which
 * the methods in this class manipulate by default. It is also possible to
 * create and bind a VertexArrayObject to manage multiple arrays.
 *
 * Each vertex attribute has these properties:
 * - Can be enabled or disabled (Only enable attrs actually used by a program)
 * - Has an instance `divisor` (usually 1 or 0 to enable/disable instancing)
 * - Have a size (1-4 values per vertex)
 * - Has a value or values that is accessible in shaders
 *
 * Attribute values are either
 * - Generic: a constant value for all vertices/instances, or
 * - Bound to a WebGLBuffer with unique values for each vertex/instance
 *
 * When binding to a WebGLBuffer it is necessary to specify the layout of
 * data in the buffer:
 * - size (1-4 values per vertex)
 * - data type (e.g. gl.FLOAT)
 * - stride, offset, and integer normalization policy can also be specified
 *
 * Note: All methods in this class take a `location` index to specify which
 * vertex attribute in the array they are operating on.
 *
 * Note: Attribute 0 can sometimes be treated specially by the driver,
 * to be safe we avoid disabling it.
 *
 * Note: WebGL2
 * - Improves support for integer attributes, both generic and buffered.
 * - Setting instance "divisors" no longer require using a WebGL extension.
 *
 */

/* eslint-disable dot-notation*/
var ERR_WEBGL2 = 'WebGL2 required';

function glGetLumaInfo(gl) {
  if (!gl.luma) {
    gl.luma = { extensions: {} };
  }
  if (gl.luma.extensions['ANGLE_instanced_arrays'] === undefined) {
    gl.luma.extensions['ANGLE_instanced_arrays'] = gl.getExtension('ANGLE_instanced_arrays');
  }
  return gl.luma;
}

// ACCESSORS

/**
 * The max number of attributes in the vertex attribute array is an
 * implementation defined limit, but never smaller than 8
 * @param {WebGLRenderingContext} gl - webgl context
 * @returns {GLuint} - (max) number of attributes in the vertex attribute array
 */
function getMaxAttributes(gl) {
  (0, _webglChecks.assertWebGLRenderingContext)(gl);
  return gl.getParameter(gl.MAX_VERTEX_ATTRIBS);
}

/**
 * Is instance divisor availble (checks for WebGL2 or ANGLE extension)
 * @param {WebGLRenderingContext} gl - webgl context
 * @returns {Boolean} - is divisor available?
 */
function hasDivisor(gl) {
  (0, _webglChecks.assertWebGLRenderingContext)(gl);
  return Boolean(gl instanceof _webglTypes.WebGL2RenderingContext || gl.getExtension(gl, 'ANGLE_instanced_arrays'));
}

/**
 * Returns true if the vertex attribute is enabled at this index.
 * @param {WebGLRenderingContext} gl - webgl context
 * @param {GLuint} location - ordinal number of the attribute
 * @returns {Boolean} - enabled status
 */
function isEnabled(gl, location) {
  return Boolean(get(gl, location, gl.VERTEX_ATTRIB_ARRAY_ENABLED));
}

/**
 * Returns the currently bound buffer
 * @param {WebGLRenderingContext} gl - webgl context
 * @param {GLuint} location - ordinal number of the attribute
 * @returns {WebGLBuffer} Returns the currently bound buffer
 */
function getBuffer(gl, location) {
  return get(gl, location, gl.VERTEX_ATTRIB_ARRAY_BUFFER_BINDING);
}

/**
 * Get values for generic vertex attributes
 * @param {WebGLRenderingContext} gl - webgl context
 * @param {GLuint} location - ordinal number of the attribute
 * @returns {Float32Array} (with 4 elements) representing the current value
 * of the vertex attribute at the given index.
 */
function getGeneric(gl, location) {
  return get(gl, gl.CURRENT_VERTEX_ATTRIB);
}

/**
 * @param {WebGLRenderingContext} gl - webgl context
 * @param {GLuint} location - ordinal number of the attribute
 */
// @returns {GLint} the size of an element of the vertex array.
function getSize(gl, location) {
  return get(location, gl.VERTEX_ATTRIB_ARRAY_SIZE);
}

/**
 * @param {WebGLRenderingContext} gl - webgl context
 * @param {GLuint} location - ordinal number of the attribute
 */
// @returns {GLenum} representing the array type.
function getType(gl, location) {
  return get(location, gl.VERTEX_ATTRIB_ARRAY_TYPE);
}

/**
 * @param {WebGLRenderingContext} gl - webgl context
 * @param {GLuint} location - ordinal number of the attribute
 */
// @returns {GLboolean} true if fixed-point data types are normalized
// for the vertex attribute array at the given index.
function isNormalized(gl, location) {
  return get(location, gl.VERTEX_ATTRIB_ARRAY_NORMALIZED);
}

/**
 * check if an integer data type in the vertex attribute at index
 * @param {WebGLRenderingContext} gl - webgl context
 * @param {GLuint} location - index of the vertex attribute.
 * @returns {GLboolean} - true if an integer data type is in the
 * vertex attribute array at the given index.
 */
function isInteger(gl, location) {
  (0, _assert2.default)(gl instanceof _webglTypes.WebGL2RenderingContext, ERR_WEBGL2);
  return get(location, gl.VERTEX_ATTRIB_ARRAY_INTEGER);
}

/**
 * @param {WebGLRenderingContext} gl - webgl context
 * @param {GLuint} location - ordinal number of the attribute
 * @returns {GLint} number of bytes between successive elements in the array.
 * 0 means that the elements are sequential.
 */
function getStride(gl, location) {
  return get(location, gl.VERTEX_ATTRIB_ARRAY_STRIDE);
}

/**
 * @param {WebGLRenderingContext} gl - webgl context
 * @param {GLuint} location - ordinal number of the attribute
 * @param {GLuint} pname - enum specifying which offset to return
 * @returns {GLuint} the address of a specified vertex attribute.
 */
function getOffset(gl, location) {
  var pname = arguments.length > 2 && arguments[2] !== undefined ? arguments[2] : gl.VERTEX_ATTRIB_ARRAY_POINTER;

  return gl.getVertexAttribOffset(location, pname);
}

/**
 * @private
 * Generic getter for information about a vertex attribute at a given position
 * @param {WebGLRenderingContext} gl - webgl context
 * @param {GLuint} location - index of the vertex attribute.
 * @param {GLenum} pname - specifies the information to query.
 * @returns {*} - requested vertex attribute information (specified by pname)
 */
function get(gl, location, pname) {
  (0, _webglChecks.assertWebGLRenderingContext)(gl);
  return gl.getVertexAttrib(location, pname);
}

// MODIFIERS

/**
 * Enable the attribute
 * Note: By default all attributes are disabled. Only attributes
 * used by a program's shaders should be enabled.
 *
 * @param {WebGLRenderingContext} gl - webgl context
 * @param {GLuint} location - ordinal number of the attribute
 */
function enable(gl, location) {
  gl.enableVertexAttribArray(location);
}

/**
 * Disable the attribute
 * Note: Only attributes used by a program's shaders should be enabled.
 *
 * @param {WebGLRenderingContext} gl - webgl context
 * @param {GLuint} location - ordinal number of the attribute
 */
function disable(gl, location) {
  // Don't disable location 0
  if (location > 0) {
    gl.disableVertexAttribArray(location);
  }
}

/**
 * Set the frequency divisor used for instanced rendering.
 * Note: Usually simply set to 1 or 0 to enable/disable instanced rendering
 * for a specific attribute.
 *
 * @param {WebGLRenderingContext} gl - webgl context
 * @param {GLuint} location - ordinal number of the attribute
 * @param {GLuint} divisor - instances that pass between updates of attribute
 */
function setDivisor(gl, location, divisor) {
  if (gl instanceof _webglTypes.WebGL2RenderingContext) {
    gl.vertexAttribDivisor(location, divisor);
    return;
  }
  var ext = glGetLumaInfo(gl).extensions['ANGLE_instanced_arrays'];
  if (ext) {
    ext.vertexAttribDivisorANGLE(location, divisor);
    return;
  }
  // Accept divisor 0 even if instancing is not supported (0 = no instancing)
  if (divisor !== 0) {
    throw new Error('WebGL instanced rendering not supported');
  }
}

/**
 * Returns the frequency divisor used for instanced rendering.
 * @param {WebGLRenderingContext} gl - webgl context
 * @param {GLuint} location - ordinal number of the attribute
 * @returns {GLuint} divisor
 */
function getDivisor(gl, location) {
  (0, _assert2.default)(location > 0);
  if (gl instanceof _webglTypes.WebGL2RenderingContext) {
    var divisor = get(location, gl.VERTEX_ATTRIB_ARRAY_DIVISOR);
    return divisor;
  }
  var ext = glGetLumaInfo(gl).extensions['ANGLE_instanced_arrays'];
  if (ext) {
    var _divisor = get(location, ext.VERTEX_ATTRIB_ARRAY_DIVISOR_ANGLE);
    return _divisor;
  }
  // if instancing is not available, return 0 meaning divisor has not been set
  return 0;
}

/**
 * Set a location in vertex attributes array to a buffer, specifying
 * its data layout and integer to float conversion and normalization flags
 *
 * @param {WebGLRenderingContext} gl - webgl context
 * @param {GLuint} location - ordinal number of the attribute
 * @param {WebGLBuffer|Buffer} buffer - WebGL buffer to set as value
 * @param {GLuint} target=gl.ARRAY_BUFFER - which target to bind to
 * @param {Object} layout= Optional data layout, defaults to buffer's layout
 * @param {GLuint} layout.size - number of values per element (1-4)
 * @param {GLuint} layout.type - type of values (e.g. gl.FLOAT)
 * @param {GLbool} layout.normalized=false - normalize integers to [-1,1], [0,1]
 * @param {GLuint} layout.integer=false - WebGL2 only, disable int-to-float conv
 * @param {GLuint} layout.stride=0 - supports strided arrays
 * @param {GLuint} layout.offset=0 - supports strided arrays
 */
function setBuffer() {
  var _ref = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : {};

  var gl = _ref.gl;
  var location = _ref.location;
  var buffer = _ref.buffer;
  var target = _ref.target;
  var layout = _ref.layout;

  (0, _webglChecks.assertWebGLRenderingContext)(gl);
  buffer = _buffer2.default.makeFrom(gl, buffer);

  // Copy main data characteristics from buffer
  target = (0, _context.glGet)(gl, target !== undefined ? target : buffer.target);
  layout = layout !== undefined ? layout : buffer.layout;
  (0, _assert2.default)(target, 'setBuffer needs target');
  (0, _assert2.default)(layout, 'setBuffer called on uninitialized buffer');

  // a non-zero named buffer object must be bound to the GL_ARRAY_BUFFER target
  buffer.bind({ target: gl.ARRAY_BUFFER });

  // Attach bound ARRAY_BUFFER with specified buffer format to location
  if (!layout.integer) {
    gl.vertexAttribPointer(location, layout.size, (0, _context.glGet)(gl, layout.type), layout.normalized, layout.stride, layout.offset);
  } else {
    // specifies *integer* data formats and locations of vertex attributes
    // For glVertexAttribIPointer, Values are always left as integer values.
    // Only accepts the integer types gl.BYTE, gl.UNSIGNED_BYTE,
    // gl.SHORT, gl.UNSIGNED_SHORT, gl.INT, gl.UNSIGNED_INT
    (0, _assert2.default)(gl instanceof _webglTypes.WebGL2RenderingContext, ERR_WEBGL2);
    gl.vertexAttribIPointer(location, layout.size, (0, _context.glGet)(gl, layout.type), layout.stride, layout.offset);
  }

  buffer.unbind({ target: gl.ARRAY_BUFFER });
}

/*
 * Specify values for generic vertex attributes
 * Generic vertex attributes are constant for all vertices
 * Up to 4 values depending on attribute size
 *
 * @param {WebGLRenderingContext} gl - webgl context
 * @param {GLuint} location - ordinal number of the attribute
 * @param {GLuint} divisor - instances that pass between updates of attribute
 */
function setGeneric(_ref2) {
  var gl = _ref2.gl;
  var location = _ref2.location;
  var array = _ref2.array;

  _utils.log.warn(0, 'VertexAttributes.setGeneric is not well tested');
  // throw new Error('vertex attribute size must be between 1 and 4');

  if (array instanceof Float32Array) {
    gl.vertexAttrib4fv(location, array);
  } else if (array instanceof Int32Array) {
    (0, _assert2.default)(gl instanceof _webglTypes.WebGL2RenderingContext, 'WebGL2 required');
    gl.vertexAttribI4iv(location, array);
  } else if (array instanceof Uint32Array) {
    (0, _assert2.default)(gl instanceof _webglTypes.WebGL2RenderingContext, 'WebGL2 required');
    gl.vertexAttribI4uiv(location, array);
  }
}

/*
 * Specify values for generic vertex attributes
 * Generic vertex attributes are constant for all vertices
 * Up to 4 values depending on attribute size
 *
 * @param {GLuint} location - ordinal number of the attribute
 * @param {GLuint} divisor - instances that pass between updates of attribute
 */
/* eslint-disable max-params */
function setGenericValues(gl, location, v0, v1, v2, v3) {
  _utils.log.warn(0, 'VertexAttributes.setGenericValues is not well tested');
  switch (arguments.length - 1) {
    case 1:
      gl.vertexAttrib1f(location, v0);break;
    case 2:
      gl.vertexAttrib2f(location, v0, v1);break;
    case 3:
      gl.vertexAttrib3f(location, v0, v1, v2);break;
    case 4:
      gl.vertexAttrib4f(location, v0, v1, v2, v3);break;
    default:
      throw new Error('vertex attribute size must be between 1 and 4');
  }

  // assert(gl instanceof WebGL2RenderingContext, 'WebGL2 required');
  // Looks like these will check how many arguments were supplied?
  // gl.vertexAttribI4i(location, v0, v1, v2, v3);
  // gl.vertexAttribI4ui(location, v0, v1, v2, v3);
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbIi4uLy4uL3NyYy93ZWJnbC92ZXJ0ZXgtYXR0cmlidXRlcy5qcyJdLCJuYW1lcyI6WyJnZXRNYXhBdHRyaWJ1dGVzIiwiaGFzRGl2aXNvciIsImlzRW5hYmxlZCIsImdldEJ1ZmZlciIsImdldEdlbmVyaWMiLCJnZXRTaXplIiwiZ2V0VHlwZSIsImlzTm9ybWFsaXplZCIsImlzSW50ZWdlciIsImdldFN0cmlkZSIsImdldE9mZnNldCIsImVuYWJsZSIsImRpc2FibGUiLCJzZXREaXZpc29yIiwiZ2V0RGl2aXNvciIsInNldEJ1ZmZlciIsInNldEdlbmVyaWMiLCJzZXRHZW5lcmljVmFsdWVzIiwiRVJSX1dFQkdMMiIsImdsR2V0THVtYUluZm8iLCJnbCIsImx1bWEiLCJleHRlbnNpb25zIiwidW5kZWZpbmVkIiwiZ2V0RXh0ZW5zaW9uIiwiZ2V0UGFyYW1ldGVyIiwiTUFYX1ZFUlRFWF9BVFRSSUJTIiwiQm9vbGVhbiIsImxvY2F0aW9uIiwiZ2V0IiwiVkVSVEVYX0FUVFJJQl9BUlJBWV9FTkFCTEVEIiwiVkVSVEVYX0FUVFJJQl9BUlJBWV9CVUZGRVJfQklORElORyIsIkNVUlJFTlRfVkVSVEVYX0FUVFJJQiIsIlZFUlRFWF9BVFRSSUJfQVJSQVlfU0laRSIsIlZFUlRFWF9BVFRSSUJfQVJSQVlfVFlQRSIsIlZFUlRFWF9BVFRSSUJfQVJSQVlfTk9STUFMSVpFRCIsIlZFUlRFWF9BVFRSSUJfQVJSQVlfSU5URUdFUiIsIlZFUlRFWF9BVFRSSUJfQVJSQVlfU1RSSURFIiwicG5hbWUiLCJWRVJURVhfQVRUUklCX0FSUkFZX1BPSU5URVIiLCJnZXRWZXJ0ZXhBdHRyaWJPZmZzZXQiLCJnZXRWZXJ0ZXhBdHRyaWIiLCJlbmFibGVWZXJ0ZXhBdHRyaWJBcnJheSIsImRpc2FibGVWZXJ0ZXhBdHRyaWJBcnJheSIsImRpdmlzb3IiLCJ2ZXJ0ZXhBdHRyaWJEaXZpc29yIiwiZXh0IiwidmVydGV4QXR0cmliRGl2aXNvckFOR0xFIiwiRXJyb3IiLCJWRVJURVhfQVRUUklCX0FSUkFZX0RJVklTT1IiLCJWRVJURVhfQVRUUklCX0FSUkFZX0RJVklTT1JfQU5HTEUiLCJidWZmZXIiLCJ0YXJnZXQiLCJsYXlvdXQiLCJtYWtlRnJvbSIsImJpbmQiLCJBUlJBWV9CVUZGRVIiLCJpbnRlZ2VyIiwidmVydGV4QXR0cmliUG9pbnRlciIsInNpemUiLCJ0eXBlIiwibm9ybWFsaXplZCIsInN0cmlkZSIsIm9mZnNldCIsInZlcnRleEF0dHJpYklQb2ludGVyIiwidW5iaW5kIiwiYXJyYXkiLCJ3YXJuIiwiRmxvYXQzMkFycmF5IiwidmVydGV4QXR0cmliNGZ2IiwiSW50MzJBcnJheSIsInZlcnRleEF0dHJpYkk0aXYiLCJVaW50MzJBcnJheSIsInZlcnRleEF0dHJpYkk0dWl2IiwidjAiLCJ2MSIsInYyIiwidjMiLCJhcmd1bWVudHMiLCJsZW5ndGgiLCJ2ZXJ0ZXhBdHRyaWIxZiIsInZlcnRleEF0dHJpYjJmIiwidmVydGV4QXR0cmliM2YiLCJ2ZXJ0ZXhBdHRyaWI0ZiJdLCJtYXBwaW5ncyI6Ijs7Ozs7UUFxRWdCQSxnQixHQUFBQSxnQjtRQVVBQyxVLEdBQUFBLFU7UUFjQUMsUyxHQUFBQSxTO1FBVUFDLFMsR0FBQUEsUztRQVdBQyxVLEdBQUFBLFU7UUFTQUMsTyxHQUFBQSxPO1FBU0FDLE8sR0FBQUEsTztRQVVBQyxZLEdBQUFBLFk7UUFXQUMsUyxHQUFBQSxTO1FBV0FDLFMsR0FBQUEsUztRQVVBQyxTLEdBQUFBLFM7UUE2QkFDLE0sR0FBQUEsTTtRQVdBQyxPLEdBQUFBLE87UUFnQkFDLFUsR0FBQUEsVTtRQXNCQUMsVSxHQUFBQSxVO1FBK0JBQyxTLEdBQUFBLFM7UUF3REFDLFUsR0FBQUEsVTtRQXdCQUMsZ0IsR0FBQUEsZ0I7O0FBMVdoQjs7QUFDQTs7QUFDQTs7OztBQUNBOztBQUNBOztBQUNBOzs7Ozs7QUFFQTs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7OztBQVJBO0FBZ0RBLElBQU1DLGFBQWEsaUJBQW5COztBQUVBLFNBQVNDLGFBQVQsQ0FBdUJDLEVBQXZCLEVBQTJCO0FBQ3pCLE1BQUksQ0FBQ0EsR0FBR0MsSUFBUixFQUFjO0FBQ1pELE9BQUdDLElBQUgsR0FBVSxFQUFDQyxZQUFZLEVBQWIsRUFBVjtBQUNEO0FBQ0QsTUFBSUYsR0FBR0MsSUFBSCxDQUFRQyxVQUFSLENBQW1CLHdCQUFuQixNQUFpREMsU0FBckQsRUFBZ0U7QUFDOURILE9BQUdDLElBQUgsQ0FBUUMsVUFBUixDQUFtQix3QkFBbkIsSUFDRUYsR0FBR0ksWUFBSCxDQUFnQix3QkFBaEIsQ0FERjtBQUVEO0FBQ0QsU0FBT0osR0FBR0MsSUFBVjtBQUNEOztBQUVEOztBQUVBOzs7Ozs7QUFNTyxTQUFTckIsZ0JBQVQsQ0FBMEJvQixFQUExQixFQUE4QjtBQUNuQyxnREFBNEJBLEVBQTVCO0FBQ0EsU0FBT0EsR0FBR0ssWUFBSCxDQUFnQkwsR0FBR00sa0JBQW5CLENBQVA7QUFDRDs7QUFFRDs7Ozs7QUFLTyxTQUFTekIsVUFBVCxDQUFvQm1CLEVBQXBCLEVBQXdCO0FBQzdCLGdEQUE0QkEsRUFBNUI7QUFDQSxTQUFPTyxRQUNMUCxvREFDQUEsR0FBR0ksWUFBSCxDQUFnQkosRUFBaEIsRUFBb0Isd0JBQXBCLENBRkssQ0FBUDtBQUlEOztBQUVEOzs7Ozs7QUFNTyxTQUFTbEIsU0FBVCxDQUFtQmtCLEVBQW5CLEVBQXVCUSxRQUF2QixFQUFpQztBQUN0QyxTQUFPRCxRQUFRRSxJQUFJVCxFQUFKLEVBQVFRLFFBQVIsRUFBa0JSLEdBQUdVLDJCQUFyQixDQUFSLENBQVA7QUFDRDs7QUFFRDs7Ozs7O0FBTU8sU0FBUzNCLFNBQVQsQ0FBbUJpQixFQUFuQixFQUF1QlEsUUFBdkIsRUFBaUM7QUFDdEMsU0FBT0MsSUFBSVQsRUFBSixFQUFRUSxRQUFSLEVBQWtCUixHQUFHVyxrQ0FBckIsQ0FBUDtBQUNEOztBQUVEOzs7Ozs7O0FBT08sU0FBUzNCLFVBQVQsQ0FBb0JnQixFQUFwQixFQUF3QlEsUUFBeEIsRUFBa0M7QUFDdkMsU0FBT0MsSUFBSVQsRUFBSixFQUFRQSxHQUFHWSxxQkFBWCxDQUFQO0FBQ0Q7O0FBRUQ7Ozs7QUFJQTtBQUNPLFNBQVMzQixPQUFULENBQWlCZSxFQUFqQixFQUFxQlEsUUFBckIsRUFBK0I7QUFDcEMsU0FBT0MsSUFBSUQsUUFBSixFQUFjUixHQUFHYSx3QkFBakIsQ0FBUDtBQUNEOztBQUVEOzs7O0FBSUE7QUFDTyxTQUFTM0IsT0FBVCxDQUFpQmMsRUFBakIsRUFBcUJRLFFBQXJCLEVBQStCO0FBQ3BDLFNBQU9DLElBQUlELFFBQUosRUFBY1IsR0FBR2Msd0JBQWpCLENBQVA7QUFDRDs7QUFFRDs7OztBQUlBO0FBQ0E7QUFDTyxTQUFTM0IsWUFBVCxDQUFzQmEsRUFBdEIsRUFBMEJRLFFBQTFCLEVBQW9DO0FBQ3pDLFNBQU9DLElBQUlELFFBQUosRUFBY1IsR0FBR2UsOEJBQWpCLENBQVA7QUFDRDs7QUFFRDs7Ozs7OztBQU9PLFNBQVMzQixTQUFULENBQW1CWSxFQUFuQixFQUF1QlEsUUFBdkIsRUFBaUM7QUFDdEMsd0JBQU9SLGdEQUFQLEVBQTZDRixVQUE3QztBQUNBLFNBQU9XLElBQUlELFFBQUosRUFBY1IsR0FBR2dCLDJCQUFqQixDQUFQO0FBQ0Q7O0FBRUQ7Ozs7OztBQU1PLFNBQVMzQixTQUFULENBQW1CVyxFQUFuQixFQUF1QlEsUUFBdkIsRUFBaUM7QUFDdEMsU0FBT0MsSUFBSUQsUUFBSixFQUFjUixHQUFHaUIsMEJBQWpCLENBQVA7QUFDRDs7QUFFRDs7Ozs7O0FBTU8sU0FBUzNCLFNBQVQsQ0FDTFUsRUFESyxFQUNEUSxRQURDLEVBRUw7QUFBQSxNQURjVSxLQUNkLHVFQURzQmxCLEdBQUdtQiwyQkFDekI7O0FBQ0EsU0FBT25CLEdBQUdvQixxQkFBSCxDQUF5QlosUUFBekIsRUFBbUNVLEtBQW5DLENBQVA7QUFDRDs7QUFFRDs7Ozs7Ozs7QUFRQSxTQUFTVCxHQUFULENBQWFULEVBQWIsRUFBaUJRLFFBQWpCLEVBQTJCVSxLQUEzQixFQUFrQztBQUNoQyxnREFBNEJsQixFQUE1QjtBQUNBLFNBQU9BLEdBQUdxQixlQUFILENBQW1CYixRQUFuQixFQUE2QlUsS0FBN0IsQ0FBUDtBQUNEOztBQUVEOztBQUVBOzs7Ozs7OztBQVFPLFNBQVMzQixNQUFULENBQWdCUyxFQUFoQixFQUFvQlEsUUFBcEIsRUFBOEI7QUFDbkNSLEtBQUdzQix1QkFBSCxDQUEyQmQsUUFBM0I7QUFDRDs7QUFFRDs7Ozs7OztBQU9PLFNBQVNoQixPQUFULENBQWlCUSxFQUFqQixFQUFxQlEsUUFBckIsRUFBK0I7QUFDcEM7QUFDQSxNQUFJQSxXQUFXLENBQWYsRUFBa0I7QUFDaEJSLE9BQUd1Qix3QkFBSCxDQUE0QmYsUUFBNUI7QUFDRDtBQUNGOztBQUVEOzs7Ozs7Ozs7QUFTTyxTQUFTZixVQUFULENBQW9CTyxFQUFwQixFQUF3QlEsUUFBeEIsRUFBa0NnQixPQUFsQyxFQUEyQztBQUNoRCxNQUFJeEIsZ0RBQUosRUFBMEM7QUFDeENBLE9BQUd5QixtQkFBSCxDQUF1QmpCLFFBQXZCLEVBQWlDZ0IsT0FBakM7QUFDQTtBQUNEO0FBQ0QsTUFBTUUsTUFBTTNCLGNBQWNDLEVBQWQsRUFBa0JFLFVBQWxCLENBQTZCLHdCQUE3QixDQUFaO0FBQ0EsTUFBSXdCLEdBQUosRUFBUztBQUNQQSxRQUFJQyx3QkFBSixDQUE2Qm5CLFFBQTdCLEVBQXVDZ0IsT0FBdkM7QUFDQTtBQUNEO0FBQ0Q7QUFDQSxNQUFJQSxZQUFZLENBQWhCLEVBQW1CO0FBQ2pCLFVBQU0sSUFBSUksS0FBSixDQUFVLHlDQUFWLENBQU47QUFDRDtBQUNGOztBQUVEOzs7Ozs7QUFNTyxTQUFTbEMsVUFBVCxDQUFvQk0sRUFBcEIsRUFBd0JRLFFBQXhCLEVBQWtDO0FBQ3ZDLHdCQUFPQSxXQUFXLENBQWxCO0FBQ0EsTUFBSVIsZ0RBQUosRUFBMEM7QUFDeEMsUUFBTXdCLFVBQVVmLElBQUlELFFBQUosRUFBY1IsR0FBRzZCLDJCQUFqQixDQUFoQjtBQUNBLFdBQU9MLE9BQVA7QUFDRDtBQUNELE1BQU1FLE1BQU0zQixjQUFjQyxFQUFkLEVBQWtCRSxVQUFsQixDQUE2Qix3QkFBN0IsQ0FBWjtBQUNBLE1BQUl3QixHQUFKLEVBQVM7QUFDUCxRQUFNRixXQUFVZixJQUFJRCxRQUFKLEVBQWNrQixJQUFJSSxpQ0FBbEIsQ0FBaEI7QUFDQSxXQUFPTixRQUFQO0FBQ0Q7QUFDRDtBQUNBLFNBQU8sQ0FBUDtBQUNEOztBQUVEOzs7Ozs7Ozs7Ozs7Ozs7O0FBZ0JPLFNBQVM3QixTQUFULEdBTUM7QUFBQSxpRkFBSixFQUFJOztBQUFBLE1BTE5LLEVBS00sUUFMTkEsRUFLTTtBQUFBLE1BSk5RLFFBSU0sUUFKTkEsUUFJTTtBQUFBLE1BSE51QixNQUdNLFFBSE5BLE1BR007QUFBQSxNQUZOQyxNQUVNLFFBRk5BLE1BRU07QUFBQSxNQUROQyxNQUNNLFFBRE5BLE1BQ007O0FBQ04sZ0RBQTRCakMsRUFBNUI7QUFDQStCLFdBQVMsaUJBQU9HLFFBQVAsQ0FBZ0JsQyxFQUFoQixFQUFvQitCLE1BQXBCLENBQVQ7O0FBRUE7QUFDQUMsV0FBUyxvQkFBTWhDLEVBQU4sRUFBVWdDLFdBQVc3QixTQUFYLEdBQXVCNkIsTUFBdkIsR0FBZ0NELE9BQU9DLE1BQWpELENBQVQ7QUFDQUMsV0FBU0EsV0FBVzlCLFNBQVgsR0FBdUI4QixNQUF2QixHQUFnQ0YsT0FBT0UsTUFBaEQ7QUFDQSx3QkFBT0QsTUFBUCxFQUFlLHdCQUFmO0FBQ0Esd0JBQU9DLE1BQVAsRUFBZSwwQ0FBZjs7QUFFQTtBQUNBRixTQUFPSSxJQUFQLENBQVksRUFBQ0gsUUFBUWhDLEdBQUdvQyxZQUFaLEVBQVo7O0FBRUE7QUFDQSxNQUFJLENBQUNILE9BQU9JLE9BQVosRUFBcUI7QUFDbkJyQyxPQUFHc0MsbUJBQUgsQ0FDRTlCLFFBREYsRUFFRXlCLE9BQU9NLElBRlQsRUFHRSxvQkFBTXZDLEVBQU4sRUFBVWlDLE9BQU9PLElBQWpCLENBSEYsRUFJRVAsT0FBT1EsVUFKVCxFQUtFUixPQUFPUyxNQUxULEVBTUVULE9BQU9VLE1BTlQ7QUFRRCxHQVRELE1BU087QUFDTDtBQUNBO0FBQ0E7QUFDQTtBQUNBLDBCQUFPM0MsZ0RBQVAsRUFBNkNGLFVBQTdDO0FBQ0FFLE9BQUc0QyxvQkFBSCxDQUNFcEMsUUFERixFQUVFeUIsT0FBT00sSUFGVCxFQUdFLG9CQUFNdkMsRUFBTixFQUFVaUMsT0FBT08sSUFBakIsQ0FIRixFQUlFUCxPQUFPUyxNQUpULEVBS0VULE9BQU9VLE1BTFQ7QUFPRDs7QUFFRFosU0FBT2MsTUFBUCxDQUFjLEVBQUNiLFFBQVFoQyxHQUFHb0MsWUFBWixFQUFkO0FBQ0Q7O0FBRUQ7Ozs7Ozs7OztBQVNPLFNBQVN4QyxVQUFULFFBQTJDO0FBQUEsTUFBdEJJLEVBQXNCLFNBQXRCQSxFQUFzQjtBQUFBLE1BQWxCUSxRQUFrQixTQUFsQkEsUUFBa0I7QUFBQSxNQUFSc0MsS0FBUSxTQUFSQSxLQUFROztBQUNoRCxhQUFJQyxJQUFKLENBQVMsQ0FBVCxFQUFZLGdEQUFaO0FBQ0E7O0FBRUEsTUFBSUQsaUJBQWlCRSxZQUFyQixFQUFtQztBQUNqQ2hELE9BQUdpRCxlQUFILENBQW1CekMsUUFBbkIsRUFBNkJzQyxLQUE3QjtBQUNELEdBRkQsTUFFTyxJQUFJQSxpQkFBaUJJLFVBQXJCLEVBQWlDO0FBQ3RDLDBCQUFPbEQsZ0RBQVAsRUFBNkMsaUJBQTdDO0FBQ0FBLE9BQUdtRCxnQkFBSCxDQUFvQjNDLFFBQXBCLEVBQThCc0MsS0FBOUI7QUFDRCxHQUhNLE1BR0EsSUFBSUEsaUJBQWlCTSxXQUFyQixFQUFrQztBQUN2QywwQkFBT3BELGdEQUFQLEVBQTZDLGlCQUE3QztBQUNBQSxPQUFHcUQsaUJBQUgsQ0FBcUI3QyxRQUFyQixFQUErQnNDLEtBQS9CO0FBQ0Q7QUFDRjs7QUFFRDs7Ozs7Ozs7QUFRQTtBQUNPLFNBQVNqRCxnQkFBVCxDQUEwQkcsRUFBMUIsRUFBOEJRLFFBQTlCLEVBQXdDOEMsRUFBeEMsRUFBNENDLEVBQTVDLEVBQWdEQyxFQUFoRCxFQUFvREMsRUFBcEQsRUFBd0Q7QUFDN0QsYUFBSVYsSUFBSixDQUFTLENBQVQsRUFBWSxzREFBWjtBQUNBLFVBQVFXLFVBQVVDLE1BQVYsR0FBbUIsQ0FBM0I7QUFDQSxTQUFLLENBQUw7QUFBUTNELFNBQUc0RCxjQUFILENBQWtCcEQsUUFBbEIsRUFBNEI4QyxFQUE1QixFQUFpQztBQUN6QyxTQUFLLENBQUw7QUFBUXRELFNBQUc2RCxjQUFILENBQWtCckQsUUFBbEIsRUFBNEI4QyxFQUE1QixFQUFnQ0MsRUFBaEMsRUFBcUM7QUFDN0MsU0FBSyxDQUFMO0FBQVF2RCxTQUFHOEQsY0FBSCxDQUFrQnRELFFBQWxCLEVBQTRCOEMsRUFBNUIsRUFBZ0NDLEVBQWhDLEVBQW9DQyxFQUFwQyxFQUF5QztBQUNqRCxTQUFLLENBQUw7QUFBUXhELFNBQUcrRCxjQUFILENBQWtCdkQsUUFBbEIsRUFBNEI4QyxFQUE1QixFQUFnQ0MsRUFBaEMsRUFBb0NDLEVBQXBDLEVBQXdDQyxFQUF4QyxFQUE2QztBQUNyRDtBQUNFLFlBQU0sSUFBSTdCLEtBQUosQ0FBVSwrQ0FBVixDQUFOO0FBTkY7O0FBU0E7QUFDQTtBQUNBO0FBQ0E7QUFDRCIsImZpbGUiOiJ2ZXJ0ZXgtYXR0cmlidXRlcy5qcyIsInNvdXJjZXNDb250ZW50IjpbIi8qIGVzbGludC1kaXNhYmxlIGRvdC1ub3RhdGlvbiovXG5pbXBvcnQge1dlYkdMMlJlbmRlcmluZ0NvbnRleHR9IGZyb20gJy4vd2ViZ2wtdHlwZXMnO1xuaW1wb3J0IHthc3NlcnRXZWJHTFJlbmRlcmluZ0NvbnRleHR9IGZyb20gJy4vd2ViZ2wtY2hlY2tzJztcbmltcG9ydCBCdWZmZXIgZnJvbSAnLi9idWZmZXInO1xuaW1wb3J0IHtnbEdldH0gZnJvbSAnLi9jb250ZXh0JztcbmltcG9ydCB7bG9nfSBmcm9tICcuLi91dGlscyc7XG5pbXBvcnQgYXNzZXJ0IGZyb20gJ2Fzc2VydCc7XG5cbi8qKlxuICogTWV0aG9kcyBmb3IgbWFuaXB1bGF0aW5nIHRoZSB2ZXJ0ZXggYXR0cmlidXRlcyBhcnJheSwgd2hpY2ggaXMgd2hlcmVcbiAqIHZlcnRleCBkYXRhIGlzIHN0YWdlZCBmb3Igc2hhZGVyIGV4ZWN1dGlvbi5cbiAqXG4gKiBWZXJ0ZXggYXR0cmlidXRlcyBhcmUgc3RvcmVkIGluIFwiYXJyYXlzXCIgd2l0aCBpbmRpY2VzIGZyb20gMCBhbmQgdXAuXG4gKiBEdXJpbmcgc2hhZGVyIGV4ZWN1dGlvbiwgdGhlc2UgaW5kaWNlcyAob3IgJ2xvY2F0aW9ucycpIGFyZSBtYXRjaGVkIHRvXG4gKiB0aGUgaW5kaWNlcyBhc3NpZ25lZCB0byBzaGFkZXIgYXR0cmlidXRlcyBkdXJpbmcgV2ViR0xQcm9ncmFtIGxpbmtpbmcuXG4gKlxuICogTm90ZTogVGhlIGdsb2JhbCBzdGF0ZSBjb250YWlucyBhbiBpbXBsaWNpdCB2ZXJ0ZXggYXR0cmlidXRlIGFycmF5IHdoaWNoXG4gKiB0aGUgbWV0aG9kcyBpbiB0aGlzIGNsYXNzIG1hbmlwdWxhdGUgYnkgZGVmYXVsdC4gSXQgaXMgYWxzbyBwb3NzaWJsZSB0b1xuICogY3JlYXRlIGFuZCBiaW5kIGEgVmVydGV4QXJyYXlPYmplY3QgdG8gbWFuYWdlIG11bHRpcGxlIGFycmF5cy5cbiAqXG4gKiBFYWNoIHZlcnRleCBhdHRyaWJ1dGUgaGFzIHRoZXNlIHByb3BlcnRpZXM6XG4gKiAtIENhbiBiZSBlbmFibGVkIG9yIGRpc2FibGVkIChPbmx5IGVuYWJsZSBhdHRycyBhY3R1YWxseSB1c2VkIGJ5IGEgcHJvZ3JhbSlcbiAqIC0gSGFzIGFuIGluc3RhbmNlIGBkaXZpc29yYCAodXN1YWxseSAxIG9yIDAgdG8gZW5hYmxlL2Rpc2FibGUgaW5zdGFuY2luZylcbiAqIC0gSGF2ZSBhIHNpemUgKDEtNCB2YWx1ZXMgcGVyIHZlcnRleClcbiAqIC0gSGFzIGEgdmFsdWUgb3IgdmFsdWVzIHRoYXQgaXMgYWNjZXNzaWJsZSBpbiBzaGFkZXJzXG4gKlxuICogQXR0cmlidXRlIHZhbHVlcyBhcmUgZWl0aGVyXG4gKiAtIEdlbmVyaWM6IGEgY29uc3RhbnQgdmFsdWUgZm9yIGFsbCB2ZXJ0aWNlcy9pbnN0YW5jZXMsIG9yXG4gKiAtIEJvdW5kIHRvIGEgV2ViR0xCdWZmZXIgd2l0aCB1bmlxdWUgdmFsdWVzIGZvciBlYWNoIHZlcnRleC9pbnN0YW5jZVxuICpcbiAqIFdoZW4gYmluZGluZyB0byBhIFdlYkdMQnVmZmVyIGl0IGlzIG5lY2Vzc2FyeSB0byBzcGVjaWZ5IHRoZSBsYXlvdXQgb2ZcbiAqIGRhdGEgaW4gdGhlIGJ1ZmZlcjpcbiAqIC0gc2l6ZSAoMS00IHZhbHVlcyBwZXIgdmVydGV4KVxuICogLSBkYXRhIHR5cGUgKGUuZy4gZ2wuRkxPQVQpXG4gKiAtIHN0cmlkZSwgb2Zmc2V0LCBhbmQgaW50ZWdlciBub3JtYWxpemF0aW9uIHBvbGljeSBjYW4gYWxzbyBiZSBzcGVjaWZpZWRcbiAqXG4gKiBOb3RlOiBBbGwgbWV0aG9kcyBpbiB0aGlzIGNsYXNzIHRha2UgYSBgbG9jYXRpb25gIGluZGV4IHRvIHNwZWNpZnkgd2hpY2hcbiAqIHZlcnRleCBhdHRyaWJ1dGUgaW4gdGhlIGFycmF5IHRoZXkgYXJlIG9wZXJhdGluZyBvbi5cbiAqXG4gKiBOb3RlOiBBdHRyaWJ1dGUgMCBjYW4gc29tZXRpbWVzIGJlIHRyZWF0ZWQgc3BlY2lhbGx5IGJ5IHRoZSBkcml2ZXIsXG4gKiB0byBiZSBzYWZlIHdlIGF2b2lkIGRpc2FibGluZyBpdC5cbiAqXG4gKiBOb3RlOiBXZWJHTDJcbiAqIC0gSW1wcm92ZXMgc3VwcG9ydCBmb3IgaW50ZWdlciBhdHRyaWJ1dGVzLCBib3RoIGdlbmVyaWMgYW5kIGJ1ZmZlcmVkLlxuICogLSBTZXR0aW5nIGluc3RhbmNlIFwiZGl2aXNvcnNcIiBubyBsb25nZXIgcmVxdWlyZSB1c2luZyBhIFdlYkdMIGV4dGVuc2lvbi5cbiAqXG4gKi9cblxuY29uc3QgRVJSX1dFQkdMMiA9ICdXZWJHTDIgcmVxdWlyZWQnO1xuXG5mdW5jdGlvbiBnbEdldEx1bWFJbmZvKGdsKSB7XG4gIGlmICghZ2wubHVtYSkge1xuICAgIGdsLmx1bWEgPSB7ZXh0ZW5zaW9uczoge319O1xuICB9XG4gIGlmIChnbC5sdW1hLmV4dGVuc2lvbnNbJ0FOR0xFX2luc3RhbmNlZF9hcnJheXMnXSA9PT0gdW5kZWZpbmVkKSB7XG4gICAgZ2wubHVtYS5leHRlbnNpb25zWydBTkdMRV9pbnN0YW5jZWRfYXJyYXlzJ10gPVxuICAgICAgZ2wuZ2V0RXh0ZW5zaW9uKCdBTkdMRV9pbnN0YW5jZWRfYXJyYXlzJyk7XG4gIH1cbiAgcmV0dXJuIGdsLmx1bWE7XG59XG5cbi8vIEFDQ0VTU09SU1xuXG4vKipcbiAqIFRoZSBtYXggbnVtYmVyIG9mIGF0dHJpYnV0ZXMgaW4gdGhlIHZlcnRleCBhdHRyaWJ1dGUgYXJyYXkgaXMgYW5cbiAqIGltcGxlbWVudGF0aW9uIGRlZmluZWQgbGltaXQsIGJ1dCBuZXZlciBzbWFsbGVyIHRoYW4gOFxuICogQHBhcmFtIHtXZWJHTFJlbmRlcmluZ0NvbnRleHR9IGdsIC0gd2ViZ2wgY29udGV4dFxuICogQHJldHVybnMge0dMdWludH0gLSAobWF4KSBudW1iZXIgb2YgYXR0cmlidXRlcyBpbiB0aGUgdmVydGV4IGF0dHJpYnV0ZSBhcnJheVxuICovXG5leHBvcnQgZnVuY3Rpb24gZ2V0TWF4QXR0cmlidXRlcyhnbCkge1xuICBhc3NlcnRXZWJHTFJlbmRlcmluZ0NvbnRleHQoZ2wpO1xuICByZXR1cm4gZ2wuZ2V0UGFyYW1ldGVyKGdsLk1BWF9WRVJURVhfQVRUUklCUyk7XG59XG5cbi8qKlxuICogSXMgaW5zdGFuY2UgZGl2aXNvciBhdmFpbGJsZSAoY2hlY2tzIGZvciBXZWJHTDIgb3IgQU5HTEUgZXh0ZW5zaW9uKVxuICogQHBhcmFtIHtXZWJHTFJlbmRlcmluZ0NvbnRleHR9IGdsIC0gd2ViZ2wgY29udGV4dFxuICogQHJldHVybnMge0Jvb2xlYW59IC0gaXMgZGl2aXNvciBhdmFpbGFibGU/XG4gKi9cbmV4cG9ydCBmdW5jdGlvbiBoYXNEaXZpc29yKGdsKSB7XG4gIGFzc2VydFdlYkdMUmVuZGVyaW5nQ29udGV4dChnbCk7XG4gIHJldHVybiBCb29sZWFuKFxuICAgIGdsIGluc3RhbmNlb2YgV2ViR0wyUmVuZGVyaW5nQ29udGV4dCB8fFxuICAgIGdsLmdldEV4dGVuc2lvbihnbCwgJ0FOR0xFX2luc3RhbmNlZF9hcnJheXMnKVxuICApO1xufVxuXG4vKipcbiAqIFJldHVybnMgdHJ1ZSBpZiB0aGUgdmVydGV4IGF0dHJpYnV0ZSBpcyBlbmFibGVkIGF0IHRoaXMgaW5kZXguXG4gKiBAcGFyYW0ge1dlYkdMUmVuZGVyaW5nQ29udGV4dH0gZ2wgLSB3ZWJnbCBjb250ZXh0XG4gKiBAcGFyYW0ge0dMdWludH0gbG9jYXRpb24gLSBvcmRpbmFsIG51bWJlciBvZiB0aGUgYXR0cmlidXRlXG4gKiBAcmV0dXJucyB7Qm9vbGVhbn0gLSBlbmFibGVkIHN0YXR1c1xuICovXG5leHBvcnQgZnVuY3Rpb24gaXNFbmFibGVkKGdsLCBsb2NhdGlvbikge1xuICByZXR1cm4gQm9vbGVhbihnZXQoZ2wsIGxvY2F0aW9uLCBnbC5WRVJURVhfQVRUUklCX0FSUkFZX0VOQUJMRUQpKTtcbn1cblxuLyoqXG4gKiBSZXR1cm5zIHRoZSBjdXJyZW50bHkgYm91bmQgYnVmZmVyXG4gKiBAcGFyYW0ge1dlYkdMUmVuZGVyaW5nQ29udGV4dH0gZ2wgLSB3ZWJnbCBjb250ZXh0XG4gKiBAcGFyYW0ge0dMdWludH0gbG9jYXRpb24gLSBvcmRpbmFsIG51bWJlciBvZiB0aGUgYXR0cmlidXRlXG4gKiBAcmV0dXJucyB7V2ViR0xCdWZmZXJ9IFJldHVybnMgdGhlIGN1cnJlbnRseSBib3VuZCBidWZmZXJcbiAqL1xuZXhwb3J0IGZ1bmN0aW9uIGdldEJ1ZmZlcihnbCwgbG9jYXRpb24pIHtcbiAgcmV0dXJuIGdldChnbCwgbG9jYXRpb24sIGdsLlZFUlRFWF9BVFRSSUJfQVJSQVlfQlVGRkVSX0JJTkRJTkcpO1xufVxuXG4vKipcbiAqIEdldCB2YWx1ZXMgZm9yIGdlbmVyaWMgdmVydGV4IGF0dHJpYnV0ZXNcbiAqIEBwYXJhbSB7V2ViR0xSZW5kZXJpbmdDb250ZXh0fSBnbCAtIHdlYmdsIGNvbnRleHRcbiAqIEBwYXJhbSB7R0x1aW50fSBsb2NhdGlvbiAtIG9yZGluYWwgbnVtYmVyIG9mIHRoZSBhdHRyaWJ1dGVcbiAqIEByZXR1cm5zIHtGbG9hdDMyQXJyYXl9ICh3aXRoIDQgZWxlbWVudHMpIHJlcHJlc2VudGluZyB0aGUgY3VycmVudCB2YWx1ZVxuICogb2YgdGhlIHZlcnRleCBhdHRyaWJ1dGUgYXQgdGhlIGdpdmVuIGluZGV4LlxuICovXG5leHBvcnQgZnVuY3Rpb24gZ2V0R2VuZXJpYyhnbCwgbG9jYXRpb24pIHtcbiAgcmV0dXJuIGdldChnbCwgZ2wuQ1VSUkVOVF9WRVJURVhfQVRUUklCKTtcbn1cblxuLyoqXG4gKiBAcGFyYW0ge1dlYkdMUmVuZGVyaW5nQ29udGV4dH0gZ2wgLSB3ZWJnbCBjb250ZXh0XG4gKiBAcGFyYW0ge0dMdWludH0gbG9jYXRpb24gLSBvcmRpbmFsIG51bWJlciBvZiB0aGUgYXR0cmlidXRlXG4gKi9cbi8vIEByZXR1cm5zIHtHTGludH0gdGhlIHNpemUgb2YgYW4gZWxlbWVudCBvZiB0aGUgdmVydGV4IGFycmF5LlxuZXhwb3J0IGZ1bmN0aW9uIGdldFNpemUoZ2wsIGxvY2F0aW9uKSB7XG4gIHJldHVybiBnZXQobG9jYXRpb24sIGdsLlZFUlRFWF9BVFRSSUJfQVJSQVlfU0laRSk7XG59XG5cbi8qKlxuICogQHBhcmFtIHtXZWJHTFJlbmRlcmluZ0NvbnRleHR9IGdsIC0gd2ViZ2wgY29udGV4dFxuICogQHBhcmFtIHtHTHVpbnR9IGxvY2F0aW9uIC0gb3JkaW5hbCBudW1iZXIgb2YgdGhlIGF0dHJpYnV0ZVxuICovXG4vLyBAcmV0dXJucyB7R0xlbnVtfSByZXByZXNlbnRpbmcgdGhlIGFycmF5IHR5cGUuXG5leHBvcnQgZnVuY3Rpb24gZ2V0VHlwZShnbCwgbG9jYXRpb24pIHtcbiAgcmV0dXJuIGdldChsb2NhdGlvbiwgZ2wuVkVSVEVYX0FUVFJJQl9BUlJBWV9UWVBFKTtcbn1cblxuLyoqXG4gKiBAcGFyYW0ge1dlYkdMUmVuZGVyaW5nQ29udGV4dH0gZ2wgLSB3ZWJnbCBjb250ZXh0XG4gKiBAcGFyYW0ge0dMdWludH0gbG9jYXRpb24gLSBvcmRpbmFsIG51bWJlciBvZiB0aGUgYXR0cmlidXRlXG4gKi9cbi8vIEByZXR1cm5zIHtHTGJvb2xlYW59IHRydWUgaWYgZml4ZWQtcG9pbnQgZGF0YSB0eXBlcyBhcmUgbm9ybWFsaXplZFxuLy8gZm9yIHRoZSB2ZXJ0ZXggYXR0cmlidXRlIGFycmF5IGF0IHRoZSBnaXZlbiBpbmRleC5cbmV4cG9ydCBmdW5jdGlvbiBpc05vcm1hbGl6ZWQoZ2wsIGxvY2F0aW9uKSB7XG4gIHJldHVybiBnZXQobG9jYXRpb24sIGdsLlZFUlRFWF9BVFRSSUJfQVJSQVlfTk9STUFMSVpFRCk7XG59XG5cbi8qKlxuICogY2hlY2sgaWYgYW4gaW50ZWdlciBkYXRhIHR5cGUgaW4gdGhlIHZlcnRleCBhdHRyaWJ1dGUgYXQgaW5kZXhcbiAqIEBwYXJhbSB7V2ViR0xSZW5kZXJpbmdDb250ZXh0fSBnbCAtIHdlYmdsIGNvbnRleHRcbiAqIEBwYXJhbSB7R0x1aW50fSBsb2NhdGlvbiAtIGluZGV4IG9mIHRoZSB2ZXJ0ZXggYXR0cmlidXRlLlxuICogQHJldHVybnMge0dMYm9vbGVhbn0gLSB0cnVlIGlmIGFuIGludGVnZXIgZGF0YSB0eXBlIGlzIGluIHRoZVxuICogdmVydGV4IGF0dHJpYnV0ZSBhcnJheSBhdCB0aGUgZ2l2ZW4gaW5kZXguXG4gKi9cbmV4cG9ydCBmdW5jdGlvbiBpc0ludGVnZXIoZ2wsIGxvY2F0aW9uKSB7XG4gIGFzc2VydChnbCBpbnN0YW5jZW9mIFdlYkdMMlJlbmRlcmluZ0NvbnRleHQsIEVSUl9XRUJHTDIpO1xuICByZXR1cm4gZ2V0KGxvY2F0aW9uLCBnbC5WRVJURVhfQVRUUklCX0FSUkFZX0lOVEVHRVIpO1xufVxuXG4vKipcbiAqIEBwYXJhbSB7V2ViR0xSZW5kZXJpbmdDb250ZXh0fSBnbCAtIHdlYmdsIGNvbnRleHRcbiAqIEBwYXJhbSB7R0x1aW50fSBsb2NhdGlvbiAtIG9yZGluYWwgbnVtYmVyIG9mIHRoZSBhdHRyaWJ1dGVcbiAqIEByZXR1cm5zIHtHTGludH0gbnVtYmVyIG9mIGJ5dGVzIGJldHdlZW4gc3VjY2Vzc2l2ZSBlbGVtZW50cyBpbiB0aGUgYXJyYXkuXG4gKiAwIG1lYW5zIHRoYXQgdGhlIGVsZW1lbnRzIGFyZSBzZXF1ZW50aWFsLlxuICovXG5leHBvcnQgZnVuY3Rpb24gZ2V0U3RyaWRlKGdsLCBsb2NhdGlvbikge1xuICByZXR1cm4gZ2V0KGxvY2F0aW9uLCBnbC5WRVJURVhfQVRUUklCX0FSUkFZX1NUUklERSk7XG59XG5cbi8qKlxuICogQHBhcmFtIHtXZWJHTFJlbmRlcmluZ0NvbnRleHR9IGdsIC0gd2ViZ2wgY29udGV4dFxuICogQHBhcmFtIHtHTHVpbnR9IGxvY2F0aW9uIC0gb3JkaW5hbCBudW1iZXIgb2YgdGhlIGF0dHJpYnV0ZVxuICogQHBhcmFtIHtHTHVpbnR9IHBuYW1lIC0gZW51bSBzcGVjaWZ5aW5nIHdoaWNoIG9mZnNldCB0byByZXR1cm5cbiAqIEByZXR1cm5zIHtHTHVpbnR9IHRoZSBhZGRyZXNzIG9mIGEgc3BlY2lmaWVkIHZlcnRleCBhdHRyaWJ1dGUuXG4gKi9cbmV4cG9ydCBmdW5jdGlvbiBnZXRPZmZzZXQoXG4gIGdsLCBsb2NhdGlvbiwgcG5hbWUgPSBnbC5WRVJURVhfQVRUUklCX0FSUkFZX1BPSU5URVJcbikge1xuICByZXR1cm4gZ2wuZ2V0VmVydGV4QXR0cmliT2Zmc2V0KGxvY2F0aW9uLCBwbmFtZSk7XG59XG5cbi8qKlxuICogQHByaXZhdGVcbiAqIEdlbmVyaWMgZ2V0dGVyIGZvciBpbmZvcm1hdGlvbiBhYm91dCBhIHZlcnRleCBhdHRyaWJ1dGUgYXQgYSBnaXZlbiBwb3NpdGlvblxuICogQHBhcmFtIHtXZWJHTFJlbmRlcmluZ0NvbnRleHR9IGdsIC0gd2ViZ2wgY29udGV4dFxuICogQHBhcmFtIHtHTHVpbnR9IGxvY2F0aW9uIC0gaW5kZXggb2YgdGhlIHZlcnRleCBhdHRyaWJ1dGUuXG4gKiBAcGFyYW0ge0dMZW51bX0gcG5hbWUgLSBzcGVjaWZpZXMgdGhlIGluZm9ybWF0aW9uIHRvIHF1ZXJ5LlxuICogQHJldHVybnMgeyp9IC0gcmVxdWVzdGVkIHZlcnRleCBhdHRyaWJ1dGUgaW5mb3JtYXRpb24gKHNwZWNpZmllZCBieSBwbmFtZSlcbiAqL1xuZnVuY3Rpb24gZ2V0KGdsLCBsb2NhdGlvbiwgcG5hbWUpIHtcbiAgYXNzZXJ0V2ViR0xSZW5kZXJpbmdDb250ZXh0KGdsKTtcbiAgcmV0dXJuIGdsLmdldFZlcnRleEF0dHJpYihsb2NhdGlvbiwgcG5hbWUpO1xufVxuXG4vLyBNT0RJRklFUlNcblxuLyoqXG4gKiBFbmFibGUgdGhlIGF0dHJpYnV0ZVxuICogTm90ZTogQnkgZGVmYXVsdCBhbGwgYXR0cmlidXRlcyBhcmUgZGlzYWJsZWQuIE9ubHkgYXR0cmlidXRlc1xuICogdXNlZCBieSBhIHByb2dyYW0ncyBzaGFkZXJzIHNob3VsZCBiZSBlbmFibGVkLlxuICpcbiAqIEBwYXJhbSB7V2ViR0xSZW5kZXJpbmdDb250ZXh0fSBnbCAtIHdlYmdsIGNvbnRleHRcbiAqIEBwYXJhbSB7R0x1aW50fSBsb2NhdGlvbiAtIG9yZGluYWwgbnVtYmVyIG9mIHRoZSBhdHRyaWJ1dGVcbiAqL1xuZXhwb3J0IGZ1bmN0aW9uIGVuYWJsZShnbCwgbG9jYXRpb24pIHtcbiAgZ2wuZW5hYmxlVmVydGV4QXR0cmliQXJyYXkobG9jYXRpb24pO1xufVxuXG4vKipcbiAqIERpc2FibGUgdGhlIGF0dHJpYnV0ZVxuICogTm90ZTogT25seSBhdHRyaWJ1dGVzIHVzZWQgYnkgYSBwcm9ncmFtJ3Mgc2hhZGVycyBzaG91bGQgYmUgZW5hYmxlZC5cbiAqXG4gKiBAcGFyYW0ge1dlYkdMUmVuZGVyaW5nQ29udGV4dH0gZ2wgLSB3ZWJnbCBjb250ZXh0XG4gKiBAcGFyYW0ge0dMdWludH0gbG9jYXRpb24gLSBvcmRpbmFsIG51bWJlciBvZiB0aGUgYXR0cmlidXRlXG4gKi9cbmV4cG9ydCBmdW5jdGlvbiBkaXNhYmxlKGdsLCBsb2NhdGlvbikge1xuICAvLyBEb24ndCBkaXNhYmxlIGxvY2F0aW9uIDBcbiAgaWYgKGxvY2F0aW9uID4gMCkge1xuICAgIGdsLmRpc2FibGVWZXJ0ZXhBdHRyaWJBcnJheShsb2NhdGlvbik7XG4gIH1cbn1cblxuLyoqXG4gKiBTZXQgdGhlIGZyZXF1ZW5jeSBkaXZpc29yIHVzZWQgZm9yIGluc3RhbmNlZCByZW5kZXJpbmcuXG4gKiBOb3RlOiBVc3VhbGx5IHNpbXBseSBzZXQgdG8gMSBvciAwIHRvIGVuYWJsZS9kaXNhYmxlIGluc3RhbmNlZCByZW5kZXJpbmdcbiAqIGZvciBhIHNwZWNpZmljIGF0dHJpYnV0ZS5cbiAqXG4gKiBAcGFyYW0ge1dlYkdMUmVuZGVyaW5nQ29udGV4dH0gZ2wgLSB3ZWJnbCBjb250ZXh0XG4gKiBAcGFyYW0ge0dMdWludH0gbG9jYXRpb24gLSBvcmRpbmFsIG51bWJlciBvZiB0aGUgYXR0cmlidXRlXG4gKiBAcGFyYW0ge0dMdWludH0gZGl2aXNvciAtIGluc3RhbmNlcyB0aGF0IHBhc3MgYmV0d2VlbiB1cGRhdGVzIG9mIGF0dHJpYnV0ZVxuICovXG5leHBvcnQgZnVuY3Rpb24gc2V0RGl2aXNvcihnbCwgbG9jYXRpb24sIGRpdmlzb3IpIHtcbiAgaWYgKGdsIGluc3RhbmNlb2YgV2ViR0wyUmVuZGVyaW5nQ29udGV4dCkge1xuICAgIGdsLnZlcnRleEF0dHJpYkRpdmlzb3IobG9jYXRpb24sIGRpdmlzb3IpO1xuICAgIHJldHVybjtcbiAgfVxuICBjb25zdCBleHQgPSBnbEdldEx1bWFJbmZvKGdsKS5leHRlbnNpb25zWydBTkdMRV9pbnN0YW5jZWRfYXJyYXlzJ107XG4gIGlmIChleHQpIHtcbiAgICBleHQudmVydGV4QXR0cmliRGl2aXNvckFOR0xFKGxvY2F0aW9uLCBkaXZpc29yKTtcbiAgICByZXR1cm47XG4gIH1cbiAgLy8gQWNjZXB0IGRpdmlzb3IgMCBldmVuIGlmIGluc3RhbmNpbmcgaXMgbm90IHN1cHBvcnRlZCAoMCA9IG5vIGluc3RhbmNpbmcpXG4gIGlmIChkaXZpc29yICE9PSAwKSB7XG4gICAgdGhyb3cgbmV3IEVycm9yKCdXZWJHTCBpbnN0YW5jZWQgcmVuZGVyaW5nIG5vdCBzdXBwb3J0ZWQnKTtcbiAgfVxufVxuXG4vKipcbiAqIFJldHVybnMgdGhlIGZyZXF1ZW5jeSBkaXZpc29yIHVzZWQgZm9yIGluc3RhbmNlZCByZW5kZXJpbmcuXG4gKiBAcGFyYW0ge1dlYkdMUmVuZGVyaW5nQ29udGV4dH0gZ2wgLSB3ZWJnbCBjb250ZXh0XG4gKiBAcGFyYW0ge0dMdWludH0gbG9jYXRpb24gLSBvcmRpbmFsIG51bWJlciBvZiB0aGUgYXR0cmlidXRlXG4gKiBAcmV0dXJucyB7R0x1aW50fSBkaXZpc29yXG4gKi9cbmV4cG9ydCBmdW5jdGlvbiBnZXREaXZpc29yKGdsLCBsb2NhdGlvbikge1xuICBhc3NlcnQobG9jYXRpb24gPiAwKTtcbiAgaWYgKGdsIGluc3RhbmNlb2YgV2ViR0wyUmVuZGVyaW5nQ29udGV4dCkge1xuICAgIGNvbnN0IGRpdmlzb3IgPSBnZXQobG9jYXRpb24sIGdsLlZFUlRFWF9BVFRSSUJfQVJSQVlfRElWSVNPUik7XG4gICAgcmV0dXJuIGRpdmlzb3I7XG4gIH1cbiAgY29uc3QgZXh0ID0gZ2xHZXRMdW1hSW5mbyhnbCkuZXh0ZW5zaW9uc1snQU5HTEVfaW5zdGFuY2VkX2FycmF5cyddO1xuICBpZiAoZXh0KSB7XG4gICAgY29uc3QgZGl2aXNvciA9IGdldChsb2NhdGlvbiwgZXh0LlZFUlRFWF9BVFRSSUJfQVJSQVlfRElWSVNPUl9BTkdMRSk7XG4gICAgcmV0dXJuIGRpdmlzb3I7XG4gIH1cbiAgLy8gaWYgaW5zdGFuY2luZyBpcyBub3QgYXZhaWxhYmxlLCByZXR1cm4gMCBtZWFuaW5nIGRpdmlzb3IgaGFzIG5vdCBiZWVuIHNldFxuICByZXR1cm4gMDtcbn1cblxuLyoqXG4gKiBTZXQgYSBsb2NhdGlvbiBpbiB2ZXJ0ZXggYXR0cmlidXRlcyBhcnJheSB0byBhIGJ1ZmZlciwgc3BlY2lmeWluZ1xuICogaXRzIGRhdGEgbGF5b3V0IGFuZCBpbnRlZ2VyIHRvIGZsb2F0IGNvbnZlcnNpb24gYW5kIG5vcm1hbGl6YXRpb24gZmxhZ3NcbiAqXG4gKiBAcGFyYW0ge1dlYkdMUmVuZGVyaW5nQ29udGV4dH0gZ2wgLSB3ZWJnbCBjb250ZXh0XG4gKiBAcGFyYW0ge0dMdWludH0gbG9jYXRpb24gLSBvcmRpbmFsIG51bWJlciBvZiB0aGUgYXR0cmlidXRlXG4gKiBAcGFyYW0ge1dlYkdMQnVmZmVyfEJ1ZmZlcn0gYnVmZmVyIC0gV2ViR0wgYnVmZmVyIHRvIHNldCBhcyB2YWx1ZVxuICogQHBhcmFtIHtHTHVpbnR9IHRhcmdldD1nbC5BUlJBWV9CVUZGRVIgLSB3aGljaCB0YXJnZXQgdG8gYmluZCB0b1xuICogQHBhcmFtIHtPYmplY3R9IGxheW91dD0gT3B0aW9uYWwgZGF0YSBsYXlvdXQsIGRlZmF1bHRzIHRvIGJ1ZmZlcidzIGxheW91dFxuICogQHBhcmFtIHtHTHVpbnR9IGxheW91dC5zaXplIC0gbnVtYmVyIG9mIHZhbHVlcyBwZXIgZWxlbWVudCAoMS00KVxuICogQHBhcmFtIHtHTHVpbnR9IGxheW91dC50eXBlIC0gdHlwZSBvZiB2YWx1ZXMgKGUuZy4gZ2wuRkxPQVQpXG4gKiBAcGFyYW0ge0dMYm9vbH0gbGF5b3V0Lm5vcm1hbGl6ZWQ9ZmFsc2UgLSBub3JtYWxpemUgaW50ZWdlcnMgdG8gWy0xLDFdLCBbMCwxXVxuICogQHBhcmFtIHtHTHVpbnR9IGxheW91dC5pbnRlZ2VyPWZhbHNlIC0gV2ViR0wyIG9ubHksIGRpc2FibGUgaW50LXRvLWZsb2F0IGNvbnZcbiAqIEBwYXJhbSB7R0x1aW50fSBsYXlvdXQuc3RyaWRlPTAgLSBzdXBwb3J0cyBzdHJpZGVkIGFycmF5c1xuICogQHBhcmFtIHtHTHVpbnR9IGxheW91dC5vZmZzZXQ9MCAtIHN1cHBvcnRzIHN0cmlkZWQgYXJyYXlzXG4gKi9cbmV4cG9ydCBmdW5jdGlvbiBzZXRCdWZmZXIoe1xuICBnbCxcbiAgbG9jYXRpb24sXG4gIGJ1ZmZlcixcbiAgdGFyZ2V0LFxuICBsYXlvdXRcbn0gPSB7fSkge1xuICBhc3NlcnRXZWJHTFJlbmRlcmluZ0NvbnRleHQoZ2wpO1xuICBidWZmZXIgPSBCdWZmZXIubWFrZUZyb20oZ2wsIGJ1ZmZlcik7XG5cbiAgLy8gQ29weSBtYWluIGRhdGEgY2hhcmFjdGVyaXN0aWNzIGZyb20gYnVmZmVyXG4gIHRhcmdldCA9IGdsR2V0KGdsLCB0YXJnZXQgIT09IHVuZGVmaW5lZCA/IHRhcmdldCA6IGJ1ZmZlci50YXJnZXQpO1xuICBsYXlvdXQgPSBsYXlvdXQgIT09IHVuZGVmaW5lZCA/IGxheW91dCA6IGJ1ZmZlci5sYXlvdXQ7XG4gIGFzc2VydCh0YXJnZXQsICdzZXRCdWZmZXIgbmVlZHMgdGFyZ2V0Jyk7XG4gIGFzc2VydChsYXlvdXQsICdzZXRCdWZmZXIgY2FsbGVkIG9uIHVuaW5pdGlhbGl6ZWQgYnVmZmVyJyk7XG5cbiAgLy8gYSBub24temVybyBuYW1lZCBidWZmZXIgb2JqZWN0IG11c3QgYmUgYm91bmQgdG8gdGhlIEdMX0FSUkFZX0JVRkZFUiB0YXJnZXRcbiAgYnVmZmVyLmJpbmQoe3RhcmdldDogZ2wuQVJSQVlfQlVGRkVSfSk7XG5cbiAgLy8gQXR0YWNoIGJvdW5kIEFSUkFZX0JVRkZFUiB3aXRoIHNwZWNpZmllZCBidWZmZXIgZm9ybWF0IHRvIGxvY2F0aW9uXG4gIGlmICghbGF5b3V0LmludGVnZXIpIHtcbiAgICBnbC52ZXJ0ZXhBdHRyaWJQb2ludGVyKFxuICAgICAgbG9jYXRpb24sXG4gICAgICBsYXlvdXQuc2l6ZSxcbiAgICAgIGdsR2V0KGdsLCBsYXlvdXQudHlwZSksXG4gICAgICBsYXlvdXQubm9ybWFsaXplZCxcbiAgICAgIGxheW91dC5zdHJpZGUsXG4gICAgICBsYXlvdXQub2Zmc2V0XG4gICAgKTtcbiAgfSBlbHNlIHtcbiAgICAvLyBzcGVjaWZpZXMgKmludGVnZXIqIGRhdGEgZm9ybWF0cyBhbmQgbG9jYXRpb25zIG9mIHZlcnRleCBhdHRyaWJ1dGVzXG4gICAgLy8gRm9yIGdsVmVydGV4QXR0cmliSVBvaW50ZXIsIFZhbHVlcyBhcmUgYWx3YXlzIGxlZnQgYXMgaW50ZWdlciB2YWx1ZXMuXG4gICAgLy8gT25seSBhY2NlcHRzIHRoZSBpbnRlZ2VyIHR5cGVzIGdsLkJZVEUsIGdsLlVOU0lHTkVEX0JZVEUsXG4gICAgLy8gZ2wuU0hPUlQsIGdsLlVOU0lHTkVEX1NIT1JULCBnbC5JTlQsIGdsLlVOU0lHTkVEX0lOVFxuICAgIGFzc2VydChnbCBpbnN0YW5jZW9mIFdlYkdMMlJlbmRlcmluZ0NvbnRleHQsIEVSUl9XRUJHTDIpO1xuICAgIGdsLnZlcnRleEF0dHJpYklQb2ludGVyKFxuICAgICAgbG9jYXRpb24sXG4gICAgICBsYXlvdXQuc2l6ZSxcbiAgICAgIGdsR2V0KGdsLCBsYXlvdXQudHlwZSksXG4gICAgICBsYXlvdXQuc3RyaWRlLFxuICAgICAgbGF5b3V0Lm9mZnNldFxuICAgICk7XG4gIH1cblxuICBidWZmZXIudW5iaW5kKHt0YXJnZXQ6IGdsLkFSUkFZX0JVRkZFUn0pO1xufVxuXG4vKlxuICogU3BlY2lmeSB2YWx1ZXMgZm9yIGdlbmVyaWMgdmVydGV4IGF0dHJpYnV0ZXNcbiAqIEdlbmVyaWMgdmVydGV4IGF0dHJpYnV0ZXMgYXJlIGNvbnN0YW50IGZvciBhbGwgdmVydGljZXNcbiAqIFVwIHRvIDQgdmFsdWVzIGRlcGVuZGluZyBvbiBhdHRyaWJ1dGUgc2l6ZVxuICpcbiAqIEBwYXJhbSB7V2ViR0xSZW5kZXJpbmdDb250ZXh0fSBnbCAtIHdlYmdsIGNvbnRleHRcbiAqIEBwYXJhbSB7R0x1aW50fSBsb2NhdGlvbiAtIG9yZGluYWwgbnVtYmVyIG9mIHRoZSBhdHRyaWJ1dGVcbiAqIEBwYXJhbSB7R0x1aW50fSBkaXZpc29yIC0gaW5zdGFuY2VzIHRoYXQgcGFzcyBiZXR3ZWVuIHVwZGF0ZXMgb2YgYXR0cmlidXRlXG4gKi9cbmV4cG9ydCBmdW5jdGlvbiBzZXRHZW5lcmljKHtnbCwgbG9jYXRpb24sIGFycmF5fSkge1xuICBsb2cud2FybigwLCAnVmVydGV4QXR0cmlidXRlcy5zZXRHZW5lcmljIGlzIG5vdCB3ZWxsIHRlc3RlZCcpO1xuICAvLyB0aHJvdyBuZXcgRXJyb3IoJ3ZlcnRleCBhdHRyaWJ1dGUgc2l6ZSBtdXN0IGJlIGJldHdlZW4gMSBhbmQgNCcpO1xuXG4gIGlmIChhcnJheSBpbnN0YW5jZW9mIEZsb2F0MzJBcnJheSkge1xuICAgIGdsLnZlcnRleEF0dHJpYjRmdihsb2NhdGlvbiwgYXJyYXkpO1xuICB9IGVsc2UgaWYgKGFycmF5IGluc3RhbmNlb2YgSW50MzJBcnJheSkge1xuICAgIGFzc2VydChnbCBpbnN0YW5jZW9mIFdlYkdMMlJlbmRlcmluZ0NvbnRleHQsICdXZWJHTDIgcmVxdWlyZWQnKTtcbiAgICBnbC52ZXJ0ZXhBdHRyaWJJNGl2KGxvY2F0aW9uLCBhcnJheSk7XG4gIH0gZWxzZSBpZiAoYXJyYXkgaW5zdGFuY2VvZiBVaW50MzJBcnJheSkge1xuICAgIGFzc2VydChnbCBpbnN0YW5jZW9mIFdlYkdMMlJlbmRlcmluZ0NvbnRleHQsICdXZWJHTDIgcmVxdWlyZWQnKTtcbiAgICBnbC52ZXJ0ZXhBdHRyaWJJNHVpdihsb2NhdGlvbiwgYXJyYXkpO1xuICB9XG59XG5cbi8qXG4gKiBTcGVjaWZ5IHZhbHVlcyBmb3IgZ2VuZXJpYyB2ZXJ0ZXggYXR0cmlidXRlc1xuICogR2VuZXJpYyB2ZXJ0ZXggYXR0cmlidXRlcyBhcmUgY29uc3RhbnQgZm9yIGFsbCB2ZXJ0aWNlc1xuICogVXAgdG8gNCB2YWx1ZXMgZGVwZW5kaW5nIG9uIGF0dHJpYnV0ZSBzaXplXG4gKlxuICogQHBhcmFtIHtHTHVpbnR9IGxvY2F0aW9uIC0gb3JkaW5hbCBudW1iZXIgb2YgdGhlIGF0dHJpYnV0ZVxuICogQHBhcmFtIHtHTHVpbnR9IGRpdmlzb3IgLSBpbnN0YW5jZXMgdGhhdCBwYXNzIGJldHdlZW4gdXBkYXRlcyBvZiBhdHRyaWJ1dGVcbiAqL1xuLyogZXNsaW50LWRpc2FibGUgbWF4LXBhcmFtcyAqL1xuZXhwb3J0IGZ1bmN0aW9uIHNldEdlbmVyaWNWYWx1ZXMoZ2wsIGxvY2F0aW9uLCB2MCwgdjEsIHYyLCB2Mykge1xuICBsb2cud2FybigwLCAnVmVydGV4QXR0cmlidXRlcy5zZXRHZW5lcmljVmFsdWVzIGlzIG5vdCB3ZWxsIHRlc3RlZCcpO1xuICBzd2l0Y2ggKGFyZ3VtZW50cy5sZW5ndGggLSAxKSB7XG4gIGNhc2UgMTogZ2wudmVydGV4QXR0cmliMWYobG9jYXRpb24sIHYwKTsgYnJlYWs7XG4gIGNhc2UgMjogZ2wudmVydGV4QXR0cmliMmYobG9jYXRpb24sIHYwLCB2MSk7IGJyZWFrO1xuICBjYXNlIDM6IGdsLnZlcnRleEF0dHJpYjNmKGxvY2F0aW9uLCB2MCwgdjEsIHYyKTsgYnJlYWs7XG4gIGNhc2UgNDogZ2wudmVydGV4QXR0cmliNGYobG9jYXRpb24sIHYwLCB2MSwgdjIsIHYzKTsgYnJlYWs7XG4gIGRlZmF1bHQ6XG4gICAgdGhyb3cgbmV3IEVycm9yKCd2ZXJ0ZXggYXR0cmlidXRlIHNpemUgbXVzdCBiZSBiZXR3ZWVuIDEgYW5kIDQnKTtcbiAgfVxuXG4gIC8vIGFzc2VydChnbCBpbnN0YW5jZW9mIFdlYkdMMlJlbmRlcmluZ0NvbnRleHQsICdXZWJHTDIgcmVxdWlyZWQnKTtcbiAgLy8gTG9va3MgbGlrZSB0aGVzZSB3aWxsIGNoZWNrIGhvdyBtYW55IGFyZ3VtZW50cyB3ZXJlIHN1cHBsaWVkP1xuICAvLyBnbC52ZXJ0ZXhBdHRyaWJJNGkobG9jYXRpb24sIHYwLCB2MSwgdjIsIHYzKTtcbiAgLy8gZ2wudmVydGV4QXR0cmliSTR1aShsb2NhdGlvbiwgdjAsIHYxLCB2MiwgdjMpO1xufVxuIl19