'use strict';

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.GL_BUFFER_USAGE = exports.GL_TARGETS = exports.GL_DRAW_MODES = exports.GL_INDEX_TYPES = undefined;
exports.isWebGLRenderingContext = isWebGLRenderingContext;
exports.isWebGL2RenderingContext = isWebGL2RenderingContext;
exports.assertWebGLRenderingContext = assertWebGLRenderingContext;
exports.assertWebGL2RenderingContext = assertWebGL2RenderingContext;
exports.glKey = glKey;
exports.isIndexType = isIndexType;
exports.assertIndexType = assertIndexType;
exports.isDrawMode = isDrawMode;
exports.assertDrawMode = assertDrawMode;
exports.glTypeFromArray = glTypeFromArray;
exports.assertArrayTypeMatch = assertArrayTypeMatch;
exports.glArrayFromType = glArrayFromType;

var _webglTypes = require('./webgl-types');

var _assert = require('assert');

var _assert2 = _interopRequireDefault(_assert);

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

// Helper definitions for validation of webgl parameters
/* eslint-disable no-inline-comments, max-len */
var ERR_CONTEXT = 'Invalid WebGLRenderingContext';
var ERR_WEBGL2 = 'Requires WebGL2';

function isWebGLRenderingContext(gl) {
  return gl instanceof _webglTypes.WebGLRenderingContext || gl && gl.ARRAY_BUFFER === 0x8892;
}

function isWebGL2RenderingContext(gl) {
  return gl instanceof _webglTypes.WebGL2RenderingContext || gl && gl.TEXTURE_BINDING_3D === 0x806A;
}

function assertWebGLRenderingContext(gl) {
  // Need to handle debug context
  (0, _assert2.default)(isWebGLRenderingContext(gl), ERR_CONTEXT);
}

function assertWebGL2RenderingContext(gl) {
  // Need to handle debug context
  (0, _assert2.default)(isWebGL2RenderingContext(gl), ERR_WEBGL2);
}

// INDEX TYPES

// TODO - move to glGet
function glKey(value) {
  for (var key in _webglTypes.WebGL) {
    if (_webglTypes.WebGL[key] === value) {
      return key;
    }
  }
  return String(value);
}

// For drawElements, size of indices
var GL_INDEX_TYPES = exports.GL_INDEX_TYPES = ['UNSIGNED_BYTE', 'UNSIGNED_SHORT', 'UNSIGNED_INT'].map(function (constant) {
  return _webglTypes.WebGL[constant];
});

function isIndexType(type) {
  return GL_INDEX_TYPES.indexOf(type) !== -1;
}

function assertIndexType(glType, source) {
  (0, _assert2.default)(isIndexType(glType), 'Bad index type gl.' + glKey(glType) + ' ' + source);
}

// DRAW MODES

var GL_DRAW_MODES = exports.GL_DRAW_MODES = ['POINTS', 'LINE_STRIP', 'LINE_LOOP', 'LINES', 'TRIANGLE_STRIP', 'TRIANGLE_FAN', 'TRIANGLES'].map(function (constant) {
  return _webglTypes.WebGL[constant];
});

function isDrawMode(glMode) {
  return GL_DRAW_MODES.indexOf(glMode) !== -1;
}

function assertDrawMode(glType, source) {
  (0, _assert2.default)(isDrawMode(glType), 'Bad draw mode gl.' + glKey(glType) + ' ' + source);
}

// TARGET TYPES

var GL_TARGETS = exports.GL_TARGETS = ['ARRAY_BUFFER', // vertex attributes (e.g. vertex/texture coords or color)
'ELEMENT_ARRAY_BUFFER', // Buffer used for element indices.
// For WebGL 2 contexts
'COPY_READ_BUFFER', // Buffer for copying from one buffer object to another
'COPY_WRITE_BUFFER', // Buffer for copying from one buffer object to another
'TRANSFORM_FEEDBACK_BUFFER', // Buffer for transform feedback operations
'UNIFORM_BUFFER', // Buffer used for storing uniform blocks
'PIXEL_PACK_BUFFER', // Buffer used for pixel transfer operations
'PIXEL_UNPACK_BUFFER' // Buffer used for pixel transfer operations
].map(function (constant) {
  return _webglTypes.WebGL[constant];
}).filter(function (constant) {
  return constant;
});

// USAGE TYPES

var GL_BUFFER_USAGE = exports.GL_BUFFER_USAGE = ['STATIC_DRAW', // Buffer used often and not change often. Contents are written to the buffer, but not read.
'DYNAMIC_DRAW', // Buffer used often and change often. Contents are written to the buffer, but not read.
'STREAM_DRAW', // Buffer not used often. Contents are written to the buffer, but not read.
// For WebGL 2 contexts
'STATIC_READ', // Buffer used often and not change often. Contents are read from the buffer, but not written.
'DYNAMIC_READ', // Buffer used often and change often. Contents are read from the buffer, but not written.
'STREAM_READ', // Contents of the buffer are likely to not be used often. Contents are read from the buffer, but not written.
'STATIC_COPY', // Buffer used often and not change often. Contents are neither written or read by the user.
'DYNAMIC_COPY', // Buffer used often and change often. Contents are neither written or read by the user.
'STREAM_COPY' // Buffer used often and not change often. Contents are neither written or read by the user.
].map(function (constant) {
  return _webglTypes.WebGL[constant];
}).filter(function (constant) {
  return constant;
});

function glTypeFromArray(array) {
  // Sorted in some order of likelihood to reduce amount of comparisons
  if (array instanceof Float32Array) {
    return _webglTypes.WebGL.FLOAT;
  } else if (array instanceof Uint16Array) {
    return _webglTypes.WebGL.UNSIGNED_SHORT;
  } else if (array instanceof Uint32Array) {
    return _webglTypes.WebGL.UNSIGNED_INT;
  } else if (array instanceof Uint8Array) {
    return _webglTypes.WebGL.UNSIGNED_BYTE;
  } else if (array instanceof Uint8ClampedArray) {
    return _webglTypes.WebGL.UNSIGNED_BYTE;
  } else if (array instanceof Int8Array) {
    return _webglTypes.WebGL.BYTE;
  } else if (array instanceof Int16Array) {
    return _webglTypes.WebGL.SHORT;
  } else if (array instanceof Int32Array) {
    return _webglTypes.WebGL.INT;
  }
  throw new Error('Failed to deduce WebGL type from array');
}

function assertArrayTypeMatch(array, type, source) {
  (0, _assert2.default)(type === glTypeFromArray(array), (array.constructor.name || 'Array') + ' ' + ('does not match element type gl.' + glKey(type) + ' ' + source));
}

/* eslint-disable complexity */
function glArrayFromType(glType) {
  var clamped = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : false;

  // Sorted in some order of likelihood to reduce amount of comparisons
  switch (glType) {
    case _webglTypes.WebGL.FLOAT:
      return Float32Array;
    case _webglTypes.WebGL.UNSIGNED_SHORT:
    case _webglTypes.WebGL.UNSIGNED_SHORT_5_6_5:
    case _webglTypes.WebGL.UNSIGNED_SHORT_4_4_4_4:
    case _webglTypes.WebGL.UNSIGNED_SHORT_5_5_5_1:
      return Uint16Array;
    case _webglTypes.WebGL.UNSIGNED_INT:
      // case WebGL.UNSIGNED_INT_2_10_10_10_REV:
      // case WebGL.UNSIGNED_INT_10F_11F_11F_REV:
      // case WebGL.UNSIGNED_INT_5_9_9_9_REV:
      // case WebGL.UNSIGNED_INT_24_8:
      return Uint32Array;
    case _webglTypes.WebGL.UNSIGNED_BYTE:
      return clamped ? Uint8ClampedArray : Uint8Array;
    case _webglTypes.WebGL.BYTE:
      return Int8Array;
    case _webglTypes.WebGL.SHORT:
      return Int16Array;
    case _webglTypes.WebGL.INT:
      return Int32Array;

    default:
      throw new Error('Failed to deduce type from array');
  }
}
/* eslint-enable complexity */
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbIi4uLy4uL3NyYy93ZWJnbC93ZWJnbC1jaGVja3MuanMiXSwibmFtZXMiOlsiaXNXZWJHTFJlbmRlcmluZ0NvbnRleHQiLCJpc1dlYkdMMlJlbmRlcmluZ0NvbnRleHQiLCJhc3NlcnRXZWJHTFJlbmRlcmluZ0NvbnRleHQiLCJhc3NlcnRXZWJHTDJSZW5kZXJpbmdDb250ZXh0IiwiZ2xLZXkiLCJpc0luZGV4VHlwZSIsImFzc2VydEluZGV4VHlwZSIsImlzRHJhd01vZGUiLCJhc3NlcnREcmF3TW9kZSIsImdsVHlwZUZyb21BcnJheSIsImFzc2VydEFycmF5VHlwZU1hdGNoIiwiZ2xBcnJheUZyb21UeXBlIiwiRVJSX0NPTlRFWFQiLCJFUlJfV0VCR0wyIiwiZ2wiLCJBUlJBWV9CVUZGRVIiLCJURVhUVVJFX0JJTkRJTkdfM0QiLCJ2YWx1ZSIsImtleSIsIlN0cmluZyIsIkdMX0lOREVYX1RZUEVTIiwibWFwIiwiY29uc3RhbnQiLCJ0eXBlIiwiaW5kZXhPZiIsImdsVHlwZSIsInNvdXJjZSIsIkdMX0RSQVdfTU9ERVMiLCJnbE1vZGUiLCJHTF9UQVJHRVRTIiwiZmlsdGVyIiwiR0xfQlVGRkVSX1VTQUdFIiwiYXJyYXkiLCJGbG9hdDMyQXJyYXkiLCJGTE9BVCIsIlVpbnQxNkFycmF5IiwiVU5TSUdORURfU0hPUlQiLCJVaW50MzJBcnJheSIsIlVOU0lHTkVEX0lOVCIsIlVpbnQ4QXJyYXkiLCJVTlNJR05FRF9CWVRFIiwiVWludDhDbGFtcGVkQXJyYXkiLCJJbnQ4QXJyYXkiLCJCWVRFIiwiSW50MTZBcnJheSIsIlNIT1JUIiwiSW50MzJBcnJheSIsIklOVCIsIkVycm9yIiwiY29uc3RydWN0b3IiLCJuYW1lIiwiY2xhbXBlZCIsIlVOU0lHTkVEX1NIT1JUXzVfNl81IiwiVU5TSUdORURfU0hPUlRfNF80XzRfNCIsIlVOU0lHTkVEX1NIT1JUXzVfNV81XzEiXSwibWFwcGluZ3MiOiI7Ozs7OztRQVNnQkEsdUIsR0FBQUEsdUI7UUFLQUMsd0IsR0FBQUEsd0I7UUFLQUMsMkIsR0FBQUEsMkI7UUFLQUMsNEIsR0FBQUEsNEI7UUFRQUMsSyxHQUFBQSxLO1FBZUFDLFcsR0FBQUEsVztRQUlBQyxlLEdBQUFBLGU7UUFZQUMsVSxHQUFBQSxVO1FBSUFDLGMsR0FBQUEsYztRQW1DQUMsZSxHQUFBQSxlO1FBc0JBQyxvQixHQUFBQSxvQjtRQU9BQyxlLEdBQUFBLGU7O0FBakloQjs7QUFFQTs7Ozs7O0FBSkE7QUFDQTtBQUtBLElBQU1DLGNBQWMsK0JBQXBCO0FBQ0EsSUFBTUMsYUFBYSxpQkFBbkI7O0FBRU8sU0FBU2IsdUJBQVQsQ0FBaUNjLEVBQWpDLEVBQXFDO0FBQzFDLFNBQU9BLG1EQUNKQSxNQUFNQSxHQUFHQyxZQUFILEtBQW9CLE1BRDdCO0FBRUQ7O0FBRU0sU0FBU2Qsd0JBQVQsQ0FBa0NhLEVBQWxDLEVBQXNDO0FBQzNDLFNBQU9BLG9EQUNKQSxNQUFNQSxHQUFHRSxrQkFBSCxLQUEwQixNQURuQztBQUVEOztBQUVNLFNBQVNkLDJCQUFULENBQXFDWSxFQUFyQyxFQUF5QztBQUM5QztBQUNBLHdCQUFPZCx3QkFBd0JjLEVBQXhCLENBQVAsRUFBb0NGLFdBQXBDO0FBQ0Q7O0FBRU0sU0FBU1QsNEJBQVQsQ0FBc0NXLEVBQXRDLEVBQTBDO0FBQy9DO0FBQ0Esd0JBQU9iLHlCQUF5QmEsRUFBekIsQ0FBUCxFQUFxQ0QsVUFBckM7QUFDRDs7QUFFRDs7QUFFQTtBQUNPLFNBQVNULEtBQVQsQ0FBZWEsS0FBZixFQUFzQjtBQUMzQixPQUFLLElBQU1DLEdBQVgsdUJBQXlCO0FBQ3ZCLFFBQUksa0JBQU1BLEdBQU4sTUFBZUQsS0FBbkIsRUFBMEI7QUFDeEIsYUFBT0MsR0FBUDtBQUNEO0FBQ0Y7QUFDRCxTQUFPQyxPQUFPRixLQUFQLENBQVA7QUFDRDs7QUFFRDtBQUNPLElBQU1HLDBDQUFpQixDQUM1QixlQUQ0QixFQUNYLGdCQURXLEVBQ08sY0FEUCxFQUc3QkMsR0FINkIsQ0FHekI7QUFBQSxTQUFZLGtCQUFNQyxRQUFOLENBQVo7QUFBQSxDQUh5QixDQUF2Qjs7QUFLQSxTQUFTakIsV0FBVCxDQUFxQmtCLElBQXJCLEVBQTJCO0FBQ2hDLFNBQU9ILGVBQWVJLE9BQWYsQ0FBdUJELElBQXZCLE1BQWlDLENBQUMsQ0FBekM7QUFDRDs7QUFFTSxTQUFTakIsZUFBVCxDQUF5Qm1CLE1BQXpCLEVBQWlDQyxNQUFqQyxFQUF5QztBQUM5Qyx3QkFBT3JCLFlBQVlvQixNQUFaLENBQVAseUJBQWlEckIsTUFBTXFCLE1BQU4sQ0FBakQsU0FBa0VDLE1BQWxFO0FBQ0Q7O0FBRUQ7O0FBRU8sSUFBTUMsd0NBQWdCLENBQzNCLFFBRDJCLEVBQ2pCLFlBRGlCLEVBQ0gsV0FERyxFQUNVLE9BRFYsRUFFM0IsZ0JBRjJCLEVBRVQsY0FGUyxFQUVPLFdBRlAsRUFJNUJOLEdBSjRCLENBSXhCO0FBQUEsU0FBWSxrQkFBTUMsUUFBTixDQUFaO0FBQUEsQ0FKd0IsQ0FBdEI7O0FBTUEsU0FBU2YsVUFBVCxDQUFvQnFCLE1BQXBCLEVBQTRCO0FBQ2pDLFNBQU9ELGNBQWNILE9BQWQsQ0FBc0JJLE1BQXRCLE1BQWtDLENBQUMsQ0FBMUM7QUFDRDs7QUFFTSxTQUFTcEIsY0FBVCxDQUF3QmlCLE1BQXhCLEVBQWdDQyxNQUFoQyxFQUF3QztBQUM3Qyx3QkFBT25CLFdBQVdrQixNQUFYLENBQVAsd0JBQStDckIsTUFBTXFCLE1BQU4sQ0FBL0MsU0FBZ0VDLE1BQWhFO0FBQ0Q7O0FBRUQ7O0FBRU8sSUFBTUcsa0NBQWEsQ0FDeEIsY0FEd0IsRUFDUjtBQUNoQixzQkFGd0IsRUFFQTtBQUN4QjtBQUNBLGtCQUp3QixFQUlKO0FBQ3BCLG1CQUx3QixFQUtIO0FBQ3JCLDJCQU53QixFQU1LO0FBQzdCLGdCQVB3QixFQU9OO0FBQ2xCLG1CQVJ3QixFQVFIO0FBQ3JCLHFCQVR3QixDQVNGO0FBVEUsRUFXekJSLEdBWHlCLENBV3JCO0FBQUEsU0FBWSxrQkFBTUMsUUFBTixDQUFaO0FBQUEsQ0FYcUIsRUFXUVEsTUFYUixDQVdlO0FBQUEsU0FBWVIsUUFBWjtBQUFBLENBWGYsQ0FBbkI7O0FBYVA7O0FBRU8sSUFBTVMsNENBQWtCLENBQzdCLGFBRDZCLEVBQ2Q7QUFDZixjQUY2QixFQUViO0FBQ2hCLGFBSDZCLEVBR2Q7QUFDZjtBQUNBLGFBTDZCLEVBS2Q7QUFDZixjQU42QixFQU1iO0FBQ2hCLGFBUDZCLEVBT2Q7QUFDZixhQVI2QixFQVFkO0FBQ2YsY0FUNkIsRUFTYjtBQUNoQixhQVY2QixDQVVmO0FBVmUsRUFZOUJWLEdBWjhCLENBWTFCO0FBQUEsU0FBWSxrQkFBTUMsUUFBTixDQUFaO0FBQUEsQ0FaMEIsRUFZR1EsTUFaSCxDQVlVO0FBQUEsU0FBWVIsUUFBWjtBQUFBLENBWlYsQ0FBeEI7O0FBY0EsU0FBU2IsZUFBVCxDQUF5QnVCLEtBQXpCLEVBQWdDO0FBQ3JDO0FBQ0EsTUFBSUEsaUJBQWlCQyxZQUFyQixFQUFtQztBQUNqQyxXQUFPLGtCQUFNQyxLQUFiO0FBQ0QsR0FGRCxNQUVPLElBQUlGLGlCQUFpQkcsV0FBckIsRUFBa0M7QUFDdkMsV0FBTyxrQkFBTUMsY0FBYjtBQUNELEdBRk0sTUFFQSxJQUFJSixpQkFBaUJLLFdBQXJCLEVBQWtDO0FBQ3ZDLFdBQU8sa0JBQU1DLFlBQWI7QUFDRCxHQUZNLE1BRUEsSUFBSU4saUJBQWlCTyxVQUFyQixFQUFpQztBQUN0QyxXQUFPLGtCQUFNQyxhQUFiO0FBQ0QsR0FGTSxNQUVBLElBQUlSLGlCQUFpQlMsaUJBQXJCLEVBQXdDO0FBQzdDLFdBQU8sa0JBQU1ELGFBQWI7QUFDRCxHQUZNLE1BRUEsSUFBSVIsaUJBQWlCVSxTQUFyQixFQUFnQztBQUNyQyxXQUFPLGtCQUFNQyxJQUFiO0FBQ0QsR0FGTSxNQUVBLElBQUlYLGlCQUFpQlksVUFBckIsRUFBaUM7QUFDdEMsV0FBTyxrQkFBTUMsS0FBYjtBQUNELEdBRk0sTUFFQSxJQUFJYixpQkFBaUJjLFVBQXJCLEVBQWlDO0FBQ3RDLFdBQU8sa0JBQU1DLEdBQWI7QUFDRDtBQUNELFFBQU0sSUFBSUMsS0FBSixDQUFVLHdDQUFWLENBQU47QUFDRDs7QUFFTSxTQUFTdEMsb0JBQVQsQ0FBOEJzQixLQUE5QixFQUFxQ1QsSUFBckMsRUFBMkNHLE1BQTNDLEVBQW1EO0FBQ3hELHdCQUFPSCxTQUFTZCxnQkFBZ0J1QixLQUFoQixDQUFoQixFQUNFLENBQUdBLE1BQU1pQixXQUFOLENBQWtCQyxJQUFsQixJQUEwQixPQUE3QiwrQ0FDa0M5QyxNQUFNbUIsSUFBTixDQURsQyxTQUNpREcsTUFEakQsQ0FERjtBQUdEOztBQUVEO0FBQ08sU0FBU2YsZUFBVCxDQUF5QmMsTUFBekIsRUFBa0Q7QUFBQSxNQUFqQjBCLE9BQWlCLHVFQUFQLEtBQU87O0FBQ3ZEO0FBQ0EsVUFBUTFCLE1BQVI7QUFDQSxTQUFLLGtCQUFNUyxLQUFYO0FBQ0UsYUFBT0QsWUFBUDtBQUNGLFNBQUssa0JBQU1HLGNBQVg7QUFDQSxTQUFLLGtCQUFNZ0Isb0JBQVg7QUFDQSxTQUFLLGtCQUFNQyxzQkFBWDtBQUNBLFNBQUssa0JBQU1DLHNCQUFYO0FBQ0UsYUFBT25CLFdBQVA7QUFDRixTQUFLLGtCQUFNRyxZQUFYO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDRSxhQUFPRCxXQUFQO0FBQ0YsU0FBSyxrQkFBTUcsYUFBWDtBQUNFLGFBQU9XLFVBQVVWLGlCQUFWLEdBQThCRixVQUFyQztBQUNGLFNBQUssa0JBQU1JLElBQVg7QUFDRSxhQUFPRCxTQUFQO0FBQ0YsU0FBSyxrQkFBTUcsS0FBWDtBQUNFLGFBQU9ELFVBQVA7QUFDRixTQUFLLGtCQUFNRyxHQUFYO0FBQ0UsYUFBT0QsVUFBUDs7QUFFRjtBQUNFLFlBQU0sSUFBSUUsS0FBSixDQUFVLGtDQUFWLENBQU47QUF4QkY7QUEwQkQ7QUFDRCIsImZpbGUiOiJ3ZWJnbC1jaGVja3MuanMiLCJzb3VyY2VzQ29udGVudCI6WyIvLyBIZWxwZXIgZGVmaW5pdGlvbnMgZm9yIHZhbGlkYXRpb24gb2Ygd2ViZ2wgcGFyYW1ldGVyc1xuLyogZXNsaW50LWRpc2FibGUgbm8taW5saW5lLWNvbW1lbnRzLCBtYXgtbGVuICovXG5pbXBvcnQge1dlYkdMLCBXZWJHTFJlbmRlcmluZ0NvbnRleHQsIFdlYkdMMlJlbmRlcmluZ0NvbnRleHR9XG4gIGZyb20gJy4vd2ViZ2wtdHlwZXMnO1xuaW1wb3J0IGFzc2VydCBmcm9tICdhc3NlcnQnO1xuXG5jb25zdCBFUlJfQ09OVEVYVCA9ICdJbnZhbGlkIFdlYkdMUmVuZGVyaW5nQ29udGV4dCc7XG5jb25zdCBFUlJfV0VCR0wyID0gJ1JlcXVpcmVzIFdlYkdMMic7XG5cbmV4cG9ydCBmdW5jdGlvbiBpc1dlYkdMUmVuZGVyaW5nQ29udGV4dChnbCkge1xuICByZXR1cm4gZ2wgaW5zdGFuY2VvZiBXZWJHTFJlbmRlcmluZ0NvbnRleHQgfHxcbiAgICAoZ2wgJiYgZ2wuQVJSQVlfQlVGRkVSID09PSAweDg4OTIpO1xufVxuXG5leHBvcnQgZnVuY3Rpb24gaXNXZWJHTDJSZW5kZXJpbmdDb250ZXh0KGdsKSB7XG4gIHJldHVybiBnbCBpbnN0YW5jZW9mIFdlYkdMMlJlbmRlcmluZ0NvbnRleHQgfHxcbiAgICAoZ2wgJiYgZ2wuVEVYVFVSRV9CSU5ESU5HXzNEID09PSAweDgwNkEpO1xufVxuXG5leHBvcnQgZnVuY3Rpb24gYXNzZXJ0V2ViR0xSZW5kZXJpbmdDb250ZXh0KGdsKSB7XG4gIC8vIE5lZWQgdG8gaGFuZGxlIGRlYnVnIGNvbnRleHRcbiAgYXNzZXJ0KGlzV2ViR0xSZW5kZXJpbmdDb250ZXh0KGdsKSwgRVJSX0NPTlRFWFQpO1xufVxuXG5leHBvcnQgZnVuY3Rpb24gYXNzZXJ0V2ViR0wyUmVuZGVyaW5nQ29udGV4dChnbCkge1xuICAvLyBOZWVkIHRvIGhhbmRsZSBkZWJ1ZyBjb250ZXh0XG4gIGFzc2VydChpc1dlYkdMMlJlbmRlcmluZ0NvbnRleHQoZ2wpLCBFUlJfV0VCR0wyKTtcbn1cblxuLy8gSU5ERVggVFlQRVNcblxuLy8gVE9ETyAtIG1vdmUgdG8gZ2xHZXRcbmV4cG9ydCBmdW5jdGlvbiBnbEtleSh2YWx1ZSkge1xuICBmb3IgKGNvbnN0IGtleSBpbiBXZWJHTCkge1xuICAgIGlmIChXZWJHTFtrZXldID09PSB2YWx1ZSkge1xuICAgICAgcmV0dXJuIGtleTtcbiAgICB9XG4gIH1cbiAgcmV0dXJuIFN0cmluZyh2YWx1ZSk7XG59XG5cbi8vIEZvciBkcmF3RWxlbWVudHMsIHNpemUgb2YgaW5kaWNlc1xuZXhwb3J0IGNvbnN0IEdMX0lOREVYX1RZUEVTID0gW1xuICAnVU5TSUdORURfQllURScsICdVTlNJR05FRF9TSE9SVCcsICdVTlNJR05FRF9JTlQnXG5dXG4ubWFwKGNvbnN0YW50ID0+IFdlYkdMW2NvbnN0YW50XSk7XG5cbmV4cG9ydCBmdW5jdGlvbiBpc0luZGV4VHlwZSh0eXBlKSB7XG4gIHJldHVybiBHTF9JTkRFWF9UWVBFUy5pbmRleE9mKHR5cGUpICE9PSAtMTtcbn1cblxuZXhwb3J0IGZ1bmN0aW9uIGFzc2VydEluZGV4VHlwZShnbFR5cGUsIHNvdXJjZSkge1xuICBhc3NlcnQoaXNJbmRleFR5cGUoZ2xUeXBlKSwgYEJhZCBpbmRleCB0eXBlIGdsLiR7Z2xLZXkoZ2xUeXBlKX0gJHtzb3VyY2V9YCk7XG59XG5cbi8vIERSQVcgTU9ERVNcblxuZXhwb3J0IGNvbnN0IEdMX0RSQVdfTU9ERVMgPSBbXG4gICdQT0lOVFMnLCAnTElORV9TVFJJUCcsICdMSU5FX0xPT1AnLCAnTElORVMnLFxuICAnVFJJQU5HTEVfU1RSSVAnLCAnVFJJQU5HTEVfRkFOJywgJ1RSSUFOR0xFUydcbl1cbi5tYXAoY29uc3RhbnQgPT4gV2ViR0xbY29uc3RhbnRdKTtcblxuZXhwb3J0IGZ1bmN0aW9uIGlzRHJhd01vZGUoZ2xNb2RlKSB7XG4gIHJldHVybiBHTF9EUkFXX01PREVTLmluZGV4T2YoZ2xNb2RlKSAhPT0gLTE7XG59XG5cbmV4cG9ydCBmdW5jdGlvbiBhc3NlcnREcmF3TW9kZShnbFR5cGUsIHNvdXJjZSkge1xuICBhc3NlcnQoaXNEcmF3TW9kZShnbFR5cGUpLCBgQmFkIGRyYXcgbW9kZSBnbC4ke2dsS2V5KGdsVHlwZSl9ICR7c291cmNlfWApO1xufVxuXG4vLyBUQVJHRVQgVFlQRVNcblxuZXhwb3J0IGNvbnN0IEdMX1RBUkdFVFMgPSBbXG4gICdBUlJBWV9CVUZGRVInLCAvLyB2ZXJ0ZXggYXR0cmlidXRlcyAoZS5nLiB2ZXJ0ZXgvdGV4dHVyZSBjb29yZHMgb3IgY29sb3IpXG4gICdFTEVNRU5UX0FSUkFZX0JVRkZFUicsIC8vIEJ1ZmZlciB1c2VkIGZvciBlbGVtZW50IGluZGljZXMuXG4gIC8vIEZvciBXZWJHTCAyIGNvbnRleHRzXG4gICdDT1BZX1JFQURfQlVGRkVSJywgLy8gQnVmZmVyIGZvciBjb3B5aW5nIGZyb20gb25lIGJ1ZmZlciBvYmplY3QgdG8gYW5vdGhlclxuICAnQ09QWV9XUklURV9CVUZGRVInLCAvLyBCdWZmZXIgZm9yIGNvcHlpbmcgZnJvbSBvbmUgYnVmZmVyIG9iamVjdCB0byBhbm90aGVyXG4gICdUUkFOU0ZPUk1fRkVFREJBQ0tfQlVGRkVSJywgLy8gQnVmZmVyIGZvciB0cmFuc2Zvcm0gZmVlZGJhY2sgb3BlcmF0aW9uc1xuICAnVU5JRk9STV9CVUZGRVInLCAvLyBCdWZmZXIgdXNlZCBmb3Igc3RvcmluZyB1bmlmb3JtIGJsb2Nrc1xuICAnUElYRUxfUEFDS19CVUZGRVInLCAvLyBCdWZmZXIgdXNlZCBmb3IgcGl4ZWwgdHJhbnNmZXIgb3BlcmF0aW9uc1xuICAnUElYRUxfVU5QQUNLX0JVRkZFUicgLy8gQnVmZmVyIHVzZWQgZm9yIHBpeGVsIHRyYW5zZmVyIG9wZXJhdGlvbnNcbl1cbi5tYXAoY29uc3RhbnQgPT4gV2ViR0xbY29uc3RhbnRdKS5maWx0ZXIoY29uc3RhbnQgPT4gY29uc3RhbnQpO1xuXG4vLyBVU0FHRSBUWVBFU1xuXG5leHBvcnQgY29uc3QgR0xfQlVGRkVSX1VTQUdFID0gW1xuICAnU1RBVElDX0RSQVcnLCAvLyBCdWZmZXIgdXNlZCBvZnRlbiBhbmQgbm90IGNoYW5nZSBvZnRlbi4gQ29udGVudHMgYXJlIHdyaXR0ZW4gdG8gdGhlIGJ1ZmZlciwgYnV0IG5vdCByZWFkLlxuICAnRFlOQU1JQ19EUkFXJywgLy8gQnVmZmVyIHVzZWQgb2Z0ZW4gYW5kIGNoYW5nZSBvZnRlbi4gQ29udGVudHMgYXJlIHdyaXR0ZW4gdG8gdGhlIGJ1ZmZlciwgYnV0IG5vdCByZWFkLlxuICAnU1RSRUFNX0RSQVcnLCAvLyBCdWZmZXIgbm90IHVzZWQgb2Z0ZW4uIENvbnRlbnRzIGFyZSB3cml0dGVuIHRvIHRoZSBidWZmZXIsIGJ1dCBub3QgcmVhZC5cbiAgLy8gRm9yIFdlYkdMIDIgY29udGV4dHNcbiAgJ1NUQVRJQ19SRUFEJywgLy8gQnVmZmVyIHVzZWQgb2Z0ZW4gYW5kIG5vdCBjaGFuZ2Ugb2Z0ZW4uIENvbnRlbnRzIGFyZSByZWFkIGZyb20gdGhlIGJ1ZmZlciwgYnV0IG5vdCB3cml0dGVuLlxuICAnRFlOQU1JQ19SRUFEJywgLy8gQnVmZmVyIHVzZWQgb2Z0ZW4gYW5kIGNoYW5nZSBvZnRlbi4gQ29udGVudHMgYXJlIHJlYWQgZnJvbSB0aGUgYnVmZmVyLCBidXQgbm90IHdyaXR0ZW4uXG4gICdTVFJFQU1fUkVBRCcsIC8vIENvbnRlbnRzIG9mIHRoZSBidWZmZXIgYXJlIGxpa2VseSB0byBub3QgYmUgdXNlZCBvZnRlbi4gQ29udGVudHMgYXJlIHJlYWQgZnJvbSB0aGUgYnVmZmVyLCBidXQgbm90IHdyaXR0ZW4uXG4gICdTVEFUSUNfQ09QWScsIC8vIEJ1ZmZlciB1c2VkIG9mdGVuIGFuZCBub3QgY2hhbmdlIG9mdGVuLiBDb250ZW50cyBhcmUgbmVpdGhlciB3cml0dGVuIG9yIHJlYWQgYnkgdGhlIHVzZXIuXG4gICdEWU5BTUlDX0NPUFknLCAvLyBCdWZmZXIgdXNlZCBvZnRlbiBhbmQgY2hhbmdlIG9mdGVuLiBDb250ZW50cyBhcmUgbmVpdGhlciB3cml0dGVuIG9yIHJlYWQgYnkgdGhlIHVzZXIuXG4gICdTVFJFQU1fQ09QWScgLy8gQnVmZmVyIHVzZWQgb2Z0ZW4gYW5kIG5vdCBjaGFuZ2Ugb2Z0ZW4uIENvbnRlbnRzIGFyZSBuZWl0aGVyIHdyaXR0ZW4gb3IgcmVhZCBieSB0aGUgdXNlci5cbl1cbi5tYXAoY29uc3RhbnQgPT4gV2ViR0xbY29uc3RhbnRdKS5maWx0ZXIoY29uc3RhbnQgPT4gY29uc3RhbnQpO1xuXG5leHBvcnQgZnVuY3Rpb24gZ2xUeXBlRnJvbUFycmF5KGFycmF5KSB7XG4gIC8vIFNvcnRlZCBpbiBzb21lIG9yZGVyIG9mIGxpa2VsaWhvb2QgdG8gcmVkdWNlIGFtb3VudCBvZiBjb21wYXJpc29uc1xuICBpZiAoYXJyYXkgaW5zdGFuY2VvZiBGbG9hdDMyQXJyYXkpIHtcbiAgICByZXR1cm4gV2ViR0wuRkxPQVQ7XG4gIH0gZWxzZSBpZiAoYXJyYXkgaW5zdGFuY2VvZiBVaW50MTZBcnJheSkge1xuICAgIHJldHVybiBXZWJHTC5VTlNJR05FRF9TSE9SVDtcbiAgfSBlbHNlIGlmIChhcnJheSBpbnN0YW5jZW9mIFVpbnQzMkFycmF5KSB7XG4gICAgcmV0dXJuIFdlYkdMLlVOU0lHTkVEX0lOVDtcbiAgfSBlbHNlIGlmIChhcnJheSBpbnN0YW5jZW9mIFVpbnQ4QXJyYXkpIHtcbiAgICByZXR1cm4gV2ViR0wuVU5TSUdORURfQllURTtcbiAgfSBlbHNlIGlmIChhcnJheSBpbnN0YW5jZW9mIFVpbnQ4Q2xhbXBlZEFycmF5KSB7XG4gICAgcmV0dXJuIFdlYkdMLlVOU0lHTkVEX0JZVEU7XG4gIH0gZWxzZSBpZiAoYXJyYXkgaW5zdGFuY2VvZiBJbnQ4QXJyYXkpIHtcbiAgICByZXR1cm4gV2ViR0wuQllURTtcbiAgfSBlbHNlIGlmIChhcnJheSBpbnN0YW5jZW9mIEludDE2QXJyYXkpIHtcbiAgICByZXR1cm4gV2ViR0wuU0hPUlQ7XG4gIH0gZWxzZSBpZiAoYXJyYXkgaW5zdGFuY2VvZiBJbnQzMkFycmF5KSB7XG4gICAgcmV0dXJuIFdlYkdMLklOVDtcbiAgfVxuICB0aHJvdyBuZXcgRXJyb3IoJ0ZhaWxlZCB0byBkZWR1Y2UgV2ViR0wgdHlwZSBmcm9tIGFycmF5Jyk7XG59XG5cbmV4cG9ydCBmdW5jdGlvbiBhc3NlcnRBcnJheVR5cGVNYXRjaChhcnJheSwgdHlwZSwgc291cmNlKSB7XG4gIGFzc2VydCh0eXBlID09PSBnbFR5cGVGcm9tQXJyYXkoYXJyYXkpLFxuICAgIGAke2FycmF5LmNvbnN0cnVjdG9yLm5hbWUgfHwgJ0FycmF5J30gYCArXG4gICAgYGRvZXMgbm90IG1hdGNoIGVsZW1lbnQgdHlwZSBnbC4ke2dsS2V5KHR5cGUpfSAke3NvdXJjZX1gKTtcbn1cblxuLyogZXNsaW50LWRpc2FibGUgY29tcGxleGl0eSAqL1xuZXhwb3J0IGZ1bmN0aW9uIGdsQXJyYXlGcm9tVHlwZShnbFR5cGUsIGNsYW1wZWQgPSBmYWxzZSkge1xuICAvLyBTb3J0ZWQgaW4gc29tZSBvcmRlciBvZiBsaWtlbGlob29kIHRvIHJlZHVjZSBhbW91bnQgb2YgY29tcGFyaXNvbnNcbiAgc3dpdGNoIChnbFR5cGUpIHtcbiAgY2FzZSBXZWJHTC5GTE9BVDpcbiAgICByZXR1cm4gRmxvYXQzMkFycmF5O1xuICBjYXNlIFdlYkdMLlVOU0lHTkVEX1NIT1JUOlxuICBjYXNlIFdlYkdMLlVOU0lHTkVEX1NIT1JUXzVfNl81OlxuICBjYXNlIFdlYkdMLlVOU0lHTkVEX1NIT1JUXzRfNF80XzQ6XG4gIGNhc2UgV2ViR0wuVU5TSUdORURfU0hPUlRfNV81XzVfMTpcbiAgICByZXR1cm4gVWludDE2QXJyYXk7XG4gIGNhc2UgV2ViR0wuVU5TSUdORURfSU5UOlxuICAvLyBjYXNlIFdlYkdMLlVOU0lHTkVEX0lOVF8yXzEwXzEwXzEwX1JFVjpcbiAgLy8gY2FzZSBXZWJHTC5VTlNJR05FRF9JTlRfMTBGXzExRl8xMUZfUkVWOlxuICAvLyBjYXNlIFdlYkdMLlVOU0lHTkVEX0lOVF81XzlfOV85X1JFVjpcbiAgLy8gY2FzZSBXZWJHTC5VTlNJR05FRF9JTlRfMjRfODpcbiAgICByZXR1cm4gVWludDMyQXJyYXk7XG4gIGNhc2UgV2ViR0wuVU5TSUdORURfQllURTpcbiAgICByZXR1cm4gY2xhbXBlZCA/IFVpbnQ4Q2xhbXBlZEFycmF5IDogVWludDhBcnJheTtcbiAgY2FzZSBXZWJHTC5CWVRFOlxuICAgIHJldHVybiBJbnQ4QXJyYXk7XG4gIGNhc2UgV2ViR0wuU0hPUlQ6XG4gICAgcmV0dXJuIEludDE2QXJyYXk7XG4gIGNhc2UgV2ViR0wuSU5UOlxuICAgIHJldHVybiBJbnQzMkFycmF5O1xuXG4gIGRlZmF1bHQ6XG4gICAgdGhyb3cgbmV3IEVycm9yKCdGYWlsZWQgdG8gZGVkdWNlIHR5cGUgZnJvbSBhcnJheScpO1xuICB9XG59XG4vKiBlc2xpbnQtZW5hYmxlIGNvbXBsZXhpdHkgKi9cbiJdfQ==