/*
high performance StyleSheet for css-in-js systems

- uses multiple style tags behind the scenes for millions of rules
- uses `insertRule` for appending in production for *much* faster performance
- 'polyfills' on server side


// usage

import StyleSheet from 'glamor/lib/sheet'
let styleSheet = new StyleSheet()

styleSheet.inject()
- 'injects' the stylesheet into the page (or into memory if on server)

styleSheet.insert('#box { border: 1px solid red; }')
- appends a css rule into the stylesheet

styleSheet.flush()
- empties the stylesheet of all its contents


*/
import { isBrowser, isDev } from './utils';
var oldIE = (function () {
    if (isBrowser) {
        var div = document.createElement('div');
        div.innerHTML = '<!--[if lt IE 10]><i></i><![endif]-->';
        return div.getElementsByTagName('i').length === 1;
    }
})();
var ServerSheet = (function () {
    function ServerSheet() {
        this.cssRules = [];
    }
    ServerSheet.prototype.insertRule = function (rule, isImportRule) {
        if (isImportRule === void 0) { isImportRule = false; }
        if (isImportRule) {
            this.cssRules.unshift({ cssText: rule });
        }
        else {
            this.cssRules.push({ cssText: rule });
        }
    };
    ServerSheet.prototype.getCSSRules = function () {
        return this.cssRules;
    };
    ServerSheet.prototype.emptyCssRules = function () {
        this.cssRules = [];
    };
    return ServerSheet;
}());
var StyleSheet = (function () {
    function StyleSheet(speedy, maxRules) {
        if (speedy === void 0) { speedy = !isDev; }
        if (maxRules === void 0) { maxRules = (isBrowser && oldIE) ? 4000 : 65000; }
        this.tags = []; // all the <style> tags inside our dom
        this.injected = false; // determine if the <Style> tags are already injected inside the head of the do
        this.isSpeedy = speedy; // the big drawback here is that the css won't be editable in devtools
        this.sheet = undefined;
        this.tags = [];
        this.maxRules = maxRules;
        this.ruleCounter = 0;
    }
    StyleSheet.prototype.getSheet = function () {
        return sheetForTag(last(this.tags));
    };
    /**
     * create <style> tag and inject it in the dom if it's browser einvironment
     * otherwise it will create an array of cssRules within the StyleSheet object
     */
    StyleSheet.prototype.inject = function () {
        if (this.injected) {
            throw new Error('already injected stylesheet!');
        }
        if (isBrowser) {
            this.tags[0] = makeStyleTag();
        }
        else {
            // server side 'polyfill'. just enough behavior to be useful.
            this.sheet = new ServerSheet();
        }
        this.injected = true;
    };
    StyleSheet.prototype.speedy = function (speedy) {
        if (this.ruleCounter !== 0) {
            throw new Error("cannot change speedy mode after inserting any rule to sheet. Either call speedy(" + speedy + ") earlier in your app, or call flush() before speedy(" + speedy + ")");
        }
        this.isSpeedy = !!speedy;
    };
    /**
     * Insert a new css rule into the <style> tag when it's in the browser environment
     * @param rule
     */
    StyleSheet.prototype.insert = function (rule) {
        if (isBrowser) {
            // this is the ultrafast version, works across browsers
            if (this.isSpeedy && this.getSheet().insertRule) {
                this.browInsert(rule);
            }
            else {
                if (rule.indexOf('@import') !== -1) {
                    var tag = last(this.tags);
                    tag.insertBefore(document.createTextNode(rule), tag.firstChild);
                }
                else {
                    last(this.tags).appendChild(document.createTextNode(rule));
                }
            }
        }
        else {
            // server side is pretty simple
            this.sheet.insertRule(rule, rule.indexOf('@import') !== -1);
        }
        this.ruleCounter++;
        if (isBrowser && this.ruleCounter % this.maxRules === 0) {
            this.tags.push(makeStyleTag());
        }
        return this.ruleCounter - 1;
    };
    // delete(index) {
    //  // we insert a blank rule when 'deleting' so previously returned indexes remain stable
    //   return this.replace(index, '');
    // }
    StyleSheet.prototype.flush = function () {
        this.injected = false;
        if (isBrowser) {
            this.tags.forEach(function (tag) { return tag.parentNode.removeChild(tag); });
            this.tags = [];
            this.sheet = null;
            this.ruleCounter = 0;
            // todo - look for remnants in document.styleSheets
        }
        else {
            // simpler on server
            //this.sheet.emptyCssRules();
            this.sheet.emptyCssRules();
        }
    };
    StyleSheet.prototype.rules = function () {
        if (!isBrowser) {
            return this.sheet.getCSSRules();
        }
        var arr = [];
        this.tags.forEach(function (tag) { return arr.splice.apply(arr, [arr.length, 0].concat(Array.from(sheetForTag(tag).cssRules))); });
        return arr;
    };
    /**
     * Insert a new css rule into the <style> tag when it's in the browser environment
     * @param rule
     */
    StyleSheet.prototype.browInsert = function (rule) {
        // this weirdness for perf, and chrome's weird bug
        // https://stackoverflow.com/questions/20007992/chrome-suddenly-stopped-accepting-insertrule
        try {
            var sheet = this.getSheet();
            sheet.insertRule(rule, rule.indexOf('@import') !== -1 ? 0 : sheet.cssRules.length);
        }
        catch (e) {
            if (isDev) {
                // might need beter dx for this
                console.warn('whoops, illegal rule inserted', rule); // eslint-disable-line no-console
            }
        }
    };
    return StyleSheet;
}());
export { StyleSheet };
function makeStyleTag() {
    var tag = document.createElement('style');
    tag.type = 'text/css';
    tag.setAttribute('data-glamor', '');
    tag.appendChild(document.createTextNode(''));
    (document.head || document.getElementsByTagName('head')[0]).appendChild(tag);
    return tag;
}
function last(arr) {
    return arr[arr.length - 1];
}
function sheetForTag(tag) {
    if (tag.sheet) {
        return tag.sheet;
    }
    // this weirdness brought to you by firefox
    // tslint:disable-next-line:prefer-for-of
    for (var i = 0; i < document.styleSheets.length; i++) {
        if (document.styleSheets[i].ownerNode === tag) {
            return document.styleSheets[i];
        }
    }
}
//# sourceMappingURL=StyleSheet.js.map