import { hasLabels, selector, flatten, isLikeRule, clean, idFor } from '../utils/index';
import { plugins, styleSheet, keyframesPlugins } from '../index';
import { createMarkupForStyles } from 'react-css-property-operations';
import { inserted, registered, ruleCache, getRegistered } from '../cache';
import { hashify } from '../utils/hash';
import { isSelector, joinSelectors, isMediaQuery, joinMediaQueries, joinSupports, isSupports } from './helper';
/**
 * A Style object will be destructured into new object style with four main keys { plain:, selects:, medias:, supports:}
 *
 * 1- 'plain' contains the plain css styles like (color: red)
 *
 * 2- 'selects' contains all css styles that depend on selectors like (. , & , : , > ) ex: &:hover { }
 *
 * 3- 'medias' contains all css styles that depend on @media selectors
 *
 * 4- 'supports' contains all css styles that depend on @support selectors
 * @param style
 *
 * example:
 * input {label: [], color: 'red',
 *        &:hover: {color: 'blue'},
 *        @media(min-width: 300px): {color: 'green', &:hover: {color: ...}, .a & .c:{color: ...}}}
 *
 * output {plain: {color: 'red'},
 *          selects: {&:hover: {color: 'blue'}},
 *          medias: {@media(min-width: 300px): {plain: {color: ...}, selects: {&:hover: ..., .a & .c: ...}, medias: null, supports: null}},
 *          supports: null}
 *
 * Notice the deep destructuring in the medias object
 *
 */
function deconstruct(style) {
    // we can be sure it's not infinitely nested here
    var plain = null;
    var selects = null;
    var medias = null;
    var supports = null;
    Object.keys(style).forEach(function (key) {
        if (key.indexOf('&') >= 0) {
            selects = selects || {};
            selects[key] = style[key];
        }
        else if (key.indexOf('@media') === 0) {
            medias = medias || {};
            medias[key] = deconstruct(style[key]);
        }
        else if (key.indexOf('@supports') === 0) {
            supports = supports || {};
            supports[key] = deconstruct(style[key]);
        }
        else if (key === 'label') {
            if (style.label.length > 0) {
                plain = plain || {};
                plain.label = hasLabels ? style.label.join('.') : '';
            }
        }
        else {
            plain = plain || {};
            plain[key] = style[key];
        }
    });
    return { plain: plain, selects: selects, medias: medias, supports: supports };
}
/**
 * create an array of strings which contains the different styles with its selectors.
 *
 * The result could look like:
 *
 *  ['.css-1j2tyha,[data-css-1j2tyha]{color:green;}', '.css-1j2tyha:hover,[ data-css-1j2tyha]:hover{color:yellow;}']
 * @param id the hash value of the style.
 * @param style
 */
function deconstructedStyleToCSS(id, style) {
    var css = [];
    // plugins here
    var plain = style.plain, selects = style.selects, medias = style.medias, supports = style.supports;
    if (plain) {
        css.push(toCSS({ style: plain, selector: selector(id) }));
    }
    if (selects) {
        Object.keys(selects).forEach(function (key) {
            return css.push(toCSS({ style: selects[key], selector: selector(id, key) }));
        });
    }
    if (medias) {
        Object.keys(medias).forEach(function (key) {
            return css.push(key + "{" + deconstructedStyleToCSS(id, medias[key]).join('') + "}");
        });
    }
    if (supports) {
        Object.keys(supports).forEach(function (key) {
            return css.push(key + "{" + deconstructedStyleToCSS(id, supports[key]).join('') + "}");
        });
    }
    return css;
}
/**
 *
 * @param param0
 *
 * example
 * selector: .css-1j2tyha:hover,[data-css-1j2tyha]:hover'
 * style: {color: 'blue'}
 * result:'.css-1j2tyha:hover,[data-css-1j2tyha]:hover{color:blue;}'
 */
export function toCSS(_a) {
    var selector = _a.selector, style = _a.style;
    var result = plugins.transform({ selector: selector, style: style });
    return result.selector + "{" + createMarkupForStyles(result.style) + "}";
}
/**
 * Insert the style rule into the StyleSheet (in other words: insert the rule into the <style> tag)
 * @param spec
 */
function insert(spec) {
    if (!inserted.has(spec.id)) {
        inserted.add(spec.id, true);
        var deconstructed = deconstruct(spec.style);
        deconstructedStyleToCSS(spec.id, deconstructed).map(function (cssRule) { return styleSheet.insert(cssRule); });
    }
}
// mutable! modifies dest.
/**
 * build a simplified style object by combining between corrospending @media and @support queries
 * at the end we will get an object that is ready to be destructured
 * @param dest
 * @param param1
 */
function build(dest, _a) {
    var _b = _a.selector, selector = _b === void 0 ? '' : _b, _c = _a.mq, mq = _c === void 0 ? '' : _c, _d = _a.supp, supp = _d === void 0 ? '' : _d, _e = _a.src, src = _e === void 0 ? {} : _e;
    var source;
    if (!Array.isArray(src)) {
        source = [src];
    }
    else {
        source = src;
    }
    source = flatten(source);
    source.forEach(function (_src) {
        if (isLikeRule(_src)) {
            var reg = getRegistered(_src);
            if (reg.type !== 'css') {
                throw new Error('cannot merge this rule');
            }
            _src = reg.style;
        }
        _src = clean(_src);
        if (_src && _src.composes) {
            build(dest, { selector: selector, mq: mq, supp: supp, src: _src.composes });
        }
        Object.keys(_src || {}).forEach(function (key) {
            if (isSelector(key)) {
                if (key === '::placeholder') {
                    build(dest, { selector: joinSelectors(selector, '::-webkit-input-placeholder'), mq: mq, supp: supp, src: _src[key] });
                    build(dest, { selector: joinSelectors(selector, '::-moz-placeholder'), mq: mq, supp: supp, src: _src[key] });
                    build(dest, { selector: joinSelectors(selector, '::-ms-input-placeholder'), mq: mq, supp: supp, src: _src[key] });
                }
                build(dest, { selector: joinSelectors(selector, key), mq: mq, supp: supp, src: _src[key] });
            }
            else if (isMediaQuery(key)) {
                build(dest, { selector: selector, mq: joinMediaQueries(mq, key), supp: supp, src: _src[key] });
            }
            else if (isSupports(key)) {
                build(dest, { selector: selector, mq: mq, supp: joinSupports(supp, key), src: _src[key] });
            }
            else if (key === 'composes') {
                // ignore, we already dealth with it
            }
            else {
                var _dest = dest;
                if (supp) {
                    _dest[supp] = _dest[supp] || {};
                    _dest = _dest[supp];
                }
                if (mq) {
                    _dest[mq] = _dest[mq] || {};
                    _dest = _dest[mq];
                }
                if (selector) {
                    _dest[selector] = _dest[selector] || {};
                    _dest = _dest[selector];
                }
                if (key === 'label') {
                    if (hasLabels) {
                        dest.label = dest.label.concat(_src.label);
                    }
                }
                else {
                    _dest[key] = _src[key];
                }
            }
        });
    });
}
// let cachedCss = (typeof WeakMap !== 'undefined') ? multiIndexCache(_css) : _css;
export function generateCss(rules) {
    // hard to type because before build() label is a string, after
    var style = { label: [] };
    build(style, { src: rules }); // mutative! but worth it.
    var spec = {
        id: hashify(style),
        style: style,
        label: hasLabels ? style.label.join('.') : '',
        type: 'css'
    };
    return toRule(spec);
}
/**
 * get the actual output for the css function, the result will look similer to:
 * {data-css-1j2tyha: ''}
 * @param spec
 */
function toRule(spec) {
    registered.add(spec.id, spec);
    insert(spec);
    if (ruleCache.has(spec.id)) {
        return ruleCache.get(spec.id);
    }
    var ret = (_a = {}, _a["data-css-" + spec.id] = hasLabels ? spec.label || '' : '', _a);
    Object.defineProperty(ret, 'toString', {
        enumerable: false, value: function () { return 'css-' + spec.id; }
    });
    ruleCache.add(spec.id, ret);
    return ret;
    var _a;
}
export function cssFor() {
    var rules = [];
    for (var _i = 0; _i < arguments.length; _i++) {
        rules[_i] = arguments[_i];
    }
    var r = clean(rules);
    return r ? r.map(function (r) {
        var style = { label: [] };
        build(style, { src: r }); // mutative! but worth it.
        return deconstructedStyleToCSS(hashify(style), deconstruct(style)).join('');
    }).join('') : '';
}
export function attribsFor() {
    var rules = [];
    for (var _i = 0; _i < arguments.length; _i++) {
        rules[_i] = arguments[_i];
    }
    var r = clean(rules);
    var htmlAttributes = r ? r.map(function (rule) {
        idFor(rule); // throwaway check for rule
        var key = Object.keys(rule)[0], value = rule[key];
        return key + "=\"" + (value || '') + "\"";
    }).join(' ') : '';
    return htmlAttributes;
}
export function insertKeyframe(spec) {
    if (!inserted.has(spec.id)) {
        var inner_1 = Object.keys(spec.keyframes).map(function (kf) {
            var result = keyframesPlugins.transform({ id: spec.id, name: kf, style: spec.keyframes[kf] });
            return result.name + "{" + createMarkupForStyles(result.style) + "}";
        }).join('');
        ['-webkit-', '-moz-', '-o-', ''].forEach(function (prefix) {
            return styleSheet.insert("@" + prefix + "keyframes " + (spec.name + '_' + spec.id) + "{" + inner_1 + "}");
        });
        inserted.add(spec.id, true);
    }
}
export function insertFontFace(spec) {
    if (!inserted.has(spec.id)) {
        styleSheet.insert("@font-face{" + createMarkupForStyles(spec.font) + "}");
        inserted.add(spec.id, true);
    }
}
//# sourceMappingURL=index.js.map