/**
 * Copyright 2015 Google Inc. All Rights Reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
import { AxiosPromise } from 'axios';
import { Compute, JWT, OAuth2Client, UserRefreshClient } from 'google-auth-library';
import { BodyResponseCallback, GlobalOptions, GoogleConfigurable, MethodOptions } from '../../shared/src';
export declare namespace cloudbuild_v1 {
    interface Options extends GlobalOptions {
        version: 'v1';
    }
    /**
     * Cloud Container Builder API
     *
     * Builds container images in the cloud.
     *
     * @example
     * const {google} = require('googleapis');
     * const cloudbuild = google.cloudbuild('v1');
     *
     * @namespace cloudbuild
     * @type {Function}
     * @version v1
     * @variation v1
     * @param {object=} options Options for Cloudbuild
     */
    class Cloudbuild {
        _options: GlobalOptions;
        google?: GoogleConfigurable;
        root: this;
        operations: Resource$Operations;
        projects: Resource$Projects;
        constructor(options: GlobalOptions, google?: GoogleConfigurable);
        getRoot(): this;
    }
    /**
     * Files in the workspace to upload to Cloud Storage upon successful
     * completion of all build steps.
     */
    interface Schema$ArtifactObjects {
        /**
         * Cloud Storage bucket and optional object path, in the form
         * &quot;gs://bucket/path/to/somewhere/&quot;. (see [Bucket Name
         * Requirements](https://cloud.google.com/storage/docs/bucket-naming#requirements)).
         * Files in the workspace matching any path pattern will be uploaded to
         * Cloud Storage with this location as a prefix.
         */
        location?: string;
        /**
         * Path globs used to match files in the build&#39;s workspace.
         */
        paths?: string[];
        /**
         * Stores timing information for pushing all artifact objects. @OutputOnly
         */
        timing?: Schema$TimeSpan;
    }
    /**
     * An artifact that was uploaded during a build. This is a single record in
     * the artifact manifest JSON file.
     */
    interface Schema$ArtifactResult {
        /**
         * The file hash of the artifact.
         */
        fileHash?: Schema$FileHashes[];
        /**
         * The path of an artifact in a Google Cloud Storage bucket, with the
         * generation number. For example,
         * `gs://mybucket/path/to/output.jar#generation`.
         */
        location?: string;
    }
    /**
     * Artifacts produced by a build that should be uploaded upon successful
     * completion of all build steps.
     */
    interface Schema$Artifacts {
        /**
         * A list of images to be pushed upon the successful completion of all build
         * steps.  The images will be pushed using the builder service account&#39;s
         * credentials.  The digests of the pushed images will be stored in the
         * Build resource&#39;s results field.  If any of the images fail to be
         * pushed, the build is marked FAILURE.
         */
        images?: string[];
        /**
         * A list of objects to be uploaded to Cloud Storage upon successful
         * completion of all build steps.  Files in the workspace matching specified
         * paths globs will be uploaded to the specified Cloud Storage location
         * using the builder service account&#39;s credentials.  The location and
         * generation of the uploaded objects will be stored in the Build
         * resource&#39;s results field.  If any objects fail to be pushed, the
         * build is marked FAILURE.
         */
        objects?: Schema$ArtifactObjects;
    }
    /**
     * A build resource in the Container Builder API.  At a high level, a `Build`
     * describes where to find source code, how to build it (for example, the
     * builder image to run on the source), and where to store the built
     * artifacts.  Fields can include the following variables, which will be
     * expanded when the build is created:  - $PROJECT_ID: the project ID of the
     * build. - $BUILD_ID: the autogenerated ID of the build. - $REPO_NAME: the
     * source repository name specified by RepoSource. - $BRANCH_NAME: the branch
     * name specified by RepoSource. - $TAG_NAME: the tag name specified by
     * RepoSource. - $REVISION_ID or $COMMIT_SHA: the commit SHA specified by
     * RepoSource or   resolved from the specified branch or tag. - $SHORT_SHA:
     * first 7 characters of $REVISION_ID or $COMMIT_SHA.
     */
    interface Schema$Build {
        /**
         * Artifacts produced by the build that should be uploaded upon successful
         * completion of all build steps.
         */
        artifacts?: Schema$Artifacts;
        /**
         * The ID of the `BuildTrigger` that triggered this build, if it was
         * triggered automatically. @OutputOnly
         */
        buildTriggerId?: string;
        /**
         * Time at which the request to create the build was received. @OutputOnly
         */
        createTime?: string;
        /**
         * Time at which execution of the build was finished.  The difference
         * between finish_time and start_time is the duration of the build&#39;s
         * execution. @OutputOnly
         */
        finishTime?: string;
        /**
         * Unique identifier of the build. @OutputOnly
         */
        id?: string;
        /**
         * A list of images to be pushed upon the successful completion of all build
         * steps.  The images are pushed using the builder service account&#39;s
         * credentials.  The digests of the pushed images will be stored in the
         * `Build` resource&#39;s results field.  If any of the images fail to be
         * pushed, the build status is marked `FAILURE`.
         */
        images?: string[];
        /**
         * Google Cloud Storage bucket where logs should be written (see [Bucket
         * Name
         * Requirements](https://cloud.google.com/storage/docs/bucket-naming#requirements)).
         * Logs file names will be of the format
         * `${logs_bucket}/log-${build_id}.txt`.
         */
        logsBucket?: string;
        /**
         * URL to logs for this build in Google Cloud Console. @OutputOnly
         */
        logUrl?: string;
        /**
         * Special options for this build.
         */
        options?: Schema$BuildOptions;
        /**
         * ID of the project. @OutputOnly.
         */
        projectId?: string;
        /**
         * Results of the build. @OutputOnly
         */
        results?: Schema$Results;
        /**
         * Secrets to decrypt using Cloud Key Management Service.
         */
        secrets?: Schema$Secret[];
        /**
         * The location of the source files to build.
         */
        source?: Schema$Source;
        /**
         * A permanent fixed identifier for source. @OutputOnly
         */
        sourceProvenance?: Schema$SourceProvenance;
        /**
         * Time at which execution of the build was started. @OutputOnly
         */
        startTime?: string;
        /**
         * Status of the build. @OutputOnly
         */
        status?: string;
        /**
         * Customer-readable message about the current status. @OutputOnly
         */
        statusDetail?: string;
        /**
         * Required. The operations to be performed on the workspace.
         */
        steps?: Schema$BuildStep[];
        /**
         * Substitutions data for `Build` resource.
         */
        substitutions?: any;
        /**
         * Tags for annotation of a `Build`. These are not docker tags.
         */
        tags?: string[];
        /**
         * Amount of time that this build should be allowed to run, to second
         * granularity. If this amount of time elapses, work on the build will cease
         * and the build status will be `TIMEOUT`.  Default time is ten minutes.
         */
        timeout?: string;
        /**
         * Stores timing information for phases of the build. Valid keys are:  *
         * BUILD: time to execute all build steps * PUSH: time to push all specified
         * images. * FETCHSOURCE: time to fetch source.  If the build does not
         * specify source or images, these keys will not be included. @OutputOnly
         */
        timing?: any;
    }
    /**
     * Metadata for build operations.
     */
    interface Schema$BuildOperationMetadata {
        /**
         * The build that the operation is tracking.
         */
        build?: Schema$Build;
    }
    /**
     * Optional arguments to enable specific features of builds.
     */
    interface Schema$BuildOptions {
        /**
         * Requested disk size for the VM that runs the build. Note that this is
         * *NOT* &quot;disk free&quot;; some of the space will be used by the
         * operating system and build utilities. Also note that this is the minimum
         * disk size that will be allocated for the build -- the build may run with
         * a larger disk than requested. At present, the maximum disk size is
         * 1000GB; builds that request more than the maximum are rejected with an
         * error.
         */
        diskSizeGb?: string;
        /**
         * Option to define build log streaming behavior to Google Cloud Storage.
         */
        logStreamingOption?: string;
        /**
         * Compute Engine machine type on which to run the build.
         */
        machineType?: string;
        /**
         * Requested verifiability options.
         */
        requestedVerifyOption?: string;
        /**
         * Requested hash for SourceProvenance.
         */
        sourceProvenanceHash?: string[];
        /**
         * Option to specify behavior when there is an error in the substitution
         * checks.
         */
        substitutionOption?: string;
    }
    /**
     * A step in the build pipeline.
     */
    interface Schema$BuildStep {
        /**
         * A list of arguments that will be presented to the step when it is
         * started.  If the image used to run the step&#39;s container has an
         * entrypoint, the `args` are used as arguments to that entrypoint. If the
         * image does not define an entrypoint, the first element in args is used as
         * the entrypoint, and the remainder will be used as arguments.
         */
        args?: string[];
        /**
         * Working directory to use when running this step&#39;s container.  If this
         * value is a relative path, it is relative to the build&#39;s working
         * directory. If this value is absolute, it may be outside the build&#39;s
         * working directory, in which case the contents of the path may not be
         * persisted across build step executions, unless a `volume` for that path
         * is specified.  If the build specifies a `RepoSource` with `dir` and a
         * step with a `dir`, which specifies an absolute path, the `RepoSource`
         * `dir` is ignored for the step&#39;s execution.
         */
        dir?: string;
        /**
         * Entrypoint to be used instead of the build step image&#39;s default
         * entrypoint. If unset, the image&#39;s default entrypoint is used.
         */
        entrypoint?: string;
        /**
         * A list of environment variable definitions to be used when running a
         * step.  The elements are of the form &quot;KEY=VALUE&quot; for the
         * environment variable &quot;KEY&quot; being given the value
         * &quot;VALUE&quot;.
         */
        env?: string[];
        /**
         * Unique identifier for this build step, used in `wait_for` to reference
         * this build step as a dependency.
         */
        id?: string;
        /**
         * Required. The name of the container image that will run this particular
         * build step.  If the image is available in the host&#39;s Docker
         * daemon&#39;s cache, it will be run directly. If not, the host will
         * attempt to pull the image first, using the builder service account&#39;s
         * credentials if necessary.  The Docker daemon&#39;s cache will already
         * have the latest versions of all of the officially supported build steps
         * ([https://github.com/GoogleCloudPlatform/cloud-builders](https://github.com/GoogleCloudPlatform/cloud-builders)).
         * The Docker daemon will also have cached many of the layers for some
         * popular images, like &quot;ubuntu&quot;, &quot;debian&quot;, but they
         * will be refreshed at the time you attempt to use them.  If you built an
         * image in a previous build step, it will be stored in the host&#39;s
         * Docker daemon&#39;s cache and is available to use as the name for a later
         * build step.
         */
        name?: string;
        /**
         * A list of environment variables which are encrypted using a Cloud Key
         * Management Service crypto key. These values must be specified in the
         * build&#39;s `Secret`.
         */
        secretEnv?: string[];
        /**
         * Status of the build step. At this time, build step status is only updated
         * on build completion; step status is not updated in real-time as the build
         * progresses. @OutputOnly
         */
        status?: string;
        /**
         * Time limit for executing this build step. If not defined, the step has no
         * time limit and will be allowed to continue to run until either it
         * completes or the build itself times out.
         */
        timeout?: string;
        /**
         * Stores timing information for executing this build step. @OutputOnly
         */
        timing?: Schema$TimeSpan;
        /**
         * List of volumes to mount into the build step.  Each volume will be
         * created as an empty volume prior to execution of the build step. Upon
         * completion of the build, volumes and their contents will be discarded.
         * Using a named volume in only one step is not valid as it is indicative of
         * a mis-configured build request.
         */
        volumes?: Schema$Volume[];
        /**
         * The ID(s) of the step(s) that this build step depends on. This build step
         * will not start until all the build steps in `wait_for` have completed
         * successfully. If `wait_for` is empty, this build step will start when all
         * previous build steps in the `Build.Steps` list have completed
         * successfully.
         */
        waitFor?: string[];
    }
    /**
     * Configuration for an automated build in response to source repository
     * changes.
     */
    interface Schema$BuildTrigger {
        /**
         * Contents of the build template.
         */
        build?: Schema$Build;
        /**
         * Time when the trigger was created.  @OutputOnly
         */
        createTime?: string;
        /**
         * Human-readable description of this trigger.
         */
        description?: string;
        /**
         * If true, the trigger will never result in a build.
         */
        disabled?: boolean;
        /**
         * Path, from the source root, to a file whose contents is used for the
         * template.
         */
        filename?: string;
        /**
         * Unique identifier of the trigger.  @OutputOnly
         */
        id?: string;
        /**
         * Substitutions data for Build resource.
         */
        substitutions?: any;
        /**
         * Template describing the types of source changes to trigger a build.
         * Branch and tag names in trigger templates are interpreted as regular
         * expressions. Any branch or tag change that matches that regular
         * expression will trigger a build.
         */
        triggerTemplate?: Schema$RepoSource;
    }
    /**
     * An image built by the pipeline.
     */
    interface Schema$BuiltImage {
        /**
         * Docker Registry 2.0 digest.
         */
        digest?: string;
        /**
         * Name used to push the container image to Google Container Registry, as
         * presented to `docker push`.
         */
        name?: string;
        /**
         * Stores timing information for pushing the specified image. @OutputOnly
         */
        pushTiming?: Schema$TimeSpan;
    }
    /**
     * Request to cancel an ongoing build.
     */
    interface Schema$CancelBuildRequest {
    }
    /**
     * The request message for Operations.CancelOperation.
     */
    interface Schema$CancelOperationRequest {
    }
    /**
     * A generic empty message that you can re-use to avoid defining duplicated
     * empty messages in your APIs. A typical example is to use it as the request
     * or the response type of an API method. For instance:      service Foo { rpc
     * Bar(google.protobuf.Empty) returns (google.protobuf.Empty);     }  The JSON
     * representation for `Empty` is empty JSON object `{}`.
     */
    interface Schema$Empty {
    }
    /**
     * Container message for hashes of byte content of files, used in
     * SourceProvenance messages to verify integrity of source input to the build.
     */
    interface Schema$FileHashes {
        /**
         * Collection of file hashes.
         */
        fileHash?: Schema$Hash[];
    }
    /**
     * Container message for hash values.
     */
    interface Schema$Hash {
        /**
         * The type of hash that was performed.
         */
        type?: string;
        /**
         * The hash value.
         */
        value?: string;
    }
    /**
     * Response including listed builds.
     */
    interface Schema$ListBuildsResponse {
        /**
         * Builds will be sorted by `create_time`, descending.
         */
        builds?: Schema$Build[];
        /**
         * Token to receive the next page of results.
         */
        nextPageToken?: string;
    }
    /**
     * Response containing existing `BuildTriggers`.
     */
    interface Schema$ListBuildTriggersResponse {
        /**
         * `BuildTriggers` for the project, sorted by `create_time` descending.
         */
        triggers?: Schema$BuildTrigger[];
    }
    /**
     * The response message for Operations.ListOperations.
     */
    interface Schema$ListOperationsResponse {
        /**
         * The standard List next-page token.
         */
        nextPageToken?: string;
        /**
         * A list of operations that matches the specified filter in the request.
         */
        operations?: Schema$Operation[];
    }
    /**
     * This resource represents a long-running operation that is the result of a
     * network API call.
     */
    interface Schema$Operation {
        /**
         * If the value is `false`, it means the operation is still in progress. If
         * `true`, the operation is completed, and either `error` or `response` is
         * available.
         */
        done?: boolean;
        /**
         * The error result of the operation in case of failure or cancellation.
         */
        error?: Schema$Status;
        /**
         * Service-specific metadata associated with the operation.  It typically
         * contains progress information and common metadata such as create time.
         * Some services might not provide such metadata.  Any method that returns a
         * long-running operation should document the metadata type, if any.
         */
        metadata?: any;
        /**
         * The server-assigned name, which is only unique within the same service
         * that originally returns it. If you use the default HTTP mapping, the
         * `name` should have the format of `operations/some/unique/name`.
         */
        name?: string;
        /**
         * The normal response of the operation in case of success.  If the original
         * method returns no data on success, such as `Delete`, the response is
         * `google.protobuf.Empty`.  If the original method is standard
         * `Get`/`Create`/`Update`, the response should be the resource.  For other
         * methods, the response should have the type `XxxResponse`, where `Xxx` is
         * the original method name.  For example, if the original method name is
         * `TakeSnapshot()`, the inferred response type is `TakeSnapshotResponse`.
         */
        response?: any;
    }
    /**
     * Location of the source in a Google Cloud Source Repository.
     */
    interface Schema$RepoSource {
        /**
         * Name of the branch to build.
         */
        branchName?: string;
        /**
         * Explicit commit SHA to build.
         */
        commitSha?: string;
        /**
         * Directory, relative to the source root, in which to run the build.  This
         * must be a relative path. If a step&#39;s `dir` is specified and is an
         * absolute path, this value is ignored for that step&#39;s execution.
         */
        dir?: string;
        /**
         * ID of the project that owns the Cloud Source Repository. If omitted, the
         * project ID requesting the build is assumed.
         */
        projectId?: string;
        /**
         * Name of the Cloud Source Repository. If omitted, the name
         * &quot;default&quot; is assumed.
         */
        repoName?: string;
        /**
         * Name of the tag to build.
         */
        tagName?: string;
    }
    /**
     * Artifacts created by the build pipeline.
     */
    interface Schema$Results {
        /**
         * Path to the artifact manifest. Only populated when artifacts are
         * uploaded.
         */
        artifactManifest?: string;
        /**
         * List of build step digests, in the order corresponding to build step
         * indices.
         */
        buildStepImages?: string[];
        /**
         * Container images that were built as a part of the build.
         */
        images?: Schema$BuiltImage[];
        /**
         * Number of artifacts uploaded. Only populated when artifacts are uploaded.
         */
        numArtifacts?: string;
    }
    /**
     * Specifies a build to retry.
     */
    interface Schema$RetryBuildRequest {
    }
    /**
     * Pairs a set of secret environment variables containing encrypted values
     * with the Cloud KMS key to use to decrypt the value.
     */
    interface Schema$Secret {
        /**
         * Cloud KMS key name to use to decrypt these envs.
         */
        kmsKeyName?: string;
        /**
         * Map of environment variable name to its encrypted value.  Secret
         * environment variables must be unique across all of a build&#39;s secrets,
         * and must be used by at least one build step. Values can be at most 1 KB
         * in size. There can be at most ten secret values across all of a
         * build&#39;s secrets.
         */
        secretEnv?: any;
    }
    /**
     * Location of the source in a supported storage service.
     */
    interface Schema$Source {
        /**
         * If provided, get the source from this location in a Cloud Source
         * Repository.
         */
        repoSource?: Schema$RepoSource;
        /**
         * If provided, get the source from this location in Google Cloud Storage.
         */
        storageSource?: Schema$StorageSource;
    }
    /**
     * Provenance of the source. Ways to find the original source, or verify that
     * some source was used for this build.
     */
    interface Schema$SourceProvenance {
        /**
         * Hash(es) of the build source, which can be used to verify that the
         * original source integrity was maintained in the build. Note that
         * `FileHashes` will only be populated if `BuildOptions` has requested a
         * `SourceProvenanceHash`.  The keys to this map are file paths used as
         * build source and the values contain the hash values for those files.  If
         * the build source came in a single package such as a gzipped tarfile
         * (`.tar.gz`), the `FileHash` will be for the single path to that file.
         * @OutputOnly
         */
        fileHashes?: any;
        /**
         * A copy of the build&#39;s `source.repo_source`, if exists, with any
         * revisions resolved.
         */
        resolvedRepoSource?: Schema$RepoSource;
        /**
         * A copy of the build&#39;s `source.storage_source`, if exists, with any
         * generations resolved.
         */
        resolvedStorageSource?: Schema$StorageSource;
    }
    /**
     * The `Status` type defines a logical error model that is suitable for
     * different programming environments, including REST APIs and RPC APIs. It is
     * used by [gRPC](https://github.com/grpc). The error model is designed to be:
     * - Simple to use and understand for most users - Flexible enough to meet
     * unexpected needs  # Overview  The `Status` message contains three pieces of
     * data: error code, error message, and error details. The error code should
     * be an enum value of google.rpc.Code, but it may accept additional error
     * codes if needed.  The error message should be a developer-facing English
     * message that helps developers *understand* and *resolve* the error. If a
     * localized user-facing error message is needed, put the localized message in
     * the error details or localize it in the client. The optional error details
     * may contain arbitrary information about the error. There is a predefined
     * set of error detail types in the package `google.rpc` that can be used for
     * common error conditions.  # Language mapping  The `Status` message is the
     * logical representation of the error model, but it is not necessarily the
     * actual wire format. When the `Status` message is exposed in different
     * client libraries and different wire protocols, it can be mapped
     * differently. For example, it will likely be mapped to some exceptions in
     * Java, but more likely mapped to some error codes in C.  # Other uses  The
     * error model and the `Status` message can be used in a variety of
     * environments, either with or without APIs, to provide a consistent
     * developer experience across different environments.  Example uses of this
     * error model include:  - Partial errors. If a service needs to return
     * partial errors to the client,     it may embed the `Status` in the normal
     * response to indicate the partial     errors.  - Workflow errors. A typical
     * workflow has multiple steps. Each step may     have a `Status` message for
     * error reporting.  - Batch operations. If a client uses batch request and
     * batch response, the     `Status` message should be used directly inside
     * batch response, one for     each error sub-response.  - Asynchronous
     * operations. If an API call embeds asynchronous operation     results in its
     * response, the status of those operations should be     represented directly
     * using the `Status` message.  - Logging. If some API errors are stored in
     * logs, the message `Status` could     be used directly after any stripping
     * needed for security/privacy reasons.
     */
    interface Schema$Status {
        /**
         * The status code, which should be an enum value of google.rpc.Code.
         */
        code?: number;
        /**
         * A list of messages that carry the error details.  There is a common set
         * of message types for APIs to use.
         */
        details?: any[];
        /**
         * A developer-facing error message, which should be in English. Any
         * user-facing error message should be localized and sent in the
         * google.rpc.Status.details field, or localized by the client.
         */
        message?: string;
    }
    /**
     * Location of the source in an archive file in Google Cloud Storage.
     */
    interface Schema$StorageSource {
        /**
         * Google Cloud Storage bucket containing the source (see [Bucket Name
         * Requirements](https://cloud.google.com/storage/docs/bucket-naming#requirements)).
         */
        bucket?: string;
        /**
         * Google Cloud Storage generation for the object. If the generation is
         * omitted, the latest generation will be used.
         */
        generation?: string;
        /**
         * Google Cloud Storage object containing the source.  This object must be a
         * gzipped archive file (`.tar.gz`) containing source to build.
         */
        object?: string;
    }
    /**
     * Start and end times for a build execution phase.
     */
    interface Schema$TimeSpan {
        /**
         * End of time span.
         */
        endTime?: string;
        /**
         * Start of time span.
         */
        startTime?: string;
    }
    /**
     * Volume describes a Docker container volume which is mounted into build
     * steps in order to persist files across build step execution.
     */
    interface Schema$Volume {
        /**
         * Name of the volume to mount.  Volume names must be unique per build step
         * and must be valid names for Docker volumes. Each named volume must be
         * used by at least two build steps.
         */
        name?: string;
        /**
         * Path at which to mount the volume.  Paths must be absolute and cannot
         * conflict with other volume paths on the same build step or with certain
         * reserved volume paths.
         */
        path?: string;
    }
    class Resource$Operations {
        root: Cloudbuild;
        constructor(root: Cloudbuild);
        getRoot(): Cloudbuild;
        /**
         * cloudbuild.operations.cancel
         * @desc Starts asynchronous cancellation on a long-running operation.  The
         * server makes a best effort to cancel the operation, but success is not
         * guaranteed.  If the server doesn't support this method, it returns
         * `google.rpc.Code.UNIMPLEMENTED`.  Clients can use Operations.GetOperation
         * or other methods to check whether the cancellation succeeded or whether
         * the operation completed despite cancellation. On successful cancellation,
         * the operation is not deleted; instead, it becomes an operation with an
         * Operation.error value with a google.rpc.Status.code of 1, corresponding
         * to `Code.CANCELLED`.
         * @alias cloudbuild.operations.cancel
         * @memberOf! ()
         *
         * @param {object} params Parameters for request
         * @param {string} params.name The name of the operation resource to be cancelled.
         * @param {().CancelOperationRequest} params.resource Request body data
         * @param {object} [options] Optionally override request options, such as `url`, `method`, and `encoding`.
         * @param {callback} callback The callback that handles the response.
         * @return {object} Request object
         */
        cancel(params?: Params$Resource$Operations$Cancel, options?: MethodOptions): AxiosPromise<Schema$Empty>;
        cancel(params: Params$Resource$Operations$Cancel, options: MethodOptions | BodyResponseCallback<Schema$Empty>, callback: BodyResponseCallback<Schema$Empty>): void;
        cancel(params: Params$Resource$Operations$Cancel, callback: BodyResponseCallback<Schema$Empty>): void;
        cancel(callback: BodyResponseCallback<Schema$Empty>): void;
        /**
         * cloudbuild.operations.get
         * @desc Gets the latest state of a long-running operation.  Clients can use
         * this method to poll the operation result at intervals as recommended by
         * the API service.
         * @alias cloudbuild.operations.get
         * @memberOf! ()
         *
         * @param {object} params Parameters for request
         * @param {string} params.name The name of the operation resource.
         * @param {object} [options] Optionally override request options, such as `url`, `method`, and `encoding`.
         * @param {callback} callback The callback that handles the response.
         * @return {object} Request object
         */
        get(params?: Params$Resource$Operations$Get, options?: MethodOptions): AxiosPromise<Schema$Operation>;
        get(params: Params$Resource$Operations$Get, options: MethodOptions | BodyResponseCallback<Schema$Operation>, callback: BodyResponseCallback<Schema$Operation>): void;
        get(params: Params$Resource$Operations$Get, callback: BodyResponseCallback<Schema$Operation>): void;
        get(callback: BodyResponseCallback<Schema$Operation>): void;
        /**
         * cloudbuild.operations.list
         * @desc Lists operations that match the specified filter in the request. If
         * the server doesn't support this method, it returns `UNIMPLEMENTED`. NOTE:
         * the `name` binding allows API services to override the binding to use
         * different resource name schemes, such as `users/x/operations`. To
         * override the binding, API services can add a binding such as
         * `"/v1/{name=users/x}/operations"` to their service configuration. For
         * backwards compatibility, the default name includes the operations
         * collection id, however overriding users must ensure the name binding is
         * the parent resource, without the operations collection id.
         * @alias cloudbuild.operations.list
         * @memberOf! ()
         *
         * @param {object} params Parameters for request
         * @param {string=} params.filter The standard list filter.
         * @param {string} params.name The name of the operation's parent resource.
         * @param {integer=} params.pageSize The standard list page size.
         * @param {string=} params.pageToken The standard list page token.
         * @param {object} [options] Optionally override request options, such as `url`, `method`, and `encoding`.
         * @param {callback} callback The callback that handles the response.
         * @return {object} Request object
         */
        list(params?: Params$Resource$Operations$List, options?: MethodOptions): AxiosPromise<Schema$ListOperationsResponse>;
        list(params: Params$Resource$Operations$List, options: MethodOptions | BodyResponseCallback<Schema$ListOperationsResponse>, callback: BodyResponseCallback<Schema$ListOperationsResponse>): void;
        list(params: Params$Resource$Operations$List, callback: BodyResponseCallback<Schema$ListOperationsResponse>): void;
        list(callback: BodyResponseCallback<Schema$ListOperationsResponse>): void;
    }
    interface Params$Resource$Operations$Cancel {
        /**
         * Auth client or API Key for the request
         */
        auth?: string | OAuth2Client | JWT | Compute | UserRefreshClient;
        /**
         * The name of the operation resource to be cancelled.
         */
        name?: string;
        /**
         * Request body metadata
         */
        requestBody?: Schema$CancelOperationRequest;
    }
    interface Params$Resource$Operations$Get {
        /**
         * Auth client or API Key for the request
         */
        auth?: string | OAuth2Client | JWT | Compute | UserRefreshClient;
        /**
         * The name of the operation resource.
         */
        name?: string;
    }
    interface Params$Resource$Operations$List {
        /**
         * Auth client or API Key for the request
         */
        auth?: string | OAuth2Client | JWT | Compute | UserRefreshClient;
        /**
         * The standard list filter.
         */
        filter?: string;
        /**
         * The name of the operation's parent resource.
         */
        name?: string;
        /**
         * The standard list page size.
         */
        pageSize?: number;
        /**
         * The standard list page token.
         */
        pageToken?: string;
    }
    class Resource$Projects {
        root: Cloudbuild;
        builds: Resource$Projects$Builds;
        triggers: Resource$Projects$Triggers;
        constructor(root: Cloudbuild);
        getRoot(): Cloudbuild;
    }
    class Resource$Projects$Builds {
        root: Cloudbuild;
        constructor(root: Cloudbuild);
        getRoot(): Cloudbuild;
        /**
         * cloudbuild.projects.builds.cancel
         * @desc Cancels a build in progress.
         * @alias cloudbuild.projects.builds.cancel
         * @memberOf! ()
         *
         * @param {object} params Parameters for request
         * @param {string} params.id ID of the build.
         * @param {string} params.projectId ID of the project.
         * @param {().CancelBuildRequest} params.resource Request body data
         * @param {object} [options] Optionally override request options, such as `url`, `method`, and `encoding`.
         * @param {callback} callback The callback that handles the response.
         * @return {object} Request object
         */
        cancel(params?: Params$Resource$Projects$Builds$Cancel, options?: MethodOptions): AxiosPromise<Schema$Build>;
        cancel(params: Params$Resource$Projects$Builds$Cancel, options: MethodOptions | BodyResponseCallback<Schema$Build>, callback: BodyResponseCallback<Schema$Build>): void;
        cancel(params: Params$Resource$Projects$Builds$Cancel, callback: BodyResponseCallback<Schema$Build>): void;
        cancel(callback: BodyResponseCallback<Schema$Build>): void;
        /**
         * cloudbuild.projects.builds.create
         * @desc Starts a build with the specified configuration.  This method
         * returns a long-running `Operation`, which includes the build ID. Pass the
         * build ID to `GetBuild` to determine the build status (such as `SUCCESS`
         * or `FAILURE`).
         * @alias cloudbuild.projects.builds.create
         * @memberOf! ()
         *
         * @param {object} params Parameters for request
         * @param {string} params.projectId ID of the project.
         * @param {().Build} params.resource Request body data
         * @param {object} [options] Optionally override request options, such as `url`, `method`, and `encoding`.
         * @param {callback} callback The callback that handles the response.
         * @return {object} Request object
         */
        create(params?: Params$Resource$Projects$Builds$Create, options?: MethodOptions): AxiosPromise<Schema$Operation>;
        create(params: Params$Resource$Projects$Builds$Create, options: MethodOptions | BodyResponseCallback<Schema$Operation>, callback: BodyResponseCallback<Schema$Operation>): void;
        create(params: Params$Resource$Projects$Builds$Create, callback: BodyResponseCallback<Schema$Operation>): void;
        create(callback: BodyResponseCallback<Schema$Operation>): void;
        /**
         * cloudbuild.projects.builds.get
         * @desc Returns information about a previously requested build.  The
         * `Build` that is returned includes its status (such as `SUCCESS`,
         * `FAILURE`, or `WORKING`), and timing information.
         * @alias cloudbuild.projects.builds.get
         * @memberOf! ()
         *
         * @param {object} params Parameters for request
         * @param {string} params.id ID of the build.
         * @param {string} params.projectId ID of the project.
         * @param {object} [options] Optionally override request options, such as `url`, `method`, and `encoding`.
         * @param {callback} callback The callback that handles the response.
         * @return {object} Request object
         */
        get(params?: Params$Resource$Projects$Builds$Get, options?: MethodOptions): AxiosPromise<Schema$Build>;
        get(params: Params$Resource$Projects$Builds$Get, options: MethodOptions | BodyResponseCallback<Schema$Build>, callback: BodyResponseCallback<Schema$Build>): void;
        get(params: Params$Resource$Projects$Builds$Get, callback: BodyResponseCallback<Schema$Build>): void;
        get(callback: BodyResponseCallback<Schema$Build>): void;
        /**
         * cloudbuild.projects.builds.list
         * @desc Lists previously requested builds.  Previously requested builds may
         * still be in-progress, or may have finished successfully or
         * unsuccessfully.
         * @alias cloudbuild.projects.builds.list
         * @memberOf! ()
         *
         * @param {object} params Parameters for request
         * @param {string=} params.filter The raw filter text to constrain the results.
         * @param {integer=} params.pageSize Number of results to return in the list.
         * @param {string=} params.pageToken Token to provide to skip to a particular spot in the list.
         * @param {string} params.projectId ID of the project.
         * @param {object} [options] Optionally override request options, such as `url`, `method`, and `encoding`.
         * @param {callback} callback The callback that handles the response.
         * @return {object} Request object
         */
        list(params?: Params$Resource$Projects$Builds$List, options?: MethodOptions): AxiosPromise<Schema$ListBuildsResponse>;
        list(params: Params$Resource$Projects$Builds$List, options: MethodOptions | BodyResponseCallback<Schema$ListBuildsResponse>, callback: BodyResponseCallback<Schema$ListBuildsResponse>): void;
        list(params: Params$Resource$Projects$Builds$List, callback: BodyResponseCallback<Schema$ListBuildsResponse>): void;
        list(callback: BodyResponseCallback<Schema$ListBuildsResponse>): void;
        /**
         * cloudbuild.projects.builds.retry
         * @desc Creates a new build based on the specified build.  This method
         * creates a new build using the original build request, which may or may
         * not result in an identical build.  For triggered builds:  * Triggered
         * builds resolve to a precise revision; therefore a retry of a triggered
         * build will result in a build that uses the same revision.  For
         * non-triggered builds that specify `RepoSource`:  * If the original build
         * built from the tip of a branch, the retried build will build from the tip
         * of that branch, which may not be the same revision as the original build.
         * * If the original build specified a commit sha or revision ID, the
         * retried build will use the identical source.  For builds that specify
         * `StorageSource`:  * If the original build pulled source from Google Cloud
         * Storage without specifying the generation of the object, the new build
         * will use the current object, which may be different from the original
         * build source. * If the original build pulled source from Cloud Storage
         * and specified the generation of the object, the new build will attempt to
         * use the same object, which may or may not be available depending on the
         * bucket's lifecycle management settings.
         * @alias cloudbuild.projects.builds.retry
         * @memberOf! ()
         *
         * @param {object} params Parameters for request
         * @param {string} params.id Build ID of the original build.
         * @param {string} params.projectId ID of the project.
         * @param {().RetryBuildRequest} params.resource Request body data
         * @param {object} [options] Optionally override request options, such as `url`, `method`, and `encoding`.
         * @param {callback} callback The callback that handles the response.
         * @return {object} Request object
         */
        retry(params?: Params$Resource$Projects$Builds$Retry, options?: MethodOptions): AxiosPromise<Schema$Operation>;
        retry(params: Params$Resource$Projects$Builds$Retry, options: MethodOptions | BodyResponseCallback<Schema$Operation>, callback: BodyResponseCallback<Schema$Operation>): void;
        retry(params: Params$Resource$Projects$Builds$Retry, callback: BodyResponseCallback<Schema$Operation>): void;
        retry(callback: BodyResponseCallback<Schema$Operation>): void;
    }
    interface Params$Resource$Projects$Builds$Cancel {
        /**
         * Auth client or API Key for the request
         */
        auth?: string | OAuth2Client | JWT | Compute | UserRefreshClient;
        /**
         * ID of the build.
         */
        id?: string;
        /**
         * ID of the project.
         */
        projectId?: string;
        /**
         * Request body metadata
         */
        requestBody?: Schema$CancelBuildRequest;
    }
    interface Params$Resource$Projects$Builds$Create {
        /**
         * Auth client or API Key for the request
         */
        auth?: string | OAuth2Client | JWT | Compute | UserRefreshClient;
        /**
         * ID of the project.
         */
        projectId?: string;
        /**
         * Request body metadata
         */
        requestBody?: Schema$Build;
    }
    interface Params$Resource$Projects$Builds$Get {
        /**
         * Auth client or API Key for the request
         */
        auth?: string | OAuth2Client | JWT | Compute | UserRefreshClient;
        /**
         * ID of the build.
         */
        id?: string;
        /**
         * ID of the project.
         */
        projectId?: string;
    }
    interface Params$Resource$Projects$Builds$List {
        /**
         * Auth client or API Key for the request
         */
        auth?: string | OAuth2Client | JWT | Compute | UserRefreshClient;
        /**
         * The raw filter text to constrain the results.
         */
        filter?: string;
        /**
         * Number of results to return in the list.
         */
        pageSize?: number;
        /**
         * Token to provide to skip to a particular spot in the list.
         */
        pageToken?: string;
        /**
         * ID of the project.
         */
        projectId?: string;
    }
    interface Params$Resource$Projects$Builds$Retry {
        /**
         * Auth client or API Key for the request
         */
        auth?: string | OAuth2Client | JWT | Compute | UserRefreshClient;
        /**
         * Build ID of the original build.
         */
        id?: string;
        /**
         * ID of the project.
         */
        projectId?: string;
        /**
         * Request body metadata
         */
        requestBody?: Schema$RetryBuildRequest;
    }
    class Resource$Projects$Triggers {
        root: Cloudbuild;
        constructor(root: Cloudbuild);
        getRoot(): Cloudbuild;
        /**
         * cloudbuild.projects.triggers.create
         * @desc Creates a new `BuildTrigger`.  This API is experimental.
         * @alias cloudbuild.projects.triggers.create
         * @memberOf! ()
         *
         * @param {object} params Parameters for request
         * @param {string} params.projectId ID of the project for which to configure automatic builds.
         * @param {().BuildTrigger} params.resource Request body data
         * @param {object} [options] Optionally override request options, such as `url`, `method`, and `encoding`.
         * @param {callback} callback The callback that handles the response.
         * @return {object} Request object
         */
        create(params?: Params$Resource$Projects$Triggers$Create, options?: MethodOptions): AxiosPromise<Schema$BuildTrigger>;
        create(params: Params$Resource$Projects$Triggers$Create, options: MethodOptions | BodyResponseCallback<Schema$BuildTrigger>, callback: BodyResponseCallback<Schema$BuildTrigger>): void;
        create(params: Params$Resource$Projects$Triggers$Create, callback: BodyResponseCallback<Schema$BuildTrigger>): void;
        create(callback: BodyResponseCallback<Schema$BuildTrigger>): void;
        /**
         * cloudbuild.projects.triggers.delete
         * @desc Deletes a `BuildTrigger` by its project ID and trigger ID.  This
         * API is experimental.
         * @alias cloudbuild.projects.triggers.delete
         * @memberOf! ()
         *
         * @param {object} params Parameters for request
         * @param {string} params.projectId ID of the project that owns the trigger.
         * @param {string} params.triggerId ID of the `BuildTrigger` to delete.
         * @param {object} [options] Optionally override request options, such as `url`, `method`, and `encoding`.
         * @param {callback} callback The callback that handles the response.
         * @return {object} Request object
         */
        delete(params?: Params$Resource$Projects$Triggers$Delete, options?: MethodOptions): AxiosPromise<Schema$Empty>;
        delete(params: Params$Resource$Projects$Triggers$Delete, options: MethodOptions | BodyResponseCallback<Schema$Empty>, callback: BodyResponseCallback<Schema$Empty>): void;
        delete(params: Params$Resource$Projects$Triggers$Delete, callback: BodyResponseCallback<Schema$Empty>): void;
        delete(callback: BodyResponseCallback<Schema$Empty>): void;
        /**
         * cloudbuild.projects.triggers.get
         * @desc Returns information about a `BuildTrigger`.  This API is
         * experimental.
         * @alias cloudbuild.projects.triggers.get
         * @memberOf! ()
         *
         * @param {object} params Parameters for request
         * @param {string} params.projectId ID of the project that owns the trigger.
         * @param {string} params.triggerId ID of the `BuildTrigger` to get.
         * @param {object} [options] Optionally override request options, such as `url`, `method`, and `encoding`.
         * @param {callback} callback The callback that handles the response.
         * @return {object} Request object
         */
        get(params?: Params$Resource$Projects$Triggers$Get, options?: MethodOptions): AxiosPromise<Schema$BuildTrigger>;
        get(params: Params$Resource$Projects$Triggers$Get, options: MethodOptions | BodyResponseCallback<Schema$BuildTrigger>, callback: BodyResponseCallback<Schema$BuildTrigger>): void;
        get(params: Params$Resource$Projects$Triggers$Get, callback: BodyResponseCallback<Schema$BuildTrigger>): void;
        get(callback: BodyResponseCallback<Schema$BuildTrigger>): void;
        /**
         * cloudbuild.projects.triggers.list
         * @desc Lists existing `BuildTrigger`s.  This API is experimental.
         * @alias cloudbuild.projects.triggers.list
         * @memberOf! ()
         *
         * @param {object} params Parameters for request
         * @param {string} params.projectId ID of the project for which to list BuildTriggers.
         * @param {object} [options] Optionally override request options, such as `url`, `method`, and `encoding`.
         * @param {callback} callback The callback that handles the response.
         * @return {object} Request object
         */
        list(params?: Params$Resource$Projects$Triggers$List, options?: MethodOptions): AxiosPromise<Schema$ListBuildTriggersResponse>;
        list(params: Params$Resource$Projects$Triggers$List, options: MethodOptions | BodyResponseCallback<Schema$ListBuildTriggersResponse>, callback: BodyResponseCallback<Schema$ListBuildTriggersResponse>): void;
        list(params: Params$Resource$Projects$Triggers$List, callback: BodyResponseCallback<Schema$ListBuildTriggersResponse>): void;
        list(callback: BodyResponseCallback<Schema$ListBuildTriggersResponse>): void;
        /**
         * cloudbuild.projects.triggers.patch
         * @desc Updates a `BuildTrigger` by its project ID and trigger ID.  This
         * API is experimental.
         * @alias cloudbuild.projects.triggers.patch
         * @memberOf! ()
         *
         * @param {object} params Parameters for request
         * @param {string} params.projectId ID of the project that owns the trigger.
         * @param {string} params.triggerId ID of the `BuildTrigger` to update.
         * @param {().BuildTrigger} params.resource Request body data
         * @param {object} [options] Optionally override request options, such as `url`, `method`, and `encoding`.
         * @param {callback} callback The callback that handles the response.
         * @return {object} Request object
         */
        patch(params?: Params$Resource$Projects$Triggers$Patch, options?: MethodOptions): AxiosPromise<Schema$BuildTrigger>;
        patch(params: Params$Resource$Projects$Triggers$Patch, options: MethodOptions | BodyResponseCallback<Schema$BuildTrigger>, callback: BodyResponseCallback<Schema$BuildTrigger>): void;
        patch(params: Params$Resource$Projects$Triggers$Patch, callback: BodyResponseCallback<Schema$BuildTrigger>): void;
        patch(callback: BodyResponseCallback<Schema$BuildTrigger>): void;
        /**
         * cloudbuild.projects.triggers.run
         * @desc Runs a `BuildTrigger` at a particular source revision.
         * @alias cloudbuild.projects.triggers.run
         * @memberOf! ()
         *
         * @param {object} params Parameters for request
         * @param {string} params.projectId ID of the project.
         * @param {string} params.triggerId ID of the trigger.
         * @param {().RepoSource} params.resource Request body data
         * @param {object} [options] Optionally override request options, such as `url`, `method`, and `encoding`.
         * @param {callback} callback The callback that handles the response.
         * @return {object} Request object
         */
        run(params?: Params$Resource$Projects$Triggers$Run, options?: MethodOptions): AxiosPromise<Schema$Operation>;
        run(params: Params$Resource$Projects$Triggers$Run, options: MethodOptions | BodyResponseCallback<Schema$Operation>, callback: BodyResponseCallback<Schema$Operation>): void;
        run(params: Params$Resource$Projects$Triggers$Run, callback: BodyResponseCallback<Schema$Operation>): void;
        run(callback: BodyResponseCallback<Schema$Operation>): void;
    }
    interface Params$Resource$Projects$Triggers$Create {
        /**
         * Auth client or API Key for the request
         */
        auth?: string | OAuth2Client | JWT | Compute | UserRefreshClient;
        /**
         * ID of the project for which to configure automatic builds.
         */
        projectId?: string;
        /**
         * Request body metadata
         */
        requestBody?: Schema$BuildTrigger;
    }
    interface Params$Resource$Projects$Triggers$Delete {
        /**
         * Auth client or API Key for the request
         */
        auth?: string | OAuth2Client | JWT | Compute | UserRefreshClient;
        /**
         * ID of the project that owns the trigger.
         */
        projectId?: string;
        /**
         * ID of the `BuildTrigger` to delete.
         */
        triggerId?: string;
    }
    interface Params$Resource$Projects$Triggers$Get {
        /**
         * Auth client or API Key for the request
         */
        auth?: string | OAuth2Client | JWT | Compute | UserRefreshClient;
        /**
         * ID of the project that owns the trigger.
         */
        projectId?: string;
        /**
         * ID of the `BuildTrigger` to get.
         */
        triggerId?: string;
    }
    interface Params$Resource$Projects$Triggers$List {
        /**
         * Auth client or API Key for the request
         */
        auth?: string | OAuth2Client | JWT | Compute | UserRefreshClient;
        /**
         * ID of the project for which to list BuildTriggers.
         */
        projectId?: string;
    }
    interface Params$Resource$Projects$Triggers$Patch {
        /**
         * Auth client or API Key for the request
         */
        auth?: string | OAuth2Client | JWT | Compute | UserRefreshClient;
        /**
         * ID of the project that owns the trigger.
         */
        projectId?: string;
        /**
         * ID of the `BuildTrigger` to update.
         */
        triggerId?: string;
        /**
         * Request body metadata
         */
        requestBody?: Schema$BuildTrigger;
    }
    interface Params$Resource$Projects$Triggers$Run {
        /**
         * Auth client or API Key for the request
         */
        auth?: string | OAuth2Client | JWT | Compute | UserRefreshClient;
        /**
         * ID of the project.
         */
        projectId?: string;
        /**
         * ID of the trigger.
         */
        triggerId?: string;
        /**
         * Request body metadata
         */
        requestBody?: Schema$RepoSource;
    }
}
