"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const EventEmitter = require("events");
const getStream = require("get-stream");
const PCancelable = require("p-cancelable");
const is_1 = require("@sindresorhus/is");
const errors_1 = require("./errors");
const normalize_arguments_1 = require("./normalize-arguments");
const request_as_event_emitter_1 = require("./request-as-event-emitter");
const parseBody = (body, responseType, statusCode) => {
    if (responseType === 'json' && is_1.default.string(body)) {
        return statusCode === 204 ? '' : JSON.parse(body);
    }
    if (responseType === 'buffer' && is_1.default.string(body)) {
        return Buffer.from(body);
    }
    if (responseType === 'text') {
        return String(body);
    }
    if (responseType === 'default') {
        return body;
    }
    throw new Error(`Failed to parse body of type '${typeof body}' as '${responseType}'`);
};
function asPromise(options) {
    const proxy = new EventEmitter();
    let finalResponse;
    // @ts-ignore `.json()`, `.buffer()` and `.text()` are added later
    const promise = new PCancelable((resolve, reject, onCancel) => {
        const emitter = request_as_event_emitter_1.default(options);
        onCancel(emitter.abort);
        const emitError = async (error) => {
            try {
                for (const hook of options.hooks.beforeError) {
                    // eslint-disable-next-line no-await-in-loop
                    error = await hook(error);
                }
                reject(error);
            }
            catch (error_) {
                reject(error_);
            }
        };
        emitter.on('response', async (response) => {
            var _a;
            proxy.emit('response', response);
            try {
                response.body = await getStream(response, { encoding: options.encoding });
            }
            catch (error) {
                emitError(new errors_1.ReadError(error, options));
                return;
            }
            if ((_a = response.req) === null || _a === void 0 ? void 0 : _a.aborted) {
                // Canceled while downloading - will throw a `CancelError` or `TimeoutError` error
                return;
            }
            try {
                for (const [index, hook] of options.hooks.afterResponse.entries()) {
                    // @ts-ignore Promise is not assignable to CancelableRequest
                    // eslint-disable-next-line no-await-in-loop
                    response = await hook(response, async (updatedOptions) => {
                        updatedOptions = normalize_arguments_1.normalizeArguments(normalize_arguments_1.mergeOptions(options, {
                            ...updatedOptions,
                            retry: {
                                calculateDelay: () => 0
                            },
                            throwHttpErrors: false,
                            responseType: 'text',
                            resolveBodyOnly: false
                        }));
                        // Remove any further hooks for that request, because we'll call them anyway.
                        // The loop continues. We don't want duplicates (asPromise recursion).
                        updatedOptions.hooks.afterResponse = options.hooks.afterResponse.slice(0, index);
                        for (const hook of options.hooks.beforeRetry) {
                            // eslint-disable-next-line no-await-in-loop
                            await hook(updatedOptions);
                        }
                        const promise = asPromise(updatedOptions);
                        onCancel(() => {
                            promise.catch(() => { });
                            promise.cancel();
                        });
                        return promise;
                    });
                }
            }
            catch (error) {
                emitError(error);
                return;
            }
            const { statusCode } = response;
            finalResponse = {
                body: response.body,
                statusCode
            };
            try {
                response.body = parseBody(response.body, options.responseType, response.statusCode);
            }
            catch (error) {
                if (statusCode >= 200 && statusCode < 300) {
                    const parseError = new errors_1.ParseError(error, response, options);
                    emitError(parseError);
                    return;
                }
            }
            const limitStatusCode = options.followRedirect ? 299 : 399;
            if (statusCode !== 304 && (statusCode < 200 || statusCode > limitStatusCode)) {
                const error = new errors_1.HTTPError(response, options);
                if (!emitter.retry(error)) {
                    if (options.throwHttpErrors) {
                        emitError(error);
                        return;
                    }
                    resolve(options.resolveBodyOnly ? response.body : response);
                }
                return;
            }
            resolve(options.resolveBodyOnly ? response.body : response);
        });
        emitter.once('error', reject);
        request_as_event_emitter_1.proxyEvents(proxy, emitter);
    });
    promise.on = (name, fn) => {
        proxy.on(name, fn);
        return promise;
    };
    const shortcut = (responseType) => {
        // eslint-disable-next-line promise/prefer-await-to-then
        const newPromise = promise.then(() => parseBody(finalResponse.body, responseType, finalResponse.statusCode));
        Object.defineProperties(newPromise, Object.getOwnPropertyDescriptors(promise));
        return newPromise;
    };
    promise.json = () => {
        if (is_1.default.undefined(options.headers.accept)) {
            options.headers.accept = 'application/json';
        }
        return shortcut('json');
    };
    promise.buffer = () => shortcut('buffer');
    promise.text = () => shortcut('text');
    return promise;
}
exports.default = asPromise;
