(function (global, factory) {
  typeof exports === 'object' && typeof module !== 'undefined' ? factory(exports) :
  typeof define === 'function' && define.amd ? define('GOVUKFrontend.SkipLink', ['exports'], factory) :
  (global = typeof globalThis !== 'undefined' ? globalThis : global || self, factory((global.GOVUKFrontend = global.GOVUKFrontend || {}, global.GOVUKFrontend.SkipLink = {})));
})(this, (function (exports) { 'use strict';

  function getFragmentFromUrl(url) {
    if (url.indexOf('#') === -1) {
      return undefined;
    }
    return url.split('#').pop();
  }
  function isSupported($scope = document.body) {
    if (!$scope) {
      return false;
    }
    return $scope.classList.contains('govuk-frontend-supported');
  }

  /**
   * Schema for component config
   *
   * @typedef {object} Schema
   * @property {SchemaCondition[]} [anyOf] - List of schema conditions
   */

  /**
   * Schema condition for component config
   *
   * @typedef {object} SchemaCondition
   * @property {string[]} required - List of required config fields
   * @property {string} errorMessage - Error message when required config fields not provided
   */

  class GOVUKFrontendError extends Error {
    constructor(...args) {
      super(...args);
      this.name = 'GOVUKFrontendError';
    }
  }
  class SupportError extends GOVUKFrontendError {
    /**
     * Checks if GOV.UK Frontend is supported on this page
     *
     * @param {HTMLElement | null} [$scope] - HTML element `<body>` checked for browser support
     */
    constructor($scope = document.body) {
      super($scope ? 'GOV.UK Frontend is not supported in this browser' : 'GOV.UK Frontend initialised without `<script type="module">`');
      this.name = 'SupportError';
    }
  }
  class ElementError extends GOVUKFrontendError {
    constructor(messageOrOptions) {
      let message = typeof messageOrOptions === 'string' ? messageOrOptions : '';
      if (typeof messageOrOptions === 'object') {
        const {
          componentName,
          identifier,
          element,
          expectedType
        } = messageOrOptions;
        message = `${componentName}: ${identifier}`;
        message += element ? ` is not of type ${expectedType || 'HTMLElement'}` : ' not found';
      }
      super(message);
      this.name = 'ElementError';
    }
  }

  class GOVUKFrontendComponent {
    constructor() {
      this.checkSupport();
    }
    checkSupport() {
      if (!isSupported()) {
        throw new SupportError();
      }
    }
  }

  /**
   * Skip link component
   *
   * @preserve
   */
  class SkipLink extends GOVUKFrontendComponent {
    /**
     * @param {Element | null} $module - HTML element to use for skip link
     * @throws {ElementError} when $module is not set or the wrong type
     * @throws {ElementError} when $module.hash does not contain a hash
     * @throws {ElementError} when the linked element is missing or the wrong type
     */
    constructor($module) {
      super();
      this.$module = void 0;
      this.$linkedElement = void 0;
      this.linkedElementListener = false;
      if (!($module instanceof HTMLAnchorElement)) {
        throw new ElementError({
          componentName: 'Skip link',
          element: $module,
          expectedType: 'HTMLAnchorElement',
          identifier: 'Root element (`$module`)'
        });
      }
      this.$module = $module;
      this.$linkedElement = this.getLinkedElement();
      this.$module.addEventListener('click', () => this.focusLinkedElement());
    }
    getLinkedElement() {
      const linkedElementId = getFragmentFromUrl(this.$module.hash);
      if (!linkedElementId) {
        throw new ElementError('Skip link: Root element (`$module`) attribute (`href`) has no URL fragment');
      }
      const $linkedElement = document.getElementById(linkedElementId);
      if (!$linkedElement) {
        throw new ElementError({
          componentName: 'Skip link',
          element: $linkedElement,
          identifier: `Target content (\`id="${linkedElementId}"\`)`
        });
      }
      return $linkedElement;
    }
    focusLinkedElement() {
      if (!this.$linkedElement.getAttribute('tabindex')) {
        this.$linkedElement.setAttribute('tabindex', '-1');
        this.$linkedElement.classList.add('govuk-skip-link-focused-element');
        if (!this.linkedElementListener) {
          this.$linkedElement.addEventListener('blur', () => this.removeFocusProperties());
          this.linkedElementListener = true;
        }
      }
      this.$linkedElement.focus();
    }
    removeFocusProperties() {
      this.$linkedElement.removeAttribute('tabindex');
      this.$linkedElement.classList.remove('govuk-skip-link-focused-element');
    }
  }
  SkipLink.moduleName = 'govuk-skip-link';

  exports.SkipLink = SkipLink;

}));
//# sourceMappingURL=skip-link.bundle.js.map
