import { visit } from 'graphql';
import { meros } from 'meros';
import { createClient } from 'graphql-ws';
import { SubscriptionClient } from 'subscriptions-transport-ws';
import { isAsyncIterable, makeAsyncIterableIteratorFromSink, } from '@n1ru4l/push-pull-async-iterable-iterator';
export const isSubscriptionWithName = (document, name) => {
    let isSubscription = false;
    visit(document, {
        OperationDefinition(node) {
            if (name === node.name?.value) {
                if (node.operation === 'subscription') {
                    isSubscription = true;
                }
            }
        },
    });
    return isSubscription;
};
export const createSimpleFetcher = (options, httpFetch) => async (graphQLParams, fetcherOpts) => {
    const data = await httpFetch(options.url, {
        method: 'POST',
        body: JSON.stringify(graphQLParams),
        headers: {
            'content-type': 'application/json',
            ...options.headers,
            ...fetcherOpts?.headers,
        },
    });
    return data.json();
};
export const createWebsocketsFetcherFromUrl = (url, connectionParams) => {
    let wsClient = null;
    let legacyClient = null;
    if (url) {
        try {
            try {
                wsClient = createClient({
                    url,
                    connectionParams,
                });
                if (!wsClient) {
                    legacyClient = new SubscriptionClient(url, { connectionParams });
                }
            }
            catch (err) {
                legacyClient = new SubscriptionClient(url, { connectionParams });
            }
        }
        catch (err) {
            console.error(`Error creating websocket client for:\n${url}\n\n${err}`);
        }
    }
    if (wsClient) {
        return createWebsocketsFetcherFromClient(wsClient);
    }
    else if (legacyClient) {
        return createLegacyWebsocketsFetcher(legacyClient);
    }
    else if (url) {
        throw Error('subscriptions client failed to initialize');
    }
};
export const createWebsocketsFetcherFromClient = (wsClient) => (graphQLParams) => makeAsyncIterableIteratorFromSink(sink => wsClient.subscribe(graphQLParams, sink));
export const createLegacyWebsocketsFetcher = (legacyWsClient) => (graphQLParams) => {
    const observable = legacyWsClient.request(graphQLParams);
    return makeAsyncIterableIteratorFromSink(sink => observable.subscribe(sink).unsubscribe);
};
export const createMultipartFetcher = (options, httpFetch) => async function* (graphQLParams, fetcherOpts) {
    const response = await httpFetch(options.url, {
        method: 'POST',
        body: JSON.stringify(graphQLParams),
        headers: {
            'content-type': 'application/json',
            accept: 'application/json, multipart/mixed',
            ...options.headers,
            ...fetcherOpts?.headers,
        },
    }).then(response => meros(response, {
        multiple: true,
    }));
    if (!isAsyncIterable(response)) {
        return yield response.json();
    }
    for await (const chunk of response) {
        if (chunk.some(part => !part.json)) {
            const message = chunk.map(part => `Headers::\n${part.headers}\n\nBody::\n${part.body}`);
            throw new Error(`Expected multipart chunks to be of json type. got:\n${message}`);
        }
        yield chunk.map(part => part.body);
    }
};
//# sourceMappingURL=lib.js.map