import { useMemo, useCallback } from "react";
import { debounce } from "../utility/debounce.js";
function getDefaultTabState({
  defaultQuery,
  defaultHeaders,
  headers,
  defaultTabs,
  query,
  variables,
  storage,
  shouldPersistHeaders
}) {
  const storedState = storage == null ? void 0 : storage.get(STORAGE_KEY);
  try {
    if (!storedState) {
      throw new Error("Storage for tabs is empty");
    }
    const parsed = JSON.parse(storedState);
    const headersForHash = shouldPersistHeaders ? headers : void 0;
    if (isTabsState(parsed)) {
      const expectedHash = hashFromTabContents({
        query,
        variables,
        headers: headersForHash
      });
      let matchingTabIndex = -1;
      for (let index = 0; index < parsed.tabs.length; index++) {
        const tab = parsed.tabs[index];
        tab.hash = hashFromTabContents({
          query: tab.query,
          variables: tab.variables,
          headers: tab.headers
        });
        if (tab.hash === expectedHash) {
          matchingTabIndex = index;
        }
      }
      if (matchingTabIndex >= 0) {
        parsed.activeTabIndex = matchingTabIndex;
      } else {
        const operationName = query ? fuzzyExtractOperationName(query) : null;
        parsed.tabs.push({
          id: guid(),
          hash: expectedHash,
          title: operationName || DEFAULT_TITLE,
          query,
          variables,
          headers,
          operationName,
          response: null
        });
        parsed.activeTabIndex = parsed.tabs.length - 1;
      }
      return parsed;
    }
    throw new Error("Storage for tabs is invalid");
  } catch {
    return {
      activeTabIndex: 0,
      tabs: (defaultTabs || [{
        query: query ?? defaultQuery,
        variables,
        headers: headers ?? defaultHeaders
      }]).map(createTab)
    };
  }
}
function isTabsState(obj) {
  return obj && typeof obj === "object" && !Array.isArray(obj) && hasNumberKey(obj, "activeTabIndex") && "tabs" in obj && Array.isArray(obj.tabs) && obj.tabs.every(isTabState);
}
function isTabState(obj) {
  return obj && typeof obj === "object" && !Array.isArray(obj) && hasStringKey(obj, "id") && hasStringKey(obj, "title") && hasStringOrNullKey(obj, "query") && hasStringOrNullKey(obj, "variables") && hasStringOrNullKey(obj, "headers") && hasStringOrNullKey(obj, "operationName") && hasStringOrNullKey(obj, "response");
}
function hasNumberKey(obj, key) {
  return key in obj && typeof obj[key] === "number";
}
function hasStringKey(obj, key) {
  return key in obj && typeof obj[key] === "string";
}
function hasStringOrNullKey(obj, key) {
  return key in obj && (typeof obj[key] === "string" || obj[key] === null);
}
function useSynchronizeActiveTabValues({
  queryEditor,
  variableEditor,
  headerEditor,
  responseEditor
}) {
  return useCallback((state) => {
    const query = (queryEditor == null ? void 0 : queryEditor.getValue()) ?? null;
    const variables = (variableEditor == null ? void 0 : variableEditor.getValue()) ?? null;
    const headers = (headerEditor == null ? void 0 : headerEditor.getValue()) ?? null;
    const operationName = (queryEditor == null ? void 0 : queryEditor.operationName) ?? null;
    const response = (responseEditor == null ? void 0 : responseEditor.getValue()) ?? null;
    return setPropertiesInActiveTab(state, {
      query,
      variables,
      headers,
      response,
      operationName
    });
  }, [queryEditor, variableEditor, headerEditor, responseEditor]);
}
function serializeTabState(tabState, shouldPersistHeaders = false) {
  return JSON.stringify(tabState, (key, value) => key === "hash" || key === "response" || !shouldPersistHeaders && key === "headers" ? null : value);
}
function useStoreTabs({
  storage,
  shouldPersistHeaders
}) {
  const store = useMemo(() => debounce(500, (value) => {
    storage == null ? void 0 : storage.set(STORAGE_KEY, value);
  }), [storage]);
  return useCallback((currentState) => {
    store(serializeTabState(currentState, shouldPersistHeaders));
  }, [shouldPersistHeaders, store]);
}
function useSetEditorValues({
  queryEditor,
  variableEditor,
  headerEditor,
  responseEditor,
  defaultHeaders
}) {
  return useCallback(({
    query,
    variables,
    headers,
    response
  }) => {
    queryEditor == null ? void 0 : queryEditor.setValue(query ?? "");
    variableEditor == null ? void 0 : variableEditor.setValue(variables ?? "");
    headerEditor == null ? void 0 : headerEditor.setValue(headers ?? defaultHeaders ?? "");
    responseEditor == null ? void 0 : responseEditor.setValue(response ?? "");
  }, [headerEditor, queryEditor, responseEditor, variableEditor, defaultHeaders]);
}
function createTab({
  query = null,
  variables = null,
  headers = null
} = {}) {
  const operationName = query ? fuzzyExtractOperationName(query) : null;
  return {
    id: guid(),
    hash: hashFromTabContents({
      query,
      variables,
      headers
    }),
    title: operationName || DEFAULT_TITLE,
    query,
    variables,
    headers,
    operationName,
    response: null
  };
}
function setPropertiesInActiveTab(state, partialTab) {
  return {
    ...state,
    tabs: state.tabs.map((tab, index) => {
      if (index !== state.activeTabIndex) {
        return tab;
      }
      const newTab = {
        ...tab,
        ...partialTab
      };
      return {
        ...newTab,
        hash: hashFromTabContents(newTab),
        title: newTab.operationName || (newTab.query ? fuzzyExtractOperationName(newTab.query) : void 0) || DEFAULT_TITLE
      };
    })
  };
}
function guid() {
  const s4 = () => {
    return Math.floor((1 + Math.random()) * 65536).toString(16).slice(1);
  };
  return `${s4()}${s4()}-${s4()}-${s4()}-${s4()}-${s4()}${s4()}${s4()}`;
}
function hashFromTabContents(args) {
  return [args.query ?? "", args.variables ?? "", args.headers ?? ""].join("|");
}
function fuzzyExtractOperationName(str) {
  const regex = /^(?!#).*(query|subscription|mutation)\s+([a-zA-Z0-9_]+)/m;
  const match = regex.exec(str);
  return (match == null ? void 0 : match[2]) ?? null;
}
function clearHeadersFromTabs(storage) {
  const persistedTabs = storage == null ? void 0 : storage.get(STORAGE_KEY);
  if (persistedTabs) {
    const parsedTabs = JSON.parse(persistedTabs);
    storage == null ? void 0 : storage.set(STORAGE_KEY, JSON.stringify(parsedTabs, (key, value) => key === "headers" ? null : value));
  }
}
const DEFAULT_TITLE = "<untitled>";
const STORAGE_KEY = "tabState";
export {
  STORAGE_KEY,
  clearHeadersFromTabs,
  createTab,
  fuzzyExtractOperationName,
  getDefaultTabState,
  serializeTabState,
  setPropertiesInActiveTab,
  useSetEditorValues,
  useStoreTabs,
  useSynchronizeActiveTabValues
};
//# sourceMappingURL=tabs.js.map
