import { mergeAst } from "@graphiql/toolkit";
import { print } from "graphql";
import { storageStore } from "./storage.js";
import { serializeTabState, clearHeadersFromTabs, setPropertiesInActiveTab, createTab } from "../utility/tabs.js";
import { STORAGE_KEY } from "../constants.js";
import { formatJSONC } from "../utility/jsonc.js";
import { debounce } from "../utility/debounce.js";
const createEditorSlice = (initial) => (set, get) => {
  function setEditorValues({
    query,
    variables,
    headers,
    response
  }) {
    const {
      queryEditor,
      variableEditor,
      headerEditor,
      responseEditor,
      defaultHeaders
    } = get();
    queryEditor == null ? void 0 : queryEditor.setValue(query ?? "");
    variableEditor == null ? void 0 : variableEditor.setValue(variables ?? "");
    headerEditor == null ? void 0 : headerEditor.setValue(headers ?? defaultHeaders ?? "");
    responseEditor == null ? void 0 : responseEditor.setValue(response ?? "");
  }
  function synchronizeActiveTabValues(tabsState) {
    const {
      queryEditor,
      variableEditor,
      headerEditor,
      responseEditor,
      operationName
    } = get();
    return setPropertiesInActiveTab(tabsState, {
      query: (queryEditor == null ? void 0 : queryEditor.getValue()) ?? null,
      variables: (variableEditor == null ? void 0 : variableEditor.getValue()) ?? null,
      headers: (headerEditor == null ? void 0 : headerEditor.getValue()) ?? null,
      response: (responseEditor == null ? void 0 : responseEditor.getValue()) ?? null,
      operationName: operationName ?? null
    });
  }
  const $actions = {
    addTab() {
      set(({
        defaultHeaders,
        onTabChange,
        tabs,
        activeTabIndex,
        actions
      }) => {
        const updatedValues = synchronizeActiveTabValues({
          tabs,
          activeTabIndex
        });
        const updated = {
          tabs: [...updatedValues.tabs, createTab({
            headers: defaultHeaders
          })],
          activeTabIndex: updatedValues.tabs.length
        };
        actions.storeTabs(updated);
        setEditorValues(updated.tabs[updated.activeTabIndex]);
        onTabChange == null ? void 0 : onTabChange(updated);
        return updated;
      });
    },
    changeTab(index) {
      set(({
        actions,
        onTabChange,
        tabs
      }) => {
        actions.stop();
        const updated = {
          tabs,
          activeTabIndex: index
        };
        actions.storeTabs(updated);
        setEditorValues(updated.tabs[updated.activeTabIndex]);
        onTabChange == null ? void 0 : onTabChange(updated);
        return updated;
      });
    },
    moveTab(newOrder) {
      set(({
        onTabChange,
        actions,
        tabs,
        activeTabIndex
      }) => {
        const activeTab = tabs[activeTabIndex];
        const updated = {
          tabs: newOrder,
          activeTabIndex: newOrder.indexOf(activeTab)
        };
        actions.storeTabs(updated);
        setEditorValues(updated.tabs[updated.activeTabIndex]);
        onTabChange == null ? void 0 : onTabChange(updated);
        return updated;
      });
    },
    closeTab(index) {
      set(({
        activeTabIndex,
        onTabChange,
        actions,
        tabs
      }) => {
        if (activeTabIndex === index) {
          actions.stop();
        }
        const updated = {
          tabs: tabs.filter((_tab, i) => index !== i),
          activeTabIndex: Math.max(activeTabIndex - 1, 0)
        };
        actions.storeTabs(updated);
        setEditorValues(updated.tabs[updated.activeTabIndex]);
        onTabChange == null ? void 0 : onTabChange(updated);
        return updated;
      });
    },
    updateActiveTabValues(partialTab) {
      set(({
        activeTabIndex,
        tabs,
        onTabChange,
        actions
      }) => {
        const updated = setPropertiesInActiveTab({
          tabs,
          activeTabIndex
        }, partialTab);
        actions.storeTabs(updated);
        onTabChange == null ? void 0 : onTabChange(updated);
        return updated;
      });
    },
    setEditor({
      headerEditor,
      queryEditor,
      responseEditor,
      variableEditor
    }) {
      const entries = Object.entries({
        headerEditor,
        queryEditor,
        responseEditor,
        variableEditor
      }).filter(([_key, value]) => value);
      const newState = Object.fromEntries(entries);
      set(newState);
    },
    setOperationName(operationName) {
      set(({
        onEditOperationName,
        actions
      }) => {
        actions.updateActiveTabValues({
          operationName
        });
        onEditOperationName == null ? void 0 : onEditOperationName(operationName);
        return {
          operationName
        };
      });
    },
    setShouldPersistHeaders(persist) {
      const {
        headerEditor,
        tabs,
        activeTabIndex
      } = get();
      const {
        storage
      } = storageStore.getState();
      if (persist) {
        storage.set(STORAGE_KEY.headers, (headerEditor == null ? void 0 : headerEditor.getValue()) ?? "");
        const serializedTabs = serializeTabState({
          tabs,
          activeTabIndex
        }, true);
        storage.set(STORAGE_KEY.tabs, serializedTabs);
      } else {
        storage.set(STORAGE_KEY.headers, "");
        clearHeadersFromTabs();
      }
      storage.set(STORAGE_KEY.persistHeaders, persist.toString());
      set({
        shouldPersistHeaders: persist
      });
    },
    storeTabs({
      tabs,
      activeTabIndex
    }) {
      const {
        storage
      } = storageStore.getState();
      const {
        shouldPersistHeaders
      } = get();
      const store = debounce(500, (value) => {
        storage.set(STORAGE_KEY.tabs, value);
      });
      store(serializeTabState({
        tabs,
        activeTabIndex
      }, shouldPersistHeaders));
    },
    setOperationFacts({
      documentAST,
      operationName,
      operations
    }) {
      set({
        documentAST,
        operationName,
        operations
      });
    },
    async copyQuery() {
      const {
        queryEditor,
        onCopyQuery
      } = get();
      if (!queryEditor) {
        return;
      }
      const query = queryEditor.getValue();
      onCopyQuery == null ? void 0 : onCopyQuery(query);
      try {
        await navigator.clipboard.writeText(query);
      } catch (error) {
        console.warn("Failed to copy query!", error);
      }
    },
    async prettifyEditors() {
      const {
        queryEditor,
        headerEditor,
        variableEditor,
        onPrettifyQuery
      } = get();
      if (variableEditor) {
        try {
          const content = variableEditor.getValue();
          const formatted = await formatJSONC(content);
          if (formatted !== content) {
            variableEditor.setValue(formatted);
          }
        } catch (error) {
          console.warn("Parsing variables JSON failed, skip prettification.", error);
        }
      }
      if (headerEditor) {
        try {
          const content = headerEditor.getValue();
          const formatted = await formatJSONC(content);
          if (formatted !== content) {
            headerEditor.setValue(formatted);
          }
        } catch (error) {
          console.warn("Parsing headers JSON failed, skip prettification.", error);
        }
      }
      if (!queryEditor) {
        return;
      }
      try {
        const content = queryEditor.getValue();
        const formatted = await onPrettifyQuery(content);
        if (formatted !== content) {
          queryEditor.setValue(formatted);
        }
      } catch (error) {
        console.warn("Parsing query failed, skip prettification.", error);
      }
    },
    mergeQuery() {
      const {
        queryEditor,
        documentAST,
        schema
      } = get();
      const query = queryEditor == null ? void 0 : queryEditor.getValue();
      if (!documentAST || !query) {
        return;
      }
      queryEditor.setValue(print(mergeAst(documentAST, schema)));
    }
  };
  return {
    ...initial,
    actions: $actions
  };
};
export {
  createEditorSlice
};
//# sourceMappingURL=editor.js.map
