import { STORAGE_KEY } from "../constants.js";
import { storageStore } from "../stores/storage.js";
function getDefaultTabState({
  defaultQuery,
  defaultHeaders,
  headers,
  query,
  variables,
  defaultTabs = [{
    query: query ?? defaultQuery,
    variables,
    headers: headers ?? defaultHeaders
  }],
  shouldPersistHeaders
}) {
  const {
    storage
  } = storageStore.getState();
  const storedState = storage.get(STORAGE_KEY.tabs);
  try {
    if (!storedState) {
      throw new Error("Storage for tabs is empty");
    }
    const parsed = JSON.parse(storedState);
    const headersForHash = shouldPersistHeaders ? headers : void 0;
    if (isTabsState(parsed)) {
      const expectedHash = hashFromTabContents({
        query,
        variables,
        headers: headersForHash
      });
      let matchingTabIndex = -1;
      for (let index = 0; index < parsed.tabs.length; index++) {
        const tab = parsed.tabs[index];
        tab.hash = hashFromTabContents({
          query: tab.query,
          variables: tab.variables,
          headers: tab.headers
        });
        if (tab.hash === expectedHash) {
          matchingTabIndex = index;
        }
      }
      if (matchingTabIndex >= 0) {
        parsed.activeTabIndex = matchingTabIndex;
      } else {
        const operationName = query ? fuzzyExtractOperationName(query) : null;
        parsed.tabs.push({
          id: guid(),
          hash: expectedHash,
          title: operationName || DEFAULT_TITLE,
          query,
          variables,
          headers,
          operationName,
          response: null
        });
        parsed.activeTabIndex = parsed.tabs.length - 1;
      }
      return parsed;
    }
    throw new Error("Storage for tabs is invalid");
  } catch {
    return {
      activeTabIndex: 0,
      tabs: defaultTabs.map(createTab)
    };
  }
}
function isTabsState(obj) {
  return obj && typeof obj === "object" && !Array.isArray(obj) && hasNumberKey(obj, "activeTabIndex") && "tabs" in obj && Array.isArray(obj.tabs) && obj.tabs.every(isTabState);
}
function isTabState(obj) {
  return obj && typeof obj === "object" && !Array.isArray(obj) && hasStringKey(obj, "id") && hasStringKey(obj, "title") && hasStringOrNullKey(obj, "query") && hasStringOrNullKey(obj, "variables") && hasStringOrNullKey(obj, "headers") && hasStringOrNullKey(obj, "operationName") && hasStringOrNullKey(obj, "response");
}
function hasNumberKey(obj, key) {
  return key in obj && typeof obj[key] === "number";
}
function hasStringKey(obj, key) {
  return key in obj && typeof obj[key] === "string";
}
function hasStringOrNullKey(obj, key) {
  return key in obj && (typeof obj[key] === "string" || obj[key] === null);
}
function serializeTabState(tabState, shouldPersistHeaders = false) {
  return JSON.stringify(tabState, (key, value) => key === "hash" || key === "response" || !shouldPersistHeaders && key === "headers" ? null : value);
}
function createTab({
  query = null,
  variables = null,
  headers = null
} = {}) {
  const operationName = query ? fuzzyExtractOperationName(query) : null;
  return {
    id: guid(),
    hash: hashFromTabContents({
      query,
      variables,
      headers
    }),
    title: operationName || DEFAULT_TITLE,
    query,
    variables,
    headers,
    operationName,
    response: null
  };
}
function setPropertiesInActiveTab(state, partialTab) {
  return {
    ...state,
    tabs: state.tabs.map((tab, index) => {
      if (index !== state.activeTabIndex) {
        return tab;
      }
      const newTab = {
        ...tab,
        ...partialTab
      };
      return {
        ...newTab,
        hash: hashFromTabContents(newTab),
        title: newTab.operationName || (newTab.query ? fuzzyExtractOperationName(newTab.query) : void 0) || DEFAULT_TITLE
      };
    })
  };
}
function guid() {
  const s4 = () => {
    return Math.floor((1 + Math.random()) * 65536).toString(16).slice(1);
  };
  return `${s4()}${s4()}-${s4()}-${s4()}-${s4()}-${s4()}${s4()}${s4()}`;
}
function hashFromTabContents(args) {
  return [args.query ?? "", args.variables ?? "", args.headers ?? ""].join("|");
}
function fuzzyExtractOperationName(str) {
  const regex = /^(?!#).*(query|subscription|mutation)\s+([a-zA-Z0-9_]+)/m;
  const match = regex.exec(str);
  return (match == null ? void 0 : match[2]) ?? null;
}
function clearHeadersFromTabs() {
  const {
    storage
  } = storageStore.getState();
  const persistedTabs = storage.get(STORAGE_KEY.tabs);
  if (persistedTabs) {
    const parsedTabs = JSON.parse(persistedTabs);
    storage.set(STORAGE_KEY.tabs, JSON.stringify(parsedTabs, (key, value) => key === "headers" ? null : value));
  }
}
const DEFAULT_TITLE = "<untitled>";
export {
  clearHeadersFromTabs,
  createTab,
  fuzzyExtractOperationName,
  getDefaultTabState,
  serializeTabState,
  setPropertiesInActiveTab
};
//# sourceMappingURL=tabs.js.map
