"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
var lodash_1 = require("lodash");
var builtinTypes = ['String', 'Float', 'Int', 'Boolean', 'ID'];
/**
 * Post processing of all imported type definitions. Loops over each of the
 * imported type definitions, and processes it using collectNewTypeDefinitions.
 *
 * @param allDefinitions All definitions from all schemas
 * @param definitionPool Current definitions (from first schema)
 * @param newTypeDefinitions All imported definitions
 * @returns Final collection of type definitions for the resulting schema
 */
function completeDefinitionPool(allDefinitions, definitionPool, newTypeDefinitions) {
    var visitedDefinitions = {};
    while (newTypeDefinitions.length > 0) {
        var schemaMap = lodash_1.keyBy(allDefinitions, function (d) { return d.name.value; });
        var newDefinition = newTypeDefinitions.shift();
        if (visitedDefinitions[newDefinition.name.value]) {
            continue;
        }
        var collectedTypedDefinitions = collectNewTypeDefinitions(allDefinitions, definitionPool, newDefinition, schemaMap);
        newTypeDefinitions.push.apply(newTypeDefinitions, collectedTypedDefinitions);
        definitionPool.push.apply(definitionPool, collectedTypedDefinitions);
        visitedDefinitions[newDefinition.name.value] = true;
    }
    return lodash_1.uniqBy(definitionPool, 'name.value');
}
exports.completeDefinitionPool = completeDefinitionPool;
/**
 * Processes a single type definition, and performs a number of checks:
 * - Add missing interface implementations
 * - Add missing referenced types
 * - Remove unused type definitions
 *
 * @param allDefinitions All definitions from all schemas
 * (only used to find missing interface implementations)
 * @param definitionPool Resulting definitions
 * @param newDefinition All imported definitions
 * @param schemaMap Map of all definitions for easy lookup
 * @returns All relevant type definitions to add to the final schema
 */
function collectNewTypeDefinitions(allDefinitions, definitionPool, newDefinition, schemaMap) {
    var newTypeDefinitions = [];
    if (newDefinition.kind === 'InputObjectTypeDefinition') {
        newDefinition.fields.forEach(function (field) {
            var namedType = getNamedType(field.type);
            var typeName = namedType.name.value;
            // collect missing argument input types
            if (!definitionPool.some(function (d) { return d.name.value === typeName; }) &&
                !builtinTypes.includes(typeName)) {
                var argTypeMatch = schemaMap[typeName];
                if (!argTypeMatch) {
                    throw new Error("Field " + field.name.value + ": Couldn't find type " + typeName + " in any of the schemas.");
                }
                newTypeDefinitions.push(argTypeMatch);
            }
        });
    }
    if (newDefinition.kind === 'InterfaceTypeDefinition') {
        var interfaceName_1 = newDefinition.name.value;
        newDefinition.fields.forEach(function (field) {
            var namedType = getNamedType(field.type);
            var typeName = namedType.name.value;
            if (!definitionPool.some(function (d) { return d.name.value === typeName; }) &&
                !builtinTypes.includes(typeName)) {
                var schemaType = schemaMap[typeName];
                if (!schemaType) {
                    throw new Error("Field " + field.name.value + ": Couldn't find type " + typeName + " in any of the schemas.");
                }
                newTypeDefinitions.push(schemaType);
            }
        });
        var interfaceImplementations = allDefinitions.filter(function (d) {
            return d.kind === 'ObjectTypeDefinition' &&
                d.interfaces.some(function (i) { return i.name.value === interfaceName_1; });
        });
        newTypeDefinitions.push.apply(newTypeDefinitions, interfaceImplementations);
    }
    if (newDefinition.kind === 'UnionTypeDefinition') {
        newDefinition.types.forEach(function (type) {
            if (!definitionPool.some(function (d) { return d.name.value === type.name.value; })) {
                var typeName = type.name.value;
                var typeMatch = schemaMap[typeName];
                if (!typeMatch) {
                    throw new Error("Couldn't find type " + typeName + " in any of the schemas.");
                }
                newTypeDefinitions.push(schemaMap[type.name.value]);
            }
        });
    }
    if (newDefinition.kind === 'ObjectTypeDefinition') {
        // collect missing interfaces
        newDefinition.interfaces.forEach(function (int) {
            if (!definitionPool.some(function (d) { return d.name.value === int.name.value; })) {
                var interfaceName = int.name.value;
                var interfaceMatch = schemaMap[interfaceName];
                if (!interfaceMatch) {
                    throw new Error("Couldn't find interface " + interfaceName + " in any of the schemas.");
                }
                newTypeDefinitions.push(schemaMap[int.name.value]);
            }
        });
        // iterate over all fields
        newDefinition.fields.forEach(function (field) {
            var namedType = getNamedType(field.type);
            var typeName = namedType.name.value;
            // collect missing argument input types
            field.arguments.forEach(function (argument) {
                var argType = getNamedType(argument.type);
                var argTypeName = argType.name.value;
                if (!definitionPool.some(function (d) { return d.name.value === argTypeName; }) &&
                    !builtinTypes.includes(argTypeName)) {
                    var argTypeMatch = schemaMap[argTypeName];
                    if (!argTypeMatch) {
                        throw new Error("Field " + field.name.value + ": Couldn't find type " + argTypeName + " in any of the schemas.");
                    }
                    newTypeDefinitions.push(argTypeMatch);
                }
            });
            // collect missing field types
            if (!definitionPool.some(function (d) { return d.name.value === typeName; }) &&
                !builtinTypes.includes(typeName)) {
                var schemaType = schemaMap[typeName];
                if (!schemaType) {
                    throw new Error("Field " + field.name.value + ": Couldn't find type " + typeName + " in any of the schemas.");
                }
                newTypeDefinitions.push(schemaType);
            }
        });
    }
    return newTypeDefinitions;
}
/**
 * Nested visitor for a type node to get to the final NamedType
 *
 * @param {TypeNode} type Type node to get NamedTypeNode for
 * @returns {NamedTypeNode} The found NamedTypeNode
 */
function getNamedType(type) {
    if (type.kind === 'NamedType') {
        return type;
    }
    else {
        return getNamedType(type.type);
    }
}
//# sourceMappingURL=definition.js.map