"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
var fs = require("fs");
var graphql_1 = require("graphql");
var lodash_1 = require("lodash");
var path = require("path");
var definition_1 = require("./definition");
/**
 * Read a schema file from disk
 *
 * @param f Filename
 * @returns File contents
 */
var read = function (f) { return fs.readFileSync(f, { encoding: 'utf8' }); };
/**
 * Parse a single import line and extract imported types and schema filename
 *
 * @param importLine Import line
 * @returns Processed import line
 */
function parseImportLine(importLine) {
    // Apply regex to import line
    var matches = importLine.match(/^import (\*|(.*)) from ('|")(.*)('|")$/);
    if (matches.length !== 6) {
        throw new Error("Too few regex matches: " + matches);
    }
    // Extract matches into named variables
    var wildcard = matches[1], importsString = matches[2], from = matches[4];
    // Extract imported types
    var imports = wildcard === '*' ? ['*'] : importsString.split(',').map(function (d) { return d.trim(); });
    // Return information about the import line
    return { imports: imports, from: from };
}
exports.parseImportLine = parseImportLine;
/**
 * Parse a schema and analyze all import lines
 *
 * @param sdl Schema to parse
 * @returns Array with collection of imports per import line (file)
 */
function parseSDL(sdl) {
    return sdl
        .split('\n')
        .map(function (l) { return l.trim(); })
        .filter(function (l) { return l.startsWith('# import') || l.startsWith('#import'); })
        .map(function (l) { return l.replace('#', '').trim(); })
        .map(parseImportLine);
}
exports.parseSDL = parseSDL;
/**
 * Main entry point. Recursively process all import statement in a schema
 *
 * @param filePath File path to the initial schema file
 * @returns Single bundled schema with all imported types
 */
function importSchema(filePath) {
    var sdl = read(filePath);
    var document = graphql_1.parse(sdl, { noLocation: true });
    // Recursively process the imports, starting by importing all types from the initial schema
    var _a = collectDefinitions(['*'], sdl, path.resolve(filePath)), allDefinitions = _a.allDefinitions, typeDefinitions = _a.typeDefinitions;
    // Post processing of the final schema (missing types, unused types, etc.)
    document.definitions = definition_1.completeDefinitionPool(lodash_1.flatten(allDefinitions), typeDefinitions[0], lodash_1.flatten(typeDefinitions));
    // Return the schema as string
    return graphql_1.print(document);
}
exports.importSchema = importSchema;
/**
 * Recursively process all schema files. Keeps track of both the filtered
 * type definitions, and all type definitions, because they might be needed
 * in post-processing (to add missing types)
 *
 * @param imports Types specified in the import statement
 * @param sdl Current schema
 * @param filePath File location for current schema
 * @param Tracking of processed schemas (for circular dependencies)
 * @param Tracking of imported type definitions per schema
 * @param Tracking of all type definitions per schema
 * @returns Both the collection of all type definitions, and the collection of imported type definitions
 */
function collectDefinitions(imports, sdl, filePath, processedFiles, typeDefinitions, allDefinitions) {
    if (processedFiles === void 0) { processedFiles = new Set(); }
    if (typeDefinitions === void 0) { typeDefinitions = []; }
    if (allDefinitions === void 0) { allDefinitions = []; }
    var key = path.resolve(filePath);
    var dirname = path.dirname(filePath);
    // Get TypeDefinitionNodes from current schema
    var document = graphql_1.parse(sdl);
    // Add all definitions to running total
    allDefinitions.push(filterTypeDefinitions(document.definitions));
    // Filter TypeDefinitionNodes by type and defined imports
    var currentTypeDefinitions = filterImportedDefinitions(imports, document.definitions);
    // Add typedefinitions to running total
    typeDefinitions.push(currentTypeDefinitions);
    // Mark file as processed (for circular dependency cases)
    processedFiles.add(key);
    // Read imports from current file
    var rawModules = parseSDL(sdl);
    // Process each file (recursively)
    rawModules.forEach(function (m) {
        // If it was not yet processed (in case of circular dependencies)
        var moduleFilePath = path.resolve(path.join(dirname, m.from));
        if (!processedFiles.has(moduleFilePath)) {
            collectDefinitions(m.imports, read(moduleFilePath), moduleFilePath, processedFiles, typeDefinitions, allDefinitions);
        }
    });
    // Return the maps of type definitions from each file
    return { allDefinitions: allDefinitions, typeDefinitions: typeDefinitions };
}
/**
 * Filter the types loaded from a schema, first by relevant types,
 * then by the types specified in the import statement.
 *
 * @param imports Types specified in the import statement
 * @param typeDefinitions All definitions from a schema
 * @returns Filtered collection of type definitions
 */
function filterImportedDefinitions(imports, typeDefinitions) {
    var filteredDefinitions = filterTypeDefinitions(typeDefinitions);
    if (imports.includes('*')) {
        return filteredDefinitions;
    }
    else {
        return filteredDefinitions.filter(function (d) { return imports.includes(d.name.value); });
    }
}
/**
 * Filter relevant definitions from schema
 *
 * @param definitions All definitions from a schema
 * @returns Relevant type definitions
 */
function filterTypeDefinitions(definitions) {
    var validKinds = [
        'ScalarTypeDefinition',
        'ObjectTypeDefinition',
        'InterfaceTypeDefinition',
        'EnumTypeDefinition',
        'UnionTypeDefinition',
        'InputObjectTypeDefinition'
    ];
    return definitions
        .filter(function (d) { return validKinds.includes(d.kind); })
        .map(function (d) { return d; });
}
//# sourceMappingURL=index.js.map