"use strict";
var __assign = (this && this.__assign) || function () {
    __assign = Object.assign || function(t) {
        for (var s, i = 1, n = arguments.length; i < n; i++) {
            s = arguments[i];
            for (var p in s) if (Object.prototype.hasOwnProperty.call(s, p))
                t[p] = s[p];
        }
        return t;
    };
    return __assign.apply(this, arguments);
};
Object.defineProperty(exports, "__esModule", { value: true });
var vscode_languageserver_types_1 = require("vscode-languageserver-types");
var graphql_1 = require("graphql");
var graphql_language_service_parser_1 = require("graphql-language-service-parser");
var autocompleteUtils_1 = require("./autocompleteUtils");
function getAutocompleteSuggestions(schema, queryText, cursor, contextToken) {
    var token = contextToken || getTokenAtPosition(queryText, cursor);
    var state = token.state.kind === 'Invalid' ? token.state.prevState : token.state;
    if (!state) {
        return [];
    }
    var kind = state.kind;
    var step = state.step;
    var typeInfo = getTypeInfo(schema, token.state);
    if (kind === graphql_language_service_parser_1.RuleKinds.DOCUMENT) {
        return autocompleteUtils_1.hintList(token, [
            { label: 'query', kind: vscode_languageserver_types_1.CompletionItemKind.Function },
            { label: 'mutation', kind: vscode_languageserver_types_1.CompletionItemKind.Function },
            { label: 'subscription', kind: vscode_languageserver_types_1.CompletionItemKind.Function },
            { label: 'fragment', kind: vscode_languageserver_types_1.CompletionItemKind.Function },
            { label: '{', kind: vscode_languageserver_types_1.CompletionItemKind.Constructor },
        ]);
    }
    if (kind === graphql_language_service_parser_1.RuleKinds.SELECTION_SET ||
        kind === graphql_language_service_parser_1.RuleKinds.FIELD ||
        kind === graphql_language_service_parser_1.RuleKinds.ALIASED_FIELD) {
        return getSuggestionsForFieldNames(token, typeInfo, schema, kind);
    }
    if (kind === graphql_language_service_parser_1.RuleKinds.ARGUMENTS ||
        (kind === graphql_language_service_parser_1.RuleKinds.ARGUMENT && step === 0)) {
        var argDefs = typeInfo.argDefs;
        if (argDefs) {
            return autocompleteUtils_1.hintList(token, argDefs.map(function (argDef) {
                var _a;
                return ({
                    label: argDef.name,
                    detail: String(argDef.type),
                    documentation: (_a = argDef.description) !== null && _a !== void 0 ? _a : undefined,
                    kind: vscode_languageserver_types_1.CompletionItemKind.Variable,
                });
            }));
        }
    }
    if (kind === graphql_language_service_parser_1.RuleKinds.OBJECT_VALUE ||
        (kind === graphql_language_service_parser_1.RuleKinds.OBJECT_FIELD && step === 0)) {
        if (typeInfo.objectFieldDefs) {
            var objectFields = autocompleteUtils_1.objectValues(typeInfo.objectFieldDefs);
            var completionKind_1 = kind === graphql_language_service_parser_1.RuleKinds.OBJECT_VALUE
                ? vscode_languageserver_types_1.CompletionItemKind.Value
                : vscode_languageserver_types_1.CompletionItemKind.Field;
            return autocompleteUtils_1.hintList(token, objectFields.map(function (field) {
                var _a;
                return ({
                    label: field.name,
                    detail: String(field.type),
                    documentation: (_a = field.description) !== null && _a !== void 0 ? _a : undefined,
                    kind: completionKind_1,
                });
            }));
        }
    }
    if (kind === graphql_language_service_parser_1.RuleKinds.ENUM_VALUE ||
        (kind === graphql_language_service_parser_1.RuleKinds.LIST_VALUE && step === 1) ||
        (kind === graphql_language_service_parser_1.RuleKinds.OBJECT_FIELD && step === 2) ||
        (kind === graphql_language_service_parser_1.RuleKinds.ARGUMENT && step === 2)) {
        return getSuggestionsForInputValues(token, typeInfo, kind);
    }
    if ((kind === graphql_language_service_parser_1.RuleKinds.TYPE_CONDITION && step === 1) ||
        (kind === graphql_language_service_parser_1.RuleKinds.NAMED_TYPE &&
            state.prevState != null &&
            state.prevState.kind === graphql_language_service_parser_1.RuleKinds.TYPE_CONDITION)) {
        return getSuggestionsForFragmentTypeConditions(token, typeInfo, schema, kind);
    }
    if (kind === graphql_language_service_parser_1.RuleKinds.FRAGMENT_SPREAD && step === 1) {
        return getSuggestionsForFragmentSpread(token, typeInfo, schema, queryText, kind);
    }
    if ((kind === graphql_language_service_parser_1.RuleKinds.VARIABLE_DEFINITION && step === 2) ||
        (kind === graphql_language_service_parser_1.RuleKinds.LIST_TYPE && step === 1) ||
        (kind === graphql_language_service_parser_1.RuleKinds.NAMED_TYPE &&
            state.prevState &&
            (state.prevState.kind === graphql_language_service_parser_1.RuleKinds.VARIABLE_DEFINITION ||
                state.prevState.kind === graphql_language_service_parser_1.RuleKinds.LIST_TYPE))) {
        return getSuggestionsForVariableDefinition(token, schema, kind);
    }
    if (kind === graphql_language_service_parser_1.RuleKinds.DIRECTIVE) {
        return getSuggestionsForDirective(token, state, schema, kind);
    }
    return [];
}
exports.getAutocompleteSuggestions = getAutocompleteSuggestions;
function getSuggestionsForFieldNames(token, typeInfo, schema, _kind) {
    if (typeInfo.parentType) {
        var parentType = typeInfo.parentType;
        var fields = [];
        if ('getFields' in parentType) {
            fields = autocompleteUtils_1.objectValues(parentType.getFields());
        }
        if (graphql_1.isCompositeType(parentType)) {
            fields.push(graphql_1.TypeNameMetaFieldDef);
        }
        if (parentType === schema.getQueryType()) {
            fields.push(graphql_1.SchemaMetaFieldDef, graphql_1.TypeMetaFieldDef);
        }
        return autocompleteUtils_1.hintList(token, fields.map(function (field, index) {
            var _a;
            return ({
                sortText: String(index) + field.name,
                label: field.name,
                detail: String(field.type),
                documentation: (_a = field.description) !== null && _a !== void 0 ? _a : undefined,
                deprecated: field.isDeprecated,
                isDeprecated: field.isDeprecated,
                deprecationReason: field.deprecationReason,
                kind: vscode_languageserver_types_1.CompletionItemKind.Field,
            });
        }));
    }
    return [];
}
function getSuggestionsForInputValues(token, typeInfo, _kind) {
    var namedInputType = graphql_1.getNamedType(typeInfo.inputType);
    if (namedInputType instanceof graphql_1.GraphQLEnumType) {
        var values = namedInputType.getValues();
        return autocompleteUtils_1.hintList(token, values.map(function (value) {
            var _a;
            return ({
                label: value.name,
                detail: String(namedInputType),
                documentation: (_a = value.description) !== null && _a !== void 0 ? _a : undefined,
                deprecated: value.isDeprecated,
                isDeprecated: value.isDeprecated,
                deprecationReason: value.deprecationReason,
                kind: vscode_languageserver_types_1.CompletionItemKind.EnumMember,
            });
        }));
    }
    else if (namedInputType === graphql_1.GraphQLBoolean) {
        return autocompleteUtils_1.hintList(token, [
            {
                label: 'true',
                detail: String(graphql_1.GraphQLBoolean),
                documentation: 'Not false.',
                kind: vscode_languageserver_types_1.CompletionItemKind.Variable,
            },
            {
                label: 'false',
                detail: String(graphql_1.GraphQLBoolean),
                documentation: 'Not true.',
                kind: vscode_languageserver_types_1.CompletionItemKind.Variable,
            },
        ]);
    }
    return [];
}
function getSuggestionsForFragmentTypeConditions(token, typeInfo, schema, _kind) {
    var possibleTypes;
    if (typeInfo.parentType) {
        if (graphql_1.isAbstractType(typeInfo.parentType)) {
            var abstractType = graphql_1.assertAbstractType(typeInfo.parentType);
            var possibleObjTypes = schema.getPossibleTypes(abstractType);
            var possibleIfaceMap_1 = Object.create(null);
            possibleObjTypes.forEach(function (type) {
                type.getInterfaces().forEach(function (iface) {
                    possibleIfaceMap_1[iface.name] = iface;
                });
            });
            possibleTypes = possibleObjTypes.concat(autocompleteUtils_1.objectValues(possibleIfaceMap_1));
        }
        else {
            possibleTypes = [typeInfo.parentType];
        }
    }
    else {
        var typeMap = schema.getTypeMap();
        possibleTypes = autocompleteUtils_1.objectValues(typeMap).filter(graphql_1.isCompositeType);
    }
    return autocompleteUtils_1.hintList(token, possibleTypes.map(function (type) {
        var namedType = graphql_1.getNamedType(type);
        return {
            label: String(type),
            documentation: (namedType && namedType.description) || '',
            kind: vscode_languageserver_types_1.CompletionItemKind.Field,
        };
    }));
}
function getSuggestionsForFragmentSpread(token, typeInfo, schema, queryText, _kind) {
    var typeMap = schema.getTypeMap();
    var defState = autocompleteUtils_1.getDefinitionState(token.state);
    var fragments = getFragmentDefinitions(queryText);
    var relevantFrags = fragments.filter(function (frag) {
        return typeMap[frag.typeCondition.name.value] &&
            !(defState &&
                defState.kind === graphql_language_service_parser_1.RuleKinds.FRAGMENT_DEFINITION &&
                defState.name === frag.name.value) &&
            graphql_1.isCompositeType(typeInfo.parentType) &&
            graphql_1.isCompositeType(typeMap[frag.typeCondition.name.value]) &&
            graphql_1.doTypesOverlap(schema, typeInfo.parentType, typeMap[frag.typeCondition.name.value]);
    });
    return autocompleteUtils_1.hintList(token, relevantFrags.map(function (frag) { return ({
        label: frag.name.value,
        detail: String(typeMap[frag.typeCondition.name.value]),
        documentation: "fragment " + frag.name.value + " on " + frag.typeCondition.name.value,
        kind: vscode_languageserver_types_1.CompletionItemKind.Field,
    }); }));
}
function getFragmentDefinitions(queryText) {
    var fragmentDefs = [];
    runOnlineParser(queryText, function (_, state) {
        if (state.kind === graphql_language_service_parser_1.RuleKinds.FRAGMENT_DEFINITION &&
            state.name &&
            state.type) {
            fragmentDefs.push({
                kind: graphql_language_service_parser_1.RuleKinds.FRAGMENT_DEFINITION,
                name: {
                    kind: 'Name',
                    value: state.name,
                },
                selectionSet: {
                    kind: graphql_language_service_parser_1.RuleKinds.SELECTION_SET,
                    selections: [],
                },
                typeCondition: {
                    kind: graphql_language_service_parser_1.RuleKinds.NAMED_TYPE,
                    name: {
                        kind: 'Name',
                        value: state.type,
                    },
                },
            });
        }
    });
    return fragmentDefs;
}
exports.getFragmentDefinitions = getFragmentDefinitions;
function getSuggestionsForVariableDefinition(token, schema, _kind) {
    var inputTypeMap = schema.getTypeMap();
    var inputTypes = autocompleteUtils_1.objectValues(inputTypeMap).filter(graphql_1.isInputType);
    return autocompleteUtils_1.hintList(token, inputTypes.map(function (type) { return ({
        label: type.name,
        documentation: type.description,
        kind: vscode_languageserver_types_1.CompletionItemKind.Variable,
    }); }));
}
function getSuggestionsForDirective(token, state, schema, _kind) {
    if (state.prevState && state.prevState.kind) {
        var directives = schema
            .getDirectives()
            .filter(function (directive) { return canUseDirective(state.prevState, directive); });
        return autocompleteUtils_1.hintList(token, directives.map(function (directive) { return ({
            label: directive.name,
            documentation: directive.description || '',
            kind: vscode_languageserver_types_1.CompletionItemKind.Function,
        }); }));
    }
    return [];
}
function getTokenAtPosition(queryText, cursor) {
    var styleAtCursor = null;
    var stateAtCursor = null;
    var stringAtCursor = null;
    var token = runOnlineParser(queryText, function (stream, state, style, index) {
        if (index === cursor.line) {
            if (stream.getCurrentPosition() >= cursor.character) {
                styleAtCursor = style;
                stateAtCursor = __assign({}, state);
                stringAtCursor = stream.current();
                return 'BREAK';
            }
        }
    });
    return {
        start: token.start,
        end: token.end,
        string: stringAtCursor || token.string,
        state: stateAtCursor || token.state,
        style: styleAtCursor || token.style,
    };
}
exports.getTokenAtPosition = getTokenAtPosition;
function runOnlineParser(queryText, callback) {
    var lines = queryText.split('\n');
    var parser = graphql_language_service_parser_1.onlineParser();
    var state = parser.startState();
    var style = '';
    var stream = new graphql_language_service_parser_1.CharacterStream('');
    for (var i = 0; i < lines.length; i++) {
        stream = new graphql_language_service_parser_1.CharacterStream(lines[i]);
        while (!stream.eol()) {
            style = parser.token(stream, state);
            var code = callback(stream, state, style, i);
            if (code === 'BREAK') {
                break;
            }
        }
        callback(stream, state, style, i);
        if (!state.kind) {
            state = parser.startState();
        }
    }
    return {
        start: stream.getStartOfToken(),
        end: stream.getCurrentPosition(),
        string: stream.current(),
        state: state,
        style: style,
    };
}
exports.runOnlineParser = runOnlineParser;
function canUseDirective(state, directive) {
    if (!state || !state.kind) {
        return false;
    }
    var kind = state.kind;
    var locations = directive.locations;
    switch (kind) {
        case graphql_language_service_parser_1.RuleKinds.QUERY:
            return locations.indexOf('QUERY') !== -1;
        case graphql_language_service_parser_1.RuleKinds.MUTATION:
            return locations.indexOf('MUTATION') !== -1;
        case graphql_language_service_parser_1.RuleKinds.SUBSCRIPTION:
            return locations.indexOf('SUBSCRIPTION') !== -1;
        case graphql_language_service_parser_1.RuleKinds.FIELD:
        case graphql_language_service_parser_1.RuleKinds.ALIASED_FIELD:
            return locations.indexOf('FIELD') !== -1;
        case graphql_language_service_parser_1.RuleKinds.FRAGMENT_DEFINITION:
            return locations.indexOf('FRAGMENT_DEFINITION') !== -1;
        case graphql_language_service_parser_1.RuleKinds.FRAGMENT_SPREAD:
            return locations.indexOf('FRAGMENT_SPREAD') !== -1;
        case graphql_language_service_parser_1.RuleKinds.INLINE_FRAGMENT:
            return locations.indexOf('INLINE_FRAGMENT') !== -1;
        case graphql_language_service_parser_1.RuleKinds.SCHEMA_DEF:
            return locations.indexOf('SCHEMA') !== -1;
        case graphql_language_service_parser_1.RuleKinds.SCALAR_DEF:
            return locations.indexOf('SCALAR') !== -1;
        case graphql_language_service_parser_1.RuleKinds.OBJECT_TYPE_DEF:
            return locations.indexOf('OBJECT') !== -1;
        case graphql_language_service_parser_1.RuleKinds.FIELD_DEF:
            return locations.indexOf('FIELD_DEFINITION') !== -1;
        case graphql_language_service_parser_1.RuleKinds.INTERFACE_DEF:
            return locations.indexOf('INTERFACE') !== -1;
        case graphql_language_service_parser_1.RuleKinds.UNION_DEF:
            return locations.indexOf('UNION') !== -1;
        case graphql_language_service_parser_1.RuleKinds.ENUM_DEF:
            return locations.indexOf('ENUM') !== -1;
        case graphql_language_service_parser_1.RuleKinds.ENUM_VALUE:
            return locations.indexOf('ENUM_VALUE') !== -1;
        case graphql_language_service_parser_1.RuleKinds.INPUT_DEF:
            return locations.indexOf('INPUT_OBJECT') !== -1;
        case graphql_language_service_parser_1.RuleKinds.INPUT_VALUE_DEF:
            var prevStateKind = state.prevState && state.prevState.kind;
            switch (prevStateKind) {
                case graphql_language_service_parser_1.RuleKinds.ARGUMENTS_DEF:
                    return locations.indexOf('ARGUMENT_DEFINITION') !== -1;
                case graphql_language_service_parser_1.RuleKinds.INPUT_DEF:
                    return locations.indexOf('INPUT_FIELD_DEFINITION') !== -1;
            }
    }
    return false;
}
exports.canUseDirective = canUseDirective;
function getTypeInfo(schema, tokenState) {
    var argDef;
    var argDefs;
    var directiveDef;
    var enumValue;
    var fieldDef;
    var inputType;
    var objectFieldDefs;
    var parentType;
    var type;
    autocompleteUtils_1.forEachState(tokenState, function (state) {
        switch (state.kind) {
            case graphql_language_service_parser_1.RuleKinds.QUERY:
            case 'ShortQuery':
                type = schema.getQueryType();
                break;
            case graphql_language_service_parser_1.RuleKinds.MUTATION:
                type = schema.getMutationType();
                break;
            case graphql_language_service_parser_1.RuleKinds.SUBSCRIPTION:
                type = schema.getSubscriptionType();
                break;
            case graphql_language_service_parser_1.RuleKinds.INLINE_FRAGMENT:
            case graphql_language_service_parser_1.RuleKinds.FRAGMENT_DEFINITION:
                if (state.type) {
                    type = schema.getType(state.type);
                }
                break;
            case graphql_language_service_parser_1.RuleKinds.FIELD:
            case graphql_language_service_parser_1.RuleKinds.ALIASED_FIELD:
                if (!type || !state.name) {
                    fieldDef = null;
                }
                else {
                    fieldDef = parentType
                        ? autocompleteUtils_1.getFieldDef(schema, parentType, state.name)
                        : null;
                    type = fieldDef ? fieldDef.type : null;
                }
                break;
            case graphql_language_service_parser_1.RuleKinds.SELECTION_SET:
                parentType = graphql_1.getNamedType(type);
                break;
            case graphql_language_service_parser_1.RuleKinds.DIRECTIVE:
                directiveDef = state.name ? schema.getDirective(state.name) : null;
                break;
            case graphql_language_service_parser_1.RuleKinds.ARGUMENTS:
                if (!state.prevState) {
                    argDefs = null;
                }
                else {
                    switch (state.prevState.kind) {
                        case graphql_language_service_parser_1.RuleKinds.FIELD:
                            argDefs = fieldDef && fieldDef.args;
                            break;
                        case graphql_language_service_parser_1.RuleKinds.DIRECTIVE:
                            argDefs = directiveDef && directiveDef.args;
                            break;
                        case graphql_language_service_parser_1.RuleKinds.ALIASED_FIELD:
                            var name_1 = state.prevState && state.prevState.name;
                            if (!name_1) {
                                argDefs = null;
                                break;
                            }
                            var field = parentType
                                ? autocompleteUtils_1.getFieldDef(schema, parentType, name_1)
                                : null;
                            if (!field) {
                                argDefs = null;
                                break;
                            }
                            argDefs = field.args;
                            break;
                        default:
                            argDefs = null;
                            break;
                    }
                }
                break;
            case graphql_language_service_parser_1.RuleKinds.ARGUMENT:
                if (argDefs) {
                    for (var i = 0; i < argDefs.length; i++) {
                        if (argDefs[i].name === state.name) {
                            argDef = argDefs[i];
                            break;
                        }
                    }
                }
                inputType = argDef && argDef.type;
                break;
            case graphql_language_service_parser_1.RuleKinds.ENUM_VALUE:
                var enumType = graphql_1.getNamedType(inputType);
                enumValue =
                    enumType instanceof graphql_1.GraphQLEnumType
                        ? find(enumType.getValues(), function (val) { return val.value === state.name; })
                        : null;
                break;
            case graphql_language_service_parser_1.RuleKinds.LIST_VALUE:
                var nullableType = graphql_1.getNullableType(inputType);
                inputType =
                    nullableType instanceof graphql_1.GraphQLList ? nullableType.ofType : null;
                break;
            case graphql_language_service_parser_1.RuleKinds.OBJECT_VALUE:
                var objectType = graphql_1.getNamedType(inputType);
                objectFieldDefs =
                    objectType instanceof graphql_1.GraphQLInputObjectType
                        ? objectType.getFields()
                        : null;
                break;
            case graphql_language_service_parser_1.RuleKinds.OBJECT_FIELD:
                var objectField = state.name && objectFieldDefs ? objectFieldDefs[state.name] : null;
                inputType = objectField && objectField.type;
                break;
            case graphql_language_service_parser_1.RuleKinds.NAMED_TYPE:
                if (state.name) {
                    type = schema.getType(state.name);
                }
                break;
        }
    });
    return {
        argDef: argDef,
        argDefs: argDefs,
        directiveDef: directiveDef,
        enumValue: enumValue,
        fieldDef: fieldDef,
        inputType: inputType,
        objectFieldDefs: objectFieldDefs,
        parentType: parentType,
        type: type,
    };
}
exports.getTypeInfo = getTypeInfo;
function find(array, predicate) {
    for (var i = 0; i < array.length; i++) {
        if (predicate(array[i])) {
            return array[i];
        }
    }
    return null;
}
//# sourceMappingURL=getAutocompleteSuggestions.js.map