"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
var assert_1 = __importDefault(require("assert"));
var graphql_1 = require("graphql");
var graphql_language_service_parser_1 = require("graphql-language-service-parser");
var graphql_language_service_utils_1 = require("graphql-language-service-utils");
exports.SEVERITY = {
    Error: 'Error',
    Warning: 'Warning',
    Information: 'Information',
    Hint: 'Hint',
};
exports.DIAGNOSTIC_SEVERITY = (_a = {},
    _a[exports.SEVERITY.Error] = 1,
    _a[exports.SEVERITY.Warning] = 2,
    _a[exports.SEVERITY.Information] = 3,
    _a[exports.SEVERITY.Hint] = 4,
    _a);
function getDiagnostics(query, schema, customRules, isRelayCompatMode) {
    if (schema === void 0) { schema = null; }
    var ast = null;
    try {
        ast = graphql_1.parse(query);
    }
    catch (error) {
        var range = getRange(error.locations[0], query);
        return [
            {
                severity: exports.DIAGNOSTIC_SEVERITY.Error,
                message: error.message,
                source: 'GraphQL: Syntax',
                range: range,
            },
        ];
    }
    return validateQuery(ast, schema, customRules, isRelayCompatMode);
}
exports.getDiagnostics = getDiagnostics;
function validateQuery(ast, schema, customRules, isRelayCompatMode) {
    if (schema === void 0) { schema = null; }
    if (!schema) {
        return [];
    }
    var validationErrorAnnotations = mapCat(graphql_language_service_utils_1.validateWithCustomRules(schema, ast, customRules, isRelayCompatMode), function (error) { return annotations(error, exports.DIAGNOSTIC_SEVERITY.Error, 'Validation'); });
    var deprecationWarningAnnotations = !graphql_1.findDeprecatedUsages
        ? []
        : mapCat(graphql_1.findDeprecatedUsages(schema, ast), function (error) {
            return annotations(error, exports.DIAGNOSTIC_SEVERITY.Warning, 'Deprecation');
        });
    return validationErrorAnnotations.concat(deprecationWarningAnnotations);
}
exports.validateQuery = validateQuery;
function mapCat(array, mapper) {
    return Array.prototype.concat.apply([], array.map(mapper));
}
function annotations(error, severity, type) {
    if (!error.nodes) {
        return [];
    }
    var highlightedNodes = [];
    error.nodes.forEach(function (node) {
        var highlightNode = node.kind !== 'Variable' && 'name' in node && node.name !== undefined
            ? node.name
            : 'variable' in node && node.variable !== undefined
                ? node.variable
                : node;
        if (highlightNode) {
            assert_1.default(error.locations, 'GraphQL validation error requires locations.');
            var loc = error.locations[0];
            var highlightLoc = getLocation(highlightNode);
            var end = loc.column + (highlightLoc.end - highlightLoc.start);
            highlightedNodes.push({
                source: "GraphQL: " + type,
                message: error.message,
                severity: severity,
                range: new graphql_language_service_utils_1.Range(new graphql_language_service_utils_1.Position(loc.line - 1, loc.column - 1), new graphql_language_service_utils_1.Position(loc.line - 1, end)),
            });
        }
    });
    return highlightedNodes;
}
function getRange(location, queryText) {
    var parser = graphql_language_service_parser_1.onlineParser();
    var state = parser.startState();
    var lines = queryText.split('\n');
    assert_1.default(lines.length >= location.line, 'Query text must have more lines than where the error happened');
    var stream = null;
    for (var i = 0; i < location.line; i++) {
        stream = new graphql_language_service_parser_1.CharacterStream(lines[i]);
        while (!stream.eol()) {
            var style = parser.token(stream, state);
            if (style === 'invalidchar') {
                break;
            }
        }
    }
    assert_1.default(stream, 'Expected Parser stream to be available.');
    var line = location.line - 1;
    var start = stream.getStartOfToken();
    var end = stream.getCurrentPosition();
    return new graphql_language_service_utils_1.Range(new graphql_language_service_utils_1.Position(line, start), new graphql_language_service_utils_1.Position(line, end));
}
exports.getRange = getRange;
function getLocation(node) {
    var typeCastedNode = node;
    var location = typeCastedNode.loc;
    assert_1.default(location, 'Expected ASTNode to have a location.');
    return location;
}
//# sourceMappingURL=getDiagnostics.js.map