var __assign = (this && this.__assign) || function () {
    __assign = Object.assign || function(t) {
        for (var s, i = 1, n = arguments.length; i < n; i++) {
            s = arguments[i];
            for (var p in s) if (Object.prototype.hasOwnProperty.call(s, p))
                t[p] = s[p];
        }
        return t;
    };
    return __assign.apply(this, arguments);
};
import { CompletionItemKind } from 'vscode-languageserver-types';
import { GraphQLBoolean, GraphQLEnumType, GraphQLInputObjectType, GraphQLList, SchemaMetaFieldDef, TypeMetaFieldDef, TypeNameMetaFieldDef, assertAbstractType, doTypesOverlap, getNamedType, getNullableType, isAbstractType, isCompositeType, isInputType, } from 'graphql';
import { CharacterStream, onlineParser, RuleKinds, } from 'graphql-language-service-parser';
import { forEachState, getDefinitionState, getFieldDef, hintList, objectValues, } from './autocompleteUtils';
export function getAutocompleteSuggestions(schema, queryText, cursor, contextToken) {
    var token = contextToken || getTokenAtPosition(queryText, cursor);
    var state = token.state.kind === 'Invalid' ? token.state.prevState : token.state;
    if (!state) {
        return [];
    }
    var kind = state.kind;
    var step = state.step;
    var typeInfo = getTypeInfo(schema, token.state);
    if (kind === RuleKinds.DOCUMENT) {
        return hintList(token, [
            { label: 'query', kind: CompletionItemKind.Function },
            { label: 'mutation', kind: CompletionItemKind.Function },
            { label: 'subscription', kind: CompletionItemKind.Function },
            { label: 'fragment', kind: CompletionItemKind.Function },
            { label: '{', kind: CompletionItemKind.Constructor },
        ]);
    }
    if (kind === RuleKinds.SELECTION_SET ||
        kind === RuleKinds.FIELD ||
        kind === RuleKinds.ALIASED_FIELD) {
        return getSuggestionsForFieldNames(token, typeInfo, schema, kind);
    }
    if (kind === RuleKinds.ARGUMENTS ||
        (kind === RuleKinds.ARGUMENT && step === 0)) {
        var argDefs = typeInfo.argDefs;
        if (argDefs) {
            return hintList(token, argDefs.map(function (argDef) {
                var _a;
                return ({
                    label: argDef.name,
                    detail: String(argDef.type),
                    documentation: (_a = argDef.description) !== null && _a !== void 0 ? _a : undefined,
                    kind: CompletionItemKind.Variable,
                });
            }));
        }
    }
    if (kind === RuleKinds.OBJECT_VALUE ||
        (kind === RuleKinds.OBJECT_FIELD && step === 0)) {
        if (typeInfo.objectFieldDefs) {
            var objectFields = objectValues(typeInfo.objectFieldDefs);
            var completionKind_1 = kind === RuleKinds.OBJECT_VALUE
                ? CompletionItemKind.Value
                : CompletionItemKind.Field;
            return hintList(token, objectFields.map(function (field) {
                var _a;
                return ({
                    label: field.name,
                    detail: String(field.type),
                    documentation: (_a = field.description) !== null && _a !== void 0 ? _a : undefined,
                    kind: completionKind_1,
                });
            }));
        }
    }
    if (kind === RuleKinds.ENUM_VALUE ||
        (kind === RuleKinds.LIST_VALUE && step === 1) ||
        (kind === RuleKinds.OBJECT_FIELD && step === 2) ||
        (kind === RuleKinds.ARGUMENT && step === 2)) {
        return getSuggestionsForInputValues(token, typeInfo, kind);
    }
    if ((kind === RuleKinds.TYPE_CONDITION && step === 1) ||
        (kind === RuleKinds.NAMED_TYPE &&
            state.prevState != null &&
            state.prevState.kind === RuleKinds.TYPE_CONDITION)) {
        return getSuggestionsForFragmentTypeConditions(token, typeInfo, schema, kind);
    }
    if (kind === RuleKinds.FRAGMENT_SPREAD && step === 1) {
        return getSuggestionsForFragmentSpread(token, typeInfo, schema, queryText, kind);
    }
    if ((kind === RuleKinds.VARIABLE_DEFINITION && step === 2) ||
        (kind === RuleKinds.LIST_TYPE && step === 1) ||
        (kind === RuleKinds.NAMED_TYPE &&
            state.prevState &&
            (state.prevState.kind === RuleKinds.VARIABLE_DEFINITION ||
                state.prevState.kind === RuleKinds.LIST_TYPE))) {
        return getSuggestionsForVariableDefinition(token, schema, kind);
    }
    if (kind === RuleKinds.DIRECTIVE) {
        return getSuggestionsForDirective(token, state, schema, kind);
    }
    return [];
}
function getSuggestionsForFieldNames(token, typeInfo, schema, _kind) {
    if (typeInfo.parentType) {
        var parentType = typeInfo.parentType;
        var fields = [];
        if ('getFields' in parentType) {
            fields = objectValues(parentType.getFields());
        }
        if (isCompositeType(parentType)) {
            fields.push(TypeNameMetaFieldDef);
        }
        if (parentType === schema.getQueryType()) {
            fields.push(SchemaMetaFieldDef, TypeMetaFieldDef);
        }
        return hintList(token, fields.map(function (field, index) {
            var _a;
            return ({
                sortText: String(index) + field.name,
                label: field.name,
                detail: String(field.type),
                documentation: (_a = field.description) !== null && _a !== void 0 ? _a : undefined,
                deprecated: field.isDeprecated,
                isDeprecated: field.isDeprecated,
                deprecationReason: field.deprecationReason,
                kind: CompletionItemKind.Field,
            });
        }));
    }
    return [];
}
function getSuggestionsForInputValues(token, typeInfo, _kind) {
    var namedInputType = getNamedType(typeInfo.inputType);
    if (namedInputType instanceof GraphQLEnumType) {
        var values = namedInputType.getValues();
        return hintList(token, values.map(function (value) {
            var _a;
            return ({
                label: value.name,
                detail: String(namedInputType),
                documentation: (_a = value.description) !== null && _a !== void 0 ? _a : undefined,
                deprecated: value.isDeprecated,
                isDeprecated: value.isDeprecated,
                deprecationReason: value.deprecationReason,
                kind: CompletionItemKind.EnumMember,
            });
        }));
    }
    else if (namedInputType === GraphQLBoolean) {
        return hintList(token, [
            {
                label: 'true',
                detail: String(GraphQLBoolean),
                documentation: 'Not false.',
                kind: CompletionItemKind.Variable,
            },
            {
                label: 'false',
                detail: String(GraphQLBoolean),
                documentation: 'Not true.',
                kind: CompletionItemKind.Variable,
            },
        ]);
    }
    return [];
}
function getSuggestionsForFragmentTypeConditions(token, typeInfo, schema, _kind) {
    var possibleTypes;
    if (typeInfo.parentType) {
        if (isAbstractType(typeInfo.parentType)) {
            var abstractType = assertAbstractType(typeInfo.parentType);
            var possibleObjTypes = schema.getPossibleTypes(abstractType);
            var possibleIfaceMap_1 = Object.create(null);
            possibleObjTypes.forEach(function (type) {
                type.getInterfaces().forEach(function (iface) {
                    possibleIfaceMap_1[iface.name] = iface;
                });
            });
            possibleTypes = possibleObjTypes.concat(objectValues(possibleIfaceMap_1));
        }
        else {
            possibleTypes = [typeInfo.parentType];
        }
    }
    else {
        var typeMap = schema.getTypeMap();
        possibleTypes = objectValues(typeMap).filter(isCompositeType);
    }
    return hintList(token, possibleTypes.map(function (type) {
        var namedType = getNamedType(type);
        return {
            label: String(type),
            documentation: (namedType && namedType.description) || '',
            kind: CompletionItemKind.Field,
        };
    }));
}
function getSuggestionsForFragmentSpread(token, typeInfo, schema, queryText, _kind) {
    var typeMap = schema.getTypeMap();
    var defState = getDefinitionState(token.state);
    var fragments = getFragmentDefinitions(queryText);
    var relevantFrags = fragments.filter(function (frag) {
        return typeMap[frag.typeCondition.name.value] &&
            !(defState &&
                defState.kind === RuleKinds.FRAGMENT_DEFINITION &&
                defState.name === frag.name.value) &&
            isCompositeType(typeInfo.parentType) &&
            isCompositeType(typeMap[frag.typeCondition.name.value]) &&
            doTypesOverlap(schema, typeInfo.parentType, typeMap[frag.typeCondition.name.value]);
    });
    return hintList(token, relevantFrags.map(function (frag) { return ({
        label: frag.name.value,
        detail: String(typeMap[frag.typeCondition.name.value]),
        documentation: "fragment " + frag.name.value + " on " + frag.typeCondition.name.value,
        kind: CompletionItemKind.Field,
    }); }));
}
export function getFragmentDefinitions(queryText) {
    var fragmentDefs = [];
    runOnlineParser(queryText, function (_, state) {
        if (state.kind === RuleKinds.FRAGMENT_DEFINITION &&
            state.name &&
            state.type) {
            fragmentDefs.push({
                kind: RuleKinds.FRAGMENT_DEFINITION,
                name: {
                    kind: 'Name',
                    value: state.name,
                },
                selectionSet: {
                    kind: RuleKinds.SELECTION_SET,
                    selections: [],
                },
                typeCondition: {
                    kind: RuleKinds.NAMED_TYPE,
                    name: {
                        kind: 'Name',
                        value: state.type,
                    },
                },
            });
        }
    });
    return fragmentDefs;
}
function getSuggestionsForVariableDefinition(token, schema, _kind) {
    var inputTypeMap = schema.getTypeMap();
    var inputTypes = objectValues(inputTypeMap).filter(isInputType);
    return hintList(token, inputTypes.map(function (type) { return ({
        label: type.name,
        documentation: type.description,
        kind: CompletionItemKind.Variable,
    }); }));
}
function getSuggestionsForDirective(token, state, schema, _kind) {
    if (state.prevState && state.prevState.kind) {
        var directives = schema
            .getDirectives()
            .filter(function (directive) { return canUseDirective(state.prevState, directive); });
        return hintList(token, directives.map(function (directive) { return ({
            label: directive.name,
            documentation: directive.description || '',
            kind: CompletionItemKind.Function,
        }); }));
    }
    return [];
}
export function getTokenAtPosition(queryText, cursor) {
    var styleAtCursor = null;
    var stateAtCursor = null;
    var stringAtCursor = null;
    var token = runOnlineParser(queryText, function (stream, state, style, index) {
        if (index === cursor.line) {
            if (stream.getCurrentPosition() >= cursor.character) {
                styleAtCursor = style;
                stateAtCursor = __assign({}, state);
                stringAtCursor = stream.current();
                return 'BREAK';
            }
        }
    });
    return {
        start: token.start,
        end: token.end,
        string: stringAtCursor || token.string,
        state: stateAtCursor || token.state,
        style: styleAtCursor || token.style,
    };
}
export function runOnlineParser(queryText, callback) {
    var lines = queryText.split('\n');
    var parser = onlineParser();
    var state = parser.startState();
    var style = '';
    var stream = new CharacterStream('');
    for (var i = 0; i < lines.length; i++) {
        stream = new CharacterStream(lines[i]);
        while (!stream.eol()) {
            style = parser.token(stream, state);
            var code = callback(stream, state, style, i);
            if (code === 'BREAK') {
                break;
            }
        }
        callback(stream, state, style, i);
        if (!state.kind) {
            state = parser.startState();
        }
    }
    return {
        start: stream.getStartOfToken(),
        end: stream.getCurrentPosition(),
        string: stream.current(),
        state: state,
        style: style,
    };
}
export function canUseDirective(state, directive) {
    if (!state || !state.kind) {
        return false;
    }
    var kind = state.kind;
    var locations = directive.locations;
    switch (kind) {
        case RuleKinds.QUERY:
            return locations.indexOf('QUERY') !== -1;
        case RuleKinds.MUTATION:
            return locations.indexOf('MUTATION') !== -1;
        case RuleKinds.SUBSCRIPTION:
            return locations.indexOf('SUBSCRIPTION') !== -1;
        case RuleKinds.FIELD:
        case RuleKinds.ALIASED_FIELD:
            return locations.indexOf('FIELD') !== -1;
        case RuleKinds.FRAGMENT_DEFINITION:
            return locations.indexOf('FRAGMENT_DEFINITION') !== -1;
        case RuleKinds.FRAGMENT_SPREAD:
            return locations.indexOf('FRAGMENT_SPREAD') !== -1;
        case RuleKinds.INLINE_FRAGMENT:
            return locations.indexOf('INLINE_FRAGMENT') !== -1;
        case RuleKinds.SCHEMA_DEF:
            return locations.indexOf('SCHEMA') !== -1;
        case RuleKinds.SCALAR_DEF:
            return locations.indexOf('SCALAR') !== -1;
        case RuleKinds.OBJECT_TYPE_DEF:
            return locations.indexOf('OBJECT') !== -1;
        case RuleKinds.FIELD_DEF:
            return locations.indexOf('FIELD_DEFINITION') !== -1;
        case RuleKinds.INTERFACE_DEF:
            return locations.indexOf('INTERFACE') !== -1;
        case RuleKinds.UNION_DEF:
            return locations.indexOf('UNION') !== -1;
        case RuleKinds.ENUM_DEF:
            return locations.indexOf('ENUM') !== -1;
        case RuleKinds.ENUM_VALUE:
            return locations.indexOf('ENUM_VALUE') !== -1;
        case RuleKinds.INPUT_DEF:
            return locations.indexOf('INPUT_OBJECT') !== -1;
        case RuleKinds.INPUT_VALUE_DEF:
            var prevStateKind = state.prevState && state.prevState.kind;
            switch (prevStateKind) {
                case RuleKinds.ARGUMENTS_DEF:
                    return locations.indexOf('ARGUMENT_DEFINITION') !== -1;
                case RuleKinds.INPUT_DEF:
                    return locations.indexOf('INPUT_FIELD_DEFINITION') !== -1;
            }
    }
    return false;
}
export function getTypeInfo(schema, tokenState) {
    var argDef;
    var argDefs;
    var directiveDef;
    var enumValue;
    var fieldDef;
    var inputType;
    var objectFieldDefs;
    var parentType;
    var type;
    forEachState(tokenState, function (state) {
        switch (state.kind) {
            case RuleKinds.QUERY:
            case 'ShortQuery':
                type = schema.getQueryType();
                break;
            case RuleKinds.MUTATION:
                type = schema.getMutationType();
                break;
            case RuleKinds.SUBSCRIPTION:
                type = schema.getSubscriptionType();
                break;
            case RuleKinds.INLINE_FRAGMENT:
            case RuleKinds.FRAGMENT_DEFINITION:
                if (state.type) {
                    type = schema.getType(state.type);
                }
                break;
            case RuleKinds.FIELD:
            case RuleKinds.ALIASED_FIELD:
                if (!type || !state.name) {
                    fieldDef = null;
                }
                else {
                    fieldDef = parentType
                        ? getFieldDef(schema, parentType, state.name)
                        : null;
                    type = fieldDef ? fieldDef.type : null;
                }
                break;
            case RuleKinds.SELECTION_SET:
                parentType = getNamedType(type);
                break;
            case RuleKinds.DIRECTIVE:
                directiveDef = state.name ? schema.getDirective(state.name) : null;
                break;
            case RuleKinds.ARGUMENTS:
                if (!state.prevState) {
                    argDefs = null;
                }
                else {
                    switch (state.prevState.kind) {
                        case RuleKinds.FIELD:
                            argDefs = fieldDef && fieldDef.args;
                            break;
                        case RuleKinds.DIRECTIVE:
                            argDefs = directiveDef && directiveDef.args;
                            break;
                        case RuleKinds.ALIASED_FIELD:
                            var name_1 = state.prevState && state.prevState.name;
                            if (!name_1) {
                                argDefs = null;
                                break;
                            }
                            var field = parentType
                                ? getFieldDef(schema, parentType, name_1)
                                : null;
                            if (!field) {
                                argDefs = null;
                                break;
                            }
                            argDefs = field.args;
                            break;
                        default:
                            argDefs = null;
                            break;
                    }
                }
                break;
            case RuleKinds.ARGUMENT:
                if (argDefs) {
                    for (var i = 0; i < argDefs.length; i++) {
                        if (argDefs[i].name === state.name) {
                            argDef = argDefs[i];
                            break;
                        }
                    }
                }
                inputType = argDef && argDef.type;
                break;
            case RuleKinds.ENUM_VALUE:
                var enumType = getNamedType(inputType);
                enumValue =
                    enumType instanceof GraphQLEnumType
                        ? find(enumType.getValues(), function (val) { return val.value === state.name; })
                        : null;
                break;
            case RuleKinds.LIST_VALUE:
                var nullableType = getNullableType(inputType);
                inputType =
                    nullableType instanceof GraphQLList ? nullableType.ofType : null;
                break;
            case RuleKinds.OBJECT_VALUE:
                var objectType = getNamedType(inputType);
                objectFieldDefs =
                    objectType instanceof GraphQLInputObjectType
                        ? objectType.getFields()
                        : null;
                break;
            case RuleKinds.OBJECT_FIELD:
                var objectField = state.name && objectFieldDefs ? objectFieldDefs[state.name] : null;
                inputType = objectField && objectField.type;
                break;
            case RuleKinds.NAMED_TYPE:
                if (state.name) {
                    type = schema.getType(state.name);
                }
                break;
        }
    });
    return {
        argDef: argDef,
        argDefs: argDefs,
        directiveDef: directiveDef,
        enumValue: enumValue,
        fieldDef: fieldDef,
        inputType: inputType,
        objectFieldDefs: objectFieldDefs,
        parentType: parentType,
        type: type,
    };
}
function find(array, predicate) {
    for (var i = 0; i < array.length; i++) {
        if (predicate(array[i])) {
            return array[i];
        }
    }
    return null;
}
//# sourceMappingURL=getAutocompleteSuggestions.js.map