var _a;
import invariant from 'assert';
import { findDeprecatedUsages, parse } from 'graphql';
import { CharacterStream, onlineParser } from 'graphql-language-service-parser';
import { Range, validateWithCustomRules, Position, } from 'graphql-language-service-utils';
export var SEVERITY = {
    Error: 'Error',
    Warning: 'Warning',
    Information: 'Information',
    Hint: 'Hint',
};
export var DIAGNOSTIC_SEVERITY = (_a = {},
    _a[SEVERITY.Error] = 1,
    _a[SEVERITY.Warning] = 2,
    _a[SEVERITY.Information] = 3,
    _a[SEVERITY.Hint] = 4,
    _a);
export function getDiagnostics(query, schema, customRules, isRelayCompatMode) {
    if (schema === void 0) { schema = null; }
    var ast = null;
    try {
        ast = parse(query);
    }
    catch (error) {
        var range = getRange(error.locations[0], query);
        return [
            {
                severity: DIAGNOSTIC_SEVERITY.Error,
                message: error.message,
                source: 'GraphQL: Syntax',
                range: range,
            },
        ];
    }
    return validateQuery(ast, schema, customRules, isRelayCompatMode);
}
export function validateQuery(ast, schema, customRules, isRelayCompatMode) {
    if (schema === void 0) { schema = null; }
    if (!schema) {
        return [];
    }
    var validationErrorAnnotations = mapCat(validateWithCustomRules(schema, ast, customRules, isRelayCompatMode), function (error) { return annotations(error, DIAGNOSTIC_SEVERITY.Error, 'Validation'); });
    var deprecationWarningAnnotations = !findDeprecatedUsages
        ? []
        : mapCat(findDeprecatedUsages(schema, ast), function (error) {
            return annotations(error, DIAGNOSTIC_SEVERITY.Warning, 'Deprecation');
        });
    return validationErrorAnnotations.concat(deprecationWarningAnnotations);
}
function mapCat(array, mapper) {
    return Array.prototype.concat.apply([], array.map(mapper));
}
function annotations(error, severity, type) {
    if (!error.nodes) {
        return [];
    }
    var highlightedNodes = [];
    error.nodes.forEach(function (node) {
        var highlightNode = node.kind !== 'Variable' && 'name' in node && node.name !== undefined
            ? node.name
            : 'variable' in node && node.variable !== undefined
                ? node.variable
                : node;
        if (highlightNode) {
            invariant(error.locations, 'GraphQL validation error requires locations.');
            var loc = error.locations[0];
            var highlightLoc = getLocation(highlightNode);
            var end = loc.column + (highlightLoc.end - highlightLoc.start);
            highlightedNodes.push({
                source: "GraphQL: " + type,
                message: error.message,
                severity: severity,
                range: new Range(new Position(loc.line - 1, loc.column - 1), new Position(loc.line - 1, end)),
            });
        }
    });
    return highlightedNodes;
}
export function getRange(location, queryText) {
    var parser = onlineParser();
    var state = parser.startState();
    var lines = queryText.split('\n');
    invariant(lines.length >= location.line, 'Query text must have more lines than where the error happened');
    var stream = null;
    for (var i = 0; i < location.line; i++) {
        stream = new CharacterStream(lines[i]);
        while (!stream.eol()) {
            var style = parser.token(stream, state);
            if (style === 'invalidchar') {
                break;
            }
        }
    }
    invariant(stream, 'Expected Parser stream to be available.');
    var line = location.line - 1;
    var start = stream.getStartOfToken();
    var end = stream.getCurrentPosition();
    return new Range(new Position(line, start), new Position(line, end));
}
function getLocation(node) {
    var typeCastedNode = node;
    var location = typeCastedNode.loc;
    invariant(location, 'Expected ASTNode to have a location.');
    return location;
}
//# sourceMappingURL=getDiagnostics.js.map