"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.getVariablesJSONSchema = exports.defaultJSONSchemaOptions = void 0;
const graphql_1 = require("graphql");
exports.defaultJSONSchemaOptions = {
    useMarkdownDescription: false,
};
function text(into, newText) {
    into.push(newText);
}
function renderType(into, t) {
    if (graphql_1.isNonNullType(t)) {
        renderType(into, t.ofType);
        text(into, '!');
    }
    else if (graphql_1.isListType(t)) {
        text(into, '[');
        renderType(into, t.ofType);
        text(into, ']');
    }
    else {
        text(into, t.name);
    }
}
function renderTypeToString(t, useMarkdown) {
    const into = [];
    if (useMarkdown) {
        text(into, '```graphql\n');
    }
    renderType(into, t);
    if (useMarkdown) {
        text(into, '\n```');
    }
    return into.join('');
}
const scalarTypesMap = {
    Int: 'integer',
    String: 'string',
    Float: 'number',
    ID: 'string',
    Boolean: 'boolean',
    DateTime: 'string',
};
function getJSONSchemaFromGraphQLType(type, options) {
    var _a;
    let required = false;
    let definition = Object.create(null);
    const definitions = Object.create(null);
    if ('defaultValue' in type && type.defaultValue !== undefined) {
        definition.default = type.defaultValue;
    }
    if (graphql_1.isEnumType(type)) {
        definition.type = 'string';
        definition.enum = type.getValues().map(val => val.name);
    }
    if (graphql_1.isScalarType(type)) {
        definition.type = (_a = scalarTypesMap[type.name]) !== null && _a !== void 0 ? _a : 'any';
    }
    if (graphql_1.isListType(type)) {
        definition.type = 'array';
        const { definition: def, definitions: defs } = getJSONSchemaFromGraphQLType(type.ofType, options);
        if (def.$ref) {
            definition.items = { $ref: def.$ref };
        }
        else {
            definition.items = def;
        }
        if (defs) {
            Object.keys(defs).forEach(defName => {
                definitions[defName] = defs[defName];
            });
        }
    }
    if (graphql_1.isNonNullType(type)) {
        required = true;
        const { definition: def, definitions: defs } = getJSONSchemaFromGraphQLType(type.ofType, options);
        definition = def;
        if (defs) {
            Object.keys(defs).forEach(defName => {
                definitions[defName] = defs[defName];
            });
        }
    }
    if (graphql_1.isInputObjectType(type)) {
        definition.$ref = `#/definitions/${type.name}`;
        const fields = type.getFields();
        const fieldDef = {
            type: 'object',
            properties: {},
            required: [],
        };
        if (type.description) {
            fieldDef.description = type.description + `\n` + renderTypeToString(type);
            if (options === null || options === void 0 ? void 0 : options.useMarkdownDescription) {
                fieldDef.markdownDescription =
                    type.description + `\n` + renderTypeToString(type, true);
            }
        }
        else {
            fieldDef.description = renderTypeToString(type);
            if (options === null || options === void 0 ? void 0 : options.useMarkdownDescription) {
                fieldDef.markdownDescription = renderTypeToString(type, true);
            }
        }
        Object.keys(fields).forEach(fieldName => {
            const field = fields[fieldName];
            const { required: fieldRequired, definition: typeDefinition, definitions: typeDefinitions, } = getJSONSchemaFromGraphQLType(field.type, options);
            const { definition: fieldDefinition, } = getJSONSchemaFromGraphQLType(field, options);
            fieldDef.properties[fieldName] = Object.assign(Object.assign({}, typeDefinition), fieldDefinition);
            const renderedField = renderTypeToString(field.type);
            fieldDef.properties[fieldName].description = field.description
                ? field.description + '\n' + renderedField
                : renderedField;
            if (options === null || options === void 0 ? void 0 : options.useMarkdownDescription) {
                const renderedFieldMarkdown = renderTypeToString(field.type, true);
                fieldDef.properties[fieldName].markdownDescription = field.description
                    ? field.description + '\n' + renderedFieldMarkdown
                    : renderedFieldMarkdown;
            }
            if (fieldRequired) {
                fieldDef.required.push(fieldName);
            }
            if (typeDefinitions) {
                Object.keys(typeDefinitions).map(defName => {
                    definitions[defName] = typeDefinitions[defName];
                });
            }
        });
        definitions[type.name] = fieldDef;
    }
    if ('description' in type &&
        !graphql_1.isScalarType(type) &&
        type.description &&
        !definition.description) {
        definition.description = type.description + '\n' + renderTypeToString(type);
        if (options === null || options === void 0 ? void 0 : options.useMarkdownDescription) {
            definition.markdownDescription =
                type.description + '\n' + renderTypeToString(type, true);
        }
    }
    else {
        definition.description = renderTypeToString(type);
        if (options === null || options === void 0 ? void 0 : options.useMarkdownDescription) {
            definition.markdownDescription = renderTypeToString(type, true);
        }
    }
    return { required, definition, definitions };
}
function getVariablesJSONSchema(variableToType, options) {
    const jsonSchema = {
        $schema: 'https://json-schema.org/draft/2020-12/schema',
        type: 'object',
        properties: {},
        required: [],
    };
    if (variableToType) {
        Object.entries(variableToType).forEach(([variableName, type]) => {
            var _a;
            const { definition, required, definitions, } = getJSONSchemaFromGraphQLType(type, options);
            jsonSchema.properties[variableName] = definition;
            if (required) {
                (_a = jsonSchema.required) === null || _a === void 0 ? void 0 : _a.push(variableName);
            }
            if (definitions) {
                jsonSchema.definitions = Object.assign(Object.assign({}, jsonSchema === null || jsonSchema === void 0 ? void 0 : jsonSchema.definitions), definitions);
            }
        });
    }
    return jsonSchema;
}
exports.getVariablesJSONSchema = getVariablesJSONSchema;
//# sourceMappingURL=getVariablesJSONSchema.js.map