import { parse, visit, } from 'graphql';
import { getAutocompleteSuggestions, getDiagnostics, getHoverInformation, } from 'graphql-language-service-interface';
import { defaultSchemaLoader, defaultSchemaBuilder, } from './schemaLoader';
export class LanguageService {
    constructor({ parser, schemaLoader, schemaBuilder, schemaConfig, schemaString, parseOptions, exteralFragmentDefinitions, }) {
        this._parser = parse;
        this._schema = null;
        this._schemaResponse = null;
        this._schemaLoader = defaultSchemaLoader;
        this._schemaBuilder = defaultSchemaBuilder;
        this._schemaString = null;
        this._parseOptions = undefined;
        this._exteralFragmentDefinitionNodes = null;
        this._exteralFragmentDefinitionsString = null;
        this.getCompletion = async (_uri, documentText, position) => {
            const schema = await this.getSchema();
            if (!documentText || documentText.length < 1 || !schema) {
                return [];
            }
            return getAutocompleteSuggestions(schema, documentText, position, undefined, await this.getExternalFragmentDefinitions());
        };
        this.getDiagnostics = async (_uri, documentText, customRules) => {
            const schema = await this.getSchema();
            if (!documentText || documentText.length < 1 || !schema) {
                return [];
            }
            return getDiagnostics(documentText, schema, customRules);
        };
        this.getHover = async (_uri, documentText, position) => getHoverInformation((await this.getSchema()), documentText, position);
        this._schemaConfig = schemaConfig;
        if (parser) {
            this._parser = parser;
        }
        if (schemaLoader) {
            this._schemaLoader = schemaLoader;
        }
        if (schemaBuilder) {
            this._schemaBuilder = schemaBuilder;
        }
        if (schemaString) {
            this._schemaString = schemaString;
        }
        if (parseOptions) {
            this._parseOptions = parseOptions;
        }
        if (exteralFragmentDefinitions) {
            if (Array.isArray(exteralFragmentDefinitions)) {
                this._exteralFragmentDefinitionNodes = exteralFragmentDefinitions;
            }
            else {
                this._exteralFragmentDefinitionsString = exteralFragmentDefinitions;
            }
        }
    }
    get schema() {
        return this._schema;
    }
    async getSchema() {
        if (this.schema) {
            return this.schema;
        }
        return this.loadSchema();
    }
    async getExternalFragmentDefinitions() {
        if (!this._exteralFragmentDefinitionNodes &&
            this._exteralFragmentDefinitionsString) {
            const definitionNodes = [];
            try {
                visit(await this._parser(this._exteralFragmentDefinitionsString), {
                    FragmentDefinition(node) {
                        definitionNodes.push(node);
                    },
                });
            }
            catch (err) {
                throw Error(`Failed parsing exteralFragmentDefinitions string:\n${this._exteralFragmentDefinitionsString}`);
            }
            this._exteralFragmentDefinitionNodes = definitionNodes;
        }
        return this._exteralFragmentDefinitionNodes;
    }
    async setSchema(schema) {
        this._schemaString = schema;
        await this.loadSchema();
    }
    async getSchemaResponse() {
        if (this._schemaResponse) {
            return this._schemaResponse;
        }
        return this.loadSchemaResponse();
    }
    async loadSchemaResponse() {
        if (this._schemaString) {
            return typeof this._schemaString === 'string'
                ? this.parse(this._schemaString)
                : this._schemaString;
        }
        if (!this._schemaConfig?.uri) {
            return null;
        }
        this._schemaResponse = (await this._schemaLoader(this._schemaConfig));
        return this._schemaResponse;
    }
    async loadSchema() {
        const schemaResponse = await this.loadSchemaResponse();
        if (schemaResponse) {
            this._schema = this._schemaBuilder(schemaResponse, this._schemaConfig.buildSchemaOptions);
            return this._schema;
        }
        else {
            return null;
        }
    }
    async parse(text, options) {
        return this._parser(text, options || this._parseOptions);
    }
}
//# sourceMappingURL=LanguageService.js.map