"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.getTypeInfo = exports.canUseDirective = exports.runOnlineParser = exports.getTokenAtPosition = exports.getFragmentDefinitions = exports.getVariableCompletions = exports.getAutocompleteSuggestions = exports.SuggestionCommand = void 0;
const vscode_languageserver_types_1 = require("vscode-languageserver-types");
const graphql_1 = require("graphql");
const graphql_2 = require("graphql");
const parser_1 = require("../parser");
const autocompleteUtils_1 = require("./autocompleteUtils");
exports.SuggestionCommand = {
    command: 'editor.action.triggerSuggest',
    title: 'Suggestions',
};
const collectFragmentDefs = (op) => {
    const externalFragments = [];
    if (op) {
        try {
            graphql_2.visit(graphql_2.parse(op), {
                FragmentDefinition(def) {
                    externalFragments.push(def);
                },
            });
        }
        catch (_a) {
            return [];
        }
    }
    return externalFragments;
};
function getAutocompleteSuggestions(schema, queryText, cursor, contextToken, fragmentDefs, options) {
    var _a;
    const opts = Object.assign(Object.assign({}, options), { schema });
    const token = contextToken || getTokenAtPosition(queryText, cursor);
    const state = token.state.kind === 'Invalid' ? token.state.prevState : token.state;
    if (!state) {
        return [];
    }
    const kind = state.kind;
    const step = state.step;
    const typeInfo = getTypeInfo(schema, token.state);
    if (kind === parser_1.RuleKinds.DOCUMENT) {
        return autocompleteUtils_1.hintList(token, [
            { label: 'query', kind: vscode_languageserver_types_1.CompletionItemKind.Function },
            { label: 'mutation', kind: vscode_languageserver_types_1.CompletionItemKind.Function },
            { label: 'subscription', kind: vscode_languageserver_types_1.CompletionItemKind.Function },
            { label: 'fragment', kind: vscode_languageserver_types_1.CompletionItemKind.Function },
            { label: '{', kind: vscode_languageserver_types_1.CompletionItemKind.Constructor },
        ]);
    }
    if (kind === parser_1.RuleKinds.IMPLEMENTS ||
        (kind === parser_1.RuleKinds.NAMED_TYPE &&
            ((_a = state.prevState) === null || _a === void 0 ? void 0 : _a.kind) === parser_1.RuleKinds.IMPLEMENTS)) {
        return getSuggestionsForImplements(token, state, schema, queryText, typeInfo);
    }
    if (kind === parser_1.RuleKinds.SELECTION_SET ||
        kind === parser_1.RuleKinds.FIELD ||
        kind === parser_1.RuleKinds.ALIASED_FIELD) {
        return getSuggestionsForFieldNames(token, typeInfo, opts);
    }
    if (kind === parser_1.RuleKinds.ARGUMENTS ||
        (kind === parser_1.RuleKinds.ARGUMENT && step === 0)) {
        const argDefs = typeInfo.argDefs;
        if (argDefs) {
            return autocompleteUtils_1.hintList(token, argDefs.map((argDef) => {
                var _a;
                return ({
                    label: argDef.name,
                    insertText: argDef.name + ': ',
                    command: exports.SuggestionCommand,
                    detail: String(argDef.type),
                    documentation: (_a = argDef.description) !== null && _a !== void 0 ? _a : undefined,
                    kind: vscode_languageserver_types_1.CompletionItemKind.Variable,
                    type: argDef.type,
                });
            }));
        }
    }
    if (kind === parser_1.RuleKinds.OBJECT_VALUE ||
        (kind === parser_1.RuleKinds.OBJECT_FIELD && step === 0)) {
        if (typeInfo.objectFieldDefs) {
            const objectFields = autocompleteUtils_1.objectValues(typeInfo.objectFieldDefs);
            const completionKind = kind === parser_1.RuleKinds.OBJECT_VALUE
                ? vscode_languageserver_types_1.CompletionItemKind.Value
                : vscode_languageserver_types_1.CompletionItemKind.Field;
            return autocompleteUtils_1.hintList(token, objectFields.map(field => {
                var _a;
                return ({
                    label: field.name,
                    detail: String(field.type),
                    documentation: (_a = field.description) !== null && _a !== void 0 ? _a : undefined,
                    kind: completionKind,
                    type: field.type,
                });
            }));
        }
    }
    if (kind === parser_1.RuleKinds.ENUM_VALUE ||
        (kind === parser_1.RuleKinds.LIST_VALUE && step === 1) ||
        (kind === parser_1.RuleKinds.OBJECT_FIELD && step === 2) ||
        (kind === parser_1.RuleKinds.ARGUMENT && step === 2)) {
        return getSuggestionsForInputValues(token, typeInfo, queryText, schema);
    }
    if (kind === parser_1.RuleKinds.VARIABLE && step === 1) {
        const namedInputType = graphql_2.getNamedType(typeInfo.inputType);
        const variableDefinitions = getVariableCompletions(queryText, schema, token);
        return autocompleteUtils_1.hintList(token, variableDefinitions.filter(v => v.detail === (namedInputType === null || namedInputType === void 0 ? void 0 : namedInputType.name)));
    }
    if ((kind === parser_1.RuleKinds.TYPE_CONDITION && step === 1) ||
        (kind === parser_1.RuleKinds.NAMED_TYPE &&
            state.prevState != null &&
            state.prevState.kind === parser_1.RuleKinds.TYPE_CONDITION)) {
        return getSuggestionsForFragmentTypeConditions(token, typeInfo, schema, kind);
    }
    if (kind === parser_1.RuleKinds.FRAGMENT_SPREAD && step === 1) {
        return getSuggestionsForFragmentSpread(token, typeInfo, schema, queryText, Array.isArray(fragmentDefs)
            ? fragmentDefs
            : collectFragmentDefs(fragmentDefs));
    }
    if ((kind === parser_1.RuleKinds.VARIABLE_DEFINITION && step === 2) ||
        (kind === parser_1.RuleKinds.LIST_TYPE && step === 1) ||
        (kind === parser_1.RuleKinds.NAMED_TYPE &&
            state.prevState &&
            (state.prevState.kind === parser_1.RuleKinds.VARIABLE_DEFINITION ||
                state.prevState.kind === parser_1.RuleKinds.LIST_TYPE ||
                state.prevState.kind === parser_1.RuleKinds.NON_NULL_TYPE))) {
        return getSuggestionsForVariableDefinition(token, schema, kind);
    }
    if (kind === parser_1.RuleKinds.DIRECTIVE) {
        return getSuggestionsForDirective(token, state, schema, kind);
    }
    return [];
}
exports.getAutocompleteSuggestions = getAutocompleteSuggestions;
const insertSuffix = ` {\n  $1\n}`;
const getInsertText = (field) => {
    const type = field.type;
    if (graphql_2.isCompositeType(type)) {
        return insertSuffix;
    }
    if (graphql_1.isListType(type) && graphql_2.isCompositeType(type.ofType)) {
        return insertSuffix;
    }
    if (graphql_1.isNonNullType(type)) {
        if (graphql_2.isCompositeType(type.ofType)) {
            return insertSuffix;
        }
        if (graphql_1.isListType(type.ofType) && graphql_2.isCompositeType(type.ofType.ofType)) {
            return insertSuffix;
        }
    }
    return null;
};
function getSuggestionsForFieldNames(token, typeInfo, options) {
    var _a;
    if (typeInfo.parentType) {
        const parentType = typeInfo.parentType;
        let fields = [];
        if ('getFields' in parentType) {
            fields = autocompleteUtils_1.objectValues(parentType.getFields());
        }
        if (graphql_2.isCompositeType(parentType)) {
            fields.push(graphql_2.TypeNameMetaFieldDef);
        }
        if (parentType === ((_a = options === null || options === void 0 ? void 0 : options.schema) === null || _a === void 0 ? void 0 : _a.getQueryType())) {
            fields.push(graphql_2.SchemaMetaFieldDef, graphql_2.TypeMetaFieldDef);
        }
        return autocompleteUtils_1.hintList(token, fields.map((field, index) => {
            var _a;
            const suggestion = {
                sortText: String(index) + field.name,
                label: field.name,
                detail: String(field.type),
                documentation: (_a = field.description) !== null && _a !== void 0 ? _a : undefined,
                deprecated: Boolean(field.deprecationReason),
                isDeprecated: Boolean(field.deprecationReason),
                deprecationReason: field.deprecationReason,
                kind: vscode_languageserver_types_1.CompletionItemKind.Field,
                type: field.type,
            };
            const insertText = getInsertText(field);
            if (insertText) {
                suggestion.insertText = field.name + insertText;
                suggestion.insertTextFormat = vscode_languageserver_types_1.InsertTextFormat.Snippet;
                suggestion.command = exports.SuggestionCommand;
            }
            return suggestion;
        }));
    }
    return [];
}
function getSuggestionsForInputValues(token, typeInfo, queryText, schema) {
    const namedInputType = graphql_2.getNamedType(typeInfo.inputType);
    const queryVariables = getVariableCompletions(queryText, schema, token).filter(v => v.detail === namedInputType.name);
    if (namedInputType instanceof graphql_2.GraphQLEnumType) {
        const values = namedInputType.getValues();
        return autocompleteUtils_1.hintList(token, values
            .map((value) => {
            var _a;
            return ({
                label: value.name,
                detail: String(namedInputType),
                documentation: (_a = value.description) !== null && _a !== void 0 ? _a : undefined,
                deprecated: Boolean(value.deprecationReason),
                isDeprecated: Boolean(value.deprecationReason),
                deprecationReason: value.deprecationReason,
                kind: vscode_languageserver_types_1.CompletionItemKind.EnumMember,
                type: namedInputType,
            });
        })
            .concat(queryVariables));
    }
    else if (namedInputType === graphql_2.GraphQLBoolean) {
        return autocompleteUtils_1.hintList(token, queryVariables.concat([
            {
                label: 'true',
                detail: String(graphql_2.GraphQLBoolean),
                documentation: 'Not false.',
                kind: vscode_languageserver_types_1.CompletionItemKind.Variable,
                type: graphql_2.GraphQLBoolean,
            },
            {
                label: 'false',
                detail: String(graphql_2.GraphQLBoolean),
                documentation: 'Not true.',
                kind: vscode_languageserver_types_1.CompletionItemKind.Variable,
                type: graphql_2.GraphQLBoolean,
            },
        ]));
    }
    return queryVariables;
}
function getSuggestionsForImplements(token, tokenState, schema, documentText, typeInfo) {
    if (tokenState.needsSeperator) {
        return [];
    }
    const typeMap = schema.getTypeMap();
    const schemaInterfaces = autocompleteUtils_1.objectValues(typeMap).filter(graphql_1.isInterfaceType);
    const schemaInterfaceNames = schemaInterfaces.map(({ name }) => name);
    const inlineInterfaces = new Set();
    runOnlineParser(documentText, (_, state) => {
        var _a, _b, _c, _d, _e;
        if (state.name) {
            if (state.kind === parser_1.RuleKinds.INTERFACE_DEF &&
                !schemaInterfaceNames.includes(state.name)) {
                inlineInterfaces.add(state.name);
            }
            if (state.kind === parser_1.RuleKinds.NAMED_TYPE &&
                ((_a = state.prevState) === null || _a === void 0 ? void 0 : _a.kind) === parser_1.RuleKinds.IMPLEMENTS) {
                if (typeInfo.interfaceDef) {
                    const existingType = (_b = typeInfo.interfaceDef) === null || _b === void 0 ? void 0 : _b.getInterfaces().find(({ name }) => name === state.name);
                    if (existingType) {
                        return;
                    }
                    const type = schema.getType(state.name);
                    const interfaceConfig = (_c = typeInfo.interfaceDef) === null || _c === void 0 ? void 0 : _c.toConfig();
                    typeInfo.interfaceDef = new graphql_1.GraphQLInterfaceType(Object.assign(Object.assign({}, interfaceConfig), { interfaces: [
                            ...interfaceConfig.interfaces,
                            type ||
                                new graphql_1.GraphQLInterfaceType({ name: state.name, fields: {} }),
                        ] }));
                }
                else if (typeInfo.objectTypeDef) {
                    const existingType = (_d = typeInfo.objectTypeDef) === null || _d === void 0 ? void 0 : _d.getInterfaces().find(({ name }) => name === state.name);
                    if (existingType) {
                        return;
                    }
                    const type = schema.getType(state.name);
                    const objectTypeConfig = (_e = typeInfo.objectTypeDef) === null || _e === void 0 ? void 0 : _e.toConfig();
                    typeInfo.objectTypeDef = new graphql_1.GraphQLObjectType(Object.assign(Object.assign({}, objectTypeConfig), { interfaces: [
                            ...objectTypeConfig.interfaces,
                            type ||
                                new graphql_1.GraphQLInterfaceType({ name: state.name, fields: {} }),
                        ] }));
                }
            }
        }
    });
    const currentTypeToExtend = typeInfo.interfaceDef || typeInfo.objectTypeDef;
    const siblingInterfaces = (currentTypeToExtend === null || currentTypeToExtend === void 0 ? void 0 : currentTypeToExtend.getInterfaces()) || [];
    const siblingInterfaceNames = siblingInterfaces.map(({ name }) => name);
    const possibleInterfaces = schemaInterfaces
        .concat([...inlineInterfaces].map(name => ({ name })))
        .filter(({ name }) => name !== (currentTypeToExtend === null || currentTypeToExtend === void 0 ? void 0 : currentTypeToExtend.name) &&
        !siblingInterfaceNames.includes(name));
    return autocompleteUtils_1.hintList(token, possibleInterfaces.map(type => {
        const result = {
            label: type.name,
            kind: vscode_languageserver_types_1.CompletionItemKind.Interface,
            type,
        };
        if (type === null || type === void 0 ? void 0 : type.description) {
            result.documentation = type.description;
        }
        return result;
    }));
}
function getSuggestionsForFragmentTypeConditions(token, typeInfo, schema, _kind) {
    let possibleTypes;
    if (typeInfo.parentType) {
        if (graphql_2.isAbstractType(typeInfo.parentType)) {
            const abstractType = graphql_2.assertAbstractType(typeInfo.parentType);
            const possibleObjTypes = schema.getPossibleTypes(abstractType);
            const possibleIfaceMap = Object.create(null);
            possibleObjTypes.forEach(type => {
                type.getInterfaces().forEach(iface => {
                    possibleIfaceMap[iface.name] = iface;
                });
            });
            possibleTypes = possibleObjTypes.concat(autocompleteUtils_1.objectValues(possibleIfaceMap));
        }
        else {
            possibleTypes = [typeInfo.parentType];
        }
    }
    else {
        const typeMap = schema.getTypeMap();
        possibleTypes = autocompleteUtils_1.objectValues(typeMap).filter(graphql_2.isCompositeType);
    }
    return autocompleteUtils_1.hintList(token, possibleTypes.map(type => {
        const namedType = graphql_2.getNamedType(type);
        return {
            label: String(type),
            documentation: (namedType && namedType.description) || '',
            kind: vscode_languageserver_types_1.CompletionItemKind.Field,
        };
    }));
}
function getSuggestionsForFragmentSpread(token, typeInfo, schema, queryText, fragmentDefs) {
    if (!queryText) {
        return [];
    }
    const typeMap = schema.getTypeMap();
    const defState = autocompleteUtils_1.getDefinitionState(token.state);
    const fragments = getFragmentDefinitions(queryText);
    if (fragmentDefs && fragmentDefs.length > 0) {
        fragments.push(...fragmentDefs);
    }
    const relevantFrags = fragments.filter(frag => typeMap[frag.typeCondition.name.value] &&
        !(defState &&
            defState.kind === parser_1.RuleKinds.FRAGMENT_DEFINITION &&
            defState.name === frag.name.value) &&
        graphql_2.isCompositeType(typeInfo.parentType) &&
        graphql_2.isCompositeType(typeMap[frag.typeCondition.name.value]) &&
        graphql_2.doTypesOverlap(schema, typeInfo.parentType, typeMap[frag.typeCondition.name.value]));
    return autocompleteUtils_1.hintList(token, relevantFrags.map(frag => ({
        label: frag.name.value,
        detail: String(typeMap[frag.typeCondition.name.value]),
        documentation: `fragment ${frag.name.value} on ${frag.typeCondition.name.value}`,
        kind: vscode_languageserver_types_1.CompletionItemKind.Field,
        type: typeMap[frag.typeCondition.name.value],
    })));
}
const getParentDefinition = (state, kind) => {
    var _a, _b, _c, _d, _e, _f, _g, _h, _j, _k;
    if (((_a = state.prevState) === null || _a === void 0 ? void 0 : _a.kind) === kind) {
        return state.prevState;
    }
    if (((_c = (_b = state.prevState) === null || _b === void 0 ? void 0 : _b.prevState) === null || _c === void 0 ? void 0 : _c.kind) === kind) {
        return state.prevState.prevState;
    }
    if (((_f = (_e = (_d = state.prevState) === null || _d === void 0 ? void 0 : _d.prevState) === null || _e === void 0 ? void 0 : _e.prevState) === null || _f === void 0 ? void 0 : _f.kind) === kind) {
        return state.prevState.prevState.prevState;
    }
    if (((_k = (_j = (_h = (_g = state.prevState) === null || _g === void 0 ? void 0 : _g.prevState) === null || _h === void 0 ? void 0 : _h.prevState) === null || _j === void 0 ? void 0 : _j.prevState) === null || _k === void 0 ? void 0 : _k.kind) === kind) {
        return state.prevState.prevState.prevState.prevState;
    }
};
function getVariableCompletions(queryText, schema, token) {
    let variableName = null;
    let variableType;
    const definitions = Object.create({});
    runOnlineParser(queryText, (_, state) => {
        if ((state === null || state === void 0 ? void 0 : state.kind) === parser_1.RuleKinds.VARIABLE && state.name) {
            variableName = state.name;
        }
        if ((state === null || state === void 0 ? void 0 : state.kind) === parser_1.RuleKinds.NAMED_TYPE && variableName) {
            const parentDefinition = getParentDefinition(state, parser_1.RuleKinds.TYPE);
            if (parentDefinition === null || parentDefinition === void 0 ? void 0 : parentDefinition.type) {
                variableType = schema.getType(parentDefinition === null || parentDefinition === void 0 ? void 0 : parentDefinition.type);
            }
        }
        if (variableName && variableType) {
            if (!definitions[variableName]) {
                definitions[variableName] = {
                    detail: variableType.toString(),
                    insertText: token.string === '$' ? variableName : '$' + variableName,
                    label: variableName,
                    type: variableType,
                    kind: vscode_languageserver_types_1.CompletionItemKind.Variable,
                };
                variableName = null;
                variableType = null;
            }
        }
    });
    return autocompleteUtils_1.objectValues(definitions);
}
exports.getVariableCompletions = getVariableCompletions;
function getFragmentDefinitions(queryText) {
    const fragmentDefs = [];
    runOnlineParser(queryText, (_, state) => {
        if (state.kind === parser_1.RuleKinds.FRAGMENT_DEFINITION &&
            state.name &&
            state.type) {
            fragmentDefs.push({
                kind: parser_1.RuleKinds.FRAGMENT_DEFINITION,
                name: {
                    kind: graphql_1.Kind.NAME,
                    value: state.name,
                },
                selectionSet: {
                    kind: parser_1.RuleKinds.SELECTION_SET,
                    selections: [],
                },
                typeCondition: {
                    kind: parser_1.RuleKinds.NAMED_TYPE,
                    name: {
                        kind: graphql_1.Kind.NAME,
                        value: state.type,
                    },
                },
            });
        }
    });
    return fragmentDefs;
}
exports.getFragmentDefinitions = getFragmentDefinitions;
function getSuggestionsForVariableDefinition(token, schema, _kind) {
    const inputTypeMap = schema.getTypeMap();
    const inputTypes = autocompleteUtils_1.objectValues(inputTypeMap).filter(graphql_2.isInputType);
    return autocompleteUtils_1.hintList(token, inputTypes.map((type) => ({
        label: type.name,
        documentation: type.description,
        kind: vscode_languageserver_types_1.CompletionItemKind.Variable,
    })));
}
function getSuggestionsForDirective(token, state, schema, _kind) {
    if (state.prevState && state.prevState.kind) {
        const directives = schema
            .getDirectives()
            .filter(directive => canUseDirective(state.prevState, directive));
        return autocompleteUtils_1.hintList(token, directives.map(directive => ({
            label: directive.name,
            documentation: directive.description || '',
            kind: vscode_languageserver_types_1.CompletionItemKind.Function,
        })));
    }
    return [];
}
function getTokenAtPosition(queryText, cursor) {
    let styleAtCursor = null;
    let stateAtCursor = null;
    let stringAtCursor = null;
    const token = runOnlineParser(queryText, (stream, state, style, index) => {
        if (index === cursor.line) {
            if (stream.getCurrentPosition() >= cursor.character) {
                styleAtCursor = style;
                stateAtCursor = Object.assign({}, state);
                stringAtCursor = stream.current();
                return 'BREAK';
            }
        }
    });
    return {
        start: token.start,
        end: token.end,
        string: stringAtCursor || token.string,
        state: stateAtCursor || token.state,
        style: styleAtCursor || token.style,
    };
}
exports.getTokenAtPosition = getTokenAtPosition;
function runOnlineParser(queryText, callback) {
    const lines = queryText.split('\n');
    const parser = parser_1.onlineParser();
    let state = parser.startState();
    let style = '';
    let stream = new parser_1.CharacterStream('');
    for (let i = 0; i < lines.length; i++) {
        stream = new parser_1.CharacterStream(lines[i]);
        while (!stream.eol()) {
            style = parser.token(stream, state);
            const code = callback(stream, state, style, i);
            if (code === 'BREAK') {
                break;
            }
        }
        callback(stream, state, style, i);
        if (!state.kind) {
            state = parser.startState();
        }
    }
    return {
        start: stream.getStartOfToken(),
        end: stream.getCurrentPosition(),
        string: stream.current(),
        state,
        style,
    };
}
exports.runOnlineParser = runOnlineParser;
function canUseDirective(state, directive) {
    if (!state || !state.kind) {
        return false;
    }
    const kind = state.kind;
    const locations = directive.locations;
    switch (kind) {
        case parser_1.RuleKinds.QUERY:
            return locations.indexOf(graphql_1.DirectiveLocation.QUERY) !== -1;
        case parser_1.RuleKinds.MUTATION:
            return locations.indexOf(graphql_1.DirectiveLocation.MUTATION) !== -1;
        case parser_1.RuleKinds.SUBSCRIPTION:
            return locations.indexOf(graphql_1.DirectiveLocation.SUBSCRIPTION) !== -1;
        case parser_1.RuleKinds.FIELD:
        case parser_1.RuleKinds.ALIASED_FIELD:
            return locations.indexOf(graphql_1.DirectiveLocation.FIELD) !== -1;
        case parser_1.RuleKinds.FRAGMENT_DEFINITION:
            return locations.indexOf(graphql_1.DirectiveLocation.FRAGMENT_DEFINITION) !== -1;
        case parser_1.RuleKinds.FRAGMENT_SPREAD:
            return locations.indexOf(graphql_1.DirectiveLocation.FRAGMENT_SPREAD) !== -1;
        case parser_1.RuleKinds.INLINE_FRAGMENT:
            return locations.indexOf(graphql_1.DirectiveLocation.INLINE_FRAGMENT) !== -1;
        case parser_1.RuleKinds.SCHEMA_DEF:
            return locations.indexOf(graphql_1.DirectiveLocation.SCHEMA) !== -1;
        case parser_1.RuleKinds.SCALAR_DEF:
            return locations.indexOf(graphql_1.DirectiveLocation.SCALAR) !== -1;
        case parser_1.RuleKinds.OBJECT_TYPE_DEF:
            return locations.indexOf(graphql_1.DirectiveLocation.OBJECT) !== -1;
        case parser_1.RuleKinds.FIELD_DEF:
            return locations.indexOf(graphql_1.DirectiveLocation.FIELD_DEFINITION) !== -1;
        case parser_1.RuleKinds.INTERFACE_DEF:
            return locations.indexOf(graphql_1.DirectiveLocation.INTERFACE) !== -1;
        case parser_1.RuleKinds.UNION_DEF:
            return locations.indexOf(graphql_1.DirectiveLocation.UNION) !== -1;
        case parser_1.RuleKinds.ENUM_DEF:
            return locations.indexOf(graphql_1.DirectiveLocation.ENUM) !== -1;
        case parser_1.RuleKinds.ENUM_VALUE:
            return locations.indexOf(graphql_1.DirectiveLocation.ENUM_VALUE) !== -1;
        case parser_1.RuleKinds.INPUT_DEF:
            return locations.indexOf(graphql_1.DirectiveLocation.INPUT_OBJECT) !== -1;
        case parser_1.RuleKinds.INPUT_VALUE_DEF:
            const prevStateKind = state.prevState && state.prevState.kind;
            switch (prevStateKind) {
                case parser_1.RuleKinds.ARGUMENTS_DEF:
                    return (locations.indexOf(graphql_1.DirectiveLocation.ARGUMENT_DEFINITION) !== -1);
                case parser_1.RuleKinds.INPUT_DEF:
                    return (locations.indexOf(graphql_1.DirectiveLocation.INPUT_FIELD_DEFINITION) !== -1);
            }
    }
    return false;
}
exports.canUseDirective = canUseDirective;
function getTypeInfo(schema, tokenState) {
    let argDef;
    let argDefs;
    let directiveDef;
    let enumValue;
    let fieldDef;
    let inputType;
    let objectTypeDef;
    let objectFieldDefs;
    let parentType;
    let type;
    let interfaceDef;
    autocompleteUtils_1.forEachState(tokenState, state => {
        switch (state.kind) {
            case parser_1.RuleKinds.QUERY:
            case 'ShortQuery':
                type = schema.getQueryType();
                break;
            case parser_1.RuleKinds.MUTATION:
                type = schema.getMutationType();
                break;
            case parser_1.RuleKinds.SUBSCRIPTION:
                type = schema.getSubscriptionType();
                break;
            case parser_1.RuleKinds.INLINE_FRAGMENT:
            case parser_1.RuleKinds.FRAGMENT_DEFINITION:
                if (state.type) {
                    type = schema.getType(state.type);
                }
                break;
            case parser_1.RuleKinds.FIELD:
            case parser_1.RuleKinds.ALIASED_FIELD: {
                if (!type || !state.name) {
                    fieldDef = null;
                }
                else {
                    fieldDef = parentType
                        ? autocompleteUtils_1.getFieldDef(schema, parentType, state.name)
                        : null;
                    type = fieldDef ? fieldDef.type : null;
                }
                break;
            }
            case parser_1.RuleKinds.SELECTION_SET:
                parentType = graphql_2.getNamedType(type);
                break;
            case parser_1.RuleKinds.DIRECTIVE:
                directiveDef = state.name ? schema.getDirective(state.name) : null;
                break;
            case parser_1.RuleKinds.INTERFACE_DEF:
                if (state.name) {
                    objectTypeDef = null;
                    interfaceDef = new graphql_1.GraphQLInterfaceType({
                        name: state.name,
                        interfaces: [],
                        fields: {},
                    });
                }
                break;
            case parser_1.RuleKinds.OBJECT_TYPE_DEF:
                if (state.name) {
                    interfaceDef = null;
                    objectTypeDef = new graphql_1.GraphQLObjectType({
                        name: state.name,
                        interfaces: [],
                        fields: {},
                    });
                }
                break;
            case parser_1.RuleKinds.ARGUMENTS: {
                if (!state.prevState) {
                    argDefs = null;
                }
                else {
                    switch (state.prevState.kind) {
                        case parser_1.RuleKinds.FIELD:
                            argDefs = fieldDef && fieldDef.args;
                            break;
                        case parser_1.RuleKinds.DIRECTIVE:
                            argDefs =
                                directiveDef && directiveDef.args;
                            break;
                        case parser_1.RuleKinds.ALIASED_FIELD: {
                            const name = state.prevState && state.prevState.name;
                            if (!name) {
                                argDefs = null;
                                break;
                            }
                            const field = parentType
                                ? autocompleteUtils_1.getFieldDef(schema, parentType, name)
                                : null;
                            if (!field) {
                                argDefs = null;
                                break;
                            }
                            argDefs = field.args;
                            break;
                        }
                        default:
                            argDefs = null;
                            break;
                    }
                }
                break;
            }
            case parser_1.RuleKinds.ARGUMENT:
                if (argDefs) {
                    for (let i = 0; i < argDefs.length; i++) {
                        if (argDefs[i].name === state.name) {
                            argDef = argDefs[i];
                            break;
                        }
                    }
                }
                inputType = argDef && argDef.type;
                break;
            case parser_1.RuleKinds.ENUM_VALUE:
                const enumType = graphql_2.getNamedType(inputType);
                enumValue =
                    enumType instanceof graphql_2.GraphQLEnumType
                        ? enumType
                            .getValues()
                            .find((val) => val.value === state.name)
                        : null;
                break;
            case parser_1.RuleKinds.LIST_VALUE:
                const nullableType = graphql_2.getNullableType(inputType);
                inputType =
                    nullableType instanceof graphql_2.GraphQLList ? nullableType.ofType : null;
                break;
            case parser_1.RuleKinds.OBJECT_VALUE:
                const objectType = graphql_2.getNamedType(inputType);
                objectFieldDefs =
                    objectType instanceof graphql_2.GraphQLInputObjectType
                        ? objectType.getFields()
                        : null;
                break;
            case parser_1.RuleKinds.OBJECT_FIELD:
                const objectField = state.name && objectFieldDefs ? objectFieldDefs[state.name] : null;
                inputType = objectField && objectField.type;
                break;
            case parser_1.RuleKinds.NAMED_TYPE:
                if (state.name) {
                    type = schema.getType(state.name);
                }
                break;
        }
    });
    return {
        argDef,
        argDefs,
        directiveDef,
        enumValue,
        fieldDef,
        inputType,
        objectFieldDefs,
        parentType,
        type,
        interfaceDef,
        objectTypeDef,
    };
}
exports.getTypeInfo = getTypeInfo;
//# sourceMappingURL=getAutocompleteSuggestions.js.map