"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.getRange = exports.validateQuery = exports.getDiagnostics = exports.DIAGNOSTIC_SEVERITY = exports.SEVERITY = void 0;
const graphql_1 = require("graphql");
const graphql_2 = require("graphql");
const parser_1 = require("../parser");
const utils_1 = require("../utils");
exports.SEVERITY = {
    Error: 'Error',
    Warning: 'Warning',
    Information: 'Information',
    Hint: 'Hint',
};
exports.DIAGNOSTIC_SEVERITY = {
    [exports.SEVERITY.Error]: 1,
    [exports.SEVERITY.Warning]: 2,
    [exports.SEVERITY.Information]: 3,
    [exports.SEVERITY.Hint]: 4,
};
const invariant = (condition, message) => {
    if (!condition) {
        throw new Error(message);
    }
};
function getDiagnostics(query, schema = null, customRules, isRelayCompatMode, externalFragments) {
    let ast = null;
    if (externalFragments) {
        if (typeof externalFragments === 'string') {
            query += '\n\n' + externalFragments;
        }
        else {
            query +=
                '\n\n' +
                    externalFragments.reduce((agg, node) => {
                        agg += graphql_1.print(node) + '\n\n';
                        return agg;
                    }, '');
        }
    }
    try {
        ast = graphql_2.parse(query);
    }
    catch (error) {
        const range = getRange(error.locations[0], query);
        return [
            {
                severity: exports.DIAGNOSTIC_SEVERITY.Error,
                message: error.message,
                source: 'GraphQL: Syntax',
                range,
            },
        ];
    }
    return validateQuery(ast, schema, customRules, isRelayCompatMode);
}
exports.getDiagnostics = getDiagnostics;
function validateQuery(ast, schema = null, customRules, isRelayCompatMode) {
    if (!schema) {
        return [];
    }
    const validationErrorAnnotations = mapCat(utils_1.validateWithCustomRules(schema, ast, customRules, isRelayCompatMode), error => annotations(error, exports.DIAGNOSTIC_SEVERITY.Error, 'Validation'));
    const deprecationWarningAnnotations = mapCat(graphql_1.validate(schema, ast, [graphql_1.NoDeprecatedCustomRule]), error => annotations(error, exports.DIAGNOSTIC_SEVERITY.Warning, 'Deprecation'));
    return validationErrorAnnotations.concat(deprecationWarningAnnotations);
}
exports.validateQuery = validateQuery;
function mapCat(array, mapper) {
    return Array.prototype.concat.apply([], array.map(mapper));
}
function annotations(error, severity, type) {
    if (!error.nodes) {
        return [];
    }
    const highlightedNodes = [];
    error.nodes.forEach(node => {
        const highlightNode = node.kind !== 'Variable' && 'name' in node && node.name !== undefined
            ? node.name
            : 'variable' in node && node.variable !== undefined
                ? node.variable
                : node;
        if (highlightNode) {
            invariant(error.locations, 'GraphQL validation error requires locations.');
            const loc = error.locations[0];
            const highlightLoc = getLocation(highlightNode);
            const end = loc.column + (highlightLoc.end - highlightLoc.start);
            highlightedNodes.push({
                source: `GraphQL: ${type}`,
                message: error.message,
                severity,
                range: new utils_1.Range(new utils_1.Position(loc.line - 1, loc.column - 1), new utils_1.Position(loc.line - 1, end)),
            });
        }
    });
    return highlightedNodes;
}
function getRange(location, queryText) {
    const parser = parser_1.onlineParser();
    const state = parser.startState();
    const lines = queryText.split('\n');
    invariant(lines.length >= location.line, 'Query text must have more lines than where the error happened');
    let stream = null;
    for (let i = 0; i < location.line; i++) {
        stream = new parser_1.CharacterStream(lines[i]);
        while (!stream.eol()) {
            const style = parser.token(stream, state);
            if (style === 'invalidchar') {
                break;
            }
        }
    }
    invariant(stream, 'Expected Parser stream to be available.');
    const line = location.line - 1;
    const start = stream.getStartOfToken();
    const end = stream.getCurrentPosition();
    return new utils_1.Range(new utils_1.Position(line, start), new utils_1.Position(line, end));
}
exports.getRange = getRange;
function getLocation(node) {
    const typeCastedNode = node;
    const location = typeCastedNode.loc;
    invariant(location, 'Expected ASTNode to have a location.');
    return location;
}
//# sourceMappingURL=getDiagnostics.js.map