'use strict';

Object.defineProperty(exports, '__esModule', { value: true });

function _interopNamespace(e) {
  if (e && e.__esModule) { return e; } else {
    var n = {};
    if (e) {
      Object.keys(e).forEach(function (k) {
        var d = Object.getOwnPropertyDescriptor(e, k);
        Object.defineProperty(n, k, d.get ? d : {
          enumerable: true,
          get: function () {
            return e[k];
          }
        });
      });
    }
    n['default'] = e;
    return n;
  }
}

const graphql = require('graphql');
const path = require('path');
const common = require('@graphql-toolkit/common');
const fs = require('fs');
const graphqlTagPluck = require('@graphql-toolkit/graphql-tag-pluck');

function isSchemaText(obj) {
    return typeof obj === 'string';
}
function isWrappedSchemaJson(obj) {
    const json = obj;
    return json.data !== undefined && json.data.__schema !== undefined;
}
function isSchemaJson(obj) {
    const json = obj;
    return json !== undefined && json.__schema !== undefined;
}
function isSchemaObject(obj) {
    return obj instanceof graphql.GraphQLSchema;
}
function isSchemaAst(obj) {
    return obj.kind !== undefined;
}
function resolveExport(fileExport) {
    if (isSchemaObject(fileExport)) {
        return fileExport;
    }
    else if (isSchemaText(fileExport)) {
        return graphql.parse(fileExport);
    }
    else if (isWrappedSchemaJson(fileExport)) {
        return graphql.buildClientSchema(fileExport.data);
    }
    else if (isSchemaJson(fileExport)) {
        return graphql.buildClientSchema(fileExport);
    }
    else if (isSchemaAst(fileExport)) {
        return fileExport;
    }
    return null;
}
async function tryToLoadFromExport(rawFilePath) {
    let filePath = rawFilePath;
    try {
        filePath = common.fixWindowsPath(filePath);
        if (require && require.cache) {
            filePath = require.resolve(filePath);
            if (require.cache[filePath]) {
                delete require.cache[filePath];
            }
        }
        const rawExports = await new Promise(function (resolve) { resolve(_interopNamespace(require(filePath))); });
        if (rawExports) {
            let rawExport = rawExports.default || rawExports.schema || rawExports.typeDefs || rawExports;
            if (rawExport) {
                let exportValue = await rawExport;
                exportValue = await (exportValue.default || exportValue.schema || exportValue.typeDefs || exportValue);
                try {
                    return resolveExport(exportValue);
                }
                catch (e) {
                    throw new Error('Exported schema must be of type GraphQLSchema, text, AST, or introspection JSON.');
                }
            }
            else {
                throw new Error(`Invalid export from export file ${filePath}: missing default export or 'schema' export!`);
            }
        }
        else {
            throw new Error(`Invalid export from export file ${filePath}: empty export!`);
        }
    }
    catch (e) {
        throw new Error(`Unable to load from file "${filePath}": ${e.message}`);
    }
}
async function tryToLoadFromCodeAst(filePath, options) {
    const foundDoc = await graphqlTagPluck.gqlPluckFromFile(filePath, options && options.pluckConfig);
    if (foundDoc) {
        return graphql.parse(foundDoc);
    }
    else {
        return null;
    }
}
const CODE_FILE_EXTENSIONS = ['.ts', '.tsx', '.js', '.jsx', '.vue'];
class CodeFileLoader {
    loaderId() {
        return 'code-file';
    }
    async canLoad(pointer, options) {
        if (common.isValidPath(pointer)) {
            const extension = path.extname(pointer).toLowerCase();
            if (CODE_FILE_EXTENSIONS.includes(extension)) {
                const normalizedFilePath = path.isAbsolute(pointer) ? pointer : path.resolve(options.cwd || process.cwd(), pointer);
                if (fs.existsSync(normalizedFilePath)) {
                    return true;
                }
            }
        }
        return false;
    }
    async load(pointer, options) {
        const normalizedFilePath = path.isAbsolute(pointer) ? pointer : path.resolve(options.cwd || process.cwd(), pointer);
        try {
            const result = await tryToLoadFromCodeAst(normalizedFilePath, options);
            if (result) {
                return {
                    location: normalizedFilePath,
                    document: result,
                };
            }
        }
        catch (e) {
            common.debugLog(`Failed to load schema from code file "${normalizedFilePath}" using AST: ${e.message}`);
            throw e;
        }
        if ( !options.noRequire) {
            if (options && options.require) {
                await Promise.all(common.asArray(options.require).map(m => new Promise(function (resolve) { resolve(_interopNamespace(require(m))); })));
            }
            const schemaOrDocument = await tryToLoadFromExport(normalizedFilePath);
            if (schemaOrDocument instanceof graphql.GraphQLSchema) {
                const schema = common.fixSchemaAst(schemaOrDocument, options);
                return {
                    get document() {
                        return graphql.parse(common.printSchemaWithDirectives(schema));
                    },
                    schema,
                    location: normalizedFilePath,
                };
            }
            else {
                return {
                    document: schemaOrDocument,
                    location: normalizedFilePath,
                };
            }
        }
        return {
            document: null,
            location: normalizedFilePath,
        };
    }
}

exports.CodeFileLoader = CodeFileLoader;
