import { GraphQLSchema, parse, buildClientSchema } from 'graphql';
import { extname, isAbsolute, resolve } from 'path';
import { isValidPath, debugLog, asArray, fixSchemaAst, printSchemaWithDirectives, fixWindowsPath } from '@graphql-toolkit/common';
import { existsSync } from 'fs';
import { gqlPluckFromFile } from '@graphql-toolkit/graphql-tag-pluck';

function isSchemaText(obj) {
    return typeof obj === 'string';
}
function isWrappedSchemaJson(obj) {
    const json = obj;
    return json.data !== undefined && json.data.__schema !== undefined;
}
function isSchemaJson(obj) {
    const json = obj;
    return json !== undefined && json.__schema !== undefined;
}
function isSchemaObject(obj) {
    return obj instanceof GraphQLSchema;
}
function isSchemaAst(obj) {
    return obj.kind !== undefined;
}
function resolveExport(fileExport) {
    if (isSchemaObject(fileExport)) {
        return fileExport;
    }
    else if (isSchemaText(fileExport)) {
        return parse(fileExport);
    }
    else if (isWrappedSchemaJson(fileExport)) {
        return buildClientSchema(fileExport.data);
    }
    else if (isSchemaJson(fileExport)) {
        return buildClientSchema(fileExport);
    }
    else if (isSchemaAst(fileExport)) {
        return fileExport;
    }
    return null;
}
async function tryToLoadFromExport(rawFilePath) {
    let filePath = rawFilePath;
    try {
        filePath = fixWindowsPath(filePath);
        if (require && require.cache) {
            filePath = require.resolve(filePath);
            if (require.cache[filePath]) {
                delete require.cache[filePath];
            }
        }
        const rawExports = await import(filePath);
        if (rawExports) {
            let rawExport = rawExports.default || rawExports.schema || rawExports.typeDefs || rawExports;
            if (rawExport) {
                let exportValue = await rawExport;
                exportValue = await (exportValue.default || exportValue.schema || exportValue.typeDefs || exportValue);
                try {
                    return resolveExport(exportValue);
                }
                catch (e) {
                    throw new Error('Exported schema must be of type GraphQLSchema, text, AST, or introspection JSON.');
                }
            }
            else {
                throw new Error(`Invalid export from export file ${filePath}: missing default export or 'schema' export!`);
            }
        }
        else {
            throw new Error(`Invalid export from export file ${filePath}: empty export!`);
        }
    }
    catch (e) {
        throw new Error(`Unable to load from file "${filePath}": ${e.message}`);
    }
}
async function tryToLoadFromCodeAst(filePath, options) {
    const foundDoc = await gqlPluckFromFile(filePath, options && options.pluckConfig);
    if (foundDoc) {
        return parse(foundDoc);
    }
    else {
        return null;
    }
}
const CODE_FILE_EXTENSIONS = ['.ts', '.tsx', '.js', '.jsx', '.vue'];
class CodeFileLoader {
    loaderId() {
        return 'code-file';
    }
    async canLoad(pointer, options) {
        if (isValidPath(pointer)) {
            const extension = extname(pointer).toLowerCase();
            if (CODE_FILE_EXTENSIONS.includes(extension)) {
                const normalizedFilePath = isAbsolute(pointer) ? pointer : resolve(options.cwd || process.cwd(), pointer);
                if (existsSync(normalizedFilePath)) {
                    return true;
                }
            }
        }
        return false;
    }
    async load(pointer, options) {
        const normalizedFilePath = isAbsolute(pointer) ? pointer : resolve(options.cwd || process.cwd(), pointer);
        try {
            const result = await tryToLoadFromCodeAst(normalizedFilePath, options);
            if (result) {
                return {
                    location: normalizedFilePath,
                    document: result,
                };
            }
        }
        catch (e) {
            debugLog(`Failed to load schema from code file "${normalizedFilePath}" using AST: ${e.message}`);
            throw e;
        }
        if ( !options.noRequire) {
            if (options && options.require) {
                await Promise.all(asArray(options.require).map(m => import(m)));
            }
            const schemaOrDocument = await tryToLoadFromExport(normalizedFilePath);
            if (schemaOrDocument instanceof GraphQLSchema) {
                const schema = fixSchemaAst(schemaOrDocument, options);
                return {
                    get document() {
                        return parse(printSchemaWithDirectives(schema));
                    },
                    schema,
                    location: normalizedFilePath,
                };
            }
            else {
                return {
                    document: schemaOrDocument,
                    location: normalizedFilePath,
                };
            }
        }
        return {
            document: null,
            location: normalizedFilePath,
        };
    }
}

export { CodeFileLoader };
