import { parse, buildASTSchema, GraphQLScalarType, GraphQLObjectType, GraphQLInterfaceType, GraphQLEnumType, GraphQLUnionType, getDirectiveValues, Kind, defaultFieldResolver, GraphQLBoolean, GraphQLInt, GraphQLString, GraphQLFloat, GraphQLID, printSchema, print, printType, specifiedRules, validate } from 'graphql';
import isGlob from 'is-glob';
import { SchemaDirectiveVisitor } from '@kamilkisiela/graphql-tools';
import AggregateError from 'aggregate-error';
import { get, set } from 'lodash';

const asArray = (fns) => (Array.isArray(fns) ? fns : [fns]);
function chainFunctions(funcs) {
    if (funcs.length === 1) {
        return funcs[0];
    }
    return funcs.reduce((a, b) => (...args) => a(b(...args)));
}
function isEqual(a, b) {
    if (Array.isArray(a) && Array.isArray(b)) {
        if (a.length !== b.length)
            return false;
        for (var index = 0; index < a.length; index++) {
            if (a[index] !== b[index]) {
                return false;
            }
        }
        return true;
    }
    return a === b || (!a && !b);
}
function isNotEqual(a, b) {
    return !isEqual(a, b);
}
function isDocumentString(str) {
    // XXX: is-valid-path or is-glob treat SDL as a valid path
    // (`scalar Date` for example)
    // this why checking the extension is fast enough
    // and prevent from parsing the string in order to find out
    // if the string is a SDL
    if (/\.[a-z0-9]+$/i.test(str)) {
        return false;
    }
    try {
        parse(str);
        return true;
    }
    catch (e) {
        return false;
    }
}
const invalidPathRegex = /[‘“!#$%&+^<=>`]/;
function isValidPath(str) {
    return typeof str === 'string' && !isGlob(str) && !invalidPathRegex.test(str);
}

function debugLog(...args) {
    if (process && process.env && process.env.DEBUG && !process.env.GQL_TOOLKIT_NODEBUG) {
        console.log(...args);
    }
}

function extractFieldResolversFromObjectType(objectType, options) {
    const fieldResolvers = {};
    const fieldMap = objectType.getFields();
    let selectedFieldNames;
    if (options && options.selectedTypeDefs) {
        const invalidSchema = buildASTSchema(options.selectedTypeDefs);
        const typeMap = invalidSchema.getTypeMap();
        if (!(objectType.name in typeMap)) {
            return {};
        }
        else {
            const selectedObjectType = typeMap[objectType.name];
            selectedFieldNames = Object.keys(selectedObjectType.getFields());
        }
    }
    for (const fieldName in fieldMap) {
        if (selectedFieldNames && !selectedFieldNames.includes(fieldName)) {
            continue;
        }
        const fieldDefinition = fieldMap[fieldName];
        fieldResolvers[fieldName] = {
            subscribe: fieldDefinition.subscribe,
            resolve: fieldDefinition.resolve,
        };
    }
    if ('resolveType' in objectType) {
        fieldResolvers['__resolveType'] = objectType.resolveType;
    }
    if ('isTypeOf' in objectType) {
        fieldResolvers['__isTypeOf'] = objectType.isTypeOf;
    }
    return fieldResolvers;
}

function extractResolversFromSchema(schema, options) {
    const resolvers = {};
    const typeMap = schema.getTypeMap();
    let selectedTypeNames;
    if (options && options.selectedTypeDefs) {
        const invalidSchema = buildASTSchema(options.selectedTypeDefs);
        selectedTypeNames = Object.keys(invalidSchema.getTypeMap());
    }
    for (const typeName in typeMap) {
        if (!typeName.startsWith('__')) {
            const typeDef = typeMap[typeName];
            if (selectedTypeNames && !selectedTypeNames.includes(typeName)) {
                continue;
            }
            if (typeDef instanceof GraphQLScalarType) {
                resolvers[typeName] = typeDef;
            }
            else if (typeDef instanceof GraphQLObjectType || typeDef instanceof GraphQLInterfaceType) {
                resolvers[typeName] = extractFieldResolversFromObjectType(typeDef, {
                    selectedTypeDefs: options && options.selectedTypeDefs,
                });
            }
            else if (typeDef instanceof GraphQLEnumType) {
                const enumValues = typeDef.getValues();
                resolvers[typeName] = {};
                for (const { name, value } of enumValues) {
                    resolvers[typeName][name] = value;
                }
            }
            else if (typeDef instanceof GraphQLUnionType) {
                resolvers[typeName] = {
                    __resolveType: typeDef.resolveType,
                };
            }
        }
    }
    return resolvers;
}

const fixWindowsPath = (path) => path.split('\\').join('/');

const flattenArray = (arr) => arr.reduce((acc, next) => acc.concat(Array.isArray(next) ? flattenArray(next) : next), []);

function getDirectives(schema, node) {
    const schemaDirectives = schema && schema.getDirectives ? schema.getDirectives() : [];
    const astNode = node && node['astNode'];
    let result = {};
    if (astNode) {
        schemaDirectives.forEach((directive) => {
            const directiveValue = getDirectiveValues(directive, astNode);
            if (directiveValue !== undefined) {
                result[directive.name] = directiveValue || {};
            }
        });
    }
    return result;
}

function isObjectTypeDefinitionOrExtension(obj) {
    return obj && (obj.kind === 'ObjectTypeDefinition' || obj.kind === 'ObjectTypeExtension');
}
function parseDirectiveValue(value) {
    switch (value.kind) {
        case Kind.INT:
            return parseInt(value.value);
        case Kind.FLOAT:
            return parseFloat(value.value);
        case Kind.BOOLEAN:
            return Boolean(value.value);
        case Kind.STRING:
        case Kind.ENUM:
            return value.value;
        case Kind.LIST:
            return value.values.map(v => parseDirectiveValue(v));
        case Kind.OBJECT:
            return value.fields.reduce((prev, v) => ({ ...prev, [v.name.value]: parseDirectiveValue(v.value) }), {});
        case Kind.NULL:
            return null;
        default:
            return null;
    }
}
function getFieldsWithDirectives(documentNode) {
    const result = {};
    const allTypes = documentNode.definitions.filter(isObjectTypeDefinitionOrExtension);
    for (const type of allTypes) {
        const typeName = type.name.value;
        for (const field of type.fields) {
            if (field.directives && field.directives.length > 0) {
                const fieldName = field.name.value;
                const key = `${typeName}.${fieldName}`;
                const directives = field.directives.map(d => ({
                    name: d.name.value,
                    args: (d.arguments || []).reduce((prev, arg) => ({ ...prev, [arg.name.value]: parseDirectiveValue(arg.value) }), {}),
                }));
                result[key] = directives;
            }
        }
    }
    return result;
}

function getImplementingTypes(interfaceName, schema) {
    const allTypesMap = schema.getTypeMap();
    const result = [];
    for (const graphqlType of Object.values(allTypesMap)) {
        if (graphqlType instanceof GraphQLObjectType) {
            const allInterfaces = graphqlType.getInterfaces();
            if (allInterfaces.find(int => int.name === interfaceName)) {
                result.push(graphqlType.name);
            }
        }
    }
    return result;
}

function getSchemaDirectiveFromDirectiveResolver(directiveResolver) {
    return class extends SchemaDirectiveVisitor {
        visitFieldDefinition(field) {
            const resolver = directiveResolver;
            const originalResolver = field.resolve || defaultFieldResolver;
            const directiveArgs = this.args;
            field.resolve = (...args) => {
                const [source /* original args */, , context, info] = args;
                return resolver(async () => originalResolver.apply(field, args), source, directiveArgs, context, info);
            };
        }
    };
}

const IGNORED_SCALARS = [GraphQLBoolean.name, GraphQLInt.name, GraphQLString.name, GraphQLFloat.name, GraphQLID.name];
function printSchemaWithDirectives(schema) {
    const allTypes = schema.getTypeMap();
    const allTypesAst = Object.keys(allTypes)
        .map(key => allTypes[key].astNode)
        .filter(a => a);
    const allTypesExtensionAst = Object.keys(allTypes)
        .map(key => allTypes[key].extensionASTNodes)
        .filter(a => a)
        .reduce((prev, curr) => [...prev, ...curr], []);
    const noAstTypes = Object.keys(allTypes)
        .map(key => (IGNORED_SCALARS.includes(key) || key.startsWith('__') || allTypes[key].astNode ? null : allTypes[key]))
        .filter(a => a);
    const directivesAst = schema
        .getDirectives()
        .map(def => def.astNode)
        .filter(a => a);
    if (allTypesAst.length === 0 && directivesAst.length === 0 && allTypesExtensionAst.length === 0) {
        return printSchema(schema);
    }
    else {
        const astTypesPrinted = [...allTypesAst, ...directivesAst, ...allTypesExtensionAst].map(ast => print(ast));
        const nonAstPrinted = noAstTypes.map(p => printType(p));
        return [...astTypesPrinted, ...nonAstPrinted].join('\n');
    }
}

const DEFAULT_IGNORED_RULES = ['NoUnusedFragments', 'NoUnusedVariables', 'KnownDirectives'];
const DEFAULT_EFFECTIVE_RULES = specifiedRules.filter((f) => !DEFAULT_IGNORED_RULES.includes(f.name));
const validateGraphQlDocuments = async (schema, documentFiles, effectiveRules = DEFAULT_EFFECTIVE_RULES) => {
    const allFragments = [];
    const allFragments$ = Promise.all(documentFiles.map(async (documentFile) => {
        if (documentFile.document) {
            for (const definitionNode of documentFile.document.definitions) {
                if (definitionNode.kind === Kind.FRAGMENT_DEFINITION) {
                    allFragments.push(definitionNode);
                }
            }
        }
    }));
    await allFragments$;
    const allErrors = [];
    const allErrors$ = Promise.all(documentFiles.map(async (documentFile) => {
        const documentToValidate = {
            kind: Kind.DOCUMENT,
            definitions: [...allFragments, ...documentFile.document.definitions].filter((d, index, arr) => {
                if (d.kind === Kind.FRAGMENT_DEFINITION) {
                    const foundIndex = arr.findIndex(i => i.kind === Kind.FRAGMENT_DEFINITION && i.name.value === d.name.value);
                    if (foundIndex !== index) {
                        return false;
                    }
                }
                return true;
            }),
        };
        const errors = validate(schema, documentToValidate, effectiveRules);
        if (errors.length > 0) {
            allErrors.push({
                filePath: documentFile.location,
                errors,
            });
        }
    }));
    await allErrors$;
    return allErrors;
};
function checkValidationErrors(loadDocumentErrors) {
    if (loadDocumentErrors.length > 0) {
        const errors = [];
        for (const loadDocumentError of loadDocumentErrors) {
            for (const graphQLError of loadDocumentError.errors) {
                const error = new Error();
                error.name = 'GraphQLDocumentError';
                error.message = `${error.name}: ${graphQLError.message}`;
                error.stack = error.message;
                graphQLError.locations.forEach(location => (error.stack += `\n    at ${loadDocumentError.filePath}:${location.line}:${location.column}`));
                errors.push(error);
            }
        }
        throw new AggregateError(errors);
    }
}

function resolveRelevantMappings(resolvers, path, allMappings) {
    const splitted = path.split('.');
    if (splitted.length === 2) {
        const typeName = splitted[0];
        const fieldName = splitted[1];
        if (fieldName === '*') {
            return flattenArray(Object.keys(resolvers[typeName]).map(field => resolveRelevantMappings(resolvers, `${typeName}.${field}`, allMappings))).filter(mapItem => !allMappings[mapItem]);
        }
        else {
            const paths = [];
            if (resolvers[typeName] && resolvers[typeName][fieldName]) {
                if (resolvers[typeName][fieldName]['subscribe']) {
                    paths.push(path + '.subscribe');
                }
                if (resolvers[typeName][fieldName]['resolve']) {
                    paths.push(path + '.resolve');
                }
                if (typeof resolvers[typeName][fieldName] === 'function') {
                    paths.push(path);
                }
            }
            return paths;
        }
    }
    else if (splitted.length === 1) {
        const typeName = splitted[0];
        return flattenArray(Object.keys(resolvers[typeName]).map(fieldName => resolveRelevantMappings(resolvers, `${typeName}.${fieldName}`, allMappings)));
    }
    return [];
}
/**
 * Wraps the resolvers object with the resolvers composition objects.
 * Implemented as a simple and basic middleware mechanism.
 *
 * @param resolvers - resolvers object
 * @param mapping - resolvers composition mapping
 * @hidden
 */
function composeResolvers(resolvers, mapping = {}) {
    const mappingResult = {};
    Object.keys(mapping).map((resolverPath) => {
        if (mapping[resolverPath] instanceof Array || typeof mapping[resolverPath] === 'function') {
            const composeFns = mapping[resolverPath];
            const relevantFields = resolveRelevantMappings(resolvers, resolverPath, mapping);
            relevantFields.forEach((path) => {
                mappingResult[path] = asArray(composeFns);
            });
        }
        else {
            Object.keys(mapping[resolverPath]).map(fieldName => {
                const composeFns = mapping[resolverPath][fieldName];
                const relevantFields = resolveRelevantMappings(resolvers, resolverPath + '.' + fieldName, mapping);
                relevantFields.forEach((path) => {
                    mappingResult[path] = asArray(composeFns);
                });
            });
        }
    });
    Object.keys(mappingResult).forEach(path => {
        const fns = chainFunctions([...asArray(mappingResult[path]), () => get(resolvers, path)]);
        set(resolvers, path, fns());
    });
    return resolvers;
}

function fixSchemaAst(schema, options) {
    if (!schema.astNode) {
        Object.defineProperty(schema, 'astNode', {
            get: () => {
                return buildASTSchema(parse(printSchemaWithDirectives(schema)), {
                    commentDescriptions: true,
                    ...(options || {}),
                }).astNode;
            },
        });
        Object.defineProperty(schema, 'extensionASTNodes', {
            get: () => {
                return buildASTSchema(parse(printSchemaWithDirectives(schema)), {
                    commentDescriptions: true,
                    ...(options || {}),
                }).extensionASTNodes;
            },
        });
    }
    return schema;
}

export { asArray, chainFunctions, checkValidationErrors, composeResolvers, debugLog, extractFieldResolversFromObjectType, extractResolversFromSchema, fixSchemaAst, fixWindowsPath, flattenArray, getDirectives, getFieldsWithDirectives, getImplementingTypes, getSchemaDirectiveFromDirectiveResolver, isDocumentString, isEqual, isNotEqual, isValidPath, printSchemaWithDirectives, validateGraphQlDocuments };
