'use strict';

Object.defineProperty(exports, '__esModule', { value: true });

const path = require('path');
const graphql = require('graphql');
const common = require('@graphql-toolkit/common');
const fs = require('fs');

function stripBOM(content) {
    content = content.toString();
    // Remove byte order marker. This catches EF BB BF (the UTF-8 BOM)
    // because the buffer-to-string conversion in `fs.readFileSync()`
    // translates it to FEFF, the UTF-16 BOM.
    if (content.charCodeAt(0) === 0xfeff) {
        content = content.slice(1);
    }
    return content;
}
function parseBOM(content) {
    return JSON.parse(stripBOM(content));
}
class JsonFileLoader {
    loaderId() {
        return 'json-file';
    }
    async canLoad(pointer, options) {
        if (common.isValidPath(pointer)) {
            const extension = path.extname(pointer).toLowerCase();
            if (extension === '.json') {
                const normalizedFilePath = path.isAbsolute(pointer) ? pointer : path.resolve(options.cwd || process.cwd(), pointer);
                if (fs.existsSync(normalizedFilePath)) {
                    return true;
                }
            }
        }
        return false;
    }
    async load(pointer, options) {
        return new Promise((resolve, reject) => {
            const normalizedFilepath = path.isAbsolute(pointer) ? pointer : path.resolve(options.cwd || process.cwd(), pointer);
            if (fs.existsSync(normalizedFilepath)) {
                try {
                    const fileContent = fs.readFileSync(normalizedFilepath, 'utf8');
                    if (!fileContent) {
                        reject(`Unable to read local introspection file: ${normalizedFilepath}`);
                    }
                    let introspection = parseBOM(fileContent);
                    if (introspection['data']) {
                        introspection = introspection['data'];
                    }
                    if (!introspection.__schema) {
                        throw new Error('Invalid schema provided!');
                    }
                    const schema = graphql.buildClientSchema(introspection, options);
                    resolve({
                        location: pointer,
                        get document() {
                            return graphql.parse(common.printSchemaWithDirectives(schema));
                        },
                        schema,
                    });
                }
                catch (e) {
                    reject(e);
                }
            }
            else {
                reject(`Unable to locate local introspection file: ${normalizedFilepath}`);
            }
        });
    }
}

exports.JsonFileLoader = JsonFileLoader;
