import { extname, isAbsolute, resolve } from 'path';
import { buildClientSchema, parse } from 'graphql';
import { isValidPath, printSchemaWithDirectives } from '@graphql-toolkit/common';
import { existsSync, readFileSync } from 'fs';

function stripBOM(content) {
    content = content.toString();
    // Remove byte order marker. This catches EF BB BF (the UTF-8 BOM)
    // because the buffer-to-string conversion in `fs.readFileSync()`
    // translates it to FEFF, the UTF-16 BOM.
    if (content.charCodeAt(0) === 0xfeff) {
        content = content.slice(1);
    }
    return content;
}
function parseBOM(content) {
    return JSON.parse(stripBOM(content));
}
class JsonFileLoader {
    loaderId() {
        return 'json-file';
    }
    async canLoad(pointer, options) {
        if (isValidPath(pointer)) {
            const extension = extname(pointer).toLowerCase();
            if (extension === '.json') {
                const normalizedFilePath = isAbsolute(pointer) ? pointer : resolve(options.cwd || process.cwd(), pointer);
                if (existsSync(normalizedFilePath)) {
                    return true;
                }
            }
        }
        return false;
    }
    async load(pointer, options) {
        return new Promise((resolve$1, reject) => {
            const normalizedFilepath = isAbsolute(pointer) ? pointer : resolve(options.cwd || process.cwd(), pointer);
            if (existsSync(normalizedFilepath)) {
                try {
                    const fileContent = readFileSync(normalizedFilepath, 'utf8');
                    if (!fileContent) {
                        reject(`Unable to read local introspection file: ${normalizedFilepath}`);
                    }
                    let introspection = parseBOM(fileContent);
                    if (introspection['data']) {
                        introspection = introspection['data'];
                    }
                    if (!introspection.__schema) {
                        throw new Error('Invalid schema provided!');
                    }
                    const schema = buildClientSchema(introspection, options);
                    resolve$1({
                        location: pointer,
                        get document() {
                            return parse(printSchemaWithDirectives(schema));
                        },
                        schema,
                    });
                }
                catch (e) {
                    reject(e);
                }
            }
            else {
                reject(`Unable to locate local introspection file: ${normalizedFilepath}`);
            }
        });
    }
}

export { JsonFileLoader };
