/**
 * gridstack.component.ts 11.1.0
 * Copyright (c) 2022-2024 Alain Dumesny - see GridStack root license
 */
import { Component, ContentChildren, EventEmitter, Input, Output, ViewChild, ViewContainerRef, reflectComponentType } from '@angular/core';
import { GridStack } from 'gridstack';
import { GridstackItemComponent } from './gridstack-item.component';
import * as i0 from "@angular/core";
import * as i1 from "@angular/common";
/**
 * HTML Component Wrapper for gridstack, in combination with GridstackItemComponent for the items
 */
export class GridstackComponent {
    /** initial options for creation of the grid */
    set options(val) { this._options = val; }
    /** return the current running options */
    get options() { return this._grid?.opts || this._options || {}; }
    /** return the native element that contains grid specific fields as well */
    get el() { return this.elementRef.nativeElement; }
    /** return the GridStack class */
    get grid() { return this._grid; }
    /**
     * stores the selector -> Type mapping, so we can create items dynamically from a string.
     * Unfortunately Ng doesn't provide public access to that mapping.
     */
    static { this.selectorToType = {}; }
    /** add a list of ng Component to be mapped to selector */
    static addComponentToSelectorType(typeList) {
        typeList.forEach(type => GridstackComponent.selectorToType[GridstackComponent.getSelector(type)] = type);
    }
    /** return the ng Component selector */
    static getSelector(type) {
        return reflectComponentType(type).selector;
    }
    constructor(
    // protected readonly zone: NgZone,
    // protected readonly cd: ChangeDetectorRef,
    elementRef) {
        this.elementRef = elementRef;
        /** individual list of GridStackEvent callbacks handlers as output
         * otherwise use this.grid.on('name1 name2 name3', callback) to handle multiple at once
         * see https://github.com/gridstack/gridstack.js/blob/master/demo/events.js#L4
         *
         * Note: camel casing and 'CB' added at the end to prevent @angular-eslint/no-output-native
         * eg: 'change' would trigger the raw CustomEvent so use different name.
         */
        this.addedCB = new EventEmitter();
        this.changeCB = new EventEmitter();
        this.disableCB = new EventEmitter();
        this.dragCB = new EventEmitter();
        this.dragStartCB = new EventEmitter();
        this.dragStopCB = new EventEmitter();
        this.droppedCB = new EventEmitter();
        this.enableCB = new EventEmitter();
        this.removedCB = new EventEmitter();
        this.resizeCB = new EventEmitter();
        this.resizeStartCB = new EventEmitter();
        this.resizeStopCB = new EventEmitter();
        this.el._gridComp = this;
    }
    ngOnInit() {
        // init ourself before any template children are created since we track them below anyway - no need to double create+update widgets
        this.loaded = !!this.options?.children?.length;
        this._grid = GridStack.init(this._options, this.el);
        delete this._options; // GS has it now
        this.checkEmpty();
    }
    /** wait until after all DOM is ready to init gridstack children (after angular ngFor and sub-components run first) */
    ngAfterContentInit() {
        // track whenever the children list changes and update the layout...
        this._sub = this.gridstackItems?.changes.subscribe(() => this.updateAll());
        // ...and do this once at least unless we loaded children already
        if (!this.loaded)
            this.updateAll();
        this.hookEvents(this.grid);
    }
    ngOnDestroy() {
        this.unhookEvents(this._grid);
        this._sub?.unsubscribe();
        this._grid?.destroy();
        delete this._grid;
        delete this.el._gridComp;
        delete this.container;
        delete this.ref;
    }
    /**
     * called when the TEMPLATE list of items changes - get a list of nodes and
     * update the layout accordingly (which will take care of adding/removing items changed by Angular)
     */
    updateAll() {
        if (!this.grid)
            return;
        const layout = [];
        this.gridstackItems?.forEach(item => {
            layout.push(item.options);
            item.clearOptions();
        });
        this.grid.load(layout); // efficient that does diffs only
    }
    /** check if the grid is empty, if so show alternative content */
    checkEmpty() {
        if (!this.grid)
            return;
        const isEmpty = !this.grid.engine.nodes.length;
        if (isEmpty === this.isEmpty)
            return;
        this.isEmpty = isEmpty;
        // this.cd.detectChanges();
    }
    /** get all known events as easy to use Outputs for convenience */
    hookEvents(grid) {
        if (!grid)
            return;
        grid
            .on('added', (event, nodes) => { this.checkEmpty(); this.addedCB.emit({ event, nodes }); })
            .on('change', (event, nodes) => this.changeCB.emit({ event, nodes }))
            .on('disable', (event) => this.disableCB.emit({ event }))
            .on('drag', (event, el) => this.dragCB.emit({ event, el }))
            .on('dragstart', (event, el) => this.dragStartCB.emit({ event, el }))
            .on('dragstop', (event, el) => this.dragStopCB.emit({ event, el }))
            .on('dropped', (event, previousNode, newNode) => this.droppedCB.emit({ event, previousNode, newNode }))
            .on('enable', (event) => this.enableCB.emit({ event }))
            .on('removed', (event, nodes) => { this.checkEmpty(); this.removedCB.emit({ event, nodes }); })
            .on('resize', (event, el) => this.resizeCB.emit({ event, el }))
            .on('resizestart', (event, el) => this.resizeStartCB.emit({ event, el }))
            .on('resizestop', (event, el) => this.resizeStopCB.emit({ event, el }));
    }
    unhookEvents(grid) {
        if (!grid)
            return;
        grid.off('added change disable drag dragstart dragstop dropped enable removed resize resizestart resizestop');
    }
    static { this.ɵfac = i0.ɵɵngDeclareFactory({ minVersion: "12.0.0", version: "18.2.8", ngImport: i0, type: GridstackComponent, deps: [{ token: i0.ElementRef }], target: i0.ɵɵFactoryTarget.Component }); }
    static { this.ɵcmp = i0.ɵɵngDeclareComponent({ minVersion: "14.0.0", version: "18.2.8", type: GridstackComponent, selector: "gridstack", inputs: { options: "options", isEmpty: "isEmpty" }, outputs: { addedCB: "addedCB", changeCB: "changeCB", disableCB: "disableCB", dragCB: "dragCB", dragStartCB: "dragStartCB", dragStopCB: "dragStopCB", droppedCB: "droppedCB", enableCB: "enableCB", removedCB: "removedCB", resizeCB: "resizeCB", resizeStartCB: "resizeStartCB", resizeStopCB: "resizeStopCB" }, queries: [{ propertyName: "gridstackItems", predicate: GridstackItemComponent }], viewQueries: [{ propertyName: "container", first: true, predicate: ["container"], descendants: true, read: ViewContainerRef, static: true }], ngImport: i0, template: `
    <!-- content to show when when grid is empty, like instructions on how to add widgets -->
    <ng-content select="[empty-content]" *ngIf="isEmpty"></ng-content>
    <!-- where dynamic items go -->
    <ng-template #container></ng-template>
    <!-- where template items go -->
    <ng-content></ng-content>
  `, isInline: true, styles: [":host{display:block}\n"], dependencies: [{ kind: "directive", type: i1.NgIf, selector: "[ngIf]", inputs: ["ngIf", "ngIfThen", "ngIfElse"] }] }); }
}
i0.ɵɵngDeclareClassMetadata({ minVersion: "12.0.0", version: "18.2.8", ngImport: i0, type: GridstackComponent, decorators: [{
            type: Component,
            args: [{ selector: 'gridstack', template: `
    <!-- content to show when when grid is empty, like instructions on how to add widgets -->
    <ng-content select="[empty-content]" *ngIf="isEmpty"></ng-content>
    <!-- where dynamic items go -->
    <ng-template #container></ng-template>
    <!-- where template items go -->
    <ng-content></ng-content>
  `, styles: [":host{display:block}\n"] }]
        }], ctorParameters: () => [{ type: i0.ElementRef }], propDecorators: { gridstackItems: [{
                type: ContentChildren,
                args: [GridstackItemComponent]
            }], container: [{
                type: ViewChild,
                args: ['container', { read: ViewContainerRef, static: true }]
            }], options: [{
                type: Input
            }], isEmpty: [{
                type: Input
            }], addedCB: [{
                type: Output
            }], changeCB: [{
                type: Output
            }], disableCB: [{
                type: Output
            }], dragCB: [{
                type: Output
            }], dragStartCB: [{
                type: Output
            }], dragStopCB: [{
                type: Output
            }], droppedCB: [{
                type: Output
            }], enableCB: [{
                type: Output
            }], removedCB: [{
                type: Output
            }], resizeCB: [{
                type: Output
            }], resizeStartCB: [{
                type: Output
            }], resizeStopCB: [{
                type: Output
            }] } });
/**
 * can be used when a new item needs to be created, which we do as a Angular component, or deleted (skip)
 **/
export function gsCreateNgComponents(host, n, add, isGrid) {
    if (add) {
        //
        // create the component dynamically - see https://angular.io/docs/ts/latest/cookbook/dynamic-component-loader.html
        //
        if (!host)
            return;
        if (isGrid) {
            // TODO: figure out how to create ng component inside regular Div. need to access app injectors...
            // if (!container) {
            //   const hostElement: Element = host;
            //   const environmentInjector: EnvironmentInjector;
            //   grid = createComponent(GridstackComponent, {environmentInjector, hostElement})?.instance;
            // }
            const gridItemComp = host.parentElement?._gridItemComp;
            if (!gridItemComp)
                return;
            // check if gridItem has a child component with 'container' exposed to create under..
            const container = gridItemComp.childWidget?.container || gridItemComp.container;
            const gridRef = container?.createComponent(GridstackComponent);
            const grid = gridRef?.instance;
            if (!grid)
                return;
            grid.ref = gridRef;
            grid.options = n;
            return grid.el;
        }
        else {
            const gridComp = host._gridComp;
            const gridItemRef = gridComp?.container?.createComponent(GridstackItemComponent);
            const gridItem = gridItemRef?.instance;
            if (!gridItem)
                return;
            gridItem.ref = gridItemRef;
            // define what type of component to create as child, OR you can do it GridstackItemComponent template, but this is more generic
            const selector = n.selector;
            const type = selector ? GridstackComponent.selectorToType[selector] : undefined;
            if (type) {
                // shared code to create our selector component
                const createComp = () => {
                    const childWidget = gridItem.container?.createComponent(type)?.instance;
                    // if proper BaseWidget subclass, save it and load additional data
                    if (childWidget && typeof childWidget.serialize === 'function' && typeof childWidget.deserialize === 'function') {
                        gridItem.childWidget = childWidget;
                        childWidget.deserialize(n);
                    }
                };
                const lazyLoad = n.lazyLoad || n.grid?.opts?.lazyLoad && n.lazyLoad !== false;
                if (lazyLoad) {
                    if (!n.visibleObservable) {
                        n.visibleObservable = new IntersectionObserver(([entry]) => {
                            if (entry.isIntersecting) {
                                n.visibleObservable?.disconnect();
                                delete n.visibleObservable;
                                createComp();
                            }
                        });
                        window.setTimeout(() => n.visibleObservable?.observe(gridItem.el)); // wait until callee sets position attributes
                    }
                }
                else
                    createComp();
            }
            return gridItem.el;
        }
    }
    else {
        //
        // REMOVE - have to call ComponentRef:destroy() for dynamic objects to correctly remove themselves
        // Note: this will destroy all children dynamic components as well: gridItem -> childWidget
        //
        if (isGrid) {
            const grid = n.el?._gridComp;
            if (grid?.ref)
                grid.ref.destroy();
            else
                grid?.ngOnDestroy();
        }
        else {
            const gridItem = n.el?._gridItemComp;
            if (gridItem?.ref)
                gridItem.ref.destroy();
            else
                gridItem?.ngOnDestroy();
        }
    }
    return;
}
/**
 * called for each item in the grid - check if additional information needs to be saved.
 * Note: since this is options minus gridstack protected members using Utils.removeInternalForSave(),
 * this typically doesn't need to do anything. However your custom Component @Input() are now supported
 * using BaseWidget.serialize()
 */
export function gsSaveAdditionalNgInfo(n, w) {
    const gridItem = n.el?._gridItemComp;
    if (gridItem) {
        const input = gridItem.childWidget?.serialize();
        if (input) {
            w.input = input;
        }
        return;
    }
    // else check if Grid
    const grid = n.el?._gridComp;
    if (grid) {
        //.... save any custom data
    }
}
//# sourceMappingURL=data:application/json;base64,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