"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.parseFragment = parseFragment;
Object.defineProperty(exports, "parsePartialTagEnd", {
  enumerable: true,
  get: function () {
    return _partialTagsParser.parsePartialTagEnd;
  }
});

var _parse = _interopRequireDefault(require("parse5"));

var _closingTagsParser = require("./closing-tags-parser");

var _partialTagsParser = require("./partial-tags-parser");

var _emptyElements = require("../empty-elements");

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

function demandLocation(node) {
  if (node.hasOwnProperty('sourceCodeLocation')) {
    const withSource = node;
    const sourceCodeLocation = withSource.sourceCodeLocation;

    if (sourceCodeLocation) {
      return sourceCodeLocation;
    }
  }

  throw new Error('Tree must contain source info');
}

function demandElementLocation(node) {
  const sourceCodeLocation = demandLocation(node);

  if (sourceCodeLocation.hasOwnProperty('startTag')) {
    return sourceCodeLocation;
  }

  throw new Error('Expected source code info to have startTag');
}

function getPropertyValuePairs(attributes) {
  return attributes.map(attribute => [attribute.name, attribute.value]);
}

const HTML_RESERVED_CHARACTERS_REGEX = /['"&<>]/g;

function getNodeDOMOperations(node, fragment, startOffset) {
  const operations = [];
  const sourceLocation = demandLocation(node);

  if (sourceLocation.startOffset > startOffset) {
    // Handle leading garbage
    const garbage = fragment.substring(startOffset, sourceLocation.startOffset);
    const closingTags = (0, _closingTagsParser.parseClosingTags)(garbage);

    if (closingTags.remaining) {
      return undefined;
    }

    operations.push(...getOperationsFromClosingTags(closingTags.tags));
  }

  if (node.nodeName === '#text') {
    const textNode = node;
    const textSource = fragment.substring(sourceLocation.startOffset, sourceLocation.endOffset);

    if (textSource.match(HTML_RESERVED_CHARACTERS_REGEX)) {
      // parse5 will ignore unmatched closing tags inside a text node
      const closingTags = (0, _closingTagsParser.parseClosingTags)(textSource);
      operations.push(...getOperationsFromClosingTags(closingTags.tags));
      return {
        operations,
        endOffset: sourceLocation.endOffset - closingTags.remaining.length
      };
    } else {
      operations.push({
        type: 'text',
        value: {
          text: textNode.value
        }
      });
    }
  } else if (!node.nodeName.startsWith('#')) {
    // Hash prefix is used for other "special" node types like comments
    const elementNode = node;
    const elementLocation = demandElementLocation(node);

    if ((0, _emptyElements.isEmptyElement)(elementNode.tagName)) {
      operations.push({
        type: 'emptyElement',
        value: {
          propertyValuePairs: getPropertyValuePairs(elementNode.attrs),
          tagName: elementNode.tagName
        }
      });
    } else {
      operations.push({
        type: 'elementOpen',
        value: {
          propertyValuePairs: getPropertyValuePairs(elementNode.attrs),
          tagName: elementNode.tagName
        }
      });
      const contentsStartOffset = elementLocation.startTag.endOffset;
      let offset = contentsStartOffset;

      for (let node of elementNode.childNodes) {
        const intermediateResult = getNodeDOMOperations(node, fragment, offset);

        if (intermediateResult) {
          operations.push(...intermediateResult.operations);
          offset = intermediateResult.endOffset;
        } else {
          // Parse failure
          return intermediateResult;
        }
      }

      const endTagLocation = elementLocation.endTag;

      if (endTagLocation) {
        const contentsEndOffset = endTagLocation.startOffset;

        if (offset < contentsEndOffset) {
          // Garbage inside of tags
          const remaining = fragment.substring(offset, contentsEndOffset);
          const closingTags = (0, _closingTagsParser.parseClosingTags)(remaining);

          if (closingTags.remaining) {
            return undefined;
          } else {
            operations.push(...getOperationsFromClosingTags(closingTags.tags));
          }
        }

        operations.push({
          type: 'elementClose',
          value: {
            tagName: elementNode.tagName
          }
        });
      } else {
        // Can't trust sourceLocation.endOffset, there may be trailing garbage
        // that is nonetheless included in the source range
        return {
          operations,
          endOffset: offset
        };
      }
    }
  }

  return {
    operations,
    endOffset: sourceLocation.endOffset
  };
}

function parseFromAST(ast, fragment) {
  const operations = [];
  let offset = 0;

  for (let child of ast.childNodes) {
    const intermediateResult = getNodeDOMOperations(child, fragment, offset);

    if (intermediateResult) {
      operations.push(...intermediateResult.operations);
      offset = intermediateResult.endOffset;
    } else {
      // Parse failed
      return {
        type: 'invalidFragment'
      };
    }
  }

  const remaining = fragment.substring(offset);
  const closingTags = (0, _closingTagsParser.parseClosingTags)(remaining);
  operations.push(...getOperationsFromClosingTags(closingTags.tags));

  if (closingTags.remaining) {
    const partialTag = (0, _partialTagsParser.parseOpenPartialTag)(closingTags.remaining);

    if (partialTag) {
      return {
        type: 'openPartialTag',
        value: {
          leadingOperations: operations,
          tagName: partialTag.tagName,
          content: partialTag.content
        }
      };
    }

    return {
      type: 'invalidFragment'
    };
  }

  return {
    type: 'fullTags',
    value: {
      operations
    }
  };
}

function getOperationsFromClosingTags(tags) {
  return tags.map(tag => {
    switch (tag.type) {
      case 'closingTag':
        {
          return {
            type: 'elementClose',
            value: {
              tagName: tag.value.tagName
            }
          };
        }

      case 'closingTagsInterstitialText':
        {
          return {
            type: 'text',
            value: {
              text: tag.value.text
            }
          };
        }
    }
  });
}

function parseFragment(fragment) {
  if (fragment === '') {
    return {
      type: 'fullTags',
      value: {
        operations: []
      }
    };
  }

  if (fragment.indexOf('<') === -1 && fragment.indexOf('>') === -1) {
    // As an optimization, bail out early if there are no HTML tags here
    return {
      type: 'fullTags',
      value: {
        operations: [{
          type: 'text',
          value: {
            text: fragment
          }
        }]
      }
    };
  }

  const ast = _parse.default.parseFragment(fragment, {
    sourceCodeLocationInfo: true
  });

  return parseFromAST(ast, fragment);
}