"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getDefaultIdom = getDefaultIdom;
exports.checkRevision = checkRevision;
exports.template = template;
exports.wrapProgram = wrapProgram;
exports.resolvePartial = resolvePartial;
exports.invokePartial = invokePartial;
exports.noop = noop;

var Utils = _interopRequireWildcard(require("./vendor/handlebars/utils"));

var _exception = _interopRequireDefault(require("./vendor/handlebars/exception"));

var _base = require("./vendor/handlebars/base");

var _idomTextBackend = require("./core/backend/idom-text-backend");

var _attributesParser = require("./core/attributes-parser");

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

function _interopRequireWildcard(obj) { if (obj && obj.__esModule) { return obj; } else { var newObj = {}; if (obj != null) { for (var key in obj) { if (Object.prototype.hasOwnProperty.call(obj, key)) { var desc = Object.defineProperty && Object.getOwnPropertyDescriptor ? Object.getOwnPropertyDescriptor(obj, key) : {}; if (desc.get || desc.set) { Object.defineProperty(newObj, key, desc); } else { newObj[key] = obj[key]; } } } } newObj.default = obj; return newObj; } }

function _objectSpread(target) { for (var i = 1; i < arguments.length; i++) { var source = arguments[i] != null ? arguments[i] : {}; var ownKeys = Object.keys(source); if (typeof Object.getOwnPropertySymbols === 'function') { ownKeys = ownKeys.concat(Object.getOwnPropertySymbols(source).filter(function (sym) { return Object.getOwnPropertyDescriptor(source, sym).enumerable; })); } ownKeys.forEach(function (key) { _defineProperty(target, key, source[key]); }); } return target; }

function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }

/* globals require */
let defaultIdom = null;

if (typeof window !== 'undefined') {
  // Only load incremental-dom in a browser environment
  defaultIdom = require('incremental-dom');
}

function getDefaultIdom() {
  return defaultIdom;
}

function checkRevision(compilerInfo) {
  const compilerRevision = compilerInfo && compilerInfo[0] || 1,
        currentRevision = _base.COMPILER_REVISION;

  if (compilerRevision !== currentRevision) {
    if (compilerRevision < currentRevision) {
      const runtimeVersions = _base.REVISION_CHANGES[currentRevision],
            compilerVersions = _base.REVISION_CHANGES[compilerRevision];
      throw new _exception.default('Template was precompiled with an older version of Handlebars than the current runtime. ' + 'Please update your precompiler to a newer version (' + runtimeVersions + ') or downgrade your runtime to an older version (' + compilerVersions + ').');
    } else {
      // Use the embedded version info since the runtime doesn't know about this revision yet
      throw new _exception.default('Template was precompiled with a newer version of Handlebars than the current runtime. ' + 'Please update your runtime to a newer version (' + compilerInfo[1] + ').');
    }
  }
}

const BACKEND_IDOM = 'idom';
const BACKEND_TEXT = 'text';
const BACKEND_KEYWORDS = [BACKEND_IDOM, BACKEND_TEXT];

function template(templateSpec, env) {
  /* istanbul ignore next */
  if (!env) {
    throw new _exception.default('No environment passed to template');
  }

  if (!templateSpec || !templateSpec.main) {
    throw new _exception.default('Unknown template object: ' + typeof templateSpec);
  }

  templateSpec.main.decorator = templateSpec.main_d; // Note: Using env.VM references rather than local var references throughout this section to allow
  // for external users to override these as psuedo-supported APIs.

  env.VM.checkRevision(templateSpec.compiler);

  function invokePartialWrapper(partial, context, options) {
    if (options.hash) {
      context = Utils.extend({}, context, options.hash);
    }

    options = _objectSpread({}, options, {
      backend: BACKEND_IDOM,
      idom: this.idom,
      noEscape: this.noEscape
    });
    partial = env.VM.resolvePartial.call(this, partial, context, options);
    let result = env.VM.invokePartial.call(this, partial, context, options);

    if (result == null && env.compile) {
      options.partials[options.name] = env.compile(partial, templateSpec.compilerOptions, env);
      result = options.partials[options.name](context, options);
    }

    if (result != null) {
      return result;
    } else {
      throw new _exception.default('The partial ' + options.name + ' could not be compiled when running in runtime-only mode');
    }
  } // Just add water


  let container = {
    buildDynamicOpeningTag: function (name, key, contents) {
      const keyValuePairs = (0, _attributesParser.parseAttributes)(contents);
      const dynamicAttrs = [];
      let dynamicKey = undefined;

      for (let [key, value] of keyValuePairs) {
        dynamicAttrs.push(key);
        dynamicAttrs.push(value);

        if (key === 'data-idom-key') {
          dynamicKey = value;
        }
      }

      const actualKey = typeof dynamicKey === 'string' ? dynamicKey : key;
      this.idom.elementOpen(name, actualKey, undefined, ...dynamicAttrs);
    },
    strict: function (obj, name) {
      if (!(name in obj)) {
        throw new _exception.default('"' + name + '" not defined in ' + obj);
      }

      return obj[name];
    },
    lookup: function (depths, name) {
      const len = depths.length;

      for (let i = 0; i < len; i++) {
        if (depths[i] && depths[i][name] != null) {
          return depths[i][name];
        }
      }
    },
    lambda: function (current, context) {
      return typeof current === 'function' ? current.call(context) : current;
    },
    extend: Utils.extend,
    runIdomToText: _idomTextBackend.runIdomToText,
    escapeExpression: function (text) {
      if (this.noEscape) {
        return text;
      }

      return Utils.escapeExpression(text);
    },
    invokePartial: invokePartialWrapper,
    fn: function (i) {
      let ret = templateSpec[i];
      ret.decorator = templateSpec[i + '_d'];
      return ret;
    },
    programs: [],
    program: function (i, data, declaredBlockParams, blockParams, depths) {
      let programWrapper = this.programs[i],
          fn = this.fn(i);

      if (data || depths || blockParams || declaredBlockParams) {
        programWrapper = wrapProgram(this, i, fn, data, declaredBlockParams, blockParams, depths);
      } else if (!programWrapper) {
        programWrapper = this.programs[i] = wrapProgram(this, i, fn);
      }

      return programWrapper;
    },
    data: function (value, depth) {
      while (value && depth--) {
        value = value._parent;
      }

      return value;
    },
    merge: function (param, common) {
      let obj = param || common;

      if (param && common && param !== common) {
        obj = Utils.extend({}, common, param);
      }

      return obj;
    },
    appendResult: function (thunkOrText) {
      if (typeof thunkOrText === 'function') {
        // An incremental-dom thunk
        thunkOrText();
      } else if (typeof thunkOrText === 'string') {
        // Unescaped text
        this.idom.text(thunkOrText);
      }
    },
    // An empty object to use as replacement for null-contexts
    nullContext: Object.seal({}),
    noop: env.VM.noop,
    compilerInfo: templateSpec.compiler
  };

  function ret(context, options = {}) {
    let data = options.data;

    _setup(options);

    if (!options.partial && templateSpec.useData) {
      data = initData(context, data);
    }

    let depths,
        blockParams = templateSpec.useBlockParams ? [] : undefined;

    if (templateSpec.useDepths) {
      if (options.depths) {
        depths = context != options.depths[0] ? [context].concat(options.depths) : options.depths;
      } else {
        depths = [context];
      }
    }

    function main(context, options) {
      if (options.backend != null && BACKEND_KEYWORDS.indexOf(options.backend) < 0) {
        let validKeywords = BACKEND_KEYWORDS.map(keyword => `'${keyword}'`).join(', ');
        throw new _exception.default(`Expected one of ${validKeywords} for options.backend, got ${options.backend}`);
      }

      if (options.backend === BACKEND_IDOM) {
        let noEscape = options.noEscape == null ? true : options.noEscape;
        let idom = options.idom || env.idom; // Main is a thunk for incremental-dom

        return templateSpec.main(_objectSpread({}, container, {
          noEscape,
          idom
        }), context, container.helpers, container.partials, data, blockParams, depths);
      } else {
        let noEscape = options.noEscape == null ? false : options.noEscape;
        return (0, _idomTextBackend.runIdomToText)(idom => {
          let thunk = templateSpec.main(_objectSpread({}, container, {
            noEscape,
            idom
          }), context, container.helpers, container.partials, data, blockParams, depths);
          thunk();
        });
      }
    }

    main = executeDecorators(templateSpec.main, main, container, options.depths || [], data, blockParams);
    return main(context, options);
  }

  ret.isTop = true;

  function _setup(options) {
    if (!options.partial) {
      container.helpers = container.merge(options.helpers, env.helpers);

      if (templateSpec.usePartial) {
        container.partials = container.merge(options.partials, env.partials);
      }

      if (templateSpec.usePartial || templateSpec.useDecorators) {
        container.decorators = container.merge(options.decorators, env.decorators);
      }
    } else {
      container.helpers = options.helpers;
      container.partials = options.partials;
      container.decorators = options.decorators;
    }
  }

  return ret;
}

function wrapProgram(container, i, fn, data, declaredBlockParams, blockParams, depths) {
  function prog(context, options = {}) {
    let currentDepths = depths;

    if (depths && context != depths[0] && !(context === container.nullContext && depths[0] === null)) {
      currentDepths = [context].concat(depths);
    }

    return fn(container, context, container.helpers, container.partials, options.data || data, blockParams && [options.blockParams].concat(blockParams), currentDepths);
  }

  prog = executeDecorators(fn, prog, container, depths, data, blockParams);
  prog.program = i;
  prog.depth = depths ? depths.length : 0;
  prog.blockParams = declaredBlockParams || 0;
  return prog;
}

function resolvePartial(partial, context, options) {
  if (!partial) {
    if (options.name === '@partial-block') {
      partial = options.data['partial-block'];
    } else {
      partial = options.partials[options.name];
    }
  } else if (!partial.call && !options.name) {
    // This is a dynamic partial that returned a string
    options.name = partial;
    partial = options.partials[partial];
  }

  return partial;
}

function invokePartial(partial, context, options) {
  // Use the current closure context to save the partial-block if this partial
  const currentPartialBlock = options.data && options.data['partial-block'];
  options.partial = true;
  let partialBlock;

  if (options.fn && options.fn !== noop) {
    options.data = (0, _base.createFrame)(options.data); // Wrapper function to get access to currentPartialBlock from the closure

    let fn = options.fn;

    partialBlock = options.data['partial-block'] = function partialBlockWrapper(context, options = {}) {
      // Restore the partial-block from the closure for the execution of the block
      // i.e. the part inside the block of the partial call.
      options.data = (0, _base.createFrame)(options.data);
      options.data['partial-block'] = currentPartialBlock;
      return fn(context, options);
    };

    if (fn.partials) {
      options.partials = Utils.extend({}, options.partials, fn.partials);
    }
  }

  if (partial === undefined && partialBlock) {
    partial = partialBlock;
  }

  if (partial === undefined) {
    throw new _exception.default('The partial ' + options.name + ' could not be found');
  } else if (partial instanceof Function) {
    return partial(context, options);
  }
}

function noop() {}

function initData(context, data) {
  if (!data || !('root' in data)) {
    data = data ? (0, _base.createFrame)(data) : {};
    data.root = context;
  }

  return data;
}

function executeDecorators(fn, prog, container, depths, data, blockParams) {
  if (fn.decorator) {
    let props = {};
    prog = fn.decorator(prog, props, container, depths && depths[0], data, blockParams, depths);
    Utils.extend(prog, props);
  }

  return prog;
}